 /************************************************************************/
 /*                                                                      */
 /*                Centre for Speech Technology Research                 */
 /*                     University of Edinburgh, UK                      */
 /*                       Copyright (c) 1996,1997                        */
 /*                        All Rights Reserved.                          */
 /*                                                                      */
 /*  Permission to use, copy, modify, distribute this software and its   */
 /*  documentation for research, educational and individual use only, is */
 /*  hereby granted without fee, subject to the following conditions:    */
 /*   1. The code must retain the above copyright notice, this list of   */
 /*      conditions and the following disclaimer.                        */
 /*   2. Any modifications must be clearly marked as such.               */
 /*   3. Original authors' names are not deleted.                        */
 /*  This software may not be used for commercial purposes without       */
 /*  specific prior written permission from the authors.                 */
 /*                                                                      */
 /*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK       */
 /*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING     */
 /*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT  */
 /*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE    */
 /*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES   */
 /*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN  */
 /*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,         */
 /*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF      */
 /*  THIS SOFTWARE.                                                      */
 /*                                                                      */
 /*************************************************************************/

class UnitDatabase;
class UnitName;
class UnitProperties;
class UnitCatalogue;

#ifndef __UNITDATABASE_H__
#define __UNITDATABASE_H__

#include "EST.h"
#include "EST_TNamedEnum.h"
#include "EST_THash.h"
#include "EST_Pathname.h"
#include "EST_ContourType.h"
#include "EST_TVector.h"
#include "SubtypedObject.h"
#include "UnitAux.h"

class Unit;
class UnitIndex;
class UnitCatalogue;

/** Database containing synthesis information. 
  *
  * This is a complete database. It is responsible for cacheing units
  * and for knowing how to interface to the catalogue to get information
  * from disk.
  *
  * A database contains units each of which has a name, some properties
  * and a collection of synthesis information. A piece of synthesis 
  * information is either a waveform or a set of coefficiants. We
  * don't specify what each of these pieces of information is, but we 
  * provide a set of standard contents.
  *
  * Since these are big and heavy we pass arount pointers to them
  * rather than the objects themselves.
  *
  * @see UnitAux.h
  * @see Unit
  * @see UnitIndex
  * @see UnitCatalogue
  * @author Richard Caley <rjc@cstr.ed.ac.uk>
  * @version $Id: UnitDatabase.h,v 1.1 1998/08/12 09:35:52 awb Exp $
  */

class UnitDatabase : public SubtypedObject<UnitDatabase> {

public:

    /**@name Synthesis Information */
    //@{
#   define CONTENT_WAVE(N) ((N)<<1)
#   define CONTENT_COEF(N) (((N-1)<<1) | 1)
    /// Standard information types.
    enum ContentType { 
	/// Nothing
	content_none=0, 
	/// Actual speech signal.
	content_signal=CONTENT_WAVE(1), 
	/// Inverse filtered residual.
	content_residual=CONTENT_WAVE(2), 
	content_experimentalwave1=CONTENT_WAVE(5), 
	content_experimentalwave2=CONTENT_WAVE(6), 
	/// Pitchmarks
	content_pitchmarks=CONTENT_COEF(1), 
	/// LPC/Reflection/whatever coefficiants
	content_analysis=CONTENT_COEF(2),
	content_experimentalcoef1=CONTENT_COEF(5), 
	content_experimentalcoef2=CONTENT_COEF(6), 
	/// No contents after this.
	content_maximum=CONTENT_COEF(16)
    };
    typedef enum ContentType ContentType;

    /// Is this content type a waveform?
    static inline int is_waveform(int n) { return (n&1) == 0; }
    /// Is it a coefficient set?
    static inline int is_coefficients(int n) { return (n&1) == 1; }
    /// Type for Nth waveform
    static inline ContentType content_waveform(int n)  
	{ return (ContentType)CONTENT_WAVE(n);};
    /// Type for Nth coefficient set
    static inline ContentType content_coefficient(int n)  
	{ return (ContentType)CONTENT_COEF(n);};
#undef CONTENT_WAVE
#undef CONTENT_COEF
    /// Mapping from content types to names
    static  EST_TNamedEnum<ContentType> ContentTypeMap;
    //@}

    /// How the units should be accessed and cached.
    enum AccessStrategy { 
	/// Read when catalogue opened.
	as_direct, 
	/// Read when needed, keep in memory
	as_ondemand,
	/// Read when needed, delete after use.
	as_dynamic
    };
    typedef enum AccessStrategy AccessStrategy;

    /// Mapping from names to access strategies.
  static  EST_TNamedEnum<AccessStrategy> AccessStrategyMap;

    /// Type of preprocessing done on signal.
    enum SignalPreprocessing { 
	/// None done.
	sp_none 
    };
  typedef SignalPreprocessing SignalPreprocessing;

    /// Mapping from names to preprocessing types.
    static  EST_TNamedEnum<SignalPreprocessing> SignalPreprocessingMap;

    /// Sturcture describing a waveform.
    struct wave_information_s {
	/// Format it was saved in.
	EST_String format;
	/// Format of each sample.
	EST_String sample_format;
	/// What kind of thing is it?
        ContentType pm_content_type;
	/// Sample rate.
	int sample_rate;
	/// How to access it.
	AccessStrategy strategy;
	/// How it was preprocessed.
	SignalPreprocessing preprocessing;
    };

    /// Structure describing a coefficiant set.
    struct coefficient_information_s {
	/// Format it was saved in.
	EST_String format;
	/// frame spaceing.
	float frame_shift;
	/// Assumed sample rate.
	int sample_rate;
	/// How to access it.
	AccessStrategy strategy;
	/// Type of coefficiants.
	EST_ContourType coefficient_format;
	/// Where to get the time from.
	EST_String time_channel;
	/// Where to get the length from.
	EST_String length_channel;
	/// Scale factor for time channel.
	float time_scale;
    };

    /// Structure describing an index.
    struct index_information_s {
	/// Name (could be filename)
	EST_String name;
	/// Subtype of UnitIndex
	EST_String type;
	/// Pointer to it.
	UnitIndex *index;
    };
  
private:
    /// Human readable description of the database.
    EST_String p_description;
    /// Copyright information.
    EST_String p_copyright;

    /// File where the database can be found.
    EST_Pathname p_filename;

    /// Subtype of Unit which should be created for each unit.
    EST_String p_unit_type;

    /// The proneme set used by this database.
    PhoneSet *p_phoneset;

    /// Unit catalogue used to find the data.
    UnitCatalogue *p_catalogue;

    /// Indexes.
    EST_TVector<index_information_s> p_indexes;

    /// Base size of hastable used to cache units.
#   define UnitCacheHashtableSize (100)

    /// Units are cached in this hash table.
    EST_TStringHash<Unit *> p_unit_cache;

    /// Arbitrary properties associated with this database.
    KVL<EST_String, EST_String> p_properties;

protected:
    /// Take the name of a piece of synthesis information and return the numeric code.
    static ContentType contents_name_to_number(EST_String name);

    /// Default constructor.
    UnitDatabase (void) ;

    /// Set the human readable description.
    void set_description(EST_String description) 
	INLINE_IF_OK({p_description=description;});
    /// Set the copyright information.
    void set_copyright(EST_String copyright) 
	INLINE_IF_OK({p_copyright = copyright;});
    /// Say what kind of unit this database should contain.
    void set_unit_type(EST_String unit_type) 
	INLINE_IF_OK({p_unit_type = unit_type;});
    /// Assign a catalogue.
    void set_catalogue(UnitCatalogue *catalogue) 
	INLINE_IF_OK({p_catalogue = catalogue;});

    /// Assign a phoneme set.
    void set_phoneset(PhoneSet *phones) 
	INLINE_IF_OK({p_phoneset = phones;});

    /// Add an new index.
    void add_index(EST_String name, EST_String type, UnitIndex *index);
  
public:
    /// Destructor
    virtual ~UnitDatabase(void) ;


    /**@name Subtyping */
    //@{
    /// Read in from the file.
    virtual EST_read_status fill_from(FILE *stream) ;
    /// Write out to file.
    virtual EST_write_status write_to(FILE *stream) ;
    /// Record the filename.
    void set_filename(EST_String filename) 
	INLINE_IF_OK({p_filename = filename;});
    //@}

    /** Database compilation.
      * Take a (presumably empty) database and insert the
      * contents of an existing one.
      */
    virtual void fill_from(UnitDatabase &source);


    /**@name Information about the database */
    //@{
    /// Human readable description of database.
    EST_String description(void) const {return p_description;};
    /// Copyright notice for the database.
    EST_String copyright(void) const {return p_copyright;};
    /// Filename where database was found.
    EST_String filename(void) const {return p_filename;};
    /// Subtype of Unit which will be returned. 
    EST_String unit_type(void) const {return p_unit_type;};
    /// Phoneme set used by this database.
    PhoneSet *phoneset(void) const {return p_phoneset;};
    /// Unit catalogue.
    UnitCatalogue *catalogue(void) const {return p_catalogue;};
    //@}

    /**@name Synthesis information. */
    //@{
    /// Names of the waveforms available for each unit.
    virtual EST_StrVector wave_names(void) const ;
    /// Names of the coefficient sets available for each unit.
    virtual EST_StrVector coefficient_names(void) const ;
    /// Names of the indexes available for this database.
    virtual EST_StrVector index_names(void) const ;

    /// Return the information about the given waveform.
    virtual wave_information_s *wave_information(ContentType content) ;
    /// Return the information about the given coefficiant set.
    virtual coefficient_information_s *coefficient_information(ContentType content) ;
    /// Return the information about the given index.
    virtual index_information_s *index_information(EST_String type);

    /// Get a given waveform for this unit.
    virtual EST_Wave *get_wave(UnitDatabase::ContentType content, Unit &unit) ;
    /// Get a given coefficient set for this unit.
    virtual EST_Track *get_coefficients(ContentType content, EST_ContourType format, Unit &unit) ;
    /// Get the division of the unit into segments.
    virtual EST_TVector<float> *get_segments(Unit &unit) ;
    //@}

    /// Return the named unit.
    virtual const Unit *unit(const EST_String name);
    /// Return the named index.
    virtual const UnitIndex *index(const EST_String type);


    /**@name Properties */
    //@{
    /// Names of all available properties.
    virtual void property_names(EST_TList<EST_String> &list) const;
    /// Set a single property
    virtual int set_property(EST_String property, ValueType value);
    /// Get the value for a property.
    virtual ValueType property(EST_String property) const;
    //@}


    /// Human readable description of the database with some detail.
    virtual void print_description(FILE *stream);

#if defined(INCLUDE_LISP)
  /**@name Scheme interface
    * These are the functions which are imported into Scheme. 
    */
  //@{
    /// Describe the database
    static LISP lisp_print_description(LISP database, LISP file);
    /// Initialisation function.
    static void lisp_declare(void);
    //@}
#endif

    friend class SubtypedObject<UnitDatabase>;
};

#include "Unit.h"
#include "UnitIndex.h"
#endif
