/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                       Copyright (c) 1996,1997                         */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                 Authors:  Alan W Black                                */
/*                 Date   :  December 1997                               */
/*-----------------------------------------------------------------------*/
/*  Access to the Ngrammar                                               */
/*                                                                       */
/*=======================================================================*/
#include <stdio.h>
#include "festival.h"
#include "festivalP.h"

static LISP ngram_loaded_list = NIL;
static EST_Ngrammar *load_ngram(const EST_String &filename);

static LISP add_ngram(const EST_String &name,EST_Ngrammar *n);

static int tc_ngrammar = -1;

static EST_Ngrammar *get_c_ngrammar(LISP x)
{
    if (TYPEP(x,tc_ngrammar))
	return (EST_Ngrammar *)USERVAL(x);
    else
	err("wta to get_c_ngrammar",x);
    return NULL;  // err doesn't return but compilers don't know that
}

static LISP siod_make_ngrammar(EST_Ngrammar *n)
{
    if (n==0)
	return NIL;
    else
	return siod_make_typed_cell(tc_ngrammar,n);
}

static LISP lisp_load_ngram(LISP name, LISP filename)
{
    EST_Ngrammar *n;

    n = load_ngram(get_c_string(filename));
    add_ngram(get_c_string(name),n);

    return name;
}

static EST_Ngrammar *load_ngram(const EST_String &filename)
{
    EST_Ngrammar *n = new EST_Ngrammar();
    if (n->load(filename) != 0)
    {
	cerr << "Ngrammar: failed to read ngrammar from \"" <<
	    filename << "\"" << endl;
	festival_error();
    }

    return n;
}

static LISP add_ngram(const EST_String &name,EST_Ngrammar *n)
{
    EST_Ngrammar *old_n;
    LISP lpair; 

    lpair = siod_assoc_str(name,ngram_loaded_list);

    if (ngram_loaded_list == NIL)
    {   // First time round so do a little initialization 
	gc_protect(&ngram_loaded_list);
	tc_ngrammar = siod_register_user_type("Ngrammar");
    }

    LISP ng = siod_make_ngrammar(n);
    
    if (lpair == NIL)
	ngram_loaded_list = 
	    cons(cons(strintern(name),cons(ng,NIL)),ngram_loaded_list);
    else
    {
	cwarn << "Ngrammar: " << name << " recreated" << endl;
	old_n = get_c_ngrammar(car(cdr(lpair)));
	delete old_n;
	setcar(cdr(lpair),ng);
    }

    return ng;
}

EST_Ngrammar *get_ngram(const EST_String &name,const EST_String &filename)
{
    //  Find ngram named name, returns NULL if none;
    LISP lpair;
    
    lpair = siod_assoc_str(name,ngram_loaded_list);

    if (lpair == NIL)
    {
	if (filename != EST_String::Empty)
	{
	    EST_Ngrammar *n = load_ngram(filename);
	    add_ngram(name,n);
	    return n;
	}
	else
	{
	    cwarn << "Ngrammar: no ngram named \"" << name << "\"" << endl;
	    return 0;
	}
    }
    else
	return get_c_ngrammar(car(cdr(lpair)));
}

void festival_ngram_init()
{
    init_subr_2("ngram.load",lisp_load_ngram,
 "(ngram.load NAME FILENAME)\n\
  Load an ngram from FILENAME and store it named NAME for later access.");

}
