/********************************************************************************
 * Copyright (c) Des Herriott 1993, 1994
 *               Erik Kunze   1995 - 1998
 *
 * Permission to use, distribute, and sell this software and its documentation
 * for any purpose is hereby granted without fee, provided that the above
 * copyright notice appear in all copies and that both that copyright notice and
 * this permission notice appear in supporting documentation, and that the name
 * of the copyright holder not be used in advertising or publicity pertaining to
 * distribution of the software without specific, written prior permission.  The
 * copyright holder makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or implied
 * warranty. THE CODE MAY NOT BE MODIFIED OR REUSED WITHOUT PERMISSION!
 *
 * THE COPYRIGHT HOLDER DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
 * TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 *
 * Author: Des Herriott
 *         Erik Kunze
 *
 * changed by EKU
 *******************************************************************************/
#ifndef lint
static char rcsid[] = "$Id: mem.c,v 4.8 1998/02/07 17:26:16 erik Rel $";
#endif

#include <stdio.h>
#include <string.h>
#include <assert.h>
#include "config.h"
#include "z80.h"
#include "resource.h"
#include "io.h"
#include "util.h"
#ifdef AUDIO
#include "audio.h"
#endif
#ifdef XZX_IF1
#include "if1.h"
#endif
#include "loadsave.h"
#include "mem.h"


static int loadRomImage(char *, int);


int RealPageNumbers[8];
uns8 *TheMemory[8];
uns8 *RealMemory[NPAGES];
#ifdef EMULATE_ULA_DELAY
int ContendedMemory[8];
#endif
MachineSettings Machines[NMACHINES] = {
	{
		SP48ROM,
		ROM_UNKNOWN,
		312,
		224,
		69888,
		0xff
	},
	{
		SP48ROM,
		ROM_UNKNOWN,
		312,
		224,
		69888,
		0xbf
	},
	{
		SP128ROM1,
		ROM_UNKNOWN,
		311,
		228,
		70908,
		0xbf
	},
#ifdef XZX_PLUS3
	{
		SP3ROM3,
		ROM_UNKNOWN,
		311,
		228,
		70908,
		0xbf
	},
#endif
}, *Machine;


void
MemoryInit(void)
{
	int checksum, romtype, i;
	for (i = 0; i < NPAGES; i++)
	{
		RealMemory[i] = (uns8 *)Malloc(XZX_PAGESIZE, "MemoryInit");
	}
#ifdef EMULATE_ULA_DELAY
	memset(ContendedMemory, 0, sizeof(ContendedMemory));
	ContendedMemory[2] = ContendedMemory[3] = 1;
#endif
	if (!loadRomImage(GETCFG(rom48), SP48ROM))
	{
		checksum = GetRomChecksum(Machines[SP_48_3].basicRom, XZX_PAGESIZE);
		switch (checksum)
		{
			case 8686:
				romtype = ROM_STANDARD;
				break;
			case 44175:
				romtype = ROM_TK95;
				break;
			case 58561:
				romtype = ROM_GROOT;
				break;
			case 53858:
				romtype = ROM_IMC;
				break;
			default:
				Msg(M_WARN, "unknown 48K ROM image - checksum %u", checksum);
				romtype = ROM_UNKNOWN;
				break;
		}
		Machines[SP_48_2].romType = romtype;
		Machines[SP_48_3].romType = romtype;
		TpPatchRom(SP_48_3);
#ifdef XBELL_AUDIO
		SetBeepTrap(SP_48_3);
#endif
	}
	if (!loadRomImage(GETCFG(rom128_0), SP128ROM0) &&
		!loadRomImage(GETCFG(rom128_1), SP128ROM1))
	{
		checksum = GetRomChecksum(Machines[SP_128].basicRom, XZX_PAGESIZE);
		switch (checksum)
		{
			case 44809:
				romtype = ROM_128;
				break;
			case 32991:
				romtype = ROM_PLUS2;
				break;
			default:
				Msg(M_WARN, "unknown 128K ROM image - checksum %u", checksum);
				romtype = ROM_UNKNOWN;
				break;
		}
		Machines[SP_128].romType = romtype;
		TpPatchRom(SP_128);
#ifdef XBELL_AUDIO
		SetBeepTrap(SP_128);
#endif
	}
#ifdef XZX_PLUS3
	if (!loadRomImage(GETCFG(rompl3_0), SP3ROM0) &&
		!loadRomImage(GETCFG(rompl3_1), SP3ROM1) &&
		!loadRomImage(GETCFG(rompl3_2), SP3ROM2) &&
		!loadRomImage(GETCFG(rompl3_3), SP3ROM3))
	{
		checksum = GetRomChecksum(Machines[SP_3].basicRom, XZX_PAGESIZE);
		switch (checksum)
		{
			case 48934:
				romtype = ROM_PLUS3;
				break;
			default:
				Msg(M_WARN, "unknown +3 ROM image - checksum %u", checksum);
				romtype = ROM_UNKNOWN;
				break;
		}
		Machines[SP_3].romType = romtype;
		TpPatchRom(SP_3);
#ifdef XBELL_AUDIO
		SetBeepTrap(SP_3);
#endif
	}
#endif
#ifdef XZX_IF1
	if (!loadRomImage(GETCFG(if1_rom), IF1ROM))
	{
		If1PatchRom();
	}
	else
	{
		Msg(M_INFO, "IF1 emulation disabled");
		SETCFG(if1_active, 0);
	}
#endif
#ifdef XZX_MF128
	if (loadRomImage(GETCFG(mf128_rom), MF128ROM))
	{
		Msg(M_INFO, "MF128 emulation disabled");
		SETCFG(mf128_active, 0);
	}
#endif
	assert(GETCFG(machine) < NMACHINES);
	Machine = &Machines[GETCFG(machine)];
}

void
InitPages(void)
{
	switch (GETCFG(machine))
	{
		case SP_48_2:
		case SP_48_3:
			PageIn(0, SP48ROM);
			break;
		case SP_128:
			PageIn(0, SP128ROM0);
			break;
#ifdef XZX_PLUS3
		case SP_3:
			PageIn(0, SP3ROM0);
			break;
#endif
	}
	PageIn(1, RAM5);
	PageIn(2, RAM2);
	PageIn(3, RAM0);
}

void
PageIn(int frame, int rpage)
{
	int i = frame << 1;

#ifdef XZX_PLUS3
	assert(frame || rpage < RAM0 || GETCFG(machine) == SP_3);
#else
	assert(frame || rpage < RAM0);
#endif
	TheMemory[i] = RealMemory[rpage];
	RealPageNumbers[i++] = rpage;
	TheMemory[i] = RealMemory[rpage] + XZX_PAGESIZE / 2;
	switch (rpage)
	{
#ifdef XZX_MF128
		case MF128ROM:
			RealPageNumbers[i] = MF128RAM;
			break;
#endif
		default:
			RealPageNumbers[i] = rpage;
			break;
	}
#ifdef EMULATE_ULA_DELAY
	if (frame == 3)
	{
		ContendedMemory[6] = ContendedMemory[7] =
									GETCFG(machine) >= SP_128 && rpage > RAM3;
	}
#endif
#ifdef XZX_PLUS3

	if (GETCFG(machine) == SP_3 && !frame && rpage < RAM0)
	{
		PageIn(1, RAM5);
		PageIn(2, RAM2);
		PageIn(3, RAM0 + (Last0x7FFD & B_SELRAM));
	}
#endif
}

static int
loadRomImage(char *image, int page)
{
	FILE *ifp;
	size_t nread;
	if (!(ifp = Fopen(image, "rb")))
	{
		Msg(M_PERR, "couldn't open ROM image <%s> for reading", image);
		return -1;
	}
	nread = fread(RealMemory[page], 1, XZX_PAGESIZE, ifp);

	if (nread != XZX_PAGESIZE && nread != XZX_PAGESIZE / 2)
	{
		Msg(M_WARN, "ROM image <%s> is only %d bytes", image, nread);
	}
	(void)fclose(ifp);
	return 0;
}

int
GetRomChecksum(int rom, int length)
{
	int checksum = 0, tmp, i;
	for (i = 0; i < length; i++) {
		tmp = checksum & 0x01;
		checksum >>= 1;
		if (tmp)
		{
			checksum |= 0x8000;
		}
		checksum ^= RealMemory[rom][i];
	}
	return checksum;
}

void
SelectModel(int model, int reset)
{
	assert(model < NMACHINES);
	if (GETCFG(machine) != model)
	{
		SETCFG(machine, model);
		Machine = &Machines[model];
#if defined(XZX_PLUS3) && (defined(XZX_IF1) || defined(XZX_MF128))
		if (model == SP_3)
		{

#ifdef XZX_IF1
			SETCFG(if1_active, 0);
#endif
#ifdef XZX_MF128
			SETCFG(mf128_active, 0);
#endif
		}
#endif
#ifdef SUN_AUDIO
		SpeakerReInit();
#endif
	}
	if (reset)
	{

		Z80_Reset();
	}
	else
	{
		InitPages();
	}
}

