/* This file is Copyright 1993 by Clifford A. Adams */
/* sgdata.c
 *
 * mostly manipulation of the structures sg_ents and sg_groups
 */

#include "EXTERN.h"
#include "common.h"
#ifdef SCAN
#include "cache.h"
#include "final.h"	/* assert() */
#include "ng.h"		/* for NG_PREV, etc... */
#include "ngdata.h"
#include "nntp.h"
#include "trn.h"
#include "rcln.h"
#include "rcstuff.h"
#include "scan.h"
#include "scmd.h"
#include "sdisp.h"
#include "sorder.h"
#include "scanart.h"	/* sa_mode_read_elig */
#include "sgroup.h"
#include "sgmisc.h"
#include "term.h"
#include "util.h"
#ifdef SERVER
#include "server.h"
#endif
#include "INTERN.h"
#include "sgdata.h"

/* initialize current context with default values */
void
sg_init_context()
{
    sg_contexts = (SG_CONTEXT*)NULL;
    sg_max_contexts = sg_num_contexts = 0;
}

/* saves the current context then creates and initializes a new context */
void
sg_new_context()
{
    int cnum;
    int i;

    /* save the old context */
    i = sg_num_contexts-1;
    if (i>=0) {			/* there was an old one */
	s_save_context();
	sg_contexts[i].num_ents = sg_num_ents;
	sg_contexts[i].ents_alloc = sg_ents_alloc;
	sg_contexts[i].ents = sg_ents;
	/* save flags here later */
    }

    sg_num_contexts++;
    cnum = s_new_context(S_GROUP);
    s_change_context(cnum);
    i = sg_num_contexts-1;
    if (i==0) {		/* first context */
	sg_max_contexts = sg_num_contexts;
	sg_contexts = (SG_CONTEXT*)safemalloc(sizeof(SG_CONTEXT));
    }
    if (sg_num_contexts>sg_max_contexts) {
	sg_contexts = (SG_CONTEXT*)saferealloc((char*)sg_contexts,
			(i+1)*sizeof(SG_CONTEXT));
	sg_max_contexts++;
    }
    sg_contexts[i].cnum = cnum;
    sg_init_ents();

    sg_contexts[i].filename = (char*)NULL;
    sg_contexts[i].label = (char*)NULL;
    sg_contexts[i].tmpfilename = (char*)NULL;
    sg_contexts[i].title = (char*)NULL;
    /* set flags to defaults (?) later */
    sg_set_screen();
}

/* deletes the current context and restores the last saved one */
void
sg_old_context()
{
    int i;

    /* free the entries (and sub-structures) */
    sg_free_ents();
    /* delete the scan-context */
    i = sg_num_contexts-1;
    s_delete_context(sg_contexts[i].cnum);
    free(sg_contexts[i].title);
    sg_contexts[i].title = (char*)NULL;
    if (sg_contexts[i].filename) {
	free(sg_contexts[i].filename);
	sg_contexts[i].filename = (char*)NULL;
    }
    if (sg_contexts[i].label) {
	free(sg_contexts[i].label);
	sg_contexts[i].label = (char*)NULL;
    }
    if (sg_contexts[i].tmpfilename) {
	/* delete temporary files */
        if (strnEQ(sg_contexts[i].tmpfilename,"/tmp/strn",9)) {
	    UNLINK(sg_contexts[i].tmpfilename);
	}
	free(sg_contexts[i].tmpfilename);
	sg_contexts[i].tmpfilename = (char*)NULL;
    }
    sg_num_contexts--;

    /* bring back the old context */
    i = sg_num_contexts-1;
    if (i<0)
	return;
    s_change_context(sg_contexts[i].cnum);
    sg_num_ents = sg_contexts[i].num_ents;
    sg_ents_alloc = sg_contexts[i].ents_alloc;
    sg_ents = sg_contexts[i].ents;
    /* restore flags? */
}

/* text for a moderated newsgroup */
/* do this better later */
static char *modstr;

/* initializes the newsgroup array */
void	/* may change */
sg_init_groups()
{
    int ngnum;

    printf("\nReading in newsgroups.") FLUSH;
    fflush(stdout);
    modstr = savestr(getval("MODSTRING"," (moderated)"));
    for (ngnum = 0; ngnum < nextrcline; ngnum++)
	if (toread[ngnum]>=TR_UNSUB)
	    sg_add_ng(ngnum);
    sg_look_num = 0;
    printf("  Done.\n") FLUSH;
    /* don't free modstr in case groups are added online */
}

/* called when newsgroups are moved around in the rcline array. */
void
sg_reinit_groups()
{
    int i,rctmp;

    /* won't catch new newsgroups, but will keep old ones sane */
    for (i=0;i<sg_num_groups;i++) {
	rctmp =  find_ng(sg_groups[i].name);
	if ((rctmp<0) || (rctmp>=nextrcline))
	    rctmp = -1;
	sg_groups[i].rcnum = rctmp;
    }
/* Maybe I can have some kind of flag set parallel to rcline which I can
 * use to see when new newsgroups appear.  The idea is to mark each one
 * group scan know about in the loop above, then search for unmarked ones
 * which are not TR_BOGUS and add them using sg_add_ng.
 */
    sg_look_num = 0;		/* reinitialize lookahead */
    sg_changed_newsrc = FALSE;	/* back to sanity */
}

bool
sg_init_ents()
{
    sg_num_ents = sg_ents_alloc = 0;
    sg_ents = (SG_ENTRYDATA*)NULL;
    return(TRUE);
}

/* returns TRUE on success */
bool
sg_add_ng(line)
int line;	/* line of newsrc */
{
    int cur;		/* current group for working */

    /* not in .newsrc == not available for now */
    if (line >= nextrcline) {
	printf("\nsg_add_ng: line (%d) >= nextrcline (%d) (not in newsrc)\n",
		line,nextrcline) FLUSH;
	return(FALSE);
    }
    if (line < 0) {
	printf("\nsg_add_ng: line (%d) < 0 (not in newsrc)\n",
		line) FLUSH;
	return(FALSE);
    }
    sg_num_groups++;
    if (sg_num_groups > sg_groups_alloc) {
	sg_groups_alloc += 100;
	if (sg_groups_alloc==100) {	/* newly allocated */
	    sg_groups = (SG_GROUPDATA*)safemalloc(sg_groups_alloc*
					sizeof(SG_GROUPDATA));
        } else {
	    sg_groups = (SG_GROUPDATA*)saferealloc((char*)sg_groups,
			sg_groups_alloc*sizeof(SG_GROUPDATA));
	}
    }
    cur = sg_num_groups-1;
    sg_groups[cur].rcnum = line;
    sg_groups[cur].desc = (char*)NULL;
    sg_groups[cur].name = savestr(rcline[line]);
    sg_groups[cur].flags = (long)0;
    return(TRUE);
}

/* Sets the toread[] entry for a newsgroup *quietly*. */
void
sg_toread(grp,refresh)
long grp;		/* index into sg_groups */
bool_int refresh;	/* if true, check even if done before */
{
    long line;		/* index into rc arrays */

    /* don't set if done before, unless specifically requested */
    if ((sg_groups[grp].flags & 2) && !refresh)
	return;
    line = sg_groups[grp].rcnum;
    if ((line>=nextrcline) || (line<0))
	return;		/* just hope things go OK */
    moderated = nullstr;
    sg_quiet_toread = TRUE;	/* so no messages are printed */
    set_toread(line);
    sg_quiet_toread = FALSE;	/* back to normal */
    /* check to see if group is moderated */
    /* roundabout way of checking--add a flag later? */
    if (toread[line]>=TR_UNSUB)		/* good group */
	if (strEQ(moderated,modstr))	/* side effect of set_toread */
	    sg_groups[grp].flags |= 1;
    
    sg_groups[grp].flags |= 2;
}

/* only passes type and dname since other data is type-specific */
/* returns index into sg_ents */
int
sg_add_ent(type,dname)
int type;
char *dname;		/* description name */
{
    int cur;			/* current entry number */

    sg_num_ents++;
    if (sg_num_ents > sg_ents_alloc) {
	sg_ents_alloc += 100;
	if (sg_ents_alloc==100) {	/* newly allocated */
	    /* don't use number 0--just allocate it and skip it */
	    sg_num_ents = 2;
	    sg_ents = (SG_ENTRYDATA*)safemalloc(sg_ents_alloc*
					sizeof(SG_ENTRYDATA));
        } else {
	    sg_ents = (SG_ENTRYDATA*)saferealloc((char*)sg_ents,
			sg_ents_alloc*sizeof(SG_ENTRYDATA));
	}
    }
    cur = sg_num_ents-1;
    s_order_add((long)cur);	/* don't need to sort right away */
    sg_ents[cur].type = type;
    sg_ents[cur].name = savestr(dname);
    sg_ents[cur].flags = 0;
    sg_ents[cur].groupnum = 0;
    sg_ents[cur].restrict = (char*)NULL;
    return(cur);
}


/* Delete invalid entries from the sg_ents array and move the rest of
 * the entries into the "hole".
 *
 * Invalid entries are defined as those with a type < 0.
 * Saved entry numbers will *not* be valid after this call.
 * (The s_order data structures are re-initialized.)
 */
void
sg_delete_invalid_ents() {
    int numvalid;
    int i,j;
    SG_ENTRYDATA *copy,*p;

    s_order_clean();
    numvalid = 1;	/* entry 0 is always valid (although unused) */
    for (i=1;i<sg_num_ents;i++)
	if (sg_ents[i].type>0)
	    numvalid++;
    if (numvalid==sg_num_ents) {
	/* re-initialize the s_order structure */
	for (i=1;i<sg_num_ents;i++)
	    s_order_add(i);
	return;
    }
    if (numvalid==1) {	/* everything was deleted */
	sg_free_ents();
	return;
    }
    copy = (SG_ENTRYDATA*)safemalloc(numvalid*sizeof(SG_ENTRYDATA));
    j = 1;
    for (i=1;i<sg_num_ents;i++)
	if (sg_ents[i].type>0) {
	    p = &sg_ents[i];
	    copy[j].type = p->type;
	    copy[j].flags = p->flags;
	    copy[j].groupnum = p->groupnum;
	    copy[j].name = p->name;
	    copy[j].restrict = p->restrict;
	    j++;
	} else {	/* invalid, so free allocated data */
	    free(sg_ents[i].name);
	    /* invalid entries will only be newsgroups */
	}
    assert(j==numvalid);
    free(sg_ents);
    sg_ents = copy;
    sg_num_ents = sg_ents_alloc = j;
    /* re-initialize the s_order structure */
    for (i=1;i<sg_num_ents;i++)
	s_order_add(i);
}

/* consider a separate function to free a single entry? */
/* frees the entry structures for the current context */
void
sg_free_ents()
{
    int i;

    /* entry 0 is uninitialized and unused */
    for (i=1;i<sg_num_ents;i++) {
	free(sg_ents[i].name);
	switch (sg_ents[i].type) {
	    case 2:	/* filename */
	    case 3:	/* restriction list */
	    case 4:	/* virtual newsgroup */
	    case 5:	/* text */
		free(sg_ents[i].restrict);
		break;
	}
    }
    if (sg_ents)
	free(sg_ents);
    sg_num_ents = sg_ents_alloc = 0;
}

bool
sg_go_ng(name)
char *name;
{
    set_ngname(name);
    ng = find_ng(ngname);
    if (ng == nextrcline) {		/* not in .newsrc == not avail. */
	printf("\nsg_go_ng(%s): attempted to go to group not in .newsrc.\n",
		name) FLUSH;
	return(FALSE);
    }
    set_toread(ng);		/* just in case something changed */
    if (toread[ng] == TR_UNSUB) {	/* unsubscribed? */
	/* not allowed now, but might be later */
	printf("\nsg_go_ng(%s): attempted to go to unsubscribed.\n",name)
	  FLUSH;
	return(FALSE);
    }
    if ((toread[ng] == 0) && (ngmax[ng] >= abs1st[ng])) {
	sg_force_read = TRUE;
	/* Hack time.  Too many things depend on this variable */
	sa_mode_read_elig = TRUE;
    } else
	sg_force_read = FALSE;
   
    return(TRUE);
}

/* initialize the entries to cover all newsgroups */
void
sg_allgroups()
{
    int i,j;

    printf("\ngroup scan: Adding all groups to menu..."); /* */
    fflush(stdout);
    (void)sg_new_context();
    for (i=0;i<sg_num_groups;i++) {
	j = sg_add_ent(1,sg_groups[i].name);
	sg_ents[j].groupnum = i;
    }
    sg_contexts[sg_num_contexts-1].title = savestr("all");
    printf("Done.\n") FLUSH;
}

/* returns TRUE if it got a description, FALSE otherwise */
bool
sg_group_desc(ng)
int ng;			/* newsgroup number, index into sg_groups */
{
    char tmpbuf[LBUFLEN];
    char *s;
    int i;

    if (sg_desc_unavail) {
	sg_use_desc = FALSE;	/* try to prevent useless calls */
	return(FALSE);
    }
    if (sg_groups[ng].desc)
	return(TRUE);		/* already have one */
    if (sg_groups[ng].flags & 4)
	return(FALSE);		/* already tried */
    sg_groups[ng].flags |= 4;	/* so we don't try again */
#ifdef USE_NNTP
    /* look for the xgtitle command */
    sprintf(tmpbuf,"xgtitle %s",sg_groups[ng].name);
    nntp_command(tmpbuf);
    (void)nntp_gets(ser_line, sizeof(ser_line));
    if (ser_line[0] != NNTP_CLASS_OK) {
	sg_desc_unavail = TRUE;	/* don't try again */
	return(FALSE);		/* failure */
    }
    /* assume that the first reply after the status is the group */
    (void)nntp_gets(ser_line, sizeof(ser_line));
    if (ser_line[0] == '.')
	return(FALSE);		/* no desc. for this group */
    /* perhaps later actually check the group name. */
    /* skip the group name */
    for (s = ser_line;(*s && !isspace(*s));s++)
	;	/* EMPTY */
    for (;(*s && isspace(*s));s++)
	;	/* EMPTY */
    if (!*s)
	return(FALSE);		/* malformed somehow */
    sg_groups[ng].desc = savestr(s);
    /* eat up the rest of the output */
    while (((i=nntp_gets(ser_line, sizeof(ser_line))) >= 0) &&
	   (ser_line[0]!='.'))
		;	/* EMPTY */
    return(TRUE);
#else	/* !USE_NNTP */
    return(FALSE);
#endif	/* USE_NNTP */
}
#endif /* SCAN */
