
module Prelude where

-- Standard value bindings {Prelude} ----------------------------------------

infixr 9  .
infixl 7  *, /, `quot`, `rem`, `div`, `mod`, :%
infixl 6  +, -
infixr 5  :, ++
infix  4  ==, /=, <, <=, >=, >
infixr 3  &&
infixr 2  ||
infixl 1  >>, >>=
infixr 0  `seq`

----------------------------------------------------------------
-- Functions
----------------------------------------------------------------

readParen = undefined
readField = undefined
showParen = undefined
showField = undefined
lex = undefined

fst            :: (a,b) -> a
fst (x,_)       = x

snd            :: (a,b) -> b
snd (_,y)       = y

id             :: a -> a
id    x         = x

(.)            :: (b -> c) -> (a -> b) -> (a -> c)
(f . g) x       = f (g x)

flip           :: (a -> b -> c) -> b -> a -> c
flip f x y      = f y x

primitive error  :: String -> a

undefined        :: a
undefined | False = undefined

(&&), (||)  :: Bool -> Bool -> Bool
False && x   = False
True  && x   = x
False || x   = x
True  || x   = True

not         :: Bool -> Bool
not True     = False
not False    = True

otherwise   :: Bool
otherwise    = True

hugsPutStr = undefined
hugsIORun = undefined
primExitWith = undefined 
primCompAux = undefined
primPmInt = undefined
primPmInteger = undefined
primPmFlt = undefined
primPmNpk = undefined
primPmSub = undefined

----------------------------------------------------------------
-- Types
----------------------------------------------------------------

type ReadS a = String -> [(a,String)]
type ShowS   = String -> String
data Void
-- data () = () deriving (Eq, Ord, Ix, Enum, Read, Show, Bounded)
data Bool    = False | True
data Char               -- builtin datatype of ISO Latin characters
type String = [Char]    -- strings are lists of characters
data Maybe a = Nothing | Just a
data Either a b = Left a | Right b
data Ordering = LT | EQ | GT
-- data [a] = [] | a : [a] deriving (Eq, Ord)
-- data (a,b) = (a,b) deriving (Eq, Ord, Ix, Read, Show)
-- etc..
data Int      -- builtin datatype of fixed size integers
data Integer  -- builtin datatype of arbitrary size integers
data Float     -- builtin datatype of single precision floating point numbers
data Double    -- builtin datatype of double precision floating point numbers
data Integral a => Ratio a = a :% a deriving (Eq)
type Rational              = Ratio Integer
--data IO a             -- builtin datatype of IO actions
data IOError            -- builtin datatype of IO error codes
type FilePath = String  -- file pathnames are represented by strings

newtype IO a = IO ((IOError -> IOResult a) -> (a -> IOResult a) -> IOResult a)
data IOResult a 
  = Hugs_ExitWith Int
  | Hugs_SuspendThread
  | Hugs_Error    IOError
  | Hugs_Return   a

----------------------------------------------------------------
-- Classes
----------------------------------------------------------------

class Eq a where
    (==), (/=) :: a -> a -> Bool
    x /= y      = not (x==y)

class (Eq a) => Ord a where
    compare                :: a -> a -> Ordering
    (<), (<=), (>=), (>)   :: a -> a -> Bool
    max, min               :: a -> a -> a

class Bounded a where
    minBound, maxBound :: a

class (Eq a, Show a, Eval a) => Num a where
    (+), (-), (*)  :: a -> a -> a
    negate         :: a -> a
    abs, signum    :: a -> a
    fromInteger    :: Integer -> a
    fromInt        :: Int -> a

class (Num a, Ord a) => Real a where
    toRational     :: a -> Rational

class (Real a, Enum a) => Integral a where
    quot, rem, div, mod :: a -> a -> a
    quotRem, divMod     :: a -> a -> (a,a)
    even, odd           :: a -> Bool
    toInteger           :: a -> Integer
    toInt               :: a -> Int

class (Num a) => Fractional a where
    (/)          :: a -> a -> a
    recip        :: a -> a
    fromRational :: Rational -> a
    fromDouble   :: Double -> a

class (Fractional a) => Floating a where
    pi                  :: a
    exp, log, sqrt      :: a -> a
    (**), logBase       :: a -> a -> a
    sin, cos, tan       :: a -> a
    asin, acos, atan    :: a -> a
    sinh, cosh, tanh    :: a -> a
    asinh, acosh, atanh :: a -> a

class (Real a, Fractional a) => RealFrac a where
    properFraction   :: (Integral b) => a -> (b,a)
    truncate, round  :: (Integral b) => a -> b
    ceiling, floor   :: (Integral b) => a -> b

class (RealFrac a, Floating a) => RealFloat a where
    floatRadix       :: a -> Integer
    floatDigits      :: a -> Int
    floatRange       :: a -> (Int,Int)
    decodeFloat      :: a -> (Integer,Int)
    encodeFloat      :: Integer -> Int -> a
    exponent         :: a -> Int
    significand      :: a -> a
    scaleFloat       :: Int -> a -> a
    isNaN, isInfinite, isDenormalized, isNegativeZero, isIEEE
		     :: a -> Bool

class (Ord a) => Ix a where
    range                :: (a,a) -> [a]
    index                :: (a,a) -> a -> Int
    inRange              :: (a,a) -> a -> Bool
    rangeSize            :: (a,a) -> Int

class Enum a where
    toEnum               :: Int -> a
    fromEnum             :: a -> Int
    enumFrom             :: a -> [a]              -- [n..]
    enumFromThen         :: a -> a -> [a]         -- [n,m..]
    enumFromTo           :: a -> a -> [a]         -- [n..m]
    enumFromThenTo       :: a -> a -> a -> [a]    -- [n,n'..m]

class Read a where
    readsPrec :: Int -> ReadS a
    readList  :: ReadS [a]

class Show a where
    showsPrec :: Int -> a -> ShowS
    showList  :: [a] -> ShowS

class Functor f where
    map :: (a -> b) -> (f a -> f b)

class Monad m where
    (>>=)  :: m a -> (a -> m b) -> m b
    return :: a -> m a
    (>>)   :: m a -> m b -> m b

class Monad m => MonadZero m where
    zero   :: m a

class MonadZero m => MonadPlus m where
    (++)   :: m a -> m a -> m a

class Eval a where
    strict :: (a -> b) -> a -> b
    seq    :: a -> b -> b

----------------------------------------------------------------
-- Instances
----------------------------------------------------------------

instance Eq ()
instance Ord ()
instance Ix ()
instance Enum ()
instance Read ()
instance Show ()
instance Bounded ()

instance Eq Char  
instance Ord Char 
instance Enum Char
instance Ix Char
instance Read Char	
instance Show Char
instance Bounded Char

instance Functor Maybe 
instance Monad Maybe 
instance MonadZero Maybe
instance MonadPlus Maybe

instance Eq a => Eq [a]
instance Ord a => Ord [a]
instance Functor [] 
instance Monad [ ]
instance MonadZero [ ]
instance MonadPlus [ ] 
instance Read a => Read [a]
instance Show a => Show [a] 

instance Show (a -> b) 
instance Functor ((->) a)

instance Eq  Int     
instance Eq  Integer 
instance Ord Int     
instance Ord Integer 

instance Num Int
instance Bounded Int
instance Num Integer
instance Real Int
instance Real Integer
instance Integral Int
instance Integral Integer
instance Ix Int
instance Ix Integer
instance Enum Int
instance Enum Integer
instance Read Int
instance Show Int
instance Read Integer
instance Show Integer

instance Eq  Float  
instance Eq  Double 
instance Ord Float  
instance Ord Double 
instance Num Float
instance Num Double
instance Real Float
instance Real Double
instance Fractional Float
instance Fractional Double
instance Floating Float
instance Floating Double
instance RealFrac Float
instance RealFrac Double
instance RealFloat Float
instance RealFloat Double
instance Enum Float
instance Enum Double
instance Read Float
instance Show Float
instance Read Double
instance Show Double

instance Integral a => Ord (Ratio a)
instance Integral a => Num (Ratio a)
instance Integral a => Real (Ratio a)
instance Integral a => Fractional (Ratio a)
instance Integral a => RealFrac (Ratio a)
instance Integral a => Enum (Ratio a)
instance (Read a, Integral a) => Read (Ratio a)
instance Integral a => Show (Ratio a)

instance Show (IO a)
instance Functor IO 
instance Monad IO

----------------------------------------------------------------
-- End
----------------------------------------------------------------