\newpage
\section{The Basics}
\label{basics}

\begin{verbatim} 

> module Basics where
> infixr 5 :+:, :=:

\end{verbatim} 
Perhaps the most basic musical idea is that of a {\em pitch}, which
consists of a {\em pitch class} (i.e. one of 12 semi-tones) and an
{\em octave}:
\begin{verbatim} 

> type Pitch      = (PitchClass, Octave)
> data PitchClass = Cf | C | Cs | Df | D | Ds | Ef | E | Es | Ff | F | Fs
>                 | Gf | G | Gs | Af | A | As | Bf | B | Bs
>      deriving (Eq,Ord,Ix,Show)
> type Octave     = Int

\end{verbatim} 
So a {\tt Pitch} is a pair consisting of a pitch class and an octave.
Octaves are just integers, but we define a datatype for pitch classes,
since distinguishing enharmonics (such as G\# and Ab) may be important
(especially for notation!).  By convention, A440 = {\tt (A,4)}.

Musical objects are captured by the {\tt Music} datatype:\footnote{I
prefer to call these ``musical objects'' rather than ``musical
values'' because the latter may be confused with musical aesthetics.}
\begin{verbatim} 

> data Music = Note Pitch Dur [NoteAttribute]   -- a note \ atomic 
>            | Rest Dur                         -- a rest /    objects
>            | Music :+: Music                  -- sequential composition
>            | Music :=: Music                  -- parallel composition
>            | Tempo  Int Int Music             -- scale the tempo
>            | Trans  Int Music                 -- transposition
>            | Instr  IName Music               -- instrument label
>            | Player PName Music               -- player label
>            | Phrase [PhraseAttribute] Music   -- phrase attributes
>     deriving Show
>
> type Dur   = Float                            -- in whole notes
> type IName = String
> type PName = String

\end{verbatim} 
Here a {\tt Note} is its pitch paired with its duration (in number of
whole notes), along with a list of {\tt NoteAttributes} (defined
later).  A {\tt Rest} also has a duration, but of course no pitch or
other attributes.

\newpage

From these two atomic constructors we can build more complex musical
objects using the other constructors, as follows:
\begin{itemize}
\item {\tt m1 :+: m2} is the sequential composition of {\tt m1} and
{\tt m2}; i.e.\ {\tt m1} and {\tt m2} are played in sequence.
\item {\tt m1 :=: m2} is the parallel   composition of {\tt m1} and
{\tt m2}; i.e.\ {\tt m1} and {\tt m2} are played simultaneously.
\item {\tt Tempo a b m} scales the rate at which
{\tt m} is played (i.e.\ its tempo) by a factor of {\tt a/b}.
\item {\tt Trans i m} transposes {\tt m} by interval {\tt i} (in semitones).
\item {\tt Instr iname m} declares that {\tt m} is to be performed using
instrument {\tt iname}.
\item {\tt Player pname m} declares that {\tt m} is to be performed by
player {\tt pname}.
\item {\tt Phrase pas m} declares that {\tt m} is to be played using
the phrase attributes (described later) in the list {\tt pas}.
\end{itemize}

It is convenient to represent these ideas in Haskell as a recursive
datatype because we wish to not only construct musical objects, but
also take them apart, analyze their structure, print them in a
structure-preserving way, interpret them for performance purposes,
etc.

\subsection{Convenient Auxiliary Functions}
\label{auxiliaries}

For convenience we first create a few names for familiar notes,
durations, and rests, as shown in Figure \ref{note-names}.  Treating
pitches as integers is also useful in many settings, so we define some
functions for converting between {\tt Pitch} values and {\tt AbsPitch}
values (integers).  These also are shown in Figure \ref{note-names},
along with a definition of {\tt trans}, which transposes pitches
(analogous to {\tt Trans}, which transposes values of type {\tt
Music}).

\begin{exercise} Show that\ \ {\tt abspitch\ .\ pitch = id}, and,
up to enharmonic equivalences,\newline {\tt pitch\ .\ abspitch = id}.
\end{exercise}

\begin{exercise} Show that\ \ {\tt trans i (trans j p) = trans (i+j) p}.
\end{exercise}

\begin{figure}{\small
\begin{verbatim} 

> cf,c,cs,df,d,ds,ef,e,es,ff,f,fs,gf,g,gs,af,a,as,bf,b,bs :: 
>    Octave -> Dur -> [NoteAttribute] -> Music
>
> cf o = Note (Cf,o);  c o = Note (C,o);  cs o = Note (Cs,o)
> df o = Note (Df,o);  d o = Note (D,o);  ds o = Note (Ds,o)
> ef o = Note (Ef,o);  e o = Note (E,o);  es o = Note (Es,o)
> ff o = Note (Ff,o);  f o = Note (F,o);  fs o = Note (Fs,o)
> gf o = Note (Gf,o);  g o = Note (G,o);  gs o = Note (Gs,o)
> af o = Note (Af,o);  a o = Note (A,o);  as o = Note (As,o)
> bf o = Note (Bf,o);  b o = Note (B,o);  bs o = Note (Bs,o)
>
> wn,  hn,  qn,  en,  sn,  tn  :: Dur
> wnr, hnr, qnr, enr, snr, tnr :: Music
>
> wn = 1          ; wnr = Rest wn      -- whole note rest
> hn = 1/2        ; hnr = Rest hn      -- half note rest
> qn = 1/4        ; qnr = Rest qn      -- quarter note rest
> en = 1/8        ; enr = Rest en      -- eight note rest
> sn = 1/16       ; snr = Rest sn      -- sixteenth note rest
> tn = 1/32       ; tnr = Rest tn      -- thirty-second note rest
>
> pitchClass :: PitchClass -> Int
>
> pitchClass pc = case pc of
>      Cf -> -1;  C -> 0;  Cs -> 1    -- or should Cf be 11?
>      Df -> 1;   D -> 2;  Ds -> 3
>      Ef -> 3;   E -> 4;  Es -> 5
>      Ff -> 4;   F -> 5;  Fs -> 6
>      Gf -> 6;   G -> 7;  Gs -> 8
>      Af -> 8;   A -> 9;  As -> 10
>      Bf -> 10;  B -> 11; Bs -> 12   -- or should Bs be 0?
>
> type AbsPitch = Int
>
> absPitch :: Pitch -> AbsPitch
> absPitch (pc,oct) = 12*oct + pitchClass pc
>
> pitch    :: AbsPitch -> Pitch
> pitch    ap       = ( [C,Cs,D,Ds,E,F,Fs,G,Gs,A,As,B] !! mod ap 12, 
>                       quot ap 12)
>
> trans    :: Int -> Pitch -> Pitch
> trans i p = pitch (absPitch p + i)

\end{verbatim}}
\caption{Convenient note names and pitch conversion functions.}
\label{note-names}
\end{figure}

\subsection{Some Simple Examples}
\label{basic-examples}

With this modest beginning, we can already express quite a few musical
relationships simply and effectively.  For example, two common ideas
in music are the construction of notes in a horizontal fashion (a {\em
line} or {\em melody}), and in a vertical fashion (a {\em chord}):
\begin{verbatim}

> line, chord :: [Music] -> Music
> line  = foldr (:+:) (Rest 0) 
> chord = foldr (:=:) (Rest 0) 

\end{verbatim}

From the notes in the C major triad in register 4, I can now construct
a C major arpeggio and chord as well:
\begin{verbatim} 

> cMaj = map (\f->f 4 qn []) [c, e, g]  -- octave 4, quarter notes
>
> cMajArp = line  cMaj
> cMajChd = chord cMaj

\end{verbatim} 

Suppose now we wish to describe a melody {\tt m} accompanied by an
identical voice a perfect 5th higher.  In Haskore we simply write
``{\tt m :=:\ Trans 7 m}.''  Similarly, a canon-like structure
involving {\tt m} can be expressed as ``{\tt m :=:\ delay d m},''
where:
\begin{verbatim} 

> delay :: Dur -> Music -> Music
> delay d m = Rest d :+: m

\end{verbatim} 

Of course, Haskell's non-strict semantics also allows us to define
infinite musical objects.  For example, a musical object may be
repeated {\em ad nauseum} using this simple function:
\begin{verbatim} 

> repeatM :: Music -> Music
> repeatM m = m :+: repeatM m

\end{verbatim} 
Thus an infinite ostinato can be expressed in this way, and then used
in different contexts that extract only the portion that's actually
needed.

% This, however, raises an interesting problem: in the expression {\tt
% m1 :=: m2}, if either {\tt m1} or {\tt m2} is infinite, so will the
% result be infinite.  Sometimes we would rather have the result be of
% duration equal to the shorter of the two.  This is not as easy as it
% sounds, since it may require interrupting the longer one in the middle
% of a note (or notes).  In any case it is quite useful; I will provide
% its typing here, with its full definition relegated to Appendix
% \ref{misc-code}.
% \begin{verbatim} 
% 
% > (===) :: Music -> Music -> Music
% 
% \end{verbatim}
% 
% Note that we can now truncate the duration of any musical object, and
% it is convenient to define a function to do so:
% \begin{verbatim} 
% 
% > truncateM :: Dur -> Music -> Music
% > truncateM d m = Rest d === m
% 
% \end{verbatim} 

The notions of inversion, retrograde, retrograde inversion, etc. used
in 12-tone theory are also easily captured in Haskore.  First let's
define a transformation from a line created by {\tt line} to a list:
\begin{verbatim} 

> lineToList :: Music -> [Music]
> lineToList n@(Rest 0) = []
> lineToList (n :+: ns) = n : lineToList ns
>
> retro, invert, retroInvert, invertRetro :: Music -> Music
> retro    = line . reverse . lineToList
> invert m = line (map inv l)
>   where l@(Note r _ _: _)  = lineToList m
>         inv (Note p d nas) = Note (pitch (2*(absPitch r) - absPitch p)) d nas
>         inv (Rest d)       = Rest d
> retroInvert = retro  . invert
> invertRetro = invert . retro

\end{verbatim} 

\begin{exercise} Show that ``{\tt retro\ .\ retro},''
``{\tt invert\ .\ invert},'' and ``{\tt retroInvert\ .\ invertRetro}''
are the identity on values created by {\tt line}.
\end{exercise}

\begin{figure*}
\centerline{
\epsfysize=2.0in 
\epsfbox{Pics/poly.eps}
}
\caption{Nested Polyrhythms}
\label{polyrhythms}
\end{figure*}

For some rhythmical ideas, consider first a simple {\em triplet} of
eighth notes; it can be expressed as ``{\tt Tempo 3 2 m},'' where {\tt
m} is a line of 3 eighth notes.  So in fact {\tt Tempo} can be used to
create quite complex rhythmical patterns.  For example, consider the
``nested polyrhythms'' shown in Figure \ref{polyrhythms}.  They can be
expressed quite naturally in Haskore as follows (note the use of the
{\em where} clause in {\tt pr2} to capture recurring phrases):
\begin{verbatim} 

> pr1, pr2 :: Pitch -> Music
> pr1 p = Tempo 5 6 (Tempo 4 3 (mkLn 1 p qn :+:
>                               Tempo 3 2 (mkLn 3 p en :+:
>                                          mkLn 2 p sn :+:
>                                          mkLn 1 p qn    ) :+:
>                               mkLn 1 p qn) :+:
>                    Tempo 3 2 (mkLn 6 p en))
>
> pr2 p = Tempo 7 6 (m1 :+:
>                    Tempo 5 4 (mkLn 5 p en) :+:
>                    m1 :+:
>                    mkLn 2 p en)
>         where m1 = Tempo 5 4 (Tempo 3 2 m2 :+: m2)
>               m2 = mkLn 3 p en
>
> mkLn n p d = line (take n (repeat (Note p d [])))

\end{verbatim}
To play polyrhythms {\tt pr1} and {\tt pr2} in parallel using middle C
and middle G, respectively, we would do the following (middle C is in
the 5th octave):
\begin{verbatim} 

> pr12 :: Music
> pr12 = pr1 (C,5) :=: pr2 (G,5)

\end{verbatim} 

As a final example in this section, we can can compute the duration in
beats of a musical object, a notion we will need in Section
\ref{performance}, as follows:
\begin{verbatim}

> dur :: Music -> Dur
>
> dur (Note _ d _)  = d
> dur (Rest d)      = d
> dur (m1 :+: m2)   = dur m1   +   dur m2
> dur (m1 :=: m2)   = dur m1 `max` dur m2
> dur (Tempo a b m) = dur m * float b / float a
> dur (Trans  _  m) = dur m
> dur (Instr  _  m) = dur m
> dur (Player _  m) = dur m
> dur (Phrase _  m) = dur m
>
> float = fromInteger . toInteger :: Int -> Float

\end{verbatim} 
Using {\tt dur} we can define a function {\tt revM} that reverses any
{\tt Music} value (and is thus considerably more useful than {\tt
retro} defined earlier).  Note the tricky treatment of {\tt (:=:)}.
\begin{verbatim}

> revM :: Music -> Music
> revM n@(Note _ _ _)  = n
> revM r@(Rest _)      = r
> revM (Tempo i1 i2 m) = Tempo i1 i2 (revM m)
> revM (Trans i  m)    = Trans i     (revM m)
> revM (Instr i  m)    = Instr i     (revM m)
> revM (Phrase pas m)  = Phrase pas  (revM m)
> revM (m1 :+: m2)     = revM m2 :+: revM m1
> revM (m1 :=: m2)     = let d1 = dur m1
>                            d2 = dur m2
>                        in if d1>d2 then revM m1 :=:
>                                         (Rest (d1-d2) :+: revM m2)
>                                    else (Rest (d2-d1) :+: revM m1) :=:
>                                         revM m2

\end{verbatim} 

\newpage 

\begin{exercise}
Find a simple piece of music written by your favorite composer, and
transcribe it into Haskore.  In doing so, look for repeating patterns,
transposed phrases, etc. and reflect this in your code, thus revealing
deeper structural aspects of the music than that found in common
practice notation.
\end{exercise}

Appendix \ref{chick} shows the first 28 bars of Chick Corea's
``Children's Song No.~6'' encoded in Haskore.

\subsection{Phrasing and Articulation}
\label{phrasing}

Recall that the {\tt Note} constructor contained a field of {\tt
NoteAttributes}.  These are values that are attached to notes for the
purpose of notation or musical interpretation.  Likewise, the {\tt
Phrase} constructor permits one to annotate an entire musical object
with {\tt PhraseAttributes}.  These two attribute datatypes cover a
wide range of attributions found in common practice notation, and are
shown in Figure \ref{attributes}.  Beware that use of them requires
the use of a player that knows how to interpret them!  Players will be
described in more detail in Section \ref{players}.

\begin{figure}
\begin{verbatim} 

> data NoteAttribute = Volume Float        -- by convention: 0=min, 100=max
>                    | Fingering Int
>                    | Dynamics String
>      deriving Show
>
> data PhraseAttribute = Dyn Dynamic
>                      | Art Articulation
>                      | Orn Ornament
>      deriving Show
>
> data Dynamic = Accent Float | Crescendo Float | Diminuendo Float
>              | PPP | PP | P | MP | SF | MF | NF | FF | FFF | Loudness Float
>              | Ritardando Float | Accelerando Float
>      deriving Show
>
> data Articulation = Staccato Float | Legato Float | Slurred Float
>                   | Tenuto | Marcato | Pedal | Fermata | FermataDown | Breath
>                   | DownBow | UpBow | Harmonic | Pizzicato | LeftPizz 
>                   | BartokPizz | Swell | Wedge | Thumb | Stopped
>      deriving Show
>
> data Ornament = Trill | Mordent | InvMordent | DoubleMordent
>               | Turn | TrilledTurn | ShortTrill
>               | Arpeggio | ArpeggioUp | ArpeggioDown
>               | Instruction String | Head NoteHead
>      deriving Show
>
> data NoteHead = DiamondHead | SquareHead | XHead | TriangleHead
>               | TremoloHead | SlashHead | ArtHarmonic | NoHead
>      deriving Show

\end{verbatim}
\caption{Note and Phrase Attributes.}
\label{attributes}
\end{figure}

Note that some of the attributes are parameterized with a numeric
value.  This is used by a player to control the degree to which
an articulation is to be applied.  For example, we would expect {\tt
Legato 1.2} to create more of a legato feel than {\tt Legato 1.1}.
The following constants represent default values for some of the
parameterized attributes: 
\begin{verbatim}

> legato, staccato  :: Articulation
> accent, bigAccent :: Dynamic
>
> legato    = Legato 1.1
> staccato  = Staccato 0.5
> accent    = Accent 1.2
> bigAccent = Accent 1.5

\end{verbatim} 

To understand exactly how a player interprets an attribute requires
knowing how players are defined.  Haskore defines only a few simple
players, so in fact many of the attributes in Figure \ref{attributes}
are to allow the user to give appropriate interpretations of them by
her particular player.  But before looking at the structure of players
we will need to look at the notion of a {\em performance} (these two
ideas are tightly linked, which is why the {\tt Players} and {\tt
Performance} modules are mutually recursive).
