module Board where                         

import List
import IO
import IOExts
import Array


-- Support for mutable arrays

type MArray a b = Array a (IORef b)

newArr :: Ix a => (a,a) -> b -> IO (MArray a b)
newArr bds x =
  do assocs <- mapM makeAssoc (range bds)
     return (array bds assocs)
  where makeAssoc i = do r <- newIORef x
                         return (i, r)

setArr :: Ix a => (MArray a b) -> a -> b -> IO ()
setArr arr i x = writeIORef (arr ! i) x

getArr :: Ix a => (MArray a b) -> a -> IO b
getArr arr i = readIORef (arr ! i)

dumpArr :: (Ix a, Show b) => (MArray a b) -> IO ()
dumpArr arr = do sequence (map fnc (elems arr)) 
                 putStr "\n"
              where
                 fnc e = do v <- readIORef e
                            putStr (show v)


readsRef r = reads r
instance (Read a) => Read (IORef a) where
  readsPrec n = readsRef

readsArray a = reads a
instance (Read a, Read b) => Read (Array a b) where
  readsPrec n = readsArray

showsRef r = shows ""
instance (Show a) => Show (IORef a) where
  showsPrec n = showsRef

data UserAction = UA_Up | UA_Down | UA_Left | UA_Right | 
                  UA_Exit | UA_Restart | UA_Hint | UA_Solve | UA_Nothing
  deriving (Read, Show, Eq)

type Dim   = Int
type Dims  = (Dim, Dim)   -- (width,height)
type Coord = (Int, Int)	  -- (x, y)

type PieceBit    = Int
type PieceBitmap = [PieceBit]

type VertexList  = [Coord]

data PieceType = Main | Target | Regular | Static | Opening | Empty
  deriving (Read, Show, Eq, Ord, Enum, Ix, Bounded)

pieceTypeToChar :: PieceType -> Char
pieceTypeToChar typ = case typ of
  Empty   -> ' '
  Main    -> 'X'
  Regular -> '*'
  Static  -> ''
  Opening -> ''
  Target  -> 'o'

pieceTypeMovable :: PieceType -> Bool
pieceTypeMovable typ = typ == Main || typ == Regular

pieceCanMoveTo :: PieceType -> Bool
pieceCanMoveTo typ = typ == Empty || typ == Target

data PieceShape = PieceShape Dims PieceBitmap VertexList
  deriving (Read, Show, Eq)

data Piece = Piece PieceType Coord PieceShape
  deriving (Read, Show, Eq)

instance Ord (Piece) where
  Piece t0 _ _ <= Piece t1 _ _ = t0 <= t1

emptyPiece :: Piece
emptyPiece = Piece Empty (0,0) (PieceShape (0,0) [] [])

type PieceIndex  = Int
type BoardBitmap = MArray Int PieceIndex

data ReadBoard = ReadBoard Dims [Piece]
  deriving (Read, Show)

data Board = Board Dims [Piece] BoardBitmap
  deriving (Read, Show)

type MoveInfo = (Int, UserAction)


indexToPiece :: Board -> PieceIndex -> Piece
indexToPiece (Board d l bm) pi =
  if pi >= 0
    then l !! pi
    else emptyPiece


indexToPieceType :: Board -> PieceIndex -> PieceType
indexToPieceType b pi =
  let (Piece typ coord shape) = indexToPiece b pi
  in  typ


dumpBoard :: Board -> IO ()
dumpBoard b@(Board (w,h) l bm) = 
  do
    sequence (map fnc (assocs bm)) 
    putStr "\n"
  where
    fnc (i,e) = do if i `mod` w == 0 then putStr "\n" else putStr ""
                   v <- readIORef e
                   putChar (pieceTypeToChar (indexToPieceType b v))


-- Utility functions

pieceAtCoord :: Board -> Coord -> IO PieceIndex
pieceAtCoord b@(Board (w,h) l bm) (x,y) = 
  if x >= 0 && x < w && y >= 0 && y < h 
    then getArr bm (x + y*w)
    else return (-1)


toAbsIndex :: Board -> Coord -> Int
toAbsIndex b@(Board (bw,bh) l bm) (x,y) = x + y*bw


isIndexInPiece :: Board -> Piece -> Int -> Bool
isIndexInPiece b@(Board (bw,bh) l bm) p@(Piece typ (px,py) (PieceShape (pw,ph) bits vs)) index = 
  index `elem` (map ((toAbsIndex b).(mapBitmapToCoord (px,py) (pw,ph))) bits)


mapBitmapToCoord :: Coord -> Dims -> PieceBit -> Coord
mapBitmapToCoord (x,y) (w,h) bit = (x + (bit `mod` w), y + (bit `div` w))


-- apply the function over list elements while the function returns True
applyWhileTrue :: (Int -> a -> IO Bool) -> Int -> [a] -> IO Bool
applyWhileTrue fnc n []     = return True
applyWhileTrue fnc n (x:xs) = do 
  ok <- fnc n x
  -- putStrLn ("applyWhileTrue executed fnc, n: " ++ (show n) ++ " result " ++ (show ok))
  if ok then applyWhileTrue fnc (n+1) xs
        else return False


-- apply the function over list elements while the function returns False
applyWhileFalse :: (a -> Bool) -> [a] -> Bool
applyWhileFalse fnc []   = False
applyWhileFalse fnc (x:xs) = 
  if (fnc x) == False then applyWhileFalse fnc xs
                      else True


-- Playing around with bitmaps

changeBit :: Board -> Coord -> Dims -> PieceIndex -> Int -> PieceBit -> IO Bool
changeBit b@(Board (w,h) l bm) coord dims pi bi bit =
  let (x,y) = mapBitmapToCoord coord dims bit
      i     = x + y*w
  in  do flg <- getArr bm i
         -- putStrLn ("Changing bit " ++ (show i) ++ " containing " ++ 
         --          (show flg) ++ ", " ++ (show (pieceCanMoveTo (indexToPieceType b flg))) ++ 
         --          " to " ++ (show pi))
         if pi == -1
           then do setArr bm i (if (isIndexInPiece b (l !! (fromEnum Target)) i) then 1 else (-1))
                   return True
           else if pieceCanMoveTo (indexToPieceType b flg)
                  then do setArr bm i pi
                          return True
           else return False


-- expand the board to make sure we have the border around
addBorder :: ReadBoard -> IO ReadBoard
addBorder (ReadBoard (w,h) l) =
  let vertexList = [(0,0),(w+2,0),(w+2,h+2),(0,h+2),(0,1),(1,1),(1,h+1),(w+1,h+1),(w+1,1),(0,1)]
      borderBitmap = sort ([1..w] ++
                           [0,w+2..(w+2)*(h+1)] ++
                           [w+1,2*w+3..(w+2)*(h+2)-1] ++
                           [(w+2)*(h+1)+1..(w+2)*(h+2)-2]
                          )
      shiftPiece (Piece t (x,y) s) = 
        if x >= 0 && y >= 0 && x < w && y < h
          then Piece t (x+1,y+1) s
          else Piece t (x  ,y  ) s
      borderPiece = Piece Static (0,0) (PieceShape (w+2, h+2) borderBitmap vertexList)
      pieceList   = (map shiftPiece l) ++ [borderPiece]
  in  return (ReadBoard (w+2,h+2) (sort pieceList))


addPieceInitial :: Board -> PieceIndex -> Piece -> IO Bool
addPieceInitial b@(Board (w,h) l bm) n p@(Piece typ (x,y) (PieceShape (pw,ph) bits vs)) =
  if x >= 0 && y >= 0 && x < w && y < h
    then 
      if pw > 0 && ph > 0 && pw <= w && pw <= h
        then do
             -- putStrLn ("Adding piece " ++ (show n)  ++ ", " ++ (show typ) ++ " at (" ++ (show x) ++ "," ++ (show y) ++ ")")
             ok <- applyWhileTrue (changeBit b (x,y) (pw,ph) n) 0 bits
             if ok then return True
                   else do putStrLn ("Could not add piece at coordinates (" ++ (show (x-1)) ++ "," ++ (show (y-1)) ++ ")")
                           return False
        else do putStrLn ("Incorrect piece dimensions (" ++ (show pw) ++ "," ++ (show ph) ++ ")")
                return False
    else do putStrLn ("Incorrect piece coordinates (" ++ (show (x-1)) ++ "," ++ (show (y-1)) ++ ")")
            return False


-- Make sure that the Board is valid
verifyBoard :: ReadBoard -> IO (Maybe Board)
verifyBoard b@(ReadBoard (w,h) l) = do
  if w <= 3 || h < 3 
    then do 
        putStrLn "Board too small"
        return Nothing
    else do
        putStrLn "Expanding board ..."
        (ReadBoard (w,h) l) <- addBorder b
        putStrLn "Creating bitmap ..."
        bm1 <- newArr (0, w*h-1) (-1)
        putStrLn "Setting the bitmap ..."
        ok  <- applyWhileTrue (addPieceInitial (Board (w,h) l bm1)) 0 l
        putStrLn "Initialization done "
        if ok then
          let m@(Piece mt _ ms) = l !! (fromEnum Main)
              t@(Piece tt _ ts) = l !! (fromEnum Target)
              r@(Piece rt _ rs) = l !! (fromEnum Regular)
          in  if mt /= Main
                then do
                     putStrLn "No Main piece specified"
                     return Nothing
              else if tt /= Target
                then do
                   putStrLn "No Target piece specified"
                   return Nothing
              else if tt /= Target || mt /= Main || rt /= Regular
                then do
                   putStrLn "Must have exactly one Main, one Target and at least one Regular piece"
                   return Nothing
              else if (foldl (\z (Piece t _ _) -> if t == Opening then (z+1) else z) 0 l) > 1
                then do
                   putStrLn "Only one Opening piece is allowed"
                   return Nothing
              else if ts /= ms
                then do
                   putStrLn "The shapes of Main and Target are not identical"
                   return Nothing
              else return (Just (Board (w,h) l bm1))
          else return Nothing


-- Read board from a file
readBoard :: String -> IO (Maybe Board)
readBoard filename = do
  bm <- newArr (0, 0) (-1)
  catch (do handle <- openFile filename ReadMode
            str <- hGetContents handle
            if str /= "" then let b0 :: ReadBoard
                                  b0 = read str
                              in  do -- putStrLn str
                                     -- putStrLn (show b0)
                                     putStrLn "Reading the board ..."
                                     b <- verifyBoard b0
                                     hClose handle
                                     case b of Nothing -> do putStrLn "\nSetup Incorrect"
                                                             return Nothing
                                               otherwise -> return b
                         else do hClose handle
                                 return Nothing
        )
        (\e -> do putStrLn ("Cannot open " ++ filename)
                  return Nothing)


addBitsUndo :: Board -> PieceIndex -> Piece -> [Int] -> IO (Bool, [(Int, PieceIndex)])
addBitsUndo b@(Board (w,h) l bm) n (Piece typ (x,y) (PieceShape (pw,ph) bits vs)) [] =
  return (True, [])
addBitsUndo b@(Board (w,h) l bm) n p@(Piece typ (x,y) (PieceShape (pw,ph) bits vs)) (f:fs) =
  let (x1,y1) = mapBitmapToCoord (x,y) (pw,ph) f
      i     = x1 + y1*w
  in  do flg <- getArr bm i
         if (n == -1) || (pieceCanMoveTo (indexToPieceType b flg))
           then do setArr bm i n
                   (f1, u1) <- addBitsUndo b n p fs
                   return (f1, (i, flg):u1)
           else return (False, [])


addPieceUndo :: Board -> PieceIndex -> Piece -> IO (Bool, [(Int, PieceIndex)])
addPieceUndo b@(Board (w,h) l bm) n p@(Piece typ (x,y) (PieceShape (pw,ph) bits vs)) = do
  if x >= 0 && y >= 0 && x < w && y < h
    then addBitsUndo b n p bits
    else return (False, [])


removePiece :: Board -> PieceIndex -> Piece -> IO Bool
removePiece b@(Board (w,h) l bm) n (Piece typ (x,y) (PieceShape (pw,ph) bits vs)) =
  applyWhileTrue (changeBit b (x,y) (pw,ph) (-1)) 0 bits



-- return True if Coord1 has a common edge with Coord2
coordNextToCoord :: Coord -> Coord -> Bool
coordNextToCoord (x1,y1) (x2,y2) =
  (x1,y1) == (x2  , y2-1) ||
  (x1,y1) == (x2  , y2+1) ||
  (x1,y1) == (x2-1, y2  ) ||
  (x1,y1) == (x2+1, y2  )


-- return True if a bit in Piece1 has a common edge with Piece2
pieceNextToPieceBit :: Piece -> Piece -> PieceBit -> Bool
pieceNextToPieceBit (Piece t1 (x1,y1) (PieceShape (w1,h1) b1 v1))
                    (Piece t2 (x2,y2) (PieceShape (w2,h2) b2 v2)) bit =
  let c2 = mapBitmapToCoord (x2,y2) (w2,h2) bit
  in  applyWhileFalse (coordNextToCoord c2) (map (mapBitmapToCoord (x1,y1) (w1,h1)) b1)


-- return True if two piece are sharing at least one edge
pieceNextToPiece :: Piece -> Piece -> Bool
pieceNextToPiece p1@(Piece t1 (x1,y1) (PieceShape (w1,h1) b1 v1))
                 p2@(Piece t2 (x2,y2) (PieceShape (w2,h2) b2 v2)) =
  -- for each bit in p1 there must a common edge with p2
   applyWhileFalse (pieceNextToPieceBit p1 p2) b2


undoChanges :: Board -> [(Int, PieceIndex)]-> IO ()
undoChanges b@(Board (w,h) l bm) undo = sequence (map (\(i,pi) -> setArr bm i pi) undo)


canPieceMove :: Board -> PieceIndex -> (Int,Int) -> IO Bool
canPieceMove b@(Board (w,h) l bm) pi (dx,dy) = return False


dropLast :: [a] -> [a]
dropLast []     = []
dropLast (x:[]) = []
dropLast (x:xs) = x:dropLast xs


checkEliminateOpening :: Board -> PieceIndex -> IO (Bool, Board)
checkEliminateOpening b@(Board (w,h) l bm) pi =
  let opening@(Piece mt mc (PieceShape md mb mv)) = l!!pi
      main    = l!! (fromEnum Main)
  in  if (pieceNextToPiece main opening)
        then do
          removePiece b (-1) opening
          -- putStrLn "Main piece next to the opening"
          canMove <- return True
          {- 
          canMove <- (canPieceMove b pi (0, -1))
                     (canPieceMove b pi (0,  1))
                     (canPieceMove b pi (-1, 0))
                     (canPieceMove b pi ( 1, 0))
          -}
          if canMove
            then return (True, (Board (w,h) (dropLast l) bm))
            else do applyWhileTrue (changeBit b mc md pi) 0 mb
                    return (False, b)
        else do applyWhileTrue (changeBit b mc md pi) 0 mb
                return (False, b)


-- Low-level function to play a move
makeMove :: Board -> PieceIndex -> (Int,Int) -> IO (Bool, Board)
makeMove b@(Board (w,h) l bm) pi (dx,dy) =
  let pc@(Piece typ (x,y) (PieceShape (pw,ph) bits vs)) = indexToPiece b pi
  in  if pieceTypeMovable typ
        then do -- Let's try to move the piece and see what happens
                removePiece b (-1) pc
                -- dumpBoard b
                (ok, undo) <- addPieceUndo b pi (Piece typ (x+dx,y+dy) (PieceShape (pw,ph) bits vs))
                -- putStrLn (show undo)
                -- putStrLn ("pi: " ++ (show pi))
                if ok
                  then let l1 = (take pi l) ++ [(Piece typ (x+dx,y+dy) (PieceShape (pw,ph) bits vs))] ++ (drop (pi+1) l)
                       in -- do dumpBoard (Board (w,h) l1 bm)
                             return (True, (Board (w,h) l1 bm))
                  else do
                    -- putStrLn "Undoing Changes"
                    undoChanges b undo
                    -- dumpBoard b
                    -- putStrLn "Adding the piece back"
                    applyWhileTrue (changeBit b (x,y) (pw,ph) pi) 0 bits
                    return (False, b)
        else if typ == Opening then -- do dumpBoard (Board (w,h) l bm)
                                       checkEliminateOpening b pi
        else return (False, b)


-- High-level function to play a move
playMove :: Board -> MoveInfo -> IO (Bool, Board)
playMove b m@(p, ua) =
  do putStrLn ("Move: (" ++ (show p) ++ ", " ++ (show ua) ++ ")")
     case ua of
       UA_Up      -> makeMove b p ( 0,-1)
       UA_Down    -> makeMove b p ( 0, 1)
       UA_Left    -> makeMove b p (-1, 0)
       UA_Right   -> makeMove b p ( 1, 0)
       UA_Exit    -> return (False, b)
       UA_Restart -> return (False, b)
       UA_Hint    -> return (False, b)
       UA_Solve   -> return (False, b)
       UA_Nothing -> return (False, b)


boardDone :: Board -> Bool
boardDone b@(Board (w,h) l bm) =
  let m@(Piece mt mc ms) = l !! (fromEnum Main)
      t@(Piece tt tc ts) = l !! (fromEnum Target)
  in  mc == tc


r = do
  b <- readBoard "daisy"
  -- putStrLn (show b)
  case b of Nothing  -> putStrLn "\nError"
            (Just b) -> do dumpBoard b
                           putStrLn (show b)
                           -- playMove b (8, UA_Down)
                           -- dumpBoard b
                           putStrLn "\nOK"

{-
To do:
1. count moves
2. improve the test for the opening elimination
-}