{-******************************************************************
  MODULE R_INBETWEEN

    This module takes care of interpolation functions. Basically,
  given two Pics, inbetween will give you a movie gradually 
  converting from one Pic to the other Pic, using linear interpolation.
    Tween will take two Movies, and append them, interpolating n
  frames between the last Pic of the first Movie and the first Pic of
  the last Movie.      

******************************************************************-}

module Inbetween(inbetween) where

import Picture(Pic)
import qualified Picture as Pic(orig)
import Behaviour(Behaviour)
import qualified Behaviour as B
import Movie(Movie, mkMovie)
import Polygon(Poly)
import qualified Polygon as Poly
import Vector(Vec, partway, mid, vmin)
import List(transpose)

  -- inbetween takes an int and two Pics, and interpolates n Pics
  -- of interpolated Pics. 
inbetween :: Int -> Pic -> Pic -> Movie
inbetween n p1 p2 = mkMovie $ map Pic.polys $ inbetween' n (Pic.polysOf p1) (Pic.polysOf p2)
 where
  inbetween' n p1 p2 
    | length p1 == length p2
    = transpose $ zipWith (inbetweenp n) p1 p2
{-
    | otherwise
    = inbetween' n [(col,p1')] [(col,p2')]
   where 
    p1' = concat [ vs | (c,vs) <- p1]
    p2' = concat [ vs | (c,vs) <- p2]
    col = head   [ c | (c,vs) <- p1 ]
-}
                      
  -- inbetweenp takes a list of 2 Polygons ([[Vec]]) and returns a 
  -- sequence of interpolated Polygons. Should the Number of vertices 
  -- of one Polygon be less than those in the other, it splits it so 
  -- as to have two Polygons of the same length.
inbetweenp :: Int -> Poly -> Poly -> [Poly]
inbetweenp n p1 p2 = [ Poly.vecs c (zipWith (partway m n) ws1 ws2) | m <- [0..n-1] ]
 where
  c = Poly.colorOf p1
  vs1 = Poly.vecsOf p1
  vs2 = Poly.vecsOf p2

  l1 = length vs1
  l2 = length vs2

  -- generate two vertex lists of same length by expanding the shorter one
  (ws1,ws2) 
    | l1 < l2   = (split l2 vs1, vs2) 
    | l1 > l2   = (vs1, split l1 vs2) 
    | otherwise = (vs1, vs2)


  -- split splits up a Polygon so as to have the given #vertices.
split :: Int -> [Vec] -> [Vec]
split n vs 
  | n <= l     = take n vs  -- shouldn't happen
  | n > double = split n (split double vs)
  | otherwise  = v1:(mid v1 v2):(split (n-2) (v2:vss))
 where
  l = length vs
  double = 2*l - 1
  (v1:v2:vss) = vs

{-                           
  -- tween will interpolate n Pics transforming the last Pic of
  -- the first Movie into the first Pic of the second Movie, and
  -- then run the second Movie    
tween :: Int -> Movie -> Movie -> Movie         
tween n m1 []   = m1
tween n m1 m2  = m1 ++ (inbetween n (last m1) (head m2')) ++ (tail m2')       
            where m2' = B.apply (B.mov (repeat v)) m2 
                  v = vmin (Pic.orig (last m1)) (Pic.orig (head m2))

  -- tweens will take a list of Movies and append them all, interpolating
  -- n frames between every Movie.
tweens :: Int -> [Movie] -> Movie
tweens n = foldr (tween n) [] 


-}