/*
 *	XFree86 VidMode and DGA support by Jens Vaasjo <jvaasjo@iname.com>
 */

#ifdef xf86_dga

#define __XF86_C

#include"xmame.h"
#include<sys/types.h>
#include<sys/wait.h>
#include<X11/extensions/xf86dga.h>
#include<X11/extensions/xf86vmode.h>

static void PseudoColor_sysdep_modify_pen(int pen,
	unsigned char red,unsigned char green,unsigned char blue);
static void TrueColor_sysdep_modify_pen(int pen,
	unsigned char red,unsigned char green,unsigned char blue);

static void bpp8_sysdep_update_display(void);
static void bpp16_sysdep_update_display(void);

static void PseudoColor_sysdep_clear_screen(void);
static void TrueColor_sysdep_clear_screen(void);

static struct
{
	int screen;
	Visual *visual;
	int depth;
	unsigned char *addr;
	int shift_r;
	int shift_g;
	int shift_b;
	int grabbed_keybd;
	int grabbed_mouse;
	char *base_addr;
	int width;
	int bank_size;
	int ram_size;
	Colormap cmap;
	unsigned *pixels;
	void (*func_sysdep_modify_pen)(int,
		unsigned char,unsigned char,unsigned char);
	void (*func_sysdep_update_display)(void);
	void (*func_sysdep_clear_screen)(void);
	int PseudoColor_palette_dirty;
} xf86ctx = {-1,NULL,-1,NULL,-1,-1,-1,0,0,NULL,
		-1,-1,-1,0,NULL,NULL,NULL,NULL,0};

int xf86_init(void)
{
	int major,minor,event_base,error_base,flags;

	/*
	 * we can not use displayname here because XF86DGAForkApp(int) in
	 * libXxf86dga.a uses NULL so we have to use NULL as well to make
	 * sure all forked safety processes use the same display,
	 * a possible workaround would be setenv("DISPLAY",displayname,0)
	 */
	display = XOpenDisplay(NULL);
	if(!display)
	{
		/* don't use stderr_file here, it isn't assigned any
		   value yet! */
		fprintf(stderr,"XOpenDisplay failed\n");
		return OSD_NOT_OK;
	}

	xf86ctx.screen = DefaultScreen(display);

	if(!XF86DGAQueryVersion(display,&major,&minor))
	{
		fprintf(stderr,"XF86DGAQueryVersion failed\n");
		XCloseDisplay(display);
		return OSD_NOT_OK;
	}

	if(!XF86DGAQueryExtension(display,&event_base,&error_base))
	{
		fprintf(stderr,"XF86DGAQueryExtension failed\n");
		XCloseDisplay(display);
		return OSD_NOT_OK;
	}

	if(!XF86DGAQueryDirectVideo(display,xf86ctx.screen,&flags))
	{
		fprintf(stderr,"XF86DGAQueryDirectVideo failed\n");
		XCloseDisplay(display);
		return OSD_NOT_OK;
	}
	if(!(flags & XF86DGADirectPresent))
	{
		fprintf(stderr,"XF86DGADirectVideo support is not present\n");
		XCloseDisplay(display);
		return OSD_NOT_OK;
	}

	if(!XF86DGAGetVideo(display,xf86ctx.screen,
		&xf86ctx.base_addr,&xf86ctx.width,
		&xf86ctx.bank_size,&xf86ctx.ram_size))
	{
		fprintf(stderr,"XF86DGAGetVideo failed\n");
		XCloseDisplay(display);
		return OSD_NOT_OK;
	}

	if(setuid(getuid()))
	{
		perror("setuid");
		XCloseDisplay(display);
		return OSD_NOT_OK;
	}

	return OSD_OK;
}

static int xf86vidmode_check_exts(void)
{
	int major,minor,event_base,error_base;

	if(!XF86VidModeQueryVersion(display,&major,&minor))
	{
		fprintf(stderr_file,"XF86VidModeQueryVersion failed\n");
		return 1;
	}

	if(!XF86VidModeQueryExtension(display,&event_base,&error_base))
	{
		fprintf(stderr_file,"XF86VidModeQueryExtension failed\n");
		return 1;
	}

	return 0;
}

static XF86VidModeModeInfo *find_best_vidmode(void)
{
	XF86VidModeModeInfo **modes,*bestmode = NULL;
	unsigned size,bestsize = ~0;
	int i,modecount = 0;

	if(!XF86VidModeGetAllModeLines(display,xf86ctx.screen,
						&modecount,&modes))
	{
		fprintf(stderr_file,"XF86VidModeGetAllModeLines failed\n");
		return NULL;
	}

	for(i=0;i<modecount;i++)
	{
		if(visual_width  > modes[i]->hdisplay) continue;
		if(visual_height > modes[i]->vdisplay) continue;

		size = modes[i]->hdisplay * modes[i]->vdisplay;
		if(size <= bestsize)
		{
			bestsize = size;
			bestmode = modes[i];
		}
	}

	return bestmode;
}

static Bool xf86vidmodegetmodeinfo(XF86VidModeModeInfo *modeinfo)
{
	XF86VidModeModeLine modeline;
	int dotclock;
	Bool err;

	err = XF86VidModeGetModeLine(display,xf86ctx.screen,
					&dotclock,&modeline);

	modeinfo->dotclock = dotclock;
	modeinfo->hdisplay = modeline.hdisplay;
	modeinfo->hsyncstart = modeline.hsyncstart;
	modeinfo->hsyncend = modeline.hsyncend;
	modeinfo->htotal = modeline.htotal;
	modeinfo->vdisplay = modeline.vdisplay;
	modeinfo->vsyncstart = modeline.vsyncstart;
	modeinfo->vsyncend = modeline.vsyncend;
	modeinfo->vtotal = modeline.vtotal;
	modeinfo->flags = modeline.flags;
	modeinfo->privsize = modeline.privsize;
	modeinfo->private = modeline.private;

	return err;
}

static int xf86vidmodeforkapp(XF86VidModeModeInfo *mode,int vp_x,int vp_y)
{
	Display *disp;
	int status;
	pid_t pid;

	pid = fork();
	if(pid > 0)
	{
		waitpid(pid,&status,0);
		disp = XOpenDisplay(NULL);
		XF86VidModeSwitchToMode(disp,xf86ctx.screen,mode);
		XF86VidModeSetViewPort(disp,xf86ctx.screen,vp_x,vp_y);
                /* 'Mach64-hack': restores screen when screwed up */
                XF86VidModeSwitchMode(disp,xf86ctx.screen,-1);
                XF86VidModeSwitchMode(disp,xf86ctx.screen,1);
                /**************************************************/
		XSync(disp,False);
		_exit(!WIFEXITED(status));
	}

	return pid;
}

static int xf86vidmode_setup_mode_restore(void)
{
	XF86VidModeModeInfo oldmode;
	int oldviewport_x,oldviewport_y;

	if(!xf86vidmodegetmodeinfo(&oldmode))
	{
		fprintf(stderr_file,"XF86VidModeGetModeLine failed\n");
		return 1;
	}

	if(!XF86VidModeGetViewPort(display,xf86ctx.screen,
				&oldviewport_x,&oldviewport_y))
	{
		fprintf(stderr_file,"XF86VidModeGetViewPort failed\n");
		return 1;
	}

	if(xf86vidmodeforkapp(&oldmode,oldviewport_x,oldviewport_y))
	{
		perror("fork");
		return 1;
	}

	return 0;
}

static int setup_PseudoColor(void)
{
	XColor color;
	int i;

	xf86ctx.cmap = XCreateColormap(display,window,xf86ctx.visual,AllocAll);

	for(i=0;i<xf86ctx.visual->map_entries;i++)
	{
		color.pixel = i;
		color.red   = 0;
		color.green = 0;
		color.blue  = 0;
		color.flags = DoRed | DoGreen | DoBlue;

		XStoreColor(display,xf86ctx.cmap,&color);
	}

	if(!XF86DGAInstallColormap(display,xf86ctx.screen,xf86ctx.cmap))
	{
		fprintf(stderr_file,"XF86DGAInstallColormap failed\n");
		return 1;
	}

	return 0;
}

static int setup_TrueColor(void)
{
	unsigned long n;

	for(xf86ctx.shift_r = 0,n = 1 << (8 * sizeof(n) - 1);
		n && (!(n & xf86ctx.visual->red_mask));
			n >>= 1,xf86ctx.shift_r++);

	for(xf86ctx.shift_g = 0,n = 1 << (8 * sizeof(n) - 1);
		n && (!(n & xf86ctx.visual->green_mask));
			n >>= 1,xf86ctx.shift_g++);

	for(xf86ctx.shift_b = 0,n = 1 << (8 * sizeof(n) - 1);
		n && (!(n & xf86ctx.visual->blue_mask));
			n >>= 1,xf86ctx.shift_b++);

	xf86ctx.pixels = (unsigned*)calloc(0xffff,sizeof(unsigned));
	if(!xf86ctx.pixels)
	{
		perror("calloc");
		return 1;
	}

	return 0;
}

static int unsupported_check(void)
{
	if(xf86ctx.bank_size != (xf86ctx.ram_size * 1024))
	{
		fprintf(stderr_file,"banked graphics modes not supported\n");
		return 1;
	}

	switch(xf86ctx.depth)
	{
		case 8:
			if(xf86ctx.visual->class != PseudoColor)
			{
				fprintf(stderr_file,"only PseudoColor display class"
					" is supported at depth of %dbpp\n",
					xf86ctx.depth);
				return 1;
			}

			break;

		case 16:
			if(xf86ctx.visual->class != TrueColor)
			{
				fprintf(stderr_file,"only TrueColor display class"
					" is supported at depth of %dbpp\n",
					xf86ctx.depth);
				return 1;
			}

			break;

		default:
			fprintf(stderr_file,
				"unsupported depth %dbpp\n",xf86ctx.depth);
			return 1;
	}

	return 0;
}

static int setup_graphics(XF86VidModeModeInfo *modeinfo)
{
	switch(xf86ctx.visual->class)
	{
		case PseudoColor:
			if(setup_PseudoColor()) return 1;
			xf86ctx.func_sysdep_modify_pen
				= PseudoColor_sysdep_modify_pen;
			xf86ctx.func_sysdep_clear_screen
				= PseudoColor_sysdep_clear_screen;
			break;

		case TrueColor:
			if(setup_TrueColor()) return 1;
			xf86ctx.func_sysdep_modify_pen
				= TrueColor_sysdep_modify_pen;
			xf86ctx.func_sysdep_clear_screen
				= TrueColor_sysdep_clear_screen;
			break;

		default:
			fprintf(stderr_file,"display class unsupported\n");
			return 1;
	}

	switch(xf86ctx.depth)
	{
		case 8:
			xf86ctx.func_sysdep_update_display
				= bpp8_sysdep_update_display;
			break;

		case 16:
			xf86ctx.func_sysdep_update_display
				= bpp16_sysdep_update_display;
			break;

		default:
			fprintf(stderr_file,
				"unsupported depth %dbpp\n",xf86ctx.depth);
			return 1;
	}

	xf86ctx.addr  = (unsigned char*)xf86ctx.base_addr;
	xf86ctx.addr += ((modeinfo->hdisplay - visual_width) / 2)
						* (xf86ctx.depth / 8);
	xf86ctx.addr += ((modeinfo->vdisplay - visual_height) / 2)
				* xf86ctx.width * (xf86ctx.depth / 8);

	return 0;
}

int sysdep_create_display(void)
{
	XF86VidModeModeInfo *bestmode;

	xf86ctx.visual = DefaultVisual(display,xf86ctx.screen);
	xf86ctx.depth = DefaultDepth(display,xf86ctx.screen);
	window = RootWindow(display,xf86ctx.screen);

	if(unsupported_check())
	{
		osd_close_display();
		return OSD_NOT_OK;
	}

	if(xf86vidmode_check_exts())
	{
		osd_close_display();
		return OSD_NOT_OK;
	}

	bestmode = find_best_vidmode();
	if(!bestmode)
	{
		fprintf(stderr_file,"no suitable mode found\n");
		osd_close_display();
		return OSD_NOT_OK;
	}

	if(xf86vidmode_setup_mode_restore())
	{
		osd_close_display();
		return OSD_NOT_OK;
	}

	fprintf(stderr_file,"VidMode Switching To Mode: %d x %d\n",
			bestmode->hdisplay,bestmode->vdisplay);

	if(!XF86VidModeSwitchToMode(display,xf86ctx.screen,bestmode))
	{
		fprintf(stderr_file,"XF86VidModeSwitchToMode failed\n");
		osd_close_display();
		return OSD_NOT_OK;
	}

	if(!XF86VidModeSetViewPort(display,xf86ctx.screen,0,0))
	{
		fprintf(stderr_file,"XF86VidModeSetViewPort failed\n");
		osd_close_display();
		return OSD_NOT_OK;
	}

	if(XGrabKeyboard(display,window,True,
		GrabModeAsync,GrabModeAsync,CurrentTime))
	{
		fprintf(stderr_file,"XGrabKeyboard failed\n");
		osd_close_display();
		return OSD_NOT_OK;
	}
	xf86ctx.grabbed_keybd = 1;

	if(use_mouse)
	{
		if(XGrabPointer(display,window,True,
			PointerMotionMask|ButtonPressMask|ButtonReleaseMask,
			GrabModeAsync,GrabModeAsync,None,None,CurrentTime))
		{
			fprintf(stderr_file,"XGrabPointer failed\n");
			osd_close_display();
			return OSD_NOT_OK;
		}
		
		xf86ctx.grabbed_mouse = 1;
	}

#ifdef X11_JOYSTICK
	if(use_joystick) x11_joystick_init();
#endif

	if(XF86DGAForkApp(xf86ctx.screen))
	{
		perror("fork");
		osd_close_display();
		return OSD_NOT_OK;
	}

	if(!XF86DGADirectVideo(display,xf86ctx.screen,
		XF86DGADirectGraphics|XF86DGADirectMouse|XF86DGADirectKeyb))
	{
		fprintf(stderr_file,"XF86DGADirectVideo failed\n");
		osd_close_display();
		return OSD_NOT_OK;
	}

	if(!XF86DGASetViewPort(display,xf86ctx.screen,0,0))
	{
		fprintf(stderr_file,"XF86DGASetViewPort failed\n");
		osd_close_display();
		return OSD_NOT_OK;
	}

	if(setup_graphics(bestmode))
	{
		osd_close_display();
		return OSD_NOT_OK;
	}

	sysdep_clear_screen();

	return OSD_OK;
}

static unsigned long mkcolor(unsigned char r,unsigned char g,unsigned char b)
{
	unsigned long p = 0;

	p |= xf86ctx.visual->red_mask   & ((r << 24) >> xf86ctx.shift_r);
	p |= xf86ctx.visual->green_mask & ((g << 24) >> xf86ctx.shift_g);
	p |= xf86ctx.visual->blue_mask  & ((b << 24) >> xf86ctx.shift_b);

	return p;
}

static void pixels_add_color(unsigned index,unsigned long color)
{
	unsigned n;

	for(n=0;n<256;n++)
	{
		xf86ctx.pixels[(n<<8) | index] &= 0xffff0000;
		xf86ctx.pixels[(n<<8) | index] |= color;

		xf86ctx.pixels[(index<<8) | n] &= 0x0000ffff;
		xf86ctx.pixels[(index<<8) | n] |= color << 16;
	}
}

/* not needed since we have full hardware access */
void sysdep_alloc_palette(void)
{
}

static void PseudoColor_sysdep_modify_pen(int pen,
	unsigned char red,unsigned char green,unsigned char blue)
{
	XColor color;

	color.pixel = pen;
	color.red   = red   << 8;
	color.green = green << 8;
	color.blue  = blue  << 8;
	color.flags = DoRed | DoGreen | DoBlue;

	XStoreColor(display,xf86ctx.cmap,&color);
	xf86ctx.PseudoColor_palette_dirty=1;
}

static void TrueColor_sysdep_modify_pen(int pen,
	unsigned char red,unsigned char green,unsigned char blue)
{
	pixels_add_color(pen,mkcolor(red,green,blue));
}

void sysdep_modify_pen(int pen,
	unsigned char red,unsigned char green,unsigned char blue)
{
	(*xf86ctx.func_sysdep_modify_pen)(pen,red,green,blue);
}

static void bpp8_sysdep_update_display(void)
{
	unsigned char *dest = xf86ctx.addr;
	unsigned y;

	if (xf86ctx.PseudoColor_palette_dirty)
	{
		XF86DGAInstallColormap(display,xf86ctx.screen,xf86ctx.cmap);
		xf86ctx.PseudoColor_palette_dirty=0;
	}
	
	for(y=visual.min_y;y<=visual.max_y;y++,dest+=xf86ctx.width)
	{
		memcpy(dest,bitmap->line[y] + visual.min_x,visual_width);
	}
}

static void bpp16_sysdep_update_display(void)
{
	unsigned *dst,*dest = (unsigned*)xf86ctx.addr;
	unsigned short index;
	unsigned x,y;

	for(y=visual.min_y;y<=visual.max_y;y++,dest+=xf86ctx.width/2)
	{
		for(x=visual.min_x,dst=dest;x<=visual.max_x;x+=2,dst++)
		{
			index = *(unsigned short*)(bitmap->line[y] + x);
			*dst = xf86ctx.pixels[index];
		}
	}
}

void sysdep_update_display(void)
{
	(*xf86ctx.func_sysdep_update_display)();
}

static void PseudoColor_sysdep_clear_screen(void)
{
	memset(xf86ctx.base_addr,0,xf86ctx.bank_size);
}

static void TrueColor_sysdep_clear_screen(void)
{
	memset(xf86ctx.base_addr,0,xf86ctx.bank_size);
}

void sysdep_clear_screen(void)
{
	(*xf86ctx.func_sysdep_clear_screen)();
}

void osd_mark_dirty(int sx,int sy,int ex,int ey,int ui)
{
}

void osd_close_display(void)
{
	if(!display) return;

	if(xf86ctx.visual->class == PseudoColor)
	{
		if(xf86ctx.cmap) XFreeColormap(display,xf86ctx.cmap);
	}
	else if(xf86ctx.visual->class == TrueColor)
	{
		free(xf86ctx.pixels);
		xf86ctx.pixels = NULL;
	}

	if(xf86ctx.grabbed_mouse) XUngrabPointer(display,CurrentTime);
	if(xf86ctx.grabbed_keybd) XUngrabKeyboard(display,CurrentTime);

	XCloseDisplay(display);
	display = NULL;
}

#endif /*def xf86*/
