/***************************************************************************

Mappy memory map (preliminary)

CPU #1:
0000-07ff video RAM
0800-0fff color RAM
1000-177f RAM
1780-17ff sprite data 1 (sprite number & color)
1800-1f7f RAM
1f80-1fff sprite data 2 (x, y position)
2000-277f RAM
2780-27ff sprite data 3 (high bit of y, flip flags, double-size flags)
3800-3fff scroll register map
4040-43ff RAM shared with CPU #2
4800-480f custom I/O chip #1
4810-481f custom I/O chip #2
5002-5003 IRQ enable
500a-500b CPU #2 enable
8000      watchdog timer
a000-ffff ROM

CPU #2:
0000-0040 sound registers
0040-03ff RAM shared with CPU #1
2000-2001 IRQ enable
2006-2007 sound enable
e000-ffff ROM

Interrupts:
CPU #1 IRQ generated by VBLANK
CPU #2 IRQ generated by VBLANK

***************************************************************************/

#include "driver.h"
#include "vidhrdw/generic.h"



/* machine driver data & functions */
extern unsigned char *mappy_sharedram;
extern unsigned char *mappy_customio_1,*mappy_customio_2;
void mappy_init_machine(void);
void motos_init_machine(void);
int mappy_sharedram_r(int offset);
void mappy_sharedram_w(int offset,int data);
void mappy_customio_w_1(int offset,int data);
void mappy_customio_w_2(int offset,int data);
int mappy_interrupt_1(void);
int mappy_interrupt_2(void);
void mappy_interrupt_enable_1_w(int offset,int data);
void mappy_interrupt_enable_2_w(int offset,int data);
void mappy_cpu_enable_w(int offset,int data);

int mappy_sharedram_r2(int offset);
int mappy_cpu1ram_r(int offset);
int mappy_customio_r_1(int offset);
int mappy_customio_r_2(int offset);

int digdug2_sharedram_r2(int offset);
int digdug2_cpu1ram_r(int offset);
int digdug2_customio_r_1(int offset);
int digdug2_customio_r_2(int offset);

int motos_sharedram_r2(int offset);
int motos_cpu1ram_r(int offset);
int motos_customio_r_1(int offset);
int motos_customio_r_2(int offset);

int todruaga_sharedram_r2(int offset);
int todruaga_cpu1ram_r(int offset);
int todruaga_customio_r_1(int offset);
int todruaga_customio_r_2(int offset);

/* video driver data & functions */
int mappy_vh_start(void);
int motos_vh_start(void);
void mappy_vh_stop(void);
void mappy_vh_screenrefresh(struct osd_bitmap *bitmap,int full_refresh);
void mappy_videoram_w(int offset,int data);
void mappy_colorram_w(int offset,int data);
void mappy_scroll_w(int offset,int data);
void mappy_vh_convert_color_prom(unsigned char *palette, unsigned short *colortable,const unsigned char *color_prom);

/* sound driver data & functions */
extern unsigned char *mappy_soundregs;
void mappy_sound_w(int offset,int data);
void mappy_sound_enable_w(int offset,int data);


/* CPU 1 read addresses */
static struct MemoryReadAddress mappy_readmem_cpu1[] =
{
	{ 0xa000, 0xffff, MRA_ROM },                                 /* ROM code */
	{ 0x4040, 0x43ff, MRA_RAM, &mappy_sharedram },               /* shared RAM with the sound CPU */
	{ 0x4800, 0x480f, mappy_customio_r_1, &mappy_customio_1 },   /* custom I/O chip #1 interface */
	{ 0x4810, 0x481f, mappy_customio_r_2, &mappy_customio_2 },   /* custom I/O chip #2 interface */
	{ 0x0000, 0x9fff, mappy_cpu1ram_r },                         /* RAM everywhere else */

	{ -1 }  /* end of table */
};

static struct MemoryReadAddress digdug2_readmem_cpu1[] =
{
	{ 0x8000, 0xffff, MRA_ROM },                                 /* ROM code */
	{ 0x4040, 0x43ff, MRA_RAM, &mappy_sharedram },               /* shared RAM with the sound CPU */
	{ 0x4800, 0x480f, digdug2_customio_r_1, &mappy_customio_1 },   /* custom I/O chip #1 interface */
	{ 0x4810, 0x481f, digdug2_customio_r_2, &mappy_customio_2 },   /* custom I/O chip #2 interface */
	{ 0x4820, 0x4bff, MRA_RAM },                                 /* extra RAM for Dig Dug 2 */
	{ 0x0000, 0x7fff, digdug2_cpu1ram_r },                       /* RAM everywhere else */

	{ -1 }  /* end of table */
};

static struct MemoryReadAddress motos_readmem_cpu1[] =
{
	{ 0x8000, 0xffff, MRA_ROM },                                 /* ROM code */
	{ 0x4040, 0x43ff, MRA_RAM, &mappy_sharedram },               /* shared RAM with the sound CPU */
	{ 0x4800, 0x480f, motos_customio_r_1, &mappy_customio_1 },   /* custom I/O chip #1 interface */
	{ 0x4810, 0x481f, motos_customio_r_2, &mappy_customio_2 },   /* custom I/O chip #2 interface */
	{ 0x0000, 0x7fff, motos_cpu1ram_r },                       /* RAM everywhere else */

	{ -1 }  /* end of table */
};

static struct MemoryReadAddress todruaga_readmem_cpu1[] =
{
	{ 0x8000, 0xffff, MRA_ROM },                                 /* ROM code */
	{ 0x4040, 0x43ff, MRA_RAM, &mappy_sharedram },               /* shared RAM with the sound CPU */
	{ 0x4800, 0x480f, todruaga_customio_r_1, &mappy_customio_1 },   /* custom I/O chip #1 interface */
	{ 0x4810, 0x481f, todruaga_customio_r_2, &mappy_customio_2 },   /* custom I/O chip #2 interface */
	{ 0x0000, 0x7fff, todruaga_cpu1ram_r },                       /* RAM everywhere else */

	{ -1 }  /* end of table */
};


/* CPU 1 write addresses */
static struct MemoryWriteAddress writemem_cpu1[] =
{
	{ 0x1000, 0x177f, MWA_RAM },                                 /* general RAM, area 1 */
	{ 0x1800, 0x1f7f, MWA_RAM },                                 /* general RAM, area 2 */
	{ 0x2000, 0x277f, MWA_RAM },                                 /* general RAM, area 3 */
	{ 0x4040, 0x43ff, MWA_RAM },                                 /* shared RAM with the sound CPU */
	{ 0x4820, 0x4bff, MWA_RAM },                                 /* extra RAM for Dig Dug 2 */
	{ 0x0000, 0x07ff, mappy_videoram_w, &videoram, &videoram_size },/* video RAM */
	{ 0x0800, 0x0fff, mappy_colorram_w, &colorram },             /* color RAM */
	{ 0x1780, 0x17ff, MWA_RAM, &spriteram, &spriteram_size },    /* sprite RAM, area 1 */
	{ 0x1f80, 0x1fff, MWA_RAM, &spriteram_2 },                   /* sprite RAM, area 2 */
	{ 0x2780, 0x27ff, MWA_RAM, &spriteram_3 },                   /* sprite RAM, area 3 */
	{ 0x3800, 0x3fff, mappy_scroll_w },                          /* scroll registers */
	{ 0x4800, 0x480f, mappy_customio_w_1 },                      /* custom I/O chip #1 interface */
	{ 0x4810, 0x481f, mappy_customio_w_2 },                      /* custom I/O chip #2 interface */
	{ 0x5002, 0x5003, mappy_interrupt_enable_1_w },              /* interrupt enable */
	{ 0x500a, 0x500b, mappy_cpu_enable_w },                      /* sound CPU enable */
	{ 0x8000, 0x8000, MWA_NOP },                                 /* watchdog timer */
	{ 0x8000, 0xffff, MWA_ROM },                                 /* ROM code */

	{ -1 }  /* end of table */
};


/* CPU 2 read addresses */
static struct MemoryReadAddress mappy_readmem_cpu2[] =
{
	{ 0xe000, 0xffff, MRA_ROM },                                 /* ROM code */
	{ 0x0040, 0x03ff, mappy_sharedram_r2 },                      /* shared RAM with the main CPU */

	{ -1 }  /* end of table */
};

static struct MemoryReadAddress digdug2_readmem_cpu2[] =
{
	{ 0xe000, 0xffff, MRA_ROM },                                 /* ROM code */
	{ 0x0040, 0x03ff, digdug2_sharedram_r2 },                    /* shared RAM with the main CPU */

	{ -1 }  /* end of table */
};

static struct MemoryReadAddress motos_readmem_cpu2[] =
{
	{ 0xe000, 0xffff, MRA_ROM },                                 /* ROM code */
	{ 0x0040, 0x03ff, motos_sharedram_r2 },                    /* shared RAM with the main CPU */

	{ -1 }  /* end of table */
};

static struct MemoryReadAddress todruaga_readmem_cpu2[] =
{
	{ 0xe000, 0xffff, MRA_ROM },                                 /* ROM code */
	{ 0x0040, 0x03ff, motos_sharedram_r2 },                    /* shared RAM with the main CPU */

	{ -1 }  /* end of table */
};


/* CPU 2 write addresses */
static struct MemoryWriteAddress writemem_cpu2[] =
{
	{ 0x0040, 0x03ff, mappy_sharedram_w },                       /* shared RAM with the main CPU */
	{ 0x0000, 0x003f, mappy_sound_w, &mappy_soundregs },         /* sound control registers */
	{ 0x2000, 0x2001, mappy_interrupt_enable_2_w },              /* interrupt enable */
	{ 0x2006, 0x2007, mappy_sound_enable_w },                    /* sound enable */
	{ 0xe000, 0xffff, MWA_ROM },                                 /* ROM code */

	{ -1 }  /* end of table */
};


/* input from the outside world */
INPUT_PORTS_START( mappy_input_ports )
	PORT_START      /* DSW0 */
	PORT_DIPNAME( 0x03, 0x00, "Difficulty", IP_KEY_NONE )
	PORT_DIPSETTING(    0x00, "Easy" )
	PORT_DIPSETTING(    0x01, "Medium" )
	PORT_DIPSETTING(    0x02, "Hard" )
	PORT_DIPSETTING(    0x03, "Hardest" )
	PORT_BIT( 0x1c, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_DIPNAME( 0x20, 0x00, "Demo Sounds", IP_KEY_NONE )
	PORT_DIPSETTING(    0x20, "Off" )
	PORT_DIPSETTING(    0x00, "On" )
	PORT_BITX(    0x40, 0x00, IPT_DIPSWITCH_NAME | IPF_CHEAT, "Rack Test", OSD_KEY_F1, IP_JOY_NONE, 0 )
	PORT_DIPSETTING(    0x00, "Off" )
	PORT_DIPSETTING(    0x40, "On" )
	PORT_DIPNAME( 0x80, 0x00, "Freeze", IP_KEY_NONE )
	PORT_DIPSETTING(    0x00, "Off" )
	PORT_DIPSETTING(    0x80, "On" )

	PORT_START      /* DSW1 */
	PORT_DIPNAME( 0x07, 0x00, "Coins", IP_KEY_NONE )
	PORT_DIPSETTING(    0x00, "1 Coin/1 Credit" )
	PORT_DIPSETTING(    0x01, "1 Coin/2 Credits" )
	PORT_DIPSETTING(    0x02, "1 Coin/3 Credits" )
	PORT_DIPSETTING(    0x03, "1 Coin/6 Credits" )
	PORT_DIPSETTING(    0x04, "2 Coins/1 Credit" )
	PORT_DIPSETTING(    0x05, "2 Coins/3 Credits" )
	PORT_DIPSETTING(    0x06, "3 Coins/1 Credits" )
	PORT_DIPSETTING(    0x07, "3 Coins/2 Credits" )
	/* TODO: bonus scores are different for 5 lives */
	PORT_DIPNAME( 0x38, 0x00, "Bonus Life", IP_KEY_NONE )
	PORT_DIPSETTING(    0x38, "None" )
	PORT_DIPSETTING(    0x20, "20k" )
	PORT_DIPSETTING(    0x08, "20k 60k" )
	PORT_DIPSETTING(    0x00, "20k 70k" )
	PORT_DIPSETTING(    0x10, "20k 80k" )
	PORT_DIPSETTING(    0x18, "30k 100k" )
	PORT_DIPSETTING(    0x28, "20k 70k 70k" )
	PORT_DIPSETTING(    0x30, "20k 80k 80k" )
	PORT_DIPNAME( 0xc0, 0x00, "Lives", IP_KEY_NONE )
	PORT_DIPSETTING(    0x80, "1" )
	PORT_DIPSETTING(    0xc0, "2" )
	PORT_DIPSETTING(    0x00, "3" )
	PORT_DIPSETTING(    0x40, "5" )

	PORT_START      /* DSW2 */
	PORT_BIT( 0x03, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_DIPNAME( 0x04, 0x00, "Orientation", IP_KEY_NONE )
	PORT_DIPSETTING(    0x00, "Upright" )
	PORT_DIPSETTING(    0x04, "Cocktail" )
	PORT_BITX(    0x08, 0x00, IPT_DIPSWITCH_NAME | IPF_TOGGLE, "Service Mode", OSD_KEY_F2, IP_JOY_NONE, 0 )
	PORT_DIPSETTING(    0x00, "Off" )
	PORT_DIPSETTING(    0x08, "On" )
	PORT_BIT( 0xf0, IP_ACTIVE_HIGH, IPT_UNUSED )

	PORT_START      /* FAKE */
	/* The player inputs are not memory mapped, they are handled by an I/O chip. */
	/* These fake input ports are read by mappy_customio_data_r() */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT | IPF_2WAY )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT | IPF_2WAY )
	PORT_BITX(0x10, IP_ACTIVE_HIGH, IPT_BUTTON1 | IPF_IMPULSE,
			IP_NAME_DEFAULT, IP_KEY_DEFAULT, IP_JOY_DEFAULT, 1 )
	PORT_BITX(0x20, IP_ACTIVE_HIGH, IPT_BUTTON1, 0, IP_KEY_PREVIOUS, IP_JOY_PREVIOUS, 0 )
	PORT_BIT( 0xc0, IP_ACTIVE_HIGH, IPT_UNUSED )

	PORT_START      /* FAKE */
	PORT_BITX(0x01, IP_ACTIVE_HIGH, IPT_COIN1 | IPF_IMPULSE,
			IP_NAME_DEFAULT, IP_KEY_DEFAULT, IP_JOY_DEFAULT, 1 )
	PORT_BITX(0x02, IP_ACTIVE_HIGH, IPT_COIN2 | IPF_IMPULSE,
			IP_NAME_DEFAULT, IP_KEY_DEFAULT, IP_JOY_DEFAULT, 1 )
	PORT_BIT( 0x0c, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_BITX(0x10, IP_ACTIVE_HIGH, IPT_START1 | IPF_IMPULSE,
			IP_NAME_DEFAULT, IP_KEY_DEFAULT, IP_JOY_DEFAULT, 1 )
	PORT_BITX(0x20, IP_ACTIVE_HIGH, IPT_START2 | IPF_IMPULSE,
			IP_NAME_DEFAULT, IP_KEY_DEFAULT, IP_JOY_DEFAULT, 1 )
	PORT_BIT( 0xc0, IP_ACTIVE_HIGH, IPT_UNUSED )
INPUT_PORTS_END


INPUT_PORTS_START( digdug2_input_ports )
	PORT_START      /* DSW0 */
	PORT_BIT( 0x1f, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_DIPNAME( 0x20, 0x00, "Lives", IP_KEY_NONE )
	PORT_DIPSETTING(    0x00, "3" )
	PORT_DIPSETTING(    0x20, "5" )
	PORT_DIPNAME( 0xc0, 0x00, "Coins", IP_KEY_NONE )
	PORT_DIPSETTING(    0x00, "1 Coin/1 Credit" )
	PORT_DIPSETTING(    0x40, "2 Coins/1 Credits" )
	PORT_DIPSETTING(    0x80, "1 Coin/2 Credits" )
	PORT_DIPSETTING(    0xc0, "3 Coins/1 Credits" )

	PORT_START      /* DSW1 */
	PORT_DIPNAME( 0x03, 0x00, "Extend", IP_KEY_NONE )
	PORT_DIPSETTING(    0x00, "Type A" )
	PORT_DIPSETTING(    0x02, "Type B" )
	PORT_DIPSETTING(    0x01, "Type C" )
	PORT_DIPSETTING(    0x03, "Type D" )
	PORT_DIPNAME( 0x04, 0x00, "Level Select", IP_KEY_NONE )
	PORT_DIPSETTING(    0x00, "Off" )
	PORT_DIPSETTING(    0x04, "On" )
	PORT_DIPNAME( 0x08, 0x00, "Freeze", IP_KEY_NONE )
	PORT_DIPSETTING(    0x00, "Off" )
	PORT_DIPSETTING(    0x08, "On" )
	PORT_BITX(0x10, IP_ACTIVE_HIGH, IPT_BUTTON2 | IPF_IMPULSE,
			IP_NAME_DEFAULT, IP_KEY_DEFAULT, IP_JOY_DEFAULT, 1 )
	PORT_BITX(0x20, IP_ACTIVE_HIGH, IPT_BUTTON2, 0, IP_KEY_PREVIOUS, IP_JOY_PREVIOUS, 0 )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_DIPNAME( 0x80, 0x00, "Orientation", IP_KEY_NONE )
	PORT_DIPSETTING(    0x00, "Upright" )
	PORT_DIPSETTING(    0x80, "Cocktail" )

	PORT_START      /* DSW2 */
	PORT_BIT( 0x07, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_BITX(    0x08, 0x00, IPT_DIPSWITCH_NAME | IPF_TOGGLE, "Service Mode", OSD_KEY_F2, IP_JOY_NONE, 0 )
	PORT_DIPSETTING(    0x00, "Off" )
	PORT_DIPSETTING(    0x08, "On" )
	PORT_BIT( 0xf0, IP_ACTIVE_HIGH, IPT_UNUSED )

	PORT_START      /* FAKE */
	/* The player inputs are not memory mapped, they are handled by an I/O chip. */
	/* These fake input ports are read by mappy_customio_data_r() */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP | IPF_4WAY )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT | IPF_4WAY )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN | IPF_4WAY )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT | IPF_4WAY )
	PORT_BITX(0x10, IP_ACTIVE_HIGH, IPT_BUTTON1 | IPF_IMPULSE,
			IP_NAME_DEFAULT, IP_KEY_DEFAULT, IP_JOY_DEFAULT, 1 )
	PORT_BITX(0x20, IP_ACTIVE_HIGH, IPT_BUTTON1, 0, IP_KEY_PREVIOUS, IP_JOY_PREVIOUS, 0 )
	PORT_BIT( 0xc0, IP_ACTIVE_HIGH, IPT_UNUSED )

	PORT_START      /* FAKE */
	PORT_BITX(0x01, IP_ACTIVE_HIGH, IPT_COIN1 | IPF_IMPULSE,
			IP_NAME_DEFAULT, IP_KEY_DEFAULT, IP_JOY_DEFAULT, 1 )
	PORT_BITX(0x02, IP_ACTIVE_HIGH, IPT_COIN2 | IPF_IMPULSE,
			IP_NAME_DEFAULT, IP_KEY_DEFAULT, IP_JOY_DEFAULT, 1 )
	PORT_BIT( 0x0c, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_BITX(0x10, IP_ACTIVE_HIGH, IPT_START1 | IPF_IMPULSE,
			IP_NAME_DEFAULT, IP_KEY_DEFAULT, IP_JOY_DEFAULT, 1 )
	PORT_BITX(0x20, IP_ACTIVE_HIGH, IPT_START2 | IPF_IMPULSE,
			IP_NAME_DEFAULT, IP_KEY_DEFAULT, IP_JOY_DEFAULT, 1 )
	PORT_BIT( 0xc0, IP_ACTIVE_HIGH, IPT_UNUSED )
INPUT_PORTS_END


INPUT_PORTS_START( motos_input_ports )
	PORT_START      /* DSW0 */
	PORT_DIPNAME( 0x01, 0x00, "Reset", IP_KEY_NONE )
	PORT_DIPSETTING(    0x00, "Off" )
	PORT_DIPSETTING(    0x01, "On" )
	PORT_DIPNAME( 0x06, 0x00, "Coins", IP_KEY_NONE )
	PORT_DIPSETTING(    0x00, "1 Coin/1 Credit" )
	PORT_DIPSETTING(    0x02, "1 Coin/2 Credits" )
	PORT_DIPSETTING(    0x04, "2 Coins/1 Credit" )
	PORT_DIPSETTING(    0x06, "3 Coins/1 Credit" )
	PORT_DIPNAME( 0x08, 0x00, "Motos", IP_KEY_NONE )
	PORT_DIPSETTING(    0x00, "3" )
	PORT_DIPSETTING(    0x08, "5" )
	PORT_DIPNAME( 0x10, 0x00, "Rank", IP_KEY_NONE )
	PORT_DIPSETTING(    0x00, "A" )
	PORT_DIPSETTING(    0x10, "B" )
	PORT_DIPNAME( 0x60, 0x00, "Bonus life", IP_KEY_NONE )
	PORT_DIPSETTING(    0x00, "10k 30k 50k" )
	PORT_DIPSETTING(    0x20, "20k --  50k" )
	PORT_DIPSETTING(    0x40, "30k --  70k" )
	PORT_DIPSETTING(    0x60, "20k 70k" )
	PORT_DIPNAME( 0x80, 0x00, "Demo Sounds", IP_KEY_NONE )
	PORT_DIPSETTING(    0x80, "Off" )
	PORT_DIPSETTING(    0x00, "On" )

	PORT_START      /* DSW1 */
	PORT_BIT( 0x3f, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_DIPNAME( 0x40, 0x00, "Orientation", IP_KEY_NONE )
	PORT_DIPSETTING(    0x00, "Upright" )
	PORT_DIPSETTING(    0x40, "Cocktail" )
	PORT_BITX(    0x80, 0x00, IPT_DIPSWITCH_NAME | IPF_TOGGLE, "Service Mode", OSD_KEY_F2, IP_JOY_NONE, 0 )
	PORT_DIPSETTING(    0x00, "Off" )
	PORT_DIPSETTING(    0x80, "On" )

	PORT_START      /* FAKE */
	/* The player inputs are not memory mapped, they are handled by an I/O chip. */
	/* These fake input ports are read by mappy_customio_data_r() */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT | IPF_8WAY )
	PORT_BITX(0x10, IP_ACTIVE_HIGH, IPT_BUTTON1 | IPF_IMPULSE,
			IP_NAME_DEFAULT, IP_KEY_DEFAULT, IP_JOY_DEFAULT, 2 )
	PORT_BITX(0x20, IP_ACTIVE_HIGH, IPT_BUTTON1, 0, IP_KEY_PREVIOUS, IP_JOY_PREVIOUS, 0 )
	PORT_BIT( 0xc0, IP_ACTIVE_HIGH, IPT_UNUSED )

	PORT_START      /* FAKE */
	PORT_BITX(0x01, IP_ACTIVE_HIGH, IPT_COIN1 | IPF_IMPULSE,
			IP_NAME_DEFAULT, IP_KEY_DEFAULT, IP_JOY_DEFAULT, 2 )
	PORT_BITX(0x02, IP_ACTIVE_HIGH, IPT_COIN2 | IPF_IMPULSE,
			IP_NAME_DEFAULT, IP_KEY_DEFAULT, IP_JOY_DEFAULT, 2 )
	PORT_BIT( 0x0c, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_BITX(0x40, IP_ACTIVE_HIGH, IPT_START1 | IPF_IMPULSE,
			IP_NAME_DEFAULT, IP_KEY_DEFAULT, IP_JOY_DEFAULT, 2 )
	PORT_BITX(0x80, IP_ACTIVE_HIGH, IPT_START2 | IPF_IMPULSE,
			IP_NAME_DEFAULT, IP_KEY_DEFAULT, IP_JOY_DEFAULT, 2 )
	PORT_BIT( 0x30, IP_ACTIVE_HIGH, IPT_UNUSED )
INPUT_PORTS_END


INPUT_PORTS_START( todruaga_input_ports )
	PORT_START      /* DSW0 */
	PORT_BIT( 0x0f, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_DIPNAME( 0x30, 0x00, "Lives", IP_KEY_NONE )
	PORT_DIPSETTING(    0x20, "1" )
	PORT_DIPSETTING(    0x10, "2" )
	PORT_DIPSETTING(    0x00, "3" )
	PORT_DIPSETTING(    0x30, "5" )
	PORT_DIPNAME( 0xc0, 0x00, "Left Coin", IP_KEY_NONE )
	PORT_DIPSETTING(    0x00, "1 Coin/1 Credit" )
	PORT_DIPSETTING(    0x40, "2 Coins/1 Credits" )
	PORT_DIPSETTING(    0x80, "1 Coin/2 Credits" )
	PORT_DIPSETTING(    0xc0, "3 Coins/1 Credits" )

	PORT_START      /* DSW1 */
	PORT_DIPNAME( 0x01, 0x00, "Freeze", IP_KEY_NONE )
	PORT_DIPSETTING(    0x00, "Off" )
	PORT_DIPSETTING(    0x01, "On" )
	PORT_DIPNAME( 0x02, 0x00, "Reset", IP_KEY_NONE )
	PORT_DIPSETTING(    0x00, "Off" )
	PORT_DIPSETTING(    0x02, "On" )
	PORT_DIPNAME( 0x0c, 0x00, "Right Coin", IP_KEY_NONE )
	PORT_DIPSETTING(    0x00, "1 Coin/1 Credit" )
	PORT_DIPSETTING(    0x04, "2 Coins/1 Credits" )
	PORT_DIPSETTING(    0x08, "1 Coin/2 Credits" )
	PORT_DIPSETTING(    0x0c, "3 Coins/1 Credits" )
	PORT_BIT( 0x70, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_DIPNAME( 0x80, 0x00, "Orientation", IP_KEY_NONE )
	PORT_DIPSETTING(    0x00, "Upright" )
	PORT_DIPSETTING(    0x80, "Cocktail" )

	PORT_START      /* DSW2 */
	PORT_BIT( 0x07, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_BITX(    0x08, 0x00, IPT_DIPSWITCH_NAME | IPF_TOGGLE, "Service Mode", OSD_KEY_F2, IP_JOY_NONE, 0 )
	PORT_DIPSETTING(    0x00, "Off" )
	PORT_DIPSETTING(    0x08, "On" )
	PORT_BIT( 0xf0, IP_ACTIVE_HIGH, IPT_UNUSED )

	PORT_START      /* FAKE */
	/* The player inputs are not memory mapped, they are handled by an I/O chip. */
	/* These fake input ports are read by mappy_customio_data_r() */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP | IPF_4WAY )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT | IPF_4WAY )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN | IPF_4WAY )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT | IPF_4WAY )
	PORT_BITX(0x10, IP_ACTIVE_HIGH, IPT_BUTTON1 | IPF_IMPULSE,
			IP_NAME_DEFAULT, IP_KEY_DEFAULT, IP_JOY_DEFAULT, 1 )
	PORT_BITX(0x20, IP_ACTIVE_HIGH, IPT_BUTTON1, 0, IP_KEY_PREVIOUS, IP_JOY_PREVIOUS, 0 )
	PORT_BIT( 0xc0, IP_ACTIVE_HIGH, IPT_UNUSED )

	PORT_START      /* FAKE */
	PORT_BIT( 0x0f, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_BITX(0x10, IP_ACTIVE_HIGH, IPT_BUTTON2 | IPF_IMPULSE,      /* this is just a guess */
			IP_NAME_DEFAULT, IP_KEY_DEFAULT, IP_JOY_DEFAULT, 1 )
	PORT_BITX(0x20, IP_ACTIVE_HIGH, IPT_BUTTON2, 0, IP_KEY_PREVIOUS, IP_JOY_PREVIOUS, 0 )
	PORT_BIT( 0xc0, IP_ACTIVE_HIGH, IPT_UNUSED )

	PORT_START      /* FAKE */
	PORT_BITX(0x01, IP_ACTIVE_HIGH, IPT_COIN1 | IPF_IMPULSE,
			IP_NAME_DEFAULT, IP_KEY_DEFAULT, IP_JOY_DEFAULT, 1 )
	PORT_BITX(0x02, IP_ACTIVE_HIGH, IPT_COIN2 | IPF_IMPULSE,
			IP_NAME_DEFAULT, IP_KEY_DEFAULT, IP_JOY_DEFAULT, 1 )
	PORT_BIT( 0x0c, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_BITX(0x10, IP_ACTIVE_HIGH, IPT_START1 | IPF_IMPULSE,
			IP_NAME_DEFAULT, IP_KEY_DEFAULT, IP_JOY_DEFAULT, 1 )
	PORT_BITX(0x20, IP_ACTIVE_HIGH, IPT_START2 | IPF_IMPULSE,
			IP_NAME_DEFAULT, IP_KEY_DEFAULT, IP_JOY_DEFAULT, 1 )
	PORT_BIT( 0xc0, IP_ACTIVE_HIGH, IPT_UNUSED )
INPUT_PORTS_END


/* layout of the 8x8x2 character data */
static struct GfxLayout charlayout =
{
	8,8,             /* 8*8 characters */
	256,             /* 256 characters */
	2,             /* 2 bits per pixel */
	{ 0, 4 },      /* the two bitplanes for 4 pixels are packed into one byte */
	{ 7*8, 6*8, 5*8, 4*8, 3*8, 2*8, 1*8, 0*8 },   /* characters are rotated 90 degrees */
	{ 8*8+0, 8*8+1, 8*8+2, 8*8+3, 0, 1, 2, 3 },   /* bits are packed in groups of four */
	16*8           /* every char takes 16 bytes */
};


/* layout of the 16x16x4 sprite data */
static struct GfxLayout mappy_spritelayout =
{
	16,16,       /* 16*16 sprites */
	128,            /* 128 sprites */
	4,                 /* 4 bits per pixel */
	{ 0, 4, 8192*8, 8192*8+4 },     /* the two bitplanes for 4 pixels are packed into one byte */
	{ 39 * 8, 38 * 8, 37 * 8, 36 * 8, 35 * 8, 34 * 8, 33 * 8, 32 * 8,
			7 * 8, 6 * 8, 5 * 8, 4 * 8, 3 * 8, 2 * 8, 1 * 8, 0 * 8 },
	{ 0, 1, 2, 3, 8*8, 8*8+1, 8*8+2, 8*8+3, 16*8+0, 16*8+1, 16*8+2, 16*8+3,
			24*8+0, 24*8+1, 24*8+2, 24*8+3 },
	64*8    /* every sprite takes 64 bytes */
};


static struct GfxLayout digdug2_spritelayout =
{
	16,16,       /* 16*16 sprites */
	256,            /* 256 sprites */
	4,                 /* 4 bits per pixel */
	{ 0, 4, 16384*8, 16384*8+4 },   /* the two bitplanes for 4 pixels are packed into one byte */
	{ 39 * 8, 38 * 8, 37 * 8, 36 * 8, 35 * 8, 34 * 8, 33 * 8, 32 * 8,
			7 * 8, 6 * 8, 5 * 8, 4 * 8, 3 * 8, 2 * 8, 1 * 8, 0 * 8 },
	{ 0, 1, 2, 3, 8*8, 8*8+1, 8*8+2, 8*8+3, 16*8+0, 16*8+1, 16*8+2, 16*8+3,
			24*8+0, 24*8+1, 24*8+2, 24*8+3 },
	64*8    /* every sprite takes 64 bytes */
};


/* pointers to the appropriate memory locations and their associated decode structs */
static struct GfxDecodeInfo mappy_gfxdecodeinfo[] =
{
	{ 1, 0x0000, &charlayout,            0, 64 },
	{ 1, 0x1000, &mappy_spritelayout, 64*4, 16 },
	{ -1 } /* end of array */
};

static struct GfxDecodeInfo digdug2_gfxdecodeinfo[] =
{
	{ 1, 0x0000, &charlayout,              0, 64 },
	{ 1, 0x1000, &digdug2_spritelayout, 64*4, 16 },
	{ -1 } /* end of array */
};

static struct GfxDecodeInfo todruaga_gfxdecodeinfo[] =
{
	{ 1, 0x0000, &charlayout,            0, 64 },
	{ 1, 0x1000, &mappy_spritelayout, 64*4, 64 },
	{ -1 } /* end of array */
};

static struct namco_interface namco_interface =
{
	23920,	/* sample rate (approximate value) */
	8,		/* number of voices */
	48,		/* gain adjustment */
	255,	/* playback volume */
	4		/* memory region */
};



/* the machine driver: 2 6809s running at 1MHz */
static struct MachineDriver mappy_machine_driver =
{
	/* basic machine hardware */
	{
		{
			CPU_M6809,
			1100000,                        /* 1.1 Mhz */
			0,
			mappy_readmem_cpu1,writemem_cpu1,0,0,
			mappy_interrupt_1,1
		},
		{
			CPU_M6809,
			1100000,                        /* 1.1 Mhz */
			3,      /* memory region #3 */
			mappy_readmem_cpu2,writemem_cpu2,0,0,
			mappy_interrupt_2,1
		}
	},
	60, DEFAULT_60HZ_VBLANK_DURATION,	/* frames per second, vblank duration */
	100,    /* 100 CPU slices per frame - an high value to ensure proper */
			/* synchronization of the CPUs */
	mappy_init_machine,

	/* video hardware */
	28*8, 36*8, { 0*8, 28*8-1, 0*8, 36*8-1 },
	mappy_gfxdecodeinfo,
	32,64*4+16*16,
	mappy_vh_convert_color_prom,

	VIDEO_TYPE_RASTER,
	0,
	mappy_vh_start,
	mappy_vh_stop,
	mappy_vh_screenrefresh,

	/* sound hardware */
	0,0,0,0,
	{
		{
			SOUND_NAMCO,
			&namco_interface
		}
	}
};

static struct MachineDriver digdug2_machine_driver =
{
	/* basic machine hardware */
	{
		{
			CPU_M6809,
			1600000,                        /* 1.6 Mhz */
			0,
			digdug2_readmem_cpu1,writemem_cpu1,0,0,
			mappy_interrupt_1,1
		},
		{
			CPU_M6809,
			1600000,                        /* 1.6 Mhz */
			3,      /* memory region #3 */
			digdug2_readmem_cpu2,writemem_cpu2,0,0,
			mappy_interrupt_2,1
		}
	},
	60, DEFAULT_60HZ_VBLANK_DURATION,	/* frames per second, vblank duration */
	100,    /* 100 CPU slices per frame - an high value to ensure proper */
			/* synchronization of the CPUs */
	mappy_init_machine,

	/* video hardware */
	28*8, 36*8, { 0*8, 28*8-1, 0*8, 36*8-1 },
	digdug2_gfxdecodeinfo,
	32,64*4+16*16,
	mappy_vh_convert_color_prom,

	VIDEO_TYPE_RASTER,
	0,
	mappy_vh_start,
	mappy_vh_stop,
	mappy_vh_screenrefresh,

	/* sound hardware */
	0,0,0,0,
	{
		{
			SOUND_NAMCO,
			&namco_interface
		}
	}
};

static struct MachineDriver motos_machine_driver =
{
	/* basic machine hardware */
	{
		{
			CPU_M6809,
			1600000,                        /* 1.6 Mhz */
			0,
			motos_readmem_cpu1,writemem_cpu1,0,0,
			mappy_interrupt_1,1
		},
		{
			CPU_M6809,
			1600000,                        /* 1.6 Mhz */
			3,      /* memory region #3 */
			motos_readmem_cpu2,writemem_cpu2,0,0,
			mappy_interrupt_2,1
		}
	},
	60, DEFAULT_60HZ_VBLANK_DURATION,	/* frames per second, vblank duration */
	100,    /* 100 CPU slices per frame - an high value to ensure proper */
			/* synchronization of the CPUs */
	motos_init_machine,

	/* video hardware */
	28*8, 36*8, { 0*8, 28*8-1, 0*8, 36*8-1 },
	digdug2_gfxdecodeinfo,
	32,64*4+16*16,
	mappy_vh_convert_color_prom,

	VIDEO_TYPE_RASTER,
	0,
	motos_vh_start,
	mappy_vh_stop,
	mappy_vh_screenrefresh,

	/* sound hardware */
	0,0,0,0,
	{
		{
			SOUND_NAMCO,
			&namco_interface
		}
	}
};

static struct MachineDriver todruaga_machine_driver =
{
	/* basic machine hardware */
	{
		{
			CPU_M6809,
			1600000,                        /* 1.6 Mhz */
			0,
			todruaga_readmem_cpu1,writemem_cpu1,0,0,
			mappy_interrupt_1,1
		},
		{
			CPU_M6809,
			1600000,                        /* 1.6 Mhz */
			3,      /* memory region #3 */
			todruaga_readmem_cpu2,writemem_cpu2,0,0,
			mappy_interrupt_2,1
		}
	},
	60, DEFAULT_60HZ_VBLANK_DURATION,	/* frames per second, vblank duration */
	100,    /* 100 CPU slices per frame - an high value to ensure proper */
			/* synchronization of the CPUs */
	mappy_init_machine,

	/* video hardware */
	28*8, 36*8, { 0*8, 28*8-1, 0*8, 36*8-1 },
	todruaga_gfxdecodeinfo,
	32,64*4+64*16,
	mappy_vh_convert_color_prom,

	VIDEO_TYPE_RASTER,
	0,
	mappy_vh_start,
	mappy_vh_stop,
	mappy_vh_screenrefresh,

	/* sound hardware */
	0,0,0,0,
	{
		{
			SOUND_NAMCO,
			&namco_interface
		}
	}
};



/* ROM loader description */
ROM_START( mappy_rom )
	ROM_REGION(0x10000)     /* 64k for code for the first CPU  */
	ROM_LOAD( "mappy1d.64",   0xa000, 0x2000, 0x52e6c708 )
	ROM_LOAD( "mappy1c.64",   0xc000, 0x2000, 0xa958a61c )
	ROM_LOAD( "mappy1b.64",   0xe000, 0x2000, 0x203766d4 )

	ROM_REGION_DISPOSE(0x5000)      /* temporary space for graphics (disposed after conversion) */
	ROM_LOAD( "mappy3b.32",   0x0000, 0x1000, 0x16498b9f )
	ROM_LOAD( "mappy3m.64",   0x1000, 0x2000, 0xf2d9647a )
	ROM_LOAD( "mappy3n.64",   0x3000, 0x2000, 0x757cf2b6 )

	ROM_REGION(0x0220)      /* color proms */
	ROM_LOAD( "mappy.pr1",    0x0000, 0x0020, 0x56531268 ) /* palette */
	ROM_LOAD( "mappy.pr2",    0x0020, 0x0100, 0x50765082 ) /* characters */
	ROM_LOAD( "mappy.pr3",    0x0120, 0x0100, 0x5396bd78 ) /* sprites */

	ROM_REGION(0x10000)     /* 64k for the second CPU */
	ROM_LOAD( "mappy1k.64",   0xe000, 0x2000, 0x8182dd5b )

	ROM_REGION(0x0100)      /* sound prom */
	ROM_LOAD( "mappy.spr",    0x0000, 0x0100, 0x16a9166a )
ROM_END

ROM_START( mappyjp_rom )
	ROM_REGION(0x10000)     /* 64k for code for the first CPU  */
	ROM_LOAD( "mappy3.bin",   0xa000, 0x2000, 0xdb9d5ab5 )
	ROM_LOAD( "mappy1c.64",   0xc000, 0x2000, 0xa958a61c )
	ROM_LOAD( "mappy1.bin",   0xe000, 0x2000, 0x77c0b492 )

	ROM_REGION_DISPOSE(0x5000)      /* temporary space for graphics (disposed after conversion) */
	ROM_LOAD( "mappy3b.32",   0x0000, 0x1000, 0x16498b9f )
	ROM_LOAD( "mappy3m.64",   0x1000, 0x2000, 0xf2d9647a )
	ROM_LOAD( "mappy3n.64",   0x3000, 0x2000, 0x757cf2b6 )

	ROM_REGION(0x0220)      /* color proms */
	ROM_LOAD( "mappy.pr1",    0x0000, 0x0020, 0x56531268 ) /* palette */
	ROM_LOAD( "mappy.pr2",    0x0020, 0x0100, 0x50765082 ) /* characters */
	ROM_LOAD( "mappy.pr3",    0x0120, 0x0100, 0x5396bd78 ) /* sprites */

	ROM_REGION(0x10000)     /* 64k for the second CPU */
	ROM_LOAD( "mappy1k.64",   0xe000, 0x2000, 0x8182dd5b )

	ROM_REGION(0x0100)      /* sound prom */
	ROM_LOAD( "mappy.spr",    0x0000, 0x0100, 0x16a9166a )
ROM_END

ROM_START( digdug2_rom )
	ROM_REGION(0x10000)     /* 64k for code for the first CPU  */
	ROM_LOAD( "ddug2-3.bin",  0x8000, 0x4000, 0xbe7ec80b )
	ROM_LOAD( "ddug2-1.bin",  0xc000, 0x4000, 0x5c77c0d4 )

	ROM_REGION_DISPOSE(0x9000)      /* temporary space for graphics (disposed after conversion) */
	ROM_LOAD( "ddug2-3b.bin", 0x0000, 0x1000, 0xafcb4509 )
	ROM_LOAD( "ddug2-3m.bin", 0x1000, 0x4000, 0xdf1f4ad8 )
	ROM_LOAD( "ddug2-3n.bin", 0x5000, 0x4000, 0xccadb3ea )

	ROM_REGION(0x0220)      /* color proms */
	ROM_LOAD( "ddclr-5b.bin", 0x0000, 0x0020, 0x9b169db5 ) /* palette */
	ROM_LOAD( "ddclr-4c.bin", 0x0020, 0x0100, 0x55a88695 ) /* characters */
	ROM_LOAD( "ddclr-5k.bin", 0x0120, 0x0100, 0x1525a4d1 ) /* sprites */

	ROM_REGION(0x10000)     /* 64k for the second CPU */
	ROM_LOAD( "ddug2-4.bin",  0xe000, 0x2000, 0x737443b1 )

	ROM_REGION(0x0100)      /* sound prom */
	ROM_LOAD( "ddsnd.bin",    0x0000, 0x0100, 0xe0074ee2 )
ROM_END

ROM_START( motos_rom )
	ROM_REGION(0x10000)     /* 64k for code for the first CPU  */
	ROM_LOAD( "mts_1d.bin",   0x8000, 0x4000, 0x1104abb2 )
	ROM_LOAD( "mts_1b.bin",   0xc000, 0x4000, 0xb5c1c299 )

	ROM_REGION_DISPOSE(0x9000)      /* temporary space for graphics (disposed after conversion) */
	ROM_LOAD( "mts_3b.bin",   0x0000, 0x1000, 0x5d4a2a22 )
	ROM_LOAD( "mts_3m.bin",   0x1000, 0x4000, 0x2f0e396e )
	ROM_LOAD( "mts_3n.bin",   0x5000, 0x4000, 0xcf8a3b86 )

	ROM_REGION(0x0220)      /* color proms */
	ROM_LOAD( "motos.pr1",    0x0000, 0x0020, 0x71972383 ) /* palette */
	ROM_LOAD( "motos.pr2",    0x0020, 0x0100, 0x730ba7fb ) /* characters */
	ROM_LOAD( "motos.pr3",    0x0120, 0x0100, 0x7721275d ) /* sprites */

	ROM_REGION(0x10000)     /* 64k for the second CPU */
	ROM_LOAD( "mts_1k.bin",   0xe000, 0x2000, 0x55e45d21 )

	ROM_REGION(0x0100)      /* sound prom */
	ROM_LOAD( "motos.spr",    0x0000, 0x0100, 0x2accdfb4 )
ROM_END

ROM_START( todruaga_rom )
	ROM_REGION(0x10000)     /* 64k for code for the first CPU  */
	ROM_LOAD( "druaga3.bin",  0x8000, 0x4000, 0x7ab4f5b2 )
	ROM_LOAD( "druaga1.bin",  0xc000, 0x4000, 0x8c20ef10 )

	ROM_REGION_DISPOSE(0x5000)      /* temporary space for graphics (disposed after conversion) */
	ROM_LOAD( "druaga3b.bin", 0x0000, 0x1000, 0xd32b249f )
	ROM_LOAD( "druaga3m.bin", 0x1000, 0x2000, 0xe827e787 )
	ROM_LOAD( "druaga3n.bin", 0x3000, 0x2000, 0x962bd060 )

	ROM_REGION(0x0520)      /* color proms */
	ROM_LOAD( "todruaga.pr1", 0x0000, 0x0020, 0x122cc395 ) /* palette */
	ROM_LOAD( "todruaga.pr2", 0x0020, 0x0100, 0x8c661d6a ) /* characters */
	ROM_LOAD( "todruaga.pr3", 0x0120, 0x0100, 0x5bcec186 ) /* sprites */
	ROM_LOAD( "todruaga.pr4", 0x0220, 0x0100, 0xf029e5f5 ) /* sprites */
	ROM_LOAD( "todruaga.pr5", 0x0320, 0x0100, 0xecdc206c ) /* sprites */
	ROM_LOAD( "todruaga.pr6", 0x0420, 0x0100, 0x57b5ad6d ) /* sprites */

	ROM_REGION(0x10000)     /* 64k for the second CPU */
	ROM_LOAD( "druaga4.bin",  0xe000, 0x2000, 0xae9d06d9 )

	ROM_REGION(0x0100)      /* sound prom */
	ROM_LOAD( "todruaga.spr", 0x0000, 0x0100, 0x07104c40 )
ROM_END



/* load the high score table */
static int mappy_hiload(void)
{
	int writing = 0;
	void *f;
	unsigned char *RAM = Machine->memory_region[Machine->drv->cpu[0].memory_region];


	/* check if the hi score table has already been initialized */
	if (memcmp(&RAM[0x1465],"BEH",3) == 0 &&          /* check for high score initials */
	    memcmp(&RAM[0x1385],"\0\x20\0",3) == 0 &&     /* check for main high score value */
	    memcmp(&RAM[0x7ed],"\0\0\0",3) == 0)          /* see if main high score was written to screen */
	{
		if ((f = osd_fopen(Machine->gamedrv->name,0,OSD_FILETYPE_HIGHSCORE,0)) != 0)
		{
			osd_fread(f,&RAM[0x1460],40);
			osd_fclose(f);

			/* also copy over the high score */
			RAM[0x1385] = RAM[0x1460];
			RAM[0x1386] = RAM[0x1461];
			RAM[0x1387] = RAM[0x1462];
		}

		/* this is a little gross, but necessary to get the high score on-screen */
		if (!writing) writing = (RAM[0x1385] >> 4);
		mappy_videoram_w (0x7f3, writing ? (RAM[0x1385] >> 4) : ' ');
		if (!writing) writing = (RAM[0x1385] & 0x0f);
		mappy_videoram_w (0x7f2, writing ? (RAM[0x1385] & 0x0f) : ' ');
		if (!writing) writing = (RAM[0x1386] >> 4);
		mappy_videoram_w (0x7f1, writing ? (RAM[0x1386] >> 4) : ' ');
		if (!writing) writing = (RAM[0x1386] & 0x0f);
		mappy_videoram_w (0x7f0, writing ? (RAM[0x1386] & 0x0f) : ' ');
		if (!writing) writing = (RAM[0x1387] >> 4);
		mappy_videoram_w (0x7ef, writing ? (RAM[0x1387] >> 4) : ' ');
		if (!writing) writing = (RAM[0x1387] & 0x0f);
		mappy_videoram_w (0x7ee, writing ? (RAM[0x1387] & 0x0f) : ' ');
		mappy_videoram_w (0x7ed, 0);

		return 1;
	}
	else return 0; /* we can't load the hi scores yet */
}


/* save the high score table */
static void mappy_hisave(void)
{
	void *f;
	unsigned char *RAM = Machine->memory_region[Machine->drv->cpu[0].memory_region];


	if ((f = osd_fopen(Machine->gamedrv->name,0,OSD_FILETYPE_HIGHSCORE,1)) != 0)
	{
		osd_fwrite(f,&RAM[0x1460],40);
		osd_fclose(f);
	}
}


/* load the high score table */
static int digdug2_hiload(void)
{
	int writing = 0;
	void *f;
	unsigned char *RAM = Machine->memory_region[Machine->drv->cpu[0].memory_region];


	/* check if the hi score table has already been initialized */
	if (memcmp(&RAM[0x11b6]," KAZU ",6) == 0 &&         /* check for high score initials */
	    memcmp(&RAM[0x100b],"\x00\x20\x00",3) == 0 &&   /* check for main high score value */
	    memcmp(&RAM[0x7ed],"\x30\0\0",3) == 0)          /* see if main high score was written to screen */
	{
		if ((f = osd_fopen(Machine->gamedrv->name,0,OSD_FILETYPE_HIGHSCORE,0)) != 0)
		{
			osd_fread(f,&RAM[0x11b0],80);
			osd_fclose(f);

			/* also copy over the high score */
			RAM[0x100b] = (RAM[0x11b0] << 4) | RAM[0x11b1];
			RAM[0x100c] = (RAM[0x11b2] << 4) | RAM[0x11b3];
			RAM[0x100d] = (RAM[0x11b4] << 4) | RAM[0x11b5];
		}

		/* this is a little gross, but necessary to get the high score on-screen */
		if (!writing) writing = (RAM[0x11b0] & 0x0f);
		mappy_videoram_w (0x7f3, writing ? (RAM[0x11b0] & 0x0f) : ' ');
		if (!writing) writing = (RAM[0x11b1] & 0x0f);
		mappy_videoram_w (0x7f2, writing ? (RAM[0x11b1] & 0x0f) : ' ');
		if (!writing) writing = (RAM[0x11b2] & 0x0f);
		mappy_videoram_w (0x7f1, writing ? (RAM[0x11b2] & 0x0f) : ' ');
		if (!writing) writing = (RAM[0x11b3] & 0x0f);
		mappy_videoram_w (0x7f0, writing ? (RAM[0x11b3] & 0x0f) : ' ');
		if (!writing) writing = (RAM[0x11b4] & 0x0f);
		mappy_videoram_w (0x7ef, writing ? (RAM[0x11b4] & 0x0f) : ' ');
		if (!writing) writing = (RAM[0x11b5] & 0x0f);
		mappy_videoram_w (0x7ee, writing ? (RAM[0x11b5] & 0x0f) : ' ');
		mappy_videoram_w (0x7ed, 0x30);

		return 1;
	}
	else return 0; /* we can't load the hi scores yet */
}


/* save the high score table */
static void digdug2_hisave(void)
{
	void *f;
	unsigned char *RAM = Machine->memory_region[Machine->drv->cpu[0].memory_region];


	if ((f = osd_fopen(Machine->gamedrv->name,0,OSD_FILETYPE_HIGHSCORE,1)) != 0)
	{
		osd_fwrite(f,&RAM[0x11b0],80);
		osd_fclose(f);
	}
}


/* load the high score table */
static int motos_hiload(void)  /* preliminary, does not copy top high score */
{
	void *f;
	unsigned char *RAM = Machine->memory_region[Machine->drv->cpu[0].memory_region];


	/* check if the hi score table has already been initialized */
	if (memcmp(&RAM[0x2412],"\x4d\x4f\x52",3) == 0)
	{
		if ((f = osd_fopen(Machine->gamedrv->name,0,OSD_FILETYPE_HIGHSCORE,0)) != 0)
		{
			osd_fread(f,&RAM[0x2400],0xA0);
			osd_fclose(f);
		}

		return 1;
	}
	else return 0; /* we can't load the hi scores yet */
}


/* save the high score table */
static void motos_hisave(void)
{
	void *f;
	unsigned char *RAM = Machine->memory_region[Machine->drv->cpu[0].memory_region];


	if ((f = osd_fopen(Machine->gamedrv->name,0,OSD_FILETYPE_HIGHSCORE,1)) != 0)
	{
		osd_fwrite(f,&RAM[0x2400],0xA0);
		osd_fclose(f);
	}
}

static int todruaga_hiload(void)
{
	void *f;
	unsigned char *RAM = Machine->memory_region[Machine->drv->cpu[0].memory_region];


	/* check if the hi score table has already been initialized */
	if (memcmp(&RAM[0x1031],"END",3) == 0)
	{
		if ((f = osd_fopen(Machine->gamedrv->name,0,OSD_FILETYPE_HIGHSCORE,0)) != 0)
		{
			osd_fread(f,&RAM[0x102A],50);
			osd_fclose(f);
		}

		RAM[0x100b] = RAM[0x102A];
		RAM[0x100c] = RAM[0x102B];
		RAM[0x100d] = RAM[0x102C];

		return 1;
	}
	else return 0; /* we can't load the hi scores yet */
}


/* save the high score table */
static void todruaga_hisave(void)
{
	void *f;
	unsigned char *RAM = Machine->memory_region[Machine->drv->cpu[0].memory_region];


	if ((f = osd_fopen(Machine->gamedrv->name,0,OSD_FILETYPE_HIGHSCORE,1)) != 0)
	{
		osd_fwrite(f,&RAM[0x102A],50);
		osd_fclose(f);
	}
}


/* the core game driver */
struct GameDriver mappy_driver =
{
	__FILE__,
	0,
	"mappy",
	"Mappy (US)",
	"1983",
	"Namco",
	"Aaron Giles\nMirko Buffoni\nJROK",
	0,
	&mappy_machine_driver,
	0,

	mappy_rom,
	0, 0,
	0,
	0,	/* sound_prom */

	mappy_input_ports,

	PROM_MEMORY_REGION(2),0,0,
	ORIENTATION_DEFAULT,

	mappy_hiload, mappy_hisave
};

struct GameDriver mappyjp_driver =
{
	__FILE__,
	&mappy_driver,
	"mappyjp",
	"Mappy (Japan)",
	"1983",
	"Namco",
	"Aaron Giles\nMirko Buffoni\nJROK",
	0,
	&mappy_machine_driver,
	0,

	mappyjp_rom,
	0, 0,
	0,
	0,	/* sound_prom */

	mappy_input_ports,

	PROM_MEMORY_REGION(2),0,0,
	ORIENTATION_DEFAULT,

	mappy_hiload, mappy_hisave
};

struct GameDriver digdug2_driver =
{
	__FILE__,
	0,
	"digdug2",
	"Dig Dug 2",
	"1985",
	"Namco",
	"Aaron Giles\nMirko Buffoni\nJROK",
	0,
	&digdug2_machine_driver,
	0,

	digdug2_rom,
	0, 0,
	0,
	0,	/* sound_prom */

	digdug2_input_ports,

	PROM_MEMORY_REGION(2),0,0,
	ORIENTATION_DEFAULT,

	digdug2_hiload, digdug2_hisave
};

struct GameDriver motos_driver =
{
	__FILE__,
	0,
	"motos",
	"Motos",
	"1985",
	"Namco",
	"Aaron Giles\nMirko Buffoni\nJROK\nValerio Verrando",
	0,
	&motos_machine_driver,
	0,

	motos_rom,
	0, 0,
	0,
	0,	/* sound_prom */

	motos_input_ports,

	PROM_MEMORY_REGION(2),0,0,
	ORIENTATION_DEFAULT,

	motos_hiload, motos_hisave
};

struct GameDriver todruaga_driver =
{
	__FILE__,
	0,
	"todruaga",
	"Tower of Druaga",
	"1984",
	"Namco",
	"Aaron Giles\nMirko Buffoni\nJROK\nValerio Verrando",
	0,
	&todruaga_machine_driver,
	0,

	todruaga_rom,
	0, 0,
	0,
	0,	/* sound_prom */

	todruaga_input_ports,

	PROM_MEMORY_REGION(2),0,0,
	ORIENTATION_DEFAULT,

	todruaga_hiload, todruaga_hisave
};
