; file: "mix.scm", 31/03/98

(##declare
  (standard-bindings)
  (extended-bindings)
  (block)
  (fixnum)
  (not safe)
)

(define word-size (##u8vector-length '#(#f))) ; may not work in the future

;------------------------------------------------------------------------------

; This first part is the "PEVAL" benchmark (partial-evaluator)

; Utilities

(define (every? pred? l)
  (let loop ((l l))
    (or (null? l) (and (pred? (car l)) (loop (cdr l))))))

(define (some? pred? l)
  (let loop ((l l))
    (if (null? l) #f (or (pred? (car l)) (loop (cdr l))))))

(define (map2 f l1 l2)
  (let loop ((l1 l1) (l2 l2))
    (if (pair? l1)
      (cons (f (car l1) (car l2)) (loop (cdr l1) (cdr l2)))
      '())))

(define (get-last-pair l)
  (let loop ((l l))
    (let ((x (cdr l))) (if (pair? x) (loop x) l))))

; The partial evaluator.

(define (partial-evaluate proc args)
  (peval (alphatize proc '()) args))

(define (alphatize exp env) ; return a copy of 'exp' where each bound var has
  (define (alpha exp)       ; been renamed (to prevent aliasing problems)
    (cond ((const-expr? exp)
           (quot (const-value exp)))
          ((symbol? exp)
           (let ((x (assq exp env))) (if x (cdr x) exp)))
          ((or (eq? (car exp) 'if) (eq? (car exp) 'begin))
           (cons (car exp) (map alpha (cdr exp))))
          ((or (eq? (car exp) 'let) (eq? (car exp) 'letrec))
           (let ((new-env (new-variables (map car (cadr exp)) env)))
             (list (car exp)
                   (map (lambda (x)
                          (list (cdr (assq (car x) new-env))
                                (if (eq? (car exp) 'let)
                                  (alpha (cadr x))
                                  (alphatize (cadr x) new-env))))
                        (cadr exp))
                   (alphatize (caddr exp) new-env))))
          ((eq? (car exp) 'lambda)
           (let ((new-env (new-variables (cadr exp) env)))
             (list 'lambda
                   (map (lambda (x) (cdr (assq x new-env))) (cadr exp))
                   (alphatize (caddr exp) new-env))))
          (else
           (map alpha exp))))
  (alpha exp))

(define (const-expr? expr) ; is 'expr' a constant expression?
  (and (not (symbol? expr))
       (or (not (pair? expr))
           (eq? (car expr) 'quote))))

(define (const-value expr) ; return the value of a constant expression
  (if (pair? expr) ; then it must be a quoted constant
    (cadr expr)
    expr))

(define (quot val) ; make a quoted constant whose value is 'val'
  (list 'quote val))

(define (new-variables parms env)
  (append (map (lambda (x) (cons x (new-variable x))) parms) env))

(define *current-num* 0)

(define (new-variable name)
  (set! *current-num* (+ *current-num* 1))
  (string->symbol
    (string-append (symbol->string name)
                   "_"
                   (number->string *current-num*))))

; (peval proc args) will transform a procedure that is known to be called
; with constants as some of its arguments into a specialized procedure that
; is 'equivalent' but accepts only the non-constant parameters.  'proc' is the
; list representation of a lambda-expression and 'args' is a list of values,
; one for each parameter of the lambda-expression.  A special value (i.e.
; 'not-constant') is used to indicate an argument that is not a constant.
; The returned procedure is one that has as parameters the parameters of the
; original procedure which are NOT passed constants.  Constants will have been
; substituted for the constant parameters that are referenced in the body
; of the procedure.
;
; For example:
;
;   (peval
;     '(lambda (x y z) (f z x y)) ; the procedure
;     (list 1 not-constant #t))   ; the knowledge about x, y and z
;
; will return: (lambda (y) (f '#t '1 y))

(define (peval proc args)
  (simplify!
    (let ((parms (cadr proc))  ; get the parameter list
          (body (caddr proc))) ; get the body of the procedure
      (list 'lambda
            (remove-constant parms args) ; remove the constant parameters
            (beta-subst ; in the body, replace variable refs to the constant
              body      ; parameters by the corresponding constant
              (map2 (lambda (x y) (if (not-constant? y) '(()) (cons x (quot y))))
                    parms
                    args))))))

(define not-constant (list '?)) ; special value indicating non-constant parms.

(define (not-constant? x) (eq? x not-constant))

(define (remove-constant l a) ; remove from list 'l' all elements whose
  (cond ((null? l)            ; corresponding element in 'a' is a constant
         '())
        ((not-constant? (car a))
         (cons (car l) (remove-constant (cdr l) (cdr a))))
        (else
         (remove-constant (cdr l) (cdr a)))))

(define (extract-constant l a) ; extract from list 'l' all elements whose
  (cond ((null? l)             ; corresponding element in 'a' is a constant
         '())
        ((not-constant? (car a))
         (extract-constant (cdr l) (cdr a)))
        (else
         (cons (car l) (extract-constant (cdr l) (cdr a))))))

(define (beta-subst exp env) ; return a modified 'exp' where each var named in
  (define (bs exp)           ; 'env' is replaced by the corresponding expr (it
    (cond ((const-expr? exp) ; is assumed that the code has been alphatized)
           (quot (const-value exp)))
          ((symbol? exp)
           (let ((x (assq exp env))) 
             (if x (cdr x) exp)))
          ((or (eq? (car exp) 'if) (eq? (car exp) 'begin))
           (cons (car exp) (map bs (cdr exp))))
          ((or (eq? (car exp) 'let) (eq? (car exp) 'letrec))
           (list (car exp)
                 (map (lambda (x) (list (car x) (bs (cadr x)))) (cadr exp))
                 (bs (caddr exp))))
          ((eq? (car exp) 'lambda)
           (list 'lambda
                 (cadr exp)
                 (bs (caddr exp))))
          (else
           (map bs exp))))
  (bs exp))

; The expression simplifier.

(define (simplify! exp)     ; simplify the expression 'exp' destructively (it
                            ; is assumed that the code has been alphatized)
  (define (simp! where env)

    (define (s! where)
      (let ((exp (car where)))

        (cond ((const-expr? exp))  ; leave constants the way they are

              ((symbol? exp))      ; leave variable references the way they are

              ((eq? (car exp) 'if) ; dead code removal for conditionals
               (s! (cdr exp))      ; simplify the predicate
               (if (const-expr? (cadr exp)) ; is the predicate a constant?
                 (begin
                   (set-car! where
                     (if (memq (const-value (cadr exp)) '(#f ())) ; false?
                       (if (= (length exp) 3) ''() (cadddr exp))
                       (caddr exp)))
                   (s! where))
                 (for-each! s! (cddr exp)))) ; simplify consequent and alt.

              ((eq? (car exp) 'begin)
               (for-each! s! (cdr exp))
               (let loop ((exps exp)) ; remove all useless expressions
                 (if (not (null? (cddr exps))) ; not last expression?
                   (let ((x (cadr exps)))
                     (loop (if (or (const-expr? x)
                                   (symbol? x)
                                   (and (pair? x) (eq? (car x) 'lambda)))
                             (begin (set-cdr! exps (cddr exps)) exps)
                             (cdr exps))))))
               (if (null? (cddr exp)) ; only one expression in the begin?
                 (set-car! where (cadr exp))))

              ((or (eq? (car exp) 'let) (eq? (car exp) 'letrec))
               (let ((new-env (cons exp env)))
                 (define (keep i)
                   (if (>= i (length (cadar where)))
                     '()
                     (let* ((var (car (list-ref (cadar where) i)))
                            (val (cadr (assq var (cadar where))))
                            (refs (ref-count (car where) var))
                            (self-refs (ref-count val var))
                            (total-refs (- (car refs) (car self-refs)))
                            (oper-refs (- (cadr refs) (cadr self-refs))))
                       (cond ((= total-refs 0)
                              (keep (+ i 1)))
                             ((or (const-expr? val)
                                  (symbol? val)
                                  (and (pair? val)
                                       (eq? (car val) 'lambda)
                                       (= total-refs 1)
                                       (= oper-refs 1)
                                       (= (car self-refs) 0))
                                  (and (caddr refs)
                                       (= total-refs 1)))
                              (set-car! where
                                (beta-subst (car where)
                                            (list (cons var val))))
                              (keep (+ i 1)))
                             (else
                              (cons var (keep (+ i 1))))))))
                 (simp! (cddr exp) new-env)
                 (for-each! (lambda (x) (simp! (cdar x) new-env)) (cadr exp))
                 (let ((to-keep (keep 0)))
                   (if (< (length to-keep) (length (cadar where)))
                     (begin
                       (if (null? to-keep)
                         (set-car! where (caddar where))
                         (set-car! (cdar where)
                           (map (lambda (v) (assq v (cadar where))) to-keep)))
                       (s! where))
                     (if (null? to-keep)
                       (set-car! where (caddar where)))))))

              ((eq? (car exp) 'lambda)
               (simp! (cddr exp) (cons exp env)))

              (else
               (for-each! s! exp)
               (cond ((symbol? (car exp)) ; is the operator position a var ref?
                      (let ((frame (binding-frame (car exp) env)))
                        (if frame ; is it a bound variable?
                          (let ((proc (bound-expr (car exp) frame)))
                            (if (and (pair? proc)
                                     (eq? (car proc) 'lambda)
                                     (some? const-expr? (cdr exp)))
                              (let* ((args (arg-pattern (cdr exp)))
                                     (new-proc (peval proc args))
                                     (new-args (remove-constant (cdr exp) args)))
                                (set-car! where
                                  (cons (add-binding new-proc frame (car exp))
                                        new-args)))))
                          (set-car! where
                            (constant-fold-global (car exp) (cdr exp))))))
                     ((not (pair? (car exp))))
                     ((eq? (caar exp) 'lambda)
                      (set-car! where
                        (list 'let
                              (map2 list (cadar exp) (cdr exp))
                              (caddar exp)))
                      (s! where)))))))

    (s! where))

  (define (remove-empty-calls! where env)

    (define (rec! where)
      (let ((exp (car where)))

        (cond ((const-expr? exp))
              ((symbol? exp))
              ((eq? (car exp) 'if)
               (rec! (cdr exp))
               (rec! (cddr exp))
               (rec! (cdddr exp)))
              ((eq? (car exp) 'begin)
               (for-each! rec! (cdr exp)))
              ((or (eq? (car exp) 'let) (eq? (car exp) 'letrec))
               (let ((new-env (cons exp env)))
                 (remove-empty-calls! (cddr exp) new-env)
                 (for-each! (lambda (x) (remove-empty-calls! (cdar x) new-env))
                            (cadr exp))))
              ((eq? (car exp) 'lambda)
               (rec! (cddr exp)))
              (else
               (for-each! rec! (cdr exp))
               (if (and (null? (cdr exp)) (symbol? (car exp)))
                 (let ((frame (binding-frame (car exp) env)))
                   (if frame ; is it a bound variable?
                     (let ((proc (bound-expr (car exp) frame)))
                       (if (and (pair? proc)
                                (eq? (car proc) 'lambda))
                         (begin
                           (set! changed? #t)
                           (set-car! where (caddr proc))))))))))))

    (rec! where))

  (define changed? #f)

  (let ((x (list exp)))
    (let loop ()
      (set! changed? #f)
      (simp! x '())
      (remove-empty-calls! x '())
      (if changed? (loop) (car x)))))

(define (ref-count exp var) ; compute how many references to variable 'var'
  (let ((total 0)           ; are contained in 'exp'
        (oper 0)
        (always-evaled #t))
    (define (rc exp ae)
      (cond ((const-expr? exp))
            ((symbol? exp)
             (if (eq? exp var)
               (begin
                 (set! total (+ total 1))
                 (set! always-evaled (and ae always-evaled)))))
            ((eq? (car exp) 'if)
             (rc (cadr exp) ae)
             (for-each (lambda (x) (rc x #f)) (cddr exp)))
            ((eq? (car exp) 'begin)
             (for-each (lambda (x) (rc x ae)) (cdr exp)))
            ((or (eq? (car exp) 'let) (eq? (car exp) 'letrec))
             (for-each (lambda (x) (rc (cadr x) ae)) (cadr exp))
             (rc (caddr exp) ae))
            ((eq? (car exp) 'lambda)
             (rc (caddr exp) #f))
            (else
             (for-each (lambda (x) (rc x ae)) exp)
             (if (symbol? (car exp))
               (if (eq? (car exp) var) (set! oper (+ oper 1)))))))
    (rc exp #t)
    (list total oper always-evaled)))

(define (binding-frame var env)
  (cond ((null? env) #f)
        ((or (eq? (caar env) 'let) (eq? (caar env) 'letrec))
         (if (assq var (cadar env)) (car env) (binding-frame var (cdr env))))
        ((eq? (caar env) 'lambda)
         (if (memq var (cadar env)) (car env) (binding-frame var (cdr env))))
        (else
         (error "ill-formed environment"))))

(define (bound-expr var frame)
  (cond ((or (eq? (car frame) 'let) (eq? (car frame) 'letrec))
         (cadr (assq var (cadr frame))))
        ((eq? (car frame) 'lambda)
         not-constant)
        (else
         (error "ill-formed frame"))))

(define (add-binding val frame name)
  (define (find-val val bindings)
    (cond ((null? bindings) #f)
          ((equal? val (cadar bindings)) ; *kludge* equal? is not exactly what
           (caar bindings))              ; we want...
          (else
           (find-val val (cdr bindings)))))
  (or (find-val val (cadr frame))
      (let ((var (new-variable name)))
        (set-cdr! (get-last-pair (cadr frame)) (list (list var val)))
        var)))

(define (for-each! proc! l) ; call proc! on each CONS CELL in the list 'l'
  (if (not (null? l))
    (begin (proc! l) (for-each! proc! (cdr l)))))

(define (arg-pattern exps) ; return the argument pattern (i.e. the list of
  (if (null? exps)         ; constants in 'exps' but with the not-constant
    '()                    ; value wherever the corresponding expression in
    (cons (if (const-expr? (car exps)) ; 'exps' is not a constant)
            (const-value (car exps))
            not-constant)
          (arg-pattern (cdr exps)))))

; Knowledge about primitive procedures.

(define *primitives*
  (list
    (cons 'car (lambda (args)
                 (and (= (length args) 1)
                      (pair? (car args))
                      (quot (car (car args))))))
    (cons 'cdr (lambda (args)
                 (and (= (length args) 1)
                      (pair? (car args))
                      (quot (cdr (car args))))))
    (cons '+ (lambda (args)
               (and (every? number? args)
                    (quot (sum args 0)))))
    (cons '* (lambda (args)
               (and (every? number? args)
                    (quot (product args 1)))))
    (cons '- (lambda (args)
               (and (> (length args) 0)
                    (every? number? args)
                    (quot (if (null? (cdr args))
                            (- (car args))
                            (- (car args) (sum (cdr args) 0)))))))
    (cons '/ (lambda (args)
               (and (> (length args) 1)
                    (every? number? args)
                    (quot (if (null? (cdr args))
                            (/ (car args))
                            (/ (car args) (product (cdr args) 1)))))))
    (cons '< (lambda (args)
               (and (= (length args) 2)
                    (every? number? args)
                    (quot (< (car args) (cadr args))))))
    (cons '= (lambda (args)
               (and (= (length args) 2)
                    (every? number? args)
                    (quot (= (car args) (cadr args))))))
    (cons '> (lambda (args)
               (and (= (length args) 2)
                    (every? number? args)
                    (quot (> (car args) (cadr args))))))
    (cons 'eq? (lambda (args)
                 (and (= (length args) 2)
                      (quot (eq? (car args) (cadr args))))))
    (cons 'not (lambda (args)
                 (and (= (length args) 1)
                      (quot (not (car args))))))
    (cons 'null? (lambda (args)
                   (and (= (length args) 1)
                        (quot (null? (car args))))))
    (cons 'pair? (lambda (args)
                   (and (= (length args) 1)
                        (quot (pair? (car args))))))
    (cons 'symbol? (lambda (args)
                     (and (= (length args) 1)
                          (quot (symbol? (car args))))))
  )
)

(define (sum lst n)
  (if (null? lst)
    n
    (sum (cdr lst) (+ n (car lst)))))

(define (product lst n)
  (if (null? lst)
    n
    (product (cdr lst) (* n (car lst)))))

(define (reduce-global name args)
  (let ((x (assq name *primitives*)))
    (and x ((cdr x) args))))

(define (constant-fold-global name exprs)

  (define (flatten args op)
    (cond ((null? args)
           '())
          ((and (pair? (car args)) (eq? (caar args) op))
           (append (flatten (cdar args) op) (flatten (cdr args) op)))
          (else
           (cons (car args) (flatten (cdr args) op)))))

  (let ((args (if (or (eq? name '+) (eq? name '*)) ; associative ops
                (flatten exprs name)
                exprs)))
    (or (and (every? const-expr? args)
             (reduce-global name (map const-value args)))
        (let ((pattern (arg-pattern args)))
          (let ((non-const (remove-constant args pattern))
                (const (map const-value (extract-constant args pattern))))
            (cond ((eq? name '+) ; + is commutative
                   (let ((x (reduce-global '+ const)))
                     (if x
                       (let ((y (const-value x)))
                         (cons '+
                               (if (= y 0) non-const (cons x non-const))))
                       (cons name args))))
                  ((eq? name '*) ; * is commutative
                   (let ((x (reduce-global '* const)))
                     (if x
                       (let ((y (const-value x)))
                         (cons '*
                               (if (= y 1) non-const (cons x non-const))))
                       (cons name args))))
                  ((eq? name 'cons)
                   (cond ((and (const-expr? (cadr args))
                               (null? (const-value (cadr args))))
                          (list 'list (car args)))
                         ((and (pair? (cadr args))
                               (eq? (car (cadr args)) 'list))
                          (cons 'list (cons (car args) (cdr (cadr args)))))
                         (else
                          (cons name args))))
                  (else
                   (cons name args))))))))

; Examples:

(define (try-peval proc args)
  (partial-evaluate proc args))

(define example1
  '(lambda (a b c)
     (if (null? a) b (+ (car a) c))))

;(try-peval example1 (list '(10 11) not-constant '1))

(define example2
  '(lambda (x y)
     (let ((q (lambda (a b) (if (< a 0) b (- 10 b)))))
       (if (< x 0) (q (- y) (- x)) (q y x)))))

;(try-peval example2 (list not-constant '1))

(define example3
  '(lambda (l n)
     (letrec ((add-list
               (lambda (l n)
                 (if (null? l)
                   '()
                   (cons (+ (car l) n) (add-list (cdr l) n))))))
       (add-list l n))))

;(try-peval example3 (list not-constant '1))

;(try-peval example3 (list '(1 2 3) not-constant))

(define example4
  '(lambda (exp env)
     (letrec ((eval
               (lambda (exp env)
                 (letrec ((eval-list
                            (lambda (l env)
                              (if (null? l)
                                '()
                                (cons (eval (car l) env)
                                      (eval-list (cdr l) env))))))
                   (if (symbol? exp) (lookup exp env)
                     (if (not (pair? exp)) exp
                       (if (eq? (car exp) 'quote) (car (cdr exp))
                         (apply (eval (car exp) env)
                                (eval-list (cdr exp) env)))))))))
       (eval exp env))))

;(try-peval example4 (list 'x not-constant))

;(try-peval example4 (list '(f 1 2 3) not-constant))

(define example5
  '(lambda (a b)
     (letrec ((funct
               (lambda (x)
                 (+ x b (if (< x 1) 0 (funct (- x 1)))))))
       (funct a))))

;(try-peval example5 (list '5 not-constant))

(define example6
  '(lambda ()
     (letrec ((fib
               (lambda (x)
                 (if (< x 2) x (+ (fib (- x 1)) (fib (- x 2)))))))
       (fib 10))))

;(try-peval example6 '())

(define example7
  '(lambda (input)
     (letrec ((copy (lambda (in)
                      (if (pair? in)
                        (cons (copy (car in))
                              (copy (cdr in)))
                        in))))
       (copy input))))

;(try-peval example7 (list '(a b c d e f g h i j k l m n o p q r s t u v w x y z)))

(define example8
  '(lambda (input)
     (letrec ((reverse (lambda (in result)
                         (if (pair? in)
                           (reverse (cdr in) (cons (car in) result))
                           result))))
       (reverse input '()))))

;(try-peval example8 (list '(a b c d e f g h i j k l m n o p q r s t u v w x y z)))

(define (test1)
  (set! *current-num* 0)
  (pp (try-peval example1 (list '(10 11) not-constant '1)))
  (pp (try-peval example2 (list not-constant '1)))
  (pp (try-peval example3 (list not-constant '1)))
  (pp (try-peval example3 (list '(1 2 3) not-constant)))
  (pp (try-peval example4 (list 'x not-constant)))
  (pp (try-peval example4 (list '(f 1 2 3) not-constant)))
  (pp (try-peval example5 (list '5 not-constant)))
  (pp (try-peval example6 '()))
  (pp (try-peval
       example7
       (list '(a b c d e f g h i j k l m n o p q r s t u v w x y z))))
  (pp (try-peval
       example8
       (list '(a b c d e f g h i j k l m n o p q r s t u v w x y z)))))

;------------------------------------------------------------------------------

; This second part tests keyword and optional parameters

(define (show msg proc)
  (newline);  (display msg) (pp proc)
)

(define (try thunk)
  (call-with-current-continuation
    (lambda (cont)
      (##catch-all (lambda (s args)
                     (display "-----> ")
                     (write s)
                     (show " on " thunk)
                     (cont 0))
                   (lambda ()
                     (write (thunk))
                     (show " on " thunk))))))

(##define-macro (test form expect)
  `(test-form (lambda () ,form) ',expect))

(define (test-form thunk expect)
  (##gc)
  (if (equal? (thunk) expect)
    (write 'ok)
    (write (list 'expected expect)))
  (newline))

(##define-macro (err form)
  `(let ()
     (##declare (safe) (generic))
     (err-form (lambda () ,form))))

(define (err-form thunk)
  (test-form
    (lambda ()
      (call-with-current-continuation
        (lambda (abort)
          (##catch-all
            (lambda lst
              (write lst)
              (newline)
              (abort 'error))
            thunk))))
    'error))

(define (f1) 'ok)
(define (f2 a) (list a))
(define (f3 . a) (list a))
(define (f4 a . b) (list a b))

(define (f5 a #!optional) (list a))
(define (f6 a #!optional b) (list a b))
(define (f7 a #!optional (b (list a b))) (list a b))

(define (f8 a #!rest b) (list a b))

(define (f9 a #!key) (list a))
(define (f10 a #!key b) (list a b))
(define (f11 a #!key (b (list a b))) (list a b))

(define (f12 a #!optional #!rest b) (list a b))
(define (f13 a #!optional b #!rest c) (list a b c))

(define (f14 a #!optional #!key) (list a))
(define (f15 a #!optional #!key b) (list a b))
(define (f16 a #!optional #!key (b (list a b))) (list a b))
(define (f17 a #!optional b #!key) (list a b))
(define (f18 a #!optional b #!key c) (list a b c))
(define (f19 a #!optional (b (list a b c)) #!key (c (list a b c))) (list a b c))

(define (f20 a #!rest b #!key) (list a b))
(define (f21 a #!rest b #!key c) (list a b c))
(define (f22 a #!rest b #!key (c (list a b c))) (list a b c))

(define (f23 a #!optional #!rest b #!key) (list a b))
(define (f24 a #!optional #!rest b #!key c) (list a b c))
(define (f25 a #!optional #!rest b #!key (c (list a b c))) (list a b c))
(define (f26 a #!optional b #!rest c #!key) (list a b c))
(define (f27 a #!optional (b (list a b c)) #!rest c #!key) (list a b c))
(define (f28 a #!optional b #!rest c #!key d) (list a b c d))
(define (f29 a #!optional (b (list a b c d)) #!rest c #!key (d (list a b c d))) (list a b c d))

(define a #f)
(define b #f)
(define c #f)
(define d #f)

(define (start f)
  (newline)
  (show "" f))

(define (run-f1)
  (start f1)
  (try (lambda () (f1)))
  (try (lambda () (f1 1)))
  (try (lambda () (f1 1 2)))
  (try (lambda () (f1 1 2 3)))
  (try (lambda () (f1 1 2 3 4)))
  (try (lambda () (f1 x: 1 y: 2))))

(define (run-f2)
  (start f2)
  (try (lambda () (f2)))
  (try (lambda () (f2 1)))
  (try (lambda () (f2 1 2)))
  (try (lambda () (f2 1 2 3)))
  (try (lambda () (f2 1 2 3 4)))
  (try (lambda () (f2 1 x: 2 y: 3))))

(define (run-f3)
  (start f3)
  (try (lambda () (f3)))
  (try (lambda () (f3 1)))
  (try (lambda () (f3 1 2)))
  (try (lambda () (f3 1 2 3)))
  (try (lambda () (f3 1 2 3 4)))
  (try (lambda () (f3 x: 1 y: 2))))

(define (run-f4)
  (start f4)
  (try (lambda () (f4)))
  (try (lambda () (f4 1)))
  (try (lambda () (f4 1 2)))
  (try (lambda () (f4 1 2 3)))
  (try (lambda () (f4 1 2 3 4)))
  (try (lambda () (f4 1 x: 2 y: 3))))

(define (run-f5)
  (start f5)
  (try (lambda () (f5)))
  (try (lambda () (f5 1)))
  (try (lambda () (f5 1 2)))
  (try (lambda () (f5 1 2 3)))
  (try (lambda () (f5 1 2 3 4)))
  (try (lambda () (f5 1 x: 2 y: 3))))

(define (run-f6)
  (start f6)
  (try (lambda () (f6)))
  (try (lambda () (f6 1)))
  (try (lambda () (f6 1 2)))
  (try (lambda () (f6 1 2 3)))
  (try (lambda () (f6 1 2 3 4)))
  (try (lambda () (f6 1 2 x: 3 y: 4))))

(define (run-f7)
  (start f7)
  (try (lambda () (f7)))
  (try (lambda () (f7 1)))
  (try (lambda () (f7 1 2)))
  (try (lambda () (f7 1 2 3)))
  (try (lambda () (f7 1 2 3 4)))
  (try (lambda () (f7 1 2 x: 3 y: 4))))

(define (run-f8)
  (start f8)
  (try (lambda () (f8)))
  (try (lambda () (f8 1)))
  (try (lambda () (f8 1 2)))
  (try (lambda () (f8 1 2 3)))
  (try (lambda () (f8 1 2 3 4)))
  (try (lambda () (f8 1 x: 2 y: 3))))

(define (run-f9)
  (start f9)
  (try (lambda () (f9)))
  (try (lambda () (f9 1)))
  (try (lambda () (f9 1 2)))
  (try (lambda () (f9 1 2 3)))
  (try (lambda () (f9 1 x: 2 y: 3))))

(define (run-f10)
  (start f10)
  (try (lambda () (f10)))
  (try (lambda () (f10 1)))
  (try (lambda () (f10 1 2)))
  (try (lambda () (f10 1 2 3)))
  (try (lambda () (f10 1 b: 2)))
  (try (lambda () (f10 1 x: 2)))
  (try (lambda () (f10 1 x: 2 b: 3))))

(define (run-f11)
  (start f11)
  (try (lambda () (f11)))
  (try (lambda () (f11 1)))
  (try (lambda () (f11 1 2)))
  (try (lambda () (f11 1 2 3)))
  (try (lambda () (f11 1 b: 2)))
  (try (lambda () (f11 1 x: 2)))
  (try (lambda () (f11 1 x: 2 b: 3))))

(define (run-f12)
  (start f12)
  (try (lambda () (f12)))
  (try (lambda () (f12 1)))
  (try (lambda () (f12 1 2)))
  (try (lambda () (f12 1 2 3)))
  (try (lambda () (f12 1 x: 2 y: 3))))

(define (run-f13)
  (start f13)
  (try (lambda () (f13)))
  (try (lambda () (f13 1)))
  (try (lambda () (f13 1 2)))
  (try (lambda () (f13 1 2 3)))
  (try (lambda () (f13 1 2 3 4)))
  (try (lambda () (f13 1 2 x: 3 y: 4))))

(define (run-f14)
  (start f14)
  (try (lambda () (f14)))
  (try (lambda () (f14 1)))
  (try (lambda () (f14 1 2)))
  (try (lambda () (f14 1 2 3)))
  (try (lambda () (f14 1 x: 2)))
  (try (lambda () (f14 1 x: 2 y: 3))))

(define (run-f15)
  (start f15)
  (try (lambda () (f15)))
  (try (lambda () (f15 1)))
  (try (lambda () (f15 1 2)))
  (try (lambda () (f15 1 2 3)))
  (try (lambda () (f15 1 b: 2)))
  (try (lambda () (f15 1 x: 2)))
  (try (lambda () (f15 1 x: 2 b: 3))))

(define (run-f16)
  (start f16)
  (try (lambda () (f16)))
  (try (lambda () (f16 1)))
  (try (lambda () (f16 1 2)))
  (try (lambda () (f16 1 2 3)))
  (try (lambda () (f16 1 b: 2)))
  (try (lambda () (f16 1 x: 2)))
  (try (lambda () (f16 1 x: 2 b: 3))))

(define (run-f17)
  (start f17)
  (try (lambda () (f17)))
  (try (lambda () (f17 1)))
  (try (lambda () (f17 1 2)))
  (try (lambda () (f17 1 2 3)))
  (try (lambda () (f17 1 2 3 4)))
  (try (lambda () (f17 1 2 x: 3)))
  (try (lambda () (f17 1 2 x: 3 y: 4))))

(define (run-f18)
  (start f18)
  (try (lambda () (f18)))
  (try (lambda () (f18 1)))
  (try (lambda () (f18 1 2)))
  (try (lambda () (f18 1 2 3)))
  (try (lambda () (f18 1 2 3 4)))
  (try (lambda () (f18 1 2 c: 3)))
  (try (lambda () (f18 1 2 x: 3)))
  (try (lambda () (f18 1 2 x: 3 c: 4))))

(define (run-f19)
  (start f19)
  (try (lambda () (f19)))
  (try (lambda () (f19 1)))
  (try (lambda () (f19 1 2)))
  (try (lambda () (f19 1 2 3)))
  (try (lambda () (f19 1 2 3 4)))
  (try (lambda () (f19 1 2 c: 3)))
  (try (lambda () (f19 1 2 x: 3)))
  (try (lambda () (f19 1 2 x: 3 c: 4))))

(define (run-f20)
  (start f20)
  (try (lambda () (f20)))
  (try (lambda () (f20 1)))
  (try (lambda () (f20 1 2)))
  (try (lambda () (f20 1 2 3)))
  (try (lambda () (f20 1 x: 2 y: 3))))

(define (run-f21)
  (start f21)
  (try (lambda () (f21)))
  (try (lambda () (f21 1)))
  (try (lambda () (f21 1 2)))
  (try (lambda () (f21 1 2 3)))
  (try (lambda () (f21 1 c: 2)))
  (try (lambda () (f21 1 x: 2)))
  (try (lambda () (f21 1 x: 2 c: 3))))

(define (run-f22)
  (start f22)
  (try (lambda () (f22)))
  (try (lambda () (f22 1)))
  (try (lambda () (f22 1 2)))
  (try (lambda () (f22 1 2 3)))
  (try (lambda () (f22 1 c: 2)))
  (try (lambda () (f22 1 x: 2)))
  (try (lambda () (f22 1 x: 2 c: 3))))

(define (run-f23)
  (start f23)
  (try (lambda () (f23)))
  (try (lambda () (f23 1)))
  (try (lambda () (f23 1 2)))
  (try (lambda () (f23 1 2 3)))
  (try (lambda () (f23 1 x: 2 y: 3))))

(define (run-f24)
  (start f24)
  (try (lambda () (f24)))
  (try (lambda () (f24 1)))
  (try (lambda () (f24 1 2)))
  (try (lambda () (f24 1 2 3)))
  (try (lambda () (f24 1 c: 2)))
  (try (lambda () (f24 1 x: 2)))
  (try (lambda () (f24 1 x: 2 c: 3))))

(define (run-f25)
  (start f25)
  (try (lambda () (f25)))
  (try (lambda () (f25 1)))
  (try (lambda () (f25 1 2)))
  (try (lambda () (f25 1 2 3)))
  (try (lambda () (f25 1 c: 2)))
  (try (lambda () (f25 1 x: 2)))
  (try (lambda () (f25 1 x: 2 c: 3))))

(define (run-f26)
  (start f26)
  (try (lambda () (f26)))
  (try (lambda () (f26 1)))
  (try (lambda () (f26 1 2)))
  (try (lambda () (f26 1 2 3)))
  (try (lambda () (f26 1 2 3 4)))
  (try (lambda () (f26 1 2 x: 3 y: 4))))

(define (run-f27)
  (start f27)
  (try (lambda () (f27)))
  (try (lambda () (f27 1)))
  (try (lambda () (f27 1 2)))
  (try (lambda () (f27 1 2 3)))
  (try (lambda () (f27 1 2 3 4)))
  (try (lambda () (f27 1 2 c: 3)))
  (try (lambda () (f27 1 2 x: 3)))
  (try (lambda () (f27 1 2 x: 3 c: 4))))

(define (run-f28)
  (start f28)
  (try (lambda () (f28)))
  (try (lambda () (f28 1)))
  (try (lambda () (f28 1 2)))
  (try (lambda () (f28 1 2 3)))
  (try (lambda () (f28 1 2 3 4)))
  (try (lambda () (f28 1 2 c: 3)))
  (try (lambda () (f28 1 2 x: 3)))
  (try (lambda () (f28 1 2 x: 3 c: 4))))

(define (run-f29)
  (start f29)
  (try (lambda () (f29)))
  (try (lambda () (f29 1)))
  (try (lambda () (f29 1 2)))
  (try (lambda () (f29 1 2 3)))
  (try (lambda () (f29 1 2 3 4)))
  (try (lambda () (f29 1 2 c: 3)))
  (try (lambda () (f29 1 2 x: 3)))
  (try (lambda () (f29 1 2 x: 3 c: 4))))

(define (test2)

  (set! a 6)
  (set! b 7)
  (set! c 8)
  (set! d 9)

  (run-f1)
  (run-f2)
  (run-f3)
  (run-f4)
  (run-f5)
  (run-f6)
  (run-f7)
  (run-f8)
  (run-f9)
  (run-f10)
  (run-f11)
  (run-f12)
  (run-f13)
  (run-f14)
  (run-f15)
  (run-f16)
  (run-f17)
  (run-f18)
  (run-f19)
  (run-f20)
  (run-f21)
  (run-f22)
  (run-f23)
  (run-f24)
  (run-f25)
  (run-f26)
  (run-f27)
  (run-f28)
  (run-f29))

;------------------------------------------------------------------------------

; This third part tests primitives on vector-like objects

(define (run-strings)
  (test (string? "5678") #t)
  (test (string? 123456789012345678) #f)
  (test (make-string 0) "")
  (test (make-string 3) "   ")
  (test (make-string 5 #\6) "66666")
  (test (string-length (make-string 8388607)) 8388607)
  (test (string) "")
  (test (string #\5 #\6) "56")
  (test (string-length "5678") 4)
  (test (string-ref "5678" 3) #\8)
  (test (let ((x (string #\5 #\6))) (string-set! x 1 #\3) x) "53")
  (test (string->list "56") (#\5 #\6))
  (test (list->string '(#\5 #\6)) "56")
  (err (make-string -1))
  (err (make-string 8388608))
  (err (make-string 536870911))
  (err (make-string 536870912))
  (err (make-string 1.5))
  (err (make-string 1 'a))
  (err (string #\5 'b))
  (err (string-length 123456789012345678))
  (err (string-ref 123456789012345678 0))
  (err (string-ref "56" -1))
  (err (string-ref "56" 2))
  (err (string-ref "56" 536870912))
  (err (string-set! 123456789012345678 0 #\3))
  (err (let ((x (string #\5 #\6))) (string-set! x -1 #\3) x))
  (err (let ((x (string #\5 #\6))) (string-set! x 2 #\3) x))
  (err (let ((x (string #\5 #\6))) (string-set! x 536870912 #\3) x))
  (err (let ((x (string #\5 #\6))) (string-set! x 1 'a) x))
  (err (string->list 123456789012345678))
  (err (list->string 123456789012345678))
  (err (list->string '(#\5 b))))

(define (run-vectors)
  (test (vector? '#(5 6 7 8)) #t)
  (test (vector? 123456789012345678) #f)
  (test (make-vector 0) #())
  (test (make-vector 3) #(#f #f #f))
  (test (make-vector 5 'a) #(a a a a a))
  (let ((n (quotient 16777215 word-size))) (test (= n (vector-length (make-vector n))) #t))
  (test (vector) #())
  (test (vector 5 'b) #(5 b))
  (test (vector-length '#(5 6 7 8)) 4)
  (test (vector-ref '#(5 6 7 8) 3) 8)
  (test (let ((x (vector 5 6))) (vector-set! x 1 3) x) #(5 3))
  (test (vector->list '#(5 6)) (5 6))
  (test (list->vector '(5 b)) #(5 b))
  (err (make-vector -1))
  (let ((n (quotient 16777216 word-size))) (err (make-vector n)))
  (err (make-vector 536870911))
  (err (make-vector 536870912))
  (err (make-vector 1.5))
  (err (vector-length 123456789012345678))
  (err (vector-ref 123456789012345678 0))
  (err (vector-ref '#(5 6) -1))
  (err (vector-ref '#(5 6) 2))
  (err (vector-ref '#(5 6) 536870912))
  (err (vector-set! 123456789012345678 0 3))
  (err (let ((x (vector 5 6))) (vector-set! x -1 3) x))
  (err (let ((x (vector 5 6))) (vector-set! x 2 3) x))
  (err (let ((x (vector 5 6))) (vector-set! x 536870912 3) x))
  (err (vector->list 123456789012345678))
  (err (list->vector 123456789012345678)))

(define (run-u8vectors)
  (test (u8vector? '#u8(5 6 7 8)) #t)
  (test (u8vector? 123456789012345678) #f)
  (test (make-u8vector 0) #u8())
  (test (make-u8vector 3) #u8(0 0 0))
  (test (make-u8vector 5 6) #u8(6 6 6 6 6))
  (test (u8vector-length (make-u8vector 16777215)) 16777215)
  (test (u8vector) #u8())
  (test (u8vector 5 -1 -128 255) #u8(5 255 128 255))
  (test (u8vector-length '#u8(5 6 7 8)) 4)
  (test (u8vector-ref '#u8(5 6 7 8) 3) 8)
  (test (let ((x (u8vector 5 6))) (u8vector-set! x 1 3) x) #u8(5 3))
  (test (let ((x (u8vector 5 6))) (u8vector-set! x 1 -128) x) #u8(5 128))
  (test (let ((x (u8vector 5 6))) (u8vector-set! x 1 255) x) #u8(5 255))
  (test (u8vector->list '#u8(5 6)) (5 6))
  (test (list->u8vector '(5 -1 -128 255)) #u8(5 255 128 255))
  (err (make-u8vector -1))
  (err (make-u8vector 16777216))
  (err (make-u8vector 536870911))
  (err (make-u8vector 536870912))
  (err (make-u8vector 1.5))
  (err (make-u8vector 1 'a))
  (err (u8vector 5 'b))
  (err (u8vector-length 123456789012345678))
  (err (u8vector-ref 123456789012345678 0))
  (err (u8vector-ref '#u8(5 6) -1))
  (err (u8vector-ref '#u8(5 6) 2))
  (err (u8vector-ref '#u8(5 6) 536870912))
  (err (u8vector-set! 123456789012345678 0 3))
  (err (let ((x (u8vector 5 6))) (u8vector-set! x -1 3) x))
  (err (let ((x (u8vector 5 6))) (u8vector-set! x 2 3) x))
  (err (let ((x (u8vector 5 6))) (u8vector-set! x 536870912 3) x))
  (err (let ((x (u8vector 5 6))) (u8vector-set! x 1 -129) x))
  (err (let ((x (u8vector 5 6))) (u8vector-set! x 1 256) x))
  (err (let ((x (u8vector 5 6))) (u8vector-set! x 1 536870912) x))
  (err (let ((x (u8vector 5 6))) (u8vector-set! x 1 'a) x))
  (err (u8vector->list 123456789012345678))
  (err (list->u8vector 123456789012345678))
  (err (list->u8vector '(5 b)))
  (err (list->u8vector '(256))))

(define (run-u16vectors)
  (test (u16vector? '#u16(5 6 7 8)) #t)
  (test (u16vector? 123456789012345678) #f)
  (test (make-u16vector 0) #u16())
  (test (make-u16vector 3) #u16(0 0 0))
  (test (make-u16vector 5 6) #u16(6 6 6 6 6))
  (test (u16vector-length (make-u16vector 8388607)) 8388607)
  (test (u16vector) #u16())
  (test (u16vector 5 -1 -32768 65535) #u16(5 65535 32768 65535))
  (test (u16vector-length '#u16(5 6 7 8)) 4)
  (test (u16vector-ref '#u16(5 6 7 8) 3) 8)
  (test (let ((x (u16vector 5 6))) (u16vector-set! x 1 3) x) #u16(5 3))
  (test (let ((x (u16vector 5 6))) (u16vector-set! x 1 -32768) x) #u16(5 32768))
  (test (let ((x (u16vector 5 6))) (u16vector-set! x 1 65535) x) #u16(5 65535))
  (test (u16vector->list '#u16(5 6)) (5 6))
  (test (list->u16vector '(5 -1 -32768 65535)) #u16(5 65535 32768 65535))
  (err (make-u16vector -1))
  (err (make-u16vector 8388608))
  (err (make-u16vector 536870911))
  (err (make-u16vector 536870912))
  (err (make-u16vector 1.5))
  (err (make-u16vector 1 'a))
  (err (u16vector 5 'b))
  (err (u16vector-length 123456789012345678))
  (err (u16vector-ref 123456789012345678 0))
  (err (u16vector-ref '#u16(5 6) -1))
  (err (u16vector-ref '#u16(5 6) 2))
  (err (u16vector-ref '#u16(5 6) 536870912))
  (err (u16vector-set! 123456789012345678 0 3))
  (err (let ((x (u16vector 5 6))) (u16vector-set! x -1 3) x))
  (err (let ((x (u16vector 5 6))) (u16vector-set! x 2 3) x))
  (err (let ((x (u16vector 5 6))) (u16vector-set! x 536870912 3) x))
  (err (let ((x (u16vector 5 6))) (u16vector-set! x 1 -32769) x))
  (err (let ((x (u16vector 5 6))) (u16vector-set! x 1 65536) x))
  (err (let ((x (u16vector 5 6))) (u16vector-set! x 1 536870912) x))
  (err (let ((x (u16vector 5 6))) (u16vector-set! x 1 'a) x))
  (err (u16vector->list 123456789012345678))
  (err (list->u16vector 123456789012345678))
  (err (list->u16vector '(5 b)))
  (err (list->u16vector '(65536))))

(define (run-u32vectors)
  (test (u32vector? '#u32(5 6 7 8)) #t)
  (test (u32vector? 123456789012345678) #f)
  (test (make-u32vector 0) #u32())
  (test (make-u32vector 3) #u32(0 0 0))
  (test (make-u32vector 5 6) #u32(6 6 6 6 6))
  (test (u32vector-length (make-u32vector 4194303)) 4194303)
  (test (u32vector) #u32())
  (test (u32vector 5 -1 -2147483648 4294967295) #u32(5 4294967295 2147483648 4294967295))
  (test (u32vector-length '#u32(5 6 7 8)) 4)
  (test (u32vector-ref '#u32(5 6 7 8) 3) 8)
  (test (let ((x (u32vector 5 6))) (u32vector-set! x 1 3) x) #u32(5 3))
  (test (let ((x (u32vector 5 6))) (u32vector-set! x 1 -2147483648) x) #u32(5 2147483648))
  (test (let ((x (u32vector 5 6))) (u32vector-set! x 1 4294967295) x) #u32(5 4294967295))
  (test (u32vector->list '#u32(5 6)) (5 6))
  (test (list->u32vector '(5 -1 -2147483648 4294967295)) #u32(5 4294967295 2147483648 4294967295))
  (err (make-u32vector -1))
  (err (make-u32vector 4194304))
  (err (make-u32vector 536870911))
  (err (make-u32vector 536870912))
  (err (make-u32vector 1.5))
  (err (make-u32vector 1 'a))
  (err (u32vector 5 'b))
  (err (u32vector-length 123456789012345678))
  (err (u32vector-ref 123456789012345678 0))
  (err (u32vector-ref '#u32(5 6) -1))
  (err (u32vector-ref '#u32(5 6) 2))
  (err (u32vector-ref '#u32(5 6) 536870912))
  (err (u32vector-set! 123456789012345678 0 3))
  (err (let ((x (u32vector 5 6))) (u32vector-set! x -1 3) x))
  (err (let ((x (u32vector 5 6))) (u32vector-set! x 2 3) x))
  (err (let ((x (u32vector 5 6))) (u32vector-set! x 536870912 3) x))
  (err (let ((x (u32vector 5 6))) (u32vector-set! x 1 -2147483649) x))
  (err (let ((x (u32vector 5 6))) (u32vector-set! x 1 4294967296) x))
  (err (let ((x (u32vector 5 6))) (u32vector-set! x 1 'a) x))
  (err (u32vector->list 123456789012345678))
  (err (list->u32vector 123456789012345678))
  (err (list->u32vector '(5 b)))
  (err (list->u32vector '(4294967296))))

(define (run-f32vectors)
  (test (f32vector? '#f32(5. 6. 7. 8.)) #t)
  (test (f32vector? 123456789012345678) #f)
  (test (make-f32vector 0) #f32())
  (test (make-f32vector 3) #f32(0. 0. 0.))
  (test (make-f32vector 5 6.) #f32(6. 6. 6. 6. 6.))
  (test (f32vector-length (make-f32vector 4194303)) 4194303)
  (test (f32vector) #f32())
  (test (f32vector 5. 6.) #f32(5. 6.))
  (test (f32vector-length '#f32(5. 6. 7. 8.)) 4)
  (test (f32vector-ref '#f32(5. 6. 7. 8.) 3) 8.)
  (test (let ((x (f32vector 5. 6.))) (f32vector-set! x 1 3.) x) #f32(5. 3.))
  (test (f32vector->list '#f32(5. 6.)) (5. 6.))
  (test (list->f32vector '(5. 6.)) #f32(5. 6.))
  (err (make-f32vector -1))
  (err (make-f32vector 4194304))
  (err (make-f32vector 536870911))
  (err (make-f32vector 536870912))
  (err (make-f32vector 1.5))
  (err (make-f32vector 1 'a))
  (err (f32vector 5. 'b))
  (err (f32vector-length 123456789012345678))
  (err (f32vector-ref 123456789012345678 0))
  (err (f32vector-ref '#f32(5. 6.) -1))
  (err (f32vector-ref '#f32(5. 6.) 2))
  (err (f32vector-ref '#f32(5. 6.) 536870912))
  (err (f32vector-set! 123456789012345678 0 3.))
  (err (let ((x (f32vector 5. 6.))) (f32vector-set! x -1 3.) x))
  (err (let ((x (f32vector 5. 6.))) (f32vector-set! x 2 3.) x))
  (err (let ((x (f32vector 5. 6.))) (f32vector-set! x 536870912 3.) x))
  (err (let ((x (f32vector 5. 6.))) (f32vector-set! x 1 'a) x))
  (err (f32vector->list 123456789012345678))
  (err (list->f32vector 123456789012345678))
  (err (list->f32vector '(5. b))))

(define (run-f64vectors)
  (test (f64vector? '#f64(5. 6. 7. 8.)) #t)
  (test (f64vector? 123456789012345678) #f)
  (test (make-f64vector 0) #f64())
  (test (make-f64vector 3) #f64(0. 0. 0.))
  (test (make-f64vector 5 6.) #f64(6. 6. 6. 6. 6.))
  (test (f32vector-length (make-f32vector 2097151)) 2097151)
  (test (f64vector) #f64())
  (test (f64vector 5. 6.) #f64(5. 6.))
  (test (f64vector-length '#f64(5. 6. 7. 8.)) 4)
  (test (f64vector-ref '#f64(5. 6. 7. 8.) 3) 8.)
  (test (let ((x (f64vector 5. 6.))) (f64vector-set! x 1 3.) x) #f64(5. 3.))
  (test (f64vector->list '#f64(5. 6.)) (5. 6.))
  (test (list->f64vector '(5. 6.)) #f64(5. 6.))
  (err (make-f64vector -1))
  (err (make-f64vector 2097152))
  (err (make-f64vector 536870911))
  (err (make-f64vector 536870912))
  (err (make-f64vector 1.5))
  (err (make-f64vector 1 'a))
  (err (f64vector 5. 'b))
  (err (f64vector-length 123456789012345678))
  (err (f64vector-ref 123456789012345678 0))
  (err (f64vector-ref '#f64(5. 6.) -1))
  (err (f64vector-ref '#f64(5. 6.) 2))
  (err (f64vector-ref '#f64(5. 6.) 536870912))
  (err (f64vector-set! 123456789012345678 0 3.))
  (err (let ((x (f64vector 5. 6.))) (f64vector-set! x -1 3.) x))
  (err (let ((x (f64vector 5. 6.))) (f64vector-set! x 2 3.) x))
  (err (let ((x (f64vector 5. 6.))) (f64vector-set! x 536870912 3.) x))
  (err (let ((x (f64vector 5. 6.))) (f64vector-set! x 1 'a) x))
  (err (f64vector->list 123456789012345678))
  (err (list->f64vector 123456789012345678))
  (err (list->f64vector '(5. b))))

(define (test3)
  (run-strings)
  (run-vectors)
  (run-u8vectors)
  (run-u16vectors)
  (run-u32vectors)
  (run-f32vectors)
  (run-f64vectors))

;------------------------------------------------------------------------------

(##declare (separate))

(define c1 2.)
(define c2 209.177)
(define c3 1.)
(define c4 -1.)
(define c5 0.)
(define c6 -0.)
(define c7 +nan.)
(define c8 +inf.)
(define c9 -inf.)
(define c10 1e40)
(define c11 -1e-291)

(define (test4)
  (write (sqrt c1)) (newline)
  (write (round c2)) (newline)
  (write (##flonum./ c3 c5)) (newline)
  (write (##flonum./ c4 c5)) (newline)
  (write (##flonum./ c3 c6)) (newline)
  (write (##flonum./ c4 c6)) (newline)
  (write c7) (newline)
  (write (##flonum.+ c9 c8)) (newline)
  (write (##flonum.< (##flonum.log c5) -1.797693e308)) (newline)
  (write (##flonum.< 1.797693e308 (##flonum.exp c10))) (newline)
  (write (##flonum./ c11 c10)) (newline)
  (if (not (equal? (number->string (##flonum.* c3 -0.)) "-0."))
    (begin
      (display "*** warning: (##flonum.* 0. -0.) != -0." ##stderr)
      (newline ##stderr)))
  (if (not (equal? (number->string (##flonum.* c4 0.)) "-0."))
    (begin
      (display "*** warning: (##flonum.* -0. 0.) != -0." ##stderr)
      (newline ##stderr)))
  (if (##flonum.< c7 c5)
    (begin
      (display "*** warning: (##flonum.< +nan. 0.) => #t" ##stderr)
      (newline ##stderr)))
  (if (##flonum.< c5 c7)
    (begin
      (display "*** warning: (##flonum.< 0. +nan.) => #t" ##stderr)
      (newline ##stderr)))
  (if (##flonum.< c7 c8)
    (begin
      (display "*** warning: (##flonum.< +nan. +inf.) => #t" ##stderr)
      (newline ##stderr)))
  (if (##flonum.< c8 c7)
    (begin
      (display "*** warning: (##flonum.< +inf. +nan.) => #t" ##stderr)
      (newline ##stderr)))
  (if (##flonum.< c7 c9)
    (begin
      (display "*** warning: (##flonum.< +nan. -inf.) => #t" ##stderr)
      (newline ##stderr)))
  (if (##flonum.< c9 c7)
    (begin
      (display "*** warning: (##flonum.< -inf. +nan.) => #t" ##stderr)
      (newline ##stderr)))
  (if (##flonum.= c7 c8)
    (begin
      (display "*** warning: (##flonum.= +nan. +inf.) => #t" ##stderr)
      (newline ##stderr)))
  (if (##flonum.= c7 c9)
    (begin
      (display "*** warning: (##flonum.= +nan. -inf.) => #t" ##stderr)
      (newline ##stderr)))
)

;------------------------------------------------------------------------------

(##declare (generic))

(define intrs 0)
(set! ##timer-interrupt-jobs (##make-jobs));(##clear-timer-interrupt-jobs)
(##add-timer-interrupt-job (lambda () (set! intrs (##fixnum.+ intrs 1))))
(define start-time (runtime))

(test1)
(test2)

(let ((rt (real-time)))
  (if (< rt 881337930172432000) ; Fri Dec  5 11:05:30 EST 1997
    (begin
      (display "real-time is low: ")
      (write rt))
    (display "real-time OK"))
  (newline))

(let ((i intrs) (elapsed (- (runtime) start-time)))
  (display elapsed ##stderr)
  (display " secs elapsed cpu time" ##stderr)
  (newline ##stderr)
  (if (or (< i 2)
          (< (* elapsed 11) (- i 1))
          (> (* elapsed 9) (+ i 1)))
    (begin
      (display "*** possible timer problem: " ##stderr)
      (write i ##stderr)
      (display " interrupts in " ##stderr)
      (write elapsed ##stderr)
      (display " secs" ##stderr)
      (newline ##stderr))))

(test3)
(test4)

;------------------------------------------------------------------------------
