package biss.jde;

import biss.AutarkCommand;
import biss.FileLib;
import biss.ObserverSocket;
import biss.Queue;
import biss.StringLib;
import biss.awt.Checkbox;
import biss.awt.Confirmer;
import biss.awt.List;
import biss.awt.Menu;
import biss.awt.PaneDrawer;
import biss.awt.Prompter;
import java.awt.Color;
import java.awt.Rectangle;
import java.io.File;
import java.util.Enumeration;
import java.util.Observable;
import java.util.Observer;
import java.util.Vector;

/**
 * logical view to browse a set of CompileUnits (Java sources) organized
 * in directories and/or packages
 *
 * (C) 1996,97 BISS GmbH Germany, see file 'LICENSE.BISS-AWT' for details
 * @author P.C.Mehlitz
 */
public class LibBrowser
  implements Observer
{
	int Mode;
	LibViewer View = new LibViewer( getOpenRect());
	SourceLibrary Lib;
	Object SelCntr;
	Object SelItem;
	Queue StatQueue = new Queue(30);
	Queue FInfoQueue = new Queue( 50);
	final static int MOD_FILE = 1;
	final static int MOD_SCAN = 0;
	final static int MOD_TREE = 3;
	final static int MOD_TYPE = 2;
	public static Vector Instances = new Vector(5);
	PaneDrawer CntrDrawer;
	PaneDrawer ItemDrawer;

public LibBrowser () {
	Lib = SourceLibrary.getDefaultLib();
	registerObserver();

	if ( !Lib.isInitialized() )
		scanLibrary();
	else {
		View.FileBtn.setChecked( true, true);
		Lib.addAllCUsObserver( this);
	}

	CntrDrawer = new PaneDrawer( View.CntrPane, StatQueue);
	ItemDrawer = new PaneDrawer( View.ItemPane, FInfoQueue);

	Instances.addElement( this);
}

public LibBrowser ( int mode, Object selCntr) {
	this();

	if ( mode == MOD_TYPE ) 
		View.TypeBtn.setChecked( true, true);

	if ( selCntr != null ) {
		View.CntrPane.setSelection( selCntr, true);
	}
}

public void addNewCompileUnit ( String cuName ) {
	File dir;

	if ( cuName.length() == 0 ) return;
	if ( !cuName.endsWith( ".java") )
		cuName += ".java";

	if ( Mode == MOD_FILE ) {
		if ( SelCntr != null ){
			cuName = ((File)SelCntr).getAbsolutePath() + File.separator + cuName;

			CompileUnit cu = new CompileUnit( cuName);
			cu.save();
			Lib.addCompileUnit( cu, this);
			View.ItemPane.setContents( getDisplayItems());
			View.ItemPane.setSelection( cu, true);
			new CUBrowser( cu);
		}
	}
}

public void askForDeleteCUConfirm () {
	if ( Mode != MOD_FILE || !View.ItemPane.hasSelections() ) return;

	new Confirmer( "delete all selected CompileUnits", "deleteCUs", this, View);
}

void askForNewCUNames () {
	if ( !View.CntrPane.hasSelection() ) {
		View.Info.setContents( "directory selection required!");
		return;
	}
	new Prompter( "name of new CompileUnit", "addCUs", this, View);
}

void closeView () {
	CntrDrawer.stopIt();
	ItemDrawer.stopIt();

	Instances.removeElement( this);
	unregisterObserver();
}

void designItem () {
	CompileUnit cu;
	if ( SelItem == null ) return;

	if ( Mode == MOD_TYPE ) {
		TypeDecl t = (TypeDecl) SelItem;
		cu = (CompileUnit) t.CU;
		new Designer( cu, t);
	}
	else {
		cu = (CompileUnit) SelItem;
		new Designer( cu, cu.mainType() );
	}
}

void drawCntrStatistics () {
	if ( !StatQueue.empty() ) {
		Object cntr = StatQueue.getNext();
		if ( View.CntrPane.setDrawEnvironment( cntr) ) {
			String s = null;
			if ( Mode == MOD_FILE )
				s = StringLib.valueOf( Lib.getCUNumberOfDir( (File)cntr), 3);
			else if ( Mode == MOD_TYPE )
				s = StringLib.valueOf( Lib.getTypeNumberOfPackage( (String)cntr), 3);

			if ( s != null )
				DrawSupport.drawCntrStats( View.CntrPane, s);
		}
	}
}

void drawContainer () {
	if ( Mode == MOD_SCAN ) {
		DrawSupport.drawCntr( View.CntrPane, View.CntrPane.DrawObject.toString(), null);
	}
	else {
		Object e = View.CntrPane.DrawObject;
		String id = "", head = null;

		if ( Mode == MOD_FILE ) {
			File dir = (File) e;
			id = dir.getAbsolutePath();
			head = dir.getParent() + File.separatorChar;
		}
		else if ( Mode == MOD_TYPE ) {
			String pckg = (String)e;
			id = pckg;
			int i = pckg.lastIndexOf( '.');
			if ( i > 0 ) 
				head = pckg.substring( 0, i);
		}

		DrawSupport.drawCntr( View.CntrPane, id, head);
		StatQueue.appendNotifyUnique( e);
	}
}

void drawFileInfos () {
	while ( !FInfoQueue.empty() ) {
		CompileUnit cu = (CompileUnit) FInfoQueue.getNext();
		if ( Mode == MOD_FILE && View.ItemPane.setDrawEnvironment( cu) ) {
			String s = cu.fileInfoString( true);
			DrawSupport.drawCUFileInfo( View.ItemPane, s, cu.IsExpanded, cu.ExpandObservers);
		}
	}
}

public void drawItem () {
	CompileUnit cu;

	switch ( Mode ) {
	case MOD_SCAN:
	case MOD_FILE:
		cu = (CompileUnit) View.ItemPane.DrawObject;
		DrawSupport.drawCULibEntry( View.ItemPane, cu.Package, 
		                            cu.MainTypeName, cu.TypeDecls.size(),
		                            cu.IsExpanded, cu.ExpandObservers);
		if ( Mode == MOD_FILE ) FInfoQueue.appendNotifyUnique( cu);
		break;
	case MOD_TYPE:
		TypeDecl td = (TypeDecl) View.ItemPane.DrawObject;
		boolean  isIF = td instanceof InterfaceDecl;
		cu = (CompileUnit) td.CU;
		DrawSupport.drawLibTypeDecl( View.ItemPane, td.Id, cu.Package,
		                             cu.FileName, td.Mods, isIF,
		                             cu.IsExpanded, cu.ExpandObservers);
		break;
	default:
		return;
	}
}

void editItem () {
	CompileUnit cu;
	if ( SelItem == null ) return;

	if ( Mode == MOD_TYPE ) {
		TypeDecl t = (TypeDecl) SelItem;
		cu = (CompileUnit) t.CU;
		new CUBrowser( cu, t);
	}
	else {
		cu = (CompileUnit) SelItem;
		new CUBrowser( cu);
	}
}

void enableCUMenuItems ( boolean enable ) {
	Menu m = View.ItemPane.getMenu();
	if ( enable) {
		m.enableItem( "add");
		m.enableItem( "remove");
	}
	else {
		m.disableItem( "add");
		m.disableItem( "remove");
	}
}

void formatAllCUs () {
	Vector v = View.ItemPane.getContents();
	Enumeration e;
	CompileUnit cu, fullCu;
	byte buf[];
	Parser p;

	if ( Mode != MOD_FILE ) {
		View.Info.setContents( "file mode required for formatting");
		return;
	}

	for ( e=v.elements(); e.hasMoreElements(); ) {
		cu = (CompileUnit) e.nextElement();
		View.ItemPane.setSelection( cu, false);
		View.Info.setContents( "formatting " + cu.MainTypeName);

		if ( !cu.isExpanded() ) {
			buf = FileLib.getFileData( cu.SrcFile);
			p = new Parser( cu.FileName, buf);
			try {
				fullCu = p.parseCompileUnit();
				cu.expandFrom( fullCu);
				cu.format();
				cu.save();
			}
			catch ( Exception x ) {
				x.printStackTrace();
			}
		}
	}
}

Vector getDisplayCntrs () {
	switch ( Mode ) {
	case MOD_FILE: return Lib.getDirList();
	case MOD_TYPE: return Lib.getPackageList();
	}
	return null;
}

Vector getDisplayItems () {
	Vector sel = View.CntrPane.getSelections();

	if ( Mode == MOD_FILE )      return Lib.getCUsOfDirs( sel);
	else if ( Mode == MOD_TYPE ) return Lib.getTypesOfPackages( sel);
	return null;
}

static Rectangle getOpenRect () {
	Rectangle r = Jde.LibBrowserRect;
	int n = Instances.size();
	if ( n > 0 )
		r.move( 30*n, 30*n);
	return r;
}

void imageItem () {
	CompileUnit cu;
	if ( SelItem == null ) return;

	if ( Mode == MOD_TYPE ) {
		TypeDecl t = (TypeDecl) SelItem;
		cu = (CompileUnit) t.CU;
		new ImageEditor( cu, t);
	}
	else {
		cu = (CompileUnit) SelItem;
		new ImageEditor( cu, cu.mainType() );
	}
}

public static void main ( String[] args ) {
	new LibBrowser();
}

public void registerObserver ()  {
	// View.addGlobalHotKey( 'f', null, "formatAllCUs");

	Lib.addObserver( this);

	View.OsClose.addObserver( this);
	View.OsCommand.addObserver( this);

	View.ItemPane.OsDrawObject.addObserver( this);
	View.ItemPane.OsDrawPane.addObserver( this);
	View.ItemPane.OsProcess.addObserver( this);
	View.ItemPane.OsSelection.addObserver( this);
	View.ItemPane.OsCommand.addObserver( this);

	View.CntrPane.OsDrawObject.addObserver( this);
	View.CntrPane.OsDrawPane.addObserver( this);
	View.CntrPane.OsSelection.addObserver( this);
	View.CntrPane.OsCommand.addObserver( this);

	View.FileBtn.OsAction.addObserver( this);
	View.TypeBtn.OsAction.addObserver( this);
}

public void removeCompileUnits () {
	Vector list = View.ItemPane.getSelections();
	if ( Mode != MOD_FILE || list.size() == 0 ) return;

	Lib.deleteCompileUnits( list, this);
	View.ItemPane.removeAllElements( list);    
}

void scanLibrary () {
	Mode = MOD_SCAN;
	Lib.scanAsync();
}

public void setMode ( int newMode ) {
	Object oldSel = SelItem;
	Object newSel = null;

	if ( newMode == MOD_FILE ) {
		if ( oldSel != null && Mode != MOD_FILE )
			newSel = (CompileUnit) ((TypeDecl)oldSel).CU;
		enableCUMenuItems( true);
	}
	else if ( newMode == MOD_TYPE ) {
		if ( oldSel != null && Mode == MOD_FILE )
			newSel = ((CompileUnit)oldSel).mainType();
		enableCUMenuItems( false);
	}
	Mode = newMode;
	SelCntr = null;
	View.CntrPane.setContents( getDisplayCntrs());
	View.ItemPane.setContents( getDisplayItems());
	View.ItemPane.setSelection( newSel, true);
}

public void unregisterObserver ()  {
	Lib.deleteAllCUsObserver( this);
	Lib.deleteObserver( this);
}

public void update ( Observable obs, Object arg ) {

	if ( obs == View.ItemPane.OsDrawObject ) {
		drawItem();
	}
	else if ( obs == View.CntrPane.OsDrawObject ) {
		drawContainer();
	}
	else if ( obs == View.CntrPane.OsDrawPane ) {
		drawCntrStatistics();
	}
	else if ( obs == View.ItemPane.OsDrawPane ) {
		drawFileInfos();
	}
	else if ( obs == View.ItemPane.OsCommand ) {
		if ( arg.equals( "add") )
			askForNewCUNames();
		else if ( arg.equals( "remove") )
			askForDeleteCUConfirm();
		else if ( arg.equals( "edit") )
			editItem();
		else if ( arg.equals( "design") )
			designItem();
		else if ( arg.equals( "image") )
			imageItem();
	}
	else if ( obs == View.ItemPane.OsProcess ) {
		editItem();
	}
	else if ( obs == View.ItemPane.OsSelection ) {
		SelItem = View.ItemPane.getSelection();
	}
	else if ( obs == View.CntrPane.OsSelection ) {
		SelCntr = View.CntrPane.getSelection();
		View.ItemPane.setContents( getDisplayItems());
	}
	else if ( obs == View.CntrPane.OsCommand ) {
		if ( arg.equals( "clone") )
			new LibBrowser( Mode, SelCntr);
	}
	else if ( obs == View.FileBtn.OsAction ) {
		setMode( MOD_FILE);
	}
	else if ( obs == View.TypeBtn.OsAction ) {
		setMode( MOD_TYPE);
	}
	else if ( obs == Lib.OsScanStart ) {
		View.CntrPane.setContents( null);
		View.ItemPane.setContents( null);
	}
	else if ( obs == Lib.OsScanDirList ) {
		View.CntrPane.setContents( Lib.getDirList());
		View.ItemPane.setContents( new Vector(30));
	}
	else if ( obs == Lib.OsScanDir ) {
		File dir = (File) arg;
		View.CntrPane.setSelection( dir, false);
		View.ItemPane.setContents( new Vector(30));
	}
	else if ( obs == Lib.OsScanFile ) {
		File file = (File) arg;
		View.Info.setContents( "Parsing File: " + file.getAbsolutePath());
	}
	else if ( obs == Lib.OsScanCU ) {
		CompileUnit cu =  (CompileUnit) arg;
		View.ItemPane.addElement( cu);
		View.ItemPane.showLastObject();
	}
	else if ( obs == Lib.OsScanReady ) {
		View.Info.setContents( "Library Scan ready");
		View.FileBtn.setChecked( true, true);
		Lib.addAllCUsObserver( this);
	}
	else if ( obs == Lib.OsChanged ) {
		if ( arg != this ) {
			View.CntrPane.updateContents( getDisplayCntrs());
			View.ItemPane.updateContents( getDisplayItems());
		}
	}
	else if ( obs == View.OsClose ) {
		closeView();
	}
	else if ( obs == View.OsCommand ) {
		if ( "formatAllCUs".equals( arg) ) formatAllCUs();
	}

	else if ( obs instanceof CompileUnit ) {
		updateContents( (CompileUnit)obs);
	}
	else if ( obs instanceof Confirmer ) {
		if ( "deleteCUs".equals( arg) )
			removeCompileUnits();
	}
	else if ( obs instanceof Prompter ) {
		if ( "addCUs".equals( arg) )
			addNewCompileUnit( ((Prompter)obs).getContents());
	}
}

public void updateContents ( CompileUnit cu ) {
	if ( cu.hasLastChange( CompileUnit.CHG_TYPE |
	                       CompileUnit.CHG_PCKG |
	                       CompileUnit.CHG_FILE ) ) {
		View.CntrPane.updateContents( getDisplayCntrs());
		View.ItemPane.updateContents( getDisplayItems());
	}
	else {
		if ( Mode == MOD_FILE )
			View.ItemPane.redrawObject( cu);
		else if (Mode == MOD_TYPE) {
			for ( Enumeration e = cu.TypeDecls.elements(); e.hasMoreElements(); )
				View.ItemPane.redrawObject( e.nextElement());
		}
	}
}
}
