package biss.jde;

import biss.FileLib;
import biss.Hash;
import biss.Scanner;
import biss.awt.Awt;
import biss.awt.Checkbox;
import biss.awt.HotKey;
import biss.awt.HotKeyProcessor;
import biss.awt.Layouter;
import biss.awt.TopWindow;
import java.awt.Rectangle;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.PrintStream;
import java.util.Enumeration;
import java.util.Observable;
import java.util.Observer;
import java.util.Vector;

/**
 * logical view of the Designer, which is a tool to graphically specify
 * the contents/layout of application specific  TopWindow classes. You
 * can think of the designer as a graphical editor which produces specialized
 * Java sources.
 *
 * (C) 1996,97 BISS GmbH Germany, see file 'LICENSE.BISS-AWT' for details
 * @author J.H.Mehlitz
 */
public class Designer
  implements Observer, CUObserver, HotKeyProcessor
{
	TopWindow DF;
	DesignPane DP;
	DesignViewer View;
	CompileUnit CU;
	String LoFldName = "Lo";
	TypeDecl TD;
	int MthHash;

public Designer( CompileUnit cu, TypeDecl t ){
	CU = cu;
	TD = t;
	if ( isDesignable( t)) {
		View = new DesignViewer();
		registerObserver();  // opens via CU update
	}
}

public Designer( String pathName){
	try {
		byte[] buf = FileLib.getFileData( new File( pathName) );
		Parser p   = new Parser( pathName, buf);
		CU = p.parseCompileUnit();

		if ( isDesignable( CU.mainType()) ){
			View = new DesignViewer();
			openOn( CU, CU.mainType() ); 
			registerObserver();
		}
	}
	catch ( Throwable x ) { x.printStackTrace(); }

}

void addLayout(){
	int los = DP.countLayouts();
	DP.addLayout();
	View.CLayout.setRange( 0, los, los);
	if ( View.CLikeDef.getContents() == null )
		View.CLikeDef.setContents( (Vector)DP.Layouts.firstElement() );
}

void addObject(){
	String cn = View.CClass.stringSelection();
	String cf = View.CField.getContents();
	if ( (cn != null) && (cf.length() > 0) && (!DP.includesObject( cf)) ){
		if ( DP.Selection == null)
			DP.addObject( cn, cf, View.CText.getContents(), View.CUnit.isChecked());
		else
			DP.addObject( cn, cf, View.CText.getContents(), DP.Selection);
		View.COrder.updateContents();
		setSplits();
	}
}

void addObjectLike(){
	DesignObject os = (DesignObject)View.CLikeDef.getSelection();
	if ( os != null){
		DP.addObject( os.clone());
		View.COrder.updateContents();
	}
}

void clearInfo () {
	updatePosInfo( null, false);
	updateTypes( null);
	updateSplits( null);
}

void clearUnit () {
	View.CLayout.setRange( 0, 0, 0);
	View.CLikeDef.setContents( null);
	View.COrder.setContents( null);
}

public boolean expandCU ( CompileUnit cu ) {
	if ( CU != cu ) return false;

	if ( cu.SrcFile != null )
		return expandSourceCU();
	//  return expandBytecodeCU();

	return false;
}

boolean expandSourceCU () {
	try {
		byte[] buf = FileLib.getFileData( CU.SrcFile);
		Parser p = new Parser( CU.FileName, buf);
		p.storeOutput();
		CompileUnit fullCu = p.parseCompileUnit();
		CU.expandFrom( fullCu);
		return true;
	}
	catch( ParseException e) { return false; }
}

String getFieldInitParams( String fieldName){
	Vector v = TD.Datas;
	for ( int i=0; i<v.size(); i++){
		Data d = (Data)v.elementAt( i);
		if ( ! d.Id.equals( fieldName))
			continue;
		if ( d.InitExpr != null){
			int idx1 = d.InitExpr.indexOf( '(');
			if ( idx1 > -1) {
				int idx2 = d.InitExpr.lastIndexOf( ')');
				return d.InitExpr.substring( idx1+1, idx2);
			}
		}
	}
	return null;
}

String getFieldType( String fieldName){
	Vector v = TD.Datas;
	for ( int i=0; i<v.size(); i++){
		Data d = (Data)v.elementAt( i);
		if ( d.Id.equals( fieldName))
			return d.Type;
	}
	return null;
}

boolean initCompileUnit(){

	try {
		ByteArrayOutputStream o = new ByteArrayOutputStream( 300);
		PrintStream s = new PrintStream( o);

		s.println( "import biss.awt.Layouter;");
		s.println( "import biss.awt.TopWindow;");
		s.println( "import java.awt.Rectangle;");

		s.println( "public class " + CU.MainTypeName);
		s.println( "  extends TopWindow");
		s.println( "{");

		LoFldName = "Lo";
		s.println( "Layouter Lo = new Layouter( this);" );

		s.println( "public static void main( String[] args) {" );
		s.println( "  new " + CU.MainTypeName + "();" );
		s.println( "}" );

		s.println( "public " + CU.MainTypeName + "() {" );
		s.println( "  super( \"" + CU.MainTypeName + "\");" );
		s.println( "  open();" );
		s.println( "}" );

		s.println( "public void open() {\n}" );
		s.println( "public void init() {\n}" );

		s.println( "}" );

		Parser p = new Parser( o.toString() );
		CompileUnit cu = p.parseCompileUnit();
		CU.mergeTypes( cu.TypeDecls, true);
		CU.mergeImports( cu.Imports);
		TD = p.TDecl;

		s.close();
		o.close();

		return true;
	}
	catch( Throwable t) {
		t.printStackTrace();
		return false;
	}
}

boolean initFromMethod( String body){
	int mh = Hash.crcHash( body);
	boolean dlgOpen = true;

	if ( mh == MthHash)
		return false;

	Vector v = new Vector();
	Vector layouts = new Vector( 3);

	byte[] buf = new byte[ body.length()];
	body.getBytes( 0, body.length(), buf, 0);

	Scanner s = new Scanner( buf);
	String  ft;

	MthHash = mh;

	for (;;){
		s.nextToken(); ft = s.string();
		if (ft.length() == 0 )
			break;
		if ( ft.equals( LoFldName)){
			s.setStartMark();
			if ( s.character() == '.') {
				s.skip( 1); s.nextToken();
				String cn = s.string();
				if ( cn.equals( "newLayout" )){
					layouts.addElement( v );
					v = new Vector();
				}
				else if ( cn.equals( "add")) {
					DesignObject os = new DesignObject( s);
					os.setClass( getFieldType( os.FldName));
					os.setInitParams( getFieldInitParams( os.FldName));
					v.addElement( os);
				}
				else if ( cn.equals( "addLikeDefault")){
					s.nextToken();
					s.skipBlock( '(', ')' );
					String ln = s.string();
					ln = ln.substring( 1, ln.length()-1).trim();
					Vector vs = (Vector)layouts.firstElement();
					for ( Enumeration e = vs.elements(); e.hasMoreElements();){
						DesignObject os = (DesignObject)e.nextElement();
						if ( os.FldName.equals( ln)){
							v.addElement( os.clone() );
							break;
						}
					}
				}
			}
		}
		else if ( ft.equals( "openIn") ){
			dlgOpen = false;
			positionFrame( s, dlgOpen);
		}
		else if ( ft.equals( "openInDlgUnits") ){
			dlgOpen = true;
			positionFrame( s, dlgOpen);
		}

		s.skipToEOL();
	}

	layouts.addElement( v );

	if ( DF == null) newFrame( null, false);
	else             DP.setLayouts( layouts);

	View.CLayout.setRange( 0, layouts.size()-1, 0);
	DP.setLayout( 0);

	if ( dlgOpen)
		View.CUnit.setChecked( true, true);

	return true;
}

boolean isDesignable( TypeDecl t ){
	if ( t == null)
		return true;
	if ( ! CU.IsExpanded) {
		if ( ! expandSourceCU() )
			return false;
	}
	if ( t instanceof ClassDecl ) {
		if ( t.hasMethod( "open"))
			return true;
	}
	return false;
}

public static void main (String[] args) {
	if ( args.length > 0)
		new Designer( args[0]);
	else
		System.out.println( "usage: Designer <DesignSource>") ;
}

void modified ( Object arg) {
	DesignObject deo = DP.Selection;

	//List ESC ( select with zero arg)
	if ( (DP.LastAction == DesignPane.SELECT) && (deo == null) )
		DP.LastAction = DesignPane.CLEAR;

	switch( DP.LastAction) {
	case DesignPane.SELECT:
		updateTypes( deo);
		updatePosInfo( deo, true);
		updateSplits( deo.LoSpec);
		View.COrder.setSelection( deo, false);
		break;
	case DesignPane.TOGGLE:
		View.COrder.toggleMarker( arg);
		DP.MultiSels = View.COrder.markedContents();
		break;
	case DesignPane.CLEAR:
		updatePosInfo( null, false);
		updateTypes( null);
		updateSplits( null);
		View.COrder.setSelection( null, false);
		break;
	case DesignPane.SET_POS:
		updatePosInfo( deo, false);
		break;
	case DesignPane.SET_UNIT:
		updatePosInfo( deo, true);
		break;
	case DesignPane.REMOVE:
		clearInfo();
		break;
	case DesignPane.RESHAPE:
		if ( deo != null)
			updatePosInfo( deo, true);
		break;
	case DesignPane.REF_POS:
		updateSliderInfo();
		break;
	case DesignPane.REF_SWITCH:
		boolean ie = (DP.RefRect != null);
		View.CAlign.setChecked( ie, false);
		if ( ie) {
			updatePosInfo( null, true);
			updateSliderInfo();
		}
		else
			updatePosInfo( deo, true);
		break;
	case DesignPane.SET_LAYOUT:
		clearInfo();
		View.TOrder.setContents( "Components Layout: " + DP.CurLo);
		if ( DP.CurLo == 0) {
			View.CAddLike.disable();
			if ( View.CLikeDef.getContents() != null)
				View.CLikeDef.setContents( null);
		}
		else {
			View.CAddLike.enable();
			if ( View.CLikeDef.getContents() == null)
				View.CLikeDef.setContents( (Vector)DP.Layouts.firstElement() );
		}
		View.COrder.setContents( DP.Comps);
		break;
	}
}

public boolean needsExpandedCU ( CompileUnit cu ) {
	return cu == CU;
}

void newFrame( Rectangle r, boolean dlg){
	if ( r == null)
		r = new Rectangle( 100, 100, 300, 300);
	if ( DF == null){
		DF = new TopWindow( TD.Id );
		Layouter lo = new Layouter( DF);
		DP = new DesignPane();
		lo.add( DP, 0, 0, 1000, 1000, false, 0, 0, 0, 0, 0);

		if ( dlg)  DF.openInDlgUnits( r);
		else       DF.openIn( r);

		DP.addLayout();
		DP.setLayout( 0);
		DF.OsClose.addObserver( this);
		DP.OsModify.addObserver( this);
	}
	else {
		DP.cleanUp();
		DF.reshape( r.x, r.y, r.width, r.height);
	}
}

void openOn( CompileUnit cu, TypeDecl t ){
	CU = cu;
	TD = t;

	if ( CU.TypeDecls.size() == 0)
		initCompileUnit();

	View.setTitle( "Designer: [" + TD.Id + "]");

	for ( int i=0; i<TD.Datas.size(); i++){
		Data d = (Data)TD.Datas.elementAt( i);
		if ( d.Type.equals( "Layouter" )){
			LoFldName = d.Id;
			break;
		}
	}
	for ( int i=0; i<TD.Methods.size(); i++){
		biss.jde.Method m = (biss.jde.Method)TD.Methods.elementAt( i);
		if ( m.Name.equals( "open")) {
			initFromMethod( m.definition() );
			break;
		}
	}
}

void positionFrame( Scanner s, boolean dlg){
	char[] da = { ',', ')' };
	int[] pos = new int[4];

	s.skipTo('(');
	try {
		for( int i=0; i<4; i++){
			s.skip( 1); s.skipBlanks(); s.nextToken( da); 
			pos[i] = Integer.parseInt( s.string() );
		}
		newFrame( new Rectangle( pos[0], pos[1], pos[2], pos[3]), dlg);
	}
	catch( Exception e) {
		//    newFrame( null);
	}
}

void printFieldsOn( PrintStream s){
	Vector   v = DP.allObjects();

	for ( int i=0; i<v.size(); i++){
		DesignObject os = (DesignObject)v.elementAt( i);
		s.println( os.ClsPath + " " + os.FldName +
		           " = new " + os.ClsPath +
		           "(" + os.getInitParams() + ");" );
	}
}

void printImportsOn( PrintStream s){
	Vector   v = DP.allObjects();

	for ( int i=0; i<v.size(); i++){
		DesignObject os = (DesignObject)v.elementAt( i);
		s.println( "import " + os.PkgName + "." + os.ClsName + ";");
	}
}

void printMethodsOn( PrintStream s){

	Rectangle r = DF.bounds();

	s.println( "public void open() {" );
	DP.printLayoutOn( s, LoFldName);
	s.println( "\n\tinit();" );
	s.println( "\tif ( Bounds != null)" );
	s.println( "\t\topenIn( Bounds);" );
	s.println( "\telse" );

	if ( ! View.CUnit.isChecked() ) {
		s.println( "\t\topenIn( " + r.x + ", " + r.y + ", " +
		           r.width + ", " + r.height + ");" );
	}
	else {
		s.println( "\t\topenInDlgUnits( " + r.x * 100 / Awt.DlgXUnit + ", " +
		           r.y * 100 / Awt.DlgYUnit + ", " +
		           r.width * 100 / Awt.DlgXUnit + ", " +
		           r.height * 100 / Awt.DlgYUnit + ");" );
	}

	s.println( "}");
}

public void processKey( Object sel){
	if ( "shiftUp".equals( sel))
		View.COrder.shiftUp();
	else if ( "shiftDown".equals( sel))
		View.COrder.shiftDown();
}

void registerObserver(){
	CU.addObserver( this);
	View.OsClose.addObserver( this);
	View.Bm.OsCommand.addObserver( this);
	View.CClass.OsSelection.addObserver( this);
	View.CField.OsEnter.addObserver( this);
	View.CText.OsEnter.addObserver( this);
	View.CLayout.OsSpinChanged.addObserver( this);
	View.CUnit.OsAction.addObserver( this);
	View.CGrid.OsSpinChanged.addObserver( this);
	View.CPosX.OsSpinChanged.addObserver( this);
	View.CPosY.OsSpinChanged.addObserver( this);
	View.CRatio.OsAction.addObserver( this);
	View.COffset.OsAction.addObserver( this);
	View.COrig.OsAction.addObserver( this);
	View.CCorn.OsAction.addObserver( this);
	View.CClear.OsAction.addObserver( this);
	View.CAdd.OsAction.addObserver( this);
	View.CAddLike.OsAction.addObserver( this);
	View.CRem.OsAction.addObserver( this);
	View.CNewLo.OsAction.addObserver( this);
	View.CSave.OsAction.addObserver( this);
	View.CPrev.OsAction.addObserver( this);
	View.CUp.OsAction.addObserver( this);
	View.CDown.OsAction.addObserver( this);
	View.COrder.OsSelection.addObserver( this);
	View.CAlign.OsAction.addObserver( this);
	View.CLeft.OsAction.addObserver( this);
	View.CRight.OsAction.addObserver( this);
	View.CTop.OsAction.addObserver( this);
	View.CBottom.OsAction.addObserver( this);
	View.CWidth.OsAction.addObserver( this);
	View.CHeight.OsAction.addObserver( this);
	View.CHorz.OsAction.addObserver( this);
	View.CVert.OsAction.addObserver( this);
	View.CSLeft.OsAction.addObserver( this);
	View.CSRight.OsAction.addObserver( this);
	View.CSTop.OsAction.addObserver( this);
	View.CSBottom.OsAction.addObserver( this);

	View.COrder.addHotKey( 'u', HotKey.Control, this, "shiftUp");
	View.COrder.addHotKey( 'd', HotKey.Control, this, "shiftDown");
}

void renameSelection( String field){
	DesignObject os = DP.Selection;

	if ((os != null) && (field != null)){
		Data da = TD.dataAt( os.FldName);
		if ( da != null)
			TD.Datas.removeElement( da);
		os.FldName = field;
	}
}

boolean save(){
	AcceptParser ap;

	if ( (DF == null) || (CU == null) )
		return false;

	CU.backup();

	try {
		ByteArrayOutputStream o = new ByteArrayOutputStream( 300);
		PrintStream s = new PrintStream( o);

		printImportsOn( s);
		ap = new AcceptParser( o.toString() );
		ap.parseImports();
		CU.mergeImports( ap.Items);

		o.reset();
		printFieldsOn( s);
		printMethodsOn( s);
		ap = new AcceptParser( o.toString() );
		TypeDecl td = ap.parseDataOrMethod();
		TD.mergeDatas( td.Datas, null);
		TD.mergeMethods( td.Methods, null); 

		o.close();
		s.close();

		CU.save();
		CU.notifyObservers( this);
		return true;
	}
	catch( Throwable t) {
		t.printStackTrace();
		return false;
	}
}

void setSplits () {
	DP.setSelectionSplits( View.CSLeft.isChecked(),
	                       View.CSRight.isChecked(),
	                       View.CSTop.isChecked(),
	                       View.CSBottom.isChecked() );
}

boolean showPreview(){
	if ( DF == null)
		return false;
	DesignPreviewer pb = new DesignPreviewer();
	pb.buildPreview( "Preview: " + DF.getTitle(), 
	                 DP.Comps, DF.bounds() );
	pb.show();
	//  pb.save();
	return true;
}

void unregisterObserver(){
	CU.deleteObserver( this);
}

public void update( Observable obs, Object arg){
	if ( obs == CU){
		if ( arg != this)
			openOn( CU, TD);
	}
	else if ( View.Bm.OsCommand == obs){
		if ( arg.equals( "Save"))
			save();
		else if ( arg.equals( "Preview"))
			showPreview();
		else if ( arg.equals( "Add"))
			addObject();
		else if ( arg.equals( "Remove")){
			if ( DP != null)
				DP.removeSelection();
		}
	}
	else if ( obs == View.OsClose){
		unregisterObserver();
		if ( DF != null) DF.dispose();
	}
	else if ( DF == null)
		return;
	else if ( obs == DP.OsModify)
		modified( arg);
	else if ( obs == DF.OsClose){
		DP = null; DF = null;
		clearInfo();
		clearUnit();
	}
	else if ( obs == View.CAdd.OsAction)
		addObject();
	else if ( obs == View.CSave.OsAction)
		save();
	else if ( obs == View.CPrev.OsAction)
		showPreview();
	else if ( obs == View.CAddLike.OsAction)
		addObjectLike();
	else if ( obs == View.CRem.OsAction)
		DP.removeSelection();
	else if ( obs == View.CClear.OsAction)
		DP.clearSelections();
	else if ( obs == View.CNewLo.OsAction)
		addLayout();
	else if ( obs == View.CClass.OsSelection)
		DP.setSelectionClass( (String)View.CClass.getSelection() );
	else if ( obs == View.CGrid.OsSpinChanged){
		DP.Grid = View.CGrid.getContents();
		View.CDist.setSpinStep( DP.Grid);
		View.CDist.setRange( DP.Grid, DP.Grid*10, DP.Grid);
	}
	else if ( obs == View.CLayout.OsSpinChanged)
		DP.setLayout( View.CLayout.getContents());
	else if ( obs == View.CText.OsEnter)
		DP.setSelectionText( View.CText.getContents() );
	else if ( obs == View.CField.OsEnter)
		renameSelection( View.CField.getContents() );
	//    DP.setSelectionField( View.CField.getContents() );
	else if ( obs == View.CUnit.OsAction)
		DP.setUnits( View.CUnit.isChecked() );
	else if ( obs == View.CPosX.OsSpinChanged)
		updateSelectionPos( View.CPosX.getContents(), false);
	else if ( obs == View.CPosY.OsSpinChanged)
		updateSelectionPos( View.CPosY.getContents(), true);
	else if ( obs == View.CRatio.OsAction)	 updatePosInfo();
	else if ( obs == View.COffset.OsAction)	 updatePosInfo();
	else if ( obs == View.COrig.OsAction)		 updatePosInfo();
	else if ( obs == View.CCorn.OsAction)	   updatePosInfo();
	else if ( obs == View.CSLeft.OsAction)   setSplits();
	else if ( obs == View.CSRight.OsAction)  setSplits();
	else if ( obs == View.CSTop.OsAction)    setSplits();
	else if ( obs == View.CSBottom.OsAction) setSplits();
	else if ( obs == View.CUp.OsAction)			 View.COrder.shiftUp();
	else if ( obs == View.CDown.OsAction)		 View.COrder.shiftDown();
	else if ( obs == View.COrder.OsSelection){
		DP.MultiSels = View.COrder.markedContents();
		DP.selectObject( (DesignObject)arg);
	}
	else if ( obs == View.CAlign.OsAction)
		DP.showRefLines( View.CAlign.isChecked());
	else if ( obs == View.CLeft.OsAction)
		DP.alignLeft();
	else if ( obs == View.CRight.OsAction)
		DP.alignRight();
	else if ( obs == View.CTop.OsAction)
		DP.alignTop();
	else if ( obs == View.CBottom.OsAction)
		DP.alignBottom();
	else if ( obs == View.CWidth.OsAction)
		DP.assignWidth();
	else if ( obs == View.CHeight.OsAction)
		DP.assignHeight();
	else if ( obs == View.CHorz.OsAction)
		DP.equiDistX( View.CDist.getContents() );
	else if ( obs == View.CVert.OsAction)
		DP.equiDistY( View.CDist.getContents() );
}

void updatePosInfo () {
	if ( DP != null){
		if ( DP.Selection != null)
			updatePosInfo( DP.Selection, true);
	}
}

void updatePosInfo ( DesignObject deo, boolean spinUpd) {
	DesignObjLayout lo = (deo != null) ? deo.LoSpec : null;
	if ( lo == null){
		View.RatioInfo.setContents( "");
		View.OffsetInfo.setContents( "");
		View.SizeInfo.setContents( "");
		View.CPosX.clear();
		View.CPosY.clear();
	}
	else{
		View.RatioInfo.setContents( lo.ratioString() );
		View.OffsetInfo.setContents( lo.offsetString() );
		View.SizeInfo.setContents( deo.sizeString() );

		if ( spinUpd) {
			int x=0, y=0;
			if ( View.CRatio.isChecked() ){
				if ( View.COrig.isChecked() ){
					x = lo.X0; y = lo.Y0;
				}
				else if ( View.CCorn.isChecked() ) {
					x = lo.X1; y = lo.Y1;
				}
			}
			else if ( View.COffset.isChecked() ) {
				if ( View.COrig.isChecked() ){
					x = lo.OffX0; y = lo.OffY0;
				}
				else if ( View.CCorn.isChecked() ) {
					x = lo.OffX1; y = lo.OffY1;
				}
			}

			View.CPosX.setContents( x);
			View.CPosY.setContents( y);
		}
	}
}

void updateRefPos ( int pos, boolean yVal) {
	int x = yVal ? View.CPosX.getContents() : pos;
	int y = yVal ? pos : View.CPosY.getContents();
	DP.RefRect.move( x - DP.SOF, y - DP.SOF);
	DP.update(); 
}

boolean updateSelectionPos ( int pos, boolean yVal) {
	DesignObject os = (DF != null) ? DP.Selection : null;

	if ( DP.RefRect != null){
		updateRefPos( pos, yVal);
		return false;
	}

	if ( os == null)
		return false;

	DesignObjLayout ol = os.LoSpec;
	if ( View.CRatio.isChecked() )
		DP.updateSelectionRatio( pos, !View.COrig.isChecked(), yVal);
	else if ( View.COffset.isChecked() ){
		if ( View.COrig.isChecked() ){
			if ( yVal) ol.OffY0 = pos;
			else       ol.OffX0 = pos;
		}
		else if ( View.CCorn.isChecked() ){
			if ( yVal) ol.OffY1 = pos;
			else       ol.OffX1 = pos;
		}
		DP.updateSelectionRect();
	}
	return true;
}

void updateSliderInfo () {
	View.OffsetInfo.setContents( DP.refPosString() );
	View.CPosX.setContents( DP.RefRect.x + DP.SOF);
	View.CPosY.setContents( DP.RefRect.y + DP.SOF);
}

void updateSplits ( DesignObjLayout lo) {
	boolean l = (lo == null) ? false : (lo.VarSides & Layouter.LEFT) != 0;
	boolean r = (lo == null) ? false : (lo.VarSides & Layouter.RIGHT) != 0;
	boolean t = (lo == null) ? false : (lo.VarSides & Layouter.TOP) != 0;
	boolean b = (lo == null) ? false : (lo.VarSides & Layouter.BOTTOM) != 0;

	View.CSLeft.setChecked( l, false);
	View.CSRight.setChecked( r, false);
	View.CSTop.setChecked( t, false);
	View.CSBottom.setChecked( b, false);
}

void updateTypes () {
	if ( DP != null){
		if ( DP.Selection != null)
			updateTypes( DP.Selection);
	}
}

void updateTypes ( DesignObject os) {
	if ( os == null) {
		View.CField.setContents( "");
		View.CText.setContents( "\"\"");
		View.CClass.setSelection( "");
	}
	else {
		View.CClass.setSelection( os.ClsPath);
		View.CField.setContents( os.FldName);
		View.CText.setContents( os.InitParams);
	}
}
}
