package biss.awt;

import biss.Command;
import biss.CommandQueue;
import java.awt.Component;
import java.awt.Graphics;

/**
 * command class for eliminating superfluous series of paints. This might
 * occur due to the JDK 1.0.2 ScreenUpdater and native libs not using backing store.
 * The effect can block the whole app for several seconds if partially obscuring
 * toplevel windows (e.g. by dragging another toplevel window on top of a
 * Java window)
 *
 * (C) 1996,97 BISS GmbH Germany, see file 'LICENSE.BISS-AWT' for details
 * @author P.C.Mehlitz
 */
public class PaintCmd
  implements Command
{
	Component Target;
	long LastPaint;
	int Count;
	static long Delta = Awt.FilterPaints;
	boolean IsAsyncPaint = false;
	boolean SkipNextPaint = false;

/**
 * command class to filter out paint series (high frequency paint() series of
 * the same target. This is a common poblem for awt implementations wich do not
 * use backing store or X servers which do not provide this. Partly obscuring a window
 * (e.g. by slowly dragging another window accross it) may block the whole Java app
 * for several seconds (by lots of superfluous repaints). This class can be used to
 * detect these cases and to defer the painting via the command queue.
 *
 * (C) 1996,97 BISS GmbH Germany, see file 'LICENSE.BISS-AWT' for details
 * @author P.C.Mehlitz
 */
public PaintCmd ( Component target ) {
	Target = target;
	LastPaint = 0;
}

public synchronized boolean checkPaintSeries () {

	if ( SkipNextPaint ) {   // explicit order to skip this paint
		SkipNextPaint = false;
		return true;
	}

	if ( Count > 0 ){		     // already queued, skip this paint
		if ( Count == 1 ) Count++; // retain again, the series seems not to be terminated yet
		return true;
	}

	long t = System.currentTimeMillis();

	if ( IsAsyncPaint ) {    // check if this is called from execute()
		LastPaint = t;
		return false;
	}

	if ( (t - LastPaint) < Delta ) { // defer painting
		Count = 1;
		CommandQueue.append( this);
		return true;
	}
	else {                   // proceed with normal painting
		LastPaint = t;
		return false;
	}
}

public synchronized boolean execute () {

	if ( --Count > 0 )
		CommandQueue.append( this);
	else {
		IsAsyncPaint = true;

		Graphics g = Target.getGraphics();
		if ( g != null ) {
			Target.paint( g);
			g.dispose();
		}

		IsAsyncPaint = false;
	}

	return true;
}

public synchronized void forceNextPaint () {
	SkipNextPaint = false;
}

public synchronized void skipNextPaint () {
	SkipNextPaint = true;
}
}
