;* --------------------------------------------------------------------*/
;*    Copyright (c) 1992-1998 by Manuel Serrano. All rights reserved.  */
;*                                                                     */
;*                                     ,--^,                           */
;*                               _ ___/ /|/                            */
;*                           ,;'( )__, ) '                             */
;*                          ;;  //   L__.                              */
;*                          '   \   /  '                               */
;*                               ^   ^                                 */
;*                                                                     */
;*                                                                     */
;*    This program is distributed in the hope that it will be useful.  */
;*    Use and copying of this software and preparation of derivative   */
;*    works based upon this software are permitted, so long as the     */
;*    following conditions are met:                                    */
;*           o credit to the authors is acknowledged following         */
;*             current academic behaviour                              */
;*           o no fees or compensation are charged for use, copies,    */
;*             or access to this software                              */
;*           o this copyright notice is included intact.               */
;*      This software is made available AS IS, and no warranty is made */
;*      about the software or its performance.                         */
;*                                                                     */
;*      Bug descriptions, use reports, comments or suggestions are     */
;*      welcome. Send them to                                          */
;*        Manuel Serrano -- Manuel.Serrano@unice.fr                    */
;*-------------------------------------------------------------------- */
;*=====================================================================*/
;*    serrano/prgm/project/bigloo/comptime/Module/impuse.scm           */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Tue Jun  4 12:25:53 1996                          */
;*    Last change :  Wed Feb 11 09:25:06 1998 (serrano)                */
;*    -------------------------------------------------------------    */
;*    The import/usr/from clauses compilation                          */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    The module                                                       */
;*---------------------------------------------------------------------*/
(module module_impuse
   
   (include "Ast/unit.sch"
	    "Tools/trace.sch")
   
   (import  module_module
	    module_prototype
	    module_class
	    module_include
	    tools_speek
	    tools_error
	    tools_file
	    type_type
	    read_access
	    read_inline
	    ast_var
	    ast_find-gdefs
	    ast_glo-decl
	    engine_param
	    init_main)
	   
   (export  (make-import-compiler)
	    (make-use-compiler)
	    (make-from-compiler)
	    (get-imported-modules)
	    (import-with-module! ::symbol)
	    (import-parser       ::symbol prototype)))

;*---------------------------------------------------------------------*/
;*    make-import-compiler ...                                         */
;*---------------------------------------------------------------------*/
(define (make-import-compiler)
   (instantiate::ccomp (id 'import)
		       (producer impuse-producer)))
 
;*---------------------------------------------------------------------*/
;*    make-use-compiler ...                                            */
;*---------------------------------------------------------------------*/
(define (make-use-compiler)
   (instantiate::ccomp (id 'use)
		       (producer impuse-producer)
		       (finalizer impuse-finalizer)))

;*---------------------------------------------------------------------*/
;*    make-from-compiler ...                                           */
;*---------------------------------------------------------------------*/
(define (make-from-compiler)
   (instantiate::ccomp (id 'from)
		       (producer impuse-producer)
		       (consumer (lambda (module clause)
				    (impuse-producer clause)
				    '()))))

;*---------------------------------------------------------------------*/
;*    impuse-producer ...                                              */
;*---------------------------------------------------------------------*/
(define (impuse-producer clause)
   (let ((mode (car clause)))
      (match-case clause
	 ((?- . ?protos)
	  (for-each (lambda (proto) (impuse-parser proto mode)) protos))
	 (else
	  (user-error "Parse error"
		      (string-append "Illegal `"
				     (string-downcase (symbol->string mode))
				     "' clause")
		      clause
		      '())))))
   
;*---------------------------------------------------------------------*/
;*    import-all-module ...                                            */
;*---------------------------------------------------------------------*/
(define (import-all-module module::symbol mode)
   (let ((b (assq module *import-module-list*)))
      (if (pair? b)
	  (imode-vars-set! (cdr b) 'all)
	  (set! *import-module-list*
		(cons (cons module (imode module mode 'all #unspecified))
		      *import-module-list*)))))

;*---------------------------------------------------------------------*/
;*    import-1-module ...                                              */
;*---------------------------------------------------------------------*/
(define (import-1-module module::symbol var mode)
   (let ((cell (assq module *import-module-list*)))
      (if (not (pair? cell))
	  (set! *import-module-list*
		(cons (cons module (imode module mode (list var) #unspecified))
		      *import-module-list*))
	  (let ((b (cdr cell)))
	     (case (imode-vars b)
		((with)
		 (imode-mode-set! b mode)
		 (imode-vars-set! b (list var)))
		((all)
		 'nothing)
		(else
		 (imode-vars-set! b (cons var (imode-vars b)))))))))

;*---------------------------------------------------------------------*/
;*    import-with-module! ...                                          */
;*---------------------------------------------------------------------*/
(define (import-with-module! module)
   (let ((b (assq module *import-module-list*)))
      (if (not (pair? b))
	  (set! *import-module-list*
		(cons (cons module (imode module 'with '() 0))
		      *import-module-list*)))))
   
;*---------------------------------------------------------------------*/
;*    impuse-parser ...                                                */
;*    -------------------------------------------------------------    */
;*    The syntaxe of importation clause is:                            */
;*    import ::= module-name                                           */
;*            |  (module-name "file-name" *)                           */
;*            |  (variable module-name)                                */
;*            |  (variable module-name "file-name" *)                  */
;*---------------------------------------------------------------------*/
(define (impuse-parser prototype mode)
   (trace (ast 2) "impuse-parser: " prototype " " mode #\Newline)
   (match-case prototype
      (((and ?module (? symbol?)) (and ?fname (? string?)) . ?rest)
       ;; (module-name "file-name" ...)
       (let loop ((rest   rest)
		  (fnames (list fname)))
	  (cond
	     ((null? rest)
	      (add-access! module (reverse! fnames))
	      (import-all-module module mode))
	     ((string? (car rest))
	      (loop (cdr rest)
		    (cons (car rest) fnames)))
	     (else
	     (user-error "Parse error"
			 "Illegal imported clause"
			 prototype
			 '())))))
      (((and ?var (? symbol?)) (and ?module (? symbol?)))
       ;; (variable module-name)
       (import-1-module module var mode))
      (((and (? symbol?) ?v) (and (? symbol?) ?m) (and ?f (? string?)) . ?rest)
       ;; (variable module-name "file-name" ...) 
       (let loop ((rest   rest)
		  (fnames (list f)))
	  (cond
	     ((null? rest)
	      (add-access! m (reverse! fnames))
	      (import-1-module m v mode))
	     ((string? (car rest))
	      (loop (cdr rest) (cons (car rest) fnames)))
	     (else
	     (user-error "Parse error"
			 "Illegal imported clause"
			 prototype
			 '())))))
      ((and ?module (? symbol?))
       ;; module-name
       (import-all-module module mode))
      (else
       (user-error "Parse error"
		   "Illegal `import/use' clause"
		   prototype
		   '()))))

;*---------------------------------------------------------------------*/
;*    imode ...                                                        */
;*---------------------------------------------------------------------*/
(define-struct imode id mode vars checksum)

;*---------------------------------------------------------------------*/
;*    *import-module-list* ...                                         */
;*---------------------------------------------------------------------*/
(define *import-module-list* '())

;*---------------------------------------------------------------------*/
;*    *imported-module-list* ...                                       */
;*    -------------------------------------------------------------    */
;*    This variable contains all the imported modules (while           */
;*    *import-module-list* is resetted each time the import finalizer  */
;*    is invoked). This variable is used by `module_library' to        */
;*    detect module already initialized.                               */
;*---------------------------------------------------------------------*/
(define *imported-module-list* '())

;*---------------------------------------------------------------------*/
;*    get-imported-modules ...                                         */
;*---------------------------------------------------------------------*/
(define (get-imported-modules)
   *imported-module-list*)

;*---------------------------------------------------------------------*/
;*    import-finalizer ...                                             */
;*    -------------------------------------------------------------    */
;*    In the `impuse' finalizer we read all the imported modules and   */
;*    if we have to, we create a unit in order to initialize imported  */
;*    modules.                                                         */
;*---------------------------------------------------------------------*/
(define (import-finalizer)
   (set! *import-module-list* (reverse! *import-module-list*))
   (trace (ast 2) "import-finalizer: " *import-module-list* #\Newline)
   (let loop ((init* '()))
      (if (null? *import-module-list*)
	  (if (pair? init*)
	      (let loop ((init*      init*)
			 (init-call* '()))
		 (if (null? init*)
		     (list (unit 'imported-modules 12 init-call* #t))
		     (let* ((id          (imode-id (car init*)))
			    (checksum    (imode-checksum (car init*)))
			    (init-fun-id (module-initialization-id id)))
			(let ((global (import-parser id (list init-fun-id
							      'checksum::long
							      'from::string))))
			   ;; module initializer can't be reachable
			   ;; from eval. We mark this.
			   (global-evaluable?-set! global #f))
			(loop (cdr init*)
			      (cons `((@ ,init-fun-id ,id) ,checksum
							   ,(symbol->string
							     *module*))
				    init-call*)))))
	      '())
	  (let ((module (car (car *import-module-list*)))
		(imode  (cdr (car *import-module-list*))))
	     (set! *import-module-list* (cdr *import-module-list*))
	     (set! *imported-module-list* (cons module *imported-module-list*))
	     (if (not (eq? (imode-mode imode) 'with))
		 (read-imported-module imode))
	     (loop (if (memq (imode-mode imode) '(import with))
		       (cons imode init*)
		       init*))))))

;*---------------------------------------------------------------------*/
;*    impuse-finalizer ...                                             */
;*    -------------------------------------------------------------    */
;*    In the `impuse' finalizer we read all the imported modules and   */
;*    if we have to, we create a unit in order to initialize imported  */
;*    modules.                                                         */
;*---------------------------------------------------------------------*/
(define (impuse-finalizer)
   (let* ((import-finalizer (import-finalizer))
	  (inline-finalizer (inline-finalizer))
	  (finalizers       (append import-finalizer inline-finalizer)))
      (if (null? finalizers)
	  'void
	  finalizers)))

;*---------------------------------------------------------------------*/
;*    import-parser ...                                                */
;*---------------------------------------------------------------------*/
(define (import-parser module::symbol prototype)
   (let ((proto (parse-prototype prototype)))
      (if (not (pair? proto))
	  (user-error "Parse error" "Illegal prototype" prototype '())
	  (case (car proto)
	     ((sfun sifun sgfun)
	      (declare-global-sfun! (cadr proto)
				    (caddr proto)
				    module
				    'import
				    (car proto)
				    prototype))
	     ((svar)
	      (declare-global-svar! (cadr proto) module	'import	prototype))
	     ((class)
	      (declare-class! (cdr proto) module 'import #f prototype))
	     ((final-class)
	      (declare-class! (cdr proto) module 'import #t prototype))
	     ((wide-class)
	      (declare-wide-class! (cdr proto) module 'import prototype))
	     (else
	      (user-error "Parse error" "Illegal prototype" prototype '()))))))
   
;*---------------------------------------------------------------------*/
;*    read-imported-module ...                                         */
;*---------------------------------------------------------------------*/
(define (read-imported-module imode)
   (trace (ast 2) #\Newline "read-imported-module: " imode #\Newline)
   (let* ((module (imode-id imode))
	  (wanted (imode-vars imode))
	  (b      (assq module *access-table*)))
      (verbose 2 "      [reading "
	       (if (eq? (imode-mode imode) 'use) "used" "imported")
	       " module " module "]" #\Newline)
      (if (not b)
	  (user-error "read-imported-module"
		      "Can't find such module"
		      module
		      '())
	  (let* ((fnames (cdr b))
		 (fname  (find-file/path (car fnames) *load-path*)))
	     (if (not (string? fname))
		 (user-error "read-imported-module"
			     "Can't open such file"
			     (car fnames)
			     '())
		 (let ((port (open-input-file fname)))
		    (if (not (input-port? port))
			(user-error "read-imported-module"
				    "Can't open such file"
				    (car fnames)
				    '())
			(let ((handler
			       (lambda (escape proc mes obj)
				  (notify-error proc mes obj)
				  (flush-output-port (current-error-port))
				  (close-input-port port)
				  (exit-bigloo -7))))
			   (try
			    (let* ((mdecl (read port #t))
				   (prov  (append
					   (begin
					      (reset-include-consumed-directive!)
					      (reset-include-consumed-code!)
					      (consume-module! module mdecl))
					   (get-include-consumed-directive)))
				   (code   (get-include-consumed-code))
				   (check  (checksum-module mdecl)))
			       (imode-checksum-set! imode check)
			       (look-for-inline
				(if (not (pair? wanted))
				    (import-everything prov module)
				    (import-wanted prov wanted module))
				code
				port
				(cdr fnames)
				module)
			       (close-input-port port))
				handler)))))))))
   
;*---------------------------------------------------------------------*/
;*    import-everything ...                                            */
;*---------------------------------------------------------------------*/
(define (import-everything provided module::symbol)
   (let loop ((provided provided)
	      (inline   '()))
      (if (null? provided)
	  inline
	  (let ((p (import-parser module (car provided))))
	     (if (global? p)
		 (let ((val (global-value p)))
		    (loop (cdr provided)
			  (cond
			     ((or (not (global? p))
				  (not (sfun? val)))
			      inline)
			     ((eq? (sfun-class val) 'sifun)
			      (cons (cons (global-id p) 'sifun) inline))
			     (else
			      inline))))
		 (loop (cdr provided) inline))))))
	     
;*---------------------------------------------------------------------*/
;*    import-wanted ...                                                */
;*---------------------------------------------------------------------*/
(define (import-wanted provided wanted module::symbol)
   (let loop ((provided provided)
	      (inline   '())
	      (wanted   wanted))
      ;; we check that all wanted functions are in the list and in
      ;; the same time, we compute the list of all inline to be fetch.
      (cond
	 ((null? wanted) 
	  inline)
	 ((null? provided)
	  (user-error module
		      "Can't find exportation for these identifiers"
		      wanted
		      '()))
	 (else
	  (let ((proto (parse-prototype (car provided))))
	     (if (pair? proto)
		 (let ((id (cadr proto)))
		    (if (not (memq id wanted))
			(loop (cdr provided)
			      inline
			      wanted)
			(let ((p (import-parser module (car provided))))
			   (cond
			      ((global? p)
			       (loop (cdr provided)
				     (cond
					((eq? (car proto) 'sifun)
					 (cons (cons id 'sifun) inline))
					(else
					 inline))
				     (remq! id wanted)))
			      ((type? p)
			       (loop (cdr provided)
				     inline
				     (remq! id wanted)))
			      (else
			       (loop (cdr provided)
				     inline
				     (remq! id wanted)))))))
		 (loop (cdr provided)
		       inline
		       wanted)))))))

