/* alloc.c
 * (c) Alexandre Frey
 */

/*
 * This is a home-brewed memory allocator
 *
 * It is designed to allow :
 *  . memory protection without too much space loss
 *  . allocation of small segments (within a 16 bits chunk)
 *    (for example alpha .lita sections )
 *  . use of garbage collector : the code is parametrised by the
 *    basic allocation routines
 */
#include "alloc.h"

/* Are mprotect() and PROT_* standard ?? */
#include <sys/mman.h>
#include <unistd.h>

int dlink_memory_errno = DLINK_MEMORY_ENO_ERROR;

/* Initialised when dlink_memory_create
 * is first called
 */
static size_t pagesize = 0;

/* to round to the lower page boundary */
static size_t pagemask;

/*
 * A general purpose block list
 */
struct block_list 
{
  size_t size;
  void* block;
  struct block_list *next;
};

struct dlink_memory 
{
  /* The allocation/desallocation functions
   * filled by dlink_memory_create
   */
  void* (*internal_alloc)(size_t);
  void  (*internal_free )(void*);
  void* (*section_alloc )(size_t);
  void  (*section_free  )(void*);
  
  /* The linked list of all allocated chunk 
   * (by section_alloc)
   * Used when freeing the entire sections
   * Sizes are kept to allow statistics
   */
  struct block_list *allocated_chunks;

  /* The list of free blocks with read_write permission
   * It gets also the memory lost when rounding to pagesize
   */
  struct block_list *read_write_free_blocks;

  struct block_list *read_only_free_blocks;
  /* Used to apply permissions */
  struct block_list *read_only_page_list;

  struct block_list *code_free_blocks;
  struct block_list *code_page_list;
  
  /* The address of the small segment (if allocated) */
  void *small_segment;

  /*
   * ---------------------------------------------
   * |rdwrt     |                    | rdonly    |
   * ---------------------------------------------
   * ^          ^       ^            ^   	  ^
   * |----------------->|small_middle|            |
   * |------------------------------>|first_non_free_rdonly;
   * |--------->|first_free_rdwrt                 |
   * small_segment               	          |
   *                                              |small_segment + 0x10000
   * (small_middle on a page boundary)
   */
  int small_first_free_rdwrt;
  int small_first_non_free_rdonly;
  int small_middle;
  
};


dlink_memory *
dlink_memory_create (void* (*internal_alloc) (size_t),
		     void  (*internal_free ) (void*),
		     void* (*section_alloc ) (size_t),
		     void  (*section_free  ) (void*) )
{
  dlink_memory *memory;

  /* Initialise pagesize if necessary */
  if (pagesize == 0) 
    {
      pagesize = getpagesize();
      pagemask = ~(pagesize - 1);
    }
  
  /* Allocate a dlink_memory structure */
  memory = (dlink_memory*)internal_alloc (sizeof (dlink_memory));

  if (memory == NULL) 
    {
      dlink_memory_errno = DLINK_MEMORY_EOUT_OF_MEMORY;
      return NULL;
    }
  
  /* Fills the underlying memory management functions fields */
  memory->internal_alloc = internal_alloc;
  memory->internal_free  = internal_free;
  memory->section_alloc  = section_alloc;
  memory->section_free   = section_free;

  /* Fills the others fields */
  memory->allocated_chunks       = NULL;
  memory->read_write_free_blocks = NULL;
  memory->read_only_free_blocks  = NULL;
  memory->read_only_page_list    = NULL;
  memory->code_free_blocks       = NULL;
  memory->code_page_list         = NULL;
  memory->small_segment          = NULL;
  memory->small_first_free_rdwrt      = 0;
  memory->small_first_non_free_rdonly = 0;
  memory->small_middle                = 0;

  dlink_memory_errno = DLINK_MEMORY_ENO_ERROR;
  return memory;
};

#define ERROR(what) do {dlink_memory_errno = DLINK_MEMORY_E##what; \
			  return NULL; } while (0)

#define OK dlink_memory_errno = DLINK_MEMORY_ENO_ERROR;

/* add a block of size SIZE at address BLOCK to BLOCK_LIST */
static struct block_list *
block_list_cons (dlink_memory *memory,
		 void* block, size_t size, struct block_list *block_list)
{
  struct block_list *new_cell;
  new_cell = (struct block_list*)
    memory->internal_alloc(sizeof (struct block_list));

  if (new_cell == NULL) ERROR(OUT_OF_MEMORY);
  
  new_cell->size  = size;
  new_cell->block = block;
  new_cell->next  = block_list;

  OK;
  return new_cell;
}
 
/* Call the underlying memory management functions
 * to allocate a chunk of memory of size SIZE
 * update the allocated_chunks field of MEMORY
 * returns NULL when no memory left
 */
static void* 
allocate_chunk (dlink_memory *memory, size_t size) 
{
  void *chunk;
  struct block_list *new_allocated_chunks;
  
  chunk = memory->section_alloc (size);
  if (chunk == NULL) ERROR(OUT_OF_MEMORY);
  
  new_allocated_chunks = 
    block_list_cons (memory, chunk, size, memory->allocated_chunks);

  if (new_allocated_chunks == NULL) return NULL;

  memory->allocated_chunks = new_allocated_chunks;

  OK;
  return chunk;
}

/* Search the block list to find a suitable block 
 * of size greater than SIZE
 * if not found return NULL
 * if found update the block
 * if the block becomes empty, update *BLOCK_LIST_PTR
 */
static void*
allocate_in_block_list (dlink_memory *memory, size_t size, 
			struct block_list **block_list_ptr)
{
  struct block_list **walker_ptr;

  if (*block_list_ptr == NULL) return NULL;

  walker_ptr = block_list_ptr;

  while (1) 
    {
      if ((*walker_ptr)->size > size)
	{
	  /* Update the block but leave ths structure of the list unchanged */
	  (*walker_ptr)->size -= size;
	  return (*walker_ptr)->block + (*walker_ptr)->size;
	}
      if ((*walker_ptr)->size == size)
	{
	  /* Use all the block */
	  void* block = (*walker_ptr)->block;
	  struct block_list *next = (*walker_ptr)->next;
	  
	  /* Free the block structure */
	  memory->internal_free (*walker_ptr);

	  /* delete it from the list */
	  *walker_ptr = next;
	  
	  return block;
	}
      
      /* any block left ? */
      if ((*walker_ptr)->next == NULL) return NULL;

      /* then try it */
      walker_ptr = &((*walker_ptr)->next);
    }
}

/* Round the adress or size to the upper page boundary */
#define PAGE_ROUND(x) (void*)((((long)x)+pagesize-1) & pagemask)
#define PAGE_SIZE_ROUND(x) (size_t)(PAGE_ROUND(x))


/* Allocate a chunk of NPAGES on a page boundary 
 * the extra memory allocated to reach the boundary
 * is added to to read_write_free_blocks
 */
static void* 
allocate_pages (dlink_memory *memory, int npages)
{
  void* chunk;			/* The chunk really allocated */
  void* page_chunk;		/* The page chunk */

  /* If NPAGES == 1, we can perhaps get the page in the Read-Write
   * Free blocks (an extra page allocated in a previous call of
   * allocate_pages). If a Read-Write Free block is of size 
   * pagesize, it must be a whole page
   */
  if (npages == 1)
    {
      page_chunk = allocate_in_block_list (memory, pagesize,
					   &memory->read_write_free_blocks);
      if (page_chunk != NULL) return page_chunk;
    }

  /* We must allocate an extra pagesize bytes to be sure
   * to get npages within the returned chunk
   * (strictly speaking pagesize-1 but that preserves alignment)
   */
  chunk = allocate_chunk (memory, (npages + 1)*pagesize);
  if (chunk == NULL) return NULL;

  page_chunk = PAGE_ROUND (chunk);
  
  if (chunk != page_chunk)
    {
      /* We collect the beginning of chunk for read_write sections
       * (we can't protect it since it isn't in a whole page)
       */
      struct block_list * new_read_write_free_blocks
	= block_list_cons (memory, 
			   chunk, page_chunk - chunk, 
			   memory->read_write_free_blocks);
      if (new_read_write_free_blocks == NULL) return NULL;
      memory->read_write_free_blocks = new_read_write_free_blocks;
    }

  if (page_chunk + npages*pagesize != chunk + (npages+1)*pagesize)
    {
      /* Idem for the end of the chunk */
      struct block_list * new_read_write_free_blocks
	= block_list_cons (memory, 
			   page_chunk + npages*pagesize, 
			   chunk + pagesize  - page_chunk, 
			   memory->read_write_free_blocks);
      if (new_read_write_free_blocks == NULL) return NULL;
      memory->read_write_free_blocks = new_read_write_free_blocks;
    }

  OK;
  return page_chunk;

}

int
dlink_memory_allocate_small_segment (dlink_memory *memory)
{
  if (memory->small_segment == NULL)
    {
      memory->small_segment = allocate_pages (memory, 0x10000/pagesize);
      if (memory->small_segment == NULL) return 0;
      memory->small_first_free_rdwrt = 0;
      memory->small_first_non_free_rdonly = 0x10000;
      memory->small_middle = 0x10000;
    }

  return 1;
}

void* 
dlink_section_allocate (dlink_memory *memory, size_t size, int class)
{
  void* section;
  if (class & DLINK_SEC_SMALL) 
    {
      /* Allocate the small_segment if necessary */
      if (!dlink_memory_allocate_small_segment (memory)) return NULL;
      switch (class & (~DLINK_SEC_SMALL)) 
	{
	case DLINK_SEC_READ_WRITE :
	  if (memory->small_first_free_rdwrt + size 
	      > memory->small_middle )
	    ERROR (OUT_OF_SMALL);
	  section = memory->small_segment + memory->small_first_free_rdwrt;
	  memory->small_first_free_rdwrt += size;
	  OK;
	  return section;
	case DLINK_SEC_READ_ONLY :
	  /* Take care of the signs */
	  if (((memory->small_first_non_free_rdonly  - (int)size) 
	       & (int)pagemask)
	      < memory->small_first_free_rdwrt)
	    ERROR (OUT_OF_SMALL);
	  memory->small_first_non_free_rdonly -= (int)size;
	  memory->small_middle 
	    = (memory->small_first_non_free_rdonly + 1) & (int)pagemask;
	  OK;
	  return memory->small_segment + memory->small_first_non_free_rdonly;
	default :
	  ERROR (ILLEGAL_ARGUMENT);
	}
    }
  else
    {
      struct block_list ** free_blocks_ptr;
      struct block_list ** page_list_ptr;
      switch (class) 
	{
	case DLINK_SEC_READ_WRITE : 
	  free_blocks_ptr = &memory->read_write_free_blocks;
	  page_list_ptr   = NULL;
	  break;
	case DLINK_SEC_CODE :
	  free_blocks_ptr = &memory->code_free_blocks;
	  page_list_ptr   = &memory->code_page_list;
	  break;
	case DLINK_SEC_READ_ONLY :
	  free_blocks_ptr = &memory->read_only_free_blocks;
	  page_list_ptr   = &memory->read_only_page_list;
	  break;
	default :
	  ERROR(ILLEGAL_ARGUMENT);
	}
      /* First search in the free block list */
      section = allocate_in_block_list (memory, size,
					free_blocks_ptr);
      OK;
      if (section != NULL) return section;
      
      /* Is it just a read write section ? */
      if (class == DLINK_SEC_READ_WRITE)
	/* Allocate it without any particular alignment precaution */
	return allocate_chunk (memory, size);
      else
	{
	  /* Allocate a suffisant number of pages to store the section */
	  size_t new_block_size = PAGE_SIZE_ROUND(size);
	  void* new_block = allocate_pages (memory, 
					    new_block_size/pagesize);
	  struct block_list *new_page_list;
	  if (new_block == NULL) return NULL;

	  /* Update the page list */
	  new_page_list = block_list_cons (memory, 
					   new_block, new_block_size, 
					   *page_list_ptr);
	  if (new_page_list == NULL) return NULL;
	  *page_list_ptr = new_page_list;
	  			
	  /* Add a new free block with the overhead if any */
	  if (new_block_size > size) 
	    {
	      struct block_list *new_free_blocks_list;
	      new_free_blocks_list = block_list_cons (memory,
						      new_block + size,
						      new_block_size - size,
						      *free_blocks_ptr);
	      if (new_free_blocks_list == NULL) return NULL;
	      *free_blocks_ptr = new_free_blocks_list;
	    }

	  OK;
	  return new_block;
	}
    }
}


/*
 * For debugging purpose 
 */
#include <stdio.h>

static void block_list_display (struct block_list *block_list) 
{
  struct block_list *walker;
  for (walker = block_list; walker != NULL; walker = walker->next)
    printf ("%p(%lx)\n", walker->block, (long)walker->size);
}

void dlink_memory_display (dlink_memory* memory) 
{
  printf ("Read-Write Free blocks :\n");
  block_list_display (memory->read_write_free_blocks);
  printf ("Read-Only Free blocks :\n");
  block_list_display (memory->read_only_free_blocks);
  printf ("Read-Only Page list :\n");
  block_list_display (memory->read_only_page_list);
  printf ("Code Free blocks :\n");
  block_list_display (memory->code_free_blocks);
  printf ("Code Page List :\n");
  block_list_display (memory->code_page_list);
  printf ("Code Allocated Chunks:\n");
  block_list_display (memory->allocated_chunks);
}


void*
dlink_memory_get_gp (dlink_memory *memory)
{
  OK;
  if (memory->small_segment) return memory->small_segment + 0x8000;
  return NULL;
}

/* Free the structure but not the blocks themselves */
static void
free_block_list (dlink_memory *memory, struct block_list *block_list)
{
  struct block_list *walker = block_list;
  while (walker != NULL)
    {
      struct block_list *next = walker->next;
      memory->internal_free (walker);
      walker = next;
    }
}

void
dlink_memory_free (dlink_memory *memory)
{
  /* Make sure the memory is unprotected */
  dlink_memory_unprotect (memory);
  {
    /* Free the chunks */
    struct block_list *walker;
    for (walker = memory->allocated_chunks; 
	 walker != NULL; 
	 walker= walker->next)
      memory->section_free (walker->block);
  }
  /* Free the block and page lists */
  free_block_list (memory, memory->allocated_chunks);
  free_block_list (memory, memory->read_write_free_blocks);
  free_block_list (memory, memory->read_only_free_blocks);
  free_block_list (memory, memory->read_only_page_list);
  free_block_list (memory, memory->code_free_blocks);
  free_block_list (memory, memory->code_page_list);

  /* Free the small segment if necessary */
  if (memory->small_segment)
    memory->internal_free (memory->small_segment);

  /* Finally, free the dlink_memory structure itself */
  memory->internal_free (memory);
}

/* Protect all the pages in BLOCK_LIST with protection PROT */
static void
protect_page_list (struct block_list *block_list, int prot)
{
  struct block_list *walker;
  for (walker = block_list; walker != NULL; walker = walker->next)
    /* Suppose that the block is a certain number of *whole* pages */
    mprotect (walker->block, walker->size, prot);
}


/* That is the half of the motivations of this package */
void
dlink_memory_protect (dlink_memory *memory)
{
  protect_page_list (memory->read_only_page_list, PROT_READ);
  protect_page_list (memory->code_page_list, PROT_READ | PROT_EXEC);
  
  /* If necessary, protect the read_only part of the small segment */
  if (memory->small_segment)
    mprotect (memory->small_segment + memory->small_middle,
	      0x10000 - memory->small_middle,
	      PROT_READ);
}

void
dlink_memory_unprotect (dlink_memory *memory)
{
  /* Reprotect with read write */
  protect_page_list (memory->read_only_page_list, PROT_READ | PROT_WRITE);
  protect_page_list (memory->code_page_list, PROT_READ | PROT_WRITE);
  
  /* Un protect the whole small segment */
  if (memory->small_segment) 
    mprotect (memory->small_segment, 0x10000, PROT_READ | PROT_WRITE);
}


/* EOF : alloc.c */

     










