// -*- mode: cpp; mode: fold -*-
// Description								/*{{{*/
// $Id: pkglist.cc,v 1.23 1998/06/06 01:13:50 jgg Exp $
/* ######################################################################

   Package List - Base class for all package list storage methods.
   Ordered Package List - A Package List that is sensitive to order

   All this does is provide a base for the admin class allowing G++ to 
   advoid problems with multiple common vtables.. 

   This also has the effect of verifying that each header is fully stand
   alone, an important aspect in a library.
   
   ##################################################################### */
									/*}}}*/
#include <fstream.h>
#include <strutl.h>
#include <pkglib/error.h>
#include <pkglib/pkglist.h>

// PkgList::~pkgPkgList - Destructor					/*{{{*/
// ---------------------------------------------------------------------
/* Provides a base for the class */
pkgPkgList::~pkgPkgList()
{
}
									/*}}}*/
// PkgList::priv_iterator::~priv_iterator - Destructor			/*{{{*/
// ---------------------------------------------------------------------
/* Provides a base for the class */
pkgPkgList::priv_iterator::~priv_iterator()
{
}
									/*}}}*/
// PkgList::iterator::Die - Delete the internal pointer and shutdown	/*{{{*/
// ---------------------------------------------------------------------
/* This is effectively a callable destructor, we use it to provide a
   base for the iterator class. */
void pkgPkgList::iterator::Die()
{
   delete I;
   I = 0;
}
									/*}}}*/
// PkgListOrdered::~PkgListOrdered - Destructor				/*{{{*/
// ---------------------------------------------------------------------
/* Provides a base for the class */
pkgPkgListOrdered::~pkgPkgListOrdered()
{
}
									/*}}}*/
// PkgCtrlInfo::operator [] - Return a ref to the package by name	/*{{{*/
// ---------------------------------------------------------------------
/* This never fails. It will return a pointer to an internal dummy 
   PkgCtrlInfo that contains no elements if a matching package is not
   found. This way expressions like: 
    List["gcc"]["Version"].Value();
   will return "" or the version string. Since the version string can't
   really be "" it's a valid test for error.
 */
const pkgPkgCtrlInfo &pkgPkgList::operator [](string Name) const
{
   static pkgPkgCtrlInfo Dummy;
   
   pkgSPkgCtrlInfo Item = Find(Name);
   if (!Item)
      return Dummy;
   return *Item;
}
									/*}}}*/

// TextElmReader::pkgTextElmReader - Constructor			/*{{{*/
// ---------------------------------------------------------------------
/* */
pkgTextElmReader::pkgTextElmReader(ifstream &F,string FileName) : 
                 F(F), File(FileName), CurLine(1)
{
   // Grab the first line.
   F.getline(Buffer,sizeof(Buffer));
   if (F.eof() != 0)
      return;
   if (!F != 0)
      _error->Error("Line %u in package file %s is too long.(1)",CurLine,File.c_str());
   CurLine++;
   SeekPos = 0;
}
									/*}}}*/
// TextElmReader::Read - Read a formatted text element from a file	/*{{{*/
// ---------------------------------------------------------------------
/* This is the core of the parser routine, it reads a single element from
   the file, with complete handling of multi line elements. */
bool pkgTextElmReader::Read(string &Element,vector<string> &Lines)
{
   if (F.eof() != 0)
      return false;
   
   Element = string();
   
   // Strip leading blanks
   char *S = _strstrip(Buffer);
   if (*S == 0)
   {
      // This is a blank line, grab the next line and return "" for the elm
      SeekPos = F.tellg();
      F.getline(Buffer,sizeof(Buffer));
      CurLine++;
      if (F.eof() != 0)
	 return true;
      Element = string();
      return true;
   }
   
   // Locate the : separating tag/value
   char *I;
   for (I = S; *I != 0 && *I != ':'; I++);
   if (*I == 0)
      return _error->Error("Line %u in package file %s has no :.",CurLine,File.c_str());
   
   // Split the string I is the value S is the tag
   *I = 0;
   I++;
   
   // Store the element
   Element = S;
   Lines.push_back(_strstrip(I));
   
   // Parse any multiline elements
   while (F.eof() == 0)
   {
      // Grab the next line.
      SeekPos = F.tellg();
      F.getline(Buffer,sizeof(Buffer));
      CurLine++;
      if (F.eof() != 0)
	 return true;
      
      if (!F != 0)
	 return _error->Error("Line %u in package file %s is too long.(2)",CurLine,File.c_str());
      
      /* If it is an empty line then we just skip over it and terminate
       multi-line processing. This also marks the end of the package
       definition */
      if (Buffer[0] == 0 || Buffer[0] == '\r')
      {
	 return true;
      }
      
      // Check if it is still part of the multiline
      if (isspace(Buffer[0]) == 0 && Buffer[0] != '.')
      {
	 break;
      }
      
      _strtabexpand(Buffer,sizeof(Buffer));

      // Strip the line again and use an empty string if its just a .
      for (I = Buffer; *I !=0 && (*I == ' ' || *I == '.'); I++);
      if (*I == '.' || *I == 0)
	 Lines.push_back(string());
      else
	 Lines.push_back(Buffer + 1);
   }
   
   return true;
}
									/*}}}*/
// TextElmReader::Seek - Jump to a specific line			/*{{{*/
// ---------------------------------------------------------------------
/* */
void pkgTextElmReader::Seek(int Line,streampos Offset)
{
   CurLine = Line; 
   F.clear();
   F.seekg(Offset);

   F.getline(Buffer,sizeof(Buffer));
   if (F.eof() != 0)
      return;
   if (!F != 0)
      _error->Error("Line %u in package file %s is too long.(3)",CurLine,File.c_str());
}
									/*}}}*/
// TextElmReader::Read - Read a single package entry			/*{{{*/
// ---------------------------------------------------------------------
/* */
bool pkgTextElmReader::Read(pkgSPkgCtrlInfo &Info)
{
   // Prime the ID table
   pkgElmRegister();
   
   Info.New();
   
   string Element;
   vector<string> Lines;
   while (Read(Element,Lines) == true && Element.empty() == false)
   {
      // Locate the element
      pkgElementId ID(Element);
      
      // No element found, build the default one
      if (ID == 0)
	 ID = pkgElementId(Element,pkgElmMultiLine::Newer);
      
      pkgElement *Elm = ID.Create();
      if (Elm->Value(Lines[0]) == false)
	 _error->Warning("Line %u in package file %s has an invalid value.",
			 CurLine,File.c_str());
      else
	 if (Info->Add(Elm) == false)
	    _error->Warning("Line %u in package file %s has dup tag %s.",
			    CurLine,File.c_str(),Elm);
      
      // Add in the lines
      Lines.erase(Lines.begin());
      if (Lines.empty() == false)
	 if (Elm->SetLines(Lines) == false)
	    _error->Warning("Tag at line %u in package file %s is not multiline.",
			    CurLine,File.c_str());
      Lines.erase(Lines.begin(),Lines.end());
   }
   return true;
}		    
									/*}}}*/

// pkgReadTxtList - Read a list of PkgCtrlInfo from a text file		/*{{{*/
// ---------------------------------------------------------------------
/* The text file format is fairly simple, it looks something like:
   Errors are returned via _error */
bool pkgReadTxtList(string File,pkgPkgList &List)
{
   // Open the stream for reading
   ifstream F(File.c_str(),ios::in | ios::nocreate);
   if (!F != 0)
      return _error->Errno("ifstream::ifstream","Opening %s",File.c_str());
   
   pkgTextElmReader Reader(F,File);
   pkgSPkgCtrlInfo Info;
   
   while (F.eof() == false)
   {
      Reader.Read(Info);
      List.Add(Info);
   }
	 
   return true;
}
									/*}}}*/
// pkgWriteTxtList - Write a list of PkgCtrlInfo to a text file		/*{{{*/
// ---------------------------------------------------------------------
/* This outputs the PkgList in a format that is readable by pkgReadTxtList */
bool pkgWriteTxtList(string File,pkgPkgList &List)
{
   // Open the stream for reading
   ofstream F(File.c_str(),ios::out | ios::trunc);
   if (!F != 0)
      return _error->Errno("ofstream::ofstream","Opening %s",File.c_str());
   
   vector<string> Lines;
   Lines.reserve(30);
   
   // Iterate over each package
   pkgPkgList::iterator I = List.begin();
   for (;I != List.end(); I++)
   {
      // Iterate over each element
      pkgPkgCtrlInfo::const_iterator Elm = (*I)->begin();
      for (; Elm != (*I)->end(); Elm++)
      {
	 // Write the tag: value
	 F << (*Elm)->Tag() << ": " << (*Elm)->Value() << endl;
	 
	 // Write the multiline
	 (*Elm)->GetLines(Lines);
	 for (vector<string>::iterator j = Lines.begin(); j != Lines.end(); j++)
	 {
	    if ((*j).length() == 0)
	       F << " ." << endl;
	    else
	       F << " " << *j << endl;
	 }
	 
	 Lines.erase(Lines.begin(),Lines.end());
      }
      
      F << endl;
   }
}
									/*}}}*/
