// -*- mode: cpp; mode: fold -*-
// Description								/*{{{*/
// $Id: pkgelement.h,v 1.14 1998/04/05 22:50:01 jgg Exp $
/* ######################################################################

   Package Element - A single tag/value pair that describes a portion of 
                     the package.
   Element Id - Identity of a Package Element. (Tag portion)
   Element String Value - String Package Element
   Element Integer Value - Integer Pacakge Element
   Element MD5Sum - MD5Sum Element
   Element Multi-Line - Multi line string
   Element Version - Element stored as a version
   Element Pacakge List - List of packages and required versions for each
   
   The Element Id acts as a cover to the underlying static ID String ->
   Class mechanism. When constructed the pkgElementId will search a
   static table of ID's to determine if it is unique. If it is not then
   a new internal representation will be constructed for it. Otherwise
   it will be atachted to the existing internal rep. 
   
   Two forms of constructors are provided, ones that take a 
   pkgElementId::NewFunc and ones that do not. The ones that do are used to
   create the instance of the Id struct, in essance defining the 
   Tag->Creation mapping.
   
   The Package Element acts as a super class to all the derived elements
   providing a basic set of functions. Derived element classes interact
   with the Element Id to ensure that the correct derived class is 
   constructued for any given Id.
   
   ##################################################################### */
									/*}}}*/
// Header section: pkglib
#ifndef PKGLIB_PKGELEMENT_H
#define PKGLIB_PKGELEMENT_H

#include <string>
#include <map>
#include <vector>
#include <pkglib/md5.h>
#include <pkglib/version.h>

class pkgElement;

class pkgElementId
{
   /* Internal representation of an element Id.
      If binary storage is ever done then a Tag integer will be needed here */
   typedef pkgElement *(* NewFunc)(pkgElementId ID);
   struct Id
   {
      string Tag;
      NewFunc Newer;
	 
      Id(string Tag,NewFunc Newer) : Tag(Tag), Newer(Newer) {};
      Id() : Newer(0) {};
   };
   
   // Static Translation table
   static map<string,Id *,less<string> > IdMap;
   Id *ID;

   public:   

   // Comparitors
   inline bool operator ==(const pkgElementId &rhs) const {return ID == rhs.ID;};
   inline bool operator ==(int rhs) const {return rhs == 0 && ID == 0;};

   // Accessors
   inline string Tag() const {return ID == 0?string():ID->Tag;};
   inline bool Valid() const {return ID != 0;};
   inline pkgElement *Create() const {return ID->Newer(*this);};
      
   // Constructors
   pkgElementId() : ID(0) {};
   pkgElementId(const char *Name);
   pkgElementId(string Name);
   pkgElementId(string Name,NewFunc Newer);
   inline pkgElementId(const pkgElementId &rhs) : ID(rhs.ID) {};
};

class pkgElement
{
   protected:
   
   // Unique by type ID, substitute for the tag string
   pkgElementId ID;

   public:
   
   // ID handling
   inline string Tag() const {return ID.Tag();};
   inline bool operator ==(const pkgElementId &rhs) const {return rhs == ID;};
   inline bool operator !=(const pkgElement &rhs) const {return !(rhs == ID);};
   inline operator pkgElementId() const {return ID;};

   // General Accessors
   virtual bool operator ==(const pkgElement &rhs) const = 0;
   virtual string Value() const = 0;
   virtual bool Value(string Val) = 0;
   virtual void GetLines(vector<string> &/*Lines*/) const {};
   virtual bool SetLines(const vector<string> &/*Lines*/) {return false;};
   
   // Flags
   virtual bool Mutable() const {return true;};
         
   pkgElement(pkgElementId ID) : ID(ID) {};
   virtual ~pkgElement();
};

class pkgElmStringVal : public pkgElement
{   
   string IValue;
   public:

   static pkgElement *Newer(pkgElementId ID) {return new pkgElmStringVal(ID);};

   
   // General Accessors
   virtual bool operator ==(const pkgElement &rhs) const 
           {return rhs == ID && ((const pkgElmStringVal *)&rhs)->IValue == IValue;};
   virtual string Value() const {return IValue;};
   virtual bool Value(string Val) {IValue = Val;return true;};
   
   pkgElmStringVal(pkgElementId ID);
};

class pkgElmIntegerVal : public pkgElement
{   
   signed long IValue;
   public:

   static pkgElement *Newer(pkgElementId ID) {return new pkgElmIntegerVal(ID);};
   
   // General Accessors
   virtual bool operator ==(const pkgElement &rhs) const 
           {return rhs == ID && ((const pkgElmIntegerVal *)&rhs)->IValue == IValue;};
   virtual string Value() const;
   virtual bool Value(string Val);
   
   pkgElmIntegerVal(pkgElementId ID);
};

class pkgElmPackage : public pkgElmStringVal
{   
   public:

   static pkgElement *Newer(pkgElementId /*ID*/) {return new pkgElmPackage;};
   
   // Only want to disable mutation
   bool Mutable() const {return false;};
   
   pkgElmPackage();
};

class pkgElmMD5Sum : public pkgElement
{
   MD5SumValue IValue;
   public:
   
   static pkgElement *Newer(pkgElementId ID) {return new pkgElmMD5Sum(ID);};
   
   // General Accessors
   virtual bool operator ==(const pkgElement &rhs) const 
           {return rhs == ID && ((const pkgElmMD5Sum *)&rhs)->IValue == IValue;};
   virtual string Value() const {return IValue;};
   virtual bool Value(string Val) {return IValue.Set(Val);};
   
   pkgElmMD5Sum(pkgElementId ID);
};

class pkgElmMultiLine : public pkgElement
{
   string SValue;
   vector<string> IValue;
   public:
   
   static pkgElement *Newer(pkgElementId ID) {return new pkgElmMultiLine(ID);};
   
   // General Accessors
   virtual bool operator ==(const pkgElement &rhs) const 
           {return rhs == ID && ((const pkgElmMultiLine *)&rhs)->SValue == SValue;};
   virtual string Value() const {return SValue;};
   virtual bool Value(string Val) {SValue = Val;return true;};

   // Line accessors
   virtual void GetLines(vector<string> &Lines) const {Lines = IValue;};
   virtual bool SetLines(const vector<string> &Lines) {IValue = Lines; return true;};

   pkgElmMultiLine(pkgElementId ID);
};

class pkgElmVersion : public pkgElement
{
   pkgVersion IValue;
   public:
   
   static pkgElement *Newer(pkgElementId ID) {return new pkgElmVersion(ID);};
   
   // General Accessors
   virtual bool operator ==(const pkgElement &rhs) const 
           {return rhs == ID && ((const pkgElmVersion *)&rhs)->IValue == IValue;};
   virtual string Value() const {return IValue;};
   virtual bool Value(string Val) {IValue = Val;return true;};
   
   pkgElmVersion(pkgElementId ID);
};

/* Operator flags for pkgElmPkgList::Item::Operator, should fit in 8 bits
   for cache */
#define pkgOP_OR 0x10
#define pkgOP_LESSEQ 0x1
#define pkgOP_GREATEREQ 0x2
#define pkgOP_LESS 0x3
#define pkgOP_GREATER 0x4
#define pkgOP_EQUALS 0x5
#define pkgOP_NOTEQUALS 0x6

class pkgElmPkgList : public pkgElement
{
   public:
   
   struct Item
   {
      string Package;
      pkgVersion Version;
      int Operator;
      Item() : Operator(0) {};
   };
   vector<Item> List;
      
   static pkgElement *Newer(pkgElementId ID) {return new pkgElmPkgList(ID);};
   
   // General Accessors
   virtual bool operator ==(const pkgElement &rhs) const;
   virtual string Value() const;
   virtual bool Value(string Val);
   virtual bool SetLines(const vector<string> &Lines);
   
   pkgElmPkgList(pkgElementId ID);
};

// Register the usual elements
void pkgElmRegister();

#endif
