// -*- mode: cpp; mode: fold -*-
// Description								/*{{{*/
// $Id: pkgcachemerge.cc,v 1.33 1998/05/18 20:48:56 jgg Exp $
/* ######################################################################
   
   Package Cache Merge - Generator code for the package cache.
   
   Please see doc/pkglib/cache.sgml for a more detailed description of 
   this format. Also be sure to keep that file up-to-date!!
   
   This module defines the entire emitter/merger code for the package 
   cache.
   
   The public entry point is the MergePackageFile function.
   
   ##################################################################### */
									/*}}}*/
// Include Files							/*{{{*/
#include <iostream.h>
#include <fstream.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <math.h>

#include <system.h>
#include <options.h>
#include <pkglib/error.h>
#include <pkglib/pkglist.h>
#include <pkglib/pkgelement.h>
#include <pkglib/mmap.h>
#include <pkglib/pkgcachemerge.h>
									/*}}}*/

// Cache::MergeState::MergeState - Constructor				/*{{{*/
// ---------------------------------------------------------------------
/* The constructor opens the file and prepares for merging. Errors
   are reported through _error. */
pkgCache::MergeState::MergeState(string FName) : FileName(FName)
{
   Fd = open(FileName.c_str(),O_CREAT | O_RDWR,S_IRGRP | S_IWGRP | S_IRUSR | S_IWUSR | S_IROTH);
   if (Fd < 0)
   {
      _error->Errno("open","Can't open %s",FileName.c_str());
      return;
   }
   CurOffset = lseek(Fd,0,SEEK_END);

   // Check the header
   if (CurOffset != 0)
   {
      bool Okay = false;
      while (1)
      {
	 // Read the header
	 Header Head;
	 lseek(Fd,0,SEEK_SET);
	 if (read(Fd,&Head,sizeof(Head)) != sizeof(Head))
	    break;

	 // Check the signature
	 Header DefHeader;
	 if (Head.Signature != DefHeader.Signature ||
	     Head.Dirty == true ||
	     Head.MajorVersion != DefHeader.MajorVersion ||
	     Head.MinorVersion != DefHeader.MinorVersion ||
	     Head.CheckSizes(DefHeader) == false)
	    break;
	 
	 Okay = true;
	 break;
      }
      
      // The cache file is corrupted
      if (Okay == false)
      {
	 close(Fd);
	 _error->Warning("The old cache file '%s' was erased, it was corrupt or a non-matching version.",FileName.c_str());
	 if (unlink(FileName.c_str()) != 0)
	 {
	    _error->Errno("unlink","Unable to erase the old cache file '%s'.",FileName.c_str());
	    return;
	 }
	 Fd = open(FileName.c_str(),O_CREAT | O_RDWR,S_IRGRP | S_IWGRP | S_IRUSR | S_IWUSR | S_IROTH);
	 if (Fd < 0)
	 {
	    _error->Errno("open","Can't open %s",FileName.c_str());
	    return;
	 }
	 CurOffset = lseek(Fd,0,SEEK_END);
	 if (CurOffset != 0)
	 {
	    _error->Error("Empty cache file is not empty");
	    return;
	 }
      }
   }
   
   if (CurOffset == 0)
   {
      Head.Dirty = true;
      ewrite(&Head,sizeof(Head));
      CurOffset = sizeof(Head);

      // Allocate space pools now
      if (sizeof(Version) == sizeof(Package))
	  AllocatePool(sizeof(Package),2*2600);
      else
	  AllocatePool(sizeof(Package),2600);
      AllocatePool(sizeof(PackageFile),15);
      AllocatePool(sizeof(Version),(long)(1.3*2100));
      AllocatePool(sizeof(Dependency),7000);
      AllocatePool(sizeof(Provides),300);
      AllocatePool(sizeof(StringItem),400);
   }
   else
   {
      eseek(0);
      eread(&Head,sizeof(Head));
      Head.Dirty = true;
      eseek(0);
      ewrite(&Head,sizeof(Head));
      eseek(0);

      MMap Map(Fd,true,true);
      if (_error->PendingError() == true)
	 return;

      // Setup structure pointers
      const pkgCache::Header *Header = (pkgCache::Header *)Map.Data();
      const pkgCache::Package *Package = (pkgCache::Package *)Map.Data();
      const pkgCache::StringItem *StrItem = (pkgCache::StringItem *)Map.Data();
      const char *String = (char *)Map.Data();

      // Parse the package list into the map
      for (int I = 0; I != _count(Header->HashTable); I++)
      {
	 unsigned long CurPackage = Header->HashTable[I];
	 for (;CurPackage != 0; CurPackage = Package[CurPackage].NextPackage)
	    NameMap.insert(NameMap_t::value_type(String + Package[CurPackage].Name,
						 CurPackage));
      }

      // Parse the unique string list into the map
      for (unsigned long CurStr = Header->StringList;CurStr != 0;
	   CurStr = StrItem[CurStr].NextItem)
      {
	 StringMap.insert(NameMap_t::value_type(String +
			  StrItem[CurStr].String,StrItem[CurStr].String));
      }
      
      eseek(CurOffset);
   }
}
									/*}}}*/
// Cache::MergeState::~MergeState - Destructor				/*{{{*/
// ---------------------------------------------------------------------
/* Rewrite the header and close the file */
pkgCache::MergeState::~MergeState()
{
   if (Fd == -1)
      return;
   
   eseek(0);
   if (_error->PendingError() == true)
      Head.Dirty = true;
   else
      Head.Dirty = false;
   
   ewrite(&Head,sizeof(Head));
   close(Fd);   

   if (_error->PendingError() == true)
      return;
   
   // Now we reopen the file and perform version sorting
   pkgCache Cache(FileName,true,false);
   if (_error->PendingError() == false)
   {
      Cache.HeaderP->Dirty = true;
      Cache.Sync();
      SortVersions(Cache);
      GenReverseDepends(Cache);
      Cache.HeaderP->Dirty = false;
   }
}
									/*}}}*/
// Cache::MergeState::AllocatePool - Allocate a pool from the end	/*{{{*/
// ---------------------------------------------------------------------
/* This allocates a new space pool from the end of the file. Space pools
   are used to store records of similar size so that no space is wasted
   to alignment (my tests showed ~50% waste) A fixed size block of file
   is set aside for the use of objects of that size. When that block
   gets filled a new block is allocated at the end. */
int pkgCache::MergeState::AllocatePool(unsigned long Size,int Number)
{
   // Find a blank slot
   int I;
   for (I = 0; I != _count(Head.PoolStart); I++)
      if (Head.PoolAln[I] == 0 || Head.PoolAln[I] == Size)
	 break;
   
   // Couldnt find a blank spot, bail
   if (I == _count(Head.PoolStart))
   {
      _error->Error("Allocate Pool failed, this is bad cache is corrupted");
      return 0;
   }
   
   // Already allocated, do nothing.
   if (Head.PoolSize[I] != 0)
      return I;
   
   Head.PoolStart[I] = (int)(ceil((float)CurOffset/(float)Size)*Size);
   Head.PoolSize[I] = Size*Number;
   Head.PoolAln[I] = Size;
   CurOffset = Head.PoolStart[I] + Head.PoolSize[I];
   eseek(CurOffset);
   
   /* We make sure that the file is grown to the new length, otherwise
      trouble occures if we reload it and start writing */
   char Jnk = 0;
   read(Fd,&Jnk,sizeof(Jnk));
   eseek(CurOffset);
   ewrite(&Jnk,sizeof(Jnk));
   eseek(CurOffset);
	
   return I;
}
									/*}}}*/
// Cache::MergeState::eseek - lseek with error reporting		/*{{{*/
// ---------------------------------------------------------------------
/* */
unsigned long pkgCache::MergeState::eseek(unsigned long Pos)
{
   off_t Res = lseek(Fd,Pos,SEEK_SET);
   if (Res < 0 || (signed)Pos != Res)
   {
      _error->Errno("lseek","Error generating cache file %u",Pos);
      return 0;
   }
   return Res;
}
									/*}}}*/
// Cache::MergeState::eread - read with error reporting			/*{{{*/
// ---------------------------------------------------------------------
/* */
bool pkgCache::MergeState::eread(void *Data,unsigned long Size)
{
   if ((signed)Size != read(Fd,Data,Size))
      return _error->Errno("read","Error generating cache file");
   return true;
}
									/*}}}*/
// Cache::MergeState::ewrite - write with error reporting		/*{{{*/
// ---------------------------------------------------------------------
/* */
bool pkgCache::MergeState::ewrite(const void *Data,unsigned long Size)
{
   if ((signed)Size != write(Fd,Data,Size))
      return _error->Errno("write","Error generating cache file");
   return true;
}
									/*}}}*/
// Cache::MergeState::elwrite - Aligned write with error reporting	/*{{{*/
// ---------------------------------------------------------------------
/* This looks for an allocation pool of matching size and appends
   the given record to the end of the allocation. If the record of
   matching size is full then a new pool of 400 items will
   be allocated. Bad evil stuff happens if there is no pool of
   matching size is found. */
unsigned long pkgCache::MergeState::elwrite(const void *Data,
					    unsigned long Size)
{
   // Attempt to use space from the allocation pools
   int I;
   for (I = 0; I != _count(Head.PoolSize); I++)
      if (Head.PoolAln[I] == Size && Head.PoolSize[I] != 0)
	 break;

   // Allocate a new pool
   if (I == _count(Head.PoolSize))
      I = AllocatePool(Size,400);

   // Write the block into the pool
   unsigned long TargOff = Head.PoolStart[I];
   eseek(TargOff);
   ewrite(Data,Size);
   eseek(CurOffset);

   Head.PoolStart[I] += Size;
   Head.PoolSize[I] -= Size;
   
   return TargOff/Size;
}
									/*}}}*/
// Cache::MergeState::EmitString - Write a string into the cache	/*{{{*/
// ---------------------------------------------------------------------
/* This places a string into the cache file at the current location,
   which really should be at the end. It is up to the rest of the
   code to ensure any seeks always reset to CurOffset. */
unsigned long pkgCache::MergeState::EmitString(string Str)
{
   if (Str.empty() == true)
      return 0;
   
   off_t Off = CurOffset;
   const char *S = Str.c_str();
   if (ewrite(S,Str.length() + 1) == false)
      return 0;
   CurOffset += Str.length() + 1;
   
   return Off;
}
									/*}}}*/
// Cache::MergeState::EmitUniqueString - Write a unique string		/*{{{*/
// ---------------------------------------------------------------------
/* */
unsigned long pkgCache::MergeState::EmitUniqueString(string Str)
{
   // Attempt to locate the package in the map
   pair<NameMap_t::iterator, bool> Item;
   Item = StringMap.insert(NameMap_t::value_type(Str,CurOffset));
   if (Item.second == true)
   {
      StringItem Itm;
      (*Item.first).second = Itm.String = EmitString(Str);
      Itm.NextItem = Head.StringList;
      Head.StringList = elwrite(&Itm,sizeof(Itm));
   }
   
   return (*Item.first).second;
}
									/*}}}*/
// Cache::MergeState::SetVersion - Select the current version		/*{{{*/
// ---------------------------------------------------------------------
/* This locates or creates a new version for the current package. It 
   returns true if the version was found, false if it was created (or 
   error). 
 
   Unfortunately searching the file is pretty unadvoidable here. I think 
   this particular routine is going to be slow.. */
bool pkgCache::MergeState::SetVersion(string Version,Version &CurVersion,
	      unsigned long &VerOffset,unsigned long FileOffset)
{
   unsigned long Cur = CurPackage.VersionList;
   for (; Cur != 0;)
   {
      // Read the version record
      pkgCache::Version Tmp;
      eseek(Cur*sizeof(Tmp));
      eread(&Tmp,sizeof(Tmp));
      eseek(Tmp.VerStr);
      if (_error->PendingError() == true)
      {
	 eseek(CurOffset);
	 return false;
      }
      
      // Read the string, versions should not exceed 300 chars..
      char S[300];
      int Res = read(Fd,S,sizeof(S)-1);
      if (Res <= 0)
      {   
	 eseek(CurOffset);
	 return false;
      }
      S[Res] = 0;
      
      // Check for match
      if (Version == S)
      {
	 eseek(CurOffset);
	 VerOffset = Cur;
	 CurVersion = Tmp;
	 if (Tmp.File == 0)
	    break;
	 return true;
      }
      
      Cur = Tmp.NextVer;
   }  

   eseek(CurOffset);
   
   if (_error->PendingError() == true)
      return false;

   // Emit a new version record
   memset(&CurVersion,0,sizeof(CurVersion));
   CurVersion.VerStr = EmitString(Version);
   CurVersion.NextVer = CurPackage.VersionList;
   CurVersion.File = FileOffset;
   CurVersion.ParentPkg = PkgOffset;
   
   VerOffset = elwrite(&CurVersion,sizeof(CurVersion));
   CurPackage.VersionList = VerOffset;
   
   return false;
}  
									/*}}}*/
// Cache::MergeState::GetPackageIndex - Return the index for a package	/*{{{*/
// ---------------------------------------------------------------------
/* If a package does not exist then a new package will be created. The
   package map is pre-loaded when an existing file is opened to make
   package look up very fast. */
unsigned long pkgCache::MergeState::GetPackageIndex(string Name,Package &Pack,bool Read)
{
   // Attempt to locate the package in the map
   pair<NameMap_t::iterator, bool> Pkg;
   Pkg = NameMap.insert(NameMap_t::value_type(Name,0));
   if (Pkg.second == true)
   {
      // Initialize the package structure
      memset(&Pack,0,sizeof(Pack));
      Pack.Name = EmitString(Name);
      Pack.Section = EmitUniqueString("Unknown");
	 
      /* Insert the package structure into the linked list
         First we generate our hash value for the name and
       	 insert into the HashTable */
      unsigned long HashValue = Hash(Name);

      // Link it in
      Pack.NextPackage = Head.HashTable[HashValue];

      // Aquire space for the package structure
      (*Pkg.first).second = elwrite(&Pack,sizeof(Pack));
  
      // Store the new first pointer
      Head.HashTable[HashValue] = (*Pkg.first).second;
   }
   else
   {
      if (Read == false)
	 return (*Pkg.first).second;
      
      eseek((*Pkg.first).second*sizeof(Pack));
      eread(&Pack,sizeof(Pack));
      eseek(CurOffset);
   }
   
   return (*Pkg.first).second;
}
									/*}}}*/
// Cache::MergeState::ProcessDepends - Add a dependency record		/*{{{*/
// ---------------------------------------------------------------------
/* We make use of the pkgElmPkgList package list parser to process the
   dependencies. */
bool pkgCache::MergeState::ProcessDepends(vector<string> &Lines,
					  unsigned char Type)
{
   pkgElementId Id;
   pkgElmPkgList List(Id);
   string First = Lines[0];
   Lines.erase(Lines.begin());
   if (List.Value(First) == false || List.SetLines(Lines) == false)
      return _error->Error("Package file %s line %u has a malformed depends line",PkgFileName.c_str(),TopLine);
   
   for (pkgElmPkgList::Item *I = List.List.end(); I != List.List.begin();)
   {
      I--;
      Dependency Dep;
      Package Pack;
      Dep.Version = EmitString(I->Version);
      Dep.Package = GetPackageIndex(I->Package,Pack,false);
      Dep.NextDepends = CurVersion.DependsList;
      Dep.Type = Type;
      Dep.CompareOp = I->Operator;
      Dep.ParentVer = VerOffset;
      
      if (Dep.Type == pkgDEP_Conflicts && (Dep.CompareOp & pkgOP_OR) != 0)
	 return _error->Error("Package file %s line %u has a malformed conflicts line",PkgFileName.c_str(),TopLine);
      
      CurVersion.DependsList = elwrite(&Dep,sizeof(Dep));
   }
   return true;
}
									/*}}}*/
// Cache::MergeState::ProcessProvides -	Add provides records		/*{{{*/
// ---------------------------------------------------------------------
/* This scans the provides line and adds entries for the virtual package
   names into the provides table that point to the current package. 
 
 TODO: Handle versions */
bool pkgCache::MergeState::ProcessProvides(vector<string> &Lines)
{
   pkgElementId Id;
   pkgElmPkgList List(Id);
   string First = Lines[0];
   Lines.erase(Lines.begin());
   if (List.Value(First) == false || List.SetLines(Lines) == false)
      return _error->Error("Package file %s line %u has a malformed providess line",PkgFileName.c_str(),TopLine);
   
   for (pkgElmPkgList::Item *I = List.List.begin(); I != List.List.end(); I++)
   {
      // This goes in provide version someday
      if (string(I->Version).empty() != true)
	 return _error->Error("Package file %s line %u has a malformed providess line",PkgFileName.c_str(),TopLine);
	 
      Provides Prov;
      memset(&Prov,0,sizeof(Prov));
      Prov.Version = VerOffset;
      Prov.NextPkgProv = CurVersion.ProvidesList;

      /* This should never really happen, a package providing itself is
       	 silly. Though it would have meaning with provide versions. */
      Package Targ;
      if ((Prov.ParentPkg = GetPackageIndex(I->Package,Targ,true)) == PkgOffset)
      {
//	 _error->Warning("Package file %s line %u package %s has a self referencing provides, ignoring.",PkgFileName.c_str(),TopLine,I->Package.c_str());
	 return true;
      }
      
      Prov.NextProvides = Targ.ProvidesList;
      CurVersion.ProvidesList = Targ.ProvidesList = elwrite(&Prov,sizeof(Prov));

      // Write the new package structure
      off_t OldOff = CurOffset;
      eseek(Prov.ParentPkg*sizeof(Targ));
      ewrite(&Targ,sizeof(Targ));
      eseek(OldOff);
   }
   return true;
}
									/*}}}*/
// Cache::MergeState::ProcessTargetVersion - TargetVerion field		/*{{{*/
// ---------------------------------------------------------------------
/* This decodes the new TargetVersion status field. The format of the 
   field is
       TrgetVersion: version (dist)
    version = version string
    dist = name of distribution to match against (see cache.sgml)
 */
bool pkgCache::MergeState::ProcessTargetVersion(string Ver)
{
   // Isolate the first word
   string::const_iterator I = Ver.begin();
   string::const_iterator Start = I;
   for(; I != Ver.end() && *I != ' '; I++);
   if (I < Ver.end() && *I != ' ')
      return _error->Error("Malformed target version line at line %u",TopLine);

   Version Jnk;
   SetVersion(string(Start,I - Start),Jnk,CurPackage.TargetVer,0);
	      
   I++;
   if (I >= Ver.end())
      return true;
   
   if (*I != '(')
      return _error->Error("Malformed target version line at line %u",TopLine);

   I++;
   for(Start = I; I < Ver.end() && *I != ')'; I++);
   if (I >= Ver.end() || *I != ')')
      return _error->Error("Malformed target version line at line %u",TopLine);
   CurPackage.TargetDist = EmitUniqueString(string(Start,I - Start));
   
   return true;
}
									/*}}}*/
// Cache::MergeState::ProcessStatus - Process a status line		/*{{{*/
// ---------------------------------------------------------------------
/* Status lines are of the form,
     Status: want flag status
   want = unknown, install, hold, deinstall, purge
   flag = ok, reinstreq, hold, hold-reinstreq
   status = not-installed, unpacked, half-configured, uninstalled,
            half-installed, config-files, post-inst-failed, 
            removal-failed, installed
   
   Some of the above are obsolete (I think?) flag = hold-* and 
   status = post-inst-failed, removal-failed at least.
 */
bool pkgCache::MergeState::ProcessStatus(string State)
{
   // Isolate the first word
   string::const_iterator I = State.begin();
   string::const_iterator Start = I;
   for(; I != State.end() && *I != ' '; I++);
   if (I >= State.end() || *I != ' ')
      return _error->Error("Malformed status line at line %u",TopLine);

   // Process the want field
   WordList WantList[] = {{"unknown",pkgSTATE_Unkown},
                          {"install",pkgSTATE_Install},
                          {"hold",pkgSTATE_Hold},
                          {"deinstall",pkgSTATE_DeInstall},
                          {"purge",pkgSTATE_Purge}};
   if (GrabWord(string(Start,I-Start),WantList,
		_count(WantList),CurPackage.SelectedState) == false)
      return _error->Error("Malformed status line at line %u",TopLine);

   // Isloate the next word
   I++;
   Start = I;
   for(; I < State.end() && *I != ' '; I++);
   if (I >= State.end() || *I != ' ')
      return _error->Error("Malformed status line at line %u",TopLine);

   // Process the flag field
   WordList FlagList[] = {{"ok",pkgSTATE_Ok},
                          {"reinstreq",pkgSTATE_ReInstReq},
                          {"hold",pkgSTATE_Hold},
                          {"hold-reinstreq",pkgSTATE_HoldReInstReq}};
   if (GrabWord(string(Start,I-Start),FlagList,
		_count(FlagList),CurPackage.InstState) == false)
      return _error->Error("Malformed status line at line %u",TopLine);

   // Isloate the last word
   I++;
   Start = I;
   for(; I < State.end() && *I != ' '; I++);
   if (I != State.end())
      return _error->Error("Malformed status line at line %u",TopLine);

   // Process the flag field
   WordList StatusList[] = {{"not-installed",pkgSTATE_NotInstalled},
                            {"unpacked",pkgSTATE_UnPacked},
                            {"half-configured",pkgSTATE_HalfConfigured},
                            {"installed",pkgSTATE_Installed},
                            {"uninstalled",pkgSTATE_UnInstalled},
                            {"half-installed",pkgSTATE_HalfInstalled},
                            {"config-files",pkgSTATE_ConfigFiles},
                            {"post-inst-failed",pkgSTATE_HalfConfigured},
                            {"removal-failed",pkgSTATE_HalfInstalled}};
   if (GrabWord(string(Start,I-Start),StatusList,
		_count(StatusList),CurPackage.CurrentState) == false)
      return _error->Error("Malformed status line at line %u",TopLine);

   /* A Status line marks the package as indicating the current
      version as well. Only if it is actually installed.. Otherwise
      the interestingd dpkg handling of the status file creates bogus 
      entries. */
   if (!(CurPackage.CurrentState == pkgSTATE_NotInstalled ||
       CurPackage.CurrentState == pkgSTATE_ConfigFiles))
      CurPackage.CurrentVer = VerOffset;
   
   return true;
}
									/*}}}*/
// Cache::MergeState::GrabWord - Matches a word and returns		/*{{{*/
// ---------------------------------------------------------------------
/* */
bool pkgCache::MergeState::GrabWord(string Word, WordList *List, int Count,
				    unsigned char &Out)
{
   for (int C = 0; C != Count; C++)
   {
      if (Word == List[C].Str)
      {
	 Out = List[C].Val;
	 return true;
      }
   }
   return false;
}
									/*}}}*/
// Cache::MergeState::MergePacakge - Emit an entire package		/*{{{*/
// ---------------------------------------------------------------------
/* This oversees the merging of a single package enitity from the text
   file. It should be called until the file is empty. */
bool pkgCache::MergeState::MergePackage(pkgTextElmReader &Reader)
{
   string Element;
   unsigned long Offset = Reader.Position();
   TopLine = Reader.Line();
   
   Lines.erase(Lines.begin(),Lines.end());
   Reader.Read(Element,Lines);

   // Hmm, no package line, not good
   if (!(Element == "Package"))
      return _error->Error("Package file %s line %u does not start a package",PkgFileName.c_str(),Reader.Line());

   // Locate the package
   PkgOffset = GetPackageIndex(Lines[0],CurPackage);
   
   // Hm, error has been queued up
   if (_error->PendingError() == true)
      return false;

   // Search for the version string
   Lines.erase(Lines.begin(),Lines.end());
   string VerStr;
   string Arch;
   while (Reader.Read(Element,Lines) == true && Element.empty() == false)
   {
      if (Element == "Version")
	 VerStr = Lines[0];
      if (Element == "Architecture")
	 Arch = Lines[0];
      
      Lines.erase(Lines.begin(),Lines.end());
   }

   // Check if the arch is valid
   if (Arch.empty() == false && !(Arch == "all") && !(Arch == PKG_DEB_ARCH))
      return true;
   
   // Jump back to the top of the package and go over again
   Reader.Seek(TopLine,Offset);

   // Hmm, no version line..
   bool HaveVer = false;
   if (VerStr.empty() == true)
   {
      VerOffset = 0;
      HaveVer = true;
   }
   else
   {
      // Grab the version
      if (SetVersion(VerStr,CurVersion,VerOffset,FileOffset) == true)
	 HaveVer = true;
      else
      {
	 // Store the pacakge file offset
	 CurVersion.Offset = Offset;
	 HaveVer = false;
      }      
   }
   
   // Scan the rest of the package record
   while (true)
   {
      Lines.erase(Lines.begin(),Lines.end());
      if (Reader.Read(Element,Lines) == false || Element.empty() == true)
	 break;

      if (Element == "Status")
      {
	 ProcessStatus(Lines[0]);
	 continue;
      }
      
      if (Element == "TargetVersion")
      {
	 ProcessTargetVersion(Lines[0]);
	 continue;
      }

      if (HaveVer == true)
	 continue;

      if (Element == "Essential")
      {
	 if (Lines[0] == "yes")
	    CurPackage.Flags |= pkgFLAG_Essential;
	 continue;
      }

      if (Element == "Section")
      {
	 CurVersion.Section = EmitUniqueString(Lines[0]);
	 CurPackage.Section = CurVersion.Section;
	 continue;
      }

      if (Element == "Installed-Size" || Element == "installed-size")
      {
	 CurVersion.InstalledSize = atoi(Lines[0].c_str())*1000;
	 continue;
      }
    
      if (Element == "Size")
      {
	 CurVersion.Size = atoi(Lines[0].c_str());
	 continue;
      }
      
      if (Element == "Depends")
      {
	 ProcessDepends(Lines,pkgDEP_Depends);
	 continue;
      }

      if (Element == "Suggests")
      {
	 ProcessDepends(Lines,pkgDEP_Suggests);
	 continue;
      }

      if (Element == "Recommends")
      {
	 ProcessDepends(Lines,pkgDEP_Recommends);
	 continue;
      }

      if (Element == "Conflicts")
      {
	 ProcessDepends(Lines,pkgDEP_Conflicts);
	 continue;
      }

      if (Element == "Pre-Depends")
      {
	 ProcessDepends(Lines,pkgDEP_PreDepends);
	 continue;
      }
      
      if (Element == "Replaces")
      {
	 ProcessDepends(Lines,pkgDEP_Replaces);
	 continue;
      }
      
      if (Element == "Provides")
      {
	 ProcessProvides(Lines);
	 continue;
      }
      
      if (Element == "Priority")
      {
	 if (Lines[0] == "important")
	    CurVersion.Priority = pkgPRIO_Important;
	 if (Lines[0] == "required")
	    CurVersion.Priority = pkgPRIO_Required;
	 if (Lines[0] == "standard")
	    CurVersion.Priority = pkgPRIO_Standard;
	 if (Lines[0] == "optional")
	    CurVersion.Priority = pkgPRIO_Optional;
	 if (Lines[0] == "extra")
	    CurVersion.Priority = pkgPRIO_Extra;
	 
	 continue;
      }      
   }

   // Finish writing the package and version.
   off_t OldOff = CurOffset;
   eseek(PkgOffset*sizeof(CurPackage));
   ewrite(&CurPackage,sizeof(CurPackage));

   if (VerOffset != 0)
   {
      eseek(VerOffset*sizeof(CurVersion));
      ewrite(&CurVersion,sizeof(CurVersion));
   }
   
   eseek(OldOff);
   
   if (_error->PendingError() == true)
      return false;
   return true;
}
									/*}}}*/
// Cache::MergeState::SelectFile - Select which file is being merged	/*{{{*/
// ---------------------------------------------------------------------
/* This should be called before MergePackage to select the target. It 
   will automatically stat the file to determine the rest of the 
   needed info */
bool pkgCache::MergeState::SelectFile(string File,string Dist,string Version,
				      unsigned long Flags)
{
   struct stat Buf;
   if (stat(File.c_str(),&Buf) == -1)
      return _error->Errno("stat","Couldn't stat ",File.c_str());
   
   PackageFile F;
   F.FileName = EmitString(File);
   F.Distribution = EmitString(Dist);
   F.Version = EmitString(Version);
   F.Size = Buf.st_size;
   F.mtime = Buf.st_mtime;
   F.NextFile = Head.FileList;
   F.Flags = Flags;
   
   FileOffset = Head.FileList = elwrite(&F,sizeof(F));
   PkgFileName = File;
}
									/*}}}*/
// Cache::MergeState::MergeFile - Merge in a single file		/*{{{*/
// ---------------------------------------------------------------------
/* All errors are propogated via the GlobalError class. */
bool pkgCache::MergeState::MergePackageFile(string FileName,string Dist,
					    string Revision,unsigned long Flags)
{
   // Open the stream for reading
   ifstream F(FileName.c_str(),ios::in | ios::nocreate);
   if (!F != 0)
      return _error->Errno("ifstream::ifstream","Opening %s",FileName.c_str());

   SelectFile(FileName,Dist,Revision,Flags);
   
   // Read all the packages in
   pkgTextElmReader Reader(F,FileName);
   while (F.eof() == 0)
      if (MergePackage(Reader) != true)
	 return false;
   
   return true;
}
									/*}}}*/
// Cache::MergeState::SortVersions - Put the version lists in order	/*{{{*/
// ---------------------------------------------------------------------
/* This goes through all packages and sorts all of the version lists
   so that they are from highest version to lowest version. It is 
   performed during closing of the cache. There can be at most
   100 versions for simplicity of this code */
static pkgCache *CompCache = 0;
static int VerComp(const void *A,const void *B)
{
   unsigned long lhs = *(unsigned long *)A;
   unsigned long rhs = *(unsigned long *)B;
   int Res = pkgVersionCompare(CompCache->StrP + CompCache->VerP[lhs].VerStr,
			       CompCache->StrP + CompCache->VerP[rhs].VerStr);
   if (Res == 0)
      _error->Warning("Detected duplicate version");
   return Res;
}

bool pkgCache::MergeState::SortVersions(pkgCache &Cache)
{
   // Iterate over the hash table
   for (int I = 0; I != _count(Cache.HeaderP->HashTable); I++)
   {
      // Iterate over the bucket
      unsigned long Pkg = Cache.HeaderP->HashTable[I];
      for (;Pkg != 0; Pkg =  Cache.PkgP[Pkg].NextPackage)
      {
	 // Copy the version pointers into VerList
	 unsigned long VerList[100];
	 int Count = 0;
	 unsigned long Ver = Cache.PkgP[Pkg].VersionList;
	 for (;Ver != 0; Ver = Cache.VerP[Ver].NextVer)
	 {
	    VerList[Count] = Ver;
	    Count++;
	 }
	 
	 // No point
	 if (Count <= 1)
	    continue;
	 CompCache = &Cache;
	 qsort(VerList,Count,sizeof(VerList[0]),VerComp);
	 
	 // Rewrite the linked list
	 Cache.PkgP[Pkg].VersionList = VerList[Count-1];
	 for (int J = Count-1; J != 0; J--)
	    Cache.VerP[VerList[J]].NextVer = VerList[J-1];
	 Cache.VerP[VerList[0]].NextVer = 0;
      }
   }

   return true;
}
									/*}}}*/
// Cache::MergeState::GenReverseDepends - Generate the reverse dep list	/*{{{*/
// ---------------------------------------------------------------------
/* This is done here because generating the list while processing
   the package files would involve alot of extra processing. It
   is simple and safe to produce the tables after sorting the
   version list. We also use this routine to assign unqiue ID 
   numbers to each package and count the number of various
   structures in the system. */
bool pkgCache::MergeState::GenReverseDepends(pkgCache &Cache)
{
   // Iterate over the hash table, blanking the depends links
   for (int I = 0; I != _count(Cache.HeaderP->HashTable); I++)
   {
      // Iterate over the bucket
      unsigned long Pkg = Cache.HeaderP->HashTable[I];
      for (;Pkg != 0; Pkg =  Cache.PkgP[Pkg].NextPackage)
	 Cache.PkgP[Pkg].RevDepends = 0;
   }

   int Pc = 0, Vc = 0, Dc = 0;
   // Iterate over the hash table, generating the depends links
   for (int I = 0; I != _count(Cache.HeaderP->HashTable); I++)
   {
      // Iterate over the bucket
      unsigned long Pkg = Cache.HeaderP->HashTable[I];
      for (;Pkg != 0; Pkg = Cache.PkgP[Pkg].NextPackage)
      {
	 Cache.PkgP[Pkg].ID = Pc;
	 Pc++;
	 // Iterate over the version
	 unsigned long Ver = Cache.PkgP[Pkg].VersionList;
	 for (;Ver != 0; Ver = Cache.VerP[Ver].NextVer)
	 {
	    Cache.VerP[Ver].ID = Vc;
	    Vc++;
	    // Iterate over the depends
	    unsigned long Dep = Cache.VerP[Ver].DependsList;
	    for (;Dep != 0; Dep = Cache.DepP[Dep].NextDepends)
	    {
	       Cache.DepP[Dep].ID = (unsigned short)Dc;
	       Dc++;
	       // Skip duplicate additions
 		unsigned long Targ = Cache.DepP[Dep].Package;
/*             This messes up DepCache, each dependency must have a reverse,
               better not to have duplicate deps in the first place.
	       if (Cache.PkgP[Targ].RevDepends != 0 &&
		   Cache.DepP[Cache.PkgP[Targ].RevDepends].ParentVer == Ver)
	       {
		  Cache.DepP[Dep].NextRevDepends = 0;
		  continue;
	       }*/

	       Cache.DepP[Dep].NextRevDepends = Cache.PkgP[Targ].RevDepends;
	       Cache.PkgP[Targ].RevDepends = Dep;
	    }
	 }
      }
   }
   
   int Pfc = 0;
   for (int PkgF = Cache.Head().FileList; PkgF != 0; 
	PkgF = Cache.PkgFileP[PkgF].NextFile)
   {
      Cache.PkgFileP[PkgF].ID = Pfc;
      Pfc++;
   }   
   
   Cache.HeaderP->PackageCount = Pc;
   Cache.HeaderP->VersionCount = Vc;
   Cache.HeaderP->DependsCount = Dc;
   Cache.HeaderP->PackageFileCount = Pfc;
   return true;
}
									/*}}}*/
