// -*- mode: cpp; mode: fold -*-
// Description								/*{{{*/
// $Id: slangdev.cc,v 1.21 1998/06/21 03:25:07 jgg Exp $
/* ######################################################################

   SLang - SLang base Input and Graphic support.
   
   SLang's colour handling is not really acceptable for what we need. 
   They take the approach that each screen element has a fixed colour -- 
   we take the approach that each screen element has a unique colour. So a 
   Fg/Bg Pair->Index conversion scheme is used. This is not ideal as the
   graphics model requires that the foreground be independent of the 
   background so when one is set it generates a new index for the
   old pair, even though the next call might reset it.

   The input handlers collect input events from all the availble sources
   and reformats them to match our definitions of input. In doing so it
   also adapts SLang to be non-blocking and rely on our select loop. 
   
   GPM is used only to support the Linux GPM interface to mouse in console
   mode. It's abilities to handle XTerms are not used because that would
   require that it have access to stdin, which we cannot permit. XTerm's
   mouse is handled directly by the SLang input handler instead.

   We also translate the SLang keysyms into our own internal set of 
   keysyms so that we are not forced to use SLang. GPM Events are 
   converted into our somewhat more expressive event structure and some
   massaging is done here as well. XTerm's have lame mouse reporting
   ability, the most information we get is the position of the mouse
   during the click or release.
   
   After the events are converted they are directed into the root widget
   for routing to the proper child widget.
   
   ##################################################################### */
									/*}}}*/
// Include Files							/*{{{*/
#include <config.h>
#include <system.h>
#include <signal.h>
#include <slang.h>
#include <unistd.h>
#include <stdlib.h>

#include <stdio.h>

#include <deity/basic.h>
#include <deity/slangdev.h>
#ifdef HAVE_LIBGPM
#include <deity/gpmdev.h>
#endif
#include <deity/event.h>
									/*}}}*/

SLangGC *SLangGC::GC = 0;
#define SlMouseKey 0x1000
bool SLkpFd::SigWinch = false;
bool SLkpFd::XTerm = false;

// SLangGC::SLangGC - Constructor					/*{{{*/
// ---------------------------------------------------------------------
/* */
SLangGC::SLangGC()
{
   Fg = Wc_White;
   Bg = Wc_Black;
	     
   SLtt_get_terminfo();
   SLsmg_init_smg();
   SLang_init_tty(-1,0,0);
   SLkp_init();      
}
									/*}}}*/
// SLangGC::~SLangGC - Destructor					/*{{{*/
// ---------------------------------------------------------------------
/* */
SLangGC::~SLangGC()
{
   SLang_reset_tty();
   SLsmg_reset_smg();
}
									/*}}}*/
// SLangGC::Winch - SigWinch handler					/*{{{*/
// ---------------------------------------------------------------------
/* */
void SLangGC::Winch()
{
   SLtt_get_screen_size();
   SLsmg_init_smg();       
}
									/*}}}*/
// SLangGC::DrawText - Write a string					/*{{{*/
// ---------------------------------------------------------------------
/* Write the string with clipping. */
void SLangGC::DrawText(Rect Pos,Point P,const char *S,unsigned int Len,
		       unsigned long Flags)
{
   if (P.y + Pos.y >= ClipWin.h)
      return;
   if (P.x + Pos.x >= ClipWin.w)
      return;
   
   // Reposition
   Rect Loc = Pos;
   Loc.Clip(ClipWin);

   // Reposition
   if ((Flags & XCenter) == XCenter)
      P.x -= Len/2;
   if ((Flags & XRight) == XRight)
      P.x -= Len;
   if ((Flags & YBottom) == YBottom)
      P.y--;

   // The clipping here can be SO MUCH BETTER
   BFill(Loc);
   SLsmg_gotorc(ClipWin.y + P.y + Pos.y,ClipWin.x + P.x + Pos.x);
   SLsmg_write_nstring((char *)S,MIN(Len,Loc.w - P.x));
}
									/*}}}*/
// SLangGC::DrawText - Write a string					/*{{{*/
// ---------------------------------------------------------------------
/* Write the string with clipping. */
void SLangGC::DrawText(Point P,const char *S,unsigned int Len,
		       unsigned long Flags)
{
   if (P.y >= ClipWin.h)
      return;
   if (P.x >= ClipWin.w)
      return;
   
   // Reposition
   if ((Flags & XCenter) == XCenter)
      P.x -= Len/2;
   if ((Flags & XRight) == XRight)
      P.x -= Len;
   if ((Flags & YBottom) == YBottom)
      P.y--;
    
   SLsmg_gotorc(ClipWin.y + P.y,ClipWin.x + P.x);
   SLsmg_write_nstring((char *)S,MIN(Len,ClipWin.x + ClipWin.w - P.x));
}
									/*}}}*/
// SLangGC::WrapText - Compute text wrap length				/*{{{*/
// ---------------------------------------------------------------------
/* */
unsigned int SLangGC::WrapText(const char *Start,unsigned int Len,long Max)
{
   if ((unsigned)Max >= Len)
       return Len;
   return Max;
}
									/*}}}*/
// SLangGC::Box - Draw a box						/*{{{*/
// ---------------------------------------------------------------------
/* */
void SLangGC::Box(Rect Size)
{
   SLsmg_draw_box(Size.y + ClipWin.y,Size.x + ClipWin.x,Size.h,Size.w);
}
									/*}}}*/
// SLangGC::HLine - Draw a horizontal line				/*{{{*/
// ---------------------------------------------------------------------
/* */
void SLangGC::HLine(Point Pos,long Length)
{
   SLsmg_gotorc(ClipWin.y + Pos.y,ClipWin.x + Pos.x);
   SLsmg_draw_hline(Length);
}
									/*}}}*/
// SLangGC::DrawLineChar - Draws a single line character		/*{{{*/
// ---------------------------------------------------------------------
/* */
void SLangGC::DrawLineChar(Point Pos,LineChars Char)
{
   if (SLtt_Has_Alt_Charset == 0)
   {
      // Non graphic equiv table
      char Table[] = {'|','-','+','/','`','\\','\'','-','|','|','-'}; 
      SLsmg_gotorc(ClipWin.y + Pos.y,ClipWin.x + Pos.x);
      SLsmg_write_nchars(&Table[Char],1);
      return;
   }

   SLsmg_set_char_set(1);
   char Table[] = {SLSMG_HLINE_CHAR,SLSMG_VLINE_CHAR,SLSMG_PLUS_CHAR,
                   SLSMG_ULCORN_CHAR,SLSMG_LLCORN_CHAR,SLSMG_URCORN_CHAR,
                   SLSMG_LRCORN_CHAR,SLSMG_UTEE_CHAR,SLSMG_LTEE_CHAR,
                   SLSMG_RTEE_CHAR,SLSMG_DTEE_CHAR};
   SLsmg_gotorc(ClipWin.y + Pos.y,ClipWin.x + Pos.x);
   SLsmg_write_nchars(&Table[Char],1);
   SLsmg_set_char_set(0);
}
									/*}}}*/
// SLangGC::Fill - Fill an area						/*{{{*/
// ---------------------------------------------------------------------
/* */
void SLangGC::Fill(Rect Area,char C)
{
   Area.Clip(ClipWin);
   
   if (C == 0)
   {
      SLsmg_set_color(MatchColor(Fg.Text,Fg.Text));
      SLsmg_fill_region(ClipWin.y + Area.y,ClipWin.x + Area.x,Area.h,Area.w,' ');
   }
   else
      SLsmg_fill_region(ClipWin.y + Area.y,ClipWin.x + Area.x,Area.h,Area.w,C);

   if (C == 0);
      SLsmg_set_color(MatchColor(Fg.Text,Bg.Text));
}
									/*}}}*/
// SLangGC::MatchColor - Create a SLang colour number for the pair	/*{{{*/
// ---------------------------------------------------------------------
/* */
unsigned long SLangGC::MatchColor(Color::TextColor Fg,Color::TextColor Bg)
{
   if (Fg > Color::White || Bg > Color::White || Fg < Color::Black || 
       Bg < Color::Black)
      return 0;
   
   ColorRec *Empty = 0;
   for(ColorRec *I = Colors+2; I != Colors + sizeof(Colors)/sizeof(ColorRec); I++)
   {      
      if (Empty == 0 && I->Fg == 0)
	 Empty = I;
      
      if (I->Fg == Fg && I->Bg == Bg)
	 return I - Colors;
   }   
   
   // Color map
   static char *Map[] =
   {"","black","red","green","brown","blue","magenta","cyan","lightgray",
    "gray","brightred","brightgreen","yellow","brightblue","brightmagenta",
      "brightcyan","white"};
   
   Empty->Fg = Fg;
   Empty->Bg = Bg;
   SLtt_set_color(Empty - Colors,0,Map[Fg],Map[Bg]);
   return Empty - Colors;      
}
									/*}}}*/
// SLangGC::SetColor - Sets the foreground colour			/*{{{*/
// ---------------------------------------------------------------------
/* */
void SLangGC::SetColor(Color C)
{
   Fg = C;
   SLsmg_set_color(MatchColor(Fg.Text,Bg.Text));
}
									/*}}}*/
// SLangGC::Background - Set the background colour			/*{{{*/
// ---------------------------------------------------------------------
/* */
void SLangGC::Background(Color C)
{
   Bg = C;
   SLsmg_set_color(MatchColor(Fg.Text,Bg.Text));
}
									/*}}}*/
// SLangGC::Flush - Sync with the display				/*{{{*/
// ---------------------------------------------------------------------
/* */
void SLangGC::Flush()
{
   SLsmg_refresh();
}
									/*}}}*/
// SLangGC:ClipRect - Set the clipping window				/*{{{*/
// ---------------------------------------------------------------------
/* */
void SLangGC::ClipRect(Rect Win)
{
   ClipWin = Win;
}
									/*}}}*/

// SLkpFd::SLkpFd - Constructor						/*{{{*/
// ---------------------------------------------------------------------
/* Setup SLangs keymap and enable mouse processing if we are in an XTerm */
SLkpFd::SLkpFd(Widget *Root) :
            SelectLoop::Fd(STDIN_FILENO,true,false,false,true), Root(Root),
            OldButtons(0)
{
   XTerm = false;
   // Check for xterm mode
   char *TermType = getenv("TERM");   
   if (TermType != 0 && strncmp(TermType,"xterm",5) == 0)
   {
      XTerm = true;
      char *S = "\x1b[?1000h";
      write(STDOUT_FILENO,S,strlen(S));

      /* We make a SLang escape sequence for the mouse. This allows us
         and SLang's keymap stuff to live happily together. */
      SLkp_define_keysym("\x1b[M",SlMouseKey);
      
      /* Add in the XTerm bindings for Home/End. Why these are not
         in the termcap is beyond me */
      SLkp_define_keysym("\x1b[1~",SL_KEY_HOME);
      SLkp_define_keysym("\x1b[4~",SL_KEY_END);
   }
}
									/*}}}*/
// SLkpFd::~SLkpFd - Destructor						/*{{{*/
// ---------------------------------------------------------------------
/* Reset the XTerms mouse interface */
SLkpFd::~SLkpFd()
{
   // Deactiveate the mouse
   if (XTerm == true)
   {
      char *S = "\x1b[?1000l";
      write(STDOUT_FILENO,S,strlen(S));
   }

   delete SLangGC::GC;
   GenGC::GC = TextGC::GC = SLangGC::GC = 0;
}
									/*}}}*/
// SLkpFd::Read - Data is ready on stdin				/*{{{*/
// ---------------------------------------------------------------------
/* We simply call SLkp_getkey to convert the key into something we can use. 
   If it detects the XTerm mouse info pre-header (we mapped it to a key)
   then we enter XTerm mouse mode and handle that event */
bool SLkpFd::Read(SelectLoop &Owner)
{
   GrabMutex WidgetLock(Widget::WidgetLock);
   
   int Key = SLkp_getkey();
   KeyEvent KeyE;
   
   // Translate the key
   switch (Key)
   {
      // Basic translations
      case SL_KEY_UP: KeyE.Extended = KeyEvent::Up; break;
      case SL_KEY_DOWN: KeyE.Extended = KeyEvent::Down; break;
      case SL_KEY_ENTER: KeyE.Extended = KeyEvent::Enter; break;
      case SL_KEY_LEFT: KeyE.Extended = KeyEvent::Left; break;
      case SL_KEY_RIGHT: KeyE.Extended = KeyEvent::Right; break;
      case SL_KEY_PPAGE: KeyE.Extended = KeyEvent::PgUp; break;
      case SL_KEY_NPAGE: KeyE.Extended = KeyEvent::PgDown; break;
      case SL_KEY_HOME: KeyE.Extended = KeyEvent::Home; break;
      case SL_KEY_END: KeyE.Extended = KeyEvent::End; break;
      case SL_KEY_DELETE: KeyE.Extended = KeyEvent::Delete; break;
      case SL_KEY_BACKSPACE: KeyE.Extended = KeyEvent::Backspace; break;
      case '\r': KeyE.Extended = KeyEvent::Enter; break;
      
      // Handle the Xterm mouse
      case SlMouseKey:
      {
	 // Grab the rest of the data bytes
	 char Data[3];
	 if (read(FileDesc,Data,sizeof(Data)) != sizeof(Data))
	    return true;
	 Data[0] -= ' ';
	 Data[0] &= 3;
	 Data[1] -= ' ';
	 Data[2] -= ' ';
	 
	 // Recode the event into a mouse event
	 MouseEvent E;
	 memset(&E,0,sizeof(E));
	 E.Clicks = 1;
	 E.Pos = Point(Data[1]-1,Data[2]-1);
	 
	 if (Data[0] == 3)
	 {
	    if (OldButtons == 0)
	       return true;
	    E.Type = MouseEvent::ButtonUp;
	    E.Button = OldButtons;
	    E.ButtonState = OldButtons;
	    OldButtons = 0;
	    if (E.Pos == LastPos)
	       E.Type |= MouseEvent::Click;
	 }
	 else
	 {
	    E.Type = MouseEvent::ButtonDn;
	 
	    // This is less than ideal
	    if (Data[0] == 0)
	       E.Button = MouseEvent::Select;
	    if (Data[0] == 1)
	       E.Button = MouseEvent::Aux;
	    if (Data[0] == 2)
	       E.Button = MouseEvent::Menu;
//	    E.ButtonState = E.Button;
	    E.ButtonState = 0;
	    OldButtons = E.Button;
	 }	 
	 LastPos = E.Pos;

	 Root->RouteMouse(E);
	 Root->DoActions();
	 GenGC::GC->Flush();
	 return true;
      }

      // Eat any errors
      case -1:
      case -2:
      return true;

      default:
      KeyE.Key = Key;
      break;
   }
   
   // Insert the key into the widget tree
   Root->RouteKey(KeyE);
   Root->DoActions();
   GenGC::GC->Flush();
   if (Key == 'q')
      Owner.Quit = true;
   return true;
}
									/*}}}*/
// SLkpFd::Signal - Handle signals					/*{{{*/
// ---------------------------------------------------------------------
/* */
bool SLkpFd::Signal(SelectLoop &)
{
   if (SigWinch == true)
   {
      SigWinch = false;

      // Shouldnt happen
      if (SLangGC::GC == 0)
	 return true;
      
      SLangGC::GC->Winch();
      Root->Resize(Rect(0,0,SLtt_Screen_Cols,SLtt_Screen_Rows));
      Root->DoActions();
      GenGC::GC->Flush();
   }
   return true;
}
									/*}}}*/
// SLkpFd::Setup - Setup the signal handler				/*{{{*/
// ---------------------------------------------------------------------
/* */
void SLkpFd::Setup(SelectLoop &Owner)
{
   Owner.AddSignal(SIGWINCH,SigWinchHandler);
   Owner.AddDeathHandler(ResetSLang);
   Root->Resize(Rect(0,0,SLtt_Screen_Cols,SLtt_Screen_Rows));
   Root->RealizeFamily();
   Root->DoActions();
   GenGC::GC->Flush();
}
									/*}}}*/
// SLkpFd::SigWinchHandler - Handles SIGWINCH				/*{{{*/
// ---------------------------------------------------------------------
/* */
void SLkpFd::SigWinchHandler(int)
{
   SigWinch = true;
}
									/*}}}*/
// SLkpFd::ResetSLang - SLang reset signal handler			/*{{{*/
// ---------------------------------------------------------------------
/* This restores the terminal in the event of something unpleasant. */
void SLkpFd::ResetSLang(int)
{
   // Deactivate the mouse
   if (XTerm == true)
   {
      char *S = "\x1b[?1000l";
      write(STDOUT_FILENO,S,strlen(S));
   }
   
   SLang_reset_tty();
   SLsmg_reset_smg();
}
									/*}}}*/
// SLangAttachDisplay - Setup SLang					/*{{{*/
// ---------------------------------------------------------------------
/* */
Widget *SLangAttachDisplay(SelectLoop &Loop)
{
   GenGC::GC = TextGC::GC = SLangGC::GC = new SLangGC;
   GraphicGC::GC = 0;
   
   BasicWidget *Main = new BasicWidget;
   Main->Resize(Rect(0,0,80,25));
   Main->Background(Wc_Blue);
   Main->BorderWidth(0);
   
   Loop.Add(new SLkpFd(Main));
#ifdef HAVE_LIBGPM
   Loop.Add(new GPMFd(Main));
#endif

   // Cant fail to init slang
   return Main;
}
									/*}}}*/
