// -*- mode: cpp; mode: fold -*-
// Description								/*{{{*/
// $Id: selectloop.cc,v 1.8 1998/04/28 03:17:27 jgg Exp $
/* ######################################################################
   
   Select Loop - Class to allow mulitple sockets to be monitored at once
   SelectLoop::Fd - Selectable file descriptor
   
   ###################################################################### */
									/*}}}*/
// Includes								/*{{{*/
#define _BSD_SOURCE // For SA_RESTART
#include <system.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <signal.h>
#include <errno.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <deity/selectloop.h>

									/*}}}*/

// Signal Stuff								/*{{{*/
// ---------------------------------------------------------------------
/* */
SelectLoop::Sig SelectLoop::Signals[10] = {{0,0},{0,0},{0,0},{0,0},{0,0},
                                           {0,0},{0,0},{0,0},{0,0},{0,0}};
SelectLoop::SigHandler SelectLoop::DeathHandlers[10] = {0,0,0,0,0,0,0,0,0,0};

bool SelectLoop::Triged = false;

void SelectLoop::SignalHandler(int Signal)
{   
   int I = 0;
   for (; I != 10 && Signals[I].Signal != 0 && Signals[I].Signal != Signal; I++);
   if (Signals[I].Signal == Signal)
   {
      Signals[I].Handler(Signal);
      SelectLoop::Triged = true;
   }
}
									/*}}}*/
// SelectLoop::DeathHandler - Handle death signals			/*{{{*/
// ---------------------------------------------------------------------
/* We allow interested partied to clean up their mess and then exit */
void SelectLoop::DeathHandler(int Signal)
{
   for (SigHandler *I = DeathHandlers; I != DeathHandlers + 10; I++)
      if (*I != 0)
	 (**I)(Signal);
   
   // Send the signal to the default handler
   kill(getpid(),Signal);
}
									/*}}}*/
// SelectLoop::SelectLoop - Constructor					/*{{{*/
// ---------------------------------------------------------------------
/* */
SelectLoop::SelectLoop() : List(0), Quit(false)
{
   // Setup the death signal handler
   struct sigaction Act;
   memset(&Act,0,sizeof(Act));
   Act.sa_handler = DeathHandler;
   sigemptyset(&Act.sa_mask);
   Act.sa_flags = SA_RESETHAND;
   sigaction(SIGINT,&Act,0);
   sigaction(SIGSEGV,&Act,0);   
}
									/*}}}*/
// SelectLoop::~SelectLoop - Destructor					/*{{{*/
// ---------------------------------------------------------------------
/* */
SelectLoop::~SelectLoop()
{
   for (Fd *I = List; I != 0;)
   {
      Fd *Tmp = I;
      I = I->Next;
      delete Tmp;
   }   
}
									/*}}}*/
// SelectLoop::Loop - Main select loop					/*{{{*/
// ---------------------------------------------------------------------
/* */
bool SelectLoop::Loop()
{
   Quit = false;
   
   // Configure the select loop.
   int n = 0;
   fd_set rfds;
   fd_set wfds;
   fd_set efds;
   
   // Blank the FD arrays
   FD_ZERO(&rfds);
   FD_ZERO(&wfds);
   FD_ZERO(&efds);

   for (Fd *I = List; I != 0; I = I->Next)
      I->Setup(*this);

   sigset_t Set;
   for (int I = 0; I != 10 && Signals[I].Signal != 0; I++)
      sigaddset(&Set,Signals[I].Signal);
   
   while (List != 0 && Quit == false)
   {
      // Select the proper FDs
      Fd **Last = &List;
      sigprocmask(SIG_BLOCK,&Set,0);
      bool iTriged = Triged;
      Triged = false;
      sigprocmask(SIG_UNBLOCK,&Set,0);
      for (Fd *I = List; I != 0;)
      {
	 if (iTriged == true && I->SignalFlag == true)
	    I->Signal(*this);
	 
	 if (I->FileDesc < 0)
	 {
	    *Last = I->Next;
	    delete I;
	    I = *Last;
	    continue;
	 }
	 
	 if (I->ReadFlag == true)
	    FD_SET(I->FileDesc,&rfds);
	 if (I->WriteFlag == true)
	    FD_SET(I->FileDesc,&wfds);
	 if (I->ErrorFlag == true)
	    FD_SET(I->FileDesc,&efds);
	 n = MAX(n,I->FileDesc);

	 Last = &I->Next;
	 I = I->Next;	 
      }
      
      /* Allow signals to interrupt system calls.
       	 Linux man page says SA_RESTART is not POSIX. It would be possible
       	 to remove this bit of code and the next bit a few lines down
       	 but signal processing would be delayed till some event occured to
       	 break select. */
      struct sigaction Act;
      memset(&Act,0,sizeof(Act));
      Act.sa_handler = SignalHandler;
      sigemptyset(&Act.sa_mask);
      Act.sa_flags = SA_RESTART;
      for (int I = 0; I != 10 && Signals[I].Signal != 0; I++)
	 sigaction(Signals[I].Signal,&Act,0);
	   
      // Wait
      if (Triged == true || select(n+1,&rfds,&wfds,&efds,0) <= 0)
      {
	 if (errno != EINTR && Triged == false)
	    return false;
	 
	 FD_ZERO(&rfds);
	 FD_ZERO(&wfds);
	 FD_ZERO(&efds);
      }      

      // Disable signals from interrupting system calls.
      Act.sa_flags = 0;
      for (int I = 0; I != 10 && Signals[I].Signal != 0; I++)
	 sigaction(Signals[I].Signal,&Act,0);

      // Run processors
      sigprocmask(SIG_BLOCK,&Set,0);
      iTriged = Triged;
      Triged = false;
      sigprocmask(SIG_UNBLOCK,&Set,0);
      for (Fd *I = List; I != 0;)
      {
	 if (iTriged == true && I->SignalFlag == true)
	    I->Signal(*this);
	 
	 if (I->FileDesc < 0)
	    continue;
	 
	 bool Keep = true;
	 if (FD_ISSET(I->FileDesc,&rfds))
	    Keep &= I->Read(*this);
	 if (FD_ISSET(I->FileDesc,&wfds))
	    Keep &= I->Write(*this);
	 if (FD_ISSET(I->FileDesc,&efds))
	    Keep &= I->Error(*this);

	 if (Keep == false)
	 {
	    *Last = I->Next;
	    delete I;
	    I = *Last;
	 }
	 else
	 {
	    Last = &I->Next;
	    I = I->Next;
	 }	 
      }   
   }
   return true;
}
									/*}}}*/
// SelectLoop::Add - Adds a new FD to be monitored			/*{{{*/
// ---------------------------------------------------------------------
/* */
void SelectLoop::Add(Fd *FD)
{
   FD->Next = List;
   List = FD;
}
									/*}}}*/
// SelectLoop::AddSignal - Add a new signal handler			/*{{{*/
// ---------------------------------------------------------------------
/* It probably would be bad if a signal occured during this. */
void SelectLoop::AddSignal(int Sig,SelectLoop::SigHandler Handler)
{
   int I = 0;
   for (; I != 10 && Signals[I].Signal != 0; I++);
   if (I >= 10)
      return;
   Signals[I].Handler = Handler;
   Signals[I].Signal = Sig;

   struct sigaction Act;
   memset(&Act,0,sizeof(Act));
   Act.sa_handler = SignalHandler;
   sigemptyset(&Act.sa_mask);
   Act.sa_flags = 0;
   sigaction(Sig,&Act,0);
}
									/*}}}*/
// SelectLoop::AddDeathHandler - Add a death signal handler		/*{{{*/
// ---------------------------------------------------------------------
/* Add a new death handler to the chain, if the list is full things
   silently fail. */
void SelectLoop::AddDeathHandler(SigHandler Handler)
{
   for (SigHandler *I = DeathHandlers; I != DeathHandlers + 10; I++)
   {
      if (*I == 0)
      {
	 *I = Handler;
	 return;
      }
   }   
}
									/*}}}*/
// SelectLoop::Fd::Fd - Constructor					/*{{{*/
// ---------------------------------------------------------------------
/* */
SelectLoop::Fd::Fd(int Fd,bool Read,bool Write,bool Error,bool Signal) :
                  ReadFlag(Read), WriteFlag(Write), ErrorFlag(Error),
                  SignalFlag(Signal)
{
   this->FileDesc  = Fd;
}
									/*}}}*/
