// -*- mode: cpp; mode: fold -*-
// Description								/*{{{*/
// $Id: pkglist.h,v 1.16 1998/06/06 01:13:51 jgg Exp $
/* ######################################################################

   Package List - Base class for all package list storage methods.
   Ordered Package List - A Package List that is sensitive to order
   
   This class is deliberately simple. We opt to use external functions
   to perform algorithms on the package list. 
 
   STL iterators can't easially be newed, there is no mechanism for 
   the client to free them. However, by using a special class containing
   only a pointer to an internal iterator we can achive the proper semantics
   and get a virtual iterator. Derived classes define a custom priv_iterator
   and begin() functions which allow iteration over any type. To advoid
   huge speed hits constantly constructing dummy priv_iterators, end() 
   is an inline which returns a null iterator. The internal iterators knows
   that when Next() returns false the iteration is over and it should erase
   and set to null the pointer. Iteration is only forwards, -- is not 
   supported, but multiple iterators may exist. To again advoid constant
   construction of dummy priv_iterators operator ++ returns void.

   The ordering of packages is determined by the derived classes. It is
   unsafe to make any assumtions.

   Members of dpkgPkgList
     Find, [](string) - Search the package list for the named package.
     Size - Determine the number of items.
     begin/end - Constructs an iterator that will be able to traverse
                 the package list using STL syntax.
     Add - Adds a new item to the list, it's position is unknown.

   Members of dpkgPkgListOrdered
     push_front - Adds an item to the front of the list
     push_back - Adds an item to the rear of the list
     insert - Inserts an item at the given iterator position
   
   dpkgSPkgCtrlInfo designates a Smart pointer to a dpkgPkgCtrlInfo.
   
   ##################################################################### */
									/*}}}*/
// Header section: pkglib
#ifndef PKGLIB_PKGLIST_H
#define PKGLIB_PKGLIST_H

#include <pkglib/pkgctrlinfo.h>

class pkgPkgList
{
   protected:
   
   /* This is the internal iterator representation. The derived class
      constructs one of these during begin() and puts it in iterator.
      iterator then provides the required semantics and deletes the
      internal iterator at the proper time. */
   class priv_iterator
   {
      public:

      pkgSPkgCtrlInfo Cur;
      virtual bool Next() = 0;
      
      virtual ~priv_iterator();
   };
   
   public:
   
   // Basic iterator
   class iterator
   {
      // Interal iterator
      pkgPkgList::priv_iterator *I;

      void Die();
      public:

      // Dereference operators.
      inline pkgSPkgCtrlInfo operator *() {return I->Cur;};
      inline pkgCSPkgCtrlInfo operator *() const {return I->Cur;};
      
      // No distinction between pre/post is made
      inline void operator ++() {if (I->Next() == false) Die();};
      inline void operator ++(int) {operator ++();};
      
      // Comparision operators
      inline bool operator ==(const iterator &rhs) const {return I == rhs.I;};
      inline bool operator !=(const iterator &rhs) const {return I != rhs.I;};

      inline iterator(const iterator &rhs) : I(rhs.I) {};
      inline iterator(pkgPkgList::priv_iterator *I = 0) : I(I) {};
      inline ~iterator() {Die();};
   };   
   friend iterator;
      
   // Index by Name 
   virtual pkgSPkgCtrlInfo Find(string Name) const = 0;
   virtual pkgSPkgCtrlInfo Find(const pkgElement &Target) const = 0;
   const pkgPkgCtrlInfo &operator [](string Name) const;
            
   // General Info
   virtual signed long Size() = 0;

   // Basic STL iterator (should provide const versions too)
   virtual iterator begin() = 0;
   inline iterator end() const {return iterator(0);};

   // Adds Info at some undetermined location 
   virtual void Add(pkgSPkgCtrlInfo &Info) = 0;
   
   // Construct/destruct
   virtual ~pkgPkgList();
};

class pkgPkgListOrdered : public pkgPkgList
{
   public:

   // STL type ordered insertion
   virtual void push_front(pkgSPkgCtrlInfo &Item) = 0;
   virtual void push_back(pkgSPkgCtrlInfo &Item) = 0;
   virtual void insert(iterator position,pkgSPkgCtrlInfo &Item) = 0;
   virtual ~pkgPkgListOrdered();
};


// Basic text parser
class ifstream;
class pkgTextElmReader
{
   char Buffer[1024];
   ifstream &F;
   string File;
   int CurLine;
   unsigned long SeekPos;
   
   public:
   
   inline int Line() {return CurLine;};
   inline ifstream &FileStream() {return F;};
   void Seek(int Line,streampos Offset);
   inline unsigned long Position() {return SeekPos;};
   inline string FileName() {return File;};
   
   bool Read(string &Element, vector<string> &Lines);
   bool Read(pkgSPkgCtrlInfo &Info);
   
   pkgTextElmReader(ifstream &F,string FileName);
};

// Some simple IO routines
bool pkgReadTxtList(string File,pkgPkgList &List);
bool pkgWriteTxtList(string File,pkgPkgList &List);

#endif
