// -*- mode: cpp; mode: fold -*-
// Description								/*{{{*/
// $Id: graphics.cc,v 1.15 1998/01/25 06:46:09 jgg Exp $
/* ######################################################################

   Graphics - The graphics System
   GenGC - Generic Graphics Context
   TextGC - Textual GC (Character cells)
   GraphicsGC - Graphical GC

   Point and Rect provide some basic point and rectangle manipulations.
   
   ##################################################################### */
									/*}}}*/
// Include Files							/*{{{*/
#include <deity/graphics.h>
#include <deity/widget.h>
#include <system.h>

#include <iostream.h>
									/*}}}*/

GenGC *GenGC::GC = 0;
TextGC *TextGC::GC = 0;
GraphicGC *GraphicGC::GC = 0;

// GenGC::GenGC - Constructor						/*{{{*/
// ---------------------------------------------------------------------
/* */
GenGC::GenGC()
{
   EraseClippings();
}
									/*}}}*/
// GenGC::CreateRegion - Create a region for the widget			/*{{{*/
// ---------------------------------------------------------------------
/* Default action is to not support regions and to unset the region flag
   in the widget and return 0. */
void *GenGC::CreateRegion(Widget *For,void *)
{
   For->Flag(0,Widget::Region);
   return 0;
}
									/*}}}*/
// GenGC::EraseClippings - Erase all of the clipping rectangles		/*{{{*/
// ---------------------------------------------------------------------
/* AddClipping should be called after this to set a proper clipping 
   region for the region. */
void GenGC::EraseClippings()
{
   ClipStack[0] = Rect();
   ClipDepth = 0;
}
									/*}}}*/
// GenGC::AddClipping - Add a new clipping rectangle			/*{{{*/
// ---------------------------------------------------------------------
/* This pushes a new clipping rectangle onto the stack. Each rectangle 
   interescts with the previous to produce a single clip rect. */
void GenGC::AddClipping(Rect Win)
{
   if (ClipDepth == _count(ClipStack))
      return;

   ClipStack[ClipDepth] = Win;
   ClipDepth++;
   GenerateClip();
}
									/*}}}*/
// GenGC::PopClipping - Pull one clip region off the stack		/*{{{*/
// ---------------------------------------------------------------------
/* There should always be at least one clipping region set to be able
   to draw on the region */
void GenGC::PopClipping()
{
   if (ClipDepth == 0)
      return;
   ClipDepth--;
   ClipStack[ClipDepth] = Rect();
   GenerateClip();
}
									/*}}}*/
// GenGC::GenerateClip - Compute the single clipping rectangle		/*{{{*/
// ---------------------------------------------------------------------
/* This computes the final interesected clipping region. Clipping regions 
   also include coordinate translatations */
void GenGC::GenerateClip()
{
   if (ClipDepth == 0)
      return;
   
   Rect Clip = ClipStack[ClipDepth - 1];
   for (int I = ClipDepth - 2; I >= 0; I--)
   {
      Clip.Clip(ClipStack[I]);
      Clip.x += ClipStack[I].x;
      Clip.y += ClipStack[I].y;
   }

   // Tell the real GC about the new region
   ClipRect(Clip);
}
									/*}}}*/
// GenGC::DrawText - Converstion to Rect,Point draw text		/*{{{*/
// ---------------------------------------------------------------------
/* This computes the point arugment based on the bounding rectangle. */
void GenGC::DrawText(Rect Pos,const char *Start,unsigned int Len,
		     unsigned long Flags)
{
   Point P = Point(0,0);

   // Compute the anchor point.
   if ((Flags & XCenter) == XCenter)
      P.x = Pos.w/2;
   if ((Flags & XRight) == XRight)
      P.x = Pos.w;
   if ((Flags & YCenter) == YCenter)
      P.y = Pos.h/2;
   if ((Flags & YBottom) == YBottom)
      P.y = Pos.h;

   DrawText(Pos,P,Start,Len,Flags);
}
									/*}}}*/
// GenGC::FlagPosition - Compute a new rectangle corner			/*{{{*/
// ---------------------------------------------------------------------
/* As input pass a rectangle with the ul corner as the user parameter and
   the w/h as the object size. Flags indicates the meaning of the ul corner
   (Center, left right, up down, etc). This will compute a new ul corner 
   so that everything is nice and easy to draw. */
void GenGC::FlagPosition(Rect &P,unsigned long Flags)
{
   // Reposition the text
   if ((Flags & XCenter) == XCenter)
      P.x -= P.w/2;
   if ((Flags & XRight) == XRight)
      P.x -= P.w;
   if ((Flags & YCenter) == YCenter)
      P.y -= P.h/2;
   if ((Flags & YBottom) == YBottom)
      P.y -= P.h;   
}
									/*}}}*/
// GenGC::DrawWrappedText - Draws text into a box			/*{{{*/
// ---------------------------------------------------------------------
/* This simply warps the text at its longest point into a rectangle. */
long GenGC::DrawWrappedText(Rect Loc,string S,bool Blank)
{  
   // Loop over the string untill all the lines have been drawn
   long Height = ExtentText(" ").h;
   Point P = Point(Loc.x,Loc.y);
   for (const char *I = S.begin(); I < S.end();)
   {
      // Out of horizontal space
      if (P.y + Height > Loc.y + Loc.h)
	 return P.y;
      
      for (; *I == ' ' && I < S.end(); I++);
      
      // Compute the wrap point
      unsigned int Dist = GC->WrapText(I,S.end() - I,Loc.w);

      // Backtrack to the nearest separator
      const char *J = I + Dist;
      for (; J > I && *J != ' ' && *J != '-' && J != S.end(); J--);
      if (J == I)
	 J = I + Dist;
      else
	 if (J != S.end())
	     J++;
      
      DrawText(P,I,J - I);
      I = J;
      P.y += Height;
   }
   
   if (S.empty() == true || Blank == true)
      P.y += Height;
   return P.y;
}
									/*}}}*/

// TextGC::~TextGC - Provides a base for the class			/*{{{*/
// ---------------------------------------------------------------------
/* */
TextGC::~TextGC()
{
}
									/*}}}*/
// TextGC::ExtentText - Return the size of a text string		/*{{{*/
// ---------------------------------------------------------------------
/* This is simple for text mode, number of chars and 1 */
Rect TextGC::ExtentText(const char *,unsigned int Len) 
{
   return Rect(0,0,Len,1);
}
									/*}}}*/
// GraphicGC::~GraphicGC - Provides a base for the class		/*{{{*/
// ---------------------------------------------------------------------
/* */
GraphicGC::~GraphicGC()
{
}
									/*}}}*/
// GraphicGC::DrawBitmap - Converstion to Rect,Point draw bitmap	/*{{{*/
// ---------------------------------------------------------------------
/* This computes the point arugment based on the bounding rectangle. */
void GraphicGC::DrawBitmap(Rect Pos,XPMImage &Image,unsigned long Flags)
{
   Point P = Point(0,0);

   // Compute the anchor point.
   if ((Flags & XCenter) == XCenter)
      P.x = Pos.w/2;
   if ((Flags & XRight) == XRight)
      P.x = Pos.w;
   if ((Flags & YCenter) == YCenter)
      P.y = Pos.h/2;
   if ((Flags & YBottom) == YBottom)
      P.y = Pos.h;

   DrawBitmap(Pos,P,Image,Flags);
}
									/*}}}*/

// Color::Color - Constructor						/*{{{*/
// ---------------------------------------------------------------------
/* */
Color::Color() : RedLevel(0), GreenLevel(0), BlueLevel(0), Text(None)
{
}
									/*}}}*/
// SimpleFont::SimpleFont - Constructor					/*{{{*/
// ---------------------------------------------------------------------
/* */
SimpleFont::SimpleFont() : Weight(Normal), Slant(Upright)
{
}
									/*}}}*/

// Point::Point - Constructor  						/*{{{*/
// ---------------------------------------------------------------------
/* */
Point::Point()
{
   x = 0;
   y = 0;
}
									/*}}}*/
// operator <<(Point) - Stream output for a point			/*{{{*/
// ---------------------------------------------------------------------
/* */
ostream &operator << (ostream &stream,const Point &P)
{
   stream << "Point(" << P.x << ',' << P.y << ')';
   return stream;
}
									/*}}}*/

// Rect::Rect - Constructor						/*{{{*/
// ---------------------------------------------------------------------
/* */
Rect::Rect()
{
   x = 0;
   y = 0;
   w = 0;
   h = 0;
}
									/*}}}*/
// Rect::Clip - Clips this rectangle so it is inside the other		/*{{{*/
// ---------------------------------------------------------------------
/* this is a subrectangle of other. The origin of other is irrelivant
   to the clipping algorithm. */
void Rect::Clip(Rect Other)
{
   if (x > Other.w)
      x = Other.w;
   if (y > Other.h)
      y = Other.h;
   if (x < 0)
   {
      w += x;
      x = 0;
   }
   if (y < 0)
   {
      h += y;
      y = 0;
   }
   if (x + w > Other.w)
      w = Other.w - x;
   if (y + h > Other.h)
      h = Other.h - y;
}
									/*}}}*/
// Rect::Inside - Determine if a point is inside a rectangle		/*{{{*/
// ---------------------------------------------------------------------
/* */
bool Rect::Inside(Point P)
{
   if (P.x < x || P.y < y)
      return false;
   if (P.x >= x + w || P.y >= y + h)
      return false;
   return true;
}
									/*}}}*/
// Rect::Inside - Determine if a whole rectangle is within another	/*{{{*/
// ---------------------------------------------------------------------
/* */
bool Rect::Inside(Rect R)
{
   if (R.Inside(Point(x,y)) && R.Inside(Point(x+w,y+h)))
      return true;
   return false;
}
									/*}}}*/
// operator <<(Rect) - Stream output for a rect				/*{{{*/
// ---------------------------------------------------------------------
/* */
ostream &operator << (ostream &stream,const Rect &R)
{
   stream << "Rect(" << R.x << ',' << R.y << ',' << R.w << ',' << R.h << ')';
   return stream;
}
									/*}}}*/

// AbsRect::AbsRect - Constructor					/*{{{*/
// ---------------------------------------------------------------------
/* */
AbsRect::AbsRect()
{
   x1 = 0;
   y1 = 0;
   x2 = 0;
   y2 = 0;
}
									/*}}}*/
// operator <<(AbsRect) - Stream output for a rect			/*{{{*/
// ---------------------------------------------------------------------
/* */
ostream &operator << (ostream &stream,const AbsRect &R)
{
   stream << "AbsRect(" << R.x1 << ',' << R.y1 << ',' << 
      R.x2 << ',' << R.y2 << ')';
   return stream;
}
									/*}}}*/
