/*
 *  texture.c:		Texture widgets, color widgets	
 *
 *  Written by:		Ullrich Hafner
 *  
 *  Copyright (C) 1998 Ullrich Hafner <hafner@informatik.uni-wuerzburg.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 */

/*
 *  $Date: 1998/10/06 21:32:21 $
 *  $Author: hafner $
 *  $Revision: 1.29 $
 *  $State: Exp $
 */

#include "config.h"

#include <gtk/gtk.h>
#include <proplist.h>

#if HAVE_STRING_H
#	include <string.h>
#else /* not HAVE_STRING_H */
#	include <strings.h>
#endif /* not HAVE_STRING_H */

#ifdef TILE_OF_DAY      
#include "tileofday.h"
#endif /* TILE_OF_DAY */

#include "misc.h"
#include "icons.h"
#include "window.h"
#include "dialog.h"
#include "previews.h"
#include "texture.h"

#include "error.h"

/*******************************************************************************

		    global variables (imported from window.c)
  
*******************************************************************************/

extern proplist_t	*windowmaker;
extern GtkTooltips	*tooltips;
extern bool_t		changed;
extern GtkWidget	*main_window;
extern proplist_t	 *pixmap_path;
extern proplist_t	 *plist_changed;
extern proplist_t	 *pl_yes;

/*******************************************************************************

				prototypes
  
*******************************************************************************/

static void
list_change (GtkWidget *widget, GtkWidget *list);
static void
list_remove (GtkWidget *widget, GtkWidget *list);
static void
list_append (GtkWidget *widget, GtkWidget *list);
static void
list_insert (GtkWidget *widget, GtkWidget *list);
static GtkWidget *
new_list_item (proplist_t *color);
static GtkWidget *
gradient_dialog (wtexture_t *wtexture);
static void
set_pixmap (GtkWidget *widget, gpointer filesel);
static void
set_pixmap_name (GtkWidget *widget, gpointer ptr);
static void
change_path (GtkWidget *widget, gpointer filesel);
static void
file_selection (GtkWidget *widget, wtexture_t *wtexture);
static GtkWidget *
pixmap_dialog (wtexture_t *wtexture, bool_t preview_browser);
static void
set_texture (GtkWidget *widget, gpointer ptr);
static void
color_selection_ok (GtkWidget               *w,
                    GtkColorSelectionDialog *cs);
static void
verify_pixmap_name (GtkWidget *widget, wtexture_t *wtexture);
static void
update_pixmap (GtkWidget *widget, GtkWidget *pixmap);

/*******************************************************************************

				public code
  
*******************************************************************************/

void
texture_dialog (GtkWidget *box, proplist_t *key, bool_t preview_browser)
/*
 *  Generate a new texture dialog widget and pack widget into given 'box'.
 *  Associated wmaker attribute name is given by 'key'.
 *
 *  No return value.
 */ 
{
   wtexture_t *wtexture;
   GtkWidget  *gdialog, *pdialog;
   GtkWidget  *hbox;
   char	      *type;
   proplist_t *array  = PLGetDictionaryEntry (windowmaker, key);

   if (!array || PLGetNumberOfElements (array) <= 0)
      return;				/* illegal values */

   /*
    *  Convert [hvd]gradient to m[hvd]gradient (they are just a subset)
    */
   type = PLGetString (PLGetArrayElement (array, 0));
   if (strcaseeq (type + 1, "gradient"))
   {
      char	*tmp;
      char	*c1name = PLGetString (PLGetArrayElement (array, 1));
      char	*c2name = PLGetString (PLGetArrayElement (array, 2));
      GdkColor	color1, color2;

      if (!gdk_color_parse (c1name, &color1))
	 color1.red = color1.green = color1.blue = 65535; /* white */
      if (!gdk_color_parse (c2name, &color2))
	 color2.red = color2.green = color2.blue = 65535; /* white */

      color1.red   = (color1.red + color2.red) >> 1;
      color1.green = (color1.green + color2.green) >> 1;
      color1.blue  = (color1.blue + color2.blue) >> 1;

      {
	 char colstr [MAXSTRLEN];

	 sprintf (colstr, "#%02x%02x%02x", /* new color */
		  (unsigned) (color1.red   / 256 + 0.5),
		  (unsigned) (color1.green / 256 + 0.5),
		  (unsigned) (color1.blue  / 256 + 0.5));
	 PLInsertArrayElement (array, PLMakeString (colstr), 1);
      }

      tmp = Calloc (strlen (type) + 2, sizeof (char));
      strcpy (tmp, "m");
      strcat (tmp, type);
      PLRemoveArrayElement (array, 0);
      PLInsertArrayElement (array, PLMakeString (tmp), 0);
      type = PLGetString (PLGetArrayElement (array, 0));
   }
   else if (strcaseeq (type, "none"))	/* workaround */
   {
      PLRemoveArrayElement (array, 0);
      PLInsertArrayElement (array, PLMakeString ("solid"), 0);
      PLInsertArrayElement (array, PLMakeString ("black"), 1);
   };
   
   wtexture      = Calloc (1, sizeof (wtexture_t));
   wtexture->key = key;
   pdialog       = pixmap_dialog (wtexture, preview_browser);
   gdialog       = gradient_dialog (wtexture);
   
   hbox = gtk_hbox_new (FALSE, 5);
   gtk_box_pack_start (GTK_BOX (box), hbox, FALSE, FALSE, 0);
   gtk_widget_show (hbox);
		     
   /*
    *  Option menu of all available texture types
    */
   {
      const char *types [] = {"Solid color", "Horizontal gradient",
			      "Vertical gradient", "Diagonal gradient",
			      "Scaled pixmap", "Tiled pixmap", "Centered pixmap",
			      NULL};
      GtkWidget *option_menu;
      unsigned  number = 0;
      char     *type   = PLGetString (PLGetArrayElement (array, 0));
      
      if (strcaseeq (type, "solid"))
	 number = 0;
      else if (strcaseeq (type, "mhgradient"))
	 number = 1;
      else if (strcaseeq (type, "mvgradient"))
	 number = 2;
      else if (strcaseeq (type, "mdgradient"))
	 number = 3;
      else if (strcaseeq (type, "spixmap"))
	 number = 4;
      else if (strcaseeq (type, "tpixmap"))
	 number = 5;
      else if (strcaseeq (type, "cpixmap"))
	 number = 6;
      else
	 error ("Texture type `%s' not supported", type);

      {
	 GtkWidget *vbox;
	 
	 vbox = gtk_vbox_new (FALSE, 0);
	 gtk_widget_show (vbox);
	 gtk_box_pack_start (GTK_BOX (hbox), vbox, FALSE, FALSE, 0);
	 
	 /*
	  *  Texture type widget
	  */
	 {
	    GtkWidget *mframe;
	    GtkWidget *hbox;
	    
	    mframe = gtk_frame_new ("Texture type");
	    gtk_frame_set_label_align (GTK_FRAME (mframe), 0.5, 0.5);
	    gtk_container_border_width (GTK_CONTAINER (mframe), 5);
	    gtk_box_pack_start (GTK_BOX (vbox), mframe, FALSE, TRUE, 0);

	    hbox = gtk_hbox_new (FALSE, 5);
	    gtk_container_add (GTK_CONTAINER (mframe), hbox);
	    gtk_container_border_width (GTK_CONTAINER (hbox), 5);
	 
	    option_menu = generate_option_menu (NULL, types, types [number],
						set_texture, wtexture);
	    gtk_tooltips_set_tip (tooltips, option_menu,
				  "Solid color: A simple solid color.\n"
				  "Horizontal gradient: "
				  "A gradient going from left "
				  "to right, from the first to the last color.\n"
				  "Vertical gradient: A gradient going from top "
				  "to bottom, from the first to the last color.\n"
				  "Diagonal gradient: A gradient going from the "
				  "top-left corner to the bottom-right corner, "
				  "from the first to the last color.\n"
				  "Scaled pixmap: The pixmap will be scaled to "
				  "adapt to the destination's size.\n"
				  "Centered pixmap: "
				  "The pixmap will be shown centered.\n"
				  "Tiled pixmap: The pixmap will be tiled in the "
				  "destination.", NULL);  
	    gtk_box_pack_start (GTK_BOX (hbox), option_menu, FALSE, TRUE, 0);
	    gtk_widget_show_all (mframe);
	 }
	 /*
	  *  Default color widget
	  */
	 {
	    GtkWidget *color_box;
	    GtkWidget *preview;
	    GtkWidget *frame;
	    GtkWidget *mframe;
	    GtkWidget *button;
	    GtkWidget *hbox;
	    
	    hbox = gtk_hbox_new (FALSE, 5);
	    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, TRUE, 0);
	    gtk_widget_show (hbox);
	    
	    mframe = gtk_frame_new ("Default color");
	    gtk_frame_set_label_align (GTK_FRAME (mframe), 0.5, 0.5);
	    gtk_container_border_width (GTK_CONTAINER (mframe), 5);
	    gtk_box_pack_start (GTK_BOX (hbox), mframe, FALSE, TRUE, 5);

	    color_box = gtk_hbox_new (FALSE, 5);
	    gtk_container_border_width (GTK_CONTAINER (color_box), 5);
	    gtk_container_add (GTK_CONTAINER (mframe), color_box);

	    frame = gtk_frame_new (NULL);
	    gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_IN);
	    gtk_container_border_width (GTK_CONTAINER (frame), 0);
	    
	    preview = gtk_preview_new (GTK_PREVIEW_COLOR);
	    gtk_preview_size (GTK_PREVIEW (preview), 50, 25);
	    gtk_container_add (GTK_CONTAINER (frame), preview);
	    gtk_object_set_user_data (GTK_OBJECT (preview),
				      PLGetArrayElement (array,
							 number < 4 ? 1 : 2));
	    fill_preview (GTK_PREVIEW (preview),
			  PLGetString (PLGetArrayElement (array,
							  number < 4 ? 1 : 2)));

	    button = gtk_button_new ();
	    gtk_tooltips_set_tip (tooltips, button,
				  "This color is used as solid color,"
				  " if a gradient can't be drawn, "
				  "or during the pixmap scaling/tiling process.",
				  NULL);  
	    gtk_container_add (GTK_CONTAINER (button), frame);
	    gtk_container_border_width (GTK_CONTAINER (button), 0);
	    gtk_object_set_user_data (GTK_OBJECT (button),
				      (gpointer) preview);
	    gtk_box_pack_start (GTK_BOX (color_box), button, FALSE, TRUE, 5);
	    gtk_signal_connect (GTK_OBJECT (button), "pressed",
				GTK_SIGNAL_FUNC (color_selection_dialog),
				(gpointer) key);

	    gtk_widget_show_all (mframe);
	 }
      }
   }
   /*
    *  Pixmap and gradient dialogs
    */
   gtk_box_pack_start (GTK_BOX (hbox), pdialog, FALSE, FALSE, 0);
   gtk_box_pack_start (GTK_BOX (hbox), gdialog, FALSE, FALSE, 0);

#ifdef TILE_OF_DAY
   check_valid_tileofday (wtexture);
#endif /* TILE_OF_DAY */
}

void
fill_preview (GtkPreview *preview, const char *col_spec)
/*
 *  Fill a 'preview' with color 'col_spec'.
 *
 *  No return value.
 */
{
   unsigned	 y;
   GdkColor	 color;
   unsigned	 width   = GTK_WIDGET (preview)->requisition.width;
   unsigned	 height  = GTK_WIDGET (preview)->requisition.height;
   unsigned char *buffer = Calloc (width * 3, sizeof (unsigned char));
   
   if (!gdk_color_parse (col_spec, &color))
      color.red = color.green = color.blue = 65535;

   for (y = 0; y < height; y++)
   {
      unsigned i;
      unsigned char *ptr = buffer;
			   
      for (i = 0; i < width; i++)
      {
	 *ptr++ = color.red / 256;
	 *ptr++ = color.green / 256;
	 *ptr++ = color.blue / 256;
      }

      gtk_preview_draw_row (preview, buffer, 0, y, width);
   }
   Free (buffer);
   gtk_widget_draw (GTK_WIDGET (preview), NULL);
}

void
color_selection_dialog (GtkWidget *widget, proplist_t *key)
/*
 *  Generate a color selection popup window.
 *  associated user data of 'widget' must be a pointer to corresponding preview.
 *  'key' is associated wmaker attribute name/
 *
 *  No return value.
 */
{
   static GtkWidget *window = NULL;

   if (!window)
   {
      GdkColor	 color;
      GtkWidget  *preview = gtk_object_get_user_data (GTK_OBJECT (widget));
      proplist_t *value   = gtk_object_get_user_data (GTK_OBJECT (preview));
      gdouble	 rgb [4];
      GtkColorSelection		*cs;
      GtkColorSelectionDialog	*dialog;
      
      gtk_preview_set_install_cmap (TRUE);
      gtk_widget_push_visual (gtk_preview_get_visual ());
      gtk_widget_push_colormap (gtk_preview_get_cmap ());

      window = gtk_color_selection_dialog_new ("Color selection");
      dialog = GTK_COLOR_SELECTION_DIALOG (window);
      cs     = GTK_COLOR_SELECTION (dialog->colorsel);
      
      {
	 char *color_spec = PLGetString (value);

	 gtk_object_set_user_data (GTK_OBJECT (cs), (gpointer) value);

	 if (!gdk_color_parse (color_spec, &color))
	    color.red = color.green = color.blue = 65530;
      }

      rgb [0] = color.red   / 65535.0;
      rgb [1] = color.green / 65535.0;
      rgb [2] = color.blue  / 65535.0;

      gtk_color_selection_set_color (cs, rgb);
      gtk_color_selection_get_color (cs, rgb);
      gtk_color_selection_set_color (cs, rgb);
      gtk_color_selection_set_opacity (cs, FALSE);
      gtk_color_selection_set_update_policy (cs, GTK_UPDATE_CONTINUOUS);
      gtk_window_position (GTK_WINDOW (window), GTK_WIN_POS_MOUSE);

      gtk_signal_connect (GTK_OBJECT (window), "destroy",
                          GTK_SIGNAL_FUNC (gtk_widget_destroyed),
                          &window);

      gtk_signal_connect (GTK_OBJECT (dialog->ok_button),
			  "clicked",
			  GTK_SIGNAL_FUNC (color_selection_ok), window);
      gtk_signal_connect_object (GTK_OBJECT (dialog->ok_button),
				 "clicked",
				 GTK_SIGNAL_FUNC (gtk_widget_destroy),
				 GTK_OBJECT (window));
      gtk_object_set_user_data (GTK_OBJECT (dialog->ok_button), key);
      gtk_object_set_user_data (GTK_OBJECT (dialog->cancel_button), preview);

      gtk_signal_connect_object (GTK_OBJECT (dialog->cancel_button),
				 "clicked",
				 GTK_SIGNAL_FUNC (gtk_widget_destroy),
				 GTK_OBJECT (window));
      gtk_widget_pop_colormap ();
      gtk_widget_pop_visual ();
   }

   if (!GTK_WIDGET_VISIBLE (window))
      gtk_widget_show (window);
   else
      gtk_widget_destroy (window);
}

/*******************************************************************************

				private code
  
*******************************************************************************/

static void
color_selection_ok (GtkWidget               *w,
                    GtkColorSelectionDialog *cs)
/*
 *  Replace old color with new one.
 *
 *  No return value.
 *
 *  Side effects:
 *	value of wmaker attribute w->key is modified.
 *	corresponding preview color update
 */
{
   GtkColorSelection	*colorsel;
   gdouble		color [4];
   proplist_t		*value;
   proplist_t		*key;
   GtkPreview		*preview;

   key      = gtk_object_get_user_data (GTK_OBJECT (w));
   preview  = gtk_object_get_user_data (GTK_OBJECT (cs->cancel_button));
   value    = gtk_object_get_user_data (GTK_OBJECT (preview));
   colorsel = GTK_COLOR_SELECTION (cs->colorsel);
   
   gtk_color_selection_get_color (colorsel, color);
   gtk_color_selection_set_color (colorsel, color);
 
   {
      char tmp [MAXSTRLEN];

      sprintf (tmp, "#%02x%02x%02x",	/* new color */
	       (unsigned) (color [0] * 255 + 0.5),
	       (unsigned) (color [1] * 255 + 0.5),
	       (unsigned) (color [2] * 255 + 0.5));

      {
	 proplist_t *plval = PLGetDictionaryEntry (windowmaker, key);
	 if (PLIsArray (plval))		/* default color of a texture */
	 {
	    unsigned i;
	    
	    for (i = 1; i < PLGetNumberOfElements (plval); i++)
	       if (PLGetArrayElement (plval, i) == value)
	       {
		  PLRemoveArrayElement (plval, i);
		  PLInsertArrayElement (plval, PLMakeString (tmp), i);
		  gtk_object_set_user_data (GTK_OBJECT (preview),
					    PLGetArrayElement (plval, i));
		  fill_preview (preview,
				PLGetString (PLGetArrayElement (plval, i)));
		  break;
  	       }
	 }
	 else				/* color attribute */
	 {
	    plval = PLMakeString (tmp);
	    gtk_object_set_user_data (GTK_OBJECT (preview), plval);
	    fill_preview (preview, tmp);
	    PLInsertDictionaryEntry (windowmaker, key, plval);
	 }
      }
      PLInsertDictionaryEntry (plist_changed, key, pl_yes);
      changed = YES;			
   }
}

static void
set_texture (GtkWidget *widget, gpointer ptr)
/*
 *  Set new type of texture.
 *
 *  No return value.
 *
 *  Side effects:
 *	value of wmaker array attribute wtexture->key is modified.
 *	color list and pixmap image update
 */
{
   wtexture_t   *wtexture   = (wtexture_t *) ptr;
   char		*type       = gtk_object_get_user_data (GTK_OBJECT (widget));
   GtkWidget	**wgradient = wtexture->wgradient;
   GtkWidget	**wpixmap   = wtexture->wpixmap;
   proplist_t	*array	    = PLGetDictionaryEntry (windowmaker, wtexture->key);
   char		*attribute;

   attribute = PLGetString (PLGetArrayElement (array, 0));

   if (strstr (type, "pixmap"))
   {
      if (!strcaseeq (attribute + 1, "pixmap")) /* gradient texture ? */
      {
	 char *s;
	 
	 if (streq ("Horizontal gradient", type))
	    s = "mhgradient";
	 else if (streq ("Vertical gradient", type))
	    s = "mvgradient";
	 else if (streq ("Diagonal gradient", type))
	    s = "mdgradient";
	 /*
	  *  Disable gradient widgets, enable pixmap widgets
	  */
	 gtk_widget_set_sensitive (wpixmap [PIXMAP_NAME], TRUE);
	 gtk_widget_set_sensitive (wpixmap [PIXMAP_BROWSE], TRUE);
	 if (wpixmap [PIXMAP_PREVIEWS])
	    gtk_widget_set_sensitive (wpixmap [PIXMAP_PREVIEWS], TRUE);
	 if (wpixmap [PIXMAP_TILES])
	    gtk_widget_set_sensitive (wpixmap [PIXMAP_TILES], TRUE);
	 
	 /*
	  *  Default pixmap
	  */
	 PLInsertArrayElement (array, PLMakeString ("BulletHole.xpm"), 1);
	 gtk_entry_set_text (GTK_ENTRY (wpixmap [PIXMAP_NAME]), "BulletHole.xpm");
	 make_pixmap ("BulletHole.xpm", 128, 128, wpixmap [PIXMAP_PIXMAP]);
	 gtk_widget_hide (wgradient [GRADIENT_FRAME]);
	 gtk_widget_show (wpixmap [PIXMAP_FRAME]);
      }
      gtk_widget_set_sensitive (wgradient [GRADIENT_REMOVE], FALSE);
      gtk_widget_set_sensitive (wgradient [GRADIENT_INSERT], FALSE);
      gtk_widget_set_sensitive (wgradient [GRADIENT_APPEND], FALSE);
      gtk_widget_set_sensitive (wgradient [GRADIENT_CHANGE], FALSE);
      gtk_widget_hide (wgradient [GRADIENT_FRAME]);
      {
	 proplist_t *pl = PLMakeString (streq ("Tiled pixmap", type)
					? "tpixmap"
					: (streq ("Scaled pixmap", type)
					   ? "spixmap" : "cpixmap"));
	 PLRemoveArrayElement (array, 0);
	 PLInsertArrayElement (array, pl, 0);

	 PLRelease (pl);
      }

      while (g_list_length (GTK_LIST (wgradient [GRADIENT_LIST])->children))
	 list_remove (wgradient [GRADIENT_REMOVE], wgradient [GRADIENT_LIST]);
   }
   else
   {
      char *s;
      
      if (streq ("Horizontal gradient", type))
	 s = "mhgradient";
      else if (streq ("Vertical gradient", type))
	 s = "mvgradient";
      else if (streq ("Diagonal gradient", type))
	 s = "mdgradient";
      else
	 s = "solid";
      
      if (strcaseeq (attribute + 1, "pixmap")) /* pixmap texture ? */
      {
	 /*
	  *  Disable pixmap widgets
	  */
	 gtk_entry_set_text (GTK_ENTRY (wpixmap [PIXMAP_NAME]), "");
	 gtk_widget_set_sensitive (wpixmap [PIXMAP_NAME], FALSE);
	 gtk_widget_set_sensitive (wpixmap [PIXMAP_BROWSE], FALSE);
	 if (wpixmap [PIXMAP_PREVIEWS])
	    gtk_widget_set_sensitive (wpixmap [PIXMAP_PREVIEWS], FALSE);
	 if (wpixmap [PIXMAP_TILES])
	    gtk_widget_set_sensitive (wpixmap [PIXMAP_TILES], FALSE);
	 make_pixmap ("BulletHole.xpm", 128, 128, wpixmap [PIXMAP_PIXMAP]);
	 PLRemoveArrayElement (array, 1); 
	 gtk_widget_hide (wpixmap [PIXMAP_FRAME]);
      }
      {
	 proplist_t *pl = PLMakeString (s);

	 PLRemoveArrayElement (array, 0);
	 PLInsertArrayElement (array, pl, 0);

	 PLRelease (pl);
      }

      /*
       *  Enable gradient widgets
       */
      if (!strcaseeq (s, "solid"))
      {
	 gtk_widget_set_sensitive (wgradient [GRADIENT_APPEND], TRUE);
	 gtk_widget_set_sensitive (wgradient [GRADIENT_INSERT], TRUE);
	 gtk_widget_set_sensitive (wgradient [GRADIENT_CHANGE], TRUE);
	 if (g_list_length (GTK_LIST (wgradient [GRADIENT_LIST])->children) > 2)
	    gtk_widget_set_sensitive (wgradient [GRADIENT_REMOVE], TRUE);
	 else
	    while (g_list_length (GTK_LIST (wgradient [GRADIENT_LIST])->children)
		   < 2)
	       list_append (wgradient [GRADIENT_REMOVE],
			    wgradient [GRADIENT_LIST]);
	 gtk_widget_show (wgradient [GRADIENT_FRAME]);
      }
      else
      {
	 gtk_widget_set_sensitive (wgradient [GRADIENT_REMOVE], FALSE);
	 gtk_widget_set_sensitive (wgradient [GRADIENT_CHANGE], FALSE);
	 gtk_widget_set_sensitive (wgradient [GRADIENT_APPEND], FALSE);
	 gtk_widget_set_sensitive (wgradient [GRADIENT_INSERT], FALSE);
	 gtk_widget_hide (wgradient [GRADIENT_FRAME]);
	 
	 while (g_list_length (GTK_LIST (wgradient [GRADIENT_LIST])->children))
	    list_remove (wgradient [GRADIENT_REMOVE], wgradient [GRADIENT_LIST]);
      }
   }

#ifdef TILE_OF_DAY
   check_valid_tileofday (wtexture);
#endif /* TILE_OF_DAY */
   changed = YES;
   PLInsertDictionaryEntry (plist_changed, wtexture->key, pl_yes);
}

static GtkWidget *
pixmap_dialog (wtexture_t *wtexture, bool_t preview_browser)
/*
 *  Generate all widgets for the pixmap dialog.
 *
 *  Return value:
 *	Container widget of pixmap dialog.
 */
{
   GtkWidget	*table;
   GtkWidget	*frame;
   proplist_t	val       = PLGetDictionaryEntry (windowmaker, wtexture->key);
   char		*type     = PLGetString (PLGetArrayElement (val, 0));
   char		*filename = PLGetString (PLGetArrayElement (val, 1));
   
   frame = gtk_frame_new ("Pixmap");
   gtk_frame_set_label_align (GTK_FRAME (frame), 0.5, 0.5);
   gtk_container_border_width (GTK_CONTAINER (frame), 5);

   wtexture->wpixmap = Calloc (PIXMAP_LAST, sizeof (GtkWidget *));
   
   table = gtk_table_new (2, 2, FALSE);
   gtk_table_set_row_spacings (GTK_TABLE (table), 5);
   gtk_table_set_col_spacings (GTK_TABLE (table), 5);
   gtk_container_border_width (GTK_CONTAINER (table), 5);
   gtk_container_add (GTK_CONTAINER (frame), table);
   
   /*
    *  Pixmap scrolled window
    */
   if (strcaseeq (type + 1, "pixmap"))
      wtexture->wpixmap [PIXMAP_PIXMAP] = make_pixmap (filename, 128, 128, NULL);
   else
      wtexture->wpixmap [PIXMAP_PIXMAP] = make_pixmap ("BulletHole.xpm", 64, 64,
						       NULL);
   /*
    *  Scrolled window for pixmap
    */
   {
      GtkWidget *scrolled = gtk_scrolled_window_new (NULL, NULL);
      
      gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled),
				      GTK_POLICY_ALWAYS, GTK_POLICY_ALWAYS);
      gtk_widget_set_usize (scrolled, 100, 100);
      gtk_container_border_width (GTK_CONTAINER (scrolled), 5);
      gtk_container_add (GTK_CONTAINER (scrolled),
			 wtexture->wpixmap [PIXMAP_PIXMAP]);
      gtk_table_attach (GTK_TABLE (table), scrolled, 0, 1, 0, 1,
			GTK_FILL | GTK_SHRINK, GTK_FILL | GTK_SHRINK, 0, 0);
   }
   /*
    *  Filename entry
    */
   wtexture->wpixmap [PIXMAP_NAME] = gtk_entry_new ();
   gtk_widget_set_usize (wtexture->wpixmap [PIXMAP_NAME], 120, 0);
   if (strcaseeq (type + 1, "pixmap"))
      gtk_entry_set_text (GTK_ENTRY (wtexture->wpixmap [PIXMAP_NAME]), filename);
   gtk_signal_connect (GTK_OBJECT (wtexture->wpixmap [PIXMAP_NAME]), "changed",
		       GTK_SIGNAL_FUNC (set_pixmap_name), wtexture);
   gtk_signal_connect (GTK_OBJECT (wtexture->wpixmap [PIXMAP_NAME]), "activate",
		       GTK_SIGNAL_FUNC (verify_pixmap_name), (gpointer) wtexture);
   gtk_table_attach (GTK_TABLE (table), wtexture->wpixmap [PIXMAP_NAME],
		     0, 2, 1, 2, GTK_FILL | GTK_SHRINK, GTK_FILL | GTK_SHRINK,
		     0, 0);
   gtk_tooltips_set_tip (tooltips, wtexture->wpixmap [PIXMAP_NAME],
			 "Filename of the pixmap. <RETURN> checks if "
			 "the pixmap filename is valid.", NULL);

   /*
    *  Browse button
    */
   {
      GtkWidget	*bbox;
      GtkWidget	*vbox;

      vbox = gtk_vbox_new (FALSE, 5);
      gtk_table_attach (GTK_TABLE (table), vbox, 1, 2, 0, 1,
			GTK_FILL | GTK_SHRINK, GTK_FILL | GTK_SHRINK, 0, 0);
      
      bbox = gtk_vbutton_box_new ();
      gtk_box_pack_end (GTK_BOX (vbox), bbox, FALSE, TRUE, 5);
  
      wtexture->wpixmap [PIXMAP_BROWSE] = gtk_button_new_with_label ("Browser");
      gtk_signal_connect (GTK_OBJECT (wtexture->wpixmap [PIXMAP_BROWSE]),
			  "pressed",
			  GTK_SIGNAL_FUNC (file_selection), (gpointer) wtexture);
      gtk_box_pack_start (GTK_BOX (bbox), wtexture->wpixmap [PIXMAP_BROWSE],
			  FALSE, TRUE, 5);
      gtk_tooltips_set_tip (tooltips, wtexture->wpixmap [PIXMAP_BROWSE],
			    "Use file browser to select pixmap.", NULL);

      if (preview_browser)
      {
	 wtexture->wpixmap [PIXMAP_PREVIEWS]
	    = gtk_button_new_with_label ("Previews");
	 gtk_signal_connect (GTK_OBJECT (wtexture->wpixmap [PIXMAP_PREVIEWS]),
			     "pressed", GTK_SIGNAL_FUNC (image_browser),
			     (gpointer) wtexture);
	 gtk_box_pack_start (GTK_BOX (bbox), wtexture->wpixmap [PIXMAP_PREVIEWS],
			     FALSE, TRUE, 5);
	 gtk_tooltips_set_tip (tooltips, wtexture->wpixmap [PIXMAP_PREVIEWS],
			       "Use preview browser to select pixmap.", NULL);
      }
      else
	 wtexture->wpixmap [PIXMAP_PREVIEWS] = NULL;
      
#ifdef TILE_OF_DAY      
      if (strcaseeq (PLGetString (wtexture->key), "IconBack"))
      {
	 wtexture->wpixmap [PIXMAP_TILES]
	    = gtk_button_new_with_label ("Tiles");
	 gtk_box_pack_start (GTK_BOX (bbox), wtexture->wpixmap [PIXMAP_TILES],
			     FALSE, TRUE, 5);
	 gtk_signal_connect (GTK_OBJECT (wtexture->wpixmap [PIXMAP_TILES]),
			     "pressed", GTK_SIGNAL_FUNC (tileofday_dialog),
			     (gpointer) wtexture);
	 gtk_tooltips_set_tip (tooltips, wtexture->wpixmap [PIXMAP_TILES],
			       "Tiles from themes.org.", NULL);
      }
#else  /* not TILE_OF_DAY */
      wtexture->wpixmap [PIXMAP_TILES] = NULL;
#endif /* not TILE_OF_DAY */     
   }
   
   wtexture->wpixmap [PIXMAP_FRAME] = frame;
   gtk_widget_show_all (frame);

   if (!strcaseeq (type + 1, "pixmap"))
   {
      gtk_widget_hide (wtexture->wpixmap [PIXMAP_FRAME]);
      gtk_widget_set_sensitive (wtexture->wpixmap [PIXMAP_NAME], FALSE);
      gtk_widget_set_sensitive (wtexture->wpixmap [PIXMAP_BROWSE], FALSE);
      if (wtexture->wpixmap [PIXMAP_PREVIEWS])
	 gtk_widget_set_sensitive (wtexture->wpixmap [PIXMAP_PREVIEWS], FALSE);
      if (wtexture->wpixmap [PIXMAP_TILES])
	 gtk_widget_set_sensitive (wtexture->wpixmap [PIXMAP_TILES], FALSE);
   }

   return frame;
}

static void
file_selection (GtkWidget *widget, wtexture_t *wtexture)
/*
 *  Show pixmap file selection popup window.
 *
 *  No return value.
 */
{
   static GtkWidget	*filesel = NULL;

   if (!filesel)
   {
      GtkFileSelection *fs;

      /*
       *  Fileselection dialog window
       */
      filesel = gtk_file_selection_new ("Pixmap file browser");
      fs      = GTK_FILE_SELECTION (filesel);
      gtk_file_selection_hide_fileop_buttons (fs);
      gtk_window_position (GTK_WINDOW (filesel), GTK_WIN_POS_MOUSE);
      gtk_object_set_user_data (GTK_OBJECT (fs), wtexture);
      gtk_signal_connect (GTK_OBJECT (filesel), "destroy",
                          GTK_SIGNAL_FUNC (gtk_widget_destroyed),
                          &filesel);
      gtk_signal_connect (GTK_OBJECT (fs->ok_button), "clicked",
			  GTK_SIGNAL_FUNC (set_pixmap),
			  (gpointer) filesel);
      gtk_signal_connect_object (GTK_OBJECT (fs->cancel_button), "clicked",
				 GTK_SIGNAL_FUNC (gtk_widget_destroy),
				 GTK_OBJECT (filesel));
      gtk_signal_connect_object (GTK_OBJECT (fs->ok_button), "clicked",
				 GTK_SIGNAL_FUNC (gtk_widget_destroy),
				 GTK_OBJECT (filesel));
      
      /*
       *  Initial filename and pixmap display window
       */
      {
	 char *filename;

	 filename = get_pixmap_path (PLGetString (PLGetArrayElement (PLGetDictionaryEntry (windowmaker, wtexture->key), 1)));
	 if (filename)
	    gtk_file_selection_set_filename (fs, filename);
	 /*
	  *  Display of pixmap 
	  */
	 {
	    GtkWidget *frame = gtk_frame_new (NULL);
	    GtkWidget *pixmap = make_pixmap (filename ? filename :
					     "BulletHole.xpm", 64, 64, NULL);
	    GtkWidget *hbox = gtk_hbox_new (FALSE, 0);

	    frame = gtk_frame_new (NULL);
	    gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_IN);
	    gtk_container_border_width (GTK_CONTAINER (frame), 0);
	    gtk_container_add (GTK_CONTAINER (frame), pixmap);
	    
	    gtk_box_pack_start (GTK_BOX (fs->main_vbox), hbox, FALSE, FALSE, 10);
	    gtk_box_pack_start (GTK_BOX (hbox), frame, TRUE, FALSE, 10);

	    gtk_signal_connect (GTK_OBJECT (fs->selection_entry), "changed",
				GTK_SIGNAL_FUNC (update_pixmap),
				(gpointer) pixmap);

	    gtk_widget_show_all (hbox);
	 }
	 if (filename)
	    Free (filename);
      }
      /*
       *  Optionmenu pixmap path
       */
      {
	 const char	**pathlist;
	 unsigned	n;
	 GtkWidget	*box;
	 GtkWidget	**wpixmap;
	 
	 wpixmap = gtk_object_get_user_data (GTK_OBJECT (widget));
	 pathlist = Calloc (PLGetNumberOfElements (pixmap_path) + 1,
			    sizeof (char *));

	 for (n = 0; n < PLGetNumberOfElements (pixmap_path); n++)
	    pathlist [n] = PLGetString (PLGetArrayElement (pixmap_path, n));
	 pathlist [n] = NULL;

	 box = generate_option_menu ("Pixmap path", pathlist, pathlist [0],
				     change_path, filesel);
	 gtk_box_pack_start (GTK_BOX (fs->main_vbox), box, FALSE, TRUE, 5);
      }


   }
   
   if (!GTK_WIDGET_VISIBLE (filesel))
      gtk_widget_show (filesel);
   else
      gtk_widget_destroy (filesel);
}

static void
update_pixmap (GtkWidget *widget, GtkWidget *pixmap)
{
   char *path = gtk_entry_get_text (GTK_ENTRY (widget));
   char *filename;

   if (!(strstr (path, ".xpm") || strstr (path, ".jpg")
	 || strstr (path, ".jpeg") || strstr (path, ".png")
	 || strstr (path, ".tif") || strstr (path, ".XPM")
	 || strstr (path, ".JPG") || strstr (path, ".JPEG")
	 || strstr (path, ".PNG") || strstr (path, ".TIF")))
      return;
   
   filename = get_pixmap_path (path);
   if (filename)
   {
      make_pixmap (filename, 64, 64, pixmap);
      Free (filename);
   }
}

static void
change_path (GtkWidget *widget, gpointer filesel)
/*
 *  Change current path of fileselection dialog 'filesel'.
 *  'path' is given by user data of the option menu 'widget'.
 *
 *  No return value.
 */
{
   const char *path = gtk_object_get_user_data (GTK_OBJECT (widget));

   if (path [strlen (path) - 1] != '/')	/* append a slash */
   {
      char *tmp = Calloc (strlen (path) + 2, sizeof (char *));

      strcpy (tmp, path);
      strcat (tmp, "/");
      gtk_file_selection_set_filename (GTK_FILE_SELECTION (filesel), tmp);
      Free (tmp);
   }
   else
      gtk_file_selection_set_filename (GTK_FILE_SELECTION (filesel), path);
}

static void
set_pixmap (GtkWidget *widget, gpointer filesel)
/*
 *  Set pixmap name of texture.
 *
 *  No return value.
 *
 *  Side effects:
 *	value of wmaker array element 1 of attribute wtexture->key is modified.
 *	update of filename entry and pixmap
 *	
 */
{
   const char		*filename;
   GtkFileSelection	*fs       = GTK_FILE_SELECTION (filesel);
   wtexture_t		*wtexture = gtk_object_get_user_data (GTK_OBJECT (fs));
   char			*path     = gtk_file_selection_get_filename (fs);

   if ((filename = strrchr (path, '/'))) /* remove path components */
      filename++;
   else
      filename = path;
   /*
    *  Check if pixmap is in PixmapPath
    */
   {
      char *tmp = get_pixmap_path (filename);

      if (!tmp)				/* pixmap not found in PixmapPath */
	 filename = path;
      else
	 Free (tmp);
   }
   
   PLRemoveArrayElement (PLGetDictionaryEntry (windowmaker, wtexture->key), 1);
   PLInsertArrayElement (PLGetDictionaryEntry (windowmaker, wtexture->key),
			 PLMakeString ((char *) filename), 1);

   make_pixmap (filename, 128, 128, wtexture->wpixmap [PIXMAP_PIXMAP]);
   gtk_entry_set_text (GTK_ENTRY (wtexture->wpixmap [PIXMAP_NAME]), filename);

#ifdef TILE_OF_DAY
   check_valid_tileofday (wtexture);
#endif /* TILE_OF_DAY */

   PLInsertDictionaryEntry (plist_changed, wtexture->key, pl_yes);
   changed = YES;
}

static GtkWidget *
gradient_dialog (wtexture_t *wtexture)
/*
 *  Generate all widgets for the gradient dialog.
 *
 *  Return value:
 *	Container widget of gradient dialog.
 */
{
   GtkWidget  *hbox;
   GtkWidget  *frame;
   GtkWidget  *scrolled;
   proplist_t *array = PLGetDictionaryEntry (windowmaker, wtexture->key);
   char	      *type  = PLGetString (PLGetArrayElement (array, 0));
   GtkWidget  *table;
   
   wtexture->wgradient = Calloc (GRADIENT_LAST, sizeof (GtkWidget *));
 
   frame = gtk_frame_new ("Gradient");
   gtk_frame_set_label_align (GTK_FRAME (frame), 0.5, 0.5);
   gtk_container_border_width (GTK_CONTAINER (frame), 5);
   
   table = gtk_table_new (1, 1, FALSE);
   gtk_container_border_width (GTK_CONTAINER (table), 5);
   gtk_table_set_row_spacings (GTK_TABLE (table), 5);
   gtk_table_set_col_spacings (GTK_TABLE (table), 5);
   gtk_container_add (GTK_CONTAINER (frame), table);
   
   hbox = gtk_hbox_new (FALSE, 5);
   gtk_table_attach (GTK_TABLE (table), hbox, 0, 1, 0, 1,
		     GTK_FILL | GTK_SHRINK, GTK_FILL | GTK_SHRINK, 0, 0);

   /*
    *  Scrolled window for list
    */
   scrolled = gtk_scrolled_window_new (NULL, NULL);
   gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled),
				   GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
   gtk_widget_set_usize (scrolled, 63, 100);
   gtk_box_pack_start (GTK_BOX (hbox), scrolled, FALSE, FALSE, 5);
   
   /*
    *  Color list
    */
   wtexture->wgradient [GRADIENT_LIST] = gtk_list_new ();
   gtk_list_set_selection_mode (GTK_LIST (wtexture->wgradient [GRADIENT_LIST]),
				GTK_SELECTION_BROWSE);
   gtk_container_add (GTK_CONTAINER (scrolled),
		      wtexture->wgradient [GRADIENT_LIST]);
   gtk_object_set_user_data (GTK_OBJECT (wtexture->wgradient [GRADIENT_LIST]),
			     wtexture->key);
   
   /*
    *  Initial list elements
    */
   {
      unsigned  i;

      if (!strcaseeq (type + 1, "pixmap") && !strcaseeq (type, "solid"))
	 for (i = 2; i < PLGetNumberOfElements (array); i++)
	    gtk_container_add (GTK_CONTAINER (wtexture->wgradient [GRADIENT_LIST]),
			       new_list_item (PLGetArrayElement (array, i)));
   }
   gtk_list_select_item (GTK_LIST (wtexture->wgradient [GRADIENT_LIST]), 0);
   /*
    *  List manipulating buttons
    */
   {
      GtkWidget  *bbox; 
      const char *text [] = {"Remove", "Insert", "Append", "Change"};
      const char *info [] = {"Remove selected color from gradient color list.",
			     "Insert a new color before current color.",
			     "Append a new color after the last color.",
			     "Change the current color."};
      void (*fct []) (GtkWidget *, GtkWidget *list) = {list_remove, list_insert,
						       list_append, list_change};
      unsigned	i;
      
      bbox = gtk_vbutton_box_new ();
      gtk_box_pack_start (GTK_BOX (hbox), bbox, FALSE, TRUE, 5);

      for (i = 0; i < 4; i++)
      {
	 wtexture->wgradient [i + 1] = gtk_button_new_with_label (text [i]);
	 gtk_box_pack_start (GTK_BOX (bbox), wtexture->wgradient [i + 1],
			     FALSE, TRUE, 5);
	 gtk_object_set_user_data (GTK_OBJECT (wtexture->wgradient [i + 1]),
				   wtexture);
	 gtk_signal_connect (GTK_OBJECT (wtexture->wgradient [i + 1]), "clicked",
			     GTK_SIGNAL_FUNC (fct [i]),
			     (gpointer) wtexture->wgradient [0]);
	 gtk_tooltips_set_tip (tooltips, wtexture->wgradient [i + 1],
			       info [i], NULL);  
      }
   }

   wtexture->wgradient [GRADIENT_FRAME] = frame;
   gtk_widget_show_all (frame);

   if (!strcaseeq (type + 2, "gradient"))
   {
      gtk_widget_hide (wtexture->wgradient [GRADIENT_FRAME]);
      gtk_widget_set_sensitive (wtexture->wgradient [GRADIENT_REMOVE], FALSE);
      gtk_widget_set_sensitive (wtexture->wgradient [GRADIENT_INSERT], FALSE);
      gtk_widget_set_sensitive (wtexture->wgradient [GRADIENT_APPEND], FALSE);
      gtk_widget_set_sensitive (wtexture->wgradient [GRADIENT_CHANGE], FALSE);
   }
   else if (g_list_length (GTK_LIST (wtexture->wgradient [GRADIENT_LIST])->children) == 2)
      gtk_widget_set_sensitive (wtexture->wgradient [GRADIENT_REMOVE], FALSE);
   
   return frame;
}   

static GtkWidget *
new_list_item (proplist_t *color)
/*
 *  Allocate new color cell in color list.
 *
 *  Return value:
 *	list item
 */
{
   GtkWidget *list_item;
   GtkWidget *preview;
   GtkWidget *lhbox;
      
   list_item = gtk_list_item_new ();
   gtk_widget_show (list_item);
   gtk_container_border_width (GTK_CONTAINER (list_item), 2);
      
   lhbox = gtk_hbox_new (FALSE, 2);
   gtk_widget_show (lhbox);
   gtk_container_add (GTK_CONTAINER (list_item), lhbox);
			
   preview = gtk_preview_new (GTK_PREVIEW_COLOR);
   gtk_preview_size (GTK_PREVIEW (preview), 20, 15);
   gtk_box_pack_start (GTK_BOX (lhbox), preview, FALSE, TRUE, 5);
   gtk_widget_show (preview);
   fill_preview (GTK_PREVIEW (preview), PLGetString (color));
   gtk_object_set_user_data (GTK_OBJECT (list_item), preview);
   gtk_object_set_user_data (GTK_OBJECT (preview), color);

   return list_item;
}

static char *last_color = NULL;		/* save last color name */

static void
list_insert (GtkWidget *widget, GtkWidget *list)
/*
 *  Insert new color cell before current position.
 *
 *  No return value.
 *
 *  Side effects:
 *	new element is inserted in wmaker array attribute wtexture->key
 */
{
   GtkWidget  *list_item;
   int	      position;
   proplist_t *key      = gtk_object_get_user_data (GTK_OBJECT (list));
   proplist_t *array    = PLGetDictionaryEntry (windowmaker, key);
   proplist_t *element  = PLMakeString (last_color?last_color:strdup ("black"));
   wtexture_t *wtexture = gtk_object_get_user_data (GTK_OBJECT (widget));

   position = gtk_list_child_position (GTK_LIST (list),
				       (GTK_LIST (list)->selection)->data);
   
   PLInsertArrayElement (array, element, position + 2);
   list_item = new_list_item (element);
   /*
    *  Insert list item at selection
    */
   {
      GList *gl = Calloc (1, sizeof (GList));
      
      gl->next = gl->prev = NULL;
      gl->data = list_item;

      gtk_list_insert_items (GTK_LIST (list), gl, position);
   }

   if (g_list_length (GTK_LIST (list)->children) > 2)
      gtk_widget_set_sensitive (wtexture->wgradient [GRADIENT_REMOVE], TRUE);
   else
      gtk_widget_set_sensitive (wtexture->wgradient [GRADIENT_REMOVE], FALSE);

   PLInsertDictionaryEntry (plist_changed, wtexture->key, pl_yes);
   changed = YES;
}

static void
list_append (GtkWidget *widget, GtkWidget *list)
/*
 *  Append new color cell after last position.
 *
 *  No return value.
 *
 *  Side effects:
 *	new element is inserted in wmaker array attribute wtexture->key
 */
{
   proplist_t *key      = gtk_object_get_user_data (GTK_OBJECT (list));
   proplist_t *array    = PLGetDictionaryEntry (windowmaker, key);
   proplist_t *element  = PLMakeString (last_color?last_color:strdup("black"));
   wtexture_t *wtexture = gtk_object_get_user_data (GTK_OBJECT (widget));

   PLAppendArrayElement (array, element);
   
   gtk_container_add (GTK_CONTAINER (list), new_list_item (element));
   if (g_list_length (GTK_LIST (list)->children) > 2)
      gtk_widget_set_sensitive (wtexture->wgradient [GRADIENT_REMOVE], TRUE);
   else
      gtk_widget_set_sensitive (wtexture->wgradient [GRADIENT_REMOVE], FALSE);

   PLInsertDictionaryEntry (plist_changed, wtexture->key, pl_yes);
   changed = YES;
}

static void
list_remove (GtkWidget *widget, GtkWidget *list)
/*
 *  Remove color cell at current position.
 *
 *  No return value.
 *
 *  Side effects:
 *	one element of wmaker array attribute wtexture->key is removed.
 */
{
   GList      *tmp_list;
   GList      *clear_list;
   unsigned   i;
   proplist_t *key      = gtk_object_get_user_data (GTK_OBJECT (list));
   proplist_t *array    = PLGetDictionaryEntry (windowmaker, key);
   wtexture_t *wtexture = gtk_object_get_user_data (GTK_OBJECT (widget));
   
   tmp_list   = GTK_LIST (list)->selection;
   clear_list = NULL;

   /*
    *  Remove color from proplist array
    */
   for (i = 2; i < PLGetNumberOfElements (array); i++)
   {
      GtkPreview *p = gtk_object_get_user_data (GTK_OBJECT (tmp_list->data));
      
      if (PLGetArrayElement (array, i)
	  == gtk_object_get_user_data (GTK_OBJECT (p)))
      {
	 if (last_color)
	    Free (last_color);
	 last_color = strdup (PLGetString (PLGetArrayElement (array, i)));
	 PLRemoveArrayElement (array, i);
	 break;
      }
   }
   
   /*
    *  Remove list item
    */
   while (tmp_list)
   {
      clear_list = g_list_prepend (clear_list, tmp_list->data);
      tmp_list = tmp_list->next;
   }

   clear_list = g_list_reverse (clear_list);
   gtk_list_remove_items (GTK_LIST (list), clear_list);
   g_list_free (clear_list);

   if (g_list_length (GTK_LIST (list)->children))
      gtk_list_select_item (GTK_LIST (list), 0);

   if (g_list_length (GTK_LIST (list)->children) > 2)
      gtk_widget_set_sensitive (wtexture->wgradient [GRADIENT_REMOVE], TRUE);
   else
      gtk_widget_set_sensitive (wtexture->wgradient [GRADIENT_REMOVE], FALSE);

   PLInsertDictionaryEntry (plist_changed, wtexture->key, pl_yes);
   changed = YES;
}

static void
list_change (GtkWidget *widget, GtkWidget *list)
/*
 *  Popup a colorselection dialog to change selectef color cell.
 *
 *  No return value.
 *
 *  Side effects:
 *	value of wmaker array attribute wtexture->key is modified.
 */
{
   GtkWidget	*list_item;

   list_item = (GTK_LIST (list)->selection)->data;

   color_selection_dialog (list_item,
			   gtk_object_get_user_data (GTK_OBJECT (list)));
}

static void
verify_pixmap_name (GtkWidget *widget, wtexture_t *wtexture)
/*
 *  Verify if typed pixmap name is valid.
 *
 *  No return value.
 *
 *  Side effects:
 *	pixmap is changed 
 */
{
   char *filename = gtk_entry_get_text (GTK_ENTRY (widget));

   make_pixmap (filename, 128, 128, wtexture->wpixmap [PIXMAP_PIXMAP]);
#ifdef TILE_OF_DAY
   check_valid_tileofday (wtexture);
#endif /* TILE_OF_DAY */
}

static void
set_pixmap_name (GtkWidget *widget, gpointer ptr)
/*
 *  Update pixmap string value.
 *
 *  No return value.
 */
{
   wtexture_t *wtexture  = (wtexture_t *) ptr;
   proplist_t *attribute = wtexture->key;
   proplist_t *element;
   proplist_t *array = PLGetDictionaryEntry (windowmaker, attribute);

   element = PLMakeString ((char *) gtk_entry_get_text (GTK_ENTRY (widget)));

   PLRemoveArrayElement (array, 1);	/* filename */
   PLInsertArrayElement (array, element, 1);
   
#ifdef TILE_OF_DAY
   check_valid_tileofday (wtexture);
#endif /* TILE_OF_DAY */
   PLInsertDictionaryEntry (plist_changed, wtexture->key, pl_yes);
   changed = YES;
}

