/*
 * Copyright (C) 1997 and 1998 WIDE Project.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLER
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * $Id: print.c,v 1.62 1998/09/06 02:58:02 itojun Exp $
 */
/*
 * Paper size selection code is based on psutil.c by Angus J. C. Duggan
 */

#include "mgp.h"

static u_int align = AL_LEFT;
static u_char *curprefix = NULL;
static u_char *tabprefix = NULL;
static u_int lineheight;
static u_int linewidth;

#define DEFAULT_PAPER_SIZE	"a4"
#define PRINT_ASCIIFONT		"Times-Roman"
#define PRINT_ASCIITHICKFONT	"Helvetica"
#define PRINT_KANJIFONT		"Ryumin-Light-H"
#define PRINT_KANJITHICKFONT	"GothicBBB-Medium-H"

#define PSSCALEF	1.25
#define PSSCALE		(100.0 / 80.0)
#define PSBASE		(80.0 / 100.0)

static FILE *fp;
static u_int reverse;
static u_char outputfile[MAXVALLEN];
static int colorps = 0;
static int painticon = 0;
static u_int curlinenum;
static u_int curpagenum;

static struct imagepool {
	struct ctrl *image;
	int xoffset;
	int xsiz;
	int ysiz;
	int eps;
} imagepool[256];		/*enough?*/
static int nimagepool = 0;
static struct textpool {
	int pfx;
	int xoffset;
	int xsiz;
	struct fontmap *font;
	int size;
	char *text;
	u_long fore;
	u_long back;
} textpool[1024];		/*enough?*/
static int ntextpool = 0;

u_long fore;
u_long back;

#ifdef VFLIB
char *vfcap_name = NULL;	/*dummy*/
#endif
#ifdef FREETYPE
char *freetypefontdir = NULL;	/*dummy*/
char *freetypefont0 = NULL;	/*dummy*/
#endif

/* lang code */
#define ASCII	0
#define KANJI	1
#define NOPE	-1

static int maxfontid = 0;
static struct fontmap {
	int ctrl;
	int lang;
	char *font;
	char *psfont;
	int fontid;	/* flag bit for generating font map */
	int loaded;
} fontmap[] = {
#ifdef VFLIB
	{ CTL_VFONT, KANJI, "goth",		"GothicBBB-Medium-H" },
	{ CTL_VFONT, KANJI, "*",		"Ryumin-Light-H" },
#endif /*VFLIB*/
#ifdef FREETYPE
	{ CTL_TFONT, ASCII, "times.ttf",	"Times-Roman" },
	{ CTL_TFONT, ASCII, "timesi.ttf",	"Times-Italic" },
	{ CTL_TFONT, ASCII, "timesbd.ttf",	"Times-Bold" },
	{ CTL_TFONT, ASCII, "timesbi.ttf",	"Times-BoldItalic" },
	{ CTL_TFONT, ASCII, "arial.ttf",	"Helvetica" },
	{ CTL_TFONT, ASCII, "ariali.ttf",	"Helvetica-Oblique" },
	{ CTL_TFONT, ASCII, "arialbd.ttf",	"Helvetica-Bold" },
	{ CTL_TFONT, ASCII, "arialbi.ttf",	"Helvetica-BoldOblique" },
	{ CTL_TFONT, ASCII, "cour.ttf",		"Courier" },
	{ CTL_TFONT, ASCII, "couri.ttf",	"Courier-Oblique" },
	{ CTL_TFONT, ASCII, "courbd.ttf",	"Courier-Bold" },
	{ CTL_TFONT, ASCII, "courbi.ttf",	"Courier-BoldOblique" },
	{ CTL_TFONT, ASCII, "*",		"Helvetica" },	/*last resort*/
#endif /*FREETYPE*/
	{ CTL_XFONT, ASCII, "times-medium-r",	"Times-Roman" },
	{ CTL_XFONT, ASCII, "times-medium-i",	"Times-Italic" },
	{ CTL_XFONT, ASCII, "times-bold-r",	"Times-Bold" },
	{ CTL_XFONT, ASCII, "times-bold-i",	"Times-BoldItalic" },
	{ CTL_XFONT, ASCII, "helvetica-medium-r",	"Helvetica" },
	{ CTL_XFONT, ASCII, "helvetica-medium-o",	"Helvetica-Oblique" },
	{ CTL_XFONT, ASCII, "helvetica-bold-r",	"Helvetica-Bold" },
	{ CTL_XFONT, ASCII, "helvetica-bold-o",	"Helvetica-BoldOblique" },
	{ CTL_XFONT, ASCII, "courier-medium-r",	"Courier" },
	{ CTL_XFONT, ASCII, "courier-medium-o",	"Courier-Oblique" },
	{ CTL_XFONT, ASCII, "courier-bold-r",	"Courier-Bold" },
	{ CTL_XFONT, ASCII, "courier-bold-i",	"Courier-BoldOblique" },
	{ CTL_XFONT, ASCII, "times*",		"Times" },
	{ CTL_XFONT, ASCII, "helvetica*",	"Helvetica" },
	{ CTL_XFONT, ASCII, "courier*",		"Courier" },
	{ CTL_XFONT, ASCII, "*",		"Helvetica" },	/*last resort*/
	{ -1 },
};
static struct fontmap *curfont[10];	/*indexed by lang*/

static u_short kinsokutable[] = {
	0x2121, 0x2122, 0x2123, 0x2124, 0x2125, 0x2126, 0x2127, 0x2128,
	0x2129, 0x212a, 0x212b, 0x212c, 0x212d, 0x212e, 0x212f, 0x2130,
	0x2133, 0x2134, 0x2135, 0x2136, 0x213c, 0x2147, 0x2149, 0x214b,
	0x214d, 0x214f, 0x2151, 0x2153, 0x2155, 0x2157, 0x2159, 0x216b,
	0x2242, 0x2244, 0
};

typedef struct papersize {
	char *name;		/* name of paper size */
	int height, width;	/* height, width in points for LANDSCAPE */
} Paper;

static Paper papersizes[] = {
	{ "a3",		842, 1191 },	/* 29.7cm * 42cm */
	{ "a4",		595, 842 },	/* 21cm * 29.7cm */
	{ "a5",		421, 595 },	/* 14.85cm * 21cm */
	{ "b5",		516, 729 },	/* 18.2cm * 25.72cm */
	{ "A3",		842, 1191 },	/* 29.7cm * 42cm */
	{ "A4",		595, 842 },	/* 21cm * 29.7cm */
	{ "A5",		421, 595 },	/* 14.85cm * 21cm */
	{ "B5",		516, 729 },	/* 18.2cm * 25.72cm */
	{ "letter",	612, 792 },	/* 8.5in * 11in */
	{ "legal",	612, 1008 },	/* 8.5in * 14in */
	{ "ledger",	1224, 792 },	/* 17in * 11in */
	{ "tabloid",	792, 1224 },	/* 11in * 17in */
	{ "statement",	396, 612 },	/* 5.5in * 8.5in */
	{ "executive",	540, 720 },	/* 7.6in * 10in */
	{ "folio",	612, 936 },	/* 8.5in * 13in */
	{ "quarto",	610, 780 },	/* 8.5in * 10.83in */
	{ "10x14",	720, 1008 },	/* 10in * 14in */
	{ NULL,	0, 0 }
};

static Paper *paper;
static int w_width, w_height;
static int paper_xmargin = 10;
static int paper_ymargin = 5;

static Paper *findpaper __P((char *));
static void print_out __P((void));
static void scan_font __P((u_int));
static void print_page __P((u_int));
static void print_init __P((void));
static void print_pageinit __P((u_int));
static void print_pagebackground __P((void));
static void process_direc_print __P((struct ctrl *));
static void line_start __P((void));
static void line_end __P((void));
static void icon_output __P((struct textpool *));
static void line_skip __P((int, int));
static void print_bar __P((struct ctrl *));
static void print_icon __P((struct ctrl *));
static void print_line __P((u_char *, int));
static int iskinsokuchar __P((u_short));
static void print_string __P((u_char *, int));
static void print_fragment __P((u_char *, u_int, int, int));
static void text_remember __P((char *, int, int, int, int));
static void icon_remember __P((int, int, int, u_long));
static void image_remember __P((struct ctrl *, struct imagepool *));
static void image_load_print __P((char *, int, int, int, int, int));
static void print_usage __P((char *));
static int set_position __P((void));
static void setpencolor __P((u_long));
static void print_full_image __P((XImage *, Visual *));
static void print_8_image __P((XImage *));
static void print_32_image __P((XImage *, Visual *));
static void print_eps __P((FILE *, char *, int, int, int, int, int));
static struct fontmap *findfont __P((int, int, char *));
static char *fontstring __P((struct fontmap *));
static void loadfont __P((struct fontmap *));

/* image/ *.[ch] */
extern char *Paths[];
extern int NumPaths;
extern char *expandPath __P((char *));

extern int findImage __P((char *, char *));

static Paper *
findpaper(name)
	char *name;
{
	Paper *pp;
	for (pp = papersizes; pp->name; pp++) {
		if (strcmp(pp->name, name) == 0)
			return pp;
	}
	return (Paper *)NULL;
}

static void
print_out()
{
	u_int	i;
	u_int	width, height;

	width  = window_width;
	height = window_height;
	window_width  = w_width;
	window_height = w_height;

	char_size = window_height * DEFAULT_CHARSIZE / 100;

	if (outputfile[0]) {
		if ((fp = fopen(outputfile, "w")) == NULL) {
			perror("fopen");
			exit(-1);
		}
	} else
		fp = stdout;

	print_init();
	for (i = 1; i <= maxpage; i ++)
		scan_font(i);
	for (i = 1; i <= maxpage; i ++)
		print_page(i);

	fclose(fp);

	window_width  = width;
	window_height = height;
}

static void
scan_font(page)
	u_int page;
{
	u_int line;
	struct ctrl *cp;
	struct fontmap *font;

	if (mgp_flag & FL_VERBOSE)
		fprintf(fp, "%% scan_font page %d\n", page);
	for (line = 0;
	     line <= page_attribute[page].pg_linenum;
	     line++) {
		/* process normal control */
		for (cp = page_control[page][line]; cp; cp = cp->ct_next) {
			switch(cp->ct_op) {
#ifdef VFLIB
			case CTL_VFONT:
				font = findfont(cp->ct_op, KANJI,
					cp->ctc_value);
				if (font)
					curfont[KANJI] = font;
				break;
#endif /*VFLIB*/

#ifdef FREETYPE
			case CTL_TFONT:
#endif
			case CTL_XFONT:
				font = findfont(cp->ct_op, ASCII,
					cp->ctc_value);
				if (font)
					curfont[ASCII] = font;
				break;

			case CTL_TEXT:
				if (strstr(cp->ctc_value, "\033$B")
				 || strstr(cp->ctc_value, "\033$@"))
					loadfont(curfont[KANJI]);
				loadfont(curfont[ASCII]);
				break;
			default:
				break;
			}
		}
	}

	memset(curfont, 0, sizeof(curfont));
}

static void
print_page(page)
	u_int page;
{
	u_int line;
	struct ctrl *cp;

	curpagenum = page;

	print_pageinit(curpagenum);

	for (line = 0;
	     line <= page_attribute[page].pg_linenum;
	     line++) {
		curlinenum = line;
		/* process normal control */
		for (cp = page_control[page][line]; cp; cp = cp->ct_next)
			process_direc_print(cp);
	}

	fprintf(fp, "showpage\n\n");
}

static void
print_init()
{
	char **p;
	char *aligns[] = { "center", "left", "right", NULL };

	fprintf(fp, "%%!PS-Adobe-2.0 EPSF-2.0\n");
	fprintf(fp, "%%%%Creater: mgp2ps\n");
	fprintf(fp, "%%%%Title: %s\n", mgp_fname);
	fprintf(fp, "%%%%Pages: %d\n", maxpage);
	fprintf(fp, "%%%%BoundingBox: 0 0 %d %d\n", paper->height, paper->width);
	fprintf(fp, "%%%%DocumentPaperSizes: %s\n", paper->name);
	fprintf(fp, "%%%%Orientation: Landscape\n");
	fprintf(fp, "%%%%EndComments\n");

	/* define constants */
	fprintf(fp, "/XMARGIN %d def /YMARGIN %d def "
		"/WIDTH %d def /HEIGHT %d def\n",
		paper_xmargin, paper_ymargin, window_width, window_height);
	fprintf(fp, "/XBODY WIDTH XMARGIN 2 mul sub def\n");
	fprintf(fp, "/vertgap %d def /horizgap %d def\n", vert_gap, horiz_gap);

	/* define writebox */
	fprintf(fp, "/writebox {\n");
	fprintf(fp, "  XMARGIN YMARGIN -1 mul moveto "
			"0 HEIGHT -1 mul rlineto\n");
	fprintf(fp, "  WIDTH 0 rlineto 0 HEIGHT rlineto\n");
	fprintf(fp, "  WIDTH -1 mul 0 rlineto stroke\n");
	fprintf(fp, "} def\n");

	/* define writeboxfill */
	fprintf(fp, "/writeboxfill {\n");
	fprintf(fp, "  newpath XMARGIN YMARGIN -1 mul moveto "
			"0 HEIGHT -1 mul rlineto\n");
	fprintf(fp, "  WIDTH 0 rlineto 0 HEIGHT rlineto\n");
	fprintf(fp, "  WIDTH -1 mul 0 rlineto closepath eofill stroke\n");
	fprintf(fp, "} def\n");

	/* ypos = ypos - (charsize * (1 + vert_gap / 100)) */
	fprintf(fp, "/NL {\n");
	fprintf(fp, "  /ypos ypos vertgap 100 div 1 add charsize mul sub def\n");
	fprintf(fp, "} bind def\n");

	/* show with line wrapping */
	fprintf(fp, "/initcharsize { /charsize 0 def } def initcharsize\n");
	fprintf(fp, "/setcharsize {\n");
	fprintf(fp, "  dup charsize gt { dup /charsize exch def } if pop\n");
	fprintf(fp, "} def\n");

	fprintf(fp, "/updatetotlen {\n");
	fprintf(fp, "  dup totlen exch sub /totlen exch def\n");
	fprintf(fp, "} bind def\n");

	fprintf(fp, "/updatefillzero {\n");
	fprintf(fp, "  inmargin {\n");
	fprintf(fp, "    currentpoint pop /fillzero exch def\n");
	fprintf(fp, "    /inmargin false def\n");
	fprintf(fp, "  } if\n");
	fprintf(fp, "} bind def\n");

	/* centering */
	fprintf(fp, "/centerdefxpos {\n");
	fprintf(fp, "  totlen XBODY gt\n");
	fprintf(fp, "    { XMARGIN }\n");
	fprintf(fp, "    { XBODY totlen sub 2 div XMARGIN add }\n");
	fprintf(fp, "  ifelse /xpos exch def\n");
	fprintf(fp, "} bind def\n");

	/* leftfill */
	fprintf(fp, "/leftdefxpos {\n");
	fprintf(fp, "  /xpos fillzero def\n");
	fprintf(fp, "} bind def\n");

	/* rightfill */
	fprintf(fp, "/rightdefxpos {\n");
	fprintf(fp, "  totlen XBODY gt\n");
	fprintf(fp, "    { XMARGIN }\n");
	fprintf(fp, "    { XBODY totlen sub XMARGIN add }\n");
	fprintf(fp, "  ifelse /xpos exch def\n");
	fprintf(fp, "} bind def\n");

	/* check newline */
	for (p = aligns; *p; p++) {
		fprintf(fp, "/%snewlinecheck {\n", *p);
		fprintf(fp, "  currentpoint pop add XMARGIN XBODY add gt {\n");
		fprintf(fp, "    NL %sdefxpos xpos ypos charsize sub moveto\n",
			*p);
		fprintf(fp, "  } if\n");
		fprintf(fp, "} bind def\n");
	}

	/* a spell for EPS */
	fprintf(fp, "%%\n");
	fprintf(fp, "/BeginEPSF {%%def\n");
	fprintf(fp, "  /b4_Inc_state save def\n");
	fprintf(fp, "  /dict_count countdictstack def\n");
	fprintf(fp, "  /op_count count 1 sub def\n");
	fprintf(fp, "  userdict begin\n");
	fprintf(fp, "  /showpage {}def\n");
	fprintf(fp, "  0 setgray 0 setlinecap\n");
	fprintf(fp, "  1 setlinewidth 0 setlinejoin\n");
	fprintf(fp, "  10 setmiterlimit [] 0 setdash\n");
	fprintf(fp, "  newpath\n");
	fprintf(fp, "  /languagelevel where\n");
	fprintf(fp, "  {pop languagelevel\n");
	fprintf(fp, "  1 ne\n");
	fprintf(fp, "    {false setstrokeadjust\n");
	fprintf(fp, "     false setoverprint\n");
	fprintf(fp, "    }if\n");
	fprintf(fp, "  }if\n");
	fprintf(fp, "}bind def\n");
	fprintf(fp, "%%\n");
	fprintf(fp, "/EndEPSF {%%def\n");
	fprintf(fp, "  count op_count sub {pop}repeat\n");
	fprintf(fp, "  countdictstack\n");
	fprintf(fp, "  dict_count sub {end}repeat\n");
	fprintf(fp, "  b4_Inc_state restore\n");
	fprintf(fp, "}bind def\n");
	fprintf(fp, "%%\n");
}

static void
print_pageinit(page)
	u_int page;
{
	fprintf(fp, "%%%%Page: %d %d\n", page, page);
	fprintf(fp, "/ypos %d def\n", - paper_ymargin);
	fprintf(fp, "initcharsize\n");
}

static void
print_pagebackground()
{
	if (!colorps)
		fprintf(fp, "90 rotate newpath writebox\n");
	else {
		setpencolor(back);
		fprintf(fp, "90 rotate newpath writeboxfill\n");
	}
}

static void
process_direc_print(cp)
	struct ctrl *cp;
{
	struct fontmap *font;
	switch(cp->ct_op) {
	case CTL_SIZE:
		char_size = window_height * cp->cti_value / 100;
		fprintf(fp, "%d setcharsize\n", char_size);
		break;

#ifdef VFLIB
	case CTL_VFONT:
		font = findfont(cp->ct_op, KANJI, cp->ctc_value);
		if (font)
			curfont[KANJI] = font;
		break;
#endif /*VFLIB*/

#ifdef FREETYPE
	case CTL_TFONT:
#endif
	case CTL_XFONT:
		font = findfont(cp->ct_op, ASCII, cp->ctc_value);
		if (font)
			curfont[ASCII] = font;
		break;

	case CTL_VGAP:
		vert_gap = cp->cti_value;
		fprintf(fp, "/vertgap %d def\n", cp->cti_value);
		break;

	case CTL_HGAP:
		horiz_gap = cp->cti_value;
		fprintf(fp, "/horizgap %d def\n", cp->cti_value);
		break;

	case CTL_GAP:
		vert_gap = horiz_gap = cp->cti_value;
		fprintf(fp, "/vertgap %d def ", cp->cti_value);
		fprintf(fp, "/horizgap %d def\n", cp->cti_value);
		break;

	case CTL_CENTER:
		align = AL_CENTER;
		break;

	case CTL_LEFT:
		align = AL_LEFT;
		break;

	case CTL_LEFTFILL:
		align = AL_LEFTFILL0;
		break;

	case CTL_RIGHT:
		align = AL_RIGHT;
		break;

	case CTL_IMAGE:
		image_remember(cp, &imagepool[nimagepool]);
		line_skip(imagepool[nimagepool].xsiz,
			imagepool[nimagepool].ysiz);

		/* placeholder */
		textpool[ntextpool].pfx = 0;
		textpool[ntextpool].xoffset = 0;
		textpool[ntextpool].xsiz = imagepool[nimagepool].xsiz;
		textpool[ntextpool].size = imagepool[nimagepool].ysiz;
		textpool[ntextpool].font = 0;		/*image*/
		textpool[ntextpool].text = NULL;
		textpool[ntextpool].fore = fore;	/*XXX*/
		textpool[ntextpool].back = back;	/*XXX*/
		ntextpool++;

		nimagepool++;
		break;

	case CTL_BAR:
		print_bar(cp);
		break;

	case CTL_PREFIX:
		curprefix = cp->ctc_value;
		break;

	case CTL_TABPREFIX:
		tabprefix = cp->ctc_value;
		break;

	case CTL_PREFIXPOS:
		if (tabprefix)
			print_line(tabprefix, 1);
		else if (curprefix)
			print_line(curprefix, 1);
		break;

	case CTL_TEXT:
	    {
		if (!cp->ctc_value)
			break;
		print_line(cp->ctc_value, 0);
		if (align == AL_LEFTFILL0)
			align = AL_LEFTFILL1;
		break;
	    }

	case CTL_ICON:
		print_icon(cp);
		break;

	case CTL_LINESTART:
		line_start();
		break;

	case CTL_LINEEND:
		line_end();
		if (align == AL_LEFTFILL1)
			align = AL_LEFTFILL0;
		break;

	case CTL_FORE:
		fore = cp->ctl_value;
		break;

	case CTL_BACK:
		back = cp->ctl_value;
		break;

	default:
		break;
	}
}

static void
line_start()
{
	linewidth = 0;
	nimagepool = 0;
	ntextpool = 0;

	if (curlinenum == 0)
		print_pagebackground();
}

static void
line_end()
{
	int i;
	char *alignstr;

	switch (align) {
	case AL_CENTER:
		alignstr = "center";
		break;
	case AL_LEFT:
	case AL_LEFTFILL0:
	case AL_LEFTFILL1:
		alignstr = "left";
		break;
	case AL_RIGHT:
		alignstr = "right";
		break;
	default:
		alignstr = "";
		break;
	}

	/*
	 * line output starts here
	 */
	if (mgp_flag & FL_VERBOSE)
		fprintf(fp, "%% a line starts here\n");

	/*
	 * if we have nothing on the line, skip it
	 */
	if (!ntextpool && !nimagepool)
		goto done;

	/*
	 * push strings in reverse order.
	 */
	fprintf(fp, "initcharsize\n");
	fprintf(fp, "0 %% sentinel for text width computation\n");

	for (i = ntextpool - 1; 0 <= i; i--) {
		if (!textpool[i].text) {
			fprintf(fp, "%d add\n", textpool[i].xsiz);
			continue;
		}
#if 0
		loadfont(textpool[i].font);
#endif
		fprintf(fp, "%d setcharsize %d %s %s "
			"1 copy stringwidth pop 3 2 roll add\n",
			textpool[i].size, textpool[i].size,
			fontstring(textpool[i].font),
			textpool[i].text);
	}
	if (mgp_flag & FL_VERBOSE) {
		fprintf(fp, "%% stack should have: str3 str2 str1 width\n");
		fprintf(fp, "%% alignment: %s\n", alignstr);
	}

	/*
	 * now show those chars with line wrapping. brief logic is:
	 *
	 *	position adjust;
	 *	foreach i (item in textpool) {
	 *		if (item exceeds the right edge) {
	 *			carriage return;
	 *			position adjust;
	 *		}
	 *		show;
	 *	}
	 */
	fprintf(fp, "/totlen exch def\n");
	fprintf(fp, "/inmargin true def /fillzero XMARGIN def\n");
	fprintf(fp, "%sdefxpos ", alignstr);
	fprintf(fp, "xpos ypos charsize sub moveto\n");

	for (i = 0; i < ntextpool; i++) {
		if (textpool[i].text) {
#if 0
			loadfont(textpool[i].font);
#endif
			fprintf(fp, "%d %s 1 copy stringwidth pop ",
				textpool[i].size,
				fontstring(textpool[i].font));
		} else
			fprintf(fp, "%d ", textpool[i].xsiz);
		fprintf(fp, "updatetotlen ");
		fprintf(fp, "%snewlinecheck ", alignstr);

		if (colorps
		 && (i == 0
		  || (textpool[i - 1].fore != textpool[i].fore)
		  || (textpool[i - 1].back != textpool[i].back))) {
			fprintf(fp, "\n");
			setpencolor(textpool[i].fore);
		}

		if (textpool[i].text) {
			if (!textpool[i].pfx)
				fprintf(fp, "updatefillzero ");
			fprintf(fp, "show\n");
		} else {
			fprintf(fp, "\n");
			if (textpool[i].font) {
				/* icon */
				icon_output(&textpool[i]);
			} else {
				/* image */
				fprintf(fp, "%d 0 rmoveto\n", textpool[i].xsiz);
			}
		}
	}

	ntextpool = 0;

    {
	struct ctrl *cp1;

	for (i = 0; i < nimagepool; i++) {
		if (mgp_flag & FL_VERBOSE)
			fprintf(fp, "%% emit the content of imagepool\n");
		cp1 = imagepool[i].image;
		if (!cp1)
			continue;
		fprintf(fp, "xpos %d add ypos moveto\n",
			imagepool[i].xoffset);
		image_load_print(cp1->ctm_fname, cp1->ctm_numcolor,
			cp1->ctm_ximagesize, cp1->ctm_yimagesize, 0,
			cp1->ctm_zoomflag);
		fprintf(fp, "%d setcharsize\n", imagepool[i].ysiz);	/*XXX*/
	}
	nimagepool = 0;
    }

done:
	fprintf(fp, "NL\n");
	tabprefix = NULL;
}

static void
icon_output(tp)
	struct textpool *tp;
{
	int isize = tp->size;
	int csize = tp->xsiz;
	int ixoff, iyoff;
	int paintit;

	iyoff = (csize - isize) / 3;	/*XXX*/
	ixoff = tp->xoffset + iyoff;

	paintit = (painticon || colorps);

	switch ((int)tp->font) {	/*XXX*/
	case 0:
		/* XXX: image is not supported yet */
		break;
	case 1:
		fprintf(fp, "currentpoint ");
		if (paintit)
			fprintf(fp, "currentpoint newpath moveto ");
		fprintf(fp, "%d %d rmoveto ", ixoff, iyoff);
		fprintf(fp, "0 %d rlineto ", isize);
		fprintf(fp, "%d 0 rlineto ", isize);
		fprintf(fp, "0 %d rlineto ", -isize);
		fprintf(fp, "%d 0 rlineto ", -isize);
		if (paintit)
			fprintf(fp, "closepath eofill ");
		fprintf(fp, "stroke moveto\n");
		fprintf(fp, "%d 0 rmoveto\n", ixoff * 2 + isize);
		break;
	case 2:
		fprintf(fp, "currentpoint ");
		if (paintit)
			fprintf(fp, "currentpoint newpath moveto ");
		fprintf(fp, "%d %d rmoveto ", ixoff + isize, iyoff + isize/2);
		fprintf(fp, "currentpoint exch %d add exch %d 0 360 arc ",
			-isize/2, isize/2);
		if (paintit)
			fprintf(fp, "closepath eofill ");
		fprintf(fp, "stroke moveto\n");
		fprintf(fp, "%d 0 rmoveto\n", ixoff * 2 + isize);
		break;
	case 3:
		fprintf(fp, "currentpoint ");
		if (paintit)
			fprintf(fp, "currentpoint newpath moveto ");
		fprintf(fp, "%d %d rmoveto ", ixoff, iyoff);
		fprintf(fp, "%d 0 rlineto ", isize);
		fprintf(fp, "%d %d rlineto ", -isize/2, isize);
		fprintf(fp, "%d %d rlineto ", -isize/2, -isize);
		if (paintit)
			fprintf(fp, "closepath eofill ");
		fprintf(fp, "stroke moveto\n");
		fprintf(fp, "%d 0 rmoveto\n", ixoff * 2 + isize);
		break;
	case 4:
		fprintf(fp, "currentpoint ");
		if (paintit)
			fprintf(fp, "currentpoint newpath moveto ");
		fprintf(fp, "%d %d rmoveto ", ixoff, iyoff + isize);
		fprintf(fp, "%d 0 rlineto ", isize);
		fprintf(fp, "%d %d rlineto ", -isize/2, -isize);
		fprintf(fp, "%d %d rlineto ", -isize/2, isize);
		if (paintit)
			fprintf(fp, "closepath eofill ");
		fprintf(fp, "stroke moveto\n");
		fprintf(fp, "%d 0 rmoveto\n", ixoff * 2 + isize);
		break;
	case 5:
		fprintf(fp, "currentpoint ");
		if (paintit)
			fprintf(fp, "currentpoint newpath moveto ");
		fprintf(fp, "%d %d rmoveto ", ixoff, iyoff);
		fprintf(fp, "0 %d rlineto ", isize);
		fprintf(fp, "%d %d rlineto ", isize, -isize/2);
		fprintf(fp, "%d %d rlineto ", -isize, -isize/2);
		if (paintit)
			fprintf(fp, "closepath eofill ");
		fprintf(fp, "stroke moveto\n");
		fprintf(fp, "%d 0 rmoveto\n", ixoff * 2 + isize);
		break;
	case 6:
		fprintf(fp, "currentpoint ");
		if (paintit)
			fprintf(fp, "currentpoint newpath moveto ");
		fprintf(fp, "%d %d rmoveto ", ixoff + isize, iyoff);
		fprintf(fp, "0 %d rlineto ", isize);
		fprintf(fp, "%d %d rlineto ", -isize, -isize/2);
		fprintf(fp, "%d %d rlineto ", isize, -isize/2);
		if (paintit)
			fprintf(fp, "closepath eofill ");
		fprintf(fp, "stroke moveto\n");
		fprintf(fp, "%d 0 rmoveto\n", ixoff * 2 + isize);
		break;
	case 7:
		fprintf(fp, "currentpoint ");
		if (paintit)
			fprintf(fp, "currentpoint newpath moveto ");
		fprintf(fp, "%d %d rmoveto ", ixoff, iyoff + isize/2);
		fprintf(fp, "%d %d rlineto ",  isize/2,  isize/2);
		fprintf(fp, "%d %d rlineto ",  isize/2, -isize/2);
		fprintf(fp, "%d %d rlineto ", -isize/2, -isize/2);
		fprintf(fp, "%d %d rlineto ", -isize/2,  isize/2);
		if (paintit)
			fprintf(fp, "closepath eofill ");
		fprintf(fp, "stroke moveto\n");
		fprintf(fp, "%d 0 rmoveto\n", ixoff * 2 + isize);
		break;
	}
}

static void
line_skip(x, y)
	int x;
	int y;
{
	linewidth += x;
	lineheight = (lineheight < y) ? y : lineheight;
}

static void
print_bar(cp)
	struct ctrl *cp;
{
	struct ctrl_bar pbar;

	pbar.ct_width = cp->ctb_width * w_height / 1000;
	pbar.ct_start = cp->ctb_start * w_width / 100;
	pbar.ct_length = cp->ctb_length * w_width / 100;

	fprintf(fp, "%%bar color %d %d %d\n",
		cp->ctb_width, cp->ctb_start, cp->ctb_length);
	fprintf(fp, "XMARGIN ypos moveto\n");
	fprintf(fp, "%d 0 rmoveto\n", pbar.ct_start);
	fprintf(fp, "%d %d rlineto\n", 0, pbar.ct_width * -1);
	fprintf(fp, "%d %d rlineto\n", pbar.ct_length, 0);
	fprintf(fp, "%d %d rlineto\n", 0, pbar.ct_width);
	fprintf(fp, "%d %d rlineto stroke\n", pbar.ct_length * -1, 0);
	fprintf(fp, "/ypos ypos %d sub def\n",
		cp->ctb_width + VERT_GAP(char_size) / 2);
	fprintf(fp, "xpos ypos moveto\n");

	linewidth = 0;
}

static void
print_icon(cp)
	struct ctrl *cp;
{
	int i;
	int itype, isize;
	static struct ctl_words icon_words[] = {
		{ 1, 'x', "box", 3 },
		{ 2, 'x', "arc", 3 },
		{ 3, 'x', "delta1", 6 },
		{ 4, 'x', "delta2", 6 },
		{ 5, 'x', "delta3", 6 },
		{ 6, 'x', "delta4", 6 },
		{ 7, 'x', "dia", 3 },
		{ 0, 'x', NULL, 0 }
	};

	for (i = 0; icon_words[i].ctl_strlen != 0; i++) {
		if (!strncasecmp(cp->ctic_value, icon_words[i].ctl_string,
			strlen(cp->ctic_value)))
				break;
	}
	itype = icon_words[i].ctl_type;
	isize = char_size * cp->ctic_size / 100;
	icon_remember(itype, isize, linewidth, cp->ctic_color);
	linewidth += char_size;
}

static void
print_line(data, pfx)
	u_char *data;
	int pfx;
{
	if (data && *data)
		print_string(data, pfx);
}

static int
iskinsokuchar(code)
	u_short code;
{
	u_short *kinsoku;

	for (kinsoku = kinsokutable; *kinsoku; kinsoku++) {
		if (code == *kinsoku)
			return 1;
	}
	return 0;
}

static void
print_string(data, pfx)
	u_char *data;
	int pfx;
{
	u_char *p;
	u_char *q;
	int kanji = 0;
	u_int code2;

	p = data;

	while (*p && *p != '\n') {
		if (p[0] == 0x1b && p[1] == '$'
		 && (p[2] == 'B' || p[2] == '@')) {
			kanji = 1;
			p += 3;
			continue;
		}
		if (p[0] == 0x1b && p[1] == '('
		 && (p[2] == 'B' || p[2] == 'J')) {
			kanji = 0;
			p += 3;
			continue;
		}

#if 0
		if (!kanji && isspace(p[0])) {
			p++;
			linewidth +=
			continue;
		}
#endif

		if (kanji) {
			for (q = p + 2; 0x21 <= q[0] && q[0] <= 0x7e; q += 2) {
				code2 = q[0] * 256 + q[1];
				if (!iskinsokuchar(code2))
					break;
			}
		} else {
			q = p;
			while (*q && isprint(*q) && !isspace(*q))
				q++;
			if (q == p)
				q++;
			else {
				/* append spaces to the end of the word. */
				while (*q && isspace(*q))
					q++;
			}
		}

		print_fragment(p, q - p, kanji, pfx);

		p = q;
	}
}

static void
print_fragment(data, len, kanjimode, pfx)
	u_char *data;
	u_int len;
	int kanjimode;
	int pfx;
{
	u_char *p;
	u_char *q;
	char buf[4096];
	u_int code;
	int textstartpos = 0;
#define OPENBRACE(kanji)	((kanji) ? '<' : '(')
#define CLOSEBRACE(kanji)	((kanji) ? '>' : ')')
#define	DANGERLETTER(x)	\
	((x) == '(' || (x) == ')' || (x) == '\\')

	p = data;
	if (kanjimode) {
		q = &buf[1];
		while (len) {
			code = p[0] * 256 + p[1];
			sprintf(q, "%04x ", code);
			p += 2;
			len -= 2;
			q += 5;
		}
		buf[0] = OPENBRACE(kanjimode);
		*q++ = CLOSEBRACE(kanjimode);
		*q = '\0';
		text_remember(buf, KANJI, char_size, textstartpos, pfx);
	} else {
		/* must take care of escaping those "dangerous" letters */
		q = &buf[0];
		*q++ = OPENBRACE(kanjimode);
		while (len) {
			if (DANGERLETTER(p[0]))
				*q++ = '\\';
			*q++ = *p++;
			len--;
		}
		*q++ = CLOSEBRACE(kanjimode);
		*q++ = '\0';
		text_remember(buf, ASCII, char_size, textstartpos, pfx);
	}
}

static char *
checkeps(fname)
	char *fname;
{
	static char epsfile[MAXPATHLEN];
	static char fullname[MAXPATHLEN];
	char *p;

	/* rewrite file suffix, if it is not "ps" nor "eps". */
	strcpy(epsfile, fname);
	p = strrchr(epsfile, '.');
	if (p) {
		if (strcmp(p, ".ps") != 0 && strcmp(p, ".eps") != 0) {
			/* try "basename.eps" */
			strcpy(p, ".eps");
		}
	}

	fullname[0] = '\0';
	if (findImage(epsfile, fullname) == 0)
		return fullname;
	else
		return NULL;
}

static void
text_remember(text, ctype, fontsize, offset, pfx)
	char *text;
	int ctype;
	int fontsize;
	int offset;
	int pfx;
{
	/*XXX*/
	if (strcmp(text, "()") == 0)
		return;
	if (!curfont[ctype]) {
		fprintf(stderr, "cannot find proper font, skipping\n");
		return;
	}
	textpool[ntextpool].pfx = pfx;
	textpool[ntextpool].xoffset = offset;
	textpool[ntextpool].xsiz = linewidth - offset;
	textpool[ntextpool].size = fontsize;
	textpool[ntextpool].font = curfont[ctype];
	textpool[ntextpool].text = strdup(text);
	textpool[ntextpool].fore = fore;
	textpool[ntextpool].back = back;
	ntextpool++;
}

static void
icon_remember(icon, fontsize, offset, color)
	int icon;
	int fontsize;
	int offset;
	u_long color;
{
	textpool[ntextpool].pfx = 1;
	textpool[ntextpool].xoffset = offset;
	textpool[ntextpool].xsiz = char_size;
	textpool[ntextpool].size = fontsize;
	textpool[ntextpool].font = (struct fontmap *)icon;	/*XXX*/
	textpool[ntextpool].text = NULL;
	textpool[ntextpool].fore = color;
	textpool[ntextpool].back = back;	/*XXX*/
	ntextpool++;
}

static void
image_remember(cp, pool)
	struct ctrl *cp;
	struct imagepool *pool;
{
	char *epsname;
	char buf[BUFSIZ];
	Image *myimage;

	if (epsname = checkeps(cp->ctm_fname)) {
		int x1, y1, x2, y2, width, height, swidth, sheight;

		if (ps_boundingbox(epsname, &x1, &y1, &x2, &y2) < 0)
			goto noneps;

		width = x2 - x1;
		height = y2 - y1;
		swidth = (int)(width * PSSCALE);
		sheight = (int)(height * PSSCALE);

		pool->xsiz = width;
		pool->ysiz = height;
		pool->xoffset = linewidth;
		pool->image = cp;
		return;
	}

noneps:
	myimage = loadImage(cp->ctm_fname, mgp_flag & FL_VERBOSE);
	if (!myimage) {
		fprintf(stderr, "failed to open %s\n", cp->ctm_fname);
		exit(1);
	}
	pool->xsiz = myimage->width;
	pool->ysiz = myimage->height;
	freeImage(myimage);
	pool->xoffset = linewidth;
	pool->image = cp;
}

static void
image_load_print(filename, numcolor, ximagesize, yimagesize, backflag, zoomflag)
	char *filename;
	int numcolor;
	int ximagesize;
	int yimagesize;
	int backflag;
	int zoomflag;
{
	Image *image, *myimage;
	Pixmap mypixmap;
	XImageInfo *ximageinfo;
	XImage	*print_image;
	int width, height;
	int xzoomrate, yzoomrate;
	static Cursor curs;
	u_int	print_width, print_height;

	int xpos;

	if (zoomflag == 2)
		ximagesize = yimagesize = 0;

	/* hook for eps */
    {
	char *p;
	FILE *epsfp;

	p = checkeps(filename);
	if (p) {
		epsfp = fopen(p, "r");
		print_eps(epsfp, p, numcolor, ximagesize, yimagesize,
			backflag, zoomflag);
		fclose(epsfp);
		return;
	}
    }

	if (!curs)
		curs = XCreateFontCursor(display, XC_watch);
	XDefineCursor(display, window, curs); XFlush(display);

	screen = DefaultScreen(display);

	if ((myimage = loadImage(filename, verbose)) == NULL)
		exit(-1);	/* fail to load image data */
	width = myimage->width;
	height = myimage->height;

	if (numcolor)
		myimage = reduce(myimage, numcolor, verbose);

	if (ximagesize != 0 && yimagesize != 0) {
		if (!zoomflag) {
			xzoomrate = window_width * ximagesize / width;
			yzoomrate = window_height * yimagesize / height;
		} else {
			xzoomrate = ximagesize;
			yzoomrate = yimagesize;
		}
		image = myimage;
		myimage = zoom(image, xzoomrate, yzoomrate, verbose);
		freeImage(image);
	} else
		xzoomrate = yzoomrate = 100;

	if (! (ximageinfo= imageToXImage(display, screen, visual, depth,
			myimage, 0, 0, 0, verbose))) {
		fprintf(stderr, "Cannot convert Image to XImage\n");
		exit(1);
	}

	mypixmap = ximageToPixmap(display, RootWindow(display, 0), ximageinfo);
	if (!mypixmap) {
		fprintf(stderr, "Cannot create image in server\n");
		exit(1);
	}

	switch (align) {
	case AL_LEFT:
	case AL_LEFTFILL0:
	case AL_LEFTFILL1:
		xpos = 0;
		break;
	case AL_CENTER:
		xpos = (window_width - width * xzoomrate /100) / 2;
		break;
	case AL_RIGHT:
		xpos = window_width - width * xzoomrate /100;
		break;
	}

	print_width = myimage->width;
	print_height = myimage->height;

	print_image =  XGetImage(display, mypixmap, 0, 0,
					print_width, print_height,
					AllPlanes, ZPixmap);

	fprintf(fp, "gsave\n");
	fprintf(fp, "%d ypos %d sub translate\n",
		xpos + paper_xmargin, print_height);
	fprintf(fp, "%d %d scale\n", print_width, print_height);
	fprintf(fp, "%d %d 8\n", print_width, print_height);
	fprintf(fp, "[%d 0 0 %d 0 %d]\n", print_width,
			-1 * print_height, print_height);
	fprintf(fp, "{currentfile\n");
	fprintf(fp, "%d string readhexstring pop}\n", print_width * 3);
	fprintf(fp, "false 3\n");
	fprintf(fp, "colorimage\n");

	/* XXX is there any generic way of doing this? */
	switch (print_image->bits_per_pixel) {
	case 8:
		print_8_image(print_image);
		break;
	case 16:
		print_full_image(print_image, visual);
		break;
	case 32:
		print_32_image(print_image, visual);
		break;
	default:
		fprintf(stderr, "Sorry unsupported visual %d\n",
			print_image->bits_per_pixel);
		exit(-1);
	}

	fprintf(fp, "grestore\n");

	XFreePixmap(display, mypixmap);
	freeXImage(myimage, ximageinfo);
	freeImage(myimage);
	XUndefineCursor(display, window); XFlush(display);
}

static void
print_usage(name)
	char	*name;
{
	fprintf(stderr, "Usage: %s [-cirV] [-f psfile] [-p paper] "
		"[-x xmargin] [-y ymargin] mgpfile\n", name);
	exit(0);
}

static int
set_position()
{
	u_int   x;

	switch(align) {
	case AL_CENTER:
		x = (window_width - linewidth)/ 2;
		break;

	case AL_LEFT:
	case AL_LEFTFILL0:
	case AL_LEFTFILL1:
		x = 0;
		break;

	case AL_RIGHT:
		x = window_width - linewidth;
		break;

	default:
		x = 0;
		break;
	}

	return x;
}

static void
setpencolor(c)
	u_long c;
{
	/*
	 * update the color of the pen.
	 * XXX background is ignored at this moment
	 */
	XColor color;

	color.pixel = c;
	XQueryColor(display, colormap, &color);
#if 0
	if (color.red == color.green && color.green == color.blue) {
		/* XXX: invert white-to-gray colors */
		color.red = 65535 - color.red;
		color.green = 65535 - color.green;
		color.blue = 65535 - color.blue;
	}
#endif
	fprintf(fp, "%f %f %f setrgbcolor %% #%08x depth %d\n",
		color.red / 65535.0,
		color.green / 65535.0,
		color.blue / 65535.0,
		color.pixel, depth);
}

static void
print_full_image(print_image, visual)
	XImage	*print_image;
	Visual	*visual;
{
	u_int	x, y;
	u_int	r, g, b;
	u_int	count = 0;
	u_int	width, height, byte_width;
	u_short	*data;

	data = (u_short *)print_image->data;
	width = print_image->width;
	height = print_image->height;
	byte_width = print_image->bytes_per_line / sizeof(u_short);

	for (y = 0; y < height; y ++) {
		for (x = 0; x < width; x ++) {
			r = ((data[x + y * byte_width] & visual->red_mask) >> 11) & 0xff;
			g = ((data[x + y * byte_width] & visual->green_mask) >> 5) & 0xff;
			b = (data[x + y * byte_width] & visual->blue_mask) & 0xff;

			if (reverse)
				fprintf(fp, "%02x%02x%02x",
					~(r << 3) & 0xff, ~(g << 2) & 0xff, ~(b << 3) & 0xff);
			else
				fprintf(fp, "%02x%02x%02x", r << 3, g << 2, b << 3);

			count ++;
			if (count == 10) {
				count = 0;
				fprintf(fp, "\n");
			}
		}
	}
}

static void
print_32_image(print_image, visual)
	XImage	*print_image;
	Visual	*visual;
{
	u_int	x, y;
	u_int	r, g, b;
	u_int	count = 0;
	u_int	width, height;
	u_short	*data;

	data = (u_short *)print_image->data;
	width = print_image->width;
	height = print_image->height;

	for (y = 0; y < height; y ++) {
		for (x = 0; x < width; x ++) {
			unsigned long pix;
			pix = XGetPixel(print_image, x, y);
			r =  pix & visual->red_mask & 0xff;
			g = (pix & visual->green_mask >> 8)  & 0xff;
			b = (pix & visual->blue_mask  >> 16) & 0xff;

			if (reverse)
				fprintf(fp, "%02x%02x%02x",
					~r & 0xff, ~g & 0xff, ~b & 0xff);
			else
				fprintf(fp, "%02x%02x%02x", r, g, b);

			count ++;
			if (count == 10) {
				count = 0;
				fprintf(fp, "\n");
			}
		}
	}
}

static void
print_8_image(print_image)
	XImage	*print_image;
{
	u_int	x, y;
	u_int	count = 0;
	u_char	*data;
	u_int	width, height, byte_width;

	data = (u_char *)print_image->data;
	width = print_image->width;
	height = print_image->height;
	byte_width = print_image->bytes_per_line / sizeof(u_char);

	for (y = 0; y < height; y ++) {
		for (x = 0; x < width; x ++) {
			if (reverse)
				fprintf(fp, "%02x%02x%02x",
					~data[x + y * byte_width] & 0xff,
					~data[x + y * byte_width] & 0xff,
					~data[x + y * byte_width] & 0xff);
			else
				fprintf(fp, "%02x%02x%02x",
					data[x + y * byte_width],
					data[x + y * byte_width],
					data[x + y * byte_width]);

			count ++;
			if (count == 10) {
				count = 0;
				fprintf(fp, "\n");
			}
		}
	}
}

static void
print_eps(epsfp, filename, numcolor, ximagesize, yimagesize, backflag, zoomflag)
	FILE *epsfp;
	char *filename;
	int numcolor;
	int ximagesize;
	int yimagesize;
	int backflag;
	int zoomflag;
{
	char line1[BUFSIZ];
	char line2[BUFSIZ];
	char line[BUFSIZ];
	int x1, y1, x2, y2, height, width, sheight, swidth;
	int xpos;

	/*
	 * mgp scale is NOT supported.
	 *
	 * PS scale code is for visibility when printed.
	 * Kazu's epsfile should be enlarged 1.2 times.
	 * And this is a challenge to understand PS's scale...
	 */
	if (fgets(line1, sizeof(line1), epsfp) == NULL) {
		fprintf(stderr, "no first line in %s.\n", filename);
		exit(1);
	}
	if (strncmp(line1, "%!", 2) != 0) {
		fprintf(stderr, "non eps file %s used as eps.\n", filename);
		exit(1);
	}

	for (;;) {
		if (fgets(line2, sizeof(line2), epsfp) == NULL || line2[0] != '%') {
			fprintf(stderr, "no bounding box in %s.\n", filename);
			exit(1);
		}
		if (sscanf(line2, "%%%%BoundingBox: %d %d %d %d", &x1, &y1, &x2, &y2) == 4)
			break;
	}

	width = x2 - x1;
	height = y2 - y1;
	swidth = (int)(width * PSSCALE);
	sheight = (int)(height * PSSCALE);

	switch (align) {
	case AL_LEFT:
	case AL_LEFTFILL0:
	case AL_LEFTFILL1:
		xpos = 0;
		break;
	case AL_CENTER:
		xpos = (window_width - swidth) / 2;
		break;
	case AL_RIGHT:
		xpos = window_width - swidth;
		break;
	}

	fprintf(fp, "BeginEPSF\n");
	fprintf(fp, "%f %f scale\n", PSSCALEF, PSSCALEF);
	fprintf(fp, "%d %d translate\n",
		(x1 - (int)(paper_xmargin * PSBASE)) * -1,
		(y2 + (int)(paper_ymargin * PSBASE)) * -1);
	fprintf(fp, "%d %f mul ypos XMARGIN add %f mul translate\n",
		xpos, (double)PSBASE, (double)PSBASE);
	fprintf(fp, "%%%%BeginDocument: %s\n", filename);
	fputs(line1, fp);
	fputs(line2, fp);
	while (fgets(line, sizeof(line), epsfp)) {
		if (strncmp(line, "%%PageBoundingBox:", 18) == 0
		||  strncmp(line, "%%Trailer", 9) == 0)
			line[1] = '#';
		fputs(line, fp);
	}
	fprintf(fp, "%%%%EndDocument\n");
	fprintf(fp, "EndEPSF\n");
}

/*------------------------------------------------------------*/

int
main(argc, argv)
	int argc;
	char **argv;
{
	u_int opt;
	extern char *optarg;
	extern int optind;
	u_char *progname;
	static char pathbuf[MAXPATHLEN];

	progname = argv[0];

	/* set default paper size */
	paper = findpaper(DEFAULT_PAPER_SIZE);

	while ((opt = getopt(argc, argv, "ciVrf:x:y:p:")) != -1) {
		switch (opt) {
		case 'c':
			colorps++;
			break;

		case 'i':
			painticon++;
			break;

		case 'f':
			strcpy(outputfile, optarg);
			break;

		case 'r':
			reverse = 1;
			break;

		case 'V':
			mgp_flag |= FL_VERBOSE;
			break;

		case 'x':
			paper_xmargin = atoi(optarg);
			break;

		case 'y':
			paper_ymargin = atoi(optarg);
			break;

		case 'p':
			paper = findpaper(optarg);
			if (!paper) {
				fprintf(stderr,
"Paper size '%s' not recognised. Using %s instead.\n",
					optarg, DEFAULT_PAPER_SIZE);
				paper = findpaper(DEFAULT_PAPER_SIZE);
			}
			break;

		default:
			print_usage(progname);
			/*NOTREACHED*/
		}
	}

	w_width  = paper->width - paper_xmargin * 2;
	w_height = paper->height - paper_ymargin * 2;

	argc -= optind;
	argv += optind;

	if (argc != 1) {
		print_usage(progname);
		/*NOTREACHED*/
	}

	mgp_fname = argv[0];

	/* setting up the search path. */
    {
	char *p;
	loadPathsAndExts();
	strncpy(pathbuf, mgp_fname, sizeof(pathbuf));
	if (p = rindex(pathbuf, '/')) {
		*p = '\0';
        	Paths[NumPaths++]= expandPath(pathbuf);
	}
    }

	init_win1(NULL);
	load_file(mgp_fname);
	init_win2();
	fore = 0UL;
	back = (1UL << depth) - 1;
	print_out();

	exit(0);
}

static struct fontmap *
findfont(ctrl, lang, font)
	int ctrl;
	int lang;	/*simply ignored*/
	char *font;
{
	struct fontmap *p, *q;
	char *star;

	if (!font[0])
		return NULL;
	for (p = fontmap; 0 <= p->ctrl; p++) {
		if (p->ctrl != ctrl)
			continue;
		star = strchr(p->font, '*');
		if (!star && strcmp(p->font, font) == 0)
			goto found;
		if (star && strncmp(p->font, font, star - p->font) == 0)
			goto found;
	}
	return NULL;

found:
	if (p->fontid)
		return p;

	p->fontid = ++maxfontid;

	if (mgp_flag & FL_VERBOSE) {
		char fonttype;

		switch (ctrl) {
#ifdef FREETYPE
		case CTL_TFONT: fonttype = 't'; break;
#endif
#ifdef VFLIB
		case CTL_VFONT: fonttype = 'v'; break;
#endif
		case CTL_XFONT: fonttype = 'x'; break;
		default:	fonttype = '?'; break;
		}
		fprintf(fp, "%% %cfont \"%s\" seen, mapped to ps font \"%s\"\n",
			fonttype, font, p->psfont);
		if (strcmp(font, p->font) != 0) {
			fprintf(fp, "%%\t(wildcard match against \"%s\")\n",
				p->font);
		}
	}
	/* use the same font index for same PS font names */
	for (q = fontmap; 0 <= q->ctrl; q++) {
		if (strcmp(p->psfont, q->psfont) == 0)
			q->fontid = maxfontid;
	}

	return p;
}

static char *
fontstring(font)
	struct fontmap *font;
{
	static char fontname[10];

	sprintf(fontname, "F%03d", font->fontid);
	return fontname;
}

static void
loadfont(font)
	struct fontmap *font;
{
	if (font->loaded)
		return;

	/* define font calling sequence */
	if (mgp_flag & FL_VERBOSE) {
		fprintf(fp, "%% loading font \"%s\" for %s\n",
			font->psfont, font->font);
	}
	fprintf(fp, "/%s {/%s findfont exch scalefont setfont} def\n",
		fontstring(font), font->psfont);
	font->loaded = 1;
}

void
cleanup(int sig)
{
	/* dummy */
}
