/*

  gtkcookie.c --> gtk+ web browser cookie file editor

  Copyright (c) 1998 Manni Wood

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

  Manni Wood: mwood@sig.bsh.com, pq1036@110.net

  Netscape, Netscape Navigator, Netscape ONE, and the Netscape N and
  Ship's Wheel logos are registered trademarks of Netscape Communications
  Corporation in the United States and other countries.
  
 */

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <errno.h>
#include <dirent.h>
#include <sys/time.h>  /* linux-specific, gives me strptime() */
/* please note that because strptime() is not part of ANSI standard
   C, you may need to define the macro _XOPEN_SOURCE = 1 (or use
   -D_XOPEN_SOURCE when compiling). This macro tells gcc that
   you want to use features of Linux beyond the strict ANSI
   subset.

   The IRIX time.h has
   #if _XOPEN4 && _NO_ANSIMODE
   around its strptime, so perhaps the makefile on the platform
   needs -D_XOPEN4 -D_NO_ANSIMODE, but I haven't tried that yet

   On Solaris, we have this rather daunting wrapper:
   
   #if (__STDC__ == 0 && !defined(_POSIX_C_SOURCE) && !defined(_XOPEN_SOURCE)) || \
    (defined(_XOPEN_SOURCE) && _XOPEN_VERSION - 0 == 4) || \
    defined(__EXTENSIONS__)

*/
#include <gtk/gtk.h>
#include "cookie.xpm"

#define COOKIE_COLUMNS 8

/*
  "The static declaration, applied to an external variable or function,
  limits the scope of that object to the rest of the source file being
  compiled."  -- K & R, p. 83
*/

/* COOKIE FUNCTIONS
   A cookie is simply an array of pointers to gchar; each pointer to
   gchar holds the string value of one of the cookie fields:
    "Domain", - domain of the server that set the cookie
    "Suffix", - T/F: is the domain a suffix, or a full domain?
    "Path", - what path does the browser have to be in to return the cookie?
    "Secure", - T/F: is this cookie only sent over secure http?
    "Expire", - time in seconds, since the epoch, this cookie expires
    "Name", - name (or key) of the cookie
    "Value", - value of the cookie
    "Expire". - human-readable expiration date
    Please note that the final "Expire" is not kept in the cookie
    file, but is a human-readable version of the date held in the
    first "Expire".
*/

/* is a line of text from the cookie file a valid cookie? */
static gboolean is_a_cookie_line(gchar *line);

/* split a line of text into the seven cookie fields, and place
   in new_cookie */
static void split_cookie_line(gchar *line, gchar *new_cookie[COOKIE_COLUMNS]);

/* set a cookie's gchar pointers to NULL */
static void initialise_cookie(gchar *new_cookie[COOKIE_COLUMNS]);

/* free the memory pointed to by all the gchar pointers in a cookie,
   and set each gchar pointer to NULL */
static void free_cookie(gchar *c[COOKIE_COLUMNS]);

/* copy the first cookie's strings to the gchar ptrs of the second
   cookie, allocating the needed memory to do so */
static void copy_cookie(gchar *c1[COOKIE_COLUMNS], gchar *c2[COOKIE_COLUMNS]);

/* copy the cookie from a particular row in the clist to c,
   allocating the needed memory to do so */
static void copy_clist_row_to_cookie(gint row, gchar *c[COOKIE_COLUMNS]);

/* handy debugging function to print the contents of a cookie */
#ifdef _MANNI_DEBUG
static void dbg_print_cookie(gchar *c[COOKIE_COLUMNS]) {
    printf("domain: %s\n", c[0]);
    printf("suffix: %s\n", c[1]);
    printf("path: %s\n", c[2]);
    printf("secure: %s\n", c[3]);
    printf("expire: %s\n", c[4]);
    printf("name: %s\n", c[5]);
    printf("value: \"%s\"\n", c[6]);
    printf("expire: %s\n", c[7]);
}
#endif

/* CLIST FUNCTIONS
   Functions used to manipulate the list of cookies held in the
   clist in the main window.
*/

/* parse glbl_cookie_FILE and populate the main clist widget with the
   results */
static void populate_clist(gpointer data);

/* clear all items from main clist */
static void clear_clist (GtkWidget *widget, gpointer data);

/* remove a row from main clist */
static void remove_row_clist (GtkWidget *widget, gpointer data);

/* respond to a double-click in the clist widget (by calling
 create_edit_dialog) */
static gboolean select_clist_dblclk(GtkWidget *widget,
			  GdkEventButton *event,
			  gpointer func_data);

/* copy the cookie from the selected row to the global variable which
   holds the currendly-selected cookie, glbl_selected_cookie */
static void select_clist (GtkWidget *widget,
		   gint row, 
		   gint column, 
		   GdkEventButton * bevent);

/* perform a sort on the column if its title button has been clicked */
static void sort_by_column_clist (GtkWidget *widget,
			 gint column,
			 GdkEventButton * bevent);

/* inserts a new cookie in clist, and activates the edit dialog
 box for the new cookie.*/
static void insert_row_clist (GtkWidget *widget, gpointer data);

/* creates the actual clist widget, with its accompanying buttons */
static void create_clist (GtkWidget *box1);

/* copies the contents of glbl_selected_cookie into
   the correct row of the master clist (called by clicking
   "OK" on the cookie edit dialogue box) */
static void update_clist(GtkWidget *widget, gpointer window);

/* COOKIE EDIT DIALOGUE BOX FUNCTIONS */

/* pops up the cookie edit dialogue box */
static void create_edit_dialog (GtkWidget *widget);

/* whenever an editing change is made to any of the fields
 of a cookie, glbl_selected_cookie has memory re-allocated,
and strings re-copied, to reflect the changes made in
the dialogue box */
static void change_callback(GtkWidget *widget, gpointer data);

/* detects text changes in the human-readable date field of the
   cookie edit dialogue box, and changes the original expiry date
   field on the fly */
static void change_time_callback(GtkWidget *widget, gpointer data);

/* called when one of the true/false checkboxes in the cookie edit
   dialogue box changes state */
static void change_tf_callback(GtkWidget *widget, gpointer data);

/* FIND DIALOG BOX FUNCTIONS */

/* show the 'find' dialogue box */
static void handle_find_dialog_box(void);

static void do_find(void);

static void do_clear(void);

/* make glbl_last_found_row equal -1 again */
static void reset_last_found_row(void);

/* OTHER DIALOG BOX FUNCTIONS */

/* show the 'about' dialogue box */
static void handle_about_dialog_box(void);

/* show an error dialogue box, with the following message */
static void show_error_dialog (gchar *message);

/* FILE SELECTION AND SAVE FUNCTIONS */

/* respond to the "OK" button in the open file dialogue */
static void file_selection_ok (GtkWidget *w, GtkFileSelection *fs);

/* respond to the "OK" button in the save_as dialogue box */
static void save_as_ok (GtkWidget *w, GtkFileSelection *fs);

/* creates the file selection dialogue box that you see when you use
   "open" */
static void create_open_file_dialogue (void);

/* creates the "save as" dialoge box */
static void create_save_as_dialogue (void);

/* handles the save_file request from the file manu; is smart enough
   to either call write_file() if the file exists, or call
   create_save_as_dialogue if glbl_selected_filename_str is empty */
static void save_file(void);

/* actually writes all of the cookie data held in the main clist
   out into a properly formatted mozilla file whose name
   should be held in glbl_selected_filename_str */
static void write_file(void);

/* OTHER FUNCTIONS */

/* check to see if there is a netscape lockfile for the current user,
   and if so, issue a warning */
static void check_for_lockfile(void);

/* change the cursor -- seems not to be working */
/* static void set_cursor (guint c); */

/* ALL GLOBAL VARIABLES START WITH glbl_ */

/* the main programme window */
static GtkWidget *glbl_main_window;

static gchar glbl_window_title_str[1024] = "gtkcookie";

/* the master combo list */
static GtkWidget *glbl_clist;

/* we need a global pointer to point to the expire time label in
   an edit window whenever the edit window is constructed. This will
   give us a handle to the label even after the function is run. We need
   the handle to the label so that we can change the value of the label
   as handler functions are called */
static GtkWidget *glbl_selected_etl;

/* we need a time struct to throw the results of strptime() into */
static struct tm glbl_selected_time;

/* we also need a time_t global var to convert glbl_selected_time from */
static time_t glbl_selected_time_sec;

/* finally, we need an ascii string representation of the glbl_selected_time_sec.
   January 2038 is when a 32 bit int will roll over, and that 32 bit int
   seems to need 10 ascii characters to print. I'll therefore allocate 12
   characters to be safe. This won't, however, be safe if we move to 64 bit
   ints */
static gchar glbl_selected_time_str[12];

static gint glbl_clist_rows = 0;
static gint glbl_clist_selected_row = 0;
static gchar *glbl_selected_cookie[COOKIE_COLUMNS]; /* the currently selected cookie in the glbl_clist */


/* glbl_selected_filename_str will be a string holding the name of the
   file from
   a file select box, or the default cookie file location */
static gchar *glbl_selected_filename_str = NULL;
static gchar *glbl_home_directory_str = NULL; /* user's home directory */
static gchar *glbl_netscape_dot_dir_str = NULL; /* netscape dot directory */
static FILE *glbl_cookie_FILE = NULL;
    
static gchar *glbl_cookie_titles_str[] = {
    "Domain",
    "Suffix",
    "Path",
    "Secure",
    "Expire (sec)",
    "Name",
    "Value",
    "Expire"
};

/* enumerate a label for each column name so that we can use
   a column name instead of a column number. cookie[Domain] tells
   us more than cookie[0] */
enum COLUMN_NAMES {COL_DOMAIN, COL_SUFFIX, COL_PATH, COL_SECURE, COL_EXPIRE_SECONDS, COL_NAME, COL_VALUE, COL_EXPIRE_DATE};

/* sad, but true: a callback function that I attach to each text
   entry field in the cookie popup window requires a pointer to
   an int as an argument, so I have to maintain this array of pointers
   to ints just to keep track of which of the 8 text fields is being
   used. */
static gint *glbl_int_ptr[COOKIE_COLUMNS];

GdkPixmap *glbl_cookie_xpm;
GdkBitmap *glbl_cookie_xpm_mask;

/* the text entry field in the find dialog box */
GtkWidget *glbl_find_text_entry = NULL;

/* keeps track of the row of the last successful find */
static gint glbl_last_found_row = -1;

/* keeps track of the last column that was sorted. If the value
   is -1, no column has been sorted */
static gint glbl_last_sorted_column = -1;

/*
  main
*/

int main (int argc, gchar *argv[]) {

    GtkWidget *file_menu_item_holder;
    GtkWidget *menu_bar;
    GtkWidget *file_menu;
    GtkWidget *menu_items;
    GtkWidget *vbox;

    GtkWidget *help_menu_item_holder;
    GtkWidget *help_menu_items;
    GtkWidget *help_menu;

    GtkWidget *edit_menu_item_holder;
    GtkWidget *edit_menu_items;
    GtkWidget *edit_menu;

    GdkColor *transparent = NULL;

    int i;

    gtk_init (&argc, &argv);

    /* initialise the pointers to ints */
    for (i = 0; i < COOKIE_COLUMNS; i++) {
	glbl_int_ptr[i] = malloc(sizeof(int));
	*glbl_int_ptr[i] = i;
    }

    /* create a new glbl_main_window */
    glbl_main_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_widget_set_usize( GTK_WIDGET (glbl_main_window), 700, 300);
    gtk_window_set_title(GTK_WINDOW (glbl_main_window), glbl_window_title_str);
    gtk_signal_connect(GTK_OBJECT (glbl_main_window), "delete_event",
		       (GtkSignalFunc) gtk_main_quit, NULL);

    /* add the icon to the glbl_main_window */

    gtk_widget_realize(glbl_main_window);
    
    glbl_cookie_xpm = gdk_pixmap_create_from_xpm_d (glbl_main_window->window,
						    &glbl_cookie_xpm_mask, 
						    transparent, 
						    cookiepic);

    gdk_window_set_icon(glbl_main_window->window, NULL,
			glbl_cookie_xpm, glbl_cookie_xpm);

    
    /* FILE MENU */
    
    /* Init the menu-widget, and remember -- never
     * gtk_show_widget() the file_menu_item_holder widget!! 
     * This is the menu that holds the menu items, the one that
     * will pop up when you click on the "File" menu in the app */
    file_menu_item_holder = gtk_menu_new();
    
    /* Next we make the menu-entries for "file_menu_item_holder".
     * Notice the call to gtk_menu_append.  Here we are adding a list of
     * menu items to our menu.  Normally, we'd also catch the "clicked"
     * signal on each of the menu items and setup a callback for it,
     * but it's omitted here to save space. */

    /***************/
    
    /* Create a new menu-item with a name... */
    menu_items = gtk_menu_item_new_with_label("Open");
	
    /* ...and add it to the file_menu_item_holder. */
    gtk_menu_append(GTK_MENU (file_menu_item_holder), menu_items);
	
    /* Do something interesting when the menuitem is selected */
    gtk_signal_connect_object(GTK_OBJECT(menu_items), "activate",
			      GTK_SIGNAL_FUNC(create_open_file_dialogue), NULL);
	
    /* Show the widget */
    gtk_widget_show(menu_items);

    /***************/

    /***************/
    
    /* Create a new menu-item with a name... */
    menu_items = gtk_menu_item_new_with_label("Save");
	
    /* ...and add it to the file_menu_item_holder. */
    gtk_menu_append(GTK_MENU (file_menu_item_holder), menu_items);
	
    /* Do something interesting when the menuitem is selected */
    gtk_signal_connect_object(GTK_OBJECT(menu_items), "activate",
			      GTK_SIGNAL_FUNC(save_file), NULL);
	
    /* Show the widget */
    gtk_widget_show(menu_items);

    /***************/

    /***************/
    
    /* Create a new menu-item with a name... */
    menu_items = gtk_menu_item_new_with_label("Save as...");
	
    /* ...and add it to the file_menu_item_holder. */
    gtk_menu_append(GTK_MENU (file_menu_item_holder), menu_items);
	
    /* Do something interesting when the menuitem is selected */
    gtk_signal_connect_object(GTK_OBJECT(menu_items), "activate",
			      GTK_SIGNAL_FUNC(create_save_as_dialogue), NULL);
	
    /* Show the widget */
    gtk_widget_show(menu_items);

    /***************/

    /***************/
    
    /* Create a new blank menu-item (which creates a line ... */
    menu_items = gtk_menu_item_new();
	
    /* ...and add it to the file_menu_item_holder. */
    gtk_menu_append(GTK_MENU(file_menu_item_holder), menu_items);

    /* no need to attach a function to a separator... */
	
    /* Show the widget */
    gtk_widget_show(menu_items);

    /***************/

    /***************/
    
    /* Create a new menu-item with a name... */
    menu_items = gtk_menu_item_new_with_label("Exit");
	
    /* ...and add it to the file_menu_item_holder. */
    gtk_menu_append(GTK_MENU (file_menu_item_holder), menu_items);
	
    /* quit when the "Exit" menuitem is selected */
    gtk_signal_connect(GTK_OBJECT (menu_items), "activate",
		       (GtkSignalFunc) gtk_main_quit, NULL);
	
    /* Show the widget */
    gtk_widget_show(menu_items);

    /***************/
    
    /* This is the file menu, and will be the label
     * displayed on the menu bar.  There won't be a signal handler attached,
     * as it only pops up the rest of the menu when pressed. */
    file_menu = gtk_menu_item_new_with_label("File");
    
    gtk_widget_show(file_menu);
    
    /* Now we specify that we want our newly created "file_menu_item_holder" to be the menu
     * for the "file menu" */
    gtk_menu_item_set_submenu(GTK_MENU_ITEM (file_menu), file_menu_item_holder);

    /* HELP MENU */
    
    /* Init the menu-widget, and remember -- never
     * gtk_show_widget() the file_menu_item_holder widget!! 
     * This is the menu that holds the menu items, the one that
     * will pop up when you click on the "File" menu in the app */
    help_menu_item_holder = gtk_menu_new();
    
    /* Next we make the menu-entries for "file_menu_item_holder".
     * Notice the call to gtk_menu_append.  Here we are adding a list of
     * menu items to our menu.  Normally, we'd also catch the "clicked"
     * signal on each of the menu items and setup a callback for it,
     * but it's omitted here to save space. */

    /***************/
    
    /* Create a new menu-item with a name... */
    help_menu_items = gtk_menu_item_new_with_label("About...");
	
    /* ...and add it to the file_menu_item_holder. */
    gtk_menu_append(GTK_MENU (help_menu_item_holder), help_menu_items);
	
    /* Do something interesting when the menuitem is selected */
    gtk_signal_connect_object(GTK_OBJECT(help_menu_items), "activate",
			      GTK_SIGNAL_FUNC(handle_about_dialog_box), NULL);
	
    /* Show the widget */
    gtk_widget_show(help_menu_items);

    /***************/

    /* This is the file menu, and will be the label
     * displayed on the menu bar.  There won't be a signal handler attached,
     * as it only pops up the rest of the menu when pressed. */
    help_menu = gtk_menu_item_new_with_label("Help");
    
    gtk_widget_show(help_menu);
    
    /* Now we specify that we want our newly created "file_menu_item_holder" to be the menu
     * for the "file menu" */
    gtk_menu_item_set_submenu(GTK_MENU_ITEM (help_menu), help_menu_item_holder);


    /* /HELP MENU */

    /* EDIT MENU */
    
    /* Init the menu-widget, and remember -- never
     * gtk_show_widget() the file_menu_item_holder widget!! 
     * This is the menu that holds the menu items, the one that
     * will pop up when you click on the "File" menu in the app */
    edit_menu_item_holder = gtk_menu_new();
    
    /* Next we make the menu-entries for "file_menu_item_holder".
     * Notice the call to gtk_menu_append.  Here we are adding a list of
     * menu items to our menu.  Normally, we'd also catch the "clicked"
     * signal on each of the menu items and setup a callback for it,
     * but it's omitted here to save space. */

    /***************/
    
    /* Create a new menu-item with a name... */
    edit_menu_items = gtk_menu_item_new_with_label("Find...");
	
    /* ...and add it to the file_menu_item_holder. */
    gtk_menu_append(GTK_MENU (edit_menu_item_holder), edit_menu_items);
	
    /* Do something interesting when the menuitem is selected */
    gtk_signal_connect_object(GTK_OBJECT(edit_menu_items), "activate",
			      GTK_SIGNAL_FUNC(handle_find_dialog_box), NULL);
	
    /* Show the widget */
    gtk_widget_show(edit_menu_items);

    /***************/

    /* This is the file menu, and will be the label
     * displayed on the menu bar.  There won't be a signal handler attached,
     * as it only pops up the rest of the menu when pressed. */
    edit_menu = gtk_menu_item_new_with_label("Edit");
    
    gtk_widget_show(edit_menu);
    
    /* Now we specify that we want our newly created "file_menu_item_holder" to be the menu
     * for the "file menu" */
    gtk_menu_item_set_submenu(GTK_MENU_ITEM (edit_menu), edit_menu_item_holder);


    /* /EDIT MENU */
    
    /* VBOX */
    
    /* A vbox to put a menu and a button in: */
    vbox = gtk_vbox_new(FALSE, 0);
    gtk_container_add(GTK_CONTAINER(glbl_main_window), vbox);
    gtk_widget_show(vbox);
    
    /* Create a menu-bar to hold the menus and add it to our main glbl_main_window */
    menu_bar = gtk_menu_bar_new();
    gtk_box_pack_start(GTK_BOX(vbox), menu_bar, FALSE, FALSE, 2);
    gtk_widget_show(menu_bar);
    
    /* embed the glbl_clist here */
    create_clist(vbox);

    /* And finally we append the menu-item to the menu-bar -- this is the
     * "File" menu-item I have been raving about =) */
    gtk_menu_bar_append(GTK_MENU_BAR (menu_bar), file_menu);
    gtk_menu_bar_append(GTK_MENU_BAR (menu_bar), edit_menu);
    gtk_menu_bar_append(GTK_MENU_BAR (menu_bar), help_menu);

    /* always display the glbl_main_window as the last step
       so it all splashes on the screen at once. */
    gtk_widget_show(glbl_main_window);

    /* fill the glbl_clist widget with all of the cookies */
    populate_clist(glbl_clist);

    /* after populating the clist, see if there is a netscape
       lock file, and be good enough to let the user know about it */
    check_for_lockfile();

    gtk_main ();

    /* free the memory from the pointers to ints */
    for (i = 0; i < COOKIE_COLUMNS; i++) {
	free(glbl_int_ptr[i]);
    }


    return 0;
}


static void save_file (void) {
    if (glbl_selected_filename_str == NULL) {
	create_save_as_dialogue();
    } else {
	write_file();
    }

}

static void write_file (void) {
    gchar error_msg[1024];
    gchar *cell_contents;
    int i;
    int row;

    /* set the title of the main window to show the selected filename */
    sprintf(glbl_window_title_str, "gtkcookie - %s", glbl_selected_filename_str);
    gtk_window_set_title(GTK_WINDOW (glbl_main_window), glbl_window_title_str);
    
    glbl_cookie_FILE = fopen(glbl_selected_filename_str, "w");
    if (glbl_cookie_FILE == NULL) {
	sprintf(error_msg, "Can't write the cookie file\n\"%s\".\n(Probably not allowed to write\nto this directory, or\nthe filesystem is full.)", glbl_selected_filename_str);
	show_error_dialog(error_msg);
	return;
    }

    fprintf(glbl_cookie_FILE, "# Netscape HTTP Cookie File\n");
    fprintf(glbl_cookie_FILE, "# http://www.netscape.com/newsref/std/cookie_spec.html\n");
    fprintf(glbl_cookie_FILE, "# This is a generated file!  Do not edit.\n\n");

    for (row = 0; row < glbl_clist_rows; row++) {
	for (i = 0; i < (COOKIE_COLUMNS - 1); i++) {
	    gtk_clist_get_text(GTK_CLIST (glbl_clist), row, i, &cell_contents);
	    if (i < (COOKIE_COLUMNS - 2)) {
		fprintf(glbl_cookie_FILE, "%s\t", cell_contents);
	    } else {
		fprintf(glbl_cookie_FILE, "%s\n", cell_contents);
	    }
	}
    }

    fclose(glbl_cookie_FILE);
}

static void handle_about_dialog_box (void) {
    static GtkWidget *about_window = NULL;
    GtkWidget *box1;
    GtkWidget *box2;
    GtkWidget *button;
    GtkWidget *separator;

    GtkWidget *about_text_label;

    if (!about_window) {
	about_window = gtk_window_new (GTK_WINDOW_DIALOG);
	gtk_window_position(GTK_WINDOW(about_window), GTK_WIN_POS_MOUSE);


	gtk_signal_connect (GTK_OBJECT(about_window), "destroy",
			    GTK_SIGNAL_FUNC(gtk_widget_destroyed),
			    &about_window);

	/* REALLY IMPORTANT NOTE! gtk_widget_destroyed (notice the
	   past tense) seems to do something a whole lot different
	   than gtk_widget_destroy (notice the present tense)
	*/


	gtk_window_set_title (GTK_WINDOW (about_window), "About gtkcookie");
	gtk_container_border_width (GTK_CONTAINER (about_window), 0);


	box1 = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (about_window), box1);
	gtk_widget_show (box1);


	box2 = gtk_vbox_new (FALSE, 10);
	gtk_container_border_width (GTK_CONTAINER (box2), 10);
	gtk_box_pack_start (GTK_BOX (box1), box2, TRUE, TRUE, 0);
	gtk_widget_show (box2);

	about_text_label = gtk_label_new("
gtkcookie, version 0.01

Code copyright (c) 1998 Manni Wood.
Cookie icon copyright (c) 1998 Glenn Matheson.

gtkcookie comes with ABSOLUTELY NO WARRANTY.
This is free software, and you are welcome
to redistribute it under certain conditions;
for details, see the file named 'COPYING' that
came with the source code of this programme.

Netscape is a registered trademark of
Netscape Communications Corporation
in the United States and other countries.
");
	gtk_box_pack_start (GTK_BOX (box2), about_text_label, TRUE, TRUE, 0);
	gtk_widget_show (about_text_label);

	separator = gtk_hseparator_new ();
	gtk_box_pack_start (GTK_BOX (box1), separator, FALSE, TRUE, 0);
	gtk_widget_show (separator);


	box2 = gtk_vbox_new (FALSE, 10);
	gtk_container_border_width (GTK_CONTAINER (box2), 10);
	gtk_box_pack_start (GTK_BOX (box1), box2, FALSE, TRUE, 0);
	gtk_widget_show (box2);


	
	button = gtk_button_new_with_label ("Close");
	gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
				   GTK_SIGNAL_FUNC(gtk_widget_destroy),
				   GTK_OBJECT (about_window));
	gtk_box_pack_start (GTK_BOX (box2), button, TRUE, TRUE, 0);
	GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
	gtk_widget_grab_default (button);
	gtk_widget_show (button);
    }

    if (!GTK_WIDGET_VISIBLE (about_window)) {
	gtk_widget_show (about_window);
    } else {
	gtk_widget_destroy (about_window);
    }
}

static void handle_find_dialog_box (void) {
    static GtkWidget *find_window = NULL;
    GtkWidget *box1;
    GtkWidget *hbox2;
    GtkWidget *hbox1;

    GtkWidget *find_text_label;
    GtkWidget *separator;

    GtkWidget *find_button;
    GtkWidget *clear_button;
    GtkWidget *close_button;

    /* each time the find dialog box is opened, set glbl_last_found_row
       to zero, to indicate that whatever the user is about to search on,
       it is the first time he will be searching for that string */
    glbl_last_found_row = -1;

    if (!find_window) {
	find_window = gtk_window_new (GTK_WINDOW_DIALOG);
	gtk_window_position(GTK_WINDOW(find_window), GTK_WIN_POS_MOUSE);


	gtk_signal_connect (GTK_OBJECT(find_window), "destroy",
			    GTK_SIGNAL_FUNC(gtk_widget_destroyed),
			    &find_window);

	/* REALLY IMPORTANT NOTE! gtk_widget_destroyed (notice the
	   past tense) seems to do something a whole lot different
	   than gtk_widget_destroy (notice the present tense)
	*/


	gtk_window_set_title (GTK_WINDOW (find_window), "gtkcookie Search");
	gtk_container_border_width (GTK_CONTAINER (find_window), 0);


	box1 = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (find_window), box1);
	gtk_widget_show (box1);


	hbox1 = gtk_hbox_new (FALSE, 10);
	gtk_container_border_width (GTK_CONTAINER (hbox1), 10);
	gtk_box_pack_start (GTK_BOX (box1), hbox1, TRUE, TRUE, 0);
	gtk_widget_show (hbox1);


	find_text_label = gtk_label_new("Find:");
	gtk_box_pack_start (GTK_BOX (hbox1), find_text_label, TRUE, TRUE, 0);
	gtk_widget_show (find_text_label);

	glbl_find_text_entry = gtk_entry_new ();
	gtk_signal_connect(GTK_OBJECT(glbl_find_text_entry), "changed",
			   GTK_SIGNAL_FUNC(reset_last_found_row),
			   NULL);

	/* I'm a bit worried about what happends to the text inside
	   glbl_find_text_entry when this window is deleted. Is it
	   de-allocated, or memory leaked? */
 	gtk_box_pack_start (GTK_BOX (hbox1), glbl_find_text_entry, TRUE, TRUE, 0);
	gtk_widget_show (glbl_find_text_entry);

	
	separator = gtk_hseparator_new ();
	gtk_box_pack_start (GTK_BOX (box1), separator, FALSE, TRUE, 0);
	gtk_widget_show (separator);


	hbox2 = gtk_hbox_new (FALSE, 10);
	gtk_container_border_width (GTK_CONTAINER (hbox2), 10);
	gtk_box_pack_start (GTK_BOX (box1), hbox2, FALSE, TRUE, 0);
	gtk_widget_show (hbox2);


	find_button = gtk_button_new_with_label ("Find");
	gtk_signal_connect_object (GTK_OBJECT (find_button), "clicked",
				   GTK_SIGNAL_FUNC(do_find),
				   NULL );
	gtk_box_pack_start (GTK_BOX (hbox2), find_button, TRUE, TRUE, 0);
	GTK_WIDGET_SET_FLAGS (find_button, GTK_CAN_DEFAULT);
	gtk_widget_grab_default (find_button);
	gtk_widget_show (find_button);


	clear_button = gtk_button_new_with_label ("Clear");
	gtk_signal_connect_object (GTK_OBJECT (clear_button), "clicked",
				   GTK_SIGNAL_FUNC(do_clear),
				   NULL );
	gtk_box_pack_start (GTK_BOX (hbox2), clear_button, TRUE, TRUE, 0);
	GTK_WIDGET_SET_FLAGS (clear_button, GTK_CAN_DEFAULT);
	gtk_widget_show (clear_button);


	close_button = gtk_button_new_with_label ("Close");
	gtk_signal_connect_object (GTK_OBJECT (close_button), "clicked",
				   GTK_SIGNAL_FUNC(gtk_widget_destroy),
				   GTK_OBJECT (find_window));
	gtk_box_pack_start (GTK_BOX (hbox2), close_button, TRUE, TRUE, 0);
	GTK_WIDGET_SET_FLAGS (close_button, GTK_CAN_DEFAULT);
	gtk_widget_show (close_button);
    }

    if (!GTK_WIDGET_VISIBLE (find_window)) {
	gtk_widget_show (find_window);
    } else {
	gtk_widget_destroy (find_window);
    }
}

static void show_error_dialog (gchar *message) {
    static GtkWidget *error_window = NULL;
    GtkWidget *box1;
    GtkWidget *box2;
    GtkWidget *button;
    GtkWidget *separator;

    GtkWidget *about_text_label;

    if (!error_window) {
	error_window = gtk_window_new (GTK_WINDOW_DIALOG);
	gtk_window_position(GTK_WINDOW(error_window), GTK_WIN_POS_MOUSE);

	gtk_signal_connect (GTK_OBJECT (error_window), "destroy",
			    GTK_SIGNAL_FUNC(gtk_widget_destroyed),
			    &error_window);

	gtk_window_set_title (GTK_WINDOW (error_window), "Error");
	gtk_container_border_width (GTK_CONTAINER (error_window), 0);

	box1 = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (error_window), box1);
	gtk_widget_show (box1);


	box2 = gtk_vbox_new (FALSE, 10);
	gtk_container_border_width (GTK_CONTAINER (box2), 10);
	gtk_box_pack_start (GTK_BOX (box1), box2, TRUE, TRUE, 0);
	gtk_widget_show (box2);

	about_text_label = gtk_label_new(message);
	gtk_box_pack_start (GTK_BOX (box2), about_text_label, TRUE, TRUE, 0);
	gtk_widget_show (about_text_label);

	separator = gtk_hseparator_new ();
	gtk_box_pack_start (GTK_BOX (box1), separator, FALSE, TRUE, 0);
	gtk_widget_show (separator);


	box2 = gtk_vbox_new (FALSE, 10);
	gtk_container_border_width (GTK_CONTAINER (box2), 10);
	gtk_box_pack_start (GTK_BOX (box1), box2, FALSE, TRUE, 0);
	gtk_widget_show (box2);


	button = gtk_button_new_with_label ("OK");

	gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
	 			   GTK_SIGNAL_FUNC(gtk_widget_destroy),
				   GTK_OBJECT (error_window));

	gtk_box_pack_start (GTK_BOX (box2), button, TRUE, TRUE, 0);
	GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
	gtk_widget_grab_default (button);
	gtk_widget_show (button);
    }

    if (!GTK_WIDGET_VISIBLE (error_window)) {
	/* gtk_widget_realize (error_window); */
	gtk_grab_add (error_window);
	gtk_widget_show (error_window);
	/* gdk_window_raise (error_window->window); */
    } else {
	gtk_grab_remove (error_window);
	gtk_widget_destroy (error_window);
    }
}


static void clear_clist (GtkWidget *widget, gpointer data) {
    gtk_clist_clear (GTK_CLIST (data));
    glbl_clist_rows = 0;
}

static void remove_row_clist (GtkWidget *widget, gpointer data) {
    gtk_clist_remove (GTK_CLIST (data), glbl_clist_selected_row);
    glbl_clist_rows--;
}

static void sort_by_column_clist (GtkWidget *widget,
			 gint column, 
			 GdkEventButton * bevent) {
    gint pass, row;
    gchar *cell_contents_1, *cell_contents_2;

    /* the swap_row_clist function is only needed by this function,
       so I declare it in this function's scope. */
    void swap_row_clist(gint row1, gint row2);

    /* as a small but useful optimisation, let us check to see
       if the column we are about to sort on has already been
       sorted. If so, we can return right now. */
    if (column == glbl_last_sorted_column) {
	return;
    }

    /* OK, OK, I know this is only a bubble sort, but hey, it works! */

    /* find a way to make the icon a watch */
    /* set_cursor(150); seems not to be working */
    /* I freeze graphicsl updates to the clist while the sort takes place */
    gtk_clist_freeze(GTK_CLIST (glbl_clist));

    for (pass = 1; pass <= glbl_clist_rows - 1; pass++) {
	for (row = 0; row <= glbl_clist_rows - 2; row++) {
	    gtk_clist_get_text(GTK_CLIST (glbl_clist), row, column,
			       &cell_contents_1);
	    gtk_clist_get_text(GTK_CLIST (glbl_clist), row + 1, column,
			       &cell_contents_2);
	    if (strcmp(cell_contents_1, cell_contents_2) > 0) {
		swap_row_clist(row, row + 1);
	    }
	}
    }
    gtk_clist_thaw(GTK_CLIST (glbl_clist));
    /* find a way to turn the icon from a watch to a pointer */
    /* set_cursor(132); seems not to be working */
}

void swap_row_clist(gint row1, gint row2) {
    gchar *tmp_cookie[COOKIE_COLUMNS];
    gchar *cell_contents;
    gint col;

    initialise_cookie(tmp_cookie);  /* always do this with a new cookie! */

    copy_clist_row_to_cookie(row1, tmp_cookie);

    for (col = 0; col < COOKIE_COLUMNS; col++) {
	gtk_clist_get_text(GTK_CLIST (glbl_clist), row2, col,
			   &cell_contents);
	gtk_clist_set_text(GTK_CLIST (glbl_clist), row1, col,
			   cell_contents);
    }
    for (col = 0; col < COOKIE_COLUMNS; col++) {
	gtk_clist_set_text(GTK_CLIST (glbl_clist), row2, col,
			   tmp_cookie[col]);
    }
    
    /* free the memory used by the cookie before leaving subroutine! */
    free_cookie(tmp_cookie);
}

/*
  void gtk_clist_set_text( GtkCList *clist,
                           gint      row,
                           gint      column,
                           gchar    *text );
  gint gtk_clist_get_text( GtkCList  *clist,
                           gint       row,
                           gint       column,
                           gchar    **text );
*/

static gboolean select_clist_dblclk(GtkWidget *widget,
		  GdkEventButton *event,
		  gpointer func_data) {

    gint row;

    row = glbl_clist_selected_row;
    
    if (GTK_IS_CLIST(widget) && event->type == GDK_2BUTTON_PRESS) {
	create_edit_dialog(widget);
    }

    return FALSE;
}

static void select_clist (GtkWidget *widget,
	 	   gint row, 
		   gint column, 
		   GdkEventButton * bevent) {
    
    gchar *tmp_cookie[COOKIE_COLUMNS];
    gint i;

    /* initialise all of glbl_selected_cookie's gchar pointers to NULL */
    initialise_cookie(tmp_cookie);
    
    for (i = 0; i < COOKIE_COLUMNS; i++) {
	switch (gtk_clist_get_cell_type (GTK_CLIST (widget), row, i)) {
	case GTK_CELL_TEXT:
	    gtk_clist_get_text(GTK_CLIST (widget), row, i, &tmp_cookie[i]);
	    break;

	default:
	    break;
	}
    }

    /* tmp_cookie contains pointers to the strings in the clist,
       but I need a copy of the cookie whose contents I can modify
       without messing with the text in the glbl_clist. Hence, I copy
       tmp_cookie's strings to the global var, glbl_selected_cookie */
    copy_cookie(tmp_cookie, glbl_selected_cookie);

    /* IMPORTANT! set the selected row to the selected row */
    glbl_clist_selected_row = row;
}


static void insert_row_clist (GtkWidget *widget, gpointer data) {
    /* future:
       1. clear the currently selected cookie
       2. gtk_clist_insert the currently selected cookie
       3. make the just-inserted cookie the selected cookie in the
       glbl_clist (ensure all of the global "this is the selected cookie row"
       variables reflect this change)
       4. scroll the glbl_clist so that the new, selected cookie is visible
       5. call the double-click function that brings up the cookie edit
       window */
    static gchar *text[] = {
	"url.com",
	"TRUE",
	"/",
	"FALSE",
	"0",
	"key",
	"value",
	"Thu Jan  1 00:00:00 1970"
    };

    gtk_clist_insert (GTK_CLIST (data), glbl_clist_selected_row, text);
    glbl_clist_rows++;
    gtk_clist_select_row( (GtkCList*) glbl_clist,
                                 glbl_clist_selected_row,
                                 0 );
    create_edit_dialog(widget);
}


static void create_clist (GtkWidget *box1) {
    gint i;  /* iterator */

    GtkWidget *box2;
    GtkWidget *button;

    box2 = gtk_hbox_new (FALSE, 10);
    gtk_container_border_width (GTK_CONTAINER (box2), 10);
    gtk_box_pack_start (GTK_BOX (box1), box2, FALSE, FALSE, 0);
    gtk_widget_show (box2);

    /* create GtkCList here so we have a pointer to throw at the 
     * button callbacks -- more is done with it later */
    glbl_clist = gtk_clist_new_with_titles (COOKIE_COLUMNS, glbl_cookie_titles_str);

    /* control buttons */
    button = gtk_button_new_with_label ("Edit Cookie");
    gtk_box_pack_start (GTK_BOX (box2), button, TRUE, TRUE, 0);
    gtk_signal_connect (GTK_OBJECT (button),
			"clicked",
 			(GtkSignalFunc) create_edit_dialog,
			(gpointer) glbl_clist);
    gtk_widget_show (button);

    button = gtk_button_new_with_label ("Nuke Cookie");
    gtk_box_pack_start (GTK_BOX (box2), button, TRUE, TRUE, 0);
    gtk_signal_connect (GTK_OBJECT (button),
			"clicked",
			(GtkSignalFunc) remove_row_clist,
			(gpointer) glbl_clist);
    gtk_widget_show (button);

    button = gtk_button_new_with_label ("Create Cookie");
    gtk_box_pack_start (GTK_BOX (box2), button, TRUE, TRUE, 0);
    gtk_signal_connect (GTK_OBJECT (button),
			"clicked",
			(GtkSignalFunc) insert_row_clist,
			(gpointer) glbl_clist);
    gtk_widget_show (button);

    /* vbox for the list itself */
    box2 = gtk_vbox_new (FALSE, 10);
    gtk_container_border_width (GTK_CONTAINER (box2), 10);
    gtk_box_pack_start (GTK_BOX (box1), box2, TRUE, TRUE, 0);
    gtk_widget_show (box2);

    /* 
     * the rest of the glbl_clist configuration
     */
    gtk_clist_set_row_height (GTK_CLIST (glbl_clist), 20);
    /* 20 pixels high for a single row? */


    /* if the user clicks on a column title button... */
    gtk_signal_connect (GTK_OBJECT (glbl_clist),
			"click_column",
			(GtkSignalFunc) sort_by_column_clist,
			NULL);
    
    /* if the user clicks on a row... */  
    gtk_signal_connect (GTK_OBJECT (glbl_clist), 
			"select_row",
			(GtkSignalFunc) select_clist, 
			NULL);


    /* if the user double-clicks on a row... */  
    gtk_signal_connect(GTK_OBJECT(glbl_clist),
		       "button_press_event",
		       GTK_SIGNAL_FUNC(select_clist_dblclk),
		       NULL);

    
    /* make each row 80 pixels wide */
    for (i = 0; i < COOKIE_COLUMNS; i++) {
	gtk_clist_set_column_width (GTK_CLIST (glbl_clist), i, 80);
    }

    /* what is a selection mode? */
    gtk_clist_set_selection_mode (GTK_CLIST (glbl_clist), GTK_SELECTION_BROWSE);
    /* what is a policy? */
    gtk_clist_set_policy (GTK_CLIST (glbl_clist), 
			  GTK_POLICY_AUTOMATIC,
			  GTK_POLICY_AUTOMATIC);

    gtk_container_border_width (GTK_CONTAINER (glbl_clist), 5);
    gtk_box_pack_start (GTK_BOX (box2), glbl_clist, TRUE, TRUE, 0);
    gtk_widget_show (glbl_clist);
}

/* POPULATE GLBL_CLIST */

static void populate_clist(gpointer data) {
    gchar file_line[1024];  /* line from the cookie file */
    /* let's hope that this maximum length is long enough to never
       truncate a cookie value... */
    gchar *success;
    gchar *test_cookie[COOKIE_COLUMNS];
    gchar error_msg[1024];
    size_t length;

    glbl_clist_rows = 0;  /* seeing as we are starting fresh, the
			total number of rows in the glbl_clist widget is 0 */

    /* set all gchar pointers in test_cookie to zero */
    initialise_cookie(test_cookie);

    /* if the selected filename is NULL, then we look
       in the default location for the cookie file */
    if (glbl_selected_filename_str == NULL) {
	glbl_home_directory_str = getenv("HOME");
	if (glbl_home_directory_str == NULL) {
	    show_error_dialog ("Environmental variable HOME not set;\ntherefore, can't find home directory\nor .netscape directory.");
	    return;
	}
	length = strlen(glbl_home_directory_str) + strlen("/.netscape/cookies");
	glbl_selected_filename_str = calloc(length + 1, sizeof(gchar));
	strncpy(glbl_selected_filename_str, glbl_home_directory_str, strlen(glbl_home_directory_str));
	strcat(glbl_selected_filename_str, "/.netscape/cookies");
	glbl_selected_filename_str[length] = '\0';
    }

    /* set the title of the main window to show the selected filename */
    sprintf(glbl_window_title_str, "gtkcookie - %s", glbl_selected_filename_str);
    gtk_window_set_title(GTK_WINDOW (glbl_main_window), glbl_window_title_str);

    glbl_cookie_FILE = fopen(glbl_selected_filename_str, "r");
    if (glbl_cookie_FILE == NULL) {
	/* if we couldn't open the cookie file, set
	   glbl_selected_filename_str to
	   NULL so that if the user says 'save', a 'save as' will be
	   invoked instead. */
	if (glbl_selected_filename_str != NULL) {
	    free(glbl_selected_filename_str);
	    glbl_selected_filename_str = NULL;
	}
	sprintf(error_msg, "can't open the cookie file\n\"%s\"\nfor reading.", glbl_selected_filename_str);
	show_error_dialog(error_msg);
	return;
    }

    success = fgets(file_line, 1024, glbl_cookie_FILE);
    if (success && is_a_cookie_line(file_line)) {
	split_cookie_line(file_line, test_cookie);
	gtk_clist_append( (GtkCList*) data, test_cookie);
	glbl_clist_rows++;  /* do this whenever you gtk_clist_append() */
	free_cookie(test_cookie);
    }

    while (success) {
	success = fgets(file_line, 1024, glbl_cookie_FILE);
	if (success && is_a_cookie_line(file_line)) {
	    split_cookie_line(file_line, test_cookie);
	    gtk_clist_append( (GtkCList*) data, test_cookie);
	    glbl_clist_rows++;  /* do this whenever you gtk_clist_append() */
	    free_cookie(test_cookie);
	}
    }

    fclose(glbl_cookie_FILE);
}
/* COOKIE FUNCTIONS */

static gboolean is_a_cookie_line(gchar *line) {
    /* a valid cookie line has 6 tabs */
    int tab_count;
    size_t i;

    tab_count = 0;
    for (i = 0; i < strlen(line); i++) {
	if (line[i] == '\t') {
	    tab_count++;
	}
    }

    if (tab_count == 6) {
	return TRUE;
    } else {
	return FALSE;
    }
}

/* make sure each pointer to gchar in new_cookie points to
   zero. This will make free_cookie() work better, because
   it will only free non-zero pointers */
static void initialise_cookie(gchar *new_cookie[COOKIE_COLUMNS]) {
    int i;
    for (i = 0; i < COOKIE_COLUMNS; i++) {
	new_cookie[i] = NULL;
    }
}

static void split_cookie_line(gchar *line, gchar *new_cookie[COOKIE_COLUMNS]) {
    /* from the string 'line', all the following ints
       will hold the start position and length of each cookie part
    */
    gchar *line_ptr;  /* pointer that will be incremented along a string */
    size_t domain_start, domain_length;
    size_t suffix_start, suffix_length;
    size_t path_start, path_length;
    size_t secure_start, secure_length;
    size_t expire_start, expire_length;
    size_t name_start, name_length;
    size_t value_start, value_length;

    size_t tab_locations[6];
    size_t number_of_tabs_found;

    size_t line_length; /* length of 'line' not including terminating '\0' */
    size_t i;  /* iterator */

    time_t cookie_time;
    struct tm *cookie_tm_ptr;

    /* make line_ptr and line equal. Later, line_ptr will get
       incremented along the string that line points to */
    line_ptr = line;

    line_length = strlen(line);

    number_of_tabs_found = 0;

    for (i = 0; i <= line_length; i++) {
	if (line[i] == '\t') {
	    tab_locations[number_of_tabs_found] = i;
	    number_of_tabs_found++;
	}
    }

    domain_start = 0;
    suffix_start = tab_locations[0] + 1;
    path_start = tab_locations[1] + 1;
    secure_start = tab_locations[2] + 1;
    expire_start = tab_locations[3] + 1;
    name_start = tab_locations[4] + 1;
    value_start = tab_locations[5] + 1;

    domain_length = suffix_start - domain_start - 1;
    suffix_length = path_start - suffix_start - 1;
    path_length = secure_start - path_start - 1;
    secure_length = expire_start - secure_start - 1;
    expire_length = name_start - expire_start - 1;
    name_length = value_start - name_start - 1;
    value_length = line_length - value_start - 1;

    new_cookie[0] = calloc(domain_length + 1, sizeof(gchar));
    strncpy(new_cookie[0], line_ptr, domain_length);
    new_cookie[0][domain_length] = '\0';

    /* increment line_ptr so that it points to a shorter line */
    line_ptr += (domain_length + 1);
    
    new_cookie[1] = calloc(suffix_length + 1, sizeof(gchar));
    strncpy(new_cookie[1], line_ptr, suffix_length);
    new_cookie[1][suffix_length] = '\0';

    line_ptr += (suffix_length + 1);

    new_cookie[2] = calloc(path_length + 1, sizeof(gchar));
    strncpy(new_cookie[2], line_ptr, path_length);
    new_cookie[2][path_length] = '\0';


    line_ptr += (path_length + 1);

    new_cookie[3] = calloc(secure_length + 1, sizeof(gchar));
    strncpy(new_cookie[3], line_ptr, secure_length);
    new_cookie[3][secure_length] = '\0';


    line_ptr += (secure_length + 1);

    new_cookie[4] = calloc(expire_length + 1, sizeof(gchar));
    strncpy(new_cookie[4], line_ptr, expire_length);
    new_cookie[4][expire_length] = '\0';


    line_ptr += (expire_length + 1);

    new_cookie[5] = calloc(name_length + 1, sizeof(gchar));
    strncpy(new_cookie[5], line_ptr, name_length);
    new_cookie[5][name_length] = '\0';


    line_ptr += (name_length + 1);

    new_cookie[6] = calloc(value_length + 1, sizeof(gchar));
    strncpy(new_cookie[6], line_ptr, value_length);
    new_cookie[6][value_length] = '\0';


    /* a human-readable date is created and placed in slot 7 */
    cookie_time = atoi(new_cookie[4]);
    /* cookie_tm_ptr = malloc(sizeof(struct tm)); seems unnecessary */
    cookie_tm_ptr = gmtime(&cookie_time);
    /* interestingly, asctime() always returns a string 26 gchars
       long, including a '\n' and a '\0' at the end. I think I'll
       turn the '\n' into a ' ' */
    new_cookie[7] = calloc(26, sizeof(gchar));
    strncpy(new_cookie[7], asctime(cookie_tm_ptr), 26);
    /* change the '\n' at slot 24 to a ' ' */
    new_cookie[7][24] = ' ';
    /* free(cookie_tm_ptr);  causes SIGSEGV */
}

static void free_cookie(gchar *c[COOKIE_COLUMNS]) {
    int i;
    for (i = 0; i < COOKIE_COLUMNS; i++) {
	if (c[i] != NULL) {
	    free(c[i]);
	    c[i] = NULL;
	}
    }
}

static void copy_cookie(gchar *c1[COOKIE_COLUMNS], gchar *c2[COOKIE_COLUMNS]) {
    /* COPIES c1 to c2 */
    
    /* ASSUMPTION: c1 and c2's pointers are all initialised, either to NULL
       or to a string of gchar */

    int i;
    size_t field_length;

    /* if c2 was pointing to anything, it won't be for long... */
    /* Remember: free_cookie frees any memory pointed to by c2
       as well as setting all of c2's pointers to null */
    free_cookie(c2);

    for (i = 0; i < COOKIE_COLUMNS; i++) {
	if (c1[i] != NULL) {
	    field_length = strlen(c1[i]);
	    c2[i] = calloc(field_length + 1, sizeof(gchar));
	    strncpy(c2[i], c1[i], field_length);
	    c2[i][field_length] = '\0';
	}
    }
    
}


static void copy_clist_row_to_cookie(gint row, gchar *c[COOKIE_COLUMNS]) {
    int i;
    size_t field_length;
    gchar *cell_contents;

    /* if c was pointing to anything, it won't be for long... */
    /* Remember: free_cookie frees any memory pointed to by c
       as well as setting all of c's pointers to null */
    free_cookie(c);

    /* if the row is larger than the number of rows in the clist,
       we can return right now */
    if (row >= glbl_clist_rows) {
	return;
    }

    for (i = 0; i < COOKIE_COLUMNS; i++) {
	gtk_clist_get_text(GTK_CLIST (glbl_clist), row, i, &cell_contents);
	if (cell_contents != NULL) {
	    field_length = strlen(cell_contents);
	    c[i] = calloc(field_length + 1, sizeof(gchar));
	    strncpy(c[i], cell_contents, field_length);
	    c[i][field_length] = '\0';
	}
    }
}


/*
 * GtkEntry
 */

static void create_edit_dialog (GtkWidget *widget) {
    static GtkWidget *window = NULL;
    GtkWidget *box1;
    GtkWidget *box2;
    GtkWidget *table;
    GtkWidget *hbox;
    GtkWidget *text_entry[COOKIE_COLUMNS];

    GtkWidget *suffix_tf_checkbox, *secure_tf_checkbox;
    
    GtkWidget *OK_button;
    GtkWidget *Cancel_button;
    GtkWidget *separator;
    GtkWidget *label, *exp_s_label;
    gint row;
    row = glbl_clist_selected_row;

    if (!window) {
	window = gtk_window_new (GTK_WINDOW_TOPLEVEL);

	gtk_signal_connect (GTK_OBJECT (window), "destroy",
			    GTK_SIGNAL_FUNC(gtk_widget_destroyed),
			    &window);

	gtk_window_set_title (GTK_WINDOW (window), "Edit Cookie");
	gtk_container_border_width (GTK_CONTAINER (window), 0);


	box1 = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (window), box1);
	gtk_widget_show (box1);


	box2 = gtk_vbox_new (FALSE, 10);
	gtk_container_border_width (GTK_CONTAINER (box2), 10);
	gtk_box_pack_start (GTK_BOX (box1), box2, TRUE, TRUE, 0);
	gtk_widget_show (box2);

	table = gtk_table_new(9,
			      2,
                              FALSE);
	gtk_box_pack_start (GTK_BOX (box2), table, TRUE, TRUE, 0);

/* {COL_DOMAIN, COL_SUFFIX, COL_PATH, COL_SECURE, COL_EXPIRE_SECONDS, COL_NAME, COL_VALUE, COL_EXPIRE_DATE}; */

	/* COL_DOMAIN Name Edit Box */
	label = gtk_label_new(glbl_cookie_titles_str[COL_DOMAIN]);
 
	text_entry[COL_DOMAIN] = gtk_entry_new ();
 
	    /* I need to pass a pointer to an int as an argument for the
	       signal connect function. The int represents the
	       column of the bit of cookie data. That's what the
	       glbl_int_ptr[COL_DOMAIN] is for.
	       */
	
	gtk_signal_connect(GTK_OBJECT(text_entry[COL_DOMAIN]), "changed",
			   GTK_SIGNAL_FUNC(change_callback),
			   glbl_int_ptr[COL_DOMAIN]);
	gtk_entry_set_text (GTK_ENTRY (text_entry[COL_DOMAIN]),
			    glbl_selected_cookie[COL_DOMAIN]);

	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 0, 1);
	gtk_table_attach_defaults(GTK_TABLE(table), text_entry[COL_DOMAIN], 1, 2, 0, 1);
	gtk_widget_show (label);
	gtk_widget_show (text_entry[COL_DOMAIN]);


/* {COL_DOMAIN, COL_SUFFIX, COL_PATH, COL_SECURE, COL_EXPIRE_SECONDS, COL_NAME, COL_VALUE, COL_EXPIRE_DATE}; */
	/* Name Edit Box */
	label = gtk_label_new(glbl_cookie_titles_str[COL_NAME]);
 
	text_entry[COL_NAME] = gtk_entry_new ();
 
	gtk_signal_connect(GTK_OBJECT(text_entry[COL_NAME]), "changed",
			   GTK_SIGNAL_FUNC(change_callback),
			   glbl_int_ptr[COL_NAME]);
	gtk_entry_set_text (GTK_ENTRY (text_entry[COL_NAME]),
			    glbl_selected_cookie[COL_NAME]);

	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1, 2);
	gtk_table_attach_defaults(GTK_TABLE(table), text_entry[COL_NAME], 1, 2, 1, 2);
	gtk_widget_show (label);
	gtk_widget_show (text_entry[COL_NAME]);


/* {COL_DOMAIN, COL_SUFFIX, COL_PATH, COL_SECURE, COL_EXPIRE_SECONDS, COL_NAME, COL_VALUE, COL_EXPIRE_DATE}; */
	/* Value Edit Box */
	label = gtk_label_new(glbl_cookie_titles_str[COL_VALUE]);
 
	text_entry[COL_VALUE] = gtk_entry_new ();
 
	gtk_signal_connect(GTK_OBJECT(text_entry[COL_VALUE]), "changed",
			   GTK_SIGNAL_FUNC(change_callback),
			   glbl_int_ptr[COL_VALUE]);
	gtk_entry_set_text (GTK_ENTRY (text_entry[COL_VALUE]),
			    glbl_selected_cookie[COL_VALUE]);

	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 2, 3);
	gtk_table_attach_defaults(GTK_TABLE(table), text_entry[COL_VALUE], 1, 2, 2, 3);
	gtk_widget_show (label);
	gtk_widget_show (text_entry[COL_VALUE]);


/* {COL_DOMAIN, COL_SUFFIX, COL_PATH, COL_SECURE, COL_EXPIRE_SECONDS, COL_NAME, COL_VALUE, COL_EXPIRE_DATE}; */
	/* COL_PATH Edit Box */
	label = gtk_label_new(glbl_cookie_titles_str[COL_PATH]);
 
	text_entry[COL_PATH] = gtk_entry_new ();
 
	gtk_signal_connect(GTK_OBJECT(text_entry[COL_PATH]), "changed",
			   GTK_SIGNAL_FUNC(change_callback),
			   glbl_int_ptr[COL_PATH]);
	gtk_entry_set_text (GTK_ENTRY (text_entry[COL_PATH]),
			    glbl_selected_cookie[COL_PATH]);

	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 3, 4);
	gtk_table_attach_defaults(GTK_TABLE(table), text_entry[COL_PATH], 1, 2, 3, 4);
	gtk_widget_show (label);
	gtk_widget_show (text_entry[COL_PATH]);
	

/* {COL_DOMAIN, COL_SUFFIX, COL_PATH, COL_SECURE, COL_EXPIRE_SECONDS, COL_NAME, COL_VALUE, COL_EXPIRE_DATE}; */
	/* COL_SUFFIX T/F Edit Box */

	suffix_tf_checkbox = gtk_check_button_new_with_label ("Domain field is actually a suffix");
 
	gtk_signal_connect(GTK_OBJECT(suffix_tf_checkbox), "clicked",
			   GTK_SIGNAL_FUNC(change_tf_callback),
			   glbl_int_ptr[COL_SUFFIX]);

	/* turn the button on or off depending on whether the
	   cookie field is "TRUE" or "FALSE" */
	if (strcmp(glbl_selected_cookie[COL_SUFFIX], "TRUE") == 0) {
	    /* checkbox is checked */
	    gtk_toggle_button_set_state( GTK_TOGGLE_BUTTON(suffix_tf_checkbox),
					 TRUE );    
	} else {
	    /* checkbox is unchecked */
	    gtk_toggle_button_set_state( GTK_TOGGLE_BUTTON(suffix_tf_checkbox),
					 FALSE );    
	}

	gtk_table_attach_defaults(GTK_TABLE(table), suffix_tf_checkbox, 0, 2, 4, 5);
	
	gtk_widget_show (suffix_tf_checkbox);

/* {COL_DOMAIN, COL_SUFFIX, COL_PATH, COL_SECURE, COL_EXPIRE_SECONDS, COL_NAME, COL_VALUE, COL_EXPIRE_DATE}; */
	/* COL_SECURE T/F Edit Box */

	secure_tf_checkbox = gtk_check_button_new_with_label ("Make this cookie secure");
 
	gtk_signal_connect(GTK_OBJECT(secure_tf_checkbox), "clicked",
			   GTK_SIGNAL_FUNC(change_tf_callback),
			   glbl_int_ptr[COL_SECURE]);

	/* turn the button on or off depending on whether the
	   cookie field is "TRUE" or "FALSE" */
	if (strcmp(glbl_selected_cookie[COL_SECURE], "TRUE") == 0) {
	    /* checkbox is checked */
	    gtk_toggle_button_set_state( GTK_TOGGLE_BUTTON(secure_tf_checkbox),
					 TRUE );    
	} else {
	    /* checkbox is unchecked */
	    gtk_toggle_button_set_state( GTK_TOGGLE_BUTTON(secure_tf_checkbox),
					 FALSE );    
	}

	gtk_table_attach_defaults(GTK_TABLE(table), secure_tf_checkbox, 0, 2, 5, 6);
	
	gtk_widget_show (secure_tf_checkbox);


	/* Descriptive label above the human-readable date dialogue box */

	label = gtk_label_new("
Cookie expiration dates are stored as
the number of seconds since 1 Jan 1970.
As you edit the human-readble date below,
stick to the following format:
Wdy Mon dd hh:mm:ss yyyy and
gtkcookie will be able to figure
out the time in seconds for you.
");
 
	gtk_widget_show (label);
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 2, 6, 7);

/* {COL_DOMAIN, COL_SUFFIX, COL_PATH, COL_SECURE, COL_EXPIRE_SECONDS, COL_NAME, COL_VALUE, COL_EXPIRE_DATE}; */
	/* Expiry Date (in seconds) Label */
	label = gtk_label_new(glbl_cookie_titles_str[COL_EXPIRE_SECONDS]);
	exp_s_label = gtk_label_new(glbl_selected_cookie[COL_EXPIRE_SECONDS]);
	/* have our global GtkWidget pointer point to the expiry date
	   label, so that we can change the label even after this function
	   has run */
	glbl_selected_etl = exp_s_label;
 
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 7, 8);
	gtk_table_attach_defaults(GTK_TABLE(table), exp_s_label, 1, 2, 7, 8);
	gtk_widget_show (label);
	gtk_widget_show (exp_s_label);


/* {COL_DOMAIN, COL_SUFFIX, COL_PATH, COL_SECURE, COL_EXPIRE_SECONDS, COL_NAME, COL_VALUE, COL_EXPIRE_DATE}; */
	/* COL_EXPIRE_DATE Edit Box */
	label = gtk_label_new(glbl_cookie_titles_str[COL_EXPIRE_DATE]);
 
	text_entry[COL_EXPIRE_DATE] = gtk_entry_new ();
 
	/* have to find a way to pass a pointer to the expiry date in seconds
	   label to this function, so that the function can change the label.
	   In the function itself, use the Linux-specific call strptime()
	   to take care of parsing the date as it is edited. */
	gtk_signal_connect(GTK_OBJECT(text_entry[COL_EXPIRE_DATE]), "changed",
			   GTK_SIGNAL_FUNC(change_time_callback),
			   glbl_int_ptr[COL_EXPIRE_DATE]);
	gtk_entry_set_text (GTK_ENTRY (text_entry[COL_EXPIRE_DATE]),
			    glbl_selected_cookie[COL_EXPIRE_DATE]);

	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 8, 9);
	gtk_table_attach_defaults(GTK_TABLE(table), text_entry[COL_EXPIRE_DATE], 1, 2, 8, 9);
	gtk_widget_show (label);
	gtk_widget_show (text_entry[COL_EXPIRE_DATE]);

	gtk_widget_show (table);

	/* if we're done with the cookie here, nuke it and
	   see what happens */
	/* free_cookie(glbl_selected_cookie); */
	/* OK, what happens is that the cookie really does contain
	   pointers right into the glbl_clist's elements. Freeing the memory
	   that the cookie's gchar pointers point to fucks up the
	   elements in the glbl_clist */

	/* To the best of my knowlege, gtk_entry_set_text copies
	   the string from the pointer into the text widget, but
	   de-allocation of the text seems to be the responsibility
	   of the widget */
	
	separator = gtk_hseparator_new ();
	gtk_box_pack_start (GTK_BOX (box1), separator, FALSE, TRUE, 0);
	gtk_widget_show (separator);

/* buttons at bottom */

	box2 = gtk_vbox_new (FALSE, 10);
	gtk_container_border_width (GTK_CONTAINER (box2), 10);
	gtk_box_pack_start (GTK_BOX (box1), box2, FALSE, TRUE, 0);
	gtk_widget_show (box2);


	
	hbox = gtk_hbox_new (TRUE, 10);
	

	OK_button = gtk_button_new_with_label ("OK");
	gtk_signal_connect (GTK_OBJECT (OK_button), "clicked",
			    GTK_SIGNAL_FUNC(update_clist),
			    GTK_OBJECT(window));
	gtk_box_pack_start (GTK_BOX (hbox), OK_button, TRUE, TRUE, 0);
	GTK_WIDGET_SET_FLAGS (OK_button, GTK_CAN_DEFAULT);
	gtk_widget_grab_default (OK_button);
	gtk_widget_show (OK_button);
	gtk_widget_show (hbox);


	Cancel_button = gtk_button_new_with_label ("Cancel");
	gtk_signal_connect_object (GTK_OBJECT (Cancel_button), "clicked",
				   GTK_SIGNAL_FUNC(gtk_widget_destroy),
				   GTK_OBJECT (window));
	gtk_box_pack_start (GTK_BOX (hbox), Cancel_button, TRUE, TRUE, 0);
	GTK_WIDGET_SET_FLAGS (Cancel_button, GTK_CAN_DEFAULT);
	gtk_widget_show (Cancel_button);
	gtk_widget_show (hbox);

	gtk_box_pack_start (GTK_BOX (box2), hbox, TRUE, TRUE, 0);
    }

    if (!GTK_WIDGET_VISIBLE (window)) {
	gtk_widget_show (window);
    } else {
	gtk_widget_destroy (window);
    }
}


static void change_callback(GtkWidget *widget, gpointer data) {
    gchar *entry_text;
    int i;  /* the number of the cookie column */
    size_t field_length;

    i = *(int *) data;
    entry_text = gtk_entry_get_text(GTK_ENTRY(widget));

    if (entry_text != NULL) {
	field_length = strlen(entry_text);
	/* shouldn't I free the memory before re-allocating it? YES! */
	if (glbl_selected_cookie[i] != NULL) {
	    free(glbl_selected_cookie[i]);
	}
	glbl_selected_cookie[i] = calloc(field_length + 1, sizeof(gchar));
	strncpy(glbl_selected_cookie[i], entry_text, field_length);
	glbl_selected_cookie[i][field_length] = '\0';
    }
}


static void change_time_callback(GtkWidget *widget, gpointer data) {
    gchar *entry_text;
    int i;  /* the number of the cookie column */
    size_t field_length;

    i = *(int *) data;
    entry_text = gtk_entry_get_text(GTK_ENTRY(widget));

    if (entry_text != NULL) {
	field_length = strlen(entry_text);
	/* shouldn't I free the memory before re-allocating it? YES! */
	if (glbl_selected_cookie[i] != NULL) {
	    free(glbl_selected_cookie[i]);
	}
	glbl_selected_cookie[i] = calloc(field_length + 1, sizeof(gchar));
	strncpy(glbl_selected_cookie[i], entry_text, field_length);
	glbl_selected_cookie[i][field_length] = '\0';

	strptime(glbl_selected_cookie[i], "%a %b %d %T %Y", &glbl_selected_time);
	glbl_selected_time_sec = mktime(&glbl_selected_time);
	sprintf(glbl_selected_time_str, "%i", (int)glbl_selected_time_sec);
	gtk_label_set( GTK_LABEL(glbl_selected_etl), glbl_selected_time_str);
	/* gtk_label_set( GTK_LABEL(glbl_selected_etl), glbl_selected_cookie[4]); */

	/* remember, below, that glbl_selected_cookie[4] is the date in seconds */
	if (glbl_selected_time_str != NULL) {
	    field_length = strlen(glbl_selected_time_str);
	    /* shouldn't I free the memory before re-allocating it? YES! */
	    if (glbl_selected_cookie[4] != NULL) {
		free(glbl_selected_cookie[4]);
	    }
	    glbl_selected_cookie[4] = calloc(field_length + 1, sizeof(gchar));
	    strncpy(glbl_selected_cookie[4], glbl_selected_time_str, field_length);
	    glbl_selected_cookie[4][field_length] = '\0';
	}
    }
}

static void change_tf_callback(GtkWidget *widget, gpointer data) {
    int i;  /* the number of the cookie column */
    size_t field_length;
    gchar *True = "TRUE";
    gchar *False = "FALSE";
    gchar *Bool = NULL;
    
    i = *(int *) data;

    if (GTK_TOGGLE_BUTTON (widget)->active) {
	Bool = True;
    } else {
	Bool = False;
    }
    field_length = strlen(Bool);
    /* shouldn't I free the memory before re-allocating it? YES! */
    if (glbl_selected_cookie[i] != NULL) {
	free(glbl_selected_cookie[i]);
    }
    glbl_selected_cookie[i] = calloc(field_length + 1, sizeof(gchar));
    strncpy(glbl_selected_cookie[i], Bool, field_length);
    glbl_selected_cookie[i][field_length] = '\0';
}

static void update_clist(GtkWidget *widget, gpointer window) {
    int column;
    for (column = 0; column < COOKIE_COLUMNS; column++) {
	gtk_clist_set_text( (GtkCList*) glbl_clist, glbl_clist_selected_row, column,
                               glbl_selected_cookie[column] );
    }
    gtk_widget_destroy(window);
}


/* FILE SELECTION STUFF */


static void file_selection_ok (GtkWidget *w, GtkFileSelection *fs) {
    size_t length;
    gchar *filename;
    filename = gtk_file_selection_get_filename (GTK_FILE_SELECTION (fs));
    if (glbl_selected_filename_str != NULL) {
	free(glbl_selected_filename_str);
    }
    if (filename != NULL) {
	length = strlen(filename);
	glbl_selected_filename_str = calloc(length + 1, sizeof(gchar));
	strncpy(glbl_selected_filename_str, filename, length);
	glbl_selected_filename_str[length] = '\0';
    }
    gtk_widget_destroy (GTK_WIDGET (fs));
    /* clear the glbl_clist */
    clear_clist(glbl_clist, (gpointer) glbl_clist);
    populate_clist(glbl_clist);
}

static void create_open_file_dialogue (void) {
    static GtkWidget *window = NULL;

    if (!window) {
	window = gtk_file_selection_new ("Open");

	gtk_file_selection_hide_fileop_buttons (GTK_FILE_SELECTION (window));

	gtk_window_position (GTK_WINDOW (window), GTK_WIN_POS_MOUSE);

	gtk_signal_connect (GTK_OBJECT (window), "destroy",
			    GTK_SIGNAL_FUNC(gtk_widget_destroyed),
			    &window);

	gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (window)->ok_button),
			    "clicked", GTK_SIGNAL_FUNC(file_selection_ok),
			    window);
	gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION (window)->cancel_button),
				   "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
				   GTK_OBJECT (window));
      
    }
  
    if (!GTK_WIDGET_VISIBLE (window)) {
	gtk_widget_show (window);
    } else {
	gtk_widget_destroy (window);
    }
}


static void create_save_as_dialogue (void) {
    static GtkWidget *window = NULL;

    if (!window) {
	window = gtk_file_selection_new ("Save As");

	gtk_file_selection_hide_fileop_buttons (GTK_FILE_SELECTION (window));

	gtk_window_position (GTK_WINDOW (window), GTK_WIN_POS_MOUSE);

	gtk_signal_connect (GTK_OBJECT (window), "destroy",
			    GTK_SIGNAL_FUNC(gtk_widget_destroyed),
			    &window);

	gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (window)->ok_button),
			    "clicked", GTK_SIGNAL_FUNC(save_as_ok),
			    window);
	gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION (window)->cancel_button),
				   "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
				   GTK_OBJECT (window));
      
	gtk_file_selection_set_filename ( (GtkFileSelection*) window, glbl_selected_filename_str);
    }
  
    if (!GTK_WIDGET_VISIBLE (window)) {
	gtk_widget_show (window);
    } else {
	gtk_widget_destroy (window);
    }
}


static void save_as_ok (GtkWidget *w, GtkFileSelection *fs) {
    size_t length;
    gchar *filename;
    filename = gtk_file_selection_get_filename (GTK_FILE_SELECTION (fs));
    if (glbl_selected_filename_str != NULL) {
	free(glbl_selected_filename_str);
    }
    if (filename != NULL) {
	length = strlen(filename);
	glbl_selected_filename_str = calloc(length + 1, sizeof(gchar));
	strncpy(glbl_selected_filename_str, filename, length);
	glbl_selected_filename_str[length] = '\0';
    }
    gtk_widget_destroy (GTK_WIDGET (fs));

    /* set the title of the main window to show the selected filename */
    sprintf(glbl_window_title_str, "gtkcookie - %s", glbl_selected_filename_str);
    gtk_window_set_title(GTK_WINDOW (glbl_main_window), glbl_window_title_str);
    
    /* now that glbl_selected_filename_str has got the new value,
       call save_file(), which will save glbl_selected_filename_str */
    save_file();
}

static void check_for_lockfile(void) {
    gboolean lockfilefound = FALSE;
    gchar error_msg[1024];
    size_t length;
    DIR *dir;
    struct dirent *ent;
    /* glbl_home_directory_str was already set by populate_clist(),
       so dont' re do the work here */
    /* glbl_home_directory_str = getenv("HOME"); */
    if (glbl_home_directory_str == NULL) {
	show_error_dialog ("The environmental variable HOME not set;\ntherefore, we can't find home directory\nor .netscape directory, which means\nwe can't test for a lock file\nto see if Netscape is running.\nIf Netscape IS running,\nnote that any changes you make to the cookie file now\nwill be overwritten by Netscape when it exits.");
	return;
    }
    length = strlen(glbl_home_directory_str) + strlen("/.netscape");
    glbl_netscape_dot_dir_str = calloc(length + 1, sizeof(gchar));
    strncpy(glbl_netscape_dot_dir_str, glbl_home_directory_str, strlen(glbl_home_directory_str));
    strcat(glbl_netscape_dot_dir_str, "/.netscape");
    glbl_netscape_dot_dir_str[length] = '\0';

    /* open the netscape user's directory to check it for a lock file */

    if (!(dir = opendir(glbl_netscape_dot_dir_str))) {
	sprintf(error_msg, "Can't open the netscape user directory.\n\"%s\"",
		strerror(errno));
	show_error_dialog(error_msg);
	return;
    }

    /* set errno to 0, so that we can tell when readdir() fials */
    errno = 0;

    /* here, we are reading the filenames from the netscape user directory
       one filename at a time. If we get a match with "lock", we know a
       lockfile exists, and netscape is running */
    while ((ent = readdir(dir))) {
	if (strcmp(ent->d_name, "lock") == 0) {
	    lockfilefound = TRUE;
	}
    }

    if (errno) {
	sprintf(error_msg, "Error reading the netscape user directory.\n\"%s\"",
		strerror(errno));
	show_error_dialog(error_msg);
	return;
    }

    closedir(dir);

    if (lockfilefound) {
	show_error_dialog("It looks like you're running Netscape right now.\nPlease note that your changes to your netscape cookie file\nwill be lost if you leave Netscape running.\nNetscape writes the cookie file just before exiting,\nand could therefore clobber any changes you make to the file now.");
    }
}

static void do_find(void) {
    gint row;
    gint column;
    gchar *find_text_entry_contents, *cell_contents;

    find_text_entry_contents = gtk_entry_get_text(GTK_ENTRY(glbl_find_text_entry));
    
    for (row = glbl_clist_selected_row; row < glbl_clist_rows; row++) {
	for (column = 0; column < COOKIE_COLUMNS; column++) {
	    gtk_clist_get_text(GTK_CLIST (glbl_clist), row, column,
			       &cell_contents);
	    if (strstr(cell_contents, find_text_entry_contents) != NULL
		&& row > glbl_last_found_row) {
		/* in the if statement directly above, we don't count
		   a search that is successful in the last found row,
		   because that search has already been done, and the
		   user is interested in rows AFTER the current row.
		   In the case of a match after the last found row,
		   set the last found row to the currently matched row
		   so that the number is accurate for the next time
		   this function is called */
		glbl_last_found_row = row;
		gtk_clist_unselect_row(GTK_CLIST(glbl_clist), row, column);
		if (gtk_clist_row_is_visible(GTK_CLIST(glbl_clist), row) ==
					     GTK_VISIBILITY_NONE) {
		    gtk_clist_moveto(GTK_CLIST(glbl_clist), row, column,
				     0.5, 0.5);
		}
		gtk_clist_select_row(GTK_CLIST(glbl_clist), row, column);
		select_clist(glbl_clist, row, column, NULL);
		return;
	    }
	}
    }
    /* if we made it this far, we must have a not found condition! */
    show_error_dialog("Not found.");
    /* printf("not found.\n"); */
}

static void do_clear(void) {
    gtk_entry_set_text(GTK_ENTRY(glbl_find_text_entry), "");
    /* the last found row is rendered invalid when the user clears
       the text box, so reset that variable to -1 */
    glbl_last_found_row = -1;
}

static void reset_last_found_row(void) {
    glbl_last_found_row = -1;
}

/* seems not to be working
static void set_cursor (guint c) {
    GdkCursor *cursor;

    c = CLAMP (c, 0, 152);
    c &= 0xfe;

    cursor = gdk_cursor_new (c);
    gdk_window_set_cursor (glbl_main_window->window, cursor);
    gdk_cursor_destroy (cursor);
}
*/

/* end */
