/* 
** Copyright (C) 1994, 1995 Enterprise Integration Technologies Corp.
**         VeriFone Inc./Hewlett-Packard. All Rights Reserved.
** Kevin Hughes, kev@kevcom.com 3/11/94
** Kent Landfield, kent@landfield.com 4/6/97
** 
** This program and library is free software; you can redistribute it and/or 
** modify it under the terms of the GNU (Library) General Public License 
** as published by the Free Software Foundation; either version 2 
** of the License, or any later version. 
** 
** This program is distributed in the hope that it will be useful, 
** but WITHOUT ANY WARRANTY; without even the implied warranty of 
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the 
** GNU (Library) General Public License for more details. 
** 
** You should have received a copy of the GNU (Library) General Public License
** along with this program; if not, write to the Free Software 
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA 
*/

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "hypermail.h"
#include <ctype.h>
#include <errno.h>

/* 
** Used to replace invalid characters in supplied attachment filenames
*/
#define REPLACEMENT_CHAR '_'

/* 
** Path separator for attachment file path generation
*/
#define PATH_SEPARATOR '/'

typedef enum {
    ENCODE_NORMAL,
    ENCODE_QP,     /* quoted printable */

    ENCODE_MULTILINED, /* this is not a real type, but just a separator showing
                          that the types below are encoded in a way that makes
                          one line in the indata may become one or more lines
                          in the outdata */

    ENCODE_BASE64, /* base64 */
    ENCODE_UUENCODE, /* well, it seems there exist some kind of semi-standard
                        for uu-encoded attachments. */
 
    ENCODE_UNKNOWN /* must be the last one */
} EncodeType;

typedef enum {
    CONTENT_TEXT,   /* normal mails are text based default */
    CONTENT_BINARY, /* this kind we store separately and href to */
    CONTENT_HTML,   /* this is html formated text */
    CONTENT_IGNORE, /* don't care about this content */

    CONTENT_UNKNOWN /* must be the last one */
} ContentType;


int ignorecontent(char *type)
{
   return (inlist(ignore_types, type));
}

int inlinecontent(char *type)
{
   return (inlist(inline_types, type));
}

int preferedcontent(char *type)
{
  /* We let plain text remain PREFERED at all times */
  if(!strcasecmp("text/plain", type))
    return 1;

#if 0
  /* 
  ** Parsing for embedded html needs more work before we 
  ** can actually do this... 
  */
  if(prefered_types) {
    return (inlist(prefered_types, type));
  }
#endif

  return 0;
}

char *tmpname(char *dir, char *pfx)
{
    char *f, *name;
    static cntr = 0;

    char *mktemp(char *template);

    if ((name = malloc(MAXLINE)) == NULL)
          return(NULL);

    sprintf(name, "%s/%s%dXXXXXX", dir, pfx, cntr++);

    if ((f = mktemp(name)) != NULL)
        return(f);

    free(name);
    return(NULL);
}

char *safe_filename(char *name)
{
    register char *sp;
    register char *np;

    static char safe_name[129];

    np = name;
    while (*np && (*np == ' ' || *np == '\t'))
        np++;

    if (!*np)
       return(NULL);

    for (sp = safe_name, np = name; *np; ) {
         /* if valid character then store it */
         if ((*np >= 'a' && *np <= 'z') || (*np >= '0' && *np <= '9') ||
             (*np >= 'A' && *np <= 'Z') || (*np == '-') || (*np == '.') ||
             (*np == ':') || (*np == '_')) {
            *sp = *np;
         }
         else  /* Need to replace the character with a safe one */
            *sp = REPLACEMENT_CHAR;
         sp++;
         np++;
    }
    *sp = '\0';

    return(safe_name);
}

/*
** Cross-indexes - adds to a list of replies. If a message is a reply to
** another, the number of the messge it's replying to is added to the list.
** This list is searched upon printing.
*/

void crossindex()
{
    int num, status, maybereply;
    char name[NAMESTRLEN], subject[SUBJSTRLEN], email[MAILSTRLEN],
        inreply[REPYSTRLEN], date[DATESTRLEN], fromdate[DATESTRLEN],
        msgid[MSGDSTRLEN];

    num = 0;
    replylist = NULL;

    while (hashnumlookup(num,name,email,subject,inreply,date,fromdate,msgid) != NULL) {
        if (inreply[0] != '\0') {
            status = hashreplynumlookup(inreply, &maybereply);
            if (status != -1)
                replylist = (struct reply *)
                addreply(replylist,status,num,name,subject,maybereply);
        }
        num++;
    }
}

/* 
** Recursively checks for replies to replies to a message, etc.
** Replies are added to the thread list.
*/

void crossindexthread2(int num)
{
    struct reply *rp;

    for (rp = replylist; rp != NULL; rp = rp->next) {
        if (rp->msgnum == num) {
            threadlist = (struct reply *)addreply(threadlist, rp->frommsgnum, 
                           num, rp->name, rp->subject, 0);
            printedlist = (struct printed *)
                        markasprinted(printedthreadlist, rp->frommsgnum);
            crossindexthread2(rp->frommsgnum);
        }
    }
}


/*
** First, print out the threads in order by date...
** Each message number is appended to a thread list. Threads and individual
** messages are separated by a -1.
*/

void crossindexthread1(struct header *hp)
{
    int hasreply;
    struct reply *rp;

    if (hp != NULL) {
        crossindexthread1(hp->left);

        for (hasreply = 0, rp = replylist; rp != NULL; rp = rp->next) {
             if (rp->frommsgnum == hp->msgnum) {
                 hasreply = 1;
                 break;
             }
        }

        if (!hasreply && !wasprinted(printedthreadlist, hp->msgnum)) {
            threadlist = (struct reply *)
              addreply(threadlist,hp->msgnum,hp->msgnum,hp->name,hp->subject,0);
            crossindexthread2(hp->msgnum);
            threadlist = (struct reply *)addreply(threadlist,-1,-1," "," ",0);
        }

        crossindexthread1(hp->right);
    }
}

/*
** Grabs the date string from a Date: header.
*/

char *getmaildate(char *line)
{
    int i;
    int len;
    char *c;
    static char date[DATESTRLEN];

    c = (char *) strchr(line, ':');
    if ((*(c + 1) && *(c + 1) == '\n') || (*(c + 2) && *(c + 2) == '\n')) {
        strcpy(date, NODATE);
        return date;
    }
    c += 2;
    while (*c == ' ' || *c == '\t')
        c++;
    for (i = 0, len = DATESTRLEN-1; *c && *c != '\n' && i < len; c++)
        date[i++] = *c;
    date[i] = '\0';

    return date;
}

/*
** Grabs the date string from a From article separator.
*/

char *getfromdate(char *line)
{
    static char tmpdate[DATESTRLEN];
    int i;
    int len;
    char *c = NULL;

    for (i = 0; days[i] != NULL && 
        ((c = (char *) strstr(line, days[i])) == NULL); i++)
        ;
    if (days[i] == NULL)
        tmpdate[0] = '\0';
    else {
        for (i = 0, len = DATESTRLEN-1; *c && *c != '\n' && i < len; c++)
             tmpdate[i++] = *c;

        tmpdate[i] = '\0';

        if (tmpdate[16] != ':') {
                tmpdate[16] = ':';
                tmpdate[17] = '0';
                tmpdate[18] = '0';
                tmpdate[19] = ' ';
                tmpdate[20] = thisyear[0];
                tmpdate[21] = thisyear[1];
                tmpdate[22] = thisyear[2];
                tmpdate[23] = thisyear[3];
                tmpdate[24] = '\0';
        }
        sprintf(tmpdate, "%s %s", tmpdate, timezonestr);
        return tmpdate;
    }

    return tmpdate;
}


/*
** Grabs the name and email address from a From: header.
** This could get tricky; I've tried to keep it simple.
** Should be able to handle all the addresses below:
**
**   From: user                   [no @]
**   From: kent (Kent Landfield)  [no @ - with comment]
**   From: <user@node.domain>     [no text name, use email as text name]
**   From: Kent Landfield <kent>  [text name but no @]
**   From: (kent)                 [comment - no email address]
**   From: "" <kent>              [email address but null comment]
**   From:                        [blank From: line]
**   From: uu.net!kent            [uucp addresses - no comment]
**   From: uu.net!kent (kent)     [uucp addresses - with comment]
**   From: "(Joe Bloggs)" <joe@anorg.com> 
**   From: "Roy T. Fielding" <fielding@kiwi.ics.uci.edu>
**   From: kent@localhost
**   From: kent@uu.net (Kent Landfield)
**   From: (George Burgyan) <gburgyan@cybercon.com>
**   From: <gburgyan@cybercon.com> (George Burgyan) 
**   From:              Kent B. Landfield <kent@landfield.com>
**   From:      IN%"fekete+reply@c2.net" 26-JAN-1997 13:28:55.36
**   From:      IN%"vicric@panix.com"  "Vicki Richman" 13-AUG-1996 10:54:33.38
**   From:      US2RMC::"lwv26@cas.org" "Larry W. Virden, x2487" 22-OCT-1994 09:44:21.44
**   From:          Mail Delivery Subsystem <postmaster@igc.apc.org>
**   From:          Self <ehasbrouck>
**   From:         adam@eden.apana.org.au (Adam Frey)
**   From:        faqserv@penguin-lust.mit.edu
**   From:    nc0548@freebsd.netcom.com (Mark Hittinger)
**   From: "- Pam Greene, one of the *.answers moderators" <pgreene@MIT.EDU>
**   From: "Felan shena Thoron'edras" <felan@netcom.com>
**   From: David Muir Sharnoff <muir@idiom.com>
**   From: A.J.Doherty@reading.ac.uk (Andy Doherty)
**   From: Jordan Hubbard                        <jkh@vector.eikon.e-technik.tu-muenchen.de>
**   From: ZXPAN%SLACVM.BITNET@MITVMA.MIT.EDU
**   From: afs!piz!alf@uu5.psi.com (Alf the Poet)
**   From: answers@cp.tn.tudelft.nl ("Moderator *.answers")
**   From: mdw%merengue@merengue.oit.unc.edu (Matt Welsh)
**   From: bgoffe@whale.st.usm.edu (William L. Goffe)
*/

void getname(char *line, char *name, char *email)
{
    int i;
    int len;
    char *c;

    len = MAILSTRLEN-1;

    /*
    ** Zeroout data.
    */
    zero_out(email, MAILSTRLEN);
    zero_out(name, NAMESTRLEN);

    /* EMail Processing First:
    ** First, is there an '@' sign we can use as an anchor ?
    */
    if ((c = (char *) strchr(line, '@')) == NULL) {
        /* 
        ** No '@' sign here so ...
        */
        if (strchr(line, '(')) {       /* From: bob (The Big Guy) */
            c = (char *) strchr(line, ':') + 1;
            while (*c == ' ' || *c == '\t')
                c++;
            for (i = 0; *c && *c != '(' && *c != ' ' && 
                 *c != '\t' && *c != '\n' && i < len; c++)
                email[i++] = *c;
            email[i] = '\0';
        }
        else if ((c = (char *) strchr(line, '<'))) {  /* From: <kent> */
            c++;
            for (i = 0; *c && *c != '>' && *c != ' ' && 
                 *c != '\t' && *c != '\n' && i < len; c++)
                email[i++] = *c;
            email[i] = '\0';
        }
        else {
            /* 
            **    - check to see if the From: line is blank, (taken care of)
            **    - check if From: uu.net!kent formatted line
            **    - check if "From: kent" formatted line
            */
            c = (char *) strchr(line, ':') + 1;
            while (*c == ' ' || *c == '\t')
                   c++;
            for (i = 0; *c && *c != ' ' && *c != '\t' && 
                *c != '\n' && *c != ',' && i < len; c++)
                 email[i++] = *c;
            email[i] = '\0';

        }

        if (email[0] == '\0') /* Was it a junk From line ? */
            strcpy(email, NOEMAIL);

        else if (use_domainaddr) { 
            /*
            ** check if site domainizes addresses 
            ** but don't modify uucp addresses
            */
            if ((c = (char *) strchr(email, '!')) == NULL) {
                strcat(email, "@");
                strcat(email, domainaddr);
            }
        }
    }
    else {
        while (*c != ' ' && *c != '\t' && *c != '<' && *c != '"')
            c--;
        c++;
        for (i = 0; *c && *c != '>' && *c != ' ' && *c != '\t' &&
             *c != '"' && *c != '\n' && i < len; c++)
            email[i++] = *c;
        email[i] = '\0';
    }

    /*
    ** NAME Processing - Boy are there a bunch of funky formats here.
    **                   No promises... I'll do my best. Let me know
    **                   what I missed...
    */
    if (strchr(line, '(')) {
        c = (char *) strchr(line, '(') + 1;
        if (*c == '"') /* is there a comment in the comment ? */
             c++;
    }
    else if (strchr(line, '<')) {
        c = (char *) strchr(line, ':') + 1;
        while (*c == ' ' || *c == '\t' || *c == '\"')
            c++;
    }
    else {
        /*
        ** Is there an email address available 
        ** that we can use for the name ?
        */
        if (!strcmp(email, NOEMAIL))    /* No */
            strcpy(name, NONAME);
        else {
            c = email+strlen(email) - 1;
            while (isspace(*c)) *c-- = '\0';
            strcpy(name, email);        /* Yes */
        }
        return;
    }

    for (i = 0, len = NAMESTRLEN-1; *c && *c != '<' && *c != '\"' && *c != ')' && *c != '(' && *c != '\n' && i < len; c++)
        name[i++] = *c;

    if (*c == '<' || *c == '(')
        name[--i] = '\0';
    else
        name[i] = '\0';

    
    /* Bailing and taking the easy way out... */

    if (name[0] == '\0') {
        if (email[0] == '\0')
            strcpy(name, NONAME);
        else
            strcpy(name, email);
    }

     /* need to strip spaces off the end of 
     ** the email and name strings 
     */

     c = email+strlen(email) - 1;
     while (isspace(*c)) *c-- = '\0';
     c = name+strlen(name) - 1;
     while (isspace(*c)) *c-- = '\0';
}

/* 
** Grabs the message ID, like <...> from the Message-ID: header.
*/


char *getid(char *line)
{
    int i;
    int len;
    char *c;
    static char msgid[MSGDSTRLEN];
 
    if ((char *)strchr(line, '<') == NULL) {
        /* bozo alert!
        **   msg-id = "<" addr-spec ">" 
        ** try to recover as best we can
        */
        c = (char *) strchr(line, ':') + 1; /* we know this exists! */

        /* skip spaces before message ID */
        while (*c && (*c == ' ' || *c == '\t')) c++; 
    }
    else 
        c = (char *) strchr(line, '<') + 1;

    for (i = 0, len = MSGDSTRLEN -1; *c && *c != '>' && *c != '\n' && i < len; c++) {
        if (*c == '\\')
            continue;
        msgid[i++] = *c;
    }

    msgid[i] = '\0';

    if (strlen(msgid) == 0) 
        strcpy(msgid, "BOZO");

    return msgid;
}


/*
** Grabs the subject from the Subject: header.
*/

char *getsubject(char *line)
{
    int i;
    int len;
    char *c;
    static char subject[SUBJSTRLEN];

    c = (char *) strchr(line, ':') + 2;
    while (isspace(*c))
         c++;
    for (i = 0, len = SUBJSTRLEN -1; *c && *c != '\n' && i < len; c++)
         subject[i++] = *c;

    subject[i] = '\0';

    /* strip trailing spaces */


    for (i--; i >= 0 && isspace(subject[i]); i--)
         subject[i] = '\0';

    /*
    ** Fix for Exchange bug that puts "FW:" in front of forwarded 
    ** messages which are substituted by Exchange for "RE:" on replies
    */

    if (!strncmp(subject, "FW:", 3)) {
        subject[0] = 'R';
        subject[1] = 'e';
    }   

    if (subject[0] ==  '\0' || isspace(subject[0]) || subject[0] == '\n' || 
        !strcasecmp(subject, "Re:") || !strcasecmp(subject, "Re[2]:"))
        strcpy(subject, NOSUBJECT);

    return subject;
}

/*
** Grabs the message ID, or date, from the In-reply-to: header.
**
** Maybe I'm confused but....
**     What either ? Should it not be consistent and choose to return 
**     one (the msgid) as the default and fall back to date when a 
**     msgid cannot be found ?
**
** Who knows what other formats are out there...
**
** In-Reply-To: <1DD9B854E27@everett.pitt.cc.nc.us>
** In-Reply-To: <199709181645.MAA02097@mail.clark.net> from "Marcus J. Ranum" at Sep 18, 97 12:41:40 pm
** In-Reply-To: <199709181645.MAA02097@mail.clark.net> from 
** In-Reply-To: "L. Detweiler"'s message of Fri, 04 Feb 94 22:51:22 -0700 <199402050551.WAA16189@longs.lance.colostate.edu>
**
** This routine is buggy and the fix is to add a readline routine that 
** concatenates continuation lines into a single line and collapses 
** whitespace. The message id should always be returned for threading 
** purposes. Mixing message-ids and dates just does not allow for proper 
** threading lookups.
*/
 
char *getreply(char *line)
{
    int i;
    int replylen;
    int mlen;
    char *c;
    char *m;
    static char reply[REPYSTRLEN];

    replylen = REPYSTRLEN -1;
    mlen = MSGDSTRLEN -1;

    /* Check for blank line */

    /* 
    ** Check for line with " from " and " at ".  Format of the line is 
    **     <msgid> from "quoted user name" at date-string
    */

    if ((char *) strstr(line, " from ") != NULL) {
        if (((char *) strstr(line, " at ")) != NULL) {
            if ((m = (char *) strchr(line, '<')) != NULL) {
                for (m++, i = 0; *m && *m != '>' && *m != '\n' && i < replylen; m++)
                     reply[i++] = *m;
                reply[i] = '\0';
                return(reply);
            }
        }

        /* 
        ** If no 'at' the line may be a continued line or a truncated line.
        ** Both will be picked up later.
        */
    }

    /* 
    ** Check for line with " message of ".  Format of the line is 
    **     "quoted user name"'s message of date-string <msgid>
    */

    if ((c = (char *) strstr(line, "message of ")) != NULL) {
        /*
        ** Check to see if there is a message ID on the line. 
        ** If not this is a continued line and when you add a readline()
        ** function that concatenates continuation lines collapsing
        ** white space, you might want to revisit this...
        */

        if ((m = (char *) strchr(line, '<')) != NULL) {
             for (m++, i = 0; *m && *m != '>' && *m != '\n' && i < replylen; m++)
                  reply[i++] = *m;
             reply[i] = '\0';
             return(reply);
        }
  
        /* Nope... Go for the Date info... Bug...*/
        c += 11;
        while (isspace(*c))
            c++;
        if (*c == '"') 
            c++;

        for (i = 0; *c && *c != '.' && *c != '\n' && i < replylen; c++)
             reply[i++] = *c;
        reply[i] = '\0';
        return(reply);
    }

    if ((c = (char *) strstr(line, "dated: ")) != NULL) {
        c += 7;
        for (i = 0; *c && *c != '.' && *c != '\n' && i < replylen; c++)
             reply[i++] = *c;
        reply[i] = '\0';
        return(reply);
    }
 
    if ((c = (char *) strstr(line, "dated ")) != NULL) {
        c += 6;
        for (i = 0; *c && *c != '.' && *c != '\n' && i < replylen; c++)
             reply[i++] = *c;
        reply[i] = '\0';
        return(reply);
    }

    if ((c = (char *) strchr(line, '<')) != NULL ) {
        c++;
        for (i = 0; *c && *c != '>' && *c != '\n' && i < mlen; c++) {
             if (*c == '\\')
                 continue;
             reply[i++] = *c;
        }
        reply[i] = '\0';
        return(reply);
    }
 
    if ((c = (char *) strstr(line, "sage of ")) != NULL) {
        c += 8;
        if (*c == '\"')
            c++;

        for (i = 0; *c && *c != '.' && *c != '\n' && 
             *c != 'f' && i < replylen; c++)
             reply[i++] = *c;
        reply[i] = '\0';

        if (*c == 'f')
            reply[--i] = '\0';
        return(reply);
    }
 
    reply[0] = '\0';
    return(reply);
}

/*
** RFC 2047 defines MIME extensions for mail headers.
**
** This function decodes that into binary/8bit data.
**
** Example:
**   =?iso-8859-1?q?I'm_called_?= =?iso-8859-1?q?Daniel?=
**
** Should result in "I'm called Daniel", but:
**
**   =?iso-8859-1?q?I'm_called?= Daniel
**
** Should result in "I'm called Daniel" too.
**
** Returns the newly allcated string, or the previous if nothing changed 
*/

static char *mdecodeRFC2047( char *string, int length )
{
    char *iptr = string;
    char *oldptr;
    char *storage=malloc(length+1);
  
    char *output = storage;
  
    char charset[129];
    char encoding[33];
    char blurb[129];
    char equal;
    int value;
  
    char didanything=FALSE;

    while (*iptr) {
        if (!strncmp(iptr, "=?", 2) &&
            (4 == sscanf(iptr+2, "%128[^?]?%32[^?]?%128[^?]?%c",
                        charset, encoding, blurb, &equal)) &&
            ('=' == equal)) {
            /* This is a full, valid 'encoded-word'. Decode! */
            char *ptr=blurb;
  
            didanything=TRUE; /* yes, we decode something */
  
            /* we could've done this with a %n in the sscanf, but we know all
               sscanfs don't grok that */

            iptr += 2+ strlen(charset) + 1 + strlen(encoding) + 1 + strlen(blurb) + 2;
  
            if (!strcasecmp("q", encoding)) {
               /* quoted printable decoding */
  
                for ( ; *ptr; ptr++ ) {
                    switch ( *ptr ) {
                        case '=':
                          sscanf( ptr+1, "%02X", &value );
                          *output++ = value;
                          ptr += 2;
                          break;
                        case '_':
                          *output++ = ' ';
                          break;
                        default:
                          *output++ = *ptr;
                          break;
                    }
                }
            }
            else if (!strcasecmp("b", encoding)) {
                /* base64 decoding */
                int length;
                base64Decode(ptr, output, &length);
                output += length-1;
            }
            else {
                /* unsupported encoding type */
                strcpy(output, "<unknown>");
                output += 9;
            }
  
            oldptr = iptr; /* save start position */
  
            while (*iptr && isspace(*iptr))
                  iptr++; /* pass all whitespaces */
  
            /* if this is an encoded word here, we should skip the passed
               whitespaces. If it isn't an encoded-word, we should include the
               whitespaces in the output. */

            if (!strncmp(iptr, "=?", 2) &&
                (4 == sscanf(iptr+2, "%128[^?]?%32[^?]?%128[^?]?%c",
                              charset, encoding, blurb, &equal)) &&
                ('=' == equal)) {
                continue; /* this IS an encoded-word, continue from here */
            }
            else
              /* this IS NOT an encoded-word, move back to the first whitespace */
              iptr = oldptr;
        }
        else
            *output++ = *iptr++;   
    }
    *output=0;
  
    if (didanything) {
        /* this check prevents unneccessary strsav() calls if not needed */
        free(string); /* free old memory */
  
#if 0
        /* debug display */
        printf("NEW: %s\n", storage);
  
        {  
            unsigned char *f;
            puts("NEW:");
            for (f=storage; f<output; f++) {
                if (isgraph(*f))
                   printf("%c", *f);
                else
                   printf("%02X", (unsigned char)*f);
            }
            puts("");
        }  
#endif
        return storage; /* return new */
    }
    else {
      free (storage);
      return string;
    }
}

/*
** Decode this [virtual] Quoted-Printable line as defined by RFC2045.
** Written by Daniel.Stenberg@sth.frontec.se
*/

static void mdecodeQP(FILE *file, char *input, char **result, int *length)
{
    int outcount=0;
    char *buffer=input;
    unsigned char inchar;
    char *output;

    int len=strlen(input);  
    output=strsav(input);

#define QP_ENLARGE_STEP 512

    while ((inchar = *input) != '\0') {

        if (outcount>=len-1) {
            /* we need to enlarge the destination area! */
            char *newp = realloc(output, len+QP_ENLARGE_STEP);
            if (newp) {
                output = newp;
                len += QP_ENLARGE_STEP;
            }
            else
                break;
        }

        input++;
        if ('=' == inchar) {
            int value;
            if ('\n'== *input) {
                if (!fgets(buffer, MAXLINE, file))
                    break;
                input = buffer;
                continue;
            }
            else if ('=' == *input) {
                inchar='=';
                input++; /* pass this */
            }
            else if (isxdigit(*input)) {
                sscanf(input, "%02X", &value);
                inchar = (unsigned char)value;
                input+=2; /* pass the two letters */
            }
            else
                inchar='=';
        }
        output[outcount++] = inchar;
    }
    output[outcount]=0; /* zero terminate */

    *result = output;
    *length = outcount;
}

/*
** Parsing...the heart of Hypermail!
** This loads in the articles from stdin or a mailbox, adding the right
** field variables to the right structures. If readone is set, it will
** think anything it reads in is one article only.
*/

void loadheaders(char *mbox, int use_stdin, int readone, char *dir)
{
    FILE *fp;
    char name[NAMESTRLEN], email[MAILSTRLEN], date[DATESTRLEN],
        msgid[MSGDSTRLEN], subject[SUBJSTRLEN], inreply[REPYSTRLEN],
        line[MAXLINE], tmpfrom[MAXLINE], fromdate[DATESTRLEN],
        oldline[MAXLINE], *cp;
    int num, isinheader, hassubject, hasdate;

    /* -- variables for the multipart/alternative parser -- */
    struct body *origbp=NULL;     /* store the original bp */
    struct body *origlp=NULL;     /* ... and the original lp */
    char alternativeparser=FALSE; /* set when inside alternative parser mode */
    /* -- end of alternative parser variables -- */

    struct body *bp;
    struct body *lp=NULL; /* the last pointer, points to the last node in the
                             body list. Initially set to NULL since we have
                             none at the moment. */

    struct body *headp=NULL; /* stored pointer to the point where we last
                                scanned the headers of this mail. */

    char Mime_B = FALSE;
    char boundbuffer[128]="";

    struct boundary *boundp=NULL; /* This variable is used to store a stack 
                                     of boundary separators in cases with mimed 
                                     mails inside mimed mails */

    char multilinenoend=FALSE; /* This variable is set TRUE if we have read 
                                  a partial line off a multiline-encoded line, 
                                  and the next line we read is supposed to get
                                  appended to the previous one */

    int bodyflags=0;           /* This variable is set to extra flags that the 
                                  addbody() calls should OR in the flag parameter */

    char *binname=NULL;        /* file name to store binary attachments in */
    int binfile=-1;

    EncodeType decode=ENCODE_NORMAL;
    ContentType content=CONTENT_TEXT;
  
    if (!strcasecmp(mbox, "NONE") || use_stdin)
        fp = stdin;
    else if ((fp = fopen(mbox, "r")) == NULL) {
            sprintf(errmsg, "Couldn't open mail archive \"%s\".", mbox);
            progerr(NULL);
    }
  
    if (readone)
        num = bignum;
    else
        num = 0;

    hassubject = 0;
    hasdate = 0;
    isinheader = 1;
    strcpy(subject, NOSUBJECT);
    inreply[0] = '\0';
    tmpfrom[0] = '\0';
    oldline[0] = '\0';
    msgid[0] = '\0';
    bp = NULL;

    if (!readone) {
        replylist = NULL;
        subjectlist = NULL;
        authorlist = NULL;
        datelist = NULL;
    }

    if (showprogress && readone)
        printf("Reading new header...\n");

    if (showprogress && !readone) {
        if (!strcasecmp(mbox, "NONE") || use_stdin)
            printf("Loading mailbox...    ");
        else
            printf("Loading mailbox \"%s\"...    ", mbox);
    }
    while (fgets(line, MAXLINE, fp) != NULL) {
#if 0
      printf("IN: %s", line);
#endif
        if (isinheader) {
            if (!strncasecmp(line, "From ", 5))
                strcpymax(fromdate, (char *) getfromdate(line), DATESTRLEN);
            /* check for MIME */
            else if (!strncasecmp( line, "MIME-Version:", 13))
                Mime_B = TRUE;
            else if (isspace(line[0]) && ('\n' != line[0]) ) {
                /*
                ** since this begins with a whitespace, it means the 
                ** previous line is continued on this line, leave only 
                ** one space character and go! 
                */
                char *ptr=line;
                while (isspace(*ptr))
                     ptr++;
                ptr--; /* leave one space */
                *ptr=' '; /* make it a true space, no tabs here! */
#if 0
                decodeRFC2047(ptr+1, MAXLINE-(ptr+2-line));
#endif
                bp = (struct body *) addbody(bp, &lp, ptr,
                                       BODY_CONTINUE|BODY_HEADER|bodyflags);
            }
  
            else if (line[0] == '\n') {
                struct body *head;

                char savealternative;

                /* 
                ** we mark this as a header-line, and we use it to 
                ** track end-of-header displays 
                */
                bp = (struct body *) addbody(bp, &lp, line, 
                                      BODY_HEADER|bodyflags);
                isinheader--;

#if 0
                printf("HEADER status: %d\n", isinheader);
#endif

                /*
                ** This signals us that we are no longer in the header, 
                ** let's fill in all those fields we are interested in. 
                ** Parse the headers up to now and copy to the target 
                ** variables 
                */
  
                for (head = bp; head; head=head->next) {
                    if (head->header && !head->demimed && Mime_B) {
                       head->line = mdecodeRFC2047(head->line,
                                                   strlen(head->line));
                       head->demimed=TRUE; /* don't do this again */
                    }
  
                    if (head->attached || !head->header) {
                       continue;
                    }
  
                    if (!strncasecmp(head->line, "Date:", 5)) {
                        strcpymax(date, (char *)getmaildate(head->line), 
                                    DATESTRLEN);
                        hasdate = 1;
                    }
                    else if (!strncasecmp(head->line, "From:", 5)) {
                        getname(head->line, name, email);
                    }
                    else if (!strncasecmp(head->line, "Message-Id:", 11)) {
                        strcpymax(msgid,(char *)getid(head->line),MSGDSTRLEN);
                    }
                    else if (!strncasecmp(head->line, "Subject:", 8)) {
                        strcpymax(subject,(char *)getsubject(head->line), 
                                  SUBJSTRLEN);
                        hassubject = 1;
                    }
                    else if (!strncasecmp(head->line, "In-Reply-To:", 12)) {
                        strcpymax(inreply,(char *)getreply(head->line), 
                                  REPYSTRLEN);
                    }
                    else if (!strncasecmp(head->line, "References:", 11)) {
                        /*
                        ** Adding threading capability for the "References" 
                        ** header, ala RFC 822, used only for messages that 
                        ** have "References" but do not have an "In-reply-to"
                        ** field. This is partically a concession for Netscape's
                        ** email composer, which erroneously uses "References"
                        ** when it should use "In-reply-to". 
                        */
                        if (!*inreply)
                            strcpymax(inreply,(char *)getid(head->line), 
                                      REPYSTRLEN);
                    }
                }

                if (!headp)
                    headp=bp;

                savealternative = FALSE;
                
                for (head = headp; head; head=head->next) {
                  if(!head->header)
                    continue;

                    if (!strncasecmp( head->line, "Content-Type:", 13)) {
                       char *ptr=head->line+13;
                       char *boundary;
                       char type[129];
#define DISP_HREF 1
#define DISP_IMG  2
#define DISP_IGNORE 3
                       /* default is href to the attachment: */
                       char disposition=DISP_HREF;

                       while (isspace(*ptr))
                           ptr++;

                       sscanf(ptr, "%128[^;]", type);
                       if ((cp = strchr(type, '\n')) != NULL) 
                         *cp = '\0'; /* rm newlines */

                       if(alternativeparser) {
                         /* We are parsing alternatives... */

                         if(preferedcontent(type) ) {
                           /* ... this is a prefered type, we want to store
                              this [instead of the earlier one]. */
#if 0
                           struct body *next;
                           printf("%s is more fun than the previous one\n",
                                  type);
/*
** Not sure why this free section is here.
** It is causing purify to barf with massive numbers of
** "FMR: Free memory reads". When I commented it out it
** cleared up the problem with no associated memory leaked
** or difference in output. It's history for now.
*/ 
                           while(bp) {
                             next=bp->next;
                             if (bp->line) free(bp->line);
                             if (bp) free(bp);
                             bp=next;
                           }
#endif
                           headp = NULL;
                         }
                         else {
                           /* ...and this type is not a prefered one. Thus, we
                            * shall ignore it completely! */
                           disposition = DISP_IGNORE;
                         }
                       }
                       if((DISP_IGNORE==disposition) || ignorecontent(type)) {
                         /* don't save this */
                         disposition = DISP_IGNORE;
                         content = CONTENT_IGNORE;
                       }
                       else if (!strncasecmp(type, "text", 4) &&
                                 strncasecmp(type, "text/x-vcard", 12)) {
                          /* 
                          ** text follows, ignore this 
                          */
                          /* default is just plain 7/8 bit */
                          decode = ENCODE_NORMAL;

                          if (!strncasecmp(&type[4], "/html", 5))
                              content = CONTENT_HTML;
                          else
                              content = CONTENT_TEXT;
                          continue;
                       }
                       else if (!strncasecmp(type, "message/rfc822", 14)) {
                           /* 
                           ** Here comes an attached mail! This can be ugly, 
                           ** since the attached mail may very well itself 
                           ** contain attached binaries, or why not another 
                           ** attached mail? :-)
                           **
                           ** We need to store the current boundary separator 
                           ** in order to get it back when we're done parsing 
                           ** this particular mail, since each attached mail 
                           ** will have its own boundary separator that *might*
                           ** be used.
                           */
                           bp = (struct body *) addbody(bp, &lp,
                               "<P><STRONG>attached mail follows:</STRONG><HR>",
                               BODY_HTMLIZED | bodyflags);
                           bodyflags |= BODY_ATTACHED;
                           isinheader = 2;
                           continue;
                       }
                       else if (strncasecmp(type, "multipart/", 10)) {
                           /* 
                           ** This is not a multipart and not text 
                           */
                           struct body *fnamep=NULL; 
                           char acomment[256];      
                           char attachname[129]; /* listed attachment name */
                           char checkpath[256];  /* uniqueness path */
                           char *fname = NULL;    /* attachment filename */
                           char nameisuniq=FALSE; /* use the name included ?*/
                           char *file = NULL;
  
                           if ((fname = (char *)strstr(ptr, "name=")) == NULL) {
                               /* 
                               ** Name of file not specified in the
                               ** Content-Type header.  See if the 
                               ** Content-Disposition header exists and
                               ** contains the info.
                               */

                               for (fnamep = head;fnamep;fnamep=fnamep->next) {
                                   if(!fnamep->header)
                                      continue;
                                   if (!strncasecmp(fnamep->line,"Content-Disposition:", 20)) {
                                       if ((fname = (char *)strstr(fnamep->line, "filename=")) != NULL) {
                                           sscanf(fname+10, "%128[^\"]",attachname);
                                           fname = safe_filename(attachname);
                                       }
                                   }
                               }
                           }
                           else {
                               sscanf(fname+6, "%128[^\"]", attachname);
                               fname = safe_filename(attachname);
                           }

                           /*
                           ** If we found a name then lets check it out 
                           ** to see if we can use it.
                           */
                           if (fname) {
                               /*
                               ** Check if we can use this attached name 
                               ** for storing: 
                               */
                               if(strlen(fname) + strlen(dir) <= 254) {
                                 /* yes, we have room in our array */
                                 struct stat fileinfo;
                                 char alter[2]="";
                                 int counter='a';
                                 nameisuniq=TRUE;
                                 do {
                                   sprintf(checkpath,"%s%c%s%s",
                                           dir, PATH_SEPARATOR, alter, fname);
                                   /* loop while the file exist and try a few
                                      alternative file names before giving up */
                                   if(counter>='z') {
                                     nameisuniq=FALSE; /* we didn't find a unique file name */
                                     break;
                                   }
                                   sprintf(alter, "%c", counter++);
                                 } while(0 == lstat(checkpath, &fileinfo));
                               }
                           }

                           if(DISP_IGNORE != disposition) {
                             /* 
                             ** Saving of the attachments is being done inline
                             ** as they are encountered. The directories must 
                             ** exist first...
                             */
  
                             if(!nameisuniq)
                               /* get a random name */
                               binname = tmpname(dir, "bin");
                             else
                               binname = checkpath;

                             if (binname) {
                               binfile=open(binname, O_WRONLY|O_CREAT, filemode);

                               if(-1 != binfile) {
                                 chmod(binname, filemode);
                                 if(showprogress)
                                   printf("\nCreated attachment file %s mode %o\n", binname, filemode);
                               }

                               file=(char *)strrchr(binname, PATH_SEPARATOR);
                               if (file)
                                 file++; /* pass the separator */
                               else
                                 file=binname;
                             }

                             if( inlinecontent(type) ) {
                               /* if we know our browsers can show this
                                  type of context as-is, we make a <img> tag
                                  instead of <a href>! */
                               disposition = DISP_IMG;

                               sprintf(line, "%s<IMG SRC=\"%s\" ALT=\"%s\">\n",
                                       (showhr ? "<HR>\n" : ""), file,
                                       fname ? fname : "picture");
                             }
                             else {
                               sprintf(line, "%s<UL>\n<LI>%s attachment: <A HREF=\"%s\">%s</A>\n</UL>\n",
                                       (showhr ? "<HR>\n" : ""), type, 
                                       file ? file : "<void>", 
                                       fname ? fname : "stored");
  
                             }
                             /*
                             ** Print attachment comment before attachment
                             */
                             sprintf(acomment, "<!-- attachment=\"%.80s\" -->\n",
                                                   file);
                             bp = (struct body *) addbody(bp, &lp, acomment,
                                                      BODY_HTMLIZED|bodyflags);
                             bp = (struct body *) addbody(bp, &lp, line,
                                                      BODY_HTMLIZED|bodyflags);
                           }

                           if(DISP_IGNORE == disposition) {
                             content = CONTENT_IGNORE;
                           }
                           else {
                             if (binname && (binfile!=-1))
                               content = CONTENT_BINARY;
                             else
                               content = CONTENT_UNKNOWN;
                           }
                           if (!nameisuniq && binname)
                             free(binname);
            
                           continue;
                       }
                       else {
                         /*
                         ** Find the first boundary separator 
                         */
           
                         boundary=strcasestr(ptr, "boundary=");
              
                         if (boundary) {
                           boundary=(char *)strchr(boundary, '=');
                           if (boundary) {
                             boundary++;
                             while (isspace(*boundary))
                               boundary++;
                             if ('\"' == *boundary) {
                               sscanf(++boundary, "%[^\"]", boundbuffer);
                             }
                               else
                                 sscanf(boundary, "%s", boundbuffer);
                             boundary = boundbuffer;
                           }
                           
                           while (fgets(line, MAXLINE, fp)) {
                             if (!strncmp(line, "--", 2) &&
                                 !strncmp(line+2, boundbuffer, 
                                          strlen(boundbuffer))) {
                               break;
                             }
                           }
  
                           /* 
                           ** This stores the boundary string in a stack 
                           ** of strings: 
                           */
                           boundp = bound(boundp, boundbuffer);
  
                           /* printf("set new boundary: %s\n", boundp->line); */
  
                           /*
                           ** We set ourselves, "back in header" since there is
                           ** gonna come MIME headers now after the separator
                           */
                           isinheader = 1;

                           /* Daniel Stenberg started adding the
                            * "multipart/alternative" parser 13th of July
                            * 1998!  We check if this is a 'multipart/
                            * alternative' header, in which case we need to
                            * treat it very special.  
                            */

                           if(!strncasecmp(&ptr[10], "alternative", 11)) {
                              /* It *is* an alternative session!  Alternative
                              ** means there will be X parts with the same text
                              ** using different content-types. We are supposed
                              ** to take the most prefered format of the ones
                              ** used and only output that one. MIME defines
                              ** the order of the texts to start with pure text
                              ** and then continue with more and more obscure
                              ** formats. (well, it doesn't use those terms but
                              ** that's what it means! ;-)) 
                              */

                              /* How "we" are gonna deal with them:
                              **
                              ** We create a "spare" linked list body for the
                              ** very first part. Since the first part is
                              ** defined to be the most readable, we save that
                              ** in case no content-type present is prefered!
                              **
                              ** We skip all parts that are not prefered. All
                              ** prefered parts found will replace the first
                              ** one that is saved. When we reach the end of
                              ** the alternatives, we will use the last saved
                              ** one as prefered.
                              */

                             savealternative = TRUE;
#if 0
                             printf("SAVEALTERNATIVE: yes\n");
#endif           
                             
                           }

                         }
                         else
                           boundary = NULL;
                       }
                   }
                   else if (!strncasecmp(head->line, "Content-Transfer-Encoding:", 26)) {
                       char *ptr=head->line+26;
  
                       while (isspace(*ptr))
                           ptr++;
                       if (!strncasecmp(ptr, "QUOTED-PRINTABLE", 16)) {
                           decode = ENCODE_QP;
                       }
                       else if (!strncasecmp(ptr, "BASE64", 6)) {
                           decode = ENCODE_BASE64;
                       }
                       else if (!strncasecmp(ptr, "8BIT", 4)) {
                           decode = ENCODE_NORMAL;
                       }
                       else if (!strncasecmp(ptr, "7BIT", 4)) {
                           decode = ENCODE_NORMAL;
                       }
                       else if (!strncasecmp(ptr, "x-uue", 5)) {
                           decode = ENCODE_UUENCODE;
  
                           if (uudecode(fp, line, line, NULL, TRUE))
                               /*
                               ** oh gee, we failed this is chaos */
                               break;
                       }  
                       else {
                           /* this is an unknown format, we use default decoding */
                           char code[64];
  
                           sscanf(ptr, "%63s", code);
                           sprintf(line, " ('%s' encoding is not supported, stored as-is)\n", code);
  
                           bp = (struct body *) addbody(bp, &lp, line,
                                             BODY_HTMLIZED|bodyflags);
                      }  
#if 0
                       printf("DECODE set to %d\n", decode);
#endif
                  }
              }
              if (savealternative) {
                  /* let's remember 'bp' and 'lp' */

                  origbp=bp;
                  origlp=lp;

                  alternativeparser = TRUE;

                  /* restart on a new list: */
                  lp=bp=NULL;
              }
              headp = lp; /* start at this point next time */
          }
          else {
#if 0
              decodeRFC2047(line, MAXLINE);
#endif
              bp = (struct body *) addbody(bp, &lp, line, BODY_HEADER|bodyflags);
            }
        }
        else {
            if (!strncmp(line, "From ", 5) && (oldline[0]=='\n')) {
                if (readone)
                    continue;
                if (strstr(oldline, "Forwarded message:")) {
                    oldline[0] = '\0';
                    continue;
                }
                isinheader = 1;
                if (!hassubject)
                    strcpy(subject, NOSUBJECT);
                else
                    hassubject = 1;
  
                if (!hasdate)
                    strcpy(date, NODATE);
                else
                    hasdate = 1;
  
                if (inreply[0] == '\0')
                    oneunre(inreply, subject);
  
                while (rmlastlines(bp));
                addhash(num,date,name,email,msgid,subject,inreply,fromdate,bp);
                authorlist = (struct header *)
                            addheader(authorlist, num, name, subject, date, 1);
                subjectlist = (struct header *)
                            addheader(subjectlist,num,name,unre(subject),date,0);
                datelist = (struct header *)
                            addheader(datelist, num, name, subject, fromdate, 2);
                strcpymax(fromdate, (char *) getfromdate(line), DATESTRLEN);
            
                bp = NULL;
                num++;
                bodyflags=0; /* reset state flags */
     
                /* go back to default mode: */
                content = CONTENT_TEXT;
                decode = ENCODE_NORMAL;
                Mime_B = FALSE;
                headp = NULL; 
                multilinenoend = FALSE;

                if (!(num % 10) && showprogress && !readone) {
                    printf("\b\b\b\b%4d", num);
                    fflush(stdout);
                }
                inreply[0] = '\0';
#if 0
                printf("LAST: %s", line);
#endif
            }
            else { /* decode MIME complient gibberish */
                char newbuffer[MAXLINE];
                char *data;
                int datalen=-1; /* -1 means use strlen to get length */
    
                if (Mime_B) {
                    if (boundp &&
                        !strncmp(line, "--", 2) &&
                        !strncmp(line+2, boundp->line, strlen(boundp->line))) {
                      /* right at this point, we have another part coming up */
                        isinheader = 1; /* back on a kind-of-header */

#if 0
                        printf("hit %s\n", line);
#endif
                        if (!strncmp(line+2+strlen(boundp->line), "--", 2)) {
                            bp = (struct body *) addbody(bp, &lp, "\n",
                                                 BODY_HTMLIZED|bodyflags);
    

                            isinheader = 0; /* no header, the ending boundary
                                               can't have any describing
                                               headers */

#if 0
                            printf("End boundary %s\n", line);
#endif
                            boundp = bound(boundp, NULL);
                            if (!boundp) {
                                bodyflags &= ~BODY_ATTACHED;
                            }
                            if(alternativeparser) {
                              struct body *next;
                              /* we no longer have alternatives */
                              alternativeparser = FALSE;
#if 0
                              printf("We DUMP an old alternative\n");
#endif
                              while(bp) {
                                origbp = addbody(origbp, &origlp,
                                                 bp->line,
                                                 (bp->header?BODY_HEADER:0)|
                                                 (bp->html?BODY_HTMLIZED:0)|
                                                 (bp->attached?BODY_ATTACHED:0)
                                                 );
                                next= bp->next;
                                free(bp->line);
                                free(bp);
                                bp=next;
                              }
                              bp = origbp;
                              lp = origlp;

                              headp= NULL;
                            }
#if 0
                            if (boundp)
                                printf("back %s\n", boundp->line);
                            else
                                printf("back to NONE\n");
#endif
                        }
    
                        if (-1 != binfile) {
                            close(binfile);
                            binfile=-1;
                        }
                        continue;
                    }
                }
    
                switch ( decode ) {
                    case ENCODE_QP:
                         mdecodeQP(fp, line, &data, &datalen);
                         break;
                    case ENCODE_BASE64:
                         base64Decode(line, newbuffer, &datalen);
                         data = newbuffer;
                         break;
                    case ENCODE_UUENCODE:
                         uudecode(NULL, line, newbuffer, &datalen, FALSE);
                         data = newbuffer;
                         break;
                    case ENCODE_NORMAL:
                         data = line;
                         break;
                    default:
                         /* we have no clue! */
                         data = NULL;
                         break;
                }
#if 0
                printf("LINE %s\n", data);
#endif
                if (data) {
                    if ((content == CONTENT_TEXT) || (content==CONTENT_HTML)) {
                        if (decode > ENCODE_MULTILINED) {
                            /* 
                            ** This can be more than one resulting line, 
                            ** as the decoded the string may look like:
                            "#!/bin/sh\r\n\r\nhelp() {\r\n echo 'Usage: difftree"
                            */
                            char *p=data;
                            char *n;
                            char store;

#if 0
                            printf("decode type %d\n", decode);
#endif

                            while ((n = (char *)strchr(p, '\n'))) {
                                 store = n[1];
                                 n[1]=0;
#if 0
                                 printf("UNFOLDED %s", p);
#endif
                                 bp = (struct body *) addbody(bp, &lp, p,
                                             (multilinenoend?BODY_CONTINUE:0)|
                                             bodyflags);
                                 multilinenoend = FALSE; /* full line pushed */                                 n[1]=store;
                                 p = n+1;
                            }
                            if (strlen(p)) {
                                /* 
                                ** This line doesn't really end here, 
                                ** we will get another line soon that 
                                ** should get appended! 
                                */
#if 0
                                 printf("CONTINUE %s\n", p);
#endif
                                bp = (struct body *) addbody(bp, &lp, p,
                                             (multilinenoend?BODY_CONTINUE:0)|
                                             bodyflags);
    
                                /*
                                ** We want the next line to get appended to this!
                                */
                                multilinenoend = TRUE;
                            }
                        }
                        else {
                            bp = (struct body *) addbody(bp, &lp, data, 
                                           (content==CONTENT_HTML?
                                           BODY_HTMLIZED:0) | bodyflags );
                        }
#if 0
                        printf("ALIVE?\n");
#endif
                    }
                    else if (content == CONTENT_BINARY) {
                        if (-1 != binfile) {
                            if (datalen < 0)
                                datalen = strlen(data);
                
                            /*fwrite(data, datalen, 1, binfile); */
                            write(binfile, data, datalen);
                            /*bp = (struct body *) addbody(bp, "file contents");*/
                        }
                    }
    
                    if (ENCODE_QP == decode)
                        free(data); /* this was allocatd by mdecodeQP() */
                }
            }
        }
        strcpymax(oldline, line, MAXLINE);
    }

    if (!isinheader || readone) {
        if (!hassubject)
            strcpy(subject, NOSUBJECT);

        if (!hasdate)
            strcpy(date, NODATE);

        if (inreply[0] == '\0')
            oneunre(inreply, subject);

        while (rmlastlines(bp));

        addhash(num, date, name, email, msgid, subject, inreply, fromdate, bp);
        authorlist = (struct header *)
                      addheader(authorlist, num, name, subject, date, 1);
        subjectlist = (struct header *)
                      addheader(subjectlist, num, name, unre(subject), date, 0);
        datelist = (struct header *) 
                      addheader(datelist, num, name, subject, fromdate, 2);
        num++;
    }
    
    if (showprogress && !readone)
        printf("\b\b\b\b%4d articles.\n", num);
    
    if (!readone)
        bignum = num - 1;

    fclose(fp);

    crossindex();
    threadlist = NULL;
    printedthreadlist = NULL;
    crossindexthread1(datelist);
  
    /* can we clean up a bit please... */
  
    if (boundp != NULL) {
        if (boundp->line) 
            free(boundp->line);
        free(boundp);
    }
}

/*
** All this does is get all the relevant header information from the
** comment fields in existing archive files. Everything is loaded into
** structures in the exact same way as if articles were being read from
** stdin or a mailbox.
*/

void loadoldheaders(char *dir)
{
    FILE *fp;
    char name[NAMESTRLEN], email[MAILSTRLEN], date[DATESTRLEN],
        msgid[MSGDSTRLEN], subject[SUBJSTRLEN], inreply[REPYSTRLEN],
        line[MAXLINE], fromdate[DATESTRLEN], filename[MAXFILELEN];
    int num;

    struct body *bp=NULL;
    struct body *lp=NULL;

    num = 0;
    sprintf(filename, "%s%s%.4d.html", dir, 
              (dir[strlen(dir) - 1] == '/') ? "" : "/", num);

    bp = (struct body *) addbody(bp, &lp, "\0", 0);

    authorlist = subjectlist = datelist = NULL;

    if (showprogress)
        printf("Reading old headers...    ");

    /*
    ** WARNING: The following while loop is dependent on the order of the
    ** comments in the HTML files. The comments need to be in the order
    ** parsed below or this loop need to be change. (It really needs to be
    ** totally rewritten to be more robust and forgiving... Soon.
    ** 
    ** fromdate == <!-- received="Wed Jun  3 10:12:00 1998 CDT" -->
    ** date     == <!-- sent="Wed, 3 Jun 1998 10:12:07 -0500 (CDT)" -->
    ** name     == <!-- name="Kent Landfield" -->
    ** email    == <!-- email="kent@landfield.com" -->
    ** subject  == <!-- subject="Test message of the testmail mail address." -->
    ** msgid    == <!-- id="199806031512.KAA22323@landfield.com" -->
    ** inreply  == <!-- inreplyto="" -->
    */

    while ((fp = fopen(filename, "r")) != NULL) {

        /* skip all lines before the first <!-- */
        while (fgets(line,sizeof(line),fp), strncmp(line, "<!-- received", 13));

        strcpymax(fromdate, (char *) getvalue(line), DATESTRLEN);

        fgets(line, sizeof(line), fp);
        strcpymax(date, (char *) getvalue(line), DATESTRLEN);

        fgets(line, sizeof(line), fp);
        strcpymax(name, (char *) getvalue(line), NAMESTRLEN);

        fgets(line, sizeof(line), fp);
        strcpymax(email, (char *) getvalue(line), MAILSTRLEN);

        fgets(line, sizeof(line), fp);
        strcpymax(subject, (char *) unconvchars(getvalue(line)), SUBJSTRLEN);

        fgets(line, sizeof(line), fp);
        strcpymax(msgid, (char *) getvalue(line), MSGDSTRLEN);

        fgets(line, sizeof(line), fp);
        strcpymax(inreply, (char *) unconvchars(getvalue(line)), MSGDSTRLEN);

        fclose(fp);

        addhash(num, date, name, email, msgid, subject, inreply, fromdate, bp);
        authorlist = (struct header *) addheader(authorlist, num, name, subject, date, 1);
        subjectlist = (struct header *) addheader(subjectlist, num, name, unre(subject), date, 0);
        datelist = (struct header *)addheader(datelist,num,name,subject,fromdate,2);

        num++;
        if (!(num % 10) && showprogress) {
            printf("\b\b\b\b%4d", num);
            fflush(stdout);
        }

        sprintf(filename, "%s%s%.4d.html", dir,
                (dir[strlen(dir) - 1] == '/') ? "" : "/", num);
    }

    if (showprogress)
        printf("\b\b\b\b%4d articles.\n", num);

    if (!num)
        bignum = 0;
    else
        bignum = num;

    /* can we clean up a bit please... */
    if (bp != NULL) {
        if (bp->line) free(bp->line);
        free(bp);
    }
}

/*
** Adds a "Next:" link in the proper article, after the archive has been
** incrementally updated.
*/

void fixnextheader(char *dir, int num)
{
    char filename[MAXFILELEN], line[MAXLINE], name[NAMESTRLEN],
        email[MAILSTRLEN], subject[SUBJSTRLEN], inreply[REPYSTRLEN],
        date[DATESTRLEN], fromdate[DATESTRLEN], msgid[MSGDSTRLEN];
    struct body *bp, *cp, *dp, *status, *lp=NULL;
    int ul;
    FILE *fp;

    ul = 0;
    sprintf(filename, "%s%s%.4d.html", dir,
          (dir[strlen(dir) - 1] == '/') ? "" : "/", num);

    bp = NULL;
    if ((fp = fopen(filename, "r")) != NULL) {
        while ((fgets(line, MAXLINE, fp)) != NULL)
            bp = (struct body *) addbody(bp, &lp, line, 0);
    }
    else
        return;
    fclose(fp);

    cp = bp; /* save start of list to free later */

    if ((fp = fopen(filename, "w+")) != NULL) {
       while (bp != NULL) {
            fprintf(fp, "%s", bp->line);
            if (!strncmp(bp->line, "<!-- next=", 10)) {
                status = (struct body *) hashnumlookup(num + 1, name, 
                          email, subject, inreply, date, fromdate, msgid);
                if (status != NULL) {
                    if (usetable) {
                        dp = bp->next;
                        if (!strncmp(dp->line, "<UL>", 4)) {
                            fprintf(fp, "%s", dp->line);
                            ul = 1;
                        }
                    }
                    fprintf(fp, "<LI><STRONG>Next message:</STRONG> ");
                    fprintf(fp, "<A HREF=\"%.4d.html\">%s: \"%s\"</A>\n",
                                  num + 1, name, convchars(subject));
                    if (ul) {
                        bp = dp;
                        ul = 0;
                    }
                }
            }
            bp = bp->next;
       }
    }

    fclose(fp);

    /* can we clean up a bit please... */
    bp = cp;
    while (bp != NULL) {
        cp = bp->next;
        if (bp->line) free(bp->line);
        free(bp);
        bp = cp;
    }
}

/*
** Adds a "Reply:" link in the proper article, after the archive has been
** incrementally updated.
*/

void fixreplyheader(char *dir, int num)
{
    int replynum, subjmatch;
    char filename[MAXFILELEN], line[MAXLINE], name[NAMESTRLEN],
        email[MAILSTRLEN], subject[SUBJSTRLEN], inreply[REPYSTRLEN],
        date[DATESTRLEN], fromdate[DATESTRLEN], msgid[MSGDSTRLEN],
        name2[NAMESTRLEN], subject2[SUBJSTRLEN];
    struct body *bp, *cp, *status;
    struct body *lp = NULL;
    FILE *fp;

    status = (struct body *) hashnumlookup(num, name, email, subject, 
              inreply, date, fromdate, msgid);

    if (status == NULL || inreply[0] == '\0')
        return;

    if (inreply[0] != '\0') {
        replynum = hashreplylookup(inreply, name2, subject2, &subjmatch);
        if (replynum == -1)
            return;
    }

    sprintf(filename, "%s%s%.4d.html", dir, 
                       (dir[strlen(dir) - 1] == '/') ? "" : "/", replynum);

    bp = NULL;
    if ((fp = fopen(filename, "r")) != NULL) {
        while ((fgets(line, MAXLINE, fp)) != NULL)
            bp = (struct body *) addbody(bp, &lp, line, 0);
    }
    else
        return;
    fclose(fp);

    cp = bp;  /* save start of list to free later */

    if ((fp = fopen(filename, "w+")) != NULL) {
        while (bp != NULL) {
            if (!strncmp(bp->line, "<!-- reply", 10)) {
                fprintf(fp, "<LI><STRONG>Reply:</STRONG> ");
                fprintf(fp, "<A HREF=\"%.4d.html\">", num);
                fprintf(fp, "%s: \"%s\"</A>\n", name, convchars(subject));
            }
            fprintf(fp, "%s", bp->line);
            bp = bp->next;
        }
    }
    fclose(fp);

    /* can we clean up a bit please... */
    bp = cp;
    while (bp != NULL) {
        cp = bp->next;
        if (bp->line) free(bp->line);
        free(bp);
        bp = cp;
    }
}

/*
** Adds a "Next in thread:" link in the proper article, after the archive
** has been incrementally updated.
*/

void fixthreadheader(char *dir, int num)
{
    char filename[MAXFILELEN],line[MAXLINE],name[NAMESTRLEN];
    char subject[SUBJSTRLEN];
    FILE *fp;
    struct reply *rp;
    struct body *bp, *cp;
    struct body *lp=NULL;
    int threadnum = 0;

    for (rp = threadlist; rp != NULL; rp = rp->next) {
        if (rp->next != NULL && rp->next->msgnum == num && rp->msgnum != -1) {
            threadnum = rp->msgnum;
            strcpymax(name, rp->next->name, NAMESTRLEN);
            strcpymax(subject, rp->next->subject, SUBJSTRLEN);
            break;
        }
    }

    if (rp == NULL)
        return;

    sprintf(filename, "%s%s%.4d.html", dir,
    (dir[strlen(dir) - 1] == '/') ? "" : "/", threadnum);

    bp = NULL;
    if ((fp = fopen(filename, "r")) != NULL) {
        while ((fgets(line, MAXLINE, fp)) != NULL)
            bp = (struct body *) addbody(bp, &lp, line, 0);
    }
    else
        return;

    fclose(fp);

    cp = bp;  /* save start of list to free later */

    if ((fp = fopen(filename, "w+")) != NULL) {
        while (bp != NULL) {
            fprintf(fp, "%s", bp->line);
            if (!strncmp(bp->line, "<!-- nextthr", 12)) {
                fprintf(fp, "<LI><STRONG>Next in thread:</STRONG> ");
                fprintf(fp, "<A HREF=\"%.4d.html\">", num);
                fprintf(fp, "%s: \"%s\"</A>\n",
                name, convchars(subject));
            }
            bp = bp->next;
        }
    }
    fclose(fp);

    /* can we clean up a bit please... */
    bp = cp;
    while (bp != NULL) {
        cp = bp->next;
        if (bp->line) free(bp->line);
        free(bp);
        bp = cp;
    }
}
