/*
 * A tool (button) bar at the top of the window.
 */

#include <stdio.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/keysym.h>
#include <sys/wait.h>
#include <stdlib.h>

#include "arena.h"
#include "types.h"
#ifdef ARENA_DEBUG	/* QingLong.26-12-96 */
#  include "debug.h"
#endif
#include "bookmark.h"
#include "bridge.h"
#include "colour.h"
#include "display.h"
#include "editor.h"
#include "icon.h"
#include "image.h"
#include "main.h"
#include "parsehtml.h"
#include "spawn.h"
#include "status.h"
#include "style.h"
#include "toolbar.h"
#include "toolbar_types.h"
#include "toolbar_i.h"
#include "util.h"
#include "x11.h"
#include "xpm_arena.h"

#include "HTUtils.h"	/* WWW general purpose macros */
#include "HTList.h"
#include "HTAccess.h"
#ifdef TOOLBAR_ICONS
#  include "HTFormat.h"
#endif


#ifdef TOOLBAR_ICONS
#  include "abort.xpm"
#  include "back_reload_forward.xpm"
#  ifdef BOOKMARKS
#    include "bookmarx.xpm"
#  endif
#  ifdef CLONE
#    include "clone.xpm"
#  endif
#  include "edit.xpm"
#  ifdef FIND
#    include "find.xpm"
#  endif
#  include "help.xpm"
#  include "home.xpm"
#  include "open.xpm"
#  include "print.xpm"
#  include "quit.xpm"
#  include "save.xpm"
#  include "view.xpm"
#  ifdef ZOOM
#    include "zoom.xpm"
#  endif
#endif


Window ToolBarWin = None;
GC     ToolBarGC  = NULL;


static XFontStruct* pToolFontInfo = NULL;
static XRectangle ArenaToolBar = {0, 0, 0, 0};
static unsigned int ArenaMainToolBarHeight = 0;

static Button* buttons; /* typedef in types.h */
static int nButtons = 0;
static int nButtonDown;


#ifdef TOOLBAR_ICONS
ButtonData buttonData[] =
{
 {"Abort",             Arena_Abort_button,             NULL},
 {"Quit",              Arena_Quit_button,              NULL},
 {"Home",              Arena_Home_button,              NULL},
 {"Open",              Arena_Open_button,              NULL},
 {"BackReloadForward", Arena_BackReloadForward_button, NULL},
 {"SaveAs",            Arena_Save_button,              NULL},
 {"View",              Arena_View_button,              NULL},
 {"Edit",              Arena_Edit_button,              NULL},
 {"Print",             Arena_Print_button,             NULL},
#ifdef ZOOM
 {"Zoom",              Arena_Zoom_button,              NULL},
#endif
#ifdef BOOKMARKS
 {"BookX",             Arena_BookMarx_button,          NULL},
#endif
#ifdef FIND
 {"Find",              Arena_Find_button,              NULL},
#endif
#ifdef CLONE
 {"Clone",             Arena_Clone_button,             NULL},
#endif
 {"Help",              Arena_Help_button,              NULL},
 { NULL, NULL, NULL}	/* This ``NULL'' entry is really necessary! */
};
# else
ButtonData buttonData[] =
{
 {"Abort",             NULL, NULL},
 {"Quit",              NULL, NULL},
 {"Home",              NULL, NULL},
 {"Open",              NULL, NULL},
#if 1
 {"<<",                NULL, NULL},
 {"||",                NULL, NULL},
 {">>",                NULL, NULL},
#else
 {"Back",              NULL, NULL},
 {"Reload",            NULL, NULL},
 {"Forward",           NULL, NULL},
#endif
 {"Save",              NULL, NULL},
 {"View",              NULL, NULL},
 {"Edit",              NULL, NULL},
 {"Print",             NULL, NULL},
#ifdef ZOOM
 {"Zoom",              NULL, NULL},
#endif
#ifdef BOOKMARKS
 {"BookX",             NULL, NULL},
#endif
#ifdef FIND
 {"Find",              NULL, NULL},
#endif
#ifdef CLONE
 {"Clone",             NULL, NULL},
#endif
#if 1
 {"?",                 NULL, NULL},
#else
 {"Help",              NULL, NULL},
#endif
 { NULL, NULL, NULL}	/* This ``NULL'' entry is really necessary! */
};
#endif


#ifdef ARENA_FLYBY_HELP
/* The button flyby help has been implemented "somewhat" indepenently of
 * the button data above on purpose.  To isolate flyby and i10n via gettext().
 * However, if you DON'T want ANY flyby stuff, it will not be included
 * in the program!  By default, I have implemented it as "in" with "hint"
 * display... Then if you don't want hints, you only have to specify
 * --flyby none on the command line!  You may alter arena.h.in to default to
 * none, hint, terse or verbose text strings... or NONE even compiled in!
 */

/* We are specifying:
 * Button Name, Counter, Hint, Terse Hint, and  Verbose Hint.
 * The button name MUST match one of the buttonData[].name values above.
 * The counter is incremented each time that particular button has a hint
 * displayed.  It is then used to "slow down" the hint timer for THAT button
 * in the future!  The gettext_noop() is used to "mark" text for i10n ing.
 * The _(string) macro (GNU style) is used instead of gettext();
 * The N_(string) macro (again, GNU) is used insted of gettext_noop().
 */
FlyByButtonData FlyByData[] =
{
 {"Abort",             0,
  N_("Abort"),
  N_("Abort operation"),
  NULL},
 {"Quit",              0,
  N_("Quit"),
  N_("Exit Arena"),
  NULL},
 {"Home",              0,
  N_("Home"),
  N_("Go to Home URL"),
  NULL},
 {"Open",              0,
  N_("Open"),
  N_("Open URL"),
  NULL},
 {"<<",                0,
  N_("Back"),
  N_("Back in history"),
  NULL},
 {"||",                0,
  N_("Reload"),
  N_("Reload document"),
  NULL},
 {">>",                0,
  N_("Forward"),
  N_("Forward in history"),
  NULL},
 {"Back",              0,
  N_("Back"),
  N_("Back in history"),
  NULL},
 {"Reload",            0,
  N_("Reload"),
  N_("Reload document"),
  NULL},
 {"Forward",           0,
  N_("Forward"),
  N_("Forward in history"),
  NULL},
 {"BackReloadForward", 0,
  N_("Back/Reload/Forward"),
  N_("Back/Reload/Forward"),
  N_("Current History\n"
     " L) Back one\n"
     " M) Reload\n"
     " R) Forward one") },
 {"SaveAs",            0,
  N_("SaveAs"),
  N_("Save document"),
  NULL},
 {"Save",              0,
  N_("Save"),
  N_("Save document"),
  NULL},
 {"View",              0,
  N_("View"),
  N_("Toggle view"),
  NULL},
 {"Edit",              0,
  N_("Edit"),
  N_("Edit document"),
  NULL},
 {"Print",             0,
  N_("Print"),
  N_("Print document"),
  NULL},
#ifdef ZOOM
 {"Zoom",              0,
  N_("Zoom"),
  N_("Zoom In/Out"),
  N_("Zoom\n"
     " L) Increase magnification\n"
     " R) Decrease magnification") },
#endif
#ifdef BOOKMARKS
 {"BookX",             0,
  N_("Bookmarks"),
  N_("Bookmarks\n"
     "Current/Index/Add"),
  N_("Bookmarks\n"
     " L) Current folder\n"
     " M) Index folder\n"
     " R) Add URL to folder") },
#endif
#ifdef FIND
 {"Find",              0,
  N_("Find"),
  NULL,
  NULL},
#endif
#ifdef CLONE
 {"Clone",             0, NULL,
  N_("Clone"),
  NULL,
  NULL},
#endif
 {"?",                 0,
  N_("Help"),
  N_("Help URL"),
  N_("Help URL\n"
     " "HELP_URL"\n") },
 {"Help",              0,
  N_("Help"),
  N_("Help URL"),
  N_("Help URL\n"
     " "HELP_URL"\n") },

 { NULL, 0, NULL, NULL}	/* This ``NULL'' entry is really necessary! */
};
#endif /* ARENA_FLYBY_HELP  */






/* Initialize toolbar data structure.
 * Uses libwww. Should be called after libwww initialization.
 */
Bool ToolBarDataInit(void)
{
#ifdef TOOLBAR_ICONS
 extern HTFormat xpm_atom;
#endif
 extern ButtonData buttonData[];
 extern int nButtons;
 ButtonData* buttonD_i;


 nButtons = 0;

#ifdef TOOLBAR_ICONS
 if (xpm_atom)
   {
    for (buttonD_i = buttonData; buttonD_i->name; buttonD_i++)
      {
       buttonD_i->icon_type = xpm_atom;
       nButtons++;
      }

    return True;
   }
  else
   {
    for (buttonD_i = buttonData; buttonD_i->name; buttonD_i++) nButtons++;
    return False;
   }
# else
 for (buttonD_i = buttonData; buttonD_i->name; buttonD_i++) nButtons++;
 return True;
#endif
}


Bool SetToolBarButtonPosition(int theButtonNumber, int x, int y)
{
 if (0 <= theButtonNumber && theButtonNumber < nButtons)
   {
    buttons[theButtonNumber].x = x;
    buttons[theButtonNumber].y = y;
    return True;
   }
  else
   return False;
}


#if 0	/* Not used at the moment. Kept for future use.   QingLong.23-07-97 */
static Bool SetToolBarButtonSize(int theButtonNumber,
				 unsigned int w, unsigned int h)
{
 if (0 <= theButtonNumber && theButtonNumber < nButtons)
   {
    buttons[theButtonNumber].w = w;
    buttons[theButtonNumber].h = h;
    return True;
   }
  else
   return False;
}
#endif


XRectangle GetToolBarButtonGeometry(int theButtonNumber)
{
 XRectangle theXrectangle;


 if (0 <= theButtonNumber && theButtonNumber < nButtons)
   {
    theXrectangle.x      = buttons[theButtonNumber].x;
    theXrectangle.y      = buttons[theButtonNumber].y;
    theXrectangle.width  = buttons[theButtonNumber].w;
    theXrectangle.height = buttons[theButtonNumber].h;
   }
  else
   {
    theXrectangle.x      = -1;
    theXrectangle.y      = -1;
    theXrectangle.width  =  0;
    theXrectangle.height =  0;
   }

 return theXrectangle;
}


/*
 * Be careful! DisCERN GetToolBarGeometry() and GetToolBarButtonGeometry() !
 */


static Bool updateArenaStatusLineGeometry(void)
{
 extern XRectangle ArenaToolBar;
 XRectangle proposedArenaStatusLine;


 proposedArenaStatusLine.x      = (buttons[0].x + buttons[0].w +
                                   ARENA_TOOLBAR_X_GAP) +
                                  (ARENA_STATUS_X_BORDER_WIDTH +
                                   ARENA_STATUS_OUTSET_WIDTH);
 proposedArenaStatusLine.y      = ArenaMainToolBarHeight +
                                  (ARENA_STATUS_Y_BORDER_WIDTH +
				   ARENA_STATUS_OUTSET_WIDTH +
				   ARENA_STATUS_OUTSET_WIDTH);
 proposedArenaStatusLine.width  = max((ArenaToolBar.width -
				       (proposedArenaStatusLine.x +
					ARENA_STATUS_OUTSET_WIDTH +
					ARENA_STATUS_X_BORDER_WIDTH)),
				      1);
 proposedArenaStatusLine.height = max(((int)buttons[0].h -
				       (ARENA_STATUS_OUTSET_WIDTH +
					ARENA_STATUS_OUTSET_WIDTH)),
				      1);

 return SetArenaStatusLineGeometry(proposedArenaStatusLine);
}


/*
 * Updates toolbar geometry (actually only the width is currently updated).
 */
Bool UpdateToolBarGeometry(void)
{
 extern XRectangle ArenaToolBar;
 extern unsigned int ArenaMainToolBarHeight;

 ArenaToolBar.width = win_width;
 {
  int theWidth = ArenaIconBarWidth();

  if (theWidth >= 0) ArenaToolBar.width -= theWidth;
 }

 if (updateArenaStatusLineGeometry())
   {
    XRectangle currentArenaStatusLine = GetArenaStatusLineGeometry();

    buttons[0].y = currentArenaStatusLine.y +
                   ((int)currentArenaStatusLine.height - (int)buttons[0].h)/2;

    ArenaToolBar.height = currentArenaStatusLine.y +
                          currentArenaStatusLine.height +
                          (currentArenaStatusLine.y - ArenaMainToolBarHeight);
    return True;
   }
  else
   return False;
}


XRectangle GetToolBarGeometry(void)
{
 extern XRectangle ArenaToolBar;

 return ArenaToolBar;
}


void SetToolBarFont(XFontStruct* pf)
{
 extern XFontStruct* pToolFontInfo;

 pToolFontInfo = pf;
}


/*
 * Initialize Arena toolbar X stuff: create window and graphic context.
 */
Bool ToolBarInit(Window theWindow)
{
 extern XFontStruct* pToolFontInfo;
 extern unsigned int ArenaMainToolBarHeight;
 int i;


 if (pToolFontInfo == NULL) return False;

 ToolBarDataInit();

 if (buttons == NULL)
   {
    buttons = (Button *)Arena_MAlloc(nButtons * sizeof(Button), False);

    for (i = 0; i < nButtons; ++i)
      {
       buttons[i].label = buttonData[i].name;

       if (buttonData[i].icon && (buttonData[i].icon_type == xpm_atom))
	 {
	  buttons[i].icon = NewImage();
	  if (buttons[i].icon)
	    {
	     buttons[i].icon->url  = NULL;
	     buttons[i].icon->next = NULL;
	     buttons[i].icon->npixels = 0;

	     {
	      ImageXyoke* theButtonImageXcouple;

	      theButtonImageXcouple = ProcessXPM(buttonData[i].icon, depth);

	      if (theButtonImageXcouple)
		{
		 XprocessLoadedImage(theButtonImageXcouple, buttons[i].icon);
		 FreeImageXyoke(theButtonImageXcouple);
		}
	     }

	     buttons[i].w = 6 + buttons[i].icon->width;
	     buttons[i].h = 6 + buttons[i].icon->height;
	    }
	 }
        else
	 {
	  buttons[i].icon = NULL;
	  buttons[i].w = 6 + XTextWidth(pToolFontInfo,
					buttons[i].label,
					Arena_StrLen(buttons[i].label));
	  buttons[i].h = 6 + (pToolFontInfo->max_bounds.ascent +
			      pToolFontInfo->max_bounds.descent);
	 }
      }
    /* End ``for (i = 0; i < nButtons; ++i)'' */

    /* Compute the main toolbar height.
     * (Do not take `Abort' button into account)
     */
    {
     int tbh = (pToolFontInfo->max_bounds.ascent +
		pToolFontInfo->max_bounds.descent);

     for (i = 1; i < nButtons; i++) if (tbh < buttons[i].h) tbh = buttons[i].h;
     ArenaMainToolBarHeight = tbh + (ARENA_TOOLBAR_Y_BORDER_WIDTH +
				     ARENA_TOOLBAR_OUTSET_WIDTH) * 2;
    }

    /*
     * Now place the toolbar buttons. Abort button is set up separetely.
     */
    {
     int x = ARENA_TOOLBAR_X_BORDER_WIDTH + ARENA_TOOLBAR_OUTSET_WIDTH;

     buttons[0].x = x; /* Y will be set after SetArenaStatusLineGeometry(). */

     for (i = 1; i < nButtons; ++i)
       {
	buttons[i].x = x;
	buttons[i].y = (ArenaMainToolBarHeight - buttons[i].h)/2;
	x += ARENA_TOOLBAR_X_GAP + buttons[i].w;
       }
    }

    /*
     * Compute toolbar geometry (ArenaToolBar XRectangle structure).
     */
    ArenaToolBar.x = 0;
    ArenaToolBar.y = 0;
   }

 /*
  * UpdateToolBarGeometry() updates (sets) `Abort' button Y position.
  */
 if (!UpdateToolBarGeometry()) return False;

 /*
  * Now create toolbar subwindow.
  */
 if (theWindow != None && ToolBarWin == None)
   {
    ToolBarWin = CreateSubWindow(display, theWindow,
				 ArenaToolBar.x,     ArenaToolBar.y,
				 ArenaToolBar.width, ArenaToolBar.height,
				 0, windowColour);
   }

 if (ToolBarWin != None && ToolBarGC == NULL)
   ToolBarGC = XCreateGC(display, ToolBarWin, 0, NULL);

 if (ToolBarGC)
   {
    XSetFont(display, ToolBarGC, pToolFontInfo->fid);
    return (ToolBarWin != None);
   }
  else
   return False;
}


Bool DrawToolBarButton(Arena_ButtonState theButtonState, int buttonNumber)
{
 if (buttons)
   {
    if (0 <= buttonNumber && buttonNumber < nButtons)
      {
       int x, y, r;
       unsigned int w, h;
       char* s;


       r = pToolFontInfo->max_bounds.ascent + 2;

       x = buttons[buttonNumber].x;
       y = buttons[buttonNumber].y;
       w = buttons[buttonNumber].w;
       h = buttons[buttonNumber].h;
       s = buttons[buttonNumber].label;

       XSetForeground(display, ToolBarGC, labelColour);

       switch (theButtonState)
	 {
	  case Arena_Button_UP:
	    DrawOutSet(ToolBarWin, ToolBarGC, x, y, w, h);
	    break;

	  case Arena_Button_DOWN:
	    DrawInSet(ToolBarWin, ToolBarGC, x, y, w, h);
	    break;
	 }

       if (buttons[buttonNumber].icon)
	 {
	  Image* theButtonIconImage = buttons[buttonNumber].icon;


	  Arena_PutTransparentImage(display, ToolBarGC,
				    theButtonIconImage->pixmap,
				    theButtonIconImage->mask,
				    ToolBarWin,
				    x + (w - theButtonIconImage->width )/2,
				    y + (h - theButtonIconImage->height)/2,
				    theButtonIconImage->width,
				    theButtonIconImage->height);
	 }
        else
	 XDrawString(display, ToolBarWin, ToolBarGC, x + 3, y + r, s,
		     Arena_StrLen(s));

       return True;
      }
     else
      return False;
   }
  else
   return False;
}


void DisplayToolBar(void)
{
 if (UpdateToolBarGeometry())
   {
    int i;


    XSetClipMask(display, ToolBarGC, None);
    XClearWindow(display, ToolBarWin);

    DrawOutSet(ToolBarWin, ToolBarGC,
	       0, 0, ArenaToolBar.width, ArenaMainToolBarHeight);

    for (i = 0; i < nButtons; i++) DrawToolBarButton(Arena_Button_UP, i);
   }
}


void PaintLabel(char* s, unsigned long theColour)
{
 int s_l, x, y, w, h;
 XRectangle theLabelXrectangle;


 if (s)
   {
    theLabelXrectangle.x      = x = buttons[nButtons-1].x +
                                    buttons[nButtons-1].w +
                                    ARENA_TOOLBAR_X_BORDER_WIDTH;
    theLabelXrectangle.y      = y = ArenaToolBar.y +
                                    ARENA_TOOLBAR_OUTSET_WIDTH;
    theLabelXrectangle.width  = w = ArenaToolBar.width - x -
                                    ARENA_TOOLBAR_OUTSET_WIDTH * 2;
    theLabelXrectangle.height = h = ArenaMainToolBarHeight -
                                    ARENA_TOOLBAR_OUTSET_WIDTH * 2;

    if ((w > 0) && (h > 0))
      {
       XSetClipRectangles(display, ToolBarGC,
			  0, 0, &theLabelXrectangle, 1, Unsorted);

       XSetForeground(display, ToolBarGC, windowColour);
       XFillRectangle(display, ToolBarWin, ToolBarGC, x, y, w, h);

       s_l = Arena_StrLen(s);
       x += (w - XTextWidth(pToolFontInfo, s, s_l))/2;
       y += (h + (pToolFontInfo->max_bounds.ascent -
		  pToolFontInfo->max_bounds.descent))/2;
       XSetForeground(display, ToolBarGC, theColour);
       XDrawString(display, ToolBarWin, ToolBarGC, x, y, s, s_l);

       XSetClipMask(display, ToolBarGC, None);
      }
   }
}


void PaintVersion(void)
{
 PaintLabel(ParseHTMLerror ?  _("Bad HTML") : ARENA_VERSION,
	    ParseHTMLerror ? strikeColour : labelColour);
}


void DisplayStatusLine_withLabel(char* theLabel)
{
 if (UpdateToolBarGeometry())
   {
    XRectangle currentArenaStatusLine = GetArenaStatusLineGeometry();

    DrawOutSet(ToolBarWin, ToolBarGC,
	       0, ArenaMainToolBarHeight,
	       ArenaToolBar.width,
	       ArenaToolBar.height - ArenaMainToolBarHeight);

    XSetClipMask(display, StatusGC, None);
    XClearWindow(display, StatusWin);

    DrawToolBarButton(AbortButtonUp ? Arena_Button_UP : Arena_Button_DOWN, 0);
    AbortButtonChanged = 0;  /* avoid flickering buffer */

    DrawInSet(ToolBarWin, ToolBarGC,
	      currentArenaStatusLine.x - ARENA_STATUS_OUTSET_WIDTH,
	      currentArenaStatusLine.y - ARENA_STATUS_OUTSET_WIDTH,
	      currentArenaStatusLine.width  + ARENA_STATUS_OUTSET_WIDTH * 2,
	      currentArenaStatusLine.height + ARENA_STATUS_OUTSET_WIDTH * 2);

    XDrawArenaStatus();
    if (theLabel) PaintLabel(theLabel, labelColour);
   }
}


#ifdef ARENA_FLYBY_HELP
/* ToolBarWhichButton    glk.10.25.97
 * Find toolbar button that contains the point x,y.
 * Zero means... NO button at that location!
 * Please Note:  This includes the Abort button on the status bar!
 */
int ToolBarWhichButton(int x, int y)
{
 int i;

 for (i = 0; i < nButtons; i++)
   {
    if (buttons[i].x <= x && x < buttons[i].x + buttons[i].w &&
	buttons[i].y <= y && y < buttons[i].y + buttons[i].h)
      {
       return (i + 1);
      }
   }

 return 0;
}

/*
 * Get the "correct" flyby text.
 * If FlyByHint is 0, NO HINTS ARE GENERATED!
 * If FlyByHint is 1, then use the hint if available, otherwise button name.
 * If FlyByHint is 2, try to give back the most verbose text available.
 */
char *ToolBarFlyByText(int j)
{
 extern int FlyByHints;

 static char *unknown = N_("Unknown Hint");

 int len;
 int i = 0;
 char *label;

 j--;   /* we found it with WhichButton() above, so offset it */
 if (j < 0 || j >= nButtons) return _(unknown);

 label = buttons[j].label;
 len = Arena_StrLen(label);
 if (len <= 0) return _(unknown);

 while( FlyByData[i].name != NULL)
   {
    if (strncasecmp(label, FlyByData[i].name, len) == 0)
      {
       char *r = NULL;
       FlyByButtonData *b = &FlyByData[i];

       switch (FlyByHints)
	 {
	 case ARENA_FLYBY_NONE:
	 case ARENA_FLYBY_HINT:
	   r = b->hint ? b->hint : b->name;
	   break;

	 default:
	 case ARENA_FLYBY_TERSE:
	   r = b->terse ? b->terse : b->hint;
	   break;

	 case ARENA_FLYBY_VERBOSE:
	   r = b->verbose ? b->verbose : b->terse;
	   r = r ? r : b->hint;
	   break;
	 }
       return r ? _(r) : _(b->name);
      }
    i++;
   }
 return _(unknown);
}
#endif /* ARENA_FLYBY_HELP */


int ToolBarButtonDown(unsigned int MBp, int x, int y)
{
 int i = ToolBarWhichButton(x, y);


 if (i > 0 && (i == 1 || !Busy()))
   {
    nButtonDown = i;
    i--;
    DrawInSet(ToolBarWin, ToolBarGC,
	      buttons[i].x, buttons[i].y, buttons[i].w, buttons[i].h);
    XFlush(display);
    return TOOLBAR;
   }
  else
   {
    if (i > 1) Beep();
    nButtonDown = 0;  /* 0 implies no button down */
    return VOID;
   }
}


void ToolBarButtonUp(unsigned int MBp, int x, int y)
{
 int i = ToolBarWhichButton(x, y);
 int x1, y1;


 if (nButtonDown)
   {
    XRectangle rect;
    unsigned int w, h;

    nButtonDown--;

    rect.x      = x1 = buttons[nButtonDown].x;
    rect.y      = y1 = buttons[nButtonDown].y;
    rect.width  = w  = buttons[nButtonDown].w;
    rect.height = h  = buttons[nButtonDown].h;

    nButtonDown++;

    DrawOutSet(ToolBarWin, ToolBarGC, x1, y1, w, h);
    XFlush(display);
   }

 if (i == nButtonDown)
   {
    i--;
    if (i == 0)
      {
       AbortFlag = True;
      }
     else
      {
       if (strncasecmp(buttons[i].label, "quit", 4) == 0)
	 {
	  XCloseDisplay(display);
	  Exit(0);
	 }
        else
	 if (SpawnIsWaitGui())
	   {
	   /* Don't allow any buttons to work "after" this test!
	    * If child running and we are IN GUI wait mode, only buttons
	    * above this test will work!
	    */
	    nButtonDown = 0;
	    Beep();
	   }
	  else
	   if (strncasecmp(buttons[i].label, "open", 4) == 0)
	     OpenDoc(NULL);
#ifdef TOOLBAR_ICONS
	    else
	     if (strncasecmp(buttons[i].label, "backreloadforward", 17) == 0)
	       {
		if (MBp == Button1)
		  BackDoc(DocDispGC, GetDefaultDocViewWindow());
		 else
		  if (MBp == Button2)
		    ReloadDoc(CurrentDoc->url);
		   else
		    if (MBp == Button3)
		      ForwardDoc(DocDispGC, GetDefaultDocViewWindow());
	       }
#else
	      else
	       if (strncasecmp(buttons[i].label, "<<", 2) == 0 ||
		   strncasecmp(buttons[i].label, "back", 4) == 0)
		 BackDoc();
	        else
		 if (strncasecmp(buttons[i].label, "||", 2) == 0 ||
		     strncasecmp(buttons[i].label, "reload", 6) == 0)
		   ReloadDoc(CurrentDoc->url);
		  else
		   if (strncasecmp(buttons[i].label, ">>", 2) == 0 ||
		       strncasecmp(buttons[i].label, "forward", 7) == 0)
		     ForwardDoc(DocDispGC, GetDefaultDocViewWindow());
#endif
		    else
		     if (strncasecmp(buttons[i].label, "save", 4) == 0)
		       SaveDoc(NULL);
		      else
		       if (strncasecmp(buttons[i].label, "print", 5) == 0)
			 PrintDoc();
		        else
			 if (strncasecmp(buttons[i].label, "view", 4) == 0)
			   ToggleView(CurrentDoc);
			  else
			   if (strncasecmp(buttons[i].label, "edit", 4) == 0)
			     EditCurrentBuffer();
			    else
			     if (strncasecmp(buttons[i].label, "home", 4) == 0)
			       HomeDoc(DocDispGC, GetDefaultDocViewWindow());
#ifdef ZOOM
			      else
			       if (strncasecmp(buttons[i].label, "zoom", 4)
				   == 0)
				 {
				  if (MBp == Button1)
				    StyleZoomChange(CurrentDoc, 1.25);
				   else
				    if (MBp == Button3)
				      StyleZoomChange(CurrentDoc, 0.8);
				 }
#endif
#ifdef BOOKMARKS
			        else
				 if (strncasecmp(buttons[i].label, "bookx", 5)
				     == 0)
				   BMProcessButtonUp(MBp);
#endif
#ifdef FIND
				  else
				   if (strncasecmp(buttons[i].label, "find", 4)
				       == 0)
				     FindString(theString);
#endif
#ifdef CLONE
				    else
				     if (strncasecmp(buttons[i].label, "clone",
						     5)
					 == 0)
				       CloneSelf();
#endif
				      else
				       if (strncasecmp(buttons[i].label,
						       "?", 1)
					   == 0 ||
					   strncasecmp(buttons[i].label,
						       "help", 4)
					   == 0)
					 OpenDoc(HelpURL);
      }
   }

 nButtonDown = 0;
}


/* PrintDoc
 * Spawn a child in ASYNC mode.
 * Pump current doc's buffer through the pipe to the child.
 * NO TEMPORARY Files necessary! glk 11.10.97
 */
void PrintDoc(void)
{
 static int PrintBlocker = 0;

 char *p;
 int c;
 int argc = 0;
 char **argv = NULL;
 FILE *fp;
 SPAWNChildInfo *child = NULL;

#ifdef ARENA_DEBUG
 char Iam[] = "PrintDoc";
#endif

 /* See if we're trying to come back in here before we get OUT!
  * Well, sort of anyway.
  */
 if (SpawnIsBlocked(PrintBlocker))
   {
    Beep();
    return;
   }
 SpawnBlock(&PrintBlocker);

 /* Turn command line into argc/argv
  * and proceed if looks valid
  */
 if (Printer != NULL)
    argv = SpawnStringToRagged(Printer, &argc);

 if (Printer == NULL || argv == NULL || argc == 0)
   {
    Announce(_("No printer specified"));
    XUndefineCursor(display, win);
    XFlush(display);
    SpawnFreeRagged(&argc, &argv);
    SpawnUnBlock(&PrintBlocker);
    return;
   }

 /* NOTICE: we are spawning in ASYNC mode
  * And that there is an Input Pipe file descriptor specified for the child.
  * We will dump the buffer to our end of the pipe... returned in fp!
  */
 Announce("%s %s ...", _("Printing"), CurrentDoc->url ? CurrentDoc->url : "?");
 child = SpawnChild(SpawnExecvp, argv[0], argv, NULL,
		    WAIT_ASYNC, KILL_ALWAYS,
		    NULL, &fp,
		    NULL, NULL,
		    NULL, NULL,
		    NULL);

 if (!SpawnIsChildAlive(child))
   {
#ifdef ARENA_DEBUG
    Arena_TracePrint(Iam, " child not there!\n");
#endif
    Announce(_("No printing available ..."));
    XUndefineCursor(display, win);
    XFlush(display);
    SpawnFreeRagged(&argc, &argv);
    SpawnUnBlock(&PrintBlocker);
    return;
   }

 /* Dump the buffer! */
 p = buffer;
 while ((c = *p++)) putc(c, fp);
 fclose(fp);

 /* Now, wait here until the child is completely done... This may someday be
  * removed to allow total async mode, but then we have to set too many
  * blocks!  Notice: We do NOT have to ForgetChild() here... ASYNC mode
  * at creation will cause SIGCHLDHandler() to do the Forget for us!
  */
 SpawnWaitForChild(child, WAIT_GUI);

 /* We're done, go back! */
 Announce("%s", CurrentDoc->url);
 XUndefineCursor(display, win);
 XFlush(display);
 SpawnFreeRagged(&argc, &argv);
 SpawnUnBlock(&PrintBlocker);
}


#if 0	/* glk 13.10.97 We don't do postscript! */
/*
 * Save and display ps document.
 */
void DisplayPSdocument(char *buffer, long length, char *path)
{
 FILE *fp;
 char *tmpfile, *p, buf[256], cmd[256];

 p = ARENA_rIndex(path, '/');

 if (p)
   {
    sprintf(buf, "/tmp%s", p);
    tmpfile = buf;
   }
  else
   tmpfile = tempnam(".", "w3");

 p = buffer;

 if ((fp = fopen(tmpfile, "w")) == 0)
   Warn("Can't open temporary postscript: file, %s\n", tmpfile);
  else
   {
    while (length-- > 0) putc(*p++, fp);

    fclose(fp);

    sprintf(cmd, "gs %s &", tmpfile);

    system(cmd); /* W3C LAMERS!!!  This routine #if 0ed */
    /* Announce("Using gs for %s", NewDoc.url);*/
   }

#if 0
 XUndefineCursor(display, win);
 XFlush(display);
#endif
}
#endif


#if 0	/* QingLong.25-05-97 */
/*
 * Save and display document in separate window.
 */
void DisplayExtDocument(char *buffer, long length, int type, char *path)
{
 FILE *fp;
 char *tmpfile, *p, *app, cmd[256], buf[256];
#ifdef ARENA_DEBUG
 char Iam[] = "DisplayExtDocument";
#endif


 p = ARENA_rIndex(path, '/');

 if (p)
   {
    sprintf(buf, "/tmp%s", p);
    tmpfile = buf;
   }
  else
   tmpfile = tempnam(".", "w3");

 p = buffer;

 if ((fp = fopen(tmpfile, "w")) == 0)
   Warn("Can't open temporary file: %s\n", tmpfile);
  else
   {
    while (length-- > 0) putc(*p++, fp);

    fclose(fp);

    if (type == DVIDOCUMENT)
      app = "xdvi";
     else
      if (type == PSDOCUMENT)
	app = "ghostview";
       else
	if (type == XVDOCUMENT)
	  app = "xv";
         else
	  if (type == MPEGDOCUMENT)
            app = "mpeg_play";
	   else
	    if (type == AUDIODOCUMENT)
	      app = "showaudio";
	     else
	      if (type == XWDDOCUMENT)
		app = "xwud -in";
	       else
		if (type == MIMEDOCUMENT)
		  app = "xterm -e metamail";
		 else
		  app = NULL;

    if (app)
      {
       sprintf(cmd, "w3show %s %s &", app, tmpfile);
       system(cmd); /* FIXME  This routine currently #if 0ed */
       Announce("using %s to display %s ...", app, NewDoc.url);
      }
#ifdef ARENA_DEBUG	/* QingLong.26-12-96 */
     else
      {
       Arena_TracePrint(Iam,
			"In the future, %s will be displayed at this point\n",
			NewDoc.url);
      }
#endif
   }
}
#endif


/*
 * Write buffer to file.
 */
void SaveDoc(char* theFileName)
{
 static char* SaveAsString = NULL;
 long i;
 FILE* fp;
 char* p;
 char* fileName;


 if (Arena_StrLen(theFileName))
   fileName = theFileName;
  else
   if ((fileName = ArenaPromptUser(_("File name?"), SaveAsString, True)))
     {
      Free(SaveAsString);
      SaveAsString = fileName;
     }
    else
     return;

 p = buffer + hdrlen;

 if ((fp = fopen(fileName, "w")))
   {
#if 0	/* QingLong.25-09-97 */
    if (!CurrentDoc->show_raw)
      fprintf(fp, "<BASE href=\"%s\">\n", CurrentDoc->url);
#endif
    for (i = CurrentDoc->loaded_length; i > 0; --i) putc(*p++, fp);

    fclose(fp);
    Announce(_("Saved to file \"%s\""), fileName);
   }
  else
   Warn(_("Can't create file \"%s\""), fileName);

 XFlush(display);
}
