/*
 * Bookmarks for Arena. Contributed by Gregory L. Kaup <glkaup@kaup.com>.
 */

/*  Arena -- bookmark.c
 *  Code to do actual handling of bookmarks.  Mouse Buttons ...
 *
 *  ARENA_BookMarx_OpenCurrentFolder_Button
 *        Open the default folder (ultimate default is ~/.Arena/bookmarks/My).
 *
 *  ARENA_BookMarx_OpenFolderIndex_Button
 *        Open ~/.Arena/bookmarks/Index which is the INDEX of bookmark folders.
 *        When Arena opens Index, selecting a "folder" sets it as
 *        the "default".
 *
 *  ARENA_BookMarx_AddCurrentURL_Button
 *        Add current URL to current "default" bookmark folder.
 *
 *  A "bookmark folder" file (e.g. My) contains URL <space> Title records <cr>
 *  
 *  This code will automatically create all absolutely necessary files for
 *  bookmarking upon the first use of the BookX toolbar button.
 *
 *  If you need to have more than one bookmark "folder", then
 *  edit the file ~/.Arena/bookmarks/Index and add a record of
 *  NEW <space> Description <cr>
 *
 *  NOTE! You CANNOT enter a URL for the first field in the Index.
 *        One word (whitespace delimited) will be used for the folder name.
 *        The rest of the record will be used as its description.
 *  Upon using the BookX button, the NEW bookmark folder file
 *       ~/.Arena/bookmarks/NEW   will be created, and all is ready!
 */

/*
 *  Author:
 *     Greg Kaup can be reached at  <glkaup@kaup.com>
 *
 *  History (reverse):
 *   08-10-1997
 *     -Removed the standard access, fork, waitpid used during sorting.
 *      Using Spawn with WAIT_ASYNC.  This required "blocking" of BM 
 *      reentry until ALL sorts were completed!
 *   09-09-1997
 *     -Change HTChunk to send a _toCString instead of _data()
 *      to pseudoLoadAnchor().  This required _new() chunk to be used
 *      for each HTML creation.  But, DocDel() was getting rid of the
 *      data within the Chunk...
 *   15-07-1997
 *     -Began the removal of the (not so temporary) .html files of the index.
 *      This reduces security risks, and is possible since QingLong made
 *      the pseudoLoadAnchor routine available!
 *     -Removed serveral "extra" routines.  Dumb overhead for no purpose!
 *   30-06-1997
 *     -BMinit() --- constatnts initialization.
 *   27-06-1997
 *     -Took out the creation of any paths except for private bookmarks.
 *     -Use the UserProfile definition of /tmp
 *   20-06-1997
 *     -Removed several routines and moved them (renamed as well) to util.c
 *      Seem to need them elsewhere, so made them a little more general.
 *     -Did away with several defines in arena.h
 *     -BMSetupNewFolder will now create all of the directories needed for
 *      Arena (using HTLib_appName).
 *   11-06-1997
 *     -Add ability to bookmark the "mailto:" entry form... actually
 *      get the mailto address.  Not really sure we want/need this, but...
 *     -Added some signal handling around the forking to run sort.
 *   02-05-1997
 *     -Fix BAD HTML flag.  Thanks to <smartin@usit.net> for the patch.
 *     -Had to fix the <li> tags to require the <ul> </ul> pair of tags.
 *   09-04-1997
 *     -Fixed ADD BookMark so that any URL with "MarxBook" will be treated 
 *      as if it had NO title.  We should not need BM to the BM folders.
 *     -Fixed the Index so that users don't have to enter URL to a new Folder.
 *      Now an Index record can be NEW <space> A New Folder Description <cr>
 *     -Added security check of URLs during Add and HTML creation.
 *      Also look over the string sent to "system" to do the sort!
 *      Also look over the HOME value used to create files, etc.
 *   07-04-1997 by QingLong
 *     -Slightly stylized the code (mainly variables/symbols names).
 *     -Added configuration time `sort' utility path detection.
 *   05-04-1997
 *     -Fix debug stuff to match Arena developers style.
 *     -Changed the external variable debug into an unsigned long int!
 *      This required many changes to other source modules.  Also added
 *      BOOKMARKS_TRACE to the arena.h.in
 *     -Created private directory for temporary files ~/.Arena/tmp.
 *     -Fixed call to external sort to use #defined PATH and the new /tmp path.
 *     -Fixed BMGetMyHost to use the libWWW routine HTGetHostName().
 *      This removes a possible buffer overrun.  I kept this routine which
 *      calls the libWWW one, to keep static storage of the host name.
 *     -FIXED SERIOUS BUG in storage method of "URL &  Desc" record in both
 *      folders and Index.  The | was removed as the field seperator.  This
 *      allowed a "file" to be specified as 2 files and that is NOT cool.
 *      If you have a VERY early version of this, you will HAVE to edit your
 *      Index and any folders.  Remove the | and replace it with a space.
 *   04-04-1997
 *     -Original version first sent to QingLong.
 *
 *  Bugs:
 *     -Would like to fix the history stuff to NOT include the bookmark URLs.
 *     -Two people logged in under the same user id... if they both try to
 *      bookmark at the same time... PROBLEM.  This also occurs if one user
 *      id is symlinked to anothers home directory!
 */

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <signal.h>
#include <time.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <unistd.h>

#include "arena.h"
#ifdef ARENA_DEBUG
#  include "debug.h"
#endif
#include "bridge.h"
#include "bookmark.h"
#include "defs.h"
#include "html.h"
#include "main.h"
#include "status.h"
#include "types.h"
#include "util.h"
#include "x11.h"
#include "spawn.h"


#include "HTLib.h"
#include "HTInet.h"
#include "HTChunk.h"

/*
 * bookmarks.c  Internal Routine Prototypes
 */
typedef enum { BM_INDEX, BM_FOLDER } BMHtmlType;

Bool BMSetupNewFolder(char *newfolder, char *desc);
Bool BMAdd( Doc* );
char *BMFolderAsFile(const char *folder);
void BMSetDefaultFolder(char *folder);
Bool BMSortFolder(const char *folder);
Bool BMHtmlFolder(char *folder, BMHtmlType ftype);
void BMSortExitHandler(SPAWNChildInfo *child);

/*
 * internal static data
 */
static char *BMFolderName = NULL;    /* DEFAULT folder name                */
static char *BMFolderDir = NULL;     /* root to the bookmark stuff         */
static Bool BMInProgress = False;    /* If BM already in progress?         */
static Bool BMSorting = 0;           /* If background sort is running      */
static HTChunk *BMHtmlChunk = NULL;  /* mem alloced HTML of bookmarks      */

/*
 * BookMarx stuff ``global'' constants initialization.
 */
Bool BMinit(void)
{
 extern char* BMFolderDir;
 char* cTemp = NULL;
#ifdef ARENA_DEBUG
 char Iam[] = "BMinit";
#endif


/* Build path to bookmark folders "~/.Arena/bookmarks/"
 * We'll use HTLib_appName() instead of .Arena here!
 */
 if ((cTemp = ALibHomePath()))
   {
    StrAllocCat(cTemp, ".");
    StrAllocCat(cTemp, HTLib_appName());
    StrAllocCat(cTemp, "/bookmarks/");

    /* Save our directory (in static storage)
     * for filespec creations later on.  Used over and over again!
     * Should only be used internally to bookmarks.c
     */
    StrAllocCopy(BMFolderDir, cTemp);
    Free(cTemp);

#ifdef ARENA_DEBUG
    if (BOOKMARKS_TRACE)
      Arena_TracePrint(Iam, " Set BookMarx root to \"%s\"\n", BMFolderDir);
#endif

    return True;
   }
  else
    {
#ifdef ARENA_DEBUG
     if (BOOKMARKS_TRACE) Arena_TracePrint(Iam, " Failed to get home path.\n");
#endif
     return False;
    }
}


/*  First external routine...  ToolbarButtonUp calls this to procees
 *  BookX button up!  We will ignore multi button presses (2 and up)
 */
void BMProcessButtonUp(unsigned int MBp)
{
 static Bool BMSetupDone = False;  /* ONLY ONCE should we be here    */

 Doc* theDoc;

#ifdef ARENA_DEBUG
 char *Iam = "BMProcessButtonUp";
#endif

/* MultiThread fix... or ignore being here!
 * If user hits BookX button rapidly, just don't keep coming in here!
 * If we are still sorting a folder from the last time we were here... BEEP!
 */
 if (BMInProgress || SpawnIsBlocked(BMSorting))
   {
#ifdef ARENA_DEBUG
    if (BOOKMARKS_TRACE)
      Arena_TracePrint(Iam, " Already active. (button %d pressed)\n", MBp);
#endif
    Beep();
    return;
   }
 BMInProgress = True;
 ShowBusy();

/* One TIME!  Create the Arena files absolutely necessary for bookmarks!
 * Done here instead of in BMInit to reduce startup time for Arena.  Here
 * only because the user wants BMs! 
 * Use HTLib_appName() in place of Arena
 * ~/.Arena/bookmarks/Index
 *                   /My
 * And set the Default bookmark folder to My
 */
 if (!BMSetupDone)
   {
    if (BMSetupNewFolder(NULL,NULL))
      {
       Beep();
       HideBusy();
       return;
      }
    BMSetupDone = True;
   }

/*  Process a mouse button UP over the BookX toolbar button!
 *   --- Add current URL to current default bookmark folder
 *   --- Open the Index of bookmark folders... Selecting one of these will
 *       set it as the new default bookmark folder.
 *   --- Open the current default bookmark folder.
 */

#ifdef ARENA_DEBUG
 if (BOOKMARKS_TRACE)
   {
    switch (MBp)
      {
       case Button1:
	 Arena_TracePrint(Iam, " Button1 begins...\n");
	 break;
       case Button2:
	 Arena_TracePrint(Iam, " Button2 begins...\n");
	 break;
       case Button3:
	 Arena_TracePrint(Iam, " Button3 begins...\n");
	 break;
       default:
	 Arena_TracePrint(Iam, " BAD BUTTON %d\n", MBp);
	 break;
      }
   }
#endif


 switch (MBp)
   {
    case ARENA_BookMarx_AddCurrentURL_Button:
      if (BMAdd(CurrentDoc)) Beep();
      break;

    case ARENA_BookMarx_OpenFolderIndex_Button:
      if (!BMHtmlFolder(ARENA_BookMarks_INDEX, BM_INDEX))
	{
	 char *tURL = NULL;
	 StrAllocCopy(tURL, "bookmark:");
	 StrAllocCat(tURL, ARENA_BookMarks_INDEX);
	 theDoc = pseudoLoadAnchor(tURL, Arena_StrLen(tURL),
				   NULL, 0,
				   HTChunk_toCString(BMHtmlChunk),
				   HTChunk_size(BMHtmlChunk),
				   html_atom,
				   True, False, False, True, False, True,
				   NULL);
	 Free(tURL);
	 if (theDoc) break;
#ifdef ARENA_DEBUG
	 if (BOOKMARKS_TRACE) Arena_TracePrint(Iam, " theDoc = NULL / INDEX");
#endif
	}
      Beep();
      Warn(_("Cannot load bookmark folders index"));
      break;

    case ARENA_BookMarx_OpenCurrentFolder_Button:
      if (!BMHtmlFolder(BMFolderName, BM_FOLDER))
	{
	 char *tURL = NULL;
	 StrAllocCopy(tURL, "bookmark:");
	 StrAllocCat(tURL, BMFolderName);
	 theDoc = pseudoLoadAnchor(tURL, Arena_StrLen(tURL),
				   NULL, 0,
				   HTChunk_toCString(BMHtmlChunk),
				   HTChunk_size(BMHtmlChunk),
				   html_atom,
				   True, False, False,
				   True, False, True,
				   NULL);
	 Free(tURL);
	 if (theDoc) break;
#ifdef ARENA_DEBUG
	 if (BOOKMARKS_TRACE) Arena_TracePrint(Iam, " theDoc = NULL / Folder");
#endif
	}
      Beep();
      Warn(_("Bookmark folder not available"));
      break;

    default:
      Beep();
      break;
   }

#ifdef ARENA_DEBUG
 if (BOOKMARKS_TRACE) Arena_TracePrint(Iam, " Button %d ENDS...\n", MBp);
#endif
 BMInProgress = False;
 HideBusy();
 return;
}

/*  Yes, add a bookmark to the currently selected default folder.
 *  If no folder is yet "defaulted" it "WILL BE" set to "My" !
 *  Will now fail to enter the URL if there is ANY security risk to
 *  using the string specified!
 */
Bool BMAdd( Doc* d )
{ 
 char *file;
 FILE *bmfile;

#ifdef ARENA_DEBUG
 char *Iam = "BMAdd";
 if (BOOKMARKS_TRACE) Arena_TracePrint(Iam, " Enter...\n");
#endif

/* Any reason to be here yet?
 * Don't BM any null URLs or our own Bookmark files!
 */
 if ( (!d || !d->url || !d->title) ||
      (d->title && strstr(d->title, "MarxBook")))
   {
#ifdef ARENA_DEBUG
    if (BOOKMARKS_TRACE) Arena_TracePrint(Iam, " No title or URL...\n");
#endif
    Warn(_("No Title or URL available for bookmarking!"));
    return EXIT_FAILURE;
   }

/* YES... but first we will check the URL for security problems
 */
 if ( BMCheckForSecurity(d->url) )
   {
#ifdef ARENA_DEBUG
    if (BOOKMARKS_TRACE)
      Arena_TracePrint(Iam, " URL %s is security risk!\n", d->url);
#endif
    Warn(_("No Title or URL available for bookmarking!"));
    return EXIT_FAILURE;
   }

/*  OK, let's get a folder file and add it
 */
 file = BMFolderAsFile(NULL);
 if ( !file )
   {
#ifdef ARENA_DEBUG
    if (BOOKMARKS_TRACE) Arena_TracePrint(Iam, " Bad (null) Folder file\n");
#endif
    Warn(_("Bookmark folder file is bad..."));
    return EXIT_FAILURE;
   }

 bmfile = fopen(file,"a");
 if ( !bmfile)
   {
#ifdef ARENA_DEBUG
    if (BOOKMARKS_TRACE)
      Arena_TracePrint(Iam, " Failed on BM file %s\n", file);
#endif
    Warn(_("Can't open bookmark folder %s"), file);
    return EXIT_FAILURE;
   }

 if (strstr(d->title, "mailto:"))          /* kludge!  Use title        */
   fprintf(bmfile, "%s %s\n", d->title, &d->title[7]);
 else
   {
    fprintf(bmfile, "%s ", d->url);
    if (strstr(d->url, "ftp:"))            /* since ftp desc are lousy! */
      fprintf(bmfile, "%s\n", d->url);     /* usually don't even have   */
    else                                   /* a system name, just dir   */
      fprintf(bmfile, "%s\n", d->title);
   }
 fclose(bmfile);

#ifdef ARENA_DEBUG
 if (BOOKMARKS_TRACE)
   {
    Arena_TracePrint(Iam, " URL  %s\n", d->url);
    Arena_TracePrint(Iam, " Title %s\n", d->title);
   }
#endif

 /* We have added our new BM to the current folder...
  * NOW sort it.  We will hereby guarantee no duplicates.
  */
 return BMSortFolder(NULL);
}

/* Sort exit handler...
 * When we SORT the bookmark folders, we will do it in ASYNC mode...
 * i.e. Let the sort run... while we continue to run.
 * This exit handler will really only clear the flag that indicates
 * the the sort WAS running.
 */
void BMSortExitHandler(SPAWNChildInfo *child)
{
#if ARENA_DEBUG
 char Iam[] = "BMSortExitHandler";
 if (BOOKMARKS_TRACE || SPAWN_TRACE)
   Arena_TracePrint(Iam, " Enter...child @"POINTER_FORMAT"\n", child);
#endif

 if (child == NULL) return;
 if (child->exit_handler_done == True)  return;

 /* This is what we are here to do!  Since we MAY have more than one
  * sort going at the same time, decrement it only!
  */
 SpawnUnBlock( &BMSorting );

#if ARENA_DEBUG
 if (BOOKMARKS_TRACE || SPAWN_TRACE)
   Arena_TracePrint(Iam, " Leaving... %d sort ongoing\n", BMSorting);
#endif
 return;
}

/*  Sort the folder... this gets rid of all duplicates
 *  The sort is onto the same file.  The PATH to sort must be previously
 *  #defined (either by configure or by bookmarks.h), and any sort arguments
 *  need to be #defined in arena.h.in  if your requirements differ.
 */
Bool BMSortFolder( const char *folder )
{
 SPAWNChildInfo *child = NULL;

 char cFolder[MAX_PATH_LENGTH+2];  /* the child process needs these          */
 char cTemp[MAX_PATH_LENGTH+2];    /* could NOT get away with "alloc" ptrs   */

 char *args[] = { ARENA_SORT_UTILITY_PATH,           /* [0] = sort path      */
		  "-T", cTemp,                       /* temp dir spec        */
		  ARENA_BookMarks_SORT_ARGS,         /* sort flags           */
		  cFolder,                           /* input file spec      */
		  "-o", cFolder,                     /* output file spec     */
		  NULL                               /* END of Args          */
 };

/* Give the child NO environment controls what so ever!
 * Must control PATH and IFS completely
 */
 char *envp[] = { "PATH=::", "IFS=' '", NULL };

#ifdef ARENA_DEBUG
 char *Iam = "BMSortFolder";
#endif

#ifdef ARENA_DEBUG
 if (BOOKMARKS_TRACE && VERBOSE_TRACE) Arena_TracePrint(Iam, " Enter...\n");
#endif

/* Building the temporary directory to be given to sort.
 * This will be passed to sort via args... NOT through the environment!
 */
 strcpy(cTemp, HTUserProfile_tmp(HTLib_userProfile()));

/* Check over the folder file spec AND HOME+APPNAME for security purposes.
 * We will just not handle some types of specs rather than get caught with
 * our pants down!
 */
 strcpy(cFolder, BMFolderAsFile(folder));
 if (BMCheckForSecurity(cFolder)
     || BMCheckForSecurity(cTemp)
     || BMCheckForSecurity(ARENA_SORT_UTILITY_PATH))
   {
#ifdef ARENA_DEBUG
     if (BOOKMARKS_TRACE)
       Arena_TracePrint(Iam, " SECURITY on folder %s\n", cFolder);
#endif
     Warn(_("Folder %s has sorting problem!"), folder);
     Beep();
     return EXIT_FAILURE;
   }

 /* Now block out some things...
  * Really just blocking "button" presses.
  * The flag gets "cleared" by the exit handler!
  */
 SpawnBlock( &BMSorting );

 /* Start up the sort child! AND KEEP on going ahead.  The BMSorting BLOCK
  * will be checked to keep the user from using the BookX GUI button until
  * ALL sorting is done!
  */
 child = SpawnVE(WAIT_ASYNC, KILL_ALWAYS,
		 args[0], args, envp, BMSortExitHandler);

 if (!SpawnIsChildAlive(child))
   {
#ifdef ARENA_DEBUG
       Arena_TracePrint(Iam, " Fork failure for %s\n", cFolder);
#endif
     Warn(_("Can't start sort of folder %s"), folder);
     Beep();
     sleep(5);
     SpawnUnBlock(&BMSorting);
     return EXIT_FAILURE;
   }
 return EXIT_SUCCESS;
}


/* Now convert the folder to .html (in memory!).
 * ftype == 0 is used to specify the INDEX is being converted.
 * ftype != 0 for regular folder conversion.
 * This allowed all of the verification stuff to be in one place.
 */
Bool BMHtmlFolder( char *folder, BMHtmlType ftype )
{
 char fmt[25];
 FILE *bmfile;

#ifdef ARENA_DEBUG
 char *Iam = "BMHtmlFolder";
#endif
 char *cTemp = NULL;
 char *cDesc = NULL;
 unsigned long types = 0;
 int iother = 0;                             /* iother allows multi "other"  */
 int istat = 0;

 static char *typedesc[] = { "http:",   "https:", "file:", "ftp:",
			     "mailto:", "news:",  "nntp:", "wais:",
			     "gopher:", "telnet:",
			     NULL };               /* REALLY need this null  */

 /* Get our Folder opened for reading...
  */
 char *file = BMFolderAsFile(folder);
#ifdef ARENA_DEBUG
 if (BOOKMARKS_TRACE && VERBOSE_TRACE)
   Arena_TracePrint(Iam, " Open file %s\n", file);
#endif
 bmfile = fopen(file, "r");
 if (!bmfile) return EXIT_FAILURE;

 /* We will need some temporary storages for the URL and its description.
  * Also create the output chunk (mem alloc) buffer.  We will get rid of
  * it when we pass it to pseudoLoadAnchor()
  */
 BMHtmlChunk = HTChunk_new(512);

 cTemp = (char *)Arena_MAlloc(MAX_PATH_LENGTH  + 2, False);
 cDesc = (char *)Arena_MAlloc(MAX_PATH_LENGTH + 2, False);
 if (!cTemp || !cDesc || !BMHtmlChunk)
   {
    fclose(bmfile);
    Free(cTemp);
    Free(cDesc);
    HTChunk_delete(BMHtmlChunk);
#ifdef ARENA_DEBUG
    if (BOOKMARKS_TRACE) Arena_TracePrint(Iam, " MALLOCs FAILED!\n");
#endif
    return EXIT_FAILURE;
   }

 /*  Create format for fscanf used below when reading in the URL <space> Title
  *  records.  This format would result in " %128s %128[^\n] " to be used
  *  below.  Therefore keeping our temporary arrays from overflowing.
  */
 sprintf(fmt, " %%%-ds %%%-d[^\n] ",
	 MAX_PATH_LENGTH, MAX_PATH_LENGTH);

 /*  Output html and segregate the several URL types with several headers.
  *  If doing the Index Folder, change the header and BOLD our new Default.
  *  And, do NOT segregate.
  *  DON'T ouput any URL that can in any way pose a security risk!
  */
 if (ftype)
   {
    HTChunk_puts(BMHtmlChunk, "<html>\n<head>\n<title>");
    HTChunk_puts(BMHtmlChunk, BMFolderName);
    HTChunk_puts(BMHtmlChunk, " MarxBook\n</title>\n</head>\n<body>\n<h1>");
    HTChunk_puts(BMHtmlChunk, _(BMFolderName)); /* allow translations */
    HTChunk_puts(BMHtmlChunk, " ");
    HTChunk_puts(BMHtmlChunk, _("Bookmarks"));
    HTChunk_puts(BMHtmlChunk, "</h1>\n");
   }
  else
    {
     HTChunk_puts(BMHtmlChunk, "<html>\n<head>\n<title>MarxBook Folder Index\n"
		               "</title>\n</head>\n<body>\n<h1>");
     HTChunk_puts(BMHtmlChunk, _("Bookmark Folder Index"));
     HTChunk_puts(BMHtmlChunk, "</h1><ul>\n");
    }

 while (istat != EOF)
   {
    istat = fscanf(bmfile, fmt, cTemp, cDesc);
    if (istat != 2) continue;
    if ( BMCheckForSecurity(cTemp) )
      {
#ifdef ARENA_DEBUG
       if (BOOKMARKS_TRACE) 
	 Arena_TracePrint(Iam, " Security problems with %s\n", cTemp);
#endif
       continue;
      }

    if (!ftype)  /* Index... Fix URL if necessary, BOLD current */
      {
       if (strstr(cTemp, ":") || strstr(cTemp,".html"))
	 {
	  Beep();
	  Warn(_("Please edit your BM Index and remove URL specifications"));
	  return EXIT_FAILURE;
	 }

       HTChunk_puts(BMHtmlChunk, "<li><a href=bookmark:");
       HTChunk_puts(BMHtmlChunk, cTemp);
       if (strstr(cTemp, BMFolderName))
	 {
	  HTChunk_puts(BMHtmlChunk, "><b>");
	  HTChunk_puts(BMHtmlChunk, cDesc);
	  HTChunk_puts(BMHtmlChunk, "</b></a>\n");
	 }
       else
	 {
	  HTChunk_putc(BMHtmlChunk, '>');
	  HTChunk_puts(BMHtmlChunk, cDesc);
	  HTChunk_puts(BMHtmlChunk, "</a>\n");
	 }
      }
    else                           /* Regular Folder, insert special headers */
      {
       int i = 0;
       if (Arena_StrLen(cTemp) < 7) continue;     /* throw away short recs!  */
       while (typedesc[i] != NULL)          /* look for a new link type      */
	 {
	  if (strstr(cTemp, typedesc[i]))
	    {
	     if (!(types&(1<<i)))
	       {
		if (types)  HTChunk_puts(BMHtmlChunk, "</ul>");
		HTChunk_puts(BMHtmlChunk, "<h2>");
		HTChunk_puts(BMHtmlChunk, typedesc[i]);
		HTChunk_puts(BMHtmlChunk, "//");
		HTChunk_puts(BMHtmlChunk, _("Links"));
		HTChunk_puts(BMHtmlChunk, "</h2><ul>\n");
		iother = 0;                   /* allow another "other"   */
		types |= (1<<i);              /* But, not another "type" */
	       }
	     break;
	    }
	  i++;
	 }
       if (typedesc[i] == NULL && !iother)
	 {
	  if (types)  HTChunk_puts(BMHtmlChunk, "</ul>");
	  HTChunk_puts(BMHtmlChunk, "<h2>");
	  HTChunk_puts(BMHtmlChunk, _("Other"));
	  HTChunk_puts(BMHtmlChunk, "://");
	  HTChunk_puts(BMHtmlChunk, _("Links"));
	  HTChunk_puts(BMHtmlChunk, "</h2></ul>\n");
	  iother = 1;
	 }
       HTChunk_puts(BMHtmlChunk, "<li><a href=");
       HTChunk_puts(BMHtmlChunk, cTemp);
       HTChunk_putc(BMHtmlChunk, '>');
       HTChunk_puts(BMHtmlChunk, cDesc);
       HTChunk_puts(BMHtmlChunk, "</a>\n");
      }
   } /* End While on istat != EOF! */

 fclose(bmfile);
 Free(cTemp);
 Free(cDesc);
 HTChunk_puts(BMHtmlChunk, "\n</ul>\n</body>\n</html>\n");
 HTChunk_terminate(BMHtmlChunk);
 return EXIT_SUCCESS;
}

/*  FolderAsFile will give back a pointer to static allocated char string
 *  containing a full file spec to the "folder" specified... if NULL, then
 *  to the "default" folder.  Can be used to get full file spec to Index
 */
char *BMFolderAsFile(const char *foldername)
{
 static char *BMFolderFILE = NULL;

 Free(BMFolderFILE);
 StrAllocCopy(BMFolderFILE, BMFolderDir);
 StrAllocCat(BMFolderFILE, foldername ? foldername : BMFolderName);

#ifdef ARENA_DEBUG
 if (BOOKMARKS_TRACE && VERBOSE_TRACE)
   Arena_TracePrint("BMFolderAsFile", " Returning %s\n", BMFolderFILE);
#endif
 return BMFolderFILE;
}


/*  BMSetDefaultFolder
 *  Will change the default folder to new specification (if possible).
 *  if new is NULL, sets default to the ULTIMATE Default "My".
 */
void BMSetDefaultFolder(char *new)
{
#ifdef ARENA_DEBUG
 char *Iam = "BMSetDefaultFolder";
 if (BOOKMARKS_TRACE && VERBOSE_TRACE)
   Arena_TracePrint(Iam, " Entering %s\n", new);
#endif

/*  store the new default name, or use the ultimate folder */

 Free(BMFolderName);
 StrAllocCopy(BMFolderName, new ? new : ARENA_BookMarks_DEFAULT_FOLDER);

#ifdef ARENA_DEBUG
 if (BOOKMARKS_TRACE && VERBOSE_TRACE)
   Arena_TracePrint(Iam, " Returning %s\n", BMFolderName);
#endif
 return;
}

/*  BMOpenDocCheckForIndex
 *  This routine is used externally by OpenDoc!
 *  OK, here is a real kludge!
 *  I did not want to set a raft of flags to control the setting of the default
 *  bookmark folder...  SO, if the current document is the URL of the "index"
 *  of folders, and the "NEW" document to get is a "BM folder",
 *  then set IT as the "default"!  AND then display it (pseudoLoadAnchor).
 *  Any failure will just drop into "get a document"!
 */
Bool BMOpenDocCheckForIndex(char *newURL)
{
#ifdef ARENA_DEBUG
 char *Iam = "BMOpenDocCheckForIndex";
 if (BOOKMARKS_TRACE && VERBOSE_TRACE)
   {
    if (newURL) Arena_TracePrint(Iam, " Enter newURL = %s\n", newURL);
    if (CurrentDoc->url) Arena_TracePrint(Iam, " URL = %s\n", CurrentDoc->url);
   }
#endif

 if (CurrentDoc->url && newURL)
   {
    if (strstr(CurrentDoc->url, "bookmark:") &&
	strstr(CurrentDoc->url, ARENA_BookMarks_INDEX))
      {
       char *t;
       if ((t = strstr(newURL, "bookmark:")) != NULL )
	 {
	  BMSetDefaultFolder(newURL + 9);
#ifdef ARENA_DEBUG
	  if (BOOKMARKS_TRACE && VERBOSE_TRACE)
	    Arena_TracePrint(Iam, " Set = %s\n", BMFolderName);
#endif
	  if (!BMHtmlFolder(BMFolderName, BM_FOLDER))
	    {
	     Doc* theDoc;
	     theDoc = pseudoLoadAnchor(newURL, Arena_StrLen(newURL),
				       NULL, 0,
				       HTChunk_toCString(BMHtmlChunk),
				       HTChunk_size(BMHtmlChunk),
				       html_atom,
				       True, False, False,
				       True, False, True,
				       NULL);
	     if (theDoc) return EXIT_SUCCESS;
#ifdef ARENA_DEBUG
	     if (BOOKMARKS_TRACE)
	       Arena_TracePrint(Iam, " theDoc = NULL / %s", newURL);
#endif
	    }
	  Beep();
	 }
      }
   }
 return EXIT_FAILURE;
}

/*  BMSetupNewFolder
 *  Create bookmarking system of files, if necessary!
 *  This routine will be invoked only once per session when the BookX
 *  toolbar button is activated!  It is however capable of being called
 *  multiple times to create new folders.
 *  Fixed to allow the HTLib_appName() to be used instead of ~./Arena
 *  Someone may want it to create ~/.GREG ( well, maybe not everyone! )
 *  Don't allow any security risk folder names to be added!
 */
Bool BMSetupNewFolder(char *newfoldername, char *desc)
{
 int istat;
 struct stat fst;
 FILE *fp;
 char *cTemp = NULL;
#ifdef ARENA_DEBUG
 char *Iam = "BMSetupNewFolder";
#endif


/*  Check for security risks.  We have to let a NULL newfoldername pass
 *  here since this routine will create all necessary bookmark directories
 *  AND the default folder.
 */
 if (newfoldername)
   {
    if (BMCheckForSecurity(newfoldername))
      {
	Warn(_("Folder name %s is a security risk!"), newfoldername);
#ifdef ARENA_DEBUG
        if (BOOKMARKS_TRACE)
	  Arena_TracePrint(Iam, " Security BAD! %s\n", newfoldername);
#endif
	return EXIT_FAILURE;
      }
   }

/* Get directory (from static storage) */
 StrAllocCopy(cTemp, BMFolderDir);

#ifdef ARENA_DEBUG
 if (BOOKMARKS_TRACE) Arena_TracePrint(Iam, " Checking \"%s\"...\n", cTemp);
#endif

 if (!ALibCreatePath(cTemp, 0750))
   {
    Warn(_("Bookmarks will not work! \"%s\" is not a valid path"), cTemp);
    Free(cTemp);
    return EXIT_FAILURE;
   }

/*  Make very sure that an Index of folders file exists.
 *  Make sure the Index contains a ref to the Ultimate default folder      
 *  AND our new folder!
 */
 StrAllocCat(cTemp, ARENA_BookMarks_INDEX);

#ifdef ARENA_DEBUG
 if (BOOKMARKS_TRACE) Arena_TracePrint(Iam, " Checking index %s\n", cTemp);
#endif

 istat = stat(cTemp, &fst);    
 if (istat || newfoldername)
   {
    fp = fopen(cTemp, "a");
    if (!fp)
      {
       Warn(_("Bookmarks will not work! \"%s\" is not a valid Index."), cTemp);
       Free(cTemp);
       return EXIT_FAILURE;
      }
    if (istat)
      {
       fprintf(fp, "%s ", ARENA_BookMarks_DEFAULT_FOLDER);
       fprintf(fp, _("My Private Bookmarks\n"));
       fprintf(fp, "%s %s ", HTLib_appName(), HTLib_appName());
       fprintf(fp, _("Project Bookmarks\n"));
      }
    if (newfoldername)
      {
       fprintf(fp, "%s ", newfoldername);
       fprintf(fp, "%s\n", desc);
      }
    fclose(fp);
   }
 Free(cTemp);

 /*  MAKE sure that the "Ultimate default folder exists" */
 istat = stat(BMFolderAsFile(ARENA_BookMarks_DEFAULT_FOLDER), &fst);
 if (istat)
   {
    fp = fopen(BMFolderAsFile(ARENA_BookMarks_DEFAULT_FOLDER), "a");
    if (!fp)
      {
       Warn(_("Bookmarks will NOT work! DEFALUT folder is not valid."));
       return EXIT_FAILURE;
      }
    fclose(fp);
   }
 BMSortFolder(ARENA_BookMarks_DEFAULT_FOLDER);

 /*  MAKE sure that the "Arena project bookmarks" exists. */
 istat = stat(BMFolderAsFile(HTLib_appName()), &fst);
 if (istat)
   {
    fp = fopen(BMFolderAsFile(HTLib_appName()), "a");
    if (!fp)
      {
       Warn(_("Bookmarks will NOT work! appName folder is not valid."));
       return EXIT_FAILURE;
      }
    fprintf(fp, "%s %s %s URL\n", DEFAULT_URL, _("Default"), HTLib_appName());
    fprintf(fp, "%s %s %s\n", HELP_URL, HTLib_appName(), _("Help"));
    fclose(fp);
   }
 BMSortFolder(HTLib_appName());

 /*  FINALLY... we're ready to try and create the new folder!
  *  Make sure we add it to the Index                        
  */
 if (newfoldername)
   {
    fp = fopen(BMFolderAsFile(newfoldername), "a");
    if (!fp)
      {
       Warn(_("Bookmark failure! New folder is not valid"));
       return EXIT_FAILURE;
      }
    fclose(fp);
    BMSortFolder(newfoldername);
   }

 /*  We have successfully created all necessary bookmark files.
  *  We also sorted each folder since they MAY have been edited externally.
  *  Now, set the "new folder" created by this routine to be the
  *  DEFAULT folder... unless this routine was called with NULL
  *  Then the Ultimate Default folder "My" will be set!
  */
 BMSetDefaultFolder(newfoldername);
 return EXIT_SUCCESS;
}

/*  One routine to check over a string for any "faulty" stuff that might
 *  lead to a security breach...  Since the string may be passed to the system
 *  sort routine, we cannot allow any types of redirection or substitution
 *
 *  DO NOT allow <, >, <<, >>, |, $ or % substitutions
 *  or [], {}, or () for subprocess creations.
 *  I'm not sure about ^&`'\ and "" .
 *  <whitespace> is NOT ALLOWED!  That would allow multiple specs within one!
 *  The # character must be allowed, it is used for "local labels"
 *  which are internal URLs within a document.
 */
Bool BMCheckForSecurity( char * s )
{
/* NEVER allow these characters in a URL?
 * For most cases, this is true.  However!!!!
 */
 char *NoWay = " \n\t\v\b\r\f\a`!@$%^&*()=+|\[]{}<>?\\\"'; ";

/* This list of chars will be NOT be allowed when cgi-bin is part of the URL
 * I know that some other chars will have to be added to allow ? and + to
 * be passed as args to the cgi progs... % is used to encode the data before
 * transmit, and = is used for "post".
 * char *cgiOK = "?+%=";  These need to be allowed now.
 */
 char *cgiIS = "/cgi-bin/";
 char *cgiNoWay = " \n\t\v\b\r\f\a`!@$^&*()|\[]{}<>\\\"'; ";

/* Ok, we will let these through for mailto: support!
 * char *mailtoOK = " <@>"; These need to be allowed now.
 */
 char *mailtoIS = "mailto:";
 char *mailtoNoWay = "\n\t\v\b\r\f\a`!$%^&*()=+|\[]{}?\\\"';";

#ifdef ARENA_DEBUG
 char *Iam = "BMCheckForSecurity";
#endif

 if (s)
   {
     if (!strstr(s, "..") && !strstr(s, "./") && !strstr(s, "~/"))
       {
	 if (strstr(s, cgiIS) && !strpbrk(s, cgiNoWay))
	   return EXIT_SUCCESS;
	 if (strstr(s, mailtoIS) && !strpbrk(s, mailtoNoWay))
	   return EXIT_SUCCESS;
	 if (!strpbrk(s, NoWay))
	   return(EXIT_SUCCESS);
       }
   }

#ifdef ARENA_DEBUG
 if (BOOKMARKS_TRACE)
   Arena_TracePrint(Iam, " URL risky %s\n", s);
#endif
 return EXIT_FAILURE;
}
