#ifndef CARD_H
#define CARD_H

#include <time.h>
#include <glib.h>

#include "cardtypes.h"

#define PO 0
#define EXT 1
#define STREET 2
#define CITY 3
#define REGION 4
#define CODE 5
#define COUNTRY 6

#define DELADDR_MAX 7

typedef struct _group
{
	char *name;
	struct _group *parent;
} CardGroup;

typedef struct
{
	CardGroup *grp;

	int used;
	enum PropertyType type;
	enum EncodType encod;
	enum ValueType value;
	char *charset;
	char *lang;
	
	void *user_data;
} CardProperty;

typedef struct
{
	CardProperty prop;
	
	char *str;
} CardStrProperty;


/* IDENTIFICATION PROPERTIES */


typedef struct
{
	CardProperty prop;
	
	char *family;        /* Public */
	char *given;         /* John */
	char *additional;    /* Quinlan */
	char *prefix;        /* Mr. */
	char *suffix;        /* Esq. */
} CardName;

typedef struct 
{
	CardProperty prop;
	
	enum PhotoType type;
	unsigned int size;
	char *data;
} CardPhoto;

typedef struct
{
	CardProperty prop;
	
	int year;
	int month;
	int day;
} CardBDay;


/* DELIVERY ADDRESSING PROPERTIES */


typedef struct
{
	CardProperty prop;
	
	int type;
	char *data[DELADDR_MAX];
} CardDelAddr;            /* Delivery Address */

typedef struct 
{
	CardProperty prop;
	
	int type;
	char *data;
} CardDelLabel;


/* TELECOMMUNICATIONS ADDRESSING PROPERTIES */


typedef struct
{
	CardProperty prop;
	
	int type;
	char *data;
} CardPhone;

typedef struct 
{
	CardProperty prop;
	
	enum EMailType type;
	char *data;
} CardEMail;

typedef struct
{
	CardProperty prop;
	
	int sign;      /* 1 or -1 */
	int hours;     /* Mexico General is at -6:00 UTC */
	int mins;      /* sign -1, hours 6, mins 0 */
} CardTimeZone;

typedef struct
{
	CardProperty prop;
	
	float lon;
	float lat;
} CardGeoPos;


/* ORGANIZATIONAL PROPERTIES */


typedef struct
{
	CardProperty prop;
	
	char *name;
	char *unit1;
	char *unit2;
	char *unit3;
	char *unit4;
} CardOrg;


/* EXPLANATORY PROPERTIES */


typedef struct
{
	CardProperty prop;
	
	int utc;
	struct tm tm;
} CardRev;

typedef struct
{
	CardProperty prop;
	
	enum SoundType type;
	unsigned int size;
	char *data;
} CardSound;

typedef struct
{
	CardProperty prop;
	
	enum KeyType type;
	char *data;
} CardKey;

typedef struct _Card
{
	CardStrProperty fname;
	CardName        name;
	CardPhoto       photo;
	CardBDay        bday;
	
	GList          *deladdr;
	GList          *dellabel;
	
	GList          *phone;
	GList          *email;
	CardStrProperty mailer;
	
	CardTimeZone    timezn;
	CardGeoPos      geopos;
	
	CardStrProperty title;
	CardStrProperty role;
	CardPhoto       logo;
	struct _Card   *agent;
	CardOrg         org;
	
	CardStrProperty comment;
	CardRev         rev;
	CardSound       sound;
	CardStrProperty url;
	CardStrProperty uid;
	CardKey         key;
	
	void *user_data;
	int flag;
} Card;

Card *card_new(void);
void card_free(Card *crd);
CardProperty empty_CardProperty(void);
GList *card_load (GList *crdlist, char *fname);
void card_save(Card *crd, FILE *fp);
VObject *card_convert_to_vobject(Card *crd);

char *card_bday_str(CardBDay bday);
char *card_timezn_str(CardTimeZone timezn);
char *card_geopos_str(CardGeoPos geopos);

#endif
