;; $Id: dbcommon.dsl 1.25 1998/09/02 17:39:50 nwalsh Exp $
;;
;; This file is part of the Modular DocBook Stylesheet distribution.
;; See ../README or http://nwalsh.com/docbook/dsssl/
;;
;; This file contains general functions common to both print and HTML
;; versions of the DocBook stylesheets.
;;

;; If **ANY** change is made to this file, you _MUST_ alter the
;; following definition:

(define %docbook-common-version%
  "Modular DocBook Stylesheet Common Functions version 1.13")

;; === element lists ====================================================

;; these have to be functions because they have to be evaluated when
;; there is a current-node so that normalize can know what declaration
;; is in effect

(define (set-element-list)
  (list (normalize "set")))

(define (book-element-list)
  (list (normalize "book")))

(define (division-element-list)
  (list (normalize "part")))

(define (component-element-list)
  (list (normalize "preface")
	(normalize "chapter")
	(normalize "appendix") 
	(normalize "article")
	(normalize "glossary")
	(normalize "bibliography")
	(normalize "index")
	(normalize "setindex")
	(normalize "reference")
	(normalize "refentry")
	(normalize "book"))) ;; just in case nothing else matches...

(define (major-component-element-list)
  (list (normalize "preface")
	(normalize "chapter") 
	(normalize "appendix") 
	(normalize "article")
	(normalize "glossary")
	(normalize "bibliography")
	(normalize "index")
	(normalize "setindex")
	(normalize "reference")
	(normalize "refentry")
	(normalize "part")
	(normalize "book"))) ;; just in case nothing else matches...

(define (section-element-list)
  (list (normalize "sect1")
	(normalize "sect2")
	(normalize "sect3") 
	(normalize "sect4")
	(normalize "sect5")
	(normalize "simplesect")
	(normalize "refsect1") 
	(normalize "refsect2") 
	(normalize "refsect3")))

(define (block-element-list)
  (list (normalize "example") 
	(normalize "figure") 
	(normalize "table") 
	(normalize "equation") 
	(normalize "procedure")))

(define (outer-parent-list)
  (list (normalize "toc") 
	(normalize "lot") 
	(normalize "appendix") 
	(normalize "chapter") 
	(normalize "part") 
	(normalize "preface") 
	(normalize "reference")
	(normalize "bibliography") 
	(normalize "glossary") 
	(normalize "index") 
	(normalize "setindex")
	(normalize "sect1") 
	(normalize "sect2") 
	(normalize "sect3") 
	(normalize "sect4") 
	(normalize "sect5") 
	(normalize "simplesect")
	(normalize "partintro") 
	(normalize "bibliodiv") 
	(normalize "glossdiv") 
	(normalize "indexdiv")
	(normalize "refentry") 
	(normalize "refsect1") 
	(normalize "refsect2") 
	(normalize "refsect3")
	(normalize "msgtext") 
	(normalize "msgexplan")))

(define (list-element-list)
  (list (normalize "orderedlist") 
	(normalize "itemizedlist") 
	(normalize "variablelist") 
	(normalize "segmentedlist")
        (normalize "simplelist") 
	(normalize "calloutlist") 
	(normalize "step")))

;; === automatic TOC ====================================================

;; Returns #t if nd should appear in the auto TOC
(define (appears-in-auto-toc? nd)
  (if (or (equal? (gi nd) (normalize "refsect1"))
	  (have-ancestor? (normalize "refsect1") nd))
      #f
      #t))

;; # return elements of nl for which appears-in-auto-toc? is #t
(define (toc-list-filter nodelist)
  (let loop ((toclist (empty-node-list)) (nl nodelist))
    (if (node-list-empty? nl)
	toclist
	(if (appears-in-auto-toc? (node-list-first nl))
	    (loop (node-list toclist (node-list-first nl))
		  (node-list-rest nl))
	    (loop toclist (node-list-rest nl))))))
  
;; === common ===========================================================

(define (INLIST?)
  (has-ancestor-member? (current-node) (list-element-list)))

(define (INBLOCK?)
  (has-ancestor-member? (current-node) 
			(list (normalize "example") 
			      (normalize "informalexample")
			      (normalize "figure") 
			      (normalize "informalfigure")
			      (normalize "equation")
			      (normalize "informalequation")
			      (normalize "funcsynopsis")
			      (normalize "programlistingco")
			      (normalize "screenco")
			      (normalize "graphicco"))))

(define (PARNUM)
  (child-number (parent (current-node))))

(define (NESTEDFNUM n fmt)
  (if (number? n)
      (format-number n fmt)
      #f))

(define (FNUM n) (NESTEDFNUM n "1"))

(define (book-start?)
  ;; Returns #t if the current-node is in the first division or 
  ;; component of a book.
  (let ((book (ancestor (normalize "book")))
	(nd   (ancestor-member 
	       (current-node) 
	       (append (component-element-list) (division-element-list)))))
    (let loop ((ch (children book)))
      (if (node-list-empty? ch)
	  #f
	  (if (member (gi (node-list-first ch)) 
		      (append (component-element-list) (division-element-list)))
	      (node-list=? (node-list-first ch) nd)
	      (loop (node-list-rest ch)))))))

(define (first-chapter?)
  ;; Returns #t if the current-node is in the first chapter of a book
  (let* ((book (ancestor (normalize "book")))
	 (nd   (ancestor-member 
		(current-node) 
		(append (component-element-list) (division-element-list))))
	 (bookch (children book))
	 (bookcomp (expand-children bookch (list (normalize "part")))))
    (let loop ((nl bookcomp))
      (if (node-list-empty? nl)
	  #f
	  (if (equal? (gi (node-list-first nl)) (normalize "chapter"))
	      (if (node-list=? (node-list-first nl) nd)
		  #t
		  #f)
	      (loop (node-list-rest nl)))))))

;; === bibliographic ====================================================

(define (author-string #!optional (author (current-node)))
  ;; Return a formatted string representation of the contents of:
  ;; AUTHOR:
  ;;   Handles *only* Honorific, FirstName, SurName, and Lineage.
  ;;   Handles *only* the first of each.
  ;;   Format is "Honorific. FirstName SurName, Lineage"
  ;; CORPAUTHOR:
  ;;   returns (data corpauthor)
  (let* ((h_nl (select-elements (descendants author) (normalize "honorific")))
	 (f_nl (select-elements (descendants author) (normalize "firstname")))
	 (s_nl (select-elements (descendants author) (normalize "surname")))
	 (l_nl (select-elements (descendants author) (normalize "lineage")))
	 (has_h (not (node-list-empty? h_nl)))
	 (has_f (not (node-list-empty? f_nl)))
	 (has_s (not (node-list-empty? s_nl)))
	 (has_l (not (node-list-empty? l_nl))))
    (if (or (equal? (gi author) (normalize "author"))
	    (equal? (gi author) (normalize "editor"))
	    (equal? (gi author) (normalize "othercredit")))
	(string-append
	 (if has_h (string-append (data (node-list-first h_nl)) ".") "")
	 (if has_f (string-append 
		    (if has_h " " "") 
		    (data (node-list-first f_nl))) "")
	 (if has_s (string-append 
		    (if (or has_h has_f) " " "") 
		    (data (node-list-first s_nl))) "")
	 (if has_l (string-append ", " (data (node-list-first l_nl))) ""))
	(data author))))

(define (author-list-string #!optional (author (current-node)))
  ;; Return a formatted string representation of the contents of AUTHOR
  ;; *including appropriate punctuation* if the AUTHOR occurs in a list
  ;; of AUTHORs in an AUTHORGROUP:
  ;;
  ;;   John Doe
  ;; or
  ;;   John Doe and Jane Doe
  ;; or
  ;;   John Doe, Jane Doe, and A. Nonymous
  ;;

  (let* ((author-node-list (select-elements
			    (descendants 
			     (ancestor (normalize "authorgroup"))) (normalize "author")))
	 (corpauthor-node-list (select-elements
				(descendants 
				 (ancestor (normalize "authorgroup"))) (normalize "corpauthor")))
	 (othercredit-node-list (select-elements
				 (descendants 
				  (ancestor (normalize "authorgroup"))) (normalize "othercredit")))
	 (author-count (if (have-ancestor? (normalize "authorgroup") author)
			   (+ (node-list-length author-node-list)
			      (node-list-length corpauthor-node-list)
			      (node-list-length othercredit-node-list))
			   1)))
    (string-append
     (if (and (> author-count 1)
	      (last-sibling? author))
	 (string-append (gentext-and) " ")
	 "")
     (author-string)
     (if (and (> author-count 2)
	      (not (last-sibling? author)))
	 ", "
	 (if (and (> author-count 1)
		  (not (last-sibling? author)))		  
	     " "
	     "")))))

;; === procedures =======================================================

(define ($proc-hierarch-number-format$ depth)
  (case (modulo depth 5)
    ((1) "1")
    ((2) "a")
    ((3) "i")
    ((4) "A")
    (else "I")))

(define ($proc-hierarch-number$ nd seperator)
  (if (equal? (gi nd) (normalize "step"))
      (string-append
       (format-number
	(child-number nd) 
	($proc-hierarch-number-format$ ($proc-step-depth$ nd)))
       seperator)
      ""))

(define ($proc-step-depth$ nd)
  (let loop ((step nd) (depth 0))
    (if (equal? (gi step) (normalize "procedure"))
	depth
	(loop (parent step) 
	      (if (equal? (gi step) (normalize "step"))
		  (+ depth 1)
		  depth)))))

(define ($proc-step-number$ nd)
  (let* ((step (if (equal? (gi nd) (normalize "step")) nd (parent nd)))
	 (str ($proc-hierarch-number$ step "")))
    (string-append str (gentext-label-title-sep (normalize "step")))))

(define ($proc-step-xref-number$ nd)
  (let loop ((step nd) (str "") (first #t))
    (if (equal? (gi step) (normalize "procedure"))
	str
	(loop (parent step) 
	      (if (equal? (gi step) (normalize "step"))
		  (string-append 
		   ($proc-hierarch-number$ step
				      (if first
					  ""
					  (gentext-intra-label-sep (normalize "step"))))
		   str)
		  str)
	      (if (equal? (gi step) (normalize "step"))
		  #f
		  first)))))

;; === sections =========================================================

(define (SECTLEVEL #!optional (sect (current-node)))
  (if (equal? (gi sect) (normalize "simplesect"))
      ;; SimpleSect is special, it should be level "n+1", where "n" is
      ;; the level of the numbered section that contains it.  If it is
      ;; the *first* sectioning element in a chapter, make it 
      ;; %default-simplesect-level%
      (cond
       ((have-ancestor? (normalize "sect5")) 6)
       ((have-ancestor? (normalize "sect4")) 5)
       ((have-ancestor? (normalize "sect3")) 4)
       ((have-ancestor? (normalize "sect2")) 3)
       ((have-ancestor? (normalize "sect1")) 2)
       ((have-ancestor? (normalize "refsect3")) 5)
       ((have-ancestor? (normalize "refsect2")) 4)
       ((have-ancestor? (normalize "refsect1")) 3)
       (else %default-simplesect-level%))
      (cond
       ((equal? (gi sect) (normalize "sect5")) 5)
       ((equal? (gi sect) (normalize "sect4")) 4)
       ((equal? (gi sect) (normalize "sect3")) 3)
       ((equal? (gi sect) (normalize "sect2")) 2)
       ((equal? (gi sect) (normalize "sect1")) 1)
       ((equal? (gi sect) (normalize "refsect3")) 4)
       ((equal? (gi sect) (normalize "refsect2")) 3)
       ((equal? (gi sect) (normalize "refsect1")) 2)
       ((equal? (gi sect) (normalize "reference")) 1)
       (else 1))))

;; === synopsis =========================================================

;; The following definitions match those given in the reference
;; documentation for DocBook V3.0
(define	%arg-choice-opt-open-str% "[")
(define	%arg-choice-opt-close-str% "]")
(define	%arg-choice-req-open-str% "{")
(define	%arg-choice-req-close-str% "}")
(define	%arg-choice-plain-open-str% " ")
(define	%arg-choice-plain-close-str% " ")
(define	%arg-choice-def-open-str% "[")
(define	%arg-choice-def-close-str% "]")
(define	%arg-rep-repeat-str% "...")
(define	%arg-rep-norepeat-str% "")
(define	%arg-rep-def-str% "")
(define %arg-or-sep% " | ")
(define %cmdsynopsis-hanging-indent% 4pi)

;; === linking ==========================================================

;; From the DocBook V3.0 Reference entry for element XREF:
;;
;; Description
;;
;;   Cross reference link to another part of the document. XRef is empty,
;;   and has common, Linkend, and Endterm attributes.
;;
;;   Processing Expectations
;;
;;   XRef must have a Linkend, but the Endterm is optional. If it is used,
;;   the content of the element it points to is displayed as the text of
;;   the cross reference; if it is absent, the XRefLabel of the
;;   cross-referenced object is displayed.
;;
;; If neither the ENDTERM nor the XREFLABEL is present, then the cross
;; reference text is taken from the (gentext-xref-strings) function
;; in the localization file, like this
;; 
;; A cross reference to an element, the target, begins with the
;; text returned by (gentext-xref-strings (gi target)).  Within
;; that text, the following substitutions are made:
;; 
;; %p is replaced by the number of the page on which target occurs
;; %[x]g is replaced by the (gentext-element-name)
;; %[x]n is replaced by the label
;; %[x]t is replaced by the title
;; 
;; Where "x" is either absent, in which case the target is used, or
;; one of the following:
;; 
;; b = the ancestral book
;; c = the ancestral component
;; d = the ancestral division
;; k = the ancestral block
;; s = the ancestral section
;; 
;; So, %cn is replaced by the label (number) of the chapter that
;; contains target.  %st is replaced by the title of the section
;; that contains target, %g is replaced by the
;; (gentext-element-name) of the target, etc.
;; 
;; What elements constitute a book, component, division, block, and
;; section are defined by the lists *-element-list.
;; 
;; As if this wasn't confusing enough, _one_ additional level of
;; indirection is available.  If one of the special symbols, #b,
;; #c, #d, #k, or #s occurs in the (gentext-xref-strings) text,
;; then it will be replaced by the appropriate
;; (gentext-xref-strings-indirect) text depending on whether or not
;; the target and the reference occur in the same element.
;; 
;; Here's a concrete example:
;; 
;; Given
;; 
;; (define (gentext-xref-strings-indirect key)
;;   (case key
;;     (("k") '(" in %kg %kn" ""))
;;     (("s") '(" in %cg %cn" " in this %cg"))))
;; 
;; and
;; 
;; (define (gentext-xref-strings giname)
;;   (case giname
;;     (("STEP") "step %n#k")
;;     (("SECT1") "the section called %t#s")))
;; 
;; A cross reference to a step in the same procedure as the reference
;; will use the string "step %n" as the gentext-xref-strings text.
;; A cross reference to a step in another procedure will use the
;; string "step %n in %kg %kn".
;; 
;; So, a reference from step 5 to step 3 in the same procedure will
;; appear as "step 3".  A reference to step 6 in the third
;; procedure in the fourth chapter from some other place will
;; appear as "step 6 in procedure 4.3".
;; 
;; Likewise a reference to another section in the current chapter
;; will appear like this "the section called target-title in this
;; Chapter", and a reference to a section in an appendix will
;; appear like this "the section called target-title in Appendix
;; B".
;; 
;; ======================================================================

(define (auto-xref target)
  (let* ((cont-blok (ancestor-member target (block-element-list)))
	 (cont-sect (ancestor-member target (section-element-list)))
	 (cont-comp (ancestor-member target (component-element-list)))
	 (cont-divn (ancestor-member target (division-element-list)))
 	 (cont-book (ancestor-member target (book-element-list)))
	 (substitute (list
		      (list "%bg" (element-gi-sosofo cont-book))
		      (list "%bn" (element-label-sosofo cont-book))
		      (list "%bt" (element-title-xref-sosofo cont-book))
		      (list "%cg" (element-gi-sosofo cont-comp))
		      (list "%cn" (element-label-sosofo cont-comp))
		      (list "%ct" (element-title-xref-sosofo cont-comp))
		      (list "%dg" (element-gi-sosofo cont-divn))
		      (list "%dn" (element-label-sosofo cont-divn))
		      (list "%dt" (element-title-xref-sosofo cont-divn))
		      (list "%g"  (element-gi-sosofo target))
		      (list "%kg" (element-gi-sosofo cont-blok))
		      (list "%kn" (element-label-sosofo cont-blok))
		      (list "%kt" (element-title-xref-sosofo cont-blok))
		      (list "%n"  (element-label-sosofo target))
		      (list "%p"  (element-page-number-sosofo target))
		      (list "%sg" (element-gi-sosofo cont-sect))
		      (list "%sn" (element-label-sosofo cont-sect))
		      (list "%st" (element-title-xref-sosofo cont-sect))
		      (list "%t"  (element-title-xref-sosofo target))))
	 (text        (subst-xref-strings-indirect 
		       (gentext-xref-strings target)
		       (current-node)
		       target))
	 (tlist   (match-split-list text (assoc-objs substitute))))
    (make sequence
      (string-list-sosofo tlist substitute))))

;; ======================================================================

(define (subst-xref-strings-indirect string reference referent)
  (let* ((rnce-blok (ancestor-member reference (block-element-list)))
	 (rnce-sect (ancestor-member reference (section-element-list)))
	 (rnce-comp (ancestor-member reference (component-element-list)))
	 (rnce-divn (ancestor-member reference (division-element-list)))
 	 (rnce-book (ancestor-member reference (book-element-list)))

	 (rent-blok (ancestor-member referent (block-element-list)))
	 (rent-sect (ancestor-member referent (section-element-list)))
	 (rent-comp (ancestor-member referent (component-element-list)))
	 (rent-divn (ancestor-member referent (division-element-list)))
 	 (rent-book (ancestor-member referent (book-element-list)))

	 (title     (element-title-xref-sosofo referent))

	 (b (if (node-list-empty? rent-book)
		""
		(if (node-list=? rnce-book rent-book)
		    (car (cdr (gentext-xref-strings-indirect "b")))
		    (car (gentext-xref-strings-indirect "b")))))

	 (c (if (node-list-empty? rent-comp)
		""
		(if (node-list=? rnce-comp rent-comp)
		    (car (cdr (gentext-xref-strings-indirect "c")))
		    (car (gentext-xref-strings-indirect "c")))))

	 (d (if (node-list-empty? rent-divn)
		""
		(if (node-list=? rnce-divn rent-divn)
		    (car (cdr (gentext-xref-strings-indirect "d")))
		    (car (gentext-xref-strings-indirect "d")))))

	 (k (if (node-list-empty? rent-blok)
		""
		(if (node-list=? rnce-blok rent-blok)
		    (car (cdr (gentext-xref-strings-indirect "k")))
		    (car (gentext-xref-strings-indirect "k")))))

	 (s (if (node-list-empty? rent-sect)
		""
		(if (node-list=? rnce-sect rent-sect)
		    (car (cdr (gentext-xref-strings-indirect "s")))
		    (car (gentext-xref-strings-indirect "s"))))))

    (string-replace-list string 
			 (list "#b" b "#c" c "#d" d "#k" k "#s" s))))

;; ======================================================================

(define (set-autolabel nd #!optional (force-label? #f))
  "")

(define (book-autolabel nd #!optional (force-label? #f))
  "")

(define (part-autolabel nd #!optional (force-label? #f))
  (format-number (child-number nd) "I"))

(define (dedication-autolabel nd #!optional (force-label? #f))
  "")

(define (preface-autolabel nd #!optional (force-label? #f))
  "")

(define (article-autolabel nd #!optional (force-label? #f))
  "")

(define (component-number component-gi component-node)
  (let* ((book  (ancestor-member component-node  (list (normalize "book"))))
	 (comps (expand-children (children book) (list (normalize "part")))))
    (let loop ((nl comps) (count 1))
      (if (node-list-empty? nl) 
	  0
	  (if (node-list=? (node-list-first nl) component-node)
	      count
	      (if (equal? (gi (node-list-first nl))
			  (gi component-node))
		  (loop (node-list-rest nl) (+ count 1))
		  (loop (node-list-rest nl) count)))))))

(define (chapter-autolabel nd #!optional (force-label? #f))
  (if (or force-label? %chapter-autolabel%)
      (format-number (component-number "chapter" nd) "1")
      ""))

(define (appendix-autolabel nd #!optional (force-label? #f))
  (if (equal? (gi (parent nd)) (normalize "article"))
      ;; this is an appendix in an article...
      (if (or force-label? %chapter-autolabel%)
	  (string-append
	   (gentext-element-name-space nd)
	   (format-number (element-number nd) "A"))
	  "")
      ;; this is an appendix in a book...
      (if (or force-label? %chapter-autolabel%)
	  (format-number (component-number "appendix" nd) "A")
	  "")))

(define (bibliography-autolabel nd #!optional (force-label? #f))
  "")

(define (bibliodiv-autolabel nd #!optional (force-label? #f))
  "")

(define (glossary-autolabel nd #!optional (force-label? #f))
  "")

(define (glossdiv-autolabel nd #!optional (force-label? #f))
  "")

(define (index-autolabel nd #!optional (force-label? #f))
  "")

(define (setindex-autolabel nd #!optional (force-label? #f))
  "")

(define (reference-autolabel nd #!optional (force-label? #f))
  (format-number (child-number nd) "1"))

(define (refentry-autolabel nd #!optional (force-label? #f))
  (let* ((isep (gentext-intra-label-sep nd))
	 (refnamediv (select-elements (children nd) (normalize "refnamediv")))
	 (refd       (select-elements (children refnamediv) (normalize "refdescriptor")))
	 (refnames   (select-elements (children refnamediv) (normalize "refname"))))
    ""))
;;    (if (node-list-empty? refd)
;;	(if (node-list-empty? refnames)
;;	    ""
;;	    (data (node-list-first refnames)))
;;	(data refd))))

(define (section-autolabel nd #!optional (force-label? #f))
  (let* ((isep (gentext-intra-label-sep nd))
	 (haschn (not (node-list-empty? (ancestor (normalize "chapter") nd))))
	 (hasapn (not (node-list-empty? (ancestor (normalize "appendix") nd))))
	 (prefix (cond
		  (haschn (string-append 
			   (element-label (ancestor (normalize "chapter") nd)) isep))
		  (hasapn (string-append 
			   (element-label (ancestor (normalize "appendix") nd)) isep))
		  (else ""))))
    (if (or force-label? %section-autolabel%)
	(cond
	 ((equal? (gi nd) (normalize "sect1"))
	  (string-append prefix (format-number (child-number nd) "1")))
	 ((equal? (gi nd) (normalize "sect2"))
	  (string-append 
	   (element-label (ancestor (normalize "sect1") nd) force-label?)
	   isep 
	   (format-number (child-number nd) "1")))
	 ((equal? (gi nd) (normalize "sect3"))
	  (string-append
	   (element-label (ancestor (normalize "sect2") nd) force-label?)
	   isep 
	   (format-number (child-number nd) "1")))
	 ((equal? (gi nd) (normalize "sect4"))
	  (string-append
	   (element-label (ancestor (normalize "sect3") nd) force-label?)
	   isep 
	   (format-number (child-number nd) "1")))
	 ((equal? (gi nd) (normalize "sect5"))
	  (string-append 
	   (element-label (ancestor (normalize "sect4") nd) force-label?)
	   isep 
	   (format-number (child-number nd) "1")))
	 (else (string-append (gi nd) " IS NOT A SECTION!")))
	"")))
  
(define (refsection-autolabel nd #!optional (force-label? #f))
  ;; I've decided that enumerating RefSects never makes sense.
  "")

;; this code is all wrong: it's pre XML case normalization...
;; (define (refsection-autolabel nd #!optional (force-label? #f))
;;   (let* ((isep (gentext-intra-label-sep nd))
;; 	 (haschn (not (node-list-empty? (ancestor "CHAPTER" nd))))
;; 	 (hasapn (not (node-list-empty? (ancestor "APPENDIX" nd))))
;; 	 (prefix (string-append
;; 		  (element-label (ancestor "REFENTRY" nd) force-label?)
;; 		  isep)))
;;     (if (or force-label? %section-autolabel%)
;; 	(case (gi nd)
;; 	  (("REFSECT1") (string-append prefix 
;; 				       (format-number (child-number nd) "1")))
;; 	  (("REFSECT2") (string-append (element-label 
;; 					(ancestor "REFSECT1" nd) force-label?)
;; 				       isep 
;; 				       (format-number (child-number nd) "1")))
;; 	  (("REFSECT3") (string-append (element-label 
;; 					(ancestor "REFSECT2" nd) force-label?)
;; 				       isep 
;; 				       (format-number (child-number nd) "1")))
;; 	  (else (string-append (gi nd) " IS NOT A SECTION!")))
;; 	"")))

(define (step-autolabel nd #!optional (force-label? #f))
  ($proc-step-xref-number$ nd))

(define (listitem-autolabel nd #!optional (force-label? #f))
  (if (equal? (gi (parent nd)) (normalize "orderedlist"))
      (number->string (child-number nd))
      "[xref to LISTITEM only supported in ORDEREDLIST]"))

(define (sidebar-autolabel nd #!optional (force-label? #f))
  "")

(define (legalnotice-autolabel nd #!optional (force-label? #f))
  "")

(define (block-autolabel nd #!optional (force-label? #f))
  (let* ((chn (element-label (ancestor (normalize "chapter") nd)))
	 (apn (element-label (ancestor (normalize "appendix") nd)))
	 (rfn (element-label (ancestor (normalize "refentry") nd)))
	 (bkn (format-number (component-child-number nd (component-element-list)) "1")))
    (if (equal? chn "")
	(if (equal? apn "")
	    (if (equal? rfn "")
		bkn
		(string-append rfn (gentext-intra-label-sep nd) bkn))
	    (string-append apn (gentext-intra-label-sep nd) bkn))  
	(string-append chn (gentext-intra-label-sep nd) bkn))))

;; For all elements, if a LABEL attribute is present, that is the label
;; that they get.  Otherwise:
;; BOOK gets the Book volume, by book-autolabel
;; PREFACE gets "", by preface-autolabel
;; CHAPTER gets the Chapter number, by chapter-autolabel
;; APPENDIX gets the Appendix letter, by appendix-autolabel
;; REFERENCE gets "", by reference-autolabel
;; REFENTRY gets "", by refentry-autolabel
;; SECT* gets the nested section number (e.g., 1.3.5), by section-autolabel
;; REFSECT* gets the nested section number, by refsection-autolabel
;; everything else gets numbered by block-autolabel
;;
(define (element-label #!optional (nd (current-node)) (force-label? #f))
  (if (node-list-empty? nd)
      ""
      (let ((label (attribute-string (normalize "label") nd)))
	(if label
	    label
	    (cond
	     ;; Use a seperately defined assoc list?
	     ((equal? (gi nd) (normalize "appendix"))
	      (appendix-autolabel nd force-label?))
	     ((equal? (gi nd) (normalize "article"))
	      (article-autolabel nd force-label?))
	     ((equal? (gi nd) (normalize "bibliodiv"))
	      (bibliodiv-autolabel nd force-label?))
	     ((equal? (gi nd) (normalize "bibliography"))
	      (bibliography-autolabel nd force-label?))
	     ((equal? (gi nd) (normalize "book"))
	      (book-autolabel nd force-label?))
	     ((equal? (gi nd) (normalize "chapter"))
	      (chapter-autolabel nd force-label?))
	     ((equal? (gi nd) (normalize "dedication"))
	      (dedication-autolabel nd force-label?))
	     ((equal? (gi nd) (normalize "glossary"))
	      (glossary-autolabel nd force-label?))
	     ((equal? (gi nd) (normalize "glossdiv"))
	      (glossdiv-autolabel nd force-label?))
	     ((equal? (gi nd) (normalize "index"))
	      (index-autolabel nd force-label?))
	     ((equal? (gi nd) (normalize "setindex"))
	      (setindex-autolabel nd force-label?))
	     ((equal? (gi nd) (normalize "legalnotice"))
	      (legalnotice-autolabel nd force-label?))
	     ((equal? (gi nd) (normalize "listitem"))
	      (listitem-autolabel nd force-label?))
	     ((equal? (gi nd) (normalize "part"))
	      (part-autolabel nd force-label?))
	     ((equal? (gi nd) (normalize "preface"))
	      (preface-autolabel nd force-label?))
	     ((equal? (gi nd) (normalize "refentry"))
	      (refentry-autolabel nd force-label?))
	     ((equal? (gi nd) (normalize "reference"))
	      (reference-autolabel nd force-label?))
	     ((equal? (gi nd) (normalize "refsect1"))
	      (refsection-autolabel nd force-label?))
	     ((equal? (gi nd) (normalize "refsect2"))
	      (refsection-autolabel nd force-label?))
	     ((equal? (gi nd) (normalize "refsect3"))
	      (refsection-autolabel nd force-label?))
	     ((equal? (gi nd) (normalize "sect1"))
	      (section-autolabel nd force-label?))
	     ((equal? (gi nd) (normalize "sect2"))
	      (section-autolabel nd force-label?))
	     ((equal? (gi nd) (normalize "sect3"))
	      (section-autolabel nd force-label?))
	     ((equal? (gi nd) (normalize "sect4"))
	      (section-autolabel nd force-label?))
	     ((equal? (gi nd) (normalize "sect5"))
	      (section-autolabel nd force-label?))
	     ((equal? (gi nd) (normalize "set"))
	      (set-autolabel nd force-label?))
	     ((equal? (gi nd) (normalize "sidebar"))
	      (sidebar-autolabel nd force-label?))
	     ((equal? (gi nd) (normalize "simplesect"))
	      "")
	     ((equal? (gi nd) (normalize "step"))
	      (step-autolabel nd force-label?))
	     (else (block-autolabel nd force-label?)))))))

;; ======================================================================

;; Returns the element label as a sosofo
;;
(define (element-label-sosofo nd #!optional (force-label? #f))
  (if (string=? (element-label nd force-label?) "")
      (empty-sosofo)
      (make sequence
	(literal (element-label nd force-label?)))))

;; ======================================================================

(define (set-title-sosofo nd)
  (let* ((setinfo (select-elements (children nd) (normalize "setinfo")))
	 (sititles (select-elements  
		    (expand-children (children setinfo) 
				     (list (normalize "bookbiblio") 
					    (normalize "bibliomisc")
					    (normalize "biblioset")))
		    (normalize "title")))
	 (settitles (select-elements (children nd) (normalize "title")))
	 (titles   (if (node-list-empty? settitles)
		       sititles
		       settitles)))
    (if (node-list-empty? titles)
	(empty-sosofo)
	(with-mode xref-title-mode
	  (process-node-list (node-list-first titles))))))

(define (book-title-sosofo nd)
  (let* ((bookinfo (select-elements (children nd) (normalize "bookinfo")))
	 (bititles (select-elements  
		    (expand-children (children bookinfo) 
				     (list (normalize "bookbiblio") 
					   (normalize "bibliomisc")
					   (normalize "biblioset")))
		    (normalize "title")))
	 (chtitles (select-elements (children nd) (normalize "title")))
	 (titles   (if (node-list-empty? chtitles)
		       bititles
		       chtitles)))
    (if (node-list-empty? titles)
	(empty-sosofo)
	(with-mode xref-title-mode
	  (process-node-list (node-list-first titles))))))

(define (part-title-sosofo nd)
  (let* ((docinfo  (select-elements (children nd) (normalize "docinfo")))
	 (dititles (select-elements  
		    (expand-children (children docinfo) 
				     (list (normalize "bookbiblio") 
					   (normalize "bibliomisc")
					   (normalize "biblioset")))
		    (normalize "title")))
	 (chtitles (select-elements (children nd) (normalize "title")))
	 (titles   (if (node-list-empty? chtitles)
		       dititles
		       chtitles)))
    (if (node-list-empty? titles)
	(empty-sosofo)
	(with-mode xref-title-mode
	  (process-node-list (node-list-first titles))))))

(define (article-title-sosofo nd)
  (let* ((artheader (select-elements (children nd) (normalize "artheader")))
	 (titles (select-elements (children artheader) (normalize "title"))))
    (if (node-list-empty? titles)
	(empty-sosofo)
	(with-mode xref-title-mode
	  (process-node-list (node-list-first titles))))))

(define (preface-title-sosofo nd)
  (let* ((docinfo  (select-elements (children nd) (normalize "docinfo")))
	 (dititles (select-elements  
		    (expand-children (children docinfo) 
				     (list (normalize "bookbiblio") 
					   (normalize "bibliomisc")
					   (normalize "biblioset")))
		    (normalize "title")))
	 (chtitles (select-elements (children nd) (normalize "title")))
	 (titles   (if (node-list-empty? chtitles)
		       dititles
		       chtitles)))
    (if (node-list-empty? titles)
	(empty-sosofo)
	(with-mode xref-title-mode
	  (process-node-list (node-list-first titles))))))

(define (chapter-title-sosofo nd)
  (let* ((docinfo  (select-elements (children nd) (normalize "docinfo")))
	 (dititles (select-elements  
		    (expand-children (children docinfo) 
				     (list (normalize "bookbiblio") 
					   (normalize "bibliomisc")
					   (normalize "biblioset")))
		    (normalize "title")))
	 (chtitles (select-elements (children nd) (normalize "title")))
	 (titles   (if (node-list-empty? chtitles)
		       dititles
		       chtitles)))
    (if (node-list-empty? titles)
	(empty-sosofo)
	(with-mode xref-title-mode
	  (process-node-list (node-list-first titles))))))

(define (appendix-title-sosofo nd)
  (let* ((docinfo  (select-elements (children nd) (normalize "docinfo")))
	 (dititles (select-elements  
		    (expand-children (children docinfo) 
				     (list (normalize "bookbiblio") 
					   (normalize "bibliomisc")
					   (normalize "biblioset")))
		    (normalize "title")))
	 (chtitles (select-elements (children nd) (normalize "title")))
	 (titles   (if (node-list-empty? chtitles)
		       dititles
		       chtitles)))
    (if (node-list-empty? titles)
	(empty-sosofo)
	(with-mode xref-title-mode
	  (process-node-list (node-list-first titles))))))

(define (reference-title-sosofo nd)
  (let* ((docinfo  (select-elements (children nd) (normalize "docinfo")))
	 (dititles (select-elements  
		    (expand-children (children docinfo) 
				     (list (normalize "bookbiblio") 
					   (normalize "bibliomisc")
					   (normalize "biblioset")))
		    (normalize "title")))
	 (chtitles (select-elements (children nd) (normalize "title")))
	 (titles   (if (node-list-empty? chtitles)
		       dititles
		       chtitles)))
    (if (node-list-empty? titles)
	(empty-sosofo)
	(with-mode xref-title-mode
	  (process-node-list (node-list-first titles))))))

;; Returns either the REFENTRYTITLE or the first REFNAME.
;;
(define (refentry-title-sosofo nd)
  (let* ((refmeta (select-elements (descendants nd) (normalize "refmeta")))
	 (refttl  (select-elements (descendants refmeta) (normalize "refentrytitle")))
	 (refndiv (select-elements (descendants nd) (normalize "refnamediv")))
	 (refname (select-elements (descendants refndiv) (normalize "refname"))))
    (if (node-list-empty? refttl)
	(if (node-list-empty? refname)
	    (empty-sosofo)
	    (with-mode xref-title-mode
	      (process-node-list (node-list-first refname))))
	(with-mode xref-title-mode
	  (process-node-list (node-list-first refttl))))))

(define (optional-title-sosofo nd)
  (let* ((docinfo  (select-elements (children nd) (normalize "docinfo")))
	 (dititles (select-elements (children docinfo) (normalize "title")))
	 (chtitles (select-elements (children nd) (normalize "title")))
	 (titles   (if (node-list-empty? chtitles)
		       dititles
		       chtitles)))
    (if (node-list-empty? titles)
	(literal (gentext-element-name nd))
	(with-mode xref-title-mode
	  (process-node-list (node-list-first titles))))))

(define (glossary-title-sosofo nd)
  (optional-title-sosofo nd))

(define (bibliography-title-sosofo nd)
  (optional-title-sosofo nd))

(define (index-title-sosofo nd)
  (optional-title-sosofo nd))

(define (setindex-title-sosofo nd)
  (optional-title-sosofo nd))

(define (dedication-title-sosofo nd)
  (optional-title-sosofo nd))

(define (section-title-sosofo nd)
  (let* ((info     (select-elements (children nd) 
				    (list (normalize "sect1info")
					  (normalize "sect2info")
					  (normalize "sect3info")
					  (normalize "sect4info")
					  (normalize "sect5info"))))
	 (ititles  (select-elements (children info) (normalize "title")))
	 (ctitles  (select-elements (children nd) (normalize "title")))
	 (titles   (if (node-list-empty? ctitles)
		       ititles
		       ctitles)))
    (if (node-list-empty? titles)
	(empty-sosofo)
	(with-mode xref-title-mode
	  (process-node-list (node-list-first titles))))))

(define (refsection-title-sosofo nd)
  (let* ((info     (select-elements (children nd) 
				    (list (normalize "refsect1info")
					  (normalize "refsect2info") 
					  (normalize "refsect3info"))))
	 (ititles  (select-elements (children info) (normalize "title")))
	 (ctitles  (select-elements (children nd) (normalize "title")))
	 (titles   (if (node-list-empty? ctitles)
		       ititles
		       ctitles)))
    (if (node-list-empty? titles)
	(empty-sosofo)
	(with-mode xref-title-mode
	  (process-node-list (node-list-first titles))))))

(define (block-title-sosofo nd)
  (let ((titles (select-elements (children nd) (normalize "title"))))
    (if (node-list-empty? titles)
	(empty-sosofo)
	(with-mode xref-title-mode
	  (process-node-list (node-list-first titles))))))

;; Returns the title of the element as a sosofo.
;;
(define (element-title-sosofo #!optional (nd (current-node)))
  (if (node-list-empty? nd)
      (empty-sosofo)
      (cond
       ;; Use a seperately defined assoc list?
       ((equal? (gi nd) (normalize "appendix")) (appendix-title-sosofo nd))
       ((equal? (gi nd) (normalize "article")) (article-title-sosofo nd))
       ((equal? (gi nd) (normalize "bibliography")) (bibliography-title-sosofo nd))
       ((equal? (gi nd) (normalize "book")) (book-title-sosofo nd))
       ((equal? (gi nd) (normalize "chapter")) (chapter-title-sosofo nd))
       ((equal? (gi nd) (normalize "dedication")) (dedication-title-sosofo nd))
       ((equal? (gi nd) (normalize "glossary")) (glossary-title-sosofo nd))
       ((equal? (gi nd) (normalize "index")) (index-title-sosofo nd))
       ((equal? (gi nd) (normalize "setindex")) (index-title-sosofo nd))
       ((equal? (gi nd) (normalize "part")) (part-title-sosofo nd))
       ((equal? (gi nd) (normalize "preface")) (preface-title-sosofo nd))
       ((equal? (gi nd) (normalize "refentry")) (refentry-title-sosofo nd))
       ((equal? (gi nd) (normalize "reference")) (reference-title-sosofo nd))
       ((equal? (gi nd) (normalize "refsect1")) (refsection-title-sosofo nd))
       ((equal? (gi nd) (normalize "refsect2")) (refsection-title-sosofo nd))
       ((equal? (gi nd) (normalize "refsect3")) (refsection-title-sosofo nd))
       ((equal? (gi nd) (normalize "sect1")) (section-title-sosofo nd))
       ((equal? (gi nd) (normalize "sect2")) (section-title-sosofo nd))
       ((equal? (gi nd) (normalize "sect3")) (section-title-sosofo nd))
       ((equal? (gi nd) (normalize "sect4")) (section-title-sosofo nd))
       ((equal? (gi nd) (normalize "sect5")) (section-title-sosofo nd))
       ((equal? (gi nd) (normalize "set")) (set-title-sosofo nd))
       (else (block-title-sosofo nd)))))

;; ======================================================================

(define (set-title nd)
  (let* ((setinfo (select-elements (children nd) (normalize "setinfo")))
	 (titles (node-list 
		  (if setinfo
		      (select-elements (children setinfo) (normalize "title"))
		      (empty-sosofo))
		  (select-elements (children nd) (normalize "title")))))
    (if (node-list-empty? titles)
	""
	(data (node-list-first titles)))))

(define (book-title nd)
  (let* ((bookinfo (select-elements (children nd) (normalize "bookinfo")))
	 (titles (node-list 
		  (if bookinfo
		      (select-elements (children bookinfo) (normalize "title"))
		      (empty-sosofo))
		  (select-elements (children nd) (normalize "title")))))
    (if (node-list-empty? titles)
	""
	(data (node-list-first titles)))))

(define (part-title nd)
  (let ((titles (select-elements (children nd) (normalize "title"))))
    (if (node-list-empty? titles)
	""
	(data (node-list-first titles)))))

(define (article-title nd)
  (let* ((artheader (select-elements (children nd) (normalize "artheader")))
	 (titles (select-elements (children artheader) (normalize "title"))))
    (if (node-list-empty? titles)
	""
	(data (node-list-first titles)))))

(define (preface-title nd)
  (let ((titles (select-elements (children nd) (normalize "title"))))
    (if (node-list-empty? titles)
	""
	(data (node-list-first titles)))))

(define (chapter-title nd)
  (let ((titles (select-elements (children nd) (normalize "title"))))
    (if (node-list-empty? titles)
	""
	(data (node-list-first titles)))))

(define (appendix-title nd)
  (let ((titles (select-elements (children nd) (normalize "title"))))
    (if (node-list-empty? titles)
	""
	(data (node-list-first titles)))))

(define (reference-title nd)
  (let ((titles (select-elements (children nd) (normalize "title"))))
    (if (node-list-empty? titles)
	""
	(data (node-list-first titles)))))

;; Returns either the REFENTRYTITLE or the first REFNAME.
;;
(define (refentry-title nd)
  (let* ((refmeta (select-elements (descendants nd) (normalize "refmeta")))
	 (refttl  (select-elements (descendants refmeta) (normalize "refentrytitle")))
	 (refndiv (select-elements (descendants nd) (normalize "refnamediv")))
	 (refname (select-elements (descendants refndiv) (normalize "refname"))))
    (if (node-list-empty? refttl)
	(if (node-list-empty? refname)
	    ""
	    (data (node-list-first refname)))
	(data (node-list-first refttl)))))

(define (bibliography-title nd)
  (let ((titles (select-elements (children nd) (normalize "title"))))
    (if (node-list-empty? titles)
	(gentext-element-name nd)
	(data (node-list-first titles)))))

(define (glossary-title nd)
  (let ((titles (select-elements (children nd) (normalize "title"))))
    (if (node-list-empty? titles)
	(gentext-element-name nd)
	(data (node-list-first titles)))))

(define (index-title nd)
  (let ((titles (select-elements (children nd) (normalize "title"))))
    (if (node-list-empty? titles)
	(gentext-element-name nd)
	(data (node-list-first titles)))))

(define (setindex-title nd)
  (let ((titles (select-elements (children nd) (normalize "title"))))
    (if (node-list-empty? titles)
	(gentext-element-name nd)
	(data (node-list-first titles)))))

(define (dedication-title nd)
  (let ((titles (select-elements (children nd) (normalize "title"))))
    (if (node-list-empty? titles)
	(gentext-element-name nd)
	(data (node-list-first titles)))))

(define (section-title nd)
  (let ((titles (select-elements (children nd) (normalize "title"))))
    (if (node-list-empty? titles)
	""
	(data (node-list-first titles)))))

(define (refsection-title nd)
  (let ((titles (select-elements (children nd) (normalize "title"))))
    (if (node-list-empty? titles)
	""
	(data (node-list-first titles)))))

(define (block-title nd)
  (let ((titles (select-elements (children nd) (normalize "title"))))
    (if (node-list-empty? titles)
	""
	(data (node-list-first titles)))))

;; Returns the data of the title of the element
(define (element-title nd)
  (if (node-list-empty? nd)
      ""
      (cond
       ;; Use a seperately defined assoc list?
       ((equal? (gi nd) (normalize "appendix")) (appendix-title nd))
       ((equal? (gi nd) (normalize "article")) (article-title nd))
       ((equal? (gi nd) (normalize "bibliography")) (bibliography-title nd))
       ((equal? (gi nd) (normalize "book")) (book-title nd))
       ((equal? (gi nd) (normalize "chapter")) (chapter-title nd))
       ((equal? (gi nd) (normalize "dedication")) (dedication-title nd))
       ((equal? (gi nd) (normalize "glossary")) (glossary-title nd))
       ((equal? (gi nd) (normalize "index")) (index-title nd))
       ((equal? (gi nd) (normalize "setindex")) (setindex-title nd))
       ((equal? (gi nd) (normalize "part")) (part-title nd))
       ((equal? (gi nd) (normalize "preface")) (preface-title nd))
       ((equal? (gi nd) (normalize "refentry")) (refentry-title nd))
       ((equal? (gi nd) (normalize "reference")) (reference-title nd))
       ((equal? (gi nd) (normalize "refsect1")) (refsection-title nd))
       ((equal? (gi nd) (normalize "refsect2")) (refsection-title nd))
       ((equal? (gi nd) (normalize "refsect3")) (refsection-title nd))
       ((equal? (gi nd) (normalize "sect1")) (section-title nd))
       ((equal? (gi nd) (normalize "sect2")) (section-title nd))
       ((equal? (gi nd) (normalize "sect3")) (section-title nd))
       ((equal? (gi nd) (normalize "sect4")) (section-title nd))
       ((equal? (gi nd) (normalize "sect5")) (section-title nd))
       ((equal? (gi nd) (normalize "set")) (set-title nd))
       (else (block-title nd)))))

;; ======================================================================
;; Returns the element gi as a sosofo
;;
(define (element-gi-sosofo nd)
  (if (node-list-empty? nd)
      (empty-sosofo)
      (make sequence
	(literal (gentext-element-name nd)))))

;; ======================================================================

(define (titlepage-info-elements node info #!optional (intro (empty-node-list)))
  ;; Returns a node-list of the elements that might appear on a title
  ;; page.  This node-list is constructed as follows:
  ;;
  ;; 1. The "title" child of node is considered as a possibility
  ;; 2. If info is not empty, then node-list starts as the children
  ;;    of info.  If the children of info don't include a title, then
  ;;    the title from the node is added.
  ;; 3. If info is empty, then node-list starts as the children of node,
  ;;    but with "partintro" filtered out.

  (let* ((title (select-elements (children node) (normalize "title")))
	 (nl    (if (node-list-empty? info)
		    (node-list-filter-by-not-gi (children node) 
						(list (normalize "partintro")))
		    (children info)))
	 (nltitle (node-list-filter-by-gi nl (list (normalize "title")))))
    (if (node-list-empty? info)
	(node-list nl
		   intro)
	(node-list (if (node-list-empty? nltitle)
		       title
		       (empty-node-list))
		   nl
		   intro))))

;; ======================================================================

(define (info-element #!optional (nd (current-node)))
  ;; Returns the *INFO element for the nd or (empty-node-list) if no
  ;; such node exists...
  (cond
   ((equal? (gi nd) (normalize "set"))
    (select-elements (children nd) (normalize "setinfo")))
   ((equal? (gi nd) (normalize "book"))
    (select-elements (children nd) (normalize "bookinfo")))
   ((equal? (gi nd) (normalize "sect1"))
    (select-elements (children nd) (normalize "sect1info")))
   ((equal? (gi nd) (normalize "sect2"))
    (select-elements (children nd) (normalize "sect2info")))
   ((equal? (gi nd) (normalize "sect3"))
    (select-elements (children nd) (normalize "sect3info")))
   ((equal? (gi nd) (normalize "sect4"))
    (select-elements (children nd) (normalize "sect4info")))
   ((equal? (gi nd) (normalize "sect5"))
    (select-elements (children nd) (normalize "sect5info")))
   ((equal? (gi nd) (normalize "refsect1")) 
    (select-elements (children nd) (normalize "refsect1info")))
   ((equal? (gi nd) (normalize "refsect2")) 
    (select-elements (children nd) (normalize "refsect2info")))
   ((equal? (gi nd) (normalize "refsect3")) 
    (select-elements (children nd) (normalize "refsect3info")))
   ((equal? (gi nd) (normalize "refsynopsisdiv"))
    (select-elements (children nd) (normalize "refsynopsisdivinfo")))
   (else ;; BIBLIODIV, GLOSSDIV, INDEXDIV, PARTINTRO, SIMPLESECT
    (select-elements (children nd) (normalize "docinfo")))))

;; ======================================================================
