/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 *  Routines for control of ICS2101 and CS4231 (mixer part only) chips
 */

#include "driver.h"

#define CURRENT_MIX( a ) \
  ( ( card -> mixer.curr[ a ].left & 0xff ) | \
    ( ( card -> mixer.curr[ a ].right & 0xff ) << 8 ) )

static int mixer_set_recsrc( gus_card_t *card, int src );
#ifdef GUSCFG_CODEC
static void codec_set_source( gus_card_t *card, short left, short right );
static void codec_set_dev( gus_card_t *card, unsigned char dev, int left, int right );
#endif
#ifdef GUSCFG_ICSMIX
static void ics_set_dev( gus_card_t *card, unsigned char dev, int left, int right );
#endif
#ifdef GUSCFG_ESS
static void ess_set_dev( gus_card_t *card, unsigned char dev, int left, int right );
#endif

#ifdef SOUND_MIXER_INFO

static struct mixer_info gus_mixer_info = {
  "GUS",
  "Gravis UltraSound"
};

#endif

void gus_mixer_init( gus_card_t *card )
{
  card -> mixer.codec_mute_output = 1;
#ifdef GUSCFG_GF1PCM
  {
    int i;

    for ( i = 0; i < GUS_VOICES; i++ )
      {
        card -> gf1.pcm_volume[ i ] = 128;	/* max volume */
        card -> gf1.pcm_pan[ i ] = 64;		/* middle position */
      }
  }
#endif
} 

void mixer_init( gus_card_t *card )
{
  short i;

  if ( card -> ics_flag || card -> codec_flag || card -> ess_flag )	
    {
      card -> mixer.mix_devs =
      		SOUND_MASK_MIC | SOUND_MASK_LINE |
    	        SOUND_MASK_SYNTH | SOUND_MASK_PCM | SOUND_MASK_SPEAKER |
    	        SOUND_MASK_CD | SOUND_MASK_VOLUME |
    	        SOUND_MASK_IMIX;
      card -> mixer.mix_stereo_devs =
      		card -> mixer.mix_devs & ~SOUND_MASK_IMIX;
      card -> mixer.mix_rec_devs = SOUND_MASK_MIC | SOUND_MASK_LINE;
#ifdef GUSCFG_CODEC
      if ( card -> codec_flag ) 
        {
          if ( !card -> pnp_flag )
            {
              card -> mixer.mix_devs &= ~SOUND_MASK_VOLUME;
              card -> mixer.mix_stereo_devs &= ~SOUND_MASK_MIC;
            }
           else
            {
              if ( card -> gf1.memory )
                card -> mixer.mix_devs |= SOUND_MASK_LINE2;
              card -> mixer.mix_rec_devs |= SOUND_MASK_VOLUME;
            }
          card -> mixer.mix_devs |= SOUND_MASK_RECLEV | SOUND_MASK_LINE1;
          if ( card -> use_codec )
            card -> mixer.mix_rec_devs |= SOUND_MASK_SYNTH | SOUND_MASK_CD;
        }
#endif
#ifdef GUSCFG_ESS
      if ( card -> ess_flag )
        {
          card -> mixer.mix_rec_devs |= SOUND_MASK_CD;
        }
#endif
    }
   else
    {
      if ( card -> version == GUS_CARD_VERSION_ACE )
        {
          card -> mixer.mix_devs = SOUND_MASK_SYNTH | SOUND_MASK_PCM;
          card -> mixer.mix_rec_devs = 0;
        }
       else
        {
          card -> mixer.mix_devs =
      		SOUND_MASK_MIC | SOUND_MASK_LINE | 
    	        SOUND_MASK_SYNTH | SOUND_MASK_PCM;
          card -> mixer.mix_rec_devs = SOUND_MASK_MIC | SOUND_MASK_LINE;
        }
      card -> mixer.mix_stereo_devs = SOUND_MASK_PCM;
    }
  card -> mixer.mix_ctrl_reg &= 0xf8;
  card -> mixer.mix_ctrl_reg |= 0x01;
		/* disable MIC IN & LINE IN, enable LINE OUT */
  if ( card -> ics_flag || card -> codec_flag || card -> ess_flag )
    {
      card -> mixer.mix_ctrl_reg &= ~1;	/* enable LINE IN - mixer overwrite this */
      card -> mixer.mix_ctrl_reg |= 4;	/* enable MIC IN - mixer overwrite this */
    }
  OUTB( card -> mixer.mix_ctrl_reg, GUSP( card, MIXCNTRLREG ) );
  OUTB( card -> gf1.active_voice = 0, GUSP( card, GF1PAGE ) );
#ifdef GUSCFG_CODEC
  if ( card -> codec_flag )	/* CODEC */
    {
      struct GUS_STRU_MIX *mix;
      static int cdc_devices[] = { CODEC_MIC_INPUT, CODEC_LINE_INPUT, CODEC_CD_INPUT, CODEC_GF1_INPUT, CODEC_OUTPUT, CODEC_INPUT, CODEC_LOOPBACK, -1 };
      static int drv_devices[] = { MIX_MIC, MIX_LINE, MIX_CD, MIX_GF1, MIX_PCM, MIX_GAIN, MIX_LOOPBACK, -1 };
      
      for ( i = 0; cdc_devices[ i ] >= 0; i++ )
        {
          mix = &card -> mixer.curr[ drv_devices[ i ] ];
          codec_set_mute( card, cdc_devices[ i ], mix -> mute & MIX_MUTE_LEFT, mix -> mute & MIX_MUTE_RIGHT );
          codec_set_dev( card, cdc_devices[ i ], mix -> left, mix -> right );
        }
      if ( card -> pnp_flag )
        {
          codec_out( card, CODEC_MIC_INPUT, 0xc0 );
          mix = &card -> mixer.curr[ MIX_MASTER ];
          codec_set_mute( card, CODEC_MASTER_OUTPUT, mix -> mute & MIX_MUTE_LEFT, mix -> mute & MIX_MUTE_RIGHT );
          codec_set_dev( card, CODEC_MASTER_OUTPUT, mix -> left, mix -> right );
        }
      codec_set_source( card, CODEC_MIXS_LINE, CODEC_MIXS_LINE );
    }
#endif
#ifdef GUSCFG_ICSMIX
  if ( !card -> codec_flag && card -> ics_flag )	/* ICS mixer */
    {
      struct GUS_STRU_MIX *mix;
      static int ics_devices[] = { ICS_MIC_DEV, ICS_LINE_DEV, ICS_CD_DEV, ICS_GF1_DEV, ICS_MASTER_DEV, -1 };
      static int drv_devices[] = { MIX_MIC, MIX_LINE, MIX_CD, MIX_GF1, MIX_MASTER, -1 };

      ics_set_dev( card, ICS_NONE_DEV, 0, 0 );
      for ( i = 0; ics_devices[ i ] >= 0; i++ )
        {
          mix = &card -> mixer.curr[ drv_devices[ i ] ];
          ics_set_dev( card, ics_devices[ i ], mix -> left, mix -> right );
        }
    }
#endif
#ifdef GUSCFG_ESS
  if ( card -> ess_flag )
    {
      struct GUS_STRU_MIX *mix;
      static int ess_devices[] = { ESS_MIC_DEV, ESS_LINE_DEV, ESS_CD_DEV, ESS_GF1_DEV, ESS_MASTER_DEV, ESS_PCM_DEV, -1 };
      static int drv_devices[] = { MIX_MIC, MIX_LINE, MIX_CD, MIX_GF1, MIX_MASTER, MIX_PCM, -1 };
      int i;

      ess_set_dev( card, ESS_FM_DEV, 0, 0 );
      ess_set_dev( card, ESS_SPEAKER_DEV, 0, 0 );
      for ( i = 0; ess_devices[ i ] >= 0; i++ )
        {
          mix = &card -> mixer.curr[ drv_devices[ i ] ];
          ess_set_dev( card, ess_devices[ i ], mix -> left, mix -> right );
        }
    }
#endif
  card -> mixer.mix_rec_src = 0;
  mixer_set_recsrc( card, 0 );
}

static void mixer_update_gf1_master( gus_card_t *card )
{
  int i, old_level;

  old_level = card -> mixer.soft_volume_level;
  if ( (card -> mixer.curr[ MIX_GF1_MASTER ].mute & MIX_MUTE) != MIX_MUTE )
    card -> mixer.soft_volume_level = 0;
   else
    card -> mixer.soft_volume_level = 
    	( ( card -> mixer.curr[ MIX_GF1_MASTER ].left +
    	    card -> mixer.curr[ MIX_GF1_MASTER ].right ) * 127L ) / 200L;
#if 0
  printk( "soft_volume_level = %i\n", card -> mixer.soft_volume_level );
#endif
  if ( old_level != card -> mixer.soft_volume_level )
    for ( i = 0; i < GF1_VOICE_RANGES; i++ )
      if ( card -> gf1.voice_ranges[ i ].volume_change )
        card -> gf1.voice_ranges[ i ].volume_change( card );
}

static void mixer_update_gf1_pcm( gus_card_t *card, int dev )
{
  int old_level_left, old_level_right;
  
  old_level_left = card -> mixer.pcm_volume_level_left;
  old_level_right = card -> mixer.pcm_volume_level_right;
  if ( card -> mixer.curr[ dev ].mute & MIX_MUTE_LEFT )
    card -> mixer.pcm_volume_level_left = ( card -> mixer.curr[ dev ].left * 127L ) / 100L;
   else
    card -> mixer.pcm_volume_level_left = 0;
  if ( card -> mixer.curr[ dev ].mute & MIX_MUTE_RIGHT )
    card -> mixer.pcm_volume_level_right = ( card -> mixer.curr[ dev ].right * 127L ) / 100L;
   else
    card -> mixer.pcm_volume_level_right = 0;
#ifdef GUSCFG_GF1PCM
  if ( old_level_left != card -> mixer.pcm_volume_level_left ||
       old_level_right != card -> mixer.pcm_volume_level_right )
    gus_gf1_pcm_volume_update( card );
#endif
}

static void mixer_update_gf1_effect( gus_card_t *card )
{
  int old_level, i;

  old_level = card -> mixer.effect_volume_level;
  if ( (card -> mixer.curr[ MIX_EFFECTS ].mute & MIX_MUTE) != MIX_MUTE )
    card -> mixer.effect_volume_level = 0;
   else
    card -> mixer.effect_volume_level = 
    	( ( card -> mixer.curr[ MIX_EFFECTS ].left +
    	    card -> mixer.curr[ MIX_EFFECTS ].right ) * 127L ) / 200L;
  if ( old_level != card -> mixer.effect_volume_level )
    for ( i = 0; i < GF1_VOICE_RANGES; i++ )
      if ( card -> gf1.voice_ranges[ i ].volume_change )
        card -> gf1.voice_ranges[ i ].volume_change( card );
}

#ifdef GUSCFG_CODEC

#if 0

static void codec_set_mic_gain( gus_card_t *card, short left, short right )
{
  codec_outm( card, CODEC_LEFT_INPUT, ~0x20, left & 0x20 );
  codec_outm( card, CODEC_RIGHT_INPUT, ~0x20, right & 0x20 );
}

#endif

static void codec_set_source( gus_card_t *card, short left, short right )
{
  codec_outm( card, CODEC_LEFT_INPUT, (unsigned char)~0xc0, (unsigned char)left & 0xc0 );
  codec_outm( card, CODEC_RIGHT_INPUT, (unsigned char)~0xc0, (unsigned char)right & 0xc0 );
}

void codec_set_mute( gus_card_t *card, unsigned char dev, short left, short right )
{
  short tmp, *mute;

  switch ( dev ) {
    case CODEC_MIC_INPUT:	/* follow */
    case CODEC_MICE_INPUT:	tmp = MIX_MIC;		break;
    case CODEC_LINE_INPUT:	tmp = MIX_LINE;		break;
    case CODEC_CD_INPUT:	tmp = MIX_CD;		break;
    case CODEC_GF1_INPUT:	tmp = MIX_GF1;		break;
    case CODEC_INPUT:		tmp = MIX_GAIN;		break;
    case CODEC_MASTER_OUTPUT:	tmp = MIX_MASTER;	break;
    case CODEC_LOOPBACK:	tmp = MIX_LOOPBACK;	break;
    default:
      dev = CODEC_OUTPUT;
      tmp = MIX_PCM;
  }

  if ( card -> pnp_flag && dev == CODEC_MIC_INPUT ) dev = CODEC_MICE_INPUT;
 
  mute = &card -> mixer.curr[ tmp ].mute;
  if ( left >= 0 )
    {
      if ( !left )
        *mute &= ~MIX_MUTE_LEFT;
       else
        *mute |= MIX_MUTE_LEFT;
    }
  if ( right >= 0 )
    {
      if ( !right )
        *mute &= ~MIX_MUTE_RIGHT;
       else
        *mute |= MIX_MUTE_RIGHT;
    }
  
  if ( dev == CODEC_INPUT ) return;	/* I can't mute gain !!! */

  left = ( *mute & MIX_MUTE_LEFT ) ? 0x00 : 0x80;
  right = ( *mute & MIX_MUTE_RIGHT ) ? 0x00 : 0x80;

  if ( dev == CODEC_OUTPUT && card -> mixer.codec_mute_output )
    left = right = 0x80; /* !!! mute !!! */

#if 0
  PRINTK( "CODEC MUTE: dev=0x%x, left=%d, right=%d\n", dev, left, right );
#endif 

  switch ( dev ) {
    case CODEC_MIC_INPUT:
      codec_outm( card, dev + 0, 0x7f, left );
      break;
    case CODEC_MASTER_OUTPUT:
      codec_outm( card, dev + 0, 0x7f, left );
      codec_outm( card, dev + 2, 0x7f, right );
      break;
    case CODEC_LOOPBACK:
      codec_outm( card, dev + 0, 0xfe, ( left >> 7 ) ^ 0x01 );
      break;
    default:
      codec_outm( card, dev + 0, 0x7f, left );
      codec_outm( card, dev + 1, 0x7f, right );
  }
}

static void codec_set_dev( gus_card_t *card, unsigned char dev, int left, int right )
{
  short tmp;

  if ( left < 0 ) left = 0;
  if ( right < 0 ) right = 0;
  if ( left > 100 ) left = 100;
  if ( right > 100 ) right = 100;

  switch ( dev ) {
    case CODEC_MIC_INPUT:	/* follow */
    case CODEC_MICE_INPUT:	tmp = MIX_MIC; 		break;
    case CODEC_LINE_INPUT:	tmp = MIX_LINE;		break;
    case CODEC_CD_INPUT:	tmp = MIX_CD;		break;
    case CODEC_GF1_INPUT:	tmp = MIX_GF1;		break;
    case CODEC_INPUT:		tmp = MIX_GAIN;		break;
    case CODEC_MASTER_OUTPUT:	tmp = MIX_MASTER;	break;
    case CODEC_LOOPBACK:	tmp = MIX_LOOPBACK;	break;
    default:			
      dev = CODEC_OUTPUT; 
      tmp = MIX_PCM;
  }
  
  if ( card -> pnp_flag && dev == CODEC_MIC_INPUT ) dev = CODEC_MICE_INPUT;
  
  card -> mixer.curr[ tmp ].left = left;
  card -> mixer.curr[ tmp ].right = right;

  left = ( ( 100 - left ) * 999 ) / 3125;
  right = ( ( 100 - right ) * 999 ) / 3125;

#if 0
  PRINTK( "CODEC: dev=0x%x, left=%d, right=%d\n", dev, left, right );
#endif 

  switch ( dev ) {
    case CODEC_MIC_INPUT:
      codec_outm( card, dev + 0, ~0x0f, left >> 1 );
      break;
    case CODEC_INPUT:
      codec_outm( card, dev + 0, ~0x0f, 15 - ( left >> 1 ) );
      codec_outm( card, dev + 1, ~0x0f, 15 - ( right >> 1 ) );
      break;
    case CODEC_MASTER_OUTPUT:
      codec_outm( card, dev + 0, ~0x1f, left );
      codec_outm( card, dev + 2, ~0x1f, right );
      break;
    case CODEC_LOOPBACK:
      codec_outm( card, dev + 0,  0x03, ( 62 - ( left << 1 ) ) << 2 );
      break;
    default:
      codec_outm( card, dev + 0, ~0x1f, left );
      codec_outm( card, dev + 1, ~0x1f, right );
  }

  tmp = card -> mixer.curr[ tmp ].mute;
  codec_set_mute( card, dev, tmp & MIX_MUTE_LEFT, tmp & MIX_MUTE_RIGHT );
}

#endif /* GUSCFG_CODEC */

#ifdef GUSCFG_ICSMIX

static void ics_set_mix( gus_card_t *card, unsigned char dev, int left, int right )
{
  unsigned long flags;
  short addr = dev << 3;

  /* ok.. ICS mixer have different scalling */
  left = ( ( 127 * left ) + 50 ) / 100;	
  right = ( ( 127 * right ) + 50 ) / 100;
  
  if ( card -> ics_flipped && ( dev == ICS_GF1_DEV || dev == ICS_MASTER_DEV ) )
    {
      short tmp;
    
      tmp = left; left = right; right = tmp;
    }
  
#if 0
  PRINTK( "dev = %i, left = %i, right = %i\n", dev, left, right );
#endif
  CLI( &flags );
  OUTB( addr | 0, GUSP( card, MIXCNTRLPORT ) );
  OUTB( 1, GUSP( card, MIXDATAPORT ) );
  OUTB( addr | 2, GUSP( card, MIXCNTRLPORT ) );
  OUTB( (unsigned char)left, GUSP( card, MIXDATAPORT ) );
  OUTB( addr | 1, GUSP( card, MIXCNTRLPORT ) );
  OUTB( 2, GUSP( card, MIXDATAPORT ) );
  OUTB( addr | 3, GUSP( card, MIXCNTRLPORT ) );
  OUTB( (unsigned char)right, GUSP( card, MIXDATAPORT ) );
  STI( &flags );
}

static void ics_set_mute( gus_card_t *card, unsigned char dev, int left, int right )
{
  short tmp, tmp1;

  switch ( dev ) {
    case ICS_NONE_DEV:	 ics_set_mix( card, dev, left ? 0 : 100, right ? 0 : 100 ); return;
    case ICS_MIC_DEV:    tmp = MIX_MIC; 	break;
    case ICS_LINE_DEV:   tmp = MIX_LINE; 	break;
    case ICS_CD_DEV:     tmp = MIX_CD; 		break;
    case ICS_GF1_DEV:    tmp = MIX_GF1; 	break;
    default:
      dev = ICS_MASTER_DEV;
      tmp = MIX_MASTER;
  }
  
  tmp1 = card -> mixer.curr[ tmp ].mute;
  if ( left >= 0 )
    {
      if ( !left )
        tmp1 &= ~MIX_MUTE_LEFT;
       else
        tmp1 |= MIX_MUTE_LEFT;
    }
  if ( right >= 0 )
    {
      if ( !right )
        tmp1 &= ~MIX_MUTE_RIGHT;
       else
        tmp1 |= MIX_MUTE_RIGHT;
    }
  card -> mixer.curr[ tmp ].mute = tmp1;

  ics_set_mix( card, dev,
               (tmp1 & MIX_MUTE_LEFT) ? card -> mixer.curr[ tmp ].left : 0,
               (tmp1 & MIX_MUTE_RIGHT) ? card -> mixer.curr[ tmp ].right : 0 );
}

static void ics_set_dev( gus_card_t *card, unsigned char dev, int left, int right )
{
  short tmp;

  if ( left < 0 ) left = 0;
  if ( right < 0 ) right = 0;
  if ( left > 100 ) left = 100;
  if ( right > 100 ) right = 100;
  
  switch ( dev ) {
    case ICS_NONE_DEV:	 ics_set_mix( card, dev, left, right ); return;
    case ICS_MIC_DEV:    tmp = MIX_MIC; 	break;
    case ICS_LINE_DEV:   tmp = MIX_LINE; 	break;
    case ICS_CD_DEV:     tmp = MIX_CD; 		break;
    case ICS_GF1_DEV:    tmp = MIX_GF1; 	break;
    default:
      dev = ICS_MASTER_DEV;
      tmp = MIX_MASTER;
  }
  card -> mixer.curr[ tmp ].left = left;
  card -> mixer.curr[ tmp ].right = right;
  
  tmp = card -> mixer.curr[ tmp ].mute;
  ics_set_mix( card, dev,
               (tmp & MIX_MUTE_LEFT) ? left : 0,
               (tmp & MIX_MUTE_RIGHT) ? right : 0 );
}

#endif /* GUSCFG_ICSMIX */

#ifdef GUSCFG_ESS

static void ess_set_mix( gus_card_t *card, unsigned char dev, int left, int right )
{
  if ( dev == ESS_SPEAKER_DEV ) { 
    left += right;
    left = ( ( 7 * left ) + 50 ) / 100;
  } else {
    left = ( ( 15 * left ) + 50 ) / 100;
    right = ( ( 15 * right ) + 50 ) / 100;
    left <<= 4;
    left |= right;
  }
#if 0
  PRINTK( "gus: mixer - 0x%x -> 0x%x\n", dev, left );
#endif
  ess_mixer_write( card, dev, left );
}

static void ess_set_mute( gus_card_t *card, unsigned char dev, int left, int right )
{
  short tmp, tmp1;

  switch ( dev ) {
    case ESS_FM_DEV:	 ess_set_mix( card, dev, left ? 0 : 100, right ? 0 : 100 ); return;
    case ESS_SPEAKER_DEV: ess_set_mix( card, dev, left ? 0 : 100, right ? 0 : 100 ); return;
    case ESS_MIC_DEV:    tmp = MIX_MIC; 	break;
    case ESS_LINE_DEV:   tmp = MIX_LINE; 	break;
    case ESS_CD_DEV:     tmp = MIX_CD; 		break;
    case ESS_GF1_DEV:    tmp = MIX_GF1; 	break;
    case ESS_PCM_DEV:	 tmp = MIX_PCM;		break;
    default:
      dev = ESS_MASTER_DEV;
      tmp = MIX_MASTER;
  }
  
  tmp1 = card -> mixer.curr[ tmp ].mute;
  if ( left >= 0 )
    {
      if ( !left )
        tmp1 &= ~MIX_MUTE_LEFT;
       else
        tmp1 |= MIX_MUTE_LEFT;
    }
  if ( right >= 0 )
    {
      if ( !right )
        tmp1 &= ~MIX_MUTE_RIGHT;
       else
        tmp1 |= MIX_MUTE_RIGHT;
    }
  card -> mixer.curr[ tmp ].mute = tmp1;

  ess_set_mix( card, dev,
               (tmp1 & MIX_MUTE_LEFT) ? card -> mixer.curr[ tmp ].left : 0,
               (tmp1 & MIX_MUTE_RIGHT) ? card -> mixer.curr[ tmp ].right : 0 );
}

static void ess_set_dev( gus_card_t *card, unsigned char dev, int left, int right )
{
  short tmp;

  if ( left < 0 ) left = 0;
  if ( right < 0 ) right = 0;
  if ( left > 100 ) left = 100;
  if ( right > 100 ) right = 100;
  
  switch ( dev ) {
    case ESS_FM_DEV:	 ess_set_mix( card, dev, left, right ); return;
    case ESS_SPEAKER_DEV: ess_set_mix( card, dev, left, right ); return;
    case ESS_MIC_DEV:    tmp = MIX_MIC; 	break;
    case ESS_LINE_DEV:   tmp = MIX_LINE; 	break;
    case ESS_CD_DEV:     tmp = MIX_CD; 		break;
    case ESS_GF1_DEV:    tmp = MIX_GF1; 	break;
    case ESS_PCM_DEV:	 tmp = MIX_PCM;		break;
    default:
      dev = ESS_MASTER_DEV;
      tmp = MIX_MASTER;
  }
  card -> mixer.curr[ tmp ].left = left;
  card -> mixer.curr[ tmp ].right = right;
  
  tmp = card -> mixer.curr[ tmp ].mute;
  ess_set_mix( card, dev,
               (tmp & MIX_MUTE_LEFT) ? left : 0,
               (tmp & MIX_MUTE_RIGHT) ? right : 0 );
}

#endif /* GUSCFG_ICSMIX */

static int mixer_set_devs_mute( gus_card_t *card, int devs, int channel )
{
  static int mixer_set_recsrc( gus_card_t *card, int src );
  short codec_flag, ics_flag, ess_flag;
  
  codec_flag = card -> codec_flag;
  ics_flag = card -> ics_flag;
  ess_flag = card -> ess_flag;
  if ( devs == -1 )
    {
      unsigned int mix_devs;
      int result = 0;
      
      mix_devs = card -> mixer.mix_devs;
      channel = !channel ? MIX_MUTE_LEFT : MIX_MUTE_RIGHT;
      if ( (mix_devs & SOUND_MASK_MIC) && !(card -> mixer.curr[ MIX_MIC ].mute & channel) )
        result |= SOUND_MASK_MIC;
      if ( (mix_devs & SOUND_MASK_LINE) && !(card -> mixer.curr[ MIX_LINE ].mute & channel) )
        result |= SOUND_MASK_LINE;
      if ( (mix_devs & SOUND_MASK_PCM) && !(card -> mixer.curr[ MIX_PCM ].mute & channel) )
        result |= SOUND_MASK_PCM;
      if ( (mix_devs & SOUND_MASK_SPEAKER) && !(card -> mixer.curr[ MIX_PCM1 ].mute & channel) )
        result |= SOUND_MASK_SPEAKER;
      if ( (mix_devs & SOUND_MASK_CD) && !(card -> mixer.curr[ MIX_CD ].mute & channel) )
        result |= SOUND_MASK_CD;
      if ( ics_flag || codec_flag || ess_flag )
        {
          if ( (mix_devs & SOUND_MASK_SYNTH) && !(card -> mixer.curr[ MIX_GF1 ].mute & channel) )
            result |= SOUND_MASK_SYNTH;
          if ( (mix_devs & SOUND_MASK_IMIX) && !(card -> mixer.curr[ MIX_GF1_MASTER ].mute & channel) )
            result |= SOUND_MASK_IMIX;
        }
       else
        {
          if ( (mix_devs & SOUND_MASK_SYNTH) && !(card -> mixer.curr[ MIX_GF1_MASTER ].mute & channel) )
            result |= SOUND_MASK_SYNTH;
        }
      if ( (mix_devs & SOUND_MASK_VOLUME) && !(card -> mixer.curr[ MIX_MASTER ].mute & channel) )
        result |= SOUND_MASK_VOLUME;
      if ( (mix_devs & SOUND_MASK_RECLEV) && !(card -> mixer.curr[ MIX_GAIN ].mute & channel) )
        result |= SOUND_MASK_RECLEV;
      if ( (mix_devs & SOUND_MASK_LINE1) && !(card -> mixer.curr[ MIX_LOOPBACK ].mute & channel) )
        result |= SOUND_MASK_LINE1;
      if ( (mix_devs & SOUND_MASK_LINE2) && !(card -> mixer.curr[ MIX_EFFECTS ].mute & channel) )
        result |= SOUND_MASK_LINE2;
#if 0
      PRINTK( "mute result = 0x%x, channel = %i\n", result, channel );
#endif
      return result;
    }
   else
    {
      short left, right;
      int mute;

#if 0
      PRINTK( "mute devs = 0x%x, channel = %i\n", devs, channel );
#endif
      mute = ( devs & SOUND_MIXER_MUTE_FLAG ) ? 0 : 1;
      if ( !channel )
        {
          left = mute;
          right = -1;
        }
       else
        {
          left = -1;
          right = mute;
        }
#if 0
      PRINTK( "\n\n\n\nmute dev = 0x%x, mute left = %i, mute right = %i, channel = %i\n", devs, left, right, channel );
#endif
      devs &= ~SOUND_MIXER_MUTE_FLAG;
#ifdef GUSCFG_CODEC
      if ( codec_flag )
        switch ( devs ) {
          case SOUND_MIXER_MIC:
            codec_set_mute( card, CODEC_MIC_INPUT, left, right );
            break;
          case SOUND_MIXER_LINE:
            codec_set_mute( card, CODEC_LINE_INPUT, left, right );
            break;
          case SOUND_MIXER_PCM:
            codec_set_mute( card, CODEC_OUTPUT, left, right );
            break;
          case SOUND_MIXER_SPEAKER:
            if ( !mute )
              card -> mixer.curr[ MIX_PCM1 ].mute &= ~( !channel ? MIX_MUTE_LEFT : MIX_MUTE_RIGHT );
             else
              card -> mixer.curr[ MIX_PCM1 ].mute |= !channel ? MIX_MUTE_LEFT : MIX_MUTE_RIGHT;
            mixer_update_gf1_pcm( card, MIX_PCM1 );
            break;
          case SOUND_MIXER_CD:
            codec_set_mute( card, CODEC_CD_INPUT, left, right );
            break;
          case SOUND_MIXER_SYNTH:
            codec_set_mute( card, CODEC_GF1_INPUT, left, right );
            break;
          case SOUND_MIXER_VOLUME:
            if ( card -> pnp_flag )
              codec_set_mute( card, CODEC_MASTER_OUTPUT, left, right );
            break;
          case SOUND_MIXER_RECLEV:
            codec_set_mute( card, CODEC_INPUT, left, right );
            break;
          case SOUND_MIXER_IMIX:
            if ( !mute )
              card -> mixer.curr[ MIX_GF1_MASTER ].mute &= ~( !channel ? MIX_MUTE_LEFT : MIX_MUTE_RIGHT );
             else
              card -> mixer.curr[ MIX_GF1_MASTER ].mute |= !channel ? MIX_MUTE_LEFT : MIX_MUTE_RIGHT;
            mixer_update_gf1_master( card );
            break;
          case SOUND_MIXER_LINE1:
            codec_set_mute( card, CODEC_LOOPBACK, left, right );
            break;
          case SOUND_MIXER_LINE2:
            if ( !mute )
              card -> mixer.curr[ MIX_EFFECTS ].mute &= ~( !channel ? MIX_MUTE_LEFT : MIX_MUTE_RIGHT );
             else
              card -> mixer.curr[ MIX_EFFECTS ].mute |= !channel ? MIX_MUTE_LEFT : MIX_MUTE_RIGHT;
            mixer_update_gf1_effect( card );
            break;
          default:
            return -EINVAL;
        }
#endif
#ifdef GUSCFG_ICSMIX
      if ( ics_flag )
        switch ( devs ) {
          case SOUND_MIXER_MIC:
            ics_set_mute( card, ICS_MIC_DEV, left, right );
            break;
          case SOUND_MIXER_LINE:
            ics_set_mute( card, ICS_LINE_DEV, left, right );
            break;
          case SOUND_MIXER_PCM:
            if ( !mute )
              card -> mixer.curr[ MIX_PCM ].mute &= ~( !channel ? MIX_MUTE_LEFT : MIX_MUTE_RIGHT );
             else
              card -> mixer.curr[ MIX_PCM ].mute |= !channel ? MIX_MUTE_LEFT : MIX_MUTE_RIGHT;
            mixer_update_gf1_pcm( card, MIX_PCM );
            break;
          case SOUND_MIXER_CD:
            ics_set_mute( card, ICS_CD_DEV, left, right );
            break;
          case SOUND_MIXER_SYNTH:
            ics_set_mute( card, ICS_GF1_DEV, left, right );
            break;
          case SOUND_MIXER_VOLUME:
            ics_set_mute( card, ICS_MASTER_DEV, left, right );
            break;
          default:
            return -EINVAL;
        }
#endif
#ifdef GUSCFG_ESS
      if ( ess_flag )
        switch ( devs ) {
          case SOUND_MIXER_MIC:
            ess_set_mute( card, ESS_MIC_DEV, left, right );
            break;
          case SOUND_MIXER_LINE:
            ess_set_mute( card, ESS_LINE_DEV, left, right );
            break;
          case SOUND_MIXER_PCM:
            ess_set_mute( card, ESS_PCM_DEV, left, right );
            break;
          case SOUND_MIXER_SPEAKER:
            if ( !mute )
              card -> mixer.curr[ MIX_PCM1 ].mute &= ~( !channel ? MIX_MUTE_LEFT : MIX_MUTE_RIGHT );
             else
              card -> mixer.curr[ MIX_PCM1 ].mute |= !channel ? MIX_MUTE_LEFT : MIX_MUTE_RIGHT;
            mixer_update_gf1_pcm( card, MIX_PCM1 );
            break;
          case SOUND_MIXER_CD:
            ess_set_mute( card, ESS_CD_DEV, left, right );
            break;
          case SOUND_MIXER_SYNTH:
            ess_set_mute( card, ESS_GF1_DEV, left, right );
            break;
          case SOUND_MIXER_VOLUME:
            ess_set_mute( card, ESS_MASTER_DEV, left, right );
            break;
          case SOUND_MIXER_IMIX:
            if ( !mute )
              card -> mixer.curr[ MIX_GF1_MASTER ].mute &= ~( !channel ? MIX_MUTE_LEFT : MIX_MUTE_RIGHT );
             else
              card -> mixer.curr[ MIX_GF1_MASTER ].mute |= !channel ? MIX_MUTE_LEFT : MIX_MUTE_RIGHT;
            mixer_update_gf1_master( card );
            break;
          default:
            return -EINVAL;
        }
#endif
#ifdef GUSCFG_GUSSTD
      if ( !codec_flag && !ics_flag && !ess_flag )
        switch ( devs ) {
          case SOUND_MIXER_MIC:
          case SOUND_MIXER_LINE:
            if ( !mute )
              card -> mixer.curr[ devs == SOUND_MIXER_MIC ? MIX_MIC : MIX_LINE ].mute &= ~( !channel ? MIX_MUTE_LEFT : MIX_MUTE_RIGHT );
             else
              card -> mixer.curr[ devs == SOUND_MIXER_MIC ? MIX_MIC : MIX_LINE ].mute |= !channel ? MIX_MUTE_LEFT : MIX_MUTE_RIGHT;
            mixer_set_recsrc( card, card -> mixer.mix_rec_src );
            break;
          case SOUND_MIXER_PCM:
            if ( !mute )
              card -> mixer.curr[ MIX_PCM ].mute &= ~( !channel ? MIX_MUTE_LEFT : MIX_MUTE_RIGHT );
             else
              card -> mixer.curr[ MIX_PCM ].mute |= !channel ? MIX_MUTE_LEFT : MIX_MUTE_RIGHT;
            mixer_update_gf1_pcm( card, MIX_PCM );
            break;
          case SOUND_MIXER_SYNTH:
            left = !channel ? MIX_MUTE_LEFT : MIX_MUTE_RIGHT;
            if ( !mute )
              card -> mixer.curr[ MIX_GF1_MASTER ].mute &= ~left;
             else
              card -> mixer.curr[ MIX_GF1_MASTER ].mute |= left;
            mixer_update_gf1_master( card );
            break;
          default:
            return -EINVAL;
        }
#endif
      return 0;
    }
}

static int mixer_set_recsrc( gus_card_t *card, int src )
{
  unsigned int rec_src;

  if ( src >= 0 )
    {
      if ( !card -> ics_flag && !card -> codec_flag && !card -> ess_flag ) 
        {
#ifdef GUSCFG_GUSSTD
          unsigned long flags;
          
          src &= SOUND_MASK_MIC | SOUND_MASK_LINE;
          card -> mixer.mix_ctrl_reg &= ~0x07;
          card -> mixer.mix_ctrl_reg |= 1;	/* disable MIC & LINE IN */
          if ( src & SOUND_MASK_MIC )
            {
              if ( ( card -> mixer.curr[ MIX_MIC ].mute & MIX_MUTE ) == MIX_MUTE )
                card -> mixer.mix_ctrl_reg |= 4;	/* enable MIC IN */
            }
          if ( src & SOUND_MASK_LINE )
            {
              if ( ( card -> mixer.curr[ MIX_LINE ].mute & MIX_MUTE ) == MIX_MUTE )
                card -> mixer.mix_ctrl_reg &= ~1;	/* enable LINE IN */
            }
          CLI( &flags );
          OUTB( card -> mixer.mix_ctrl_reg, GUSP( card, MIXCNTRLREG ) );
          OUTB( card -> gf1.active_voice = 0, GUSP( card, GF1PAGE ) );
          STI( &flags );
#if 0
          printk( "mix_ctrl_reg = 0x%x\n", card -> mixer.mix_ctrl_reg );
#endif
          card -> mixer.mix_rec_src = src;
#endif
        }
       else
        {
          rec_src = src;
#ifdef GUSCFG_CODEC
          if ( card -> use_codec )
            {
              src = rec_src & SOUND_MASK_SYNTH ? CODEC_MIXS_GF1 : CODEC_MIXS_LINE;
              if ( rec_src & SOUND_MASK_MIC ) src |= CODEC_MIXS_MIC;
              if ( rec_src & SOUND_MASK_CD ) src = CODEC_MIXS_GF1 | CODEC_MIXS_MIC; 
              if ( rec_src & SOUND_MASK_VOLUME ) src = CODEC_MIXS_GF1 | CODEC_MIXS_MIC; 
              codec_set_source( card, src, src );
            }
#endif
#ifdef GUSCFG_ESS
          if ( card -> ess_flag )
            {
              unsigned char data;

              src = ESS_MIXS_NONE;
              if ( rec_src & SOUND_MASK_CD ) src = ESS_MIXS_CD;
              if ( rec_src & SOUND_MASK_LINE ) src = ESS_MIXS_LINE;
              if ( rec_src & SOUND_MASK_MIC ) src = ESS_MIXS_MIC;
              data = ess_mixer_read( card, ESS_RECORD_SOURCE );
              data &= ~7;
              ess_mixer_write( card, ESS_RECORD_SOURCE, data | src );
            }
#endif
          card -> mixer.mix_rec_src = rec_src;
        }
    }
  return card -> mixer.mix_rec_src;
}

static int mixer_set_mic( gus_card_t *card, int value )
{
  short left = value & 0xff;
  short right = ( value >> 8 ) & 0xff;
    
  if ( value >= 0 )
    { 
#ifdef GUSCFG_CODEC
      if ( card -> codec_flag )
        {
          if ( !card -> pnp_flag ) right = left;
          codec_set_dev( card, CODEC_MIC_INPUT, left, right );
        }
#endif
#ifdef GUSCFG_ICSMIX
      if ( card -> ics_flag ) 		/* ICS mixer */
        ics_set_dev( card, ICS_MIC_DEV, left, right );
#endif
#ifdef GUSCFG_ESS
      if ( card -> ess_flag ) 		/* ESS mixer */
        ess_set_dev( card, ESS_MIC_DEV, left, right );
#endif
#ifdef GUSCFG_GUSSTD
      if ( !card -> codec_flag && !card -> ics_flag && !card -> ess_flag )
        {
          unsigned int rec_src;
        
          left += right;
          left >>= 1;
          rec_src = card -> mixer.mix_rec_src;
          if ( left > 10 && ( rec_src & SOUND_MASK_MIC ) == 0 ) 
            mixer_set_recsrc( card, rec_src | SOUND_MASK_MIC );
          if ( left <= 10 && ( rec_src & SOUND_MASK_MIC ) != 0 )
            mixer_set_recsrc( card, rec_src & ~SOUND_MASK_MIC );
          card -> mixer.curr[ MIX_MIC ].left =
          card -> mixer.curr[ MIX_MIC ].right = left;          
        }
#endif
    }
  return CURRENT_MIX( MIX_MIC );
}

static int mixer_set_cd( gus_card_t *card, int value )
{
  short left = value & 0xff;
  short right = ( value >> 8 ) & 0xff;
  
  if ( value >= 0 )
    { 
#ifdef GUSCFG_CODEC
      if ( card -> codec_flag )		/* CODEC mixer */
        codec_set_dev( card, CODEC_CD_INPUT, left, right );
#endif
#ifdef GUSCFG_ICSMIX
      if ( card -> ics_flag )		/* ICS mixer */
        ics_set_dev( card, ICS_CD_DEV, left, right );
#endif
#ifdef GUSCFG_ESS
      if ( card -> ess_flag )		/* ESS mixer */
        ess_set_dev( card, ESS_CD_DEV, left, right );
#endif
#ifdef GUSCFG_STDGUS
      if ( !card -> codec_flag && !card -> ics_flag && !card -> ess_flag )
        return -EINVAL;
#endif
    }
  return CURRENT_MIX( MIX_CD );
}

static int mixer_set_line( gus_card_t *card, int value )
{
  short left = value & 0xff;
  short right = ( value >> 8 ) & 0xff;
  
  if ( value >= 0 )
    { 
#ifdef GUSCFG_CODEC
      if ( card -> codec_flag )		/* CODEC mixer */
        codec_set_dev( card, CODEC_LINE_INPUT, left, right );
#endif
#ifdef GUSCFG_ICSMIX
      if ( card -> ics_flag )		/* ICS mixer */
        ics_set_dev( card, ICS_LINE_DEV, left, right );
#endif
#ifdef GUSCFG_ESS
      if ( card -> ess_flag )		/* ESS mixer */
        ess_set_dev( card, ESS_LINE_DEV, left, right );
#endif
#ifdef GUSCFG_GUSSTD
      if ( !card -> codec_flag && !card -> ics_flag && !card -> ess_flag )
        {
          unsigned int rec_src;

          left += right;
          left >>= 1;
          rec_src = card -> mixer.mix_rec_src;
          if ( left > 10 && ( rec_src & SOUND_MASK_LINE ) == 0 ) 
            mixer_set_recsrc( card, rec_src | SOUND_MASK_LINE );
          if ( left < 10 && ( rec_src & SOUND_MASK_LINE ) != 0 )
            mixer_set_recsrc( card, rec_src & ~SOUND_MASK_LINE );
          card -> mixer.curr[ MIX_LINE ].left =
          card -> mixer.curr[ MIX_LINE ].right = left;
        }
#endif
    }
  return CURRENT_MIX( MIX_LINE );
}

static int mixer_set_gf1_master( gus_card_t *card, int value )
{
  short left = value & 0xff;
  short right = ( value >> 8 ) & 0xff;
  
  if ( value >= 0 )
    { 
      if ( left < 0 ) left = 0;
      if ( left > 100 ) left = 100;
      if ( right < 0 ) right = 0;
      if ( right > 100 ) right = 100;
      card -> mixer.curr[ MIX_GF1_MASTER ].left = left;
      card -> mixer.curr[ MIX_GF1_MASTER ].right = right;
      mixer_update_gf1_master( card );
    }
  return CURRENT_MIX( MIX_GF1_MASTER );
}

static int mixer_set_gf1( gus_card_t *card, int value )
{
  short left;
  short right;
  
  if ( !card -> codec_flag && !card -> ics_flag && !card -> ess_flag )
    return mixer_set_gf1_master( card, value );
  left = value & 0xff;
  right = ( value >> 8 ) & 0xff;
  if ( value >= 0 )
    { 
#ifdef GUSCFG_CODEC
      if ( card -> codec_flag )		/* CODEC mixer */
        codec_set_dev( card, CODEC_GF1_INPUT, left, right );
#endif
#ifdef GUSCFG_ICSMIX
      if ( card -> ics_flag )		/* ICS mixer */
        ics_set_dev( card, ICS_GF1_DEV, left, right );
#endif
#ifdef GUSCFG_ESS
      if ( card -> ess_flag )		/* ESS mixer */
        ess_set_dev( card, ESS_GF1_DEV, left, right );
#endif
#ifdef GUSCFG_STDGUS
      if ( !card -> codec_flag && !card -> ics_flag )
        return -EINVAL;
#endif
    }
  return CURRENT_MIX( MIX_GF1 );
}

static int mixer_set_pcm( gus_card_t *card, int value )
{
  short left = value & 0xff;
  short right = ( value >> 8 ) & 0xff;
  
  if ( value >= 0 )
    {
#ifdef GUSCFG_CODEC
      if ( card -> codec_flag )
        codec_set_dev( card, CODEC_OUTPUT, left, right );
#endif
#ifdef GUSCFG_ESS
      if ( card -> ess_flag )
        ess_set_dev( card, ESS_PCM_DEV, left, right );
#endif
      if ( !card -> codec_flag && !card -> ess_flag )
        { 
          card -> mixer.curr[ MIX_PCM ].left = left;
          card -> mixer.curr[ MIX_PCM ].right = right;
          mixer_update_gf1_pcm( card, MIX_PCM );
        }
    }
  return CURRENT_MIX( MIX_PCM );
}

static int mixer_set_pcm1( gus_card_t *card, int value )
{
  short left = value & 0xff;
  short right = ( value >> 8 ) & 0xff;
  
  if ( !card -> codec_flag && !card -> ess_flag ) return -ENODEV;
  if ( value >= 0 )
    {
      card -> mixer.curr[ MIX_PCM1 ].left = left;
      card -> mixer.curr[ MIX_PCM1 ].right = right;
      mixer_update_gf1_pcm( card, MIX_PCM1 );
    }
  return CURRENT_MIX( MIX_PCM1 );
}

static int mixer_set_gain( gus_card_t *card, int value )
{
#ifdef GUSCFG_CODEC
  short left = value & 0xff;
  short right = ( value >> 8 ) & 0xff;
  
  if ( value >= 0 )
    {
      if ( card -> codec_flag )
        codec_set_dev( card, CODEC_INPUT, left, right );
       else
        return -EINVAL;
    }
  return CURRENT_MIX( MIX_GAIN );
#else
  return -EINVAL;
#endif
}

static int mixer_set_master( gus_card_t *card, int value )
{
#if defined( GUSCFG_PNP ) || defined( GUSCFG_ICSMIX )
  short left = value & 0xff;
  short right = ( value >> 8 ) & 0xff;
  
  if ( value >= 0 )
    { 
#ifdef GUSCFG_PNP
      if ( card -> pnp_flag )
        codec_set_dev( card, CODEC_MASTER_OUTPUT, left, right );
#endif
#ifdef GUSCFG_ICSMIX
      if ( card -> ics_flag )		/* ICS mixer */
        ics_set_dev( card, ICS_MASTER_DEV, left, right );
#endif
#ifdef GUSCFG_ESS
      if ( card -> ess_flag )		/* ESS mixer */
        ess_set_dev( card, ESS_MASTER_DEV, left, right );
#endif
      if ( !card -> pnp_flag && !card -> ics_flag && !card -> ess_flag )
        return -EINVAL;
    }
  return CURRENT_MIX( MIX_MASTER );
#else
  return -EINVAL;
#endif
}

static int mixer_set_loopback( gus_card_t *card, int value )
{
#if defined( GUSCFG_CODEC )
  short left = value & 0xff;
  short right = ( value >> 8 ) & 0xff;
  
  if ( value >= 0 )
    { 
      left = right;
      codec_set_dev( card, CODEC_LOOPBACK, left, right );
    }
  return CURRENT_MIX( MIX_LOOPBACK );
#else
  return -EINVAL;
#endif
}

static int mixer_set_effect( gus_card_t *card, int value )
{
  short left = value & 0xff;
  short right = ( value >> 8 ) & 0xff;
  
  if ( value >= 0 )
    { 
      if ( left < 0 ) left = 0;
      if ( left > 100 ) left = 100;
      if ( right < 0 ) right = 0;
      if ( right > 100 ) right = 100;
      card -> mixer.curr[ MIX_EFFECTS ].left = left;
      card -> mixer.curr[ MIX_EFFECTS ].right = right;
      mixer_update_gf1_effect( card );
    }
  return CURRENT_MIX( MIX_EFFECTS );
}

static int gus_mixer_special_read( gus_card_t *card, struct GUS_MIXER_SPECIAL *special, int space )
{
  struct GUS_MIXER_SPECIAL sspecial, *old_special = NULL;

  if ( space == SP_USER )
    {
      if ( VERIFY_AREA( VERIFY_READ, special, sizeof( *special ) ) ) return -EIO;
      MEMCPY_FROMFS( &sspecial, old_special = special, sizeof( sspecial ) );
      special = &sspecial;
    }
  switch ( special -> what ) {
    case GUS_MIXER_S_NONE:
      return 0;
#ifdef GUSCFG_CODEC
    case GUS_MIXER_S_IW:
      special -> data.interwave.serial = card -> codec.interwave_serial_port >> 5;
      break;
#endif
    default:
      return -EINVAL;
  }
  if ( space == SP_USER ) 
    {
      if ( VERIFY_AREA( VERIFY_WRITE, old_special, sizeof( *special ) ) ) return -EIO;
      MEMCPY_TOFS( old_special, special, sizeof( sspecial ) );
    }
  return 0;
}

static int gus_mixer_special_write( gus_card_t *card, struct GUS_MIXER_SPECIAL *special, int space )
{
  struct GUS_MIXER_SPECIAL sspecial;

  if ( space == SP_USER )
    {
      if ( VERIFY_AREA( VERIFY_READ, special, sizeof( *special ) ) ) return -EIO;
      MEMCPY_FROMFS( &sspecial, special, sizeof( sspecial ) );
      special = &sspecial;
    }
  switch ( special -> what ) {
    case GUS_MIXER_S_NONE:
      return 0;
#ifdef GUSCFG_CODEC
    case GUS_MIXER_S_IW:
      {
        unsigned char uc = card -> codec.interwave_serial_port & 0x1f;
        uc |= special -> data.interwave.serial << 5;
        card -> codec.interwave_serial_port = uc;
        return 0;
      }
#endif
  }
  return -EINVAL;
}

int gus_ioctl_mixer( gus_card_t *card, struct file *file, unsigned int cmd, unsigned long arg )
{
  if ( ( ( cmd >> 8 ) & 0xff ) != 'M' ) return -EINVAL;
  if ( !card ) card = (gus_card_t *)file -> private_data;
#ifdef SOUND_MIXER_INFO
  if ( cmd == SOUND_MIXER_INFO )
    {
      if ( VERIFY_AREA( VERIFY_WRITE, (void *)arg, sizeof( gus_mixer_info ) ) ) return -EIO;
      MEMCPY_TOFS( (void *)arg, &gus_mixer_info, sizeof( gus_mixer_info ) );
      return 0;
    }
#endif
  switch ( cmd ) {
    case SOUND_MIXER_CARDS:
      return IOCTL_OUT( arg, gus_cards_count );
    case SOUND_MIXER_GUS_VERSION:
      return IOCTL_OUT( arg, card -> version | ( card -> daughter_flag ? 0x10000 : 0 ) );
    case SOUND_MIXER_WRITE_DEVS_LMUTE:
      return IOCTL_OUT( arg, mixer_set_devs_mute( card, IOCTL_IN( arg ), 0 ) );
    case SOUND_MIXER_WRITE_DEVS_RMUTE:
      return IOCTL_OUT( arg, mixer_set_devs_mute( card, IOCTL_IN( arg ), 1 ) );
    case SOUND_MIXER_READ_DEVS_LMUTE:
      return IOCTL_OUT( arg, mixer_set_devs_mute( card, -1, 0 ) );
    case SOUND_MIXER_READ_DEVS_RMUTE:
      return IOCTL_OUT( arg, mixer_set_devs_mute( card, -1, 1 ) );
    case SOUND_MIXER_SPECIAL_READ:
      return gus_mixer_special_read( card, (struct GUS_MIXER_SPECIAL *)arg, SP_USER );
    case SOUND_MIXER_SPECIAL_WRITE:
      return gus_mixer_special_write( card, (struct GUS_MIXER_SPECIAL *)arg, SP_USER );
  }
  if ( cmd & IOC_IN )
    switch ( cmd & 0xff ) {
      case SOUND_MIXER_RECSRC:
        return IOCTL_OUT( arg, mixer_set_recsrc( card, IOCTL_IN( arg ) & card -> mixer.mix_devs ) );
      case SOUND_MIXER_MIC:
        return IOCTL_OUT( arg, mixer_set_mic( card, IOCTL_IN( arg ) ) );
      case SOUND_MIXER_CD:
        return IOCTL_OUT( arg, mixer_set_cd( card, IOCTL_IN( arg ) ) );
      case SOUND_MIXER_LINE:
        return IOCTL_OUT( arg, mixer_set_line( card, IOCTL_IN( arg ) ) );
      case SOUND_MIXER_SYNTH:
        return IOCTL_OUT( arg, mixer_set_gf1( card, IOCTL_IN( arg ) ) );
      case SOUND_MIXER_PCM:
        return IOCTL_OUT( arg, mixer_set_pcm( card, IOCTL_IN( arg ) ) );
      case SOUND_MIXER_SPEAKER:
        return IOCTL_OUT( arg, mixer_set_pcm1( card, IOCTL_IN( arg ) ) );
      case SOUND_MIXER_RECLEV:
        return IOCTL_OUT( arg, mixer_set_gain( card, IOCTL_IN( arg ) ) );
      case SOUND_MIXER_VOLUME:
        return IOCTL_OUT( arg, mixer_set_master( card, IOCTL_IN( arg ) ) );  
      case SOUND_MIXER_IMIX:
        return IOCTL_OUT( arg, mixer_set_gf1_master( card, IOCTL_IN( arg ) ) );
      case SOUND_MIXER_LINE1:
        return IOCTL_OUT( arg, mixer_set_loopback( card, IOCTL_IN( arg ) ) );
      case SOUND_MIXER_LINE2:
        return IOCTL_OUT( arg, mixer_set_effect( card, IOCTL_IN( arg ) ) );
    }
   else
  if ( cmd & IOC_OUT )
    switch ( cmd & 0xff ) {
      case SOUND_MIXER_DEVMASK:
        return IOCTL_OUT( arg, card -> mixer.mix_devs );
      case SOUND_MIXER_STEREODEVS:
        return IOCTL_OUT( arg, card -> mixer.mix_stereo_devs ); 
      case SOUND_MIXER_RECMASK:
        return IOCTL_OUT( arg, card -> mixer.mix_rec_devs );
      case SOUND_MIXER_CAPS:
        return IOCTL_OUT( arg, 0 );
      case SOUND_MIXER_RECSRC:
        return IOCTL_OUT( arg, mixer_set_recsrc( card, -1 ) );
      case SOUND_MIXER_MIC:
        return IOCTL_OUT( arg, mixer_set_mic( card, -1 ) );
      case SOUND_MIXER_CD:
        return IOCTL_OUT( arg, mixer_set_cd( card, -1 ) );
      case SOUND_MIXER_LINE:
        return IOCTL_OUT( arg, mixer_set_line( card, -1 ) );
      case SOUND_MIXER_SYNTH:
        return IOCTL_OUT( arg, mixer_set_gf1( card, -1 ) );
      case SOUND_MIXER_PCM:
        return IOCTL_OUT( arg, mixer_set_pcm( card, -1 ) );
      case SOUND_MIXER_SPEAKER:
        return IOCTL_OUT( arg, mixer_set_pcm1( card, -1 ) );
      case SOUND_MIXER_RECLEV:
        return IOCTL_OUT( arg, mixer_set_gain( card, -1 ) );
      case SOUND_MIXER_VOLUME:
        return IOCTL_OUT( arg, mixer_set_master( card, -1 ) );
      case SOUND_MIXER_IMIX:
        return IOCTL_OUT( arg, mixer_set_gf1_master( card, -1 ) );
      case SOUND_MIXER_LINE1:
        return IOCTL_OUT( arg, mixer_set_loopback( card, -1 ) );
      case SOUND_MIXER_LINE2:
        return IOCTL_OUT( arg, mixer_set_effect( card, -1 ) );
    }
#if 0
  PRINTK( "MIXER ERR\n" );
  PRINTK( "  MIXER REQUEST: 0x%x, 0x%x\n", cmd, (int)IOCTL_IN( arg ) );
#endif
  return -EINVAL;
}

int gus_open_mixer( unsigned short minor, struct file *file )
{
  if ( minor >= GUS_MINOR_GDEVMASK ) {
    if ( (minor & GUS_MINOR_GDEVMASK) == GUS_MINOR_GMIXER1 ) return -ENODEV;
  }
  minor >>= 4;
  minor &= 7;
  if ( minor >= gus_cards_count ) return -ENODEV;
  file -> private_data = gus_cards[ minor ];
  MOD_INC_USE_COUNT;
  return 0;
}

void gus_release_mixer( struct file *file )
{
  file -> private_data = NULL;
  MOD_DEC_USE_COUNT;
}

void gus_mixer_update_level( gus_card_t *card, int device, int left, int right, int lmute, int rmute )
{
  unsigned int both;
  unsigned short mute;
  
  both = ( left & 0xff ) | ( ( right & 0xff ) << 8 );
  mute = card -> mixer.curr[ device ].mute;
  if ( lmute >= 0 )
    if ( lmute ) mute |= MIX_MUTE_LEFT; else mute &= ~MIX_MUTE_LEFT;
  if ( rmute >= 0 )
    if ( rmute ) mute |= MIX_MUTE_RIGHT; else mute &= ~MIX_MUTE_RIGHT;
  card -> mixer.curr[ device ].mute = mute;
  switch ( device ) {
    case MIX_MIC:	mixer_set_mic( card, both ); break;
    case MIX_LINE:	mixer_set_line( card, both ); break;
    case MIX_CD:	mixer_set_cd( card, both ); break;
    case MIX_GF1:	mixer_set_gf1( card, both ); break;
    case MIX_PCM:	mixer_set_pcm( card, both ); break;
    case MIX_PCM1:	mixer_set_pcm1( card, both ); break;
    case MIX_GAIN:	mixer_set_gain( card, both ); break;
    case MIX_MASTER:	mixer_set_master( card, both ); break;
    case MIX_GF1_MASTER: mixer_set_gf1_master( card, both ); break;
    case MIX_LOOPBACK:	mixer_set_loopback( card, both ); break;
    case MIX_EFFECTS:	mixer_set_effect( card, both ); break;
  }
}
