/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 *  Routines for the MIDI interface - UART 6850
 */

#include "driver.h"
#include "midi.h"

#ifdef GUSCFG_MIDI_DEVICES

static void gus_interrupt_midi_in( gus_card_t *card )
{
  int count;
  unsigned char stat, data;
  struct GUS_STRU_MIDI *midi;

  midi = &card -> midi[ GUS_MIDID_UART ];
  count = 10;
  while ( count )
    {
      stat = gf1_uart_stat( card );
      if ( !(stat & 0x01) || (stat == 0xff) )	/* data in Rx FIFO? */
        {
          count--;
          continue;
        }
      count = 100;			/* arm counter */
      data = gf1_uart_get( card );
      if ( !(midi -> flags & GUS_MIDIF_USED_IN) ) continue;
      if ( stat & 0x10 )
        {
          card -> gf1.uart_framing++;
          gus_midi_cmd_init( &midi -> rx, 0 );
          continue;
        }
      gus_midi_cmd_byte( &midi -> rx, gf1_uart_get( card ) );
      if ( stat & 0x20 )
        {
          card -> gf1.uart_overrun++;
          gus_midi_cmd_init( &midi -> rx, 0 );	/* overrun? */
        }
    }
}

static void gus_interrupt_midi_out( gus_card_t *card )
{
  struct GUS_STRU_MIDI *midi;

  midi = &card -> midi[ GUS_MIDID_UART ];
  if ( gf1_uart_stat( card ) & 0x01 )
    gus_interrupt_midi_in( card );				/* try deblock output */
  if ( ( gf1_uart_stat( card ) & 0x02 ) && midi -> tx_used )	/* Tx FIFO free? */
    {
#if 0
      PRINTK( "tx = 0x%x\n", midi -> tx_buf[ midi -> tx_tail ] );
#endif
      gf1_uart_put( card, midi -> tx_buf[ midi -> tx_tail++ ] );
      midi -> tx_tail %= midi -> tx_size;
      midi -> tx_used--;
    }
#ifdef GUSCFG_MIDI
  if ( GETLOCK( card, midi_raw_out ) & WK_SLEEP ) {
    GETLOCK( card, midi_raw_out ) &= ~WK_SLEEP;
    WAKEUP( card, midi_raw_out );
  }
#endif
  if ( !midi -> tx_used )
    {
      gf1_uart_cmd( card, card -> gf1.uart_cmd & ~0x20 );	/* disable Tx interrupt */
      if ( GETLOCK( card, uart_flush ) & WK_SLEEP )
        {
          GETLOCK( card, uart_flush ) &= ~WK_SLEEP;
          WAKEUP( card, uart_flush );
        }
    }
}

static void uart_reset( gus_card_t *card )
{
  gf1_uart_cmd( card, 0x03 );		/* reset */
  if ( card -> gf1.uart_enable )
    {
      gus_delay1( 1 );
      gf1_uart_cmd( card, 0x00 );		/* normal operations */
    }
}

static void uart_close( gus_card_t *card )
{
  gf1_uart_cmd( card, 0x03 );		/* reset */
}

static int gus_gf1_uart_init_write( gus_card_t *card )
{
  struct GUS_STRU_MIDI *midi;

  midi = &card -> midi[ GUS_MIDID_UART ];
  if ( !( midi -> flags & GUS_MIDIF_USED ) ) uart_reset( card );
  if ( ( midi -> tx_buf = gus_malloc( midi -> tx_size ) ) == NULL ) return -ENOMEM;
  card -> gf1.interrupt_handler_midi_out = gus_interrupt_midi_out;
#if 0
  printk( "write init - cmd = 0x%x, stat = 0x%x\n", card -> gf1.uart_cmd, gf1_uart_stat( card ) );
#endif
  return 0;
}

static int gus_gf1_uart_init_read( gus_card_t *card )
{
  int i;
  struct GUS_STRU_MIDI *midi;

  midi = &card -> midi[ GUS_MIDID_UART ];
  if ( !( midi -> flags & GUS_MIDIF_USED ) ) uart_reset( card );
  if ( ( midi -> rx.buf = gus_malloc( midi -> rx.size ) ) == NULL ) return -ENOMEM;
  if ( midi -> rx_size > 0 ) {
    if ( ( midi -> rx_buf = gus_malloc( midi -> rx_size ) ) == NULL ) {
      gus_free( midi -> rx.buf, midi -> rx.size );
      return -ENOMEM;
    }
  } else {
    midi -> rx_buf = NULL;
  }
  card -> gf1.interrupt_handler_midi_in = gus_interrupt_midi_in;
  if ( card -> gf1.uart_enable )
    {
      for ( i = 0; i < 1000 && ( gf1_uart_stat( card ) & 0x01 ); i++ )
        gf1_uart_get( card );	/* clean Rx */
      if ( i >= 1000 )
        PRINTK( "gus: midi uart init read - cleanup error\n" );
      gf1_uart_cmd( card, card -> gf1.uart_cmd | 0x80 );	/* enable Rx interrupts */
    }
#if 0
  printk( "read init - cmd = 0x%x, stat = 0x%x\n", card -> gf1.uart_cmd, gf1_uart_stat( card ) );
#endif
  return 0;
}

static void gus_gf1_uart_done_write( gus_card_t *card )
{
  struct GUS_STRU_MIDI *midi;

  midi = &card -> midi[ GUS_MIDID_UART ];
  while ( midi -> tx_used )
    {
      GETLOCK( card, uart_flush ) |= WK_SLEEP;
      SLEEP( card, uart_flush, 60 * HZ );
      GETLOCK( card, uart_flush ) &= ~WK_SLEEP;
      if ( TABORT( card, uart_flush ) ) break;
      if ( TIMEOUT( card, uart_flush ) ) break;
    }
  if ( ( midi -> flags & GUS_MIDIF_USED ) == GUS_MIDIF_USED_OUT )
    uart_close( card );
   else
    gf1_uart_cmd( card, card -> gf1.uart_cmd & ~0x20 );	/* disable Tx interrupts */
  gus_set_default_handlers( card, GF1_HANDLER_MIDI_OUT );
  gus_free( midi -> tx_buf, midi -> tx_size );
  midi -> tx_buf = NULL;
}

static void gus_gf1_uart_done_read( gus_card_t *card )
{
  struct GUS_STRU_MIDI *midi;

  midi = &card -> midi[ GUS_MIDID_UART ];
  if ( ( midi -> flags & GUS_MIDIF_USED ) == GUS_MIDIF_USED_IN )
    uart_close( card );
   else
    gf1_uart_cmd( card, card -> gf1.uart_cmd & ~0x80 );	/* disable Rx interrupts */
  gus_set_default_handlers( card, GF1_HANDLER_MIDI_IN );
  if ( midi -> rx_buf ) {
    gus_free( midi -> rx_buf, midi -> rx_size );
    midi -> rx_buf = NULL;
  }
  gus_free( midi -> rx.buf, midi -> rx.size );
  midi -> rx.buf = NULL;
}

static int gus_gf1_uart_putcmd( gus_card_t *card, unsigned char *buffer, unsigned int count )
{
  unsigned long flags;
  struct GUS_STRU_MIDI *midi;
  unsigned int old_tx_used;
  
  midi = &card -> midi[ GUS_MIDID_UART ];
  if ( !(midi -> flags & GUS_MIDIF_USED_OUT) ) return -EIO;
  if ( !card -> gf1.uart_enable ) return 0;	/* ignore */
  CLI( &flags );
  old_tx_used = midi -> tx_used;
  while ( count-- > 0 )
    {
      midi -> tx_buf[ midi -> tx_head++ ] = *buffer++;
      midi -> tx_head %= midi -> tx_size;
      midi -> tx_used++;
    }
  if ( !old_tx_used )
    {
      if ( gf1_uart_stat( card ) & 0x01 )
        gus_interrupt_midi_in( card );					/* try deblock output */
      if ( ( gf1_uart_stat( card ) & 0x02 ) && midi -> tx_used )	/* Tx FIFO free? */
        {
#if 0
          PRINTK( "tx = 0x%x\n", midi -> tx_buf[ midi -> tx_tail ] );
#endif
          gf1_uart_put( card, midi -> tx_buf[ midi -> tx_tail++ ] );
          midi -> tx_tail %= midi -> tx_size;
          midi -> tx_used--;
        }
      if ( midi -> tx_used )
        gf1_uart_cmd( card, card -> gf1.uart_cmd | 0x20 );		/* enable Tx interrupts */
    }
  STI( &flags );
  return 0;
}

void gus_init_gf1_uart( gus_card_t *card )
{
  struct GUS_STRU_MIDI *midi;
  
  midi = &card -> midi[ GUS_MIDID_UART ];
  midi -> flags = 0;
  midi -> init_write = gus_gf1_uart_init_write;
  midi -> init_read = gus_gf1_uart_init_read;
  midi -> done_write = gus_gf1_uart_done_write;
  midi -> done_read = gus_gf1_uart_done_read;
  midi -> putcmd = gus_gf1_uart_putcmd;
}

#endif /* GUSCFG_MIDI_DEVICES */
