
/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 *  Routines for control timers of GF1 chip
 */

#include "driver.h"

void gus_gf1_timer_init( gus_card_t *card )
{
  card -> gf1.timer_enabled = 0;
  card -> gf1.timer_owner = NULL;
  card -> gf1.timer_lock = 0;
  card -> gf1.timer_midi = 0;
}

/*
 *  timer 1
 */

static int gus_timer1_start( gus_card_t *card, unsigned short time )
{
  unsigned long flags;
  unsigned char tmp;

  tmp = ( card -> gf1.timer_enabled |= 4 );
  card -> gf1.timer_old_count1 = time;
  CLI( &flags );
  gus_write8( card, 0x46, 256 - time );		/* timer 2 count */
  gus_write8( card, 0x45, tmp );		/* enable timer 2 IRQ */
  gus_adlib_write( card, 0x04, tmp >> 2 );	/* timer 2 start */
  STI( &flags );
  return 0;
}

static int gus_timer1_stop( gus_card_t *card )
{
  unsigned char tmp;

  tmp = ( card -> gf1.timer_enabled &= ~4 );
  gus_i_write8( card, 0x45, tmp );		/* disable timer #1 */
  return 0;
}

#if 0

static int gus_timer2_start( gus_card_t *card, unsigned short time )	/* 320us */
{
  unsigned long flags;
  unsigned char tmp;

  if ( time < 1 || time > 256 ) return -EINVAL;
#if 0
  PRINTK( "gus_timer2_start: time = %d\n", time );
#endif
  card -> gf1.timer_old_count2 = time;
  tmp = ( card -> gf1.timer_enabled |= 8 );
  CLI( &flags );
  gus_write8( card, 0x47, 256 - time );		/* timer 2 count */
  gus_write8( card, 0x45, tmp );		/* enable timer 2 IRQ */
  gus_adlib_write( card, 0x04, tmp >> 2 );	/* timer 2 start */
  STI( &flags );
  card -> gf1.syn_stop_flag = 0;
  return 0;
}

#endif

static int gus_timer2_stop( gus_card_t *card )
{
  unsigned char tmp;

#if 0
  PRINTK( "gus_timer2_stop\n" );
#endif
  tmp = ( card -> gf1.timer_enabled &= ~8 );
  gus_i_write8( card, 0x45, tmp );		/* disable timer #2 */
  card -> gf1.timer_old_count2 = 1;
  return 0;
}

static inline int gus_timer_start( gus_card_t *card )
{
  int res;
 
  card -> gf1.timer_abs_tick = 0;
  card -> gf1.timer_tempo_modulo = 0;
  card -> gf1.timer_effects_ticks_current = card -> gf1.timer_effects_ticks;
  res = gus_timer1_start( card, card -> gf1.timer_tempo_value / 80L );
  card -> gf1.effect_timer = card -> gf1.timer_old_count1 * 80;
  return res;
}

static inline int gus_timer_stop( gus_card_t *card )
{
  return gus_timer1_stop( card );
}

static inline int gus_timer_continue( gus_card_t *card )
{
  return gus_timer1_start( card, card -> gf1.timer_old_count1 );
}

static int gus_timer_lock( gus_card_t *card, char *owner )
{
  if ( !card -> gf1.timer_lock )
    {
      card -> gf1.timer_owner = owner;
      card -> gf1.timer_lock = 1;
      return 0;
    }
  return -EBUSY;
}

static void gus_timer_unlock( gus_card_t *card )
{
  if ( card -> gf1.timer_lock )
    {
      card -> gf1.timer_owner = NULL;
      card -> gf1.timer_lock = 0;
    }
}

int gus_timer_set_master( gus_card_t *master )
{
  if ( master -> gf1.timer_master ) return -EBUSY;
  master -> gf1.timer_master = 1;
  MEMSET( &master -> gf1.timer_slave_cards, 0, sizeof( gus_card_t * ) * GUS_CARDS );
  master -> gf1.timer_slave_cards[ 0 ] = master;
  return 0;
}

int gus_timer_res_master( gus_card_t *master )
{
  short i;

  if ( master == NULL ) return 0;
  gus_timer_stop( master );		/* for sure */
  for ( i = 0; i < GUS_CARDS; i++ )
    if ( master -> gf1.timer_slave_cards[ i ] )
      gus_timer_res_slave( master, master -> gf1.timer_slave_cards[ i ] );
  master -> gf1.timer_master = 0;
  return 0;
}

int gus_timer_set_slave( gus_card_t *master, gus_card_t *slave )
{
  short i;
  
  if ( !slave ) return -EINVAL;
  for ( i = 0; i < GUS_CARDS; i++ )
    {
      if ( master -> gf1.timer_slave_cards[ i ] == NULL )
        {
          gus_timer1_stop( slave );	/* timers on slave card isn't used */
          gus_timer2_stop( slave );
          master -> gf1.timer_slave_cards[ i ] = slave;
          slave -> gf1.timer_master_card = master;
          slave -> gf1.timer_slave = 1;
          return 0;
        }
      if ( master -> gf1.timer_slave_cards[ i ] == slave ) return -EIO;	/* already */
    }
  return 0;
}

int gus_timer_res_slave( gus_card_t *master, gus_card_t *slave )
{
  short i;
  
  for ( i = 0; i < GUS_CARDS; i++ )
    if ( master -> gf1.timer_slave_cards[ i ] == slave )
      {
        master -> gf1.timer_slave_cards[ i ] = 0;
        slave -> gf1.timer_slave = 0;
        slave -> gf1.timer_master_card = NULL;
        return 0;
      }
  return -EIO;
}

/*
 *
 */

static void gus_interrupt_timer1( gus_card_t *card )
{
  unsigned int i;

  gus_timer1_stop( card );
  if ( card -> gf1.timer_slave ) return;
  card -> gf1.timer_tempo_modulo += card -> gf1.timer_tempo_value;
  i = card -> gf1.timer_tempo_modulo / 80L;
  card -> gf1.timer_tempo_modulo %= 80;
  gus_timer1_start( card, i );

  /*
   *  ok.. now is timer loaded..
   *  step 1. - effect processing...
   */

  if ( card -> gf1.timer_master )
    {
      for ( i = 0; i < GUS_CARDS; i++ )
        {
          gus_card_t *card1;
              
          card1 = card -> gf1.timer_slave_cards[ i ];
          if ( !card1 || !(card1 -> gf1.mode & GF1_MODE_ENGINE) ) continue;
          gus_engine_process_effects( card1 );
        }
    } 
   else
    gus_engine_process_effects( card );

  /*
   *
   */

  if ( card -> gf1.timer_effects_ticks_current > 1 )
    {
      card -> gf1.timer_effects_ticks_current--;
      return;
    }
  card -> gf1.timer_effects_ticks_current = card -> gf1.timer_effects_ticks;
#ifdef GUSCFG_OSS
  if ( card -> gf1.timer_sequencer )
    gus_sequencer_tick();
#endif
  if ( card -> gf1.timer_wait_ticks > 0 )
    {
      card -> gf1.timer_wait_ticks--;
      return;
    }

  /*
   *  step 2. - new events processing
   */

  if ( card -> gf1.timer_master )
    {
      int flag = 0;

#ifdef GUSCFG_MIDI
      if ( card -> gf1.timer_midi )
        {
          gus_midi_interrupt();			/* own routine */
          flag++;
        }
#endif
#ifdef GUSCFG_OSS
      if ( card -> gf1.timer_sequencer )
        {
          gus_sequencer_process_events( 1 );	/* own routine */
          flag++;
        }
#endif
      for ( i = 0; i < GUS_CARDS; i++ )
        {
          gus_card_t *card1;
              
          card1 = card -> gf1.timer_slave_cards[ i ];
          if ( !card1 || !(card1 -> gf1.mode & GF1_MODE_ENGINE) ) continue;
          if ( !flag ) gus_gf1_process_events( card1 );
          gus_engine_process_voices( card1 );
        }
    } 
   else
    {
      gus_gf1_process_events( card );
      gus_engine_process_voices( card );
    }

  card -> gf1.effect_timer = card -> gf1.timer_old_count1 * 80;
  card -> gf1.timer_abs_tick++;
}

void gus_timer_wait( gus_card_t *card )
{
  if ( !card -> gf1.timer_master && card -> gf1.timer_slave )
    gus_timer_stop( card -> gf1.timer_master_card );
   else
    gus_timer_stop( card );
}

void gus_timer_restart( gus_card_t *card )
{
  if ( !card -> gf1.timer_master && card -> gf1.timer_slave )
    gus_timer_continue( card -> gf1.timer_master_card );
   else
    gus_timer_continue( card );
  gus_engine_process_voices( card );
}

int gus_timer_tempo( gus_card_t *card, int tempo )
{
#if 0
  printk( "tempo = %i, timer_base = %i\n", tempo, card -> gf1.timer_base );
#endif
  if ( tempo > 400 ) tempo = 400;
  if ( tempo < 14 ) tempo = 14;
  card -> gf1.timer_tempo = tempo;
  tempo *= card -> gf1.timer_base;
  card -> gf1.timer_tempo_value = ( 100000000 + ( tempo >> 1 ) ) / tempo;
  card -> gf1.timer_effects_ticks = 1;
  if ( card -> gf1.timer_tempo_value < 400 )
    card -> gf1.timer_tempo_value = 400;
  while ( card -> gf1.timer_tempo_value >= 8000 )
    {
      card -> gf1.timer_effects_ticks <<= 1;
      card -> gf1.timer_tempo_value >>= 1;
    }
#if 0
  card -> gf1.timer_effects_ticks = 200;
#endif
#if 0
  printk( "card -> gf1.timer_tempo_value = %i, effect = %i\n", card -> gf1.timer_tempo_value, card -> gf1.timer_effects_ticks );
#endif
  return 0;
}

int gus_timer_base( gus_card_t *card, int base )
{
  card -> gf1.timer_base = base;
  return gus_timer_tempo( card, card -> gf1.timer_tempo );
}

int gus_timer_ioctl( gus_card_t *card, unsigned int cmd, unsigned long arg )
{
  int res;

  switch ( cmd ) {
    case GUS_IOCTL_TIMER_START:
#if 1
      return gus_timer_start( card );
#else
      return 0;
#endif
    case GUS_IOCTL_TIMER_STOP:
      res = gus_timer_stop( card );
      if ( card -> gf1.mode & GF1_MODE_ENGINE )
        if ( res < 0 ) 
          gus_timer1_stop( card );
         else
          res = gus_timer1_stop( card );
      return res;
    case GUS_IOCTL_TIMER_CONTINUE:
      return gus_timer_continue( card );
    case GUS_IOCTL_TIMER_BASE:
      return gus_timer_base( card, IOCTL_IN( arg ) );
    case GUS_IOCTL_TIMER_TEMPO:
      return gus_timer_tempo( card, IOCTL_IN( arg ) );
    case GUS_IOCTL_TIMER_MASTER: 
      return gus_timer_set_master( card );
    case GUS_IOCTL_TIMER_SLAVE:
      res = IOCTL_IN( arg );
      if ( res < 0 || res >= GUS_CARDS || gus_cards[ res ] == NULL ) return -ENODEV;
      return gus_timer_set_slave( card, gus_cards[ res ] );
  }
  return -EIO;
}

/*
 *
 */
 
int gus_timer_open( gus_card_t *card, char *owner )
{
  if ( gus_timer_lock( card, owner ) < 0 ) return -EBUSY;
  card -> gf1.interrupt_handler_timer1 = gus_interrupt_timer1;
#if 0		/* we don't need at this time second timer */
  card -> gf1.interrupt_handler_timer2 = gus_interrupt_timer2;
#endif
  card -> gf1.timer_wait_ticks = 0;
  card -> gf1.timer_effects_ticks_current = 0;
  card -> gf1.timer_tempo = 60;
  gus_timer_base( card, 100 );
  gus_gf1_open( card, GF1_MODE_TIMER );	/* IRQ enable */
  return 0;
}

void gus_timer_close( gus_card_t *card )
{
  gus_gf1_close( card, GF1_MODE_TIMER ); /* IRQ disable */
  gus_timer1_stop( card );
  gus_timer2_stop( card );
  gus_set_default_handlers( card, GF1_HANDLER_TIMER1 | GF1_HANDLER_TIMER2 );
  if ( card -> gf1.timer_master ) gus_timer_res_master( card );
  if ( card -> gf1.timer_slave ) gus_timer_res_slave( card -> gf1.timer_master_card, card );
  gus_timer_unlock( card );
}
 
/*
 *
 */

void gus_timer_info( gus_card_t *card, gus_info_buffer_t *buffer )
{
  if ( !card -> gf1.timer_slave ) 
    gus_iprintf( buffer,
  		"TIMERS:\n"
		"  owner         : %s%s\n"
		"  enabled       :%s%s\n"
		"  timer1 value  : %i\n"
		"  timer2 value  : %i\n"
		"  base          : %i\n"
		"  tempo         : %i (%ius)\n"
		"  effects ticks : %i\n",
		card -> gf1.timer_owner,
		card -> gf1.timer_master ? " [master]" : "",
		card -> gf1.timer_enabled & 4 ? " timer1" : "",
		card -> gf1.timer_enabled & 8 ? " timer2" : "",
		card -> gf1.timer_old_count1,
		card -> gf1.timer_old_count2,
		card -> gf1.timer_base,
		card -> gf1.timer_tempo,
		card -> gf1.timer_tempo_value,
		card -> gf1.timer_effects_ticks
	     );
   else
    gus_iprintf( buffer,
		"TIMERS:\n"
		"  owner         : %s [slave]\n",
		card -> gf1.timer_master_card -> gf1.timer_owner
	     );
}
