/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 *  INTERWAVE INTERFACE
 */

#include "driver.h"
#include "tables.h"

#if 0
#define FREE_DEBUG
#endif

static void voice_interrupt_wave( gus_card_t *card, gus_voice_t *voice );
static void voice_interrupt_volume( gus_card_t *card, gus_voice_t *voice );
static void voice_interrupt_effect( gus_card_t *card, gus_voice_t *voice );
static void voice_start( gus_card_t *card, gus_voice_t *voice );
static void voice_go( gus_card_t *card, gus_voice_t *voice );
static void voice_stop( gus_card_t *card, gus_voice_t *voice, unsigned char mode );
static void voice_control( gus_card_t *card, gus_voice_t *voice, unsigned char control );
static void voice_freq( gus_card_t *card, gus_voice_t *voice, unsigned int freq );
static void voice_volume( gus_card_t *card, gus_voice_t *voice, unsigned short volume );
static void voice_loop( gus_card_t *card, gus_voice_t *voice, unsigned int start, unsigned int end );
static void voice_ramp( gus_card_t *card, gus_voice_t *voice, unsigned char start, unsigned char end, unsigned char rate, unsigned char control );
static void voice_pos( gus_card_t *card, gus_voice_t *voice, unsigned int position );
static void voice_pan( gus_card_t *card, gus_voice_t *voice, unsigned short pan );
static void voice_lfo( gus_card_t *card, gus_voice_t *voice, unsigned char *data );
static void voice_private1( gus_card_t *card, gus_voice_t *voice, unsigned char *data );

static struct GUS_STRU_INSTRUMENT_VOICE_COMMANDS voice_commands = {
  voice_interrupt_wave,
  voice_interrupt_volume,
  voice_interrupt_effect,
  voice_start,
  voice_go,
  voice_stop,
  voice_control,
  voice_freq,
  voice_volume,
  voice_loop,
  voice_ramp,
  voice_pos,
  voice_pan,
  voice_lfo,
  voice_private1
};

static void note_stop( gus_card_t *card, gus_voice_t *voice, int wait );
static void note_wait( gus_card_t *card, gus_voice_t *voice );
static void note_off( gus_card_t *card, gus_voice_t *voice );
static void note_volume( gus_card_t *card, gus_voice_t *voice );
static void note_pitchbend( gus_card_t *card, gus_voice_t *voice );
static void note_vibrato( gus_card_t *card, gus_voice_t *voice );
static void note_tremolo( gus_card_t *card, gus_voice_t *voice );

static struct GUS_STRU_INSTRUMENT_NOTE_COMMANDS note_commands = {
  note_stop,
  note_wait,
  note_off,
  note_volume,
  note_pitchbend,
  note_vibrato,
  note_tremolo
};

static void chn_trigger_down( gus_card_t *card, gus_channel_t *channel, gus_instrument_t *instrument, unsigned char note, unsigned char velocity, unsigned char priority );
static void chn_trigger_up( gus_card_t *card, gus_note_t *note );
static void chn_control( gus_card_t *card, gus_channel_t *channel, unsigned short p1, unsigned short p2 );

static struct GUS_STRU_INSTRUMENT_CHANNEL_COMMANDS channel_commands = {
  chn_trigger_down,
  chn_trigger_up,
  chn_control
};

/*
 *
 */
 
static void layer_complete( gus_card_t *card, gus_voice_t *voice, int wait );
static void do_volume_envelope( gus_card_t *card, gus_voice_t *voice );
static void do_pitch_envelope( gus_card_t *card, gus_voice_t *voice );
static int trigger_layer( gus_card_t *card, gus_note_t *note, gus_voice_t *voice, int layer_event );

/*
 *  interrupt handling
 */
 
static void voice_interrupt_wave( gus_card_t *card, gus_voice_t *voice )
{
  if ( voice -> flags & VFLG_RUNNING )
    layer_complete( card, voice, 1 );
   else
    gus_gf1_stop_voice( card, voice -> number );
}

static void voice_interrupt_volume( gus_card_t *card, gus_voice_t *voice )
{
  if ( voice -> flags & VFLG_RUNNING )
    do_volume_envelope( card, voice );
   else
    gus_gf1_stop_voice( card, voice -> number );
}

static void voice_interrupt_effect( gus_card_t *card, gus_voice_t *voice )
{
  if ( (voice -> flags & (VFLG_RUNNING|VFLG_EFFECT_TIMER2)) == (VFLG_RUNNING|VFLG_EFFECT_TIMER2) )
    do_pitch_envelope( card, voice );
}

/*
 *
 */

#ifdef GUSCFG_INTERWAVE
static void set_volume( gus_card_t *card, unsigned short left, unsigned short right, int both )
{
  if ( left > MAX_VOLUME ) left = MAX_VOLUME;
  left <<= 4;
  if ( right > MAX_VOLUME ) right = MAX_VOLUME;
  right <<= 4;
  if ( both ) gus_write16( card, GF1_VW_OFFSET_LEFT, left );
  gus_write16( card, GF1_VW_OFFSET_LEFT_FINAL, left );
  if ( both ) gus_write16( card, GF1_VW_OFFSET_RIGHT, right );
  gus_write16( card, GF1_VW_OFFSET_RIGHT_FINAL, right );
}
#endif

#ifdef GUSCFG_INTERWAVE
static void change_effect_volume( gus_card_t *card, gus_voice_t *voice, unsigned short atten, int both )
{
  gus_channel_t *channel;
  unsigned char rev_depth, chorus_depth;

  if ( !voice -> note ) return;
  channel = voice -> note -> channel;
  if ( !channel ) return;
  if ( !card -> gf1.enh_mode ) return;
  rev_depth = chorus_depth = 0;
  if ( channel -> number == 9 )	/* drum channel */
    {
      gus_instrument_t *instrument;
    
      instrument = voice -> instrument; 
      if ( !instrument ) return;
      /* percussion's effects are multiplied against channel levels */
      switch ( instrument -> data.iw.effect1 ) {
        case GUS_INSTR_IE_REVERB:
          rev_depth = instrument -> data.iw.effect1_depth;
          break;
        case GUS_INSTR_IE_CHORUS:
          chorus_depth = instrument -> data.iw.effect1_depth;
          break;
       }
       switch ( instrument -> data.iw.effect2 ) {
         case GUS_INSTR_IE_REVERB:
           rev_depth = instrument -> data.iw.effect2_depth;
           break;
         case GUS_INSTR_IE_CHORUS:
           chorus_depth = instrument -> data.iw.effect2_depth;
           break;
       }
      rev_depth = VOLUME( rev_depth );
      chorus_depth = VOLUME( chorus_depth );
    }
  gus_set_effects_volume( card, voice,
                      atten + channel -> effect1_level + rev_depth,
                      atten + channel -> effect3_level + chorus_depth,
          	      both );
}
#endif

static void stop_note_voice( gus_card_t *card, gus_voice_t *voice )
{
  gus_engine_midi_go_note_stop( card, voice -> note, 1 );
}

static struct GUS_STRU_IW_ENV_RECORD *find_envelope(
			struct GUS_STRU_IW_ENV *env,
			unsigned char note,
			unsigned char velocity )
{
  int i = 0;
  struct GUS_STRU_IW_ENV_RECORD *record;
  
  if ( env -> index == GUS_STRU_IW_ENV_I_VELOCITY ) i = velocity; else
  if ( env -> index == GUS_STRU_IW_ENV_I_FREQUENCY ) i = note; else
    return env -> record;
  for ( record = env -> record; record; record = record -> next )
    if ( i <= record -> hirange ) break;
  return record;
}

static void layer_complete( gus_card_t *card, gus_voice_t *voice, int wait )
{
#if 0
  PRINTK( " -%i- layer complete\n", voice -> number );
#endif
  if ( wait )
    {
      note_stop( card, voice, 0 );
      note_wait( card, voice );
    }
   else
    note_stop( card, voice, 1 );
}

static void do_volume_envelope( gus_card_t *card, gus_voice_t *voice )
{
  unsigned long flags;
  short old_volume, next = 0, index, nattack, nrelease;
  unsigned short rate = 0;
  char program_next_ramp;
  static char retriggering = 0;
  
  while ( 1 ) {
    program_next_ramp = 0;
#if 0
    printk( " -%i- venv_state = %i\n", voice -> number, voice -> data.iw.venv_state );
#endif
    switch ( voice -> data.iw.venv_state ) {
      case IW_BEFORE:
        voice -> data.iw.venv_next = 0;
        voice -> data.iw.venv_rate = 20;
        voice -> data.iw.venv_index = 0;	/* initialize envelope pointer */
        voice -> data.iw.venv_state++;
        CLI( &flags );
        gf1_select_voice( card, voice -> number );
        gus_write16( card, GF1_VW_VOLUME, MIN_VOLUME );
        STI( &flags );
        break;
      case IW_ATTACK:
        index = voice -> data.iw.venv_index;
        if ( index < voice -> data.iw.ver -> nattack )
          {
            program_next_ramp = 1;
            next = voice -> data.iw.ver -> points[ index ].offset;
            rate = voice -> data.iw.ver -> points[ index ].rate;
            voice -> data.iw.venv_index++;
          }
         else
          voice -> data.iw.venv_state++;
        break;
      case IW_SUSTAIN:
        program_next_ramp = 1;
        next = voice -> data.iw.ver -> sustain_offset;
        rate = voice -> data.iw.ver -> sustain_rate;
        voice -> data.iw.venv_state++;
        break;
      case IW_VAR_RELEASE:
        index = voice -> data.iw.venv_index;
        nattack = voice -> data.iw.ver -> nattack;
        nrelease = voice -> data.iw.ver -> nrelease;
        if ( !voice -> layer )
          {
            gus_ctrl_stop( card, GF1_VB_VOLUME_CONTROL );
            voice -> data.iw.venv_state = IW_DONE;
            break;
          }
        if ( index <= nattack &&
             voice -> layer -> data.iw.venv.mode == GUS_STRU_IW_ENV_M_SUSTAIN )
          {
            if ( !(voice -> flags & VFLG_RELEASING) )
              {
                voice -> flags |= VFLG_SUSTAINING;
                gus_ctrl_stop( card, GF1_VB_VOLUME_CONTROL );
                return;
              }
          }
        if ( index <= nattack &&
             voice -> layer -> data.iw.venv.mode == GUS_STRU_IW_ENV_M_ONE_SHOT )
          index = voice -> data.iw.venv_index = nattack;
        if ( index < ( nattack + nrelease ) )
          {
            if ( index == nattack )
              {
                CLI( &flags );
                gf1_select_voice( card, voice -> number );
                gus_ctrl_stop( card, GF1_VB_VOLUME_CONTROL );
                old_volume = gus_read16( card, GF1_VW_VOLUME ) >> 8;
                STI( &flags );
                if ( voice -> layer -> data.iw.velocity_mode == GUS_LAYER_IW_V_VEL_RATE )
                  old_volume += voice -> data.iw.vel_atten >> 4;
                voice -> data.iw.rel_offset = old_volume - voice -> data.iw.ver -> sustain_offset;
                if ( voice -> data.iw.rel_offset > 0 ) voice -> data.iw.rel_offset = 0;
              }
            program_next_ramp = 1;
            next = voice -> data.iw.ver -> points[ index ].offset;
            rate = voice -> data.iw.ver -> points[ index ].rate;
            voice -> data.iw.venv_index++;
            index++;
          }
         else
          voice -> data.iw.venv_state++;
        break;
      case IW_FINAL_RELEASE:
        program_next_ramp = 1;
        next = 0;
        rate = voice -> data.iw.ver -> release_rate;
        voice -> data.iw.venv_state++;
        /* Rate of 0 shuts off enveloping, but leaves note */
        /* playing. This is useful for sampled percussion. */
        if ( !rate )
          {
            CLI( &flags );
            gf1_select_voice( card, voice -> number );
            gus_ctrl_stop( card, GF1_VB_VOLUME_CONTROL );
            old_volume = gus_read16( card, GF1_VW_VOLUME ) >> 8;
            STI( &flags );
            if ( old_volume <= MIN_OFFSET + 1 )
              layer_complete( card, voice, 1 );
            return;
          }
        break;
      case IW_DONE:
        if ( (voice -> layer -> data.iw.venv.flags & GUS_STRU_IW_ENV_F_RETRIGGER) && !retriggering )
          {
            if ( !(voice -> flags & VFLG_RELEASING) )
              {
                retriggering = 1;
                if ( voice -> layer -> data.iw.flags & GUS_LAYER_IW_F_RETRIGGER )
                  {
                    CLI( &flags );
                    gf1_select_voice( card, voice -> number );
                    gus_ctrl_stop( card, GF1_VB_VOLUME_CONTROL );
                    STI( &flags );
                    if ( trigger_layer( card, voice -> note, voice, GUS_LAYER_IW_E_RETRIG ) )
                      {
                        CLI( &flags );
                        gf1_select_voice( card, voice -> number );
                        gus_write8( card, GF1_VB_VOLUME_CONTROL, voice -> data.iw.volume_control );
                        STI( &flags );
                        retriggering = 0;
                        return;
                      }
                  }
                voice -> data.iw.venv_state = IW_BEFORE;
                retriggering = 0;
                continue;
              }
          }
        layer_complete( card, voice, 1 );
        return;
      case IW_VOLUME:
        next = voice -> data.iw.venv_next;
        rate = voice -> data.iw.venv_rate;
        voice -> data.iw.venv_state = voice -> data.iw.venv_state_prev;
        CLI( &flags );
        gf1_select_voice( card, voice -> number );
        gus_ctrl_stop( card, GF1_VB_VOLUME_CONTROL );
        old_volume = gus_read16( card, GF1_VW_VOLUME ) >> 8;
        STI( &flags );
        goto __volume;
    }
    if ( !program_next_ramp ) continue;
    CLI( &flags );
    gf1_select_voice( card, voice -> number );
    gus_ctrl_stop( card, GF1_VB_VOLUME_CONTROL );
    old_volume = gus_read16( card, GF1_VW_VOLUME ) >> 8;
    STI( &flags );
    if ( !rate ) continue;
    /* set up volume ramp */
    if ( voice -> layer -> data.iw.velocity_mode == GUS_LAYER_IW_V_VEL_RATE )
      {
        /* the volume ramp registers are 8 bits, and the     */
        /* attenuation registers are 12 bits... compensate   */
        next -= voice -> data.iw.vel_atten >> 4;
      }
    if ( voice -> data.iw.venv_state == IW_VAR_RELEASE &&
         voice -> data.iw.venv_index > voice -> data.iw.ver -> nattack )
      next += voice -> data.iw.rel_offset;
    voice -> data.iw.venv_next = next;
    voice -> data.iw.venv_rate = rate;
    if ( !card -> gf1.enh_mode )
      {
        unsigned char rmode;
        
        rmode = rate & 0xc0;
        rate &= 0x3f;
        if ( rate > 0 )
          {
            rate = (unsigned char)(((int)rate * 44100L) / card -> gf1.playback_freq);
            if ( !rate ) rate++;
            if ( rate > 0x3f ) rate = 0x3f;
          }
        rate |= rmode;
        voice -> data.iw.venv_rate = rate;
        __volume:
        next = ( ( (int)voice -> data.iw.gf1_volume * (int)next ) / 255 ) >> 8;
      }
    if ( old_volume < MIN_OFFSET ) old_volume = MIN_OFFSET;
    if ( next < MIN_OFFSET ) next = MIN_OFFSET;
    if ( next > MAX_OFFSET ) next = MAX_OFFSET;
    if ( old_volume == next ) continue;	/* already there */
    voice -> data.iw.volume_control &= ~0x83;
    voice -> data.iw.volume_control |= 0x20;
#if 0
    printk( " -%i- program envelope - old_volume = 0x%x, next = 0x%x\n", voice -> number, old_volume, next );
#endif
    CLI( &flags );
    gf1_select_voice( card, voice -> number );
    if ( old_volume > next )
      {
        gus_write8( card, GF1_VB_VOLUME_START, next );
        gus_write8( card, GF1_VB_VOLUME_END, old_volume );
        voice -> data.iw.volume_control |= 0x40;
      }
     else
      {
        gus_write8( card, GF1_VB_VOLUME_START, old_volume );
        gus_write8( card, GF1_VB_VOLUME_END, next );
        voice -> data.iw.volume_control &= ~0x40;
      }
    gus_write8( card, GF1_VB_VOLUME_RATE, rate );
    gus_write8( card, GF1_VB_VOLUME_CONTROL, voice -> data.iw.volume_control );
    STI( &flags );
    return;
  }
}

static void do_pitch_envelope( gus_card_t *card, gus_voice_t *voice )
{
  unsigned long flags;
  char program_next_ramp;
  short next = 0, index, nattack, nrelease;
  int time = 0;
  static char retriggering = 0;
  
  if ( voice -> data.iw.penv_state != IW_BEFORE && voice -> data.iw.penv_in_progress )
    {
      if ( --voice -> data.iw.penv_timer <= 0 )
        voice -> data.iw.penv_pitch = voice -> data.iw.penv_next;
       else
        {
          voice -> data.iw.penv_pitch += voice -> data.iw.penv_increment;
          goto __pitch_return;
        }
    }
  while ( 1 ) {
    program_next_ramp = 0;
    voice -> data.iw.penv_in_progress = 0;
#if 0
    printk( "state = %i\n", voice -> data.iw.penv_state );
#endif
    switch ( voice -> data.iw.penv_state ) {
      case IW_BEFORE:
        voice -> data.iw.penv_index = 0;	/* initialize envelope pointer */
        voice -> data.iw.penv_state++;
        voice -> data.iw.penv_pitch = 1024 << 16;
        break;
      case IW_ATTACK:
        index = voice -> data.iw.penv_index;
#if 0
        printk( "voice -> data.iw.per = 0x%x\n", voice -> data.iw.per );
#endif
        if ( index < voice -> data.iw.per -> nattack )
          {
            program_next_ramp = 1;
            next = voice -> data.iw.per -> points[ index ].offset;
            time = voice -> data.iw.per -> points[ index ].rate;
            voice -> data.iw.penv_index++;
          }
         else
          voice -> data.iw.penv_state++;
        break;
      case IW_SUSTAIN:
        program_next_ramp = 1;
        next = voice -> data.iw.per -> sustain_offset;
        time = voice -> data.iw.per -> sustain_rate;
        voice -> data.iw.penv_state++;
        break;
      case IW_VAR_RELEASE:
        index = voice -> data.iw.penv_index;
        nattack = voice -> data.iw.per -> nattack;
        nrelease = voice -> data.iw.per -> nrelease;
        if ( index <= nattack &&
             voice -> layer -> data.iw.penv.mode == GUS_STRU_IW_ENV_M_SUSTAIN )
          {
            if ( !(voice -> flags & VFLG_RELEASING) )
              {
                voice -> flags &= ~VFLG_EFFECT_TIMER2;
                goto __pitch_return;
              }
          }
        if ( index <= nattack &&
             voice -> layer -> data.iw.penv.mode == GUS_STRU_IW_ENV_M_ONE_SHOT )
          index = voice -> data.iw.penv_index = nattack;
        if ( index < ( nattack + nrelease ) )
          {
            program_next_ramp = 1;
            next = voice -> data.iw.per -> points[ index ].offset;
            time = voice -> data.iw.per -> points[ index ].rate;
            voice -> data.iw.penv_index++;
            index++;
          }
         else
          voice -> data.iw.penv_state++;
        break;
      case IW_FINAL_RELEASE:
        program_next_ramp = 1;
        next = 1024;
        time = voice -> data.iw.per -> release_rate;
        voice -> data.iw.penv_state++;
        /* Time of 0 shuts off enveloping, and leaves pitch        */
        /* at current offset. This is useful for indexed detuning. */
        if ( !time )
          {
            voice -> flags &= ~VFLG_EFFECT_TIMER2;
            goto __pitch_return;
          }
        break;
      case IW_DONE:
        if ( (voice -> layer -> data.iw.penv.flags & GUS_STRU_IW_ENV_F_RETRIGGER) && !retriggering )
          {
            if ( !(voice -> flags & VFLG_RELEASING) )
              {
                retriggering = 1;
                if ( voice -> layer -> data.iw.flags & GUS_LAYER_IW_F_RETRIGGER )
                  {
                    gus_gf1_stop_voice( card, voice -> number );
                    if ( trigger_layer( card, voice -> note, voice, GUS_LAYER_IW_E_RETRIG ) )
                      {
                        retriggering = 0;
                        return;
                      }
                  }
                retriggering = 0;
                voice -> data.iw.penv_state = IW_BEFORE;
                continue;
              }
          }
        voice -> flags &= ~VFLG_EFFECT_TIMER2;
        goto __pitch_return;
      case IW_VOLUME:
        break;
    }
    if ( !program_next_ramp ) continue;
#if 0
    printk( " -%i- next = %i, time = %i, pitch = %i\n", voice -> number, next, time, voice -> data.iw.penv_pitch >> 16 );
#endif
    voice -> data.iw.penv_next = ((int)next) << 16;
    if ( !time )
      {
        voice -> data.iw.penv_pitch = voice -> data.iw.penv_next;
        continue;
      }
    voice -> data.iw.penv_in_progress = 1;
#if 0
    printk( "card -> gf1.effect_timer = %i\n", card -> gf1.effect_timer );
#endif
    if ( card -> gf1.timer_slave )
      time = ( time * 20000L ) / card -> gf1.timer_master_card -> gf1.effect_timer;
     else
      time = ( time * 20000L ) / card -> gf1.effect_timer;
    voice -> data.iw.penv_timer = time;
    voice -> data.iw.penv_increment = ( voice -> data.iw.penv_next - voice -> data.iw.penv_pitch ) / time;
    voice -> flags |= VFLG_EFFECT_TIMER2;
    break;
  }
  __pitch_return:
#if 0
  printk( "pitch_envelope: -%i- fc_register = 0x%x, pitch = 0x%x\n", voice -> number, voice -> data.iw.fc_register, voice -> data.iw.penv_pitch );
#endif
  voice -> fc_register = (unsigned short)(((unsigned int)voice -> data.iw.fc_register * (voice -> data.iw.penv_pitch>>16)) >> 10);
  CLI( &flags );
  gf1_select_voice( card, voice -> number );
  gus_i_write16( card, GF1_VW_FREQUENCY, voice -> fc_register + voice -> lfo_fc );
  STI( &flags );
#if 0
  printk( "pitch_envelope: freq = 0x%x\n", gus_i_read16( card, GF1_VW_FREQUENCY ) );
#endif
}

static int trigger_layer( gus_card_t *card,
			  gus_note_t *note,
			  gus_voice_t *voice,
			  int layer_event )
{
  unsigned long flags;
  gus_channel_t *channel = note -> channel;
  gus_layer_t *layer = voice -> layer;
  gus_wave_t *wave = NULL;
  unsigned short sn, sn_interp = 0;
  unsigned char voice_control, volume_control, w_16;
  unsigned int addrs;
  short atten = 0;

#if 0
  printk( "trigger_layer: note = %i, voice = %i, event = %i\n",
  			note -> number, voice -> number, layer_event );
#endif
  if ( !channel || !layer )
    {
      if ( voice -> flags & VFLG_USED )
        gus_voice_free( card, voice );
      return 1;
    }
  if ( layer_event <= GUS_LAYER_IW_E_KDOWN ||
       layer_event == GUS_LAYER_IW_E_LEGATO )
    {
      /* do kbd_freq scale and find waveform to play */
      if ( layer -> data.iw.freq_scale != 1024 )
        {
          unsigned int l1;
          
          /* translate, scale, translate */
          l1 = ((unsigned int)note -> number - layer -> data.iw.freq_center) * layer -> data.iw.freq_scale;
          /* store the remainder */
          sn_interp = (unsigned short)(l1 & ((1 << 10) - 1));
          sn = (unsigned short)((l1 >> 10) + layer -> data.iw.freq_center);
        }
       else
        sn = note -> number;
      /* search for correct wave in wave list */
      for ( wave = layer -> wave; wave; wave = wave -> next )
        if ( (unsigned short)wave -> data.iw.high_note >= sn &&
             (unsigned short)wave -> data.iw.low_note <= sn ) break;
      if ( !wave )
        {
#ifdef FREE_DEBUG
          PRINTK( "iw free (1)\n" );
#endif
          if ( voice -> flags & VFLG_USED )
            gus_voice_free( card, voice );
          return 0;		/* ignore this layer */
        }
      voice -> wave = wave;
      voice -> data.iw.ofc_reg = gus_compute_freq( gus_scale_table[ sn ], wave -> data.iw.sample_ratio, 44100UL );
      if ( layer -> data.iw.freq_scale != 1024 )
        {
          /* inter value is percentage between this semitone and next semitone */
          /* This equation is the linear interpolation between a frequency and */
          /* another frequency one semitone highter in pitch.                  */
          /* or (1.059F - F) * interp + F or                                   */
          /* ((((K * log2(1/12) * interp) / K) + K) * original) / K            */
          /* where K is a 2^10                                                 */
          voice -> data.iw.ofc_reg = (unsigned short)(((((61L * sn_interp)>>10) + 1024L) * voice -> data.iw.ofc_reg) >> 10);
        }
#ifdef GUSCFG_INTERWAVE
      if ( !card -> gf1.enh_mode )
#endif
        voice -> data.iw.ofc_reg = ( voice -> data.iw.ofc_reg * 44100L ) / card -> gf1.playback_freq;
    }
  if ( layer_event <= GUS_LAYER_IW_E_KDOWN )
    {
      struct GUS_STRU_IW_ENV_RECORD *ver, *per;
      short pan_offset;
    
      ver = find_envelope( &layer -> data.iw.venv, note -> number, note -> velocity );
      if ( !ver || ( layer_event == GUS_LAYER_IW_E_KUP &&
                     layer -> data.iw.venv.mode != GUS_STRU_IW_ENV_M_ONE_SHOT ) )
        {
#ifdef FREE_DEBUG
          PRINTK( "iw free (2)\n" );
#endif
          if ( voice -> flags & VFLG_USED )
            gus_voice_free( card, voice );
          return 0;
        }
      per = find_envelope( &layer -> data.iw.penv, note -> number, note -> velocity );
      /* sum all attenuations together */
      /* velocity attenuation is based on x^2... since hardware has */
      /* exponential volume table, we have to use log(x^2) which */
      /* turns out to be 2log(x) or 2 * the linear volume table! */
#if 1
      voice -> data.iw.vel_atten = VOLUME( note -> velocity );
#else
      voice -> data.iw.vel_atten = MAX_VOLUME - gus_lvol_to_gvol_raw( note -> velocity << 9 );
#endif
      voice -> data.iw.layer_atten = layer -> data.iw.attenuation << 5;
      voice -> data.iw.wave_atten = wave -> data.iw.attenuation << 5;
      voice -> data.iw.ver = ver;
      voice -> data.iw.per = per;
      atten = VOLUME( card -> mixer.soft_volume_level ) +
              VOLUME( channel -> volume ) +
              VOLUME( channel -> expression ) +
              voice -> data.iw.layer_atten +
              voice -> data.iw.wave_atten;
#if 0
      printk( "====> atten = %i, cvol = %i, cexp = %i, layr = %i, wave = %i\n", 
      	      atten,
 	      VOLUME( channel -> volume ),
              VOLUME( channel -> expression ),
              voice -> data.iw.layer_atten,
              voice -> data.iw.wave_atten );
#endif
      /* pan */
      if ( layer -> data.iw.pan_freq_scale )
        pan_offset = (((short)(note -> number - 64) * layer -> data.iw.pan_freq_scale) >> 7) + (layer -> data.iw.pan - 64);
       else
        pan_offset = layer -> data.iw.pan - 64;
      voice -> data.iw.pan_offset = pan_offset;
      pan_offset = pan_offset + (short)channel -> pan;
      if ( pan_offset < 0 ) pan_offset = 0;
      if ( pan_offset > 127 ) pan_offset = 127;
      voice -> data.iw.pan = pan_offset;
    }
  /* pan offset registers */
  if ( channel -> gf1_pitchbend == 1024 )
    voice -> data.iw.fc_register = voice -> data.iw.ofc_reg;
   else
    {
      unsigned int newfc;
    
      newfc = (((unsigned int)voice -> data.iw.ofc_reg * channel -> gf1_pitchbend) >> 10);
      if ( newfc > 65535 ) newfc = 65535;
      voice -> data.iw.fc_register = newfc;
    }
  if ( layer_event != GUS_LAYER_IW_E_RETRIG )
    {
      /* if there is a pitch envelope, the fc will be set later */
      if ( !voice -> data.iw.per )
        {
          voice -> fc_register = voice -> data.iw.fc_register;
          CLI( &flags );
          gf1_select_voice( card, voice -> number );
          gus_write16( card, GF1_VW_FREQUENCY, voice -> data.iw.fc_register + voice -> lfo_fc );
          STI( &flags );
        }
      if ( layer_event == GUS_LAYER_IW_E_LEGATO )
        {
          if ( voice -> data.iw.per )
            {
              voice -> fc_register = (unsigned short)(((unsigned int)voice -> data.iw.fc_register * (voice -> data.iw.penv_pitch >> 16)) >> 10);
              CLI( &flags );
              gf1_select_voice( card, voice -> number );
              gus_write16( card, GF1_VW_FREQUENCY, voice -> fc_register + voice -> lfo_fc );
              STI( &flags );
            }
          return 1;
        }
      /* activate voice */
      if ( layer -> data.iw.velocity_mode == GUS_LAYER_IW_V_VEL_TIME )
        atten += voice -> data.iw.vel_atten;
#ifdef GUSCFG_INTERWAVE
      if ( card -> gf1.enh_mode )
        {
          unsigned char mode;
          short vlo, vro;
        
          vlo = ATTEN( (ATTEN_TABLE_SIZE-1) - voice -> data.iw.pan );
          vro = ATTEN( voice -> data.iw.pan );
          if ( voice -> data.iw.pan != ATTEN_TABLE_SIZE - 1 && voice -> data.iw.pan != 0 )
            {
              vlo >>= 1;
              vro >>= 1;
            }
          mode = 0x20;				/* offset enable */
          if ( wave -> format & GUS_WAVE_ROM ) mode |= 0x80;
          if ( wave -> format & GUS_WAVE_ULAW ) mode |= 0x40;
          CLI( &flags );
          gf1_select_voice( card, voice -> number );
          gus_write8( card, GF1_VB_MODE, mode );	/* activate voice */
          set_volume( card, atten + vlo, atten + vro, 1 );
          STI( &flags );
          change_effect_volume( card, voice, atten, 1 );
        }
       else
#endif
        {
          if ( atten > MAX_VOLUME ) atten = MAX_VOLUME;
          voice -> data.iw.gf1_volume = ( MAX_VOLUME - atten ) << 4;
          CLI( &flags );
          gf1_select_voice( card, voice -> number );
          gus_write8( card, GF1_VB_PAN, voice -> data.iw.pan >> 3 );
          STI( &flags );
        }
    }
  volume_control = 0;
  wave = voice -> wave;
  if ( wave -> format & GUS_WAVE_LOOP )
    {
      voice_control = 0x08;
      if ( wave -> format & GUS_WAVE_BIDIR )
        voice_control |= 0x10;
    }
   else
    /* non looped sound should generate interrupt when done */
    voice_control = 0x20;		/* IRQ enable */
  addrs = wave -> begin.memory;
  if ( wave -> format & GUS_WAVE_16BIT )
    {
      voice_control |= 0x04;
      w_16 = 1;
    }
   else
    w_16 = 0;
  CLI( &flags );
  gf1_select_voice( card, voice -> number );
  if ( wave -> format & GUS_WAVE_BACKWARD )
    {
      voice_control |= 0x40;
      gus_write_addr( card, GF1_VA_CURRENT, addrs + wave -> loop_end, w_16 );
    }
   else
    gus_write_addr( card, GF1_VA_CURRENT, addrs, w_16 );
  gus_write_addr( card, GF1_VA_START, addrs + wave -> loop_start, w_16 );
  gus_write_addr( card, GF1_VA_END, addrs + wave -> loop_end, w_16 );
  STI( &flags );
  voice -> data.iw.voice_control = voice_control;
  voice -> data.iw.volume_control = volume_control;
  voice -> flags |= VFLG_RUNNING;
  voice -> data.iw.venv_state = IW_BEFORE;
  voice -> data.iw.penv_state = IW_BEFORE;
  do_volume_envelope( card, voice );
  if ( !(voice -> flags & VFLG_RUNNING) )
    {
      /* volume envelope was flat at 0 volume */
#ifdef FREE_DEBUG
      PRINTK( "iw free (3)\n" );
#endif
      if ( voice -> flags & VFLG_USED )
        gus_voice_free( card, voice );
      return 0;
    }
  if ( voice -> data.iw.per ) do_pitch_envelope( card, voice );
#ifdef GUSCFG_INTERWAVE
  /* if channel vibrato or build-in vibrato then start the LFO hardware */
  if ( layer -> data.iw.vibrato.depth || channel -> vib_depth )
    note_vibrato( card, voice );
  if ( layer -> data.iw.tremolo.depth )
    note_tremolo( card, voice );
#endif
#if 0
  gus_print_voice_registers( card );
#endif
  return 1;
}

static int note_legato( gus_card_t *card, gus_channel_t *channel, unsigned char note )
{
  gus_note_t *pnote, *pnote_next;
  gus_voice_t *voice, *voice_next;
  
  for ( pnote = channel -> notes; pnote; pnote = pnote_next )
    {
      pnote_next = pnote -> next;
      if ( pnote -> flags & NFLG_RELEASING ) continue;
      pnote -> number = note;
      for ( voice = pnote -> voices; voice; voice = voice_next )
        {
          voice_next = voice -> next;
          trigger_layer( card, pnote, voice, GUS_LAYER_IW_E_LEGATO );
        }
      return 1;
    }
  return 0;
}

/*
 *  voice specific commands - not implemented for InterWave instruments
 */

static void voice_start( gus_card_t *card, gus_voice_t *voice )
{
}

static void voice_go( gus_card_t *card, gus_voice_t *voice )
{
}

static void voice_stop( gus_card_t *card, gus_voice_t *voice, unsigned char mode )
{
}

static void voice_control( gus_card_t *card, gus_voice_t *voice, unsigned char control )
{
}

static void voice_freq( gus_card_t *card, gus_voice_t *voice, unsigned int freq )
{
}

static void voice_volume( gus_card_t *card, gus_voice_t *voice, unsigned short volume )
{
}

static void voice_loop( gus_card_t *card, gus_voice_t *voice, unsigned int start, unsigned int end )
{
}

static void voice_ramp( gus_card_t *card, gus_voice_t *voice, unsigned char start, unsigned char end, unsigned char rate, unsigned char control )
{
}

static void voice_pos( gus_card_t *card, gus_voice_t *voice, unsigned int position )
{
}

static void voice_pan( gus_card_t *card, gus_voice_t *voice, unsigned short pan )
{
}

static void voice_lfo( gus_card_t *card, gus_voice_t *voice, unsigned char *data )
{
}

static void voice_private1( gus_card_t *card, gus_voice_t *voice, unsigned char *data )
{
}

/*
 *  note specific commands
 */

static void note_stop( gus_card_t *card, gus_voice_t *voice, int wait )
{
  unsigned long flags;
  unsigned short old_volume;
  
  if ( (voice -> flags & (VFLG_RUNNING | VFLG_USED)) == (VFLG_RUNNING | VFLG_USED) )
    {
      voice -> flags &= ~VFLG_EFFECT_TIMER2;
      CLI( &flags );
      gf1_select_voice( card, voice -> number );
      gus_ctrl_stop( card, GF1_VB_ADDRESS_CONTROL );
      gus_ctrl_stop( card, GF1_VB_VOLUME_CONTROL );
      old_volume = gus_read16( card, GF1_VW_VOLUME ) >> 8;
      STI( &flags );
      if ( old_volume > MIN_OFFSET )
        {
          if ( wait )
            {
              voice -> data.iw.volume_control = 0x40 | 0x20;
              /* make interrupt look like end of volume envelope */
              voice -> data.iw.venv_state = IW_DONE;
            }
           else
            voice -> data.iw.volume_control = 0x40;
          CLI( &flags );
          gf1_select_voice( card, voice -> number );
          gus_write8( card, GF1_VB_VOLUME_START, MIN_OFFSET );
          gus_write8( card, GF1_VB_VOLUME_RATE, 63 );
          gus_write8( card, GF1_VB_VOLUME_CONTROL, voice -> data.iw.volume_control );
          STI( &flags );
        }
       else
        {
          if ( wait )
            {
              /* make interrupt look like end of envelope */
              voice -> data.iw.venv_state = IW_DONE;
              voice -> flags |= VFLG_RELEASING;
              do_volume_envelope( card, voice );
            }
        }
    }
}

static void note_wait( gus_card_t *card, gus_voice_t *voice )
{
  unsigned long flags;

  if ( (voice -> flags & (VFLG_RUNNING | VFLG_USED)) == (VFLG_RUNNING | VFLG_USED) )
    {
      CLI( &flags );
      gf1_select_voice( card, voice -> number );
      /* wait for volume to stop */
      while ( ( gus_read8( card, GF1_VB_ADDRESS_CONTROL ) & 0x03 ) == 0 );
#if 0
      printk( "note wave - old volume = 0x%x\n", gus_i_read16( card, GF1_VW_VOLUME ) );
#endif
#ifdef GUSCFG_INTERWAVE
      if ( card -> gf1.enh_mode )
        {
          gus_write16( card, GF1_VW_VOLUME, 0 );
          gus_write8( card, GF1_VB_ACCUMULATOR, 0 ); /* disable effects */
          gus_write16( card, GF1_VW_EFFECT_VOLUME, 0 );
          gus_write16( card, GF1_VW_EFFECT_VOLUME_FINAL, 0 );
        }
       else
#endif
        gus_write16( card, GF1_VW_VOLUME, MIN_VOLUME );
      STI( &flags );
      gus_lfo_shutdown( card, voice -> number, GUS_LFO_VIBRATO );
      gus_lfo_shutdown( card, voice -> number, GUS_LFO_TREMOLO );
      if ( voice -> flags & VFLG_WAIT_FOR_START )
        {
#ifdef FREE_DEBUG
          PRINTK( "iw free (4)\n" );
#endif
          if ( voice -> flags & VFLG_USED )
            gus_voice_free( card, voice );
        }
       else
        gus_voice_and_note_free( card, voice );
    }
}

static void note_off( gus_card_t *card, gus_voice_t *voice )
{
  gus_note_t *note;

  note = voice -> note;
  if ( !note || !voice -> layer || (voice -> flags & VFLG_RELEASING) ) return;
  /* mark voice as releasing - even one-shots need to know when to stop */
  /* envelope and layer retriggering                                    */
  voice -> flags |= VFLG_RELEASING;
  if ( voice -> layer -> data.iw.venv.mode == GUS_STRU_IW_ENV_M_ONE_SHOT ) return;
  voice -> priority = ~0;		/* min priority */
  note -> flags &= ~NFLG_SUSTAIN;
  note -> flags |= NFLG_RELEASING;
  /* force envelope to advance to release part */
  voice -> flags &= ~VFLG_SUSTAINING;
  if ( voice -> data.iw.venv_state <= IW_VAR_RELEASE &&
       voice -> data.iw.venv_index <= voice -> data.iw.ver -> nattack )
    {
      voice -> data.iw.venv_state = IW_VAR_RELEASE;
      do_volume_envelope( card, voice );
    }
  if ( voice -> layer && voice -> data.iw.per )
    {
      if ( voice -> layer -> data.iw.penv.mode == GUS_STRU_IW_ENV_M_ONE_SHOT ) return;
      if ( voice -> data.iw.penv_state <= IW_VAR_RELEASE && voice -> data.iw.penv_index <= voice -> data.iw.per -> nattack )
        {
          voice -> flags &= ~VFLG_EFFECT_TIMER2;
          voice -> data.iw.penv_state = IW_VAR_RELEASE;
          voice -> data.iw.penv_in_progress = 0;
          voice -> data.iw.penv_pitch = voice -> data.iw.penv_next;
          do_pitch_envelope( card, voice );
        }
    }
}

static void note_volume( gus_card_t *card, gus_voice_t *voice )
{
  unsigned long flags;
  gus_channel_t *channel;
  int atten;
  short pan;
  
  if ( !voice -> note ) return;
  channel = voice -> note -> channel;
  if ( !channel ) return;
  atten = VOLUME( card -> mixer.soft_volume_level ) +
          VOLUME( channel -> volume ) +
          VOLUME( channel -> expression ) +
          voice -> data.iw.layer_atten +
          voice -> data.iw.wave_atten;
  if ( !voice -> layer ) return;
  if ( voice -> layer -> data.iw.velocity_mode == GUS_LAYER_IW_V_VEL_TIME )
    atten += voice -> data.iw.vel_atten;
  pan = voice -> data.iw.pan_offset + (short)channel -> pan;
  if ( pan < 0 ) pan = 0;
  if ( pan > 127 ) pan = 127;
#ifdef GUSCFG_INTERWAVE
  if ( card -> gf1.enh_mode )
    {
      short vlo, vro;
    
      vlo = ATTEN( (ATTEN_TABLE_SIZE-1) - pan );
      vro = ATTEN( pan );
      if ( pan != (ATTEN_TABLE_SIZE-1) && pan != 0 )
        {
          vlo >>= 1;
          vro >>= 1;
        }
      CLI( &flags );
      gf1_select_voice( card, voice -> number );
      set_volume( card, atten + vlo, atten + vro, 0 );
      STI( &flags );
      change_effect_volume( card, voice, atten, 0 );
    }
   else
#endif
    {
      CLI( &flags );
      gf1_select_voice( card, voice -> number );
      gus_write8( card, GF1_VB_PAN, pan >> 3 );
      STI( &flags );
      if ( atten > MAX_VOLUME ) atten = MAX_VOLUME;
      voice -> data.iw.gf1_volume = ( MAX_VOLUME - atten ) << 4;
      voice -> data.iw.venv_state_prev = voice -> data.iw.venv_state;
      voice -> data.iw.venv_state = IW_VOLUME;
      do_volume_envelope( card, voice );
    }
}

static void note_pitchbend( gus_card_t *card, gus_voice_t *voice )
{
  unsigned long flags;
  unsigned int newfc;
  unsigned short pitchbend;
  
  if ( !voice -> note ) return;
  if ( !voice -> note -> channel ) return;
  pitchbend = voice -> note -> channel -> gf1_pitchbend;
  if ( pitchbend == 1024 )
    voice -> data.iw.fc_register = voice -> data.iw.ofc_reg;
   else
    {
      newfc = ((unsigned int)voice -> data.iw.ofc_reg * pitchbend) >> 10;
      if ( newfc > 65535 ) newfc = 65535;
      voice -> data.iw.fc_register = (unsigned short)newfc;
    }
  voice -> fc_register =
       	voice -> data.iw.per ?
     	  (unsigned short)(((unsigned int)voice -> data.iw.fc_register * (voice -> data.iw.penv_pitch>>16))>>10) :
       	  voice -> data.iw.fc_register;
  CLI( &flags );
  gf1_select_voice( card, voice -> number );
  gus_write16( card, GF1_VW_FREQUENCY, voice -> fc_register + voice -> lfo_fc );
  STI( &flags );
}

static void note_vibrato( gus_card_t *card, gus_voice_t *voice )
{
  gus_channel_t *channel;
  short depth, bdepth, current_depth, sweep = 0, freq;
  
  if ( !voice -> note || !voice -> layer ) return;
  channel = voice -> note -> channel;
  if ( !channel ) return;
  bdepth = voice -> layer -> data.iw.vibrato.depth;
  if ( channel -> vib_depth )
    {
      if ( bdepth < 0 )
        depth = gus_compute_vibrato( -(((channel -> vib_depth >> 1) * (127 + bdepth) >> 7) - bdepth), voice -> data.iw.fc_register );
       else
        depth = gus_compute_vibrato( bdepth + ((channel -> vib_depth >> 1) * (127 - bdepth) >> 7), voice -> data.iw.fc_register );
      if ( bdepth )
        sweep = voice -> layer -> data.iw.vibrato.sweep * (127 - channel -> vib_depth) >> 7;  
    }
   else
    {
      depth = gus_compute_vibrato( bdepth, voice -> data.iw.fc_register );
      if ( depth )
        sweep = voice -> layer -> data.iw.vibrato.sweep;
    }
  current_depth = !sweep ? depth << 5 : 0;
  if ( voice -> flags & VFLG_VIBRATO_RUNNING )
    {
      if ( depth )
        gus_lfo_change_depth( card, voice -> number, GUS_LFO_VIBRATO, depth );
       else
        {
          gus_lfo_shutdown( card, voice -> number, GUS_LFO_VIBRATO );
          voice -> flags &= ~VFLG_VIBRATO_RUNNING;
        }
    }
   else
  if ( depth )
    {
      freq = voice -> layer -> data.iw.vibrato.freq;
      if ( !freq ) freq = 523;
      gus_lfo_setup( card, 
                        voice -> number, GUS_LFO_VIBRATO, freq,
        		current_depth, depth, sweep,
        		!channel -> vib_depth ?
        		  voice -> layer -> data.iw.vibrato.shape :
        		  GUS_STRU_IW_LFO_SHAPE_TRIANGLE );
      voice -> flags |= VFLG_VIBRATO_RUNNING;
    }
}

static void note_tremolo( gus_card_t *card, gus_voice_t *voice )
{
  gus_channel_t *channel;
  short depth, bdepth, sweep = 0, current_depth, freq;
 
  if ( !voice -> note || !voice -> layer ) return;
  channel = voice -> note -> channel;
  if ( !channel ) return;
  bdepth = voice -> layer -> data.iw.tremolo.depth;
  if ( channel -> trem_depth )
    {
      if ( bdepth < 0 )
        depth = -((channel -> trem_depth * (MAX_TDEPTH + bdepth) / MAX_TDEPTH) - bdepth);
       else
        depth = bdepth + (channel -> trem_depth * (MAX_TDEPTH - bdepth) / MAX_TDEPTH);
      if ( bdepth )
        sweep = voice -> layer -> data.iw.tremolo.sweep * (127 - channel -> trem_depth) >> 7;
    }
   else
    {
      depth = bdepth;
      if ( depth )
        sweep = voice -> layer -> data.iw.tremolo.sweep;
    }
  current_depth = !sweep ? depth << 5 : 0;
  if ( voice -> flags & VFLG_TREMOLO_RUNNING )
    {
      if ( depth )
        gus_lfo_change_depth( card, voice -> number, GUS_LFO_TREMOLO, depth );
       else
        {
          gus_lfo_shutdown( card, voice -> number, GUS_LFO_TREMOLO );
          voice -> flags &= ~VFLG_TREMOLO_RUNNING;
        }
    }
   else
  if ( depth )
    {
      freq = voice -> layer -> data.iw.tremolo.freq;
      if ( !freq ) freq = 523;
      gus_lfo_setup( card, voice -> number, GUS_LFO_TREMOLO,
      			freq, current_depth, depth, sweep,
      			voice -> layer -> data.iw.tremolo.shape );
      voice -> flags |= VFLG_TREMOLO_RUNNING;
    }
}

/*
 *  channel specific commands
 */

static void chn_trigger_down( gus_card_t *card, gus_channel_t *channel, gus_instrument_t *instrument, unsigned char note, unsigned char velocity, unsigned char priority )
{
  unsigned long flags;
  int idx;
  short nvoices_used = 0;
  gus_layer_t *layer;
  gus_note_t *pnote;
  gus_voice_t *voice = NULL, *voice_next;

  if ( channel -> flags & CFLG_LEGATO )
    if ( note_legato( card, channel, note ) ) return;
  if ( ( pnote = gus_note_alloc( card, channel ) ) == NULL ) return;
  pnote -> commands = &note_commands;
  pnote -> voices = NULL;
  pnote -> number = note;
  pnote -> velocity = velocity;
  pnote -> channel = channel;
  pnote -> instrument = instrument;
  pnote -> priority = priority;
  for ( layer = instrument -> info.layer, idx = 0; layer; layer = layer -> next, idx++ )
    {
      if ( layer -> data.iw.layer_event != GUS_LAYER_IW_E_KDOWN ) continue;
      if ( instrument -> layer == GUS_INSTR_L_NONE && idx > 0 ) break;
      if ( instrument -> layer == GUS_INSTR_L_VELOCITY &&
           ( velocity < layer -> data.iw.low_range ||
             velocity > layer -> data.iw.high_range ) ) continue;
      if ( instrument -> layer == GUS_INSTR_L_FREQUENCY &&
           ( note < layer -> data.iw.low_range ||
             note > layer -> data.iw.high_range ) ) continue;
      if ( ( voice = gus_voice_alloc( card, pnote, priority, stop_note_voice ) ) == NULL ) continue;
      voice -> flags |= VFLG_WAIT_FOR_START;
      voice -> commands = &voice_commands;
      voice -> note = pnote;
      voice -> layer = layer;
      voice -> instrument = instrument;
      if ( trigger_layer( card, pnote, voice, GUS_LAYER_IW_E_KDOWN ) )
        nvoices_used++;
    }
  if ( !nvoices_used )
    {
#if 0
      printk( "trigger down failed - note = %i, velocity = %i, channel = %i\n", note, velocity, channel -> number );
#endif
      gus_note_free( card, pnote );
    }
   else
    {
      CLI( &flags );
      for ( voice = pnote -> voices; voice; voice = voice_next )
        {
          voice_next = voice -> next;
          gf1_select_voice( card, voice -> number );
          gus_write8( card, GF1_VB_ADDRESS_CONTROL, voice -> data.iw.voice_control );
          voice -> flags &= ~VFLG_WAIT_FOR_START;
        }
      STI( &flags );
    }
}

static void chn_trigger_up( gus_card_t *card, gus_note_t *note )
{
  unsigned long flags;
  gus_layer_t *layer;
  gus_voice_t *voice, *voice_next;
  gus_instrument_t *instrument = note -> instrument;
  short nvoices_used, idx, velocity, note_number;

  nvoices_used = 0;
  velocity = note -> velocity;
  note_number = note -> number;
  for ( layer = instrument -> info.layer, idx = 0; layer; layer = layer -> next, idx++ )
    {
      if ( layer -> data.iw.layer_event != GUS_LAYER_IW_E_KUP ) continue;
      if ( instrument -> layer == GUS_INSTR_L_NONE && idx > 0 ) break;
      if ( instrument -> layer == GUS_INSTR_L_VELOCITY &&
           ( velocity < layer -> data.iw.low_range ||
             velocity > layer -> data.iw.high_range ) ) continue;
      if ( instrument -> layer == GUS_INSTR_L_FREQUENCY &&
           ( note_number < layer -> data.iw.low_range ||
             note_number > layer -> data.iw.high_range ) ) continue;
      if ( ( voice = gus_voice_alloc( card, note, note -> priority, stop_note_voice ) ) == NULL ) continue;
      voice -> commands = &voice_commands;
      voice -> note = note;
      voice -> layer = layer;
      voice -> instrument = instrument;
      if ( trigger_layer( card, note, voice, GUS_LAYER_IW_E_KDOWN ) )
        nvoices_used++;
    }
  if ( nvoices_used )
    {
      CLI( &flags );
      for ( voice = note -> voices; voice; voice = voice_next )
        {
          voice_next = voice -> next;
          gf1_select_voice( card, voice -> number );
          gus_write8( card, GF1_VB_ADDRESS_CONTROL, voice -> data.iw.voice_control );
        }
      STI( &flags );
    }
}

static void chn_control( gus_card_t *card, gus_channel_t *channel, unsigned short p1, unsigned short p2 )
{
  switch ( p1 ) {
    case GUS_MCTL_E1_REVERB_DEPTH:
      channel -> effect1_level = VOLUME( p2 );
      gus_engine_midi_recompute_volume( card, channel );
      break;
    case GUS_MCTL_E3_CHORUS_DEPTH:
      channel -> effect3_level = VOLUME( p2 );
      gus_engine_midi_recompute_volume( card, channel );
      break;
  }
}

/*
 *
 */
 
void gus_gf1_interwave_init( gus_card_t *card )
{
  gus_engine_instrument_register( GUS_INSTR_INTERWAVE, &voice_commands, &note_commands, &channel_commands );
}
