/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 *  Routines for control of ESS ES-1688 chip
 */
  
#include "driver.h"
#include "pcm.h"
#include "ulaw.h"

#ifdef GUSCFG_ESS

/* playback flags */

#define PFLG_NONE       0x0000
#define PFLG_USED       0x0001
#define PFLG_FLUSH      0x0002

/* record flags */

#define RFLG_NONE       0x0000
#define RFLG_USED       0x0001
#define RFLG_FLUSH      0x0002

int ess_dsp_command( gus_card_t *card, unsigned char val )
{
  int i;
  
  for ( i = 10000; i; i-- )
    if ( ( INB( ESSP( card, STATUS ) ) & 0x80 ) == 0 ) {
      OUTB( val, ESSP( card, COMMAND ) );
      return 1;
    }
#ifdef GUSCFG_DEBUG
  printk( "ess_dsp_command: timeout (0x%x)\n", val );
#endif
  return 0;
}

int ess_dsp_get_byte( gus_card_t *card )
{
  int i;
  
  for ( i = 1000; i; i-- )
    if ( INB( ESSP( card, DATA_AVAIL ) ) & 0x80 )
      return INB( ESSP( card, READ ) );
#ifdef GUSCFG_DEBUG
  PRINTK( "ess get byte failed: 0x%x = 0x%x!!!\n", ESSP( card, DATA_AVAIL ), INB( ESSP( card, DATA_AVAIL ) ) );
#endif
  return -1;
}

int ess_write( gus_card_t *card, unsigned char reg, unsigned char data )
{
  if ( !ess_dsp_command( card, reg ) ) return 0;
  return ess_dsp_command( card, data );
}

int ess_read( gus_card_t *card, unsigned char reg )
{
  /* Read a byte from an extended mode register of ES1688 */  
  if ( !ess_dsp_command( card, 0xc0 ) ) return -1;        
  if ( !ess_dsp_command( card, reg ) ) return -1;
  return ess_dsp_get_byte( card );                
}

void ess_mixer_write( gus_card_t *card, unsigned char reg, unsigned char data )
{
  unsigned long flags;
  
  CLI( &flags );
  OUTB( reg, ESSP( card, MIXER_ADDR ) );
  gus_delay2( 1 );
  OUTB( data, ESSP( card, MIXER_DATA ) );
  gus_delay2( 1 );
  STI( &flags );
}

unsigned char ess_mixer_read( gus_card_t *card, unsigned char reg )
{
  unsigned long flags;
  unsigned char result;
  
  CLI( &flags );
  OUTB( reg, ESSP( card, MIXER_ADDR ) );
  gus_delay2( 1 );
  result = INB( ESSP( card, MIXER_DATA ) );
  gus_delay2( 1 );
  STI( &flags );
  return result;
}

int ess_reset( gus_card_t *card )
{
  int i;

  OUTB( 3, ESSP( card, RESET ) );	/* valid only for ESS chips, SB -> 1 */
  gus_delay2( 1 );
  OUTB( 0, ESSP( card, RESET ) );
  gus_delay2( 3 );
  for ( i = 0; i < 1000 && !(INB( ESSP( card, DATA_AVAIL ) ) & 0x80); i++ );
  if ( INB( ESSP( card, READ ) ) != 0xaa ) {
#ifdef GUSCFG_DEBUG
    PRINTK( "ess_reset: failed!!!\n" );
#endif
    return -1;
  }
  ess_dsp_command( card, 0xc6 );	/* enable extended mode */
  return 0;
}

int ess_detect( gus_card_t *card )
{
  unsigned long flags;
  int i, major, minor;
  int cfg1, cfg2, cfg3;

  /*
   *  initialization sequence
   */

  CLI( &flags );                        /* Some ESS1688 cards need this */
  INB( ESSP( card, ENABLE1 ) );
  INB( ESSP( card, ENABLE1 ) );
  INB( ESSP( card, ENABLE1 ) );
  INB( ESSP( card, ENABLE2 ) );
  INB( ESSP( card, ENABLE1 ) );
  INB( ESSP( card, ENABLE2 ) );
  INB( ESSP( card, ENABLE1 ) );
  INB( ESSP( card, ENABLE1 ) );
  INB( ESSP( card, ENABLE2 ) );
  INB( ESSP( card, ENABLE1 ) );  
  INB( ESSP( card, ENABLE0 ) );
  STI( &flags );

  if ( ess_reset( card ) < 0 ) {
#ifdef GUSCFG_DEBUG_DETECT
    printk( "ESS: [0x%x] reset failed... 0x%x\n", card -> ess.port, INB( ESSP( card, READ ) ) );
#endif
    return -1;
  }

  ess_dsp_command( card, 0xe7 ); 	/* return identification */
  
  for ( i = 1000, major = minor = 0; i; i-- ) {
    if ( INB( ESSP( card, DATA_AVAIL ) ) & 0x80 )
      if ( major == 0 )
        major = INB( ESSP( card, READ ) );
       else
        minor = INB( ESSP( card, READ ) );
  }

#ifdef GUSCFG_DEBUG_DETECT
  printk( "ESS: [0x%x] found.. major = 0x%x, minor = 0x%x\n", card -> ess.port, major, minor );
#endif

  card -> ess.version = (major << 8) | minor;

  if ( ( card -> ess.version & 0xfff0 ) != 0x6880 ) {
#ifdef GUSCFG_DEBUG_DETECT
    printk( "ESS: [0x%x] Oops, seems that different ESS chip is detected!!!\n", card -> ess.port );
#endif
    return -1;
  }

#if 0
  for ( i = 0x00; i < 0x80; i++ ) {
    if ( (i & 0x0f) == 0 ) PRINTK( "\n[0x%02x]: ", i ); 
    PRINTK( "%02x:", ess_mixer_read( card, i ) );
  }
  PRINTK( "\n" );
#endif
#if 0
  for ( i = 0x00; i < 0x100; i++ ) {
    if ( (i & 0x0f) == 0 ) PRINTK( "\n[0x%02x]: ", i ); 
    PRINTK( "%02x:", ess_read( card, i ) );
  }
  PRINTK( "\n" );
#endif

  ess_write( card, 0xb1, 0x10 );	/* disable IRQ */
  ess_write( card, 0xb2, 0x00 );	/* disable DMA */

  /*
   *  check if we have GUS Extreme, don't try initialize other ESS cards
   */

  i = INB( ESSP( card, INIT1 ) );
#ifdef GUSCFG_DEBUG_DETECT
  printk( "ESS: [0x%x] (1) init1 = 0x%x\n", card -> ess.port, i );
#endif
  if ( !( i & 0x40 ) ) {
    return -1;
  }
  OUTB( 0, ESSP( card, INIT1 ) );
  i = INB( ESSP( card, INIT1 ) );
#ifdef GUSCFG_DEBUG_DETECT
  printk( "ESS: [0x%x] (2) init1 = 0x%x\n", card -> ess.port, i );
#endif
  if ( i != 0x40 ) {
    return -1;
  }

  /*
   * This is main stuff - enable access to GF1 chip...
   * I'm not sure, if this will work for card which have
   * ES 1688 chip in another place than 0x220.
   *
   * ULTRINIT.EXE:
   * 0x230 = 0,2,3
   * 0x240 = 2,0,1
   * 0x250 = 2,0,3
   * 0x260 = 2,2,1
   */

  switch ( card -> ess.port ) {
    case 0x220:
    case 0x230:
      card -> gf1.port = 0x240;
      cfg1 = 2; cfg2 = 0; cfg3 = 1;
      break;
    case 0x240:
    case 0x250:
      card -> gf1.port = 0x260;
      cfg1 = 2; cfg2 = 2; cfg3 = 1;
      break;
    default:
      PRINTK( "gus: ES 1688 chip found at unknown port!!!" );
      return -1;
  }

  ess_mixer_write( card, 0x40, 0x8b );
  CLI( &flags );
  OUTB( cfg1, ESSP( card, INIT1 ) );	/* 0x230 = 0 */
  OUTB( 0, 0x201 );
  OUTB( cfg2, ESSP( card, INIT1 ) );	/* 0x230 = 2 */
  OUTB( 0, 0x201 );
  OUTB( cfg3, ESSP( card, INIT1 ) );	/* 0x230,0x250 = 3 */
  STI( &flags );
  /* enable joystick, but disable OPL3 */  
  ess_mixer_write( card, 0x40, 0x89 );

  return 0;
}

int ess_init( gus_card_t *card, int start )
{
  static int irqs[ 16 ] = { -1, -1, 0, -1, -1, 1, -1, 2, -1, 0, 3, -1, -1, -1, -1, -1 };
  int cfg, irq_bits, dma, dma_bits;
  
  ess_read( card, 0xb1 );
  ess_read( card, 0xb2 );
  if ( start ) {
    cfg = 0xf0;		/* enable only DMA counter interrupt */
    irq_bits = irqs[ card -> ess.irq & 0x0f ];
    if ( irq_bits < 0 ) {
      PRINTK( "gus: bad IRQ %d for ES 1688 chip!!\n", card -> ess.irq );
      irq_bits = 0;
      cfg = 0x10;
    }
    ess_write( card, 0xb1, cfg | (irq_bits << 2) );
    cfg = 0xf0;		/* extended mode DMA enable */
    dma = card -> dmas[ GUS_DMA_ESS ] -> dma & 7;
    if ( dma > 3 || dma == 2 ) {
      PRINTK( "gus: bad DMA channel %d for ES 1688 chip!!\n", dma );
      dma_bits = 0;
      cfg = 0x00;	/* disable all DMA */
    } else {
      dma_bits = dma;
      if ( dma != 3 ) dma_bits++;
    }
    ess_write( card, 0xb2, cfg | (dma_bits << 2) );
  } else {
    ess_write( card, 0xb1, 0x10 );	/* disable IRQ */
    ess_write( card, 0xb2, 0x00 );	/* disable DMA */
  }
#if 0
  printk( "i[0xb1] = 0x%x\n", ess_read( card, 0xb1 ) );
  printk( "i[0xb2] = 0x%x\n", ess_read( card, 0xb2 ) );
#endif
  ess_read( card, 0xb1 );
  ess_read( card, 0xb2 );
  ess_reset( card );
  return 0;
}

/*
 *
 */

static void ess_open( gus_card_t *card )
{
  if ( card -> ess.mode & ESS_MODE_OPEN ) return;
  ess_reset( card );
}

static void ess_close( gus_card_t *card )
{
  if ( card -> ess.mode & ESS_MODE_OPEN ) return;
  ess_reset( card );
}

static void ess_set_rate( gus_card_t *card, int rate )
{
  int divider;
  unsigned char bits;

  if ( rate < 4000 ) rate = 4000;
  if ( rate > 48000 ) rate = 48000;
  if ( rate > 22000 ) {
    bits = 0x80;
    divider = 256 - (795500 + (rate >> 1)) / rate;
  } else {
    bits = 0x00;
    divider = 128 - (397700 + (rate >> 1)) / rate;
  }
  bits |= divider;
  ess_write( card, 0xa1, bits );
  /* set filter register */
  rate = (rate * 9) / 20;
  divider = 256 - 7160000 / (rate * 82);
  ess_write( card, 0xa2, divider );
}

static void ess_trigger( gus_card_t *card, unsigned char value )
{
  int val;

  val = ess_read( card, 0xb8 );
  if ( val < 0 ) return;		/* something is wrong */
  if ( (val & 0x0f) == value ) return;
#if 0
  printk( "trigger: val = 0x%x, value = 0x%x\n", val, value );
  printk( "trigger: residue = 0x%x\n", get_dma_residue( card -> dmas[ GUS_DMA_ESS ] -> dma ) );
#endif
  ess_write( card, 0xb8, (val & 0xf0) | value );
}

static void ess_start_playback( gus_card_t *card, gus_pcm_t *pcm )
{
  unsigned long flags;
  int dma;
  unsigned int size;
  gus_pcm_channel_t *pchn;

  pchn = &pcm -> chn[ PCM_PLAYBACK ];
  ess_reset( card );
  ess_set_rate( card, pchn -> rate );
  ess_write( card, 0xb8, 4 );	/* auto init DMA mode */
  ess_write( card, 0xa8, (ess_read( card, 0xa8 ) & ~0x03) | (3 - pchn -> voices) );
  ess_write( card, 0xb9, 2 );	/* demand mode (4 bytes/request) */
  if ( pchn -> voices == 1 ) {
    if ( !(pchn -> mode & PCM_MODE_16) ) {	/* 8. bit mono */
      ess_write( card, 0xb6, 0x80 );
      ess_write( card, 0xb7, 0x51 );
      ess_write( card, 0xb7, 0xd0 );
    } else {					/* 16. bit stereo */
      ess_write( card, 0xb6, 0x00 );
      ess_write( card, 0xb7, 0x71 );
      ess_write( card, 0xb7, 0xf4 );
    }
  } else {
    if ( !(pchn -> mode & PCM_MODE_16) ) {	/* 8. bit stereo */
      ess_write( card, 0xb6, 0x80 );
      ess_write( card, 0xb7, 0x51 );
      ess_write( card, 0xb7, 0x98 );
    } else {					/* 16. bit stereo */
      ess_write( card, 0xb6, 0x00 );
      ess_write( card, 0xb7, 0x71 );
      ess_write( card, 0xb7, 0xbc );
    }
  }
  ess_write( card, 0xb1, (ess_read( card, 0xb1 ) & 0x0f) | 0x50 );
  ess_write( card, 0xb2, (ess_read( card, 0xb2 ) & 0x0f) | 0x50 );
  ess_dsp_command( card, ESS_DSP_CMD_SPKON );
  /* --- */
  dma = card -> dmas[ GUS_DMA_ESS ] -> dma;
  CLI( &flags );
  disable_dma( dma );
  clear_dma_ff( dma );
  set_dma_mode( dma, DMA_MODE_WRITE | 0x10 );
  set_dma_addr( dma, (long)virt_to_bus( pchn -> buffer ) );
  set_dma_count( dma, size = pchn -> used_size );
  enable_dma( dma );
  STI( &flags );
  size /= pchn -> blocks;
  size = pchn -> used_size - size;
  size--;
  CLI( &flags );
  ess_write( card, 0xa4, (unsigned char)size );
  ess_write( card, 0xa5, (unsigned char)(size >> 8) );
  STI( &flags );
  PCM_LOCK( pchn, pchn -> tail );
  pchn -> flags |= PCM_FLG_TRIGGER;
  card -> ess.pflags |= PFLG_USED;
  if ( !( pchn -> flags & PCM_FLG_MMAP ) ) {
    pchn -> interrupts++;
    pchn -> processed_bytes += pchn -> block_size;
  }
  ess_trigger( card, 0x05 );			/* GO!!! */
}

static void ess_stop_playback( gus_card_t *card, gus_pcm_t *pcm )
{
  gus_pcm_channel_t *pchn;

  pchn = &pcm -> chn[ PCM_PLAYBACK ];
  ess_trigger( card, 0x00 );
  ess_dsp_command( card, 0xd0 );
  PCM_LOCKZERO( pchn );
  pchn -> flags &= ~PCM_FLG_TRIGGER;
  disable_dma( card -> dmas[ GUS_DMA_ESS ] -> dma );
  card -> ess.pflags &= ~( PFLG_USED | PFLG_FLUSH );	/* done */
}

static void ess_start_record( gus_card_t *card, gus_pcm_t *pcm )
{
  unsigned long flags;
  int dma;
  unsigned int size;
  gus_pcm_channel_t *pchn;
  
  pchn = &pcm -> chn[ PCM_RECORD ];
  ess_reset( card );
  ess_set_rate( card, pchn -> rate );
  ess_dsp_command( card, ESS_DSP_CMD_SPKOFF );
  ess_write( card, 0xb8, 0x0e ); /* auto init DMA mode */
  ess_write( card, 0xa8, (ess_read( card, 0xa8 ) & ~0x03) | (3 - pchn -> voices) );
  ess_write( card, 0xb9, 2 );	/* demand mode (4 bytes/request) */
  if ( pchn -> voices == 1 ) {
    if ( !(pchn -> mode & PCM_MODE_16) ) {	/* 8. bit mono */
      ess_write( card, 0xb7, 0x51 );
      ess_write( card, 0xb7, 0xd0 );
    } else {					/* 16. bit stereo */
      ess_write( card, 0xb7, 0x71 );
      ess_write( card, 0xb7, 0xf4 );
    }
  } else {
    if ( !(pchn -> mode & PCM_MODE_16) ) {	/* 8. bit stereo */
      ess_write( card, 0xb7, 0x51 );
      ess_write( card, 0xb7, 0x98 );
    } else {					/* 16. bit stereo */
      ess_write( card, 0xb7, 0x71 );
      ess_write( card, 0xb7, 0xbc );
    }
  }
  ess_write( card, 0xb1, (ess_read( card, 0xb1 ) & 0x0f) | 0x50 );
  ess_write( card, 0xb2, (ess_read( card, 0xb2 ) & 0x0f) | 0x50);
  /* --- */
  dma = card -> dmas[ GUS_DMA_ESS ] -> dma;
  CLI( &flags );
  disable_dma( dma );
  clear_dma_ff( dma );
  set_dma_mode( dma, DMA_MODE_READ | 0x10 );
  set_dma_addr( dma, (long)virt_to_bus( pchn -> buffer ) );
  clear_dma_ff( dma );
  set_dma_count( dma, size = pchn -> used_size );
  enable_dma( dma );
  STI( &flags );
  size /= pchn -> blocks;
  size = pchn -> used_size - size;
  size--;
  CLI( &flags );
  ess_write( card, 0xa4, (unsigned char)size );
  ess_write( card, 0xa5, (unsigned char)(size >> 8) );
  STI( &flags );
  PCM_LOCK( pchn, pchn -> tail );
  pchn -> flags |= PCM_FLG_TRIGGER;
  card -> ess.rflags |= RFLG_USED;
  if ( !( pchn -> flags & PCM_FLG_MMAP ) )
    pchn -> interrupts++;
  ess_trigger( card, 0x0f );			/* GO!!! */
}

static void ess_stop_record( gus_card_t *card, gus_pcm_t *pcm )
{
  gus_pcm_channel_t *pchn;

  pchn = &pcm -> chn[ PCM_RECORD ];
  ess_trigger( card, 0x00 );
  ess_dsp_command( card, 0xd0 );
  PCM_LOCKZERO( pchn );
  pchn -> flags &= ~PCM_FLG_TRIGGER;
  disable_dma( card -> dmas[ GUS_DMA_ESS ] -> dma );
  card -> ess.rflags &= ~( RFLG_USED | RFLG_FLUSH );	/* done */
}

void ess_interrupt( gus_card_t *card )
{
#ifdef IN_INTERRUPT_DEBUG
  static in_interrupt = 0;
#endif
  gus_pcm_t *pcm;
  gus_pcm_channel_t *pchn;

#ifdef IN_INTERRUPT_DEBUG
  if ( in_interrupt )
    PRINTK( "gus: Eiaaa, interrupt routine reentered - %d times (ess)!!!\n",
  in_interrupt++;    
#endif
#if 0
  PRINTK( "ess: interrupt!!!\n" );
#endif
  
  pcm = card -> pcm_ess;

  if ( card -> ess.pflags & PFLG_USED ) {
#ifdef GUSCFG_INTERRUPTS_PROFILE
    card -> ess.interrupt_stat_playback++;        
#endif
#if 0
    PRINTK( "irq: residue = 0x%x\n", get_dma_residue( card -> dmas[ GUS_DMA_ESS ] -> dma ) );
#endif
    pchn = &pcm -> chn[ PCM_PLAYBACK ];
    if ( !(pchn -> flags & PCM_FLG_TRIGGER) ) return;
    if ( !( pchn -> flags & PCM_FLG_MMAP ) ) {
      MEMSET( pchn -> buffer + pchn -> block_size * pchn -> tail, pchn -> neutral_byte, 16 );
      if ( pchn -> used == 1 ) {
        ess_trigger( card, 0x00 );
        pchn -> flags &= ~PCM_FLG_TRIGGER;
        if ( !(pchn -> flags & PCM_FLG_SYNC) )
          pchn -> discarded++;
      } else {
        pchn -> interrupts++;
      }
      pchn -> processed_bytes += pchn -> sizes[ pchn -> tail ];
      pchn -> sizes[ pchn -> tail++ ] = 0;
      pchn -> tail %= pchn -> blocks;
      pchn -> used--;
      if ( pchn -> used > 0 )
        PCM_LOCK( pchn, pchn -> tail );
       else
        PCM_LOCKZERO( pchn );
    } else {
      pchn -> interrupts++;
      pchn -> processed_bytes += pchn -> block_size;
      pchn -> tail++;
      pchn -> tail %= pchn -> blocks;
    }
    if ( !pchn -> used && ( card -> ess.pflags & PFLG_FLUSH ) ) {
      ess_stop_playback( card, pcm );
      card -> ess.pflags &= ~PFLG_FLUSH;
    }
    if ( pchn -> flags & PCM_FLG_SLEEP ) {
      pchn -> flags &= ~PCM_FLG_SLEEP;
      WAKEUP( pcm, playback );
    }
  }

  if ( card -> ess.rflags & RFLG_USED ) {
#ifdef GUSCFG_INTERRUPTS_PROFILE
    card -> ess.interrupt_stat_record++;        
#endif
    pchn = &pcm -> chn[ PCM_RECORD ];
    if ( !(pchn -> flags & PCM_FLG_TRIGGER) ) return;
    if ( !( pchn -> flags & PCM_FLG_MMAP ) ) {
      if ( pchn -> used < pchn -> blocks )
        pchn -> used++;
       else
        card -> ess.record_overflow++;
      pchn -> sizes[ pchn -> head ] = pchn -> block_size;
      pchn -> head++;
      pchn -> head %= pchn -> blocks;
      PCM_LOCK( pchn, pchn -> head );
    } else {
      pchn -> head++;
      pchn -> head %= pchn -> blocks;
    }
    pchn -> interrupts++;
    pchn -> processed_bytes += pchn -> block_size;
    if ( card -> ess.rflags & RFLG_FLUSH ) {  /* stop this as soon is possible */
      ess_stop_record( card, pcm );
      card -> ess.rflags &= ~RFLG_FLUSH;
    }
    if ( pchn -> flags & PCM_FLG_SLEEP ) {
      pchn -> flags &= ~PCM_FLG_SLEEP;
      WAKEUP( pcm, record );
    }
  }  

  INB( ESSP( card, DATA_AVAIL ) );	/* ack interrupt */
#ifdef IN_INTERRUPT_DEBUG
  in_interrupt--;
#endif            
}

/*
 *
 */

static int ess_open_playback( gus_pcm_t *pcm )
{
  gus_card_t *card;

  card = pcm -> card;
  if ( gus_dma_malloc( card, GUS_DMA_ESS, "ESS PCM - playback", 1 ) < 0 )
    return -ENOMEM;
  gus_pcm_set_dma_struct( card, GUS_DMA_ESS, &pcm -> chn[ PCM_PLAYBACK ] );
  ess_open( card );
  card -> ess.mode |= ESS_MODE_PLAY;
  card -> ess.pflags = RFLG_NONE;
  return 0;
}

static int ess_open_record( gus_pcm_t *pcm )
{
  gus_card_t *card;

  card = pcm -> card;
  if ( gus_dma_malloc( card, GUS_DMA_ESS, "ESS PCM - record", 1 ) < 0 )
    return -ENOMEM;
  gus_pcm_set_dma_struct( card, GUS_DMA_ESS, &pcm -> chn[ PCM_RECORD ] );
  ess_open( card );
  card -> ess.mode |= ESS_MODE_RECORD;
  card -> ess.rflags = RFLG_NONE;
  return 0;
}

static void ess_close_playback( gus_pcm_t *pcm )
{
  gus_card_t *card;

  card = pcm -> card;
  if ( card -> ess.pflags & PFLG_USED )
    ess_stop_playback( card, pcm );
  card -> ess.pflags = PFLG_NONE;
  card -> ess.mode &= ~ESS_MODE_PLAY;  
  ess_close( card );
  gus_dma_free( card, GUS_DMA_ESS, 1 );
}

static void ess_close_record( gus_pcm_t *pcm )
{
  gus_card_t *card;

  card = pcm -> card;
  if ( card -> ess.rflags & RFLG_USED )
    ess_stop_record( card, pcm );
  card -> ess.rflags = RFLG_NONE;
  card -> ess.mode &= ~ESS_MODE_RECORD;
  ess_close( card );
  gus_dma_free( card, GUS_DMA_ESS, 1 );
}

static void ess_init_playback( gus_pcm_t *pcm )
{
  gus_card_t *card;
  gus_pcm_channel_t *pchn;

  card = pcm -> card;
  pchn = &pcm -> chn[ PCM_PLAYBACK ];
  if ( !(pchn -> flags & PCM_FLG_MMAP) && !pchn -> used ) return;
  if ( !( card -> ess.pflags & PFLG_USED ) )
    ess_start_playback( card, pcm );
   else
    {
      ess_trigger( card, 0x05 );
      pchn -> flags |= PCM_FLG_TRIGGER;
    }
}

static void ess_done_playback( gus_pcm_t *pcm )
{
  gus_card_t *card;
  gus_pcm_channel_t *pchn;

#if 0
  printk( "ess_done_playback!!!\n" );
#endif
  card = pcm -> card;
  if ( card -> ess.pflags & PFLG_USED )
    {
      pchn = &pcm -> chn[ PCM_PLAYBACK ];
      card -> ess.pflags |= PFLG_FLUSH;
      while ( card -> ess.pflags & PFLG_USED )
        {
          pchn -> flags |= PCM_FLG_SLEEP;
          SLEEP( pcm, playback, HZ * 10 );
          pchn -> flags &= ~PCM_FLG_SLEEP;
          if ( card -> ess.pflags & PFLG_USED )
            {
              if ( TABORT( pcm, playback ) )
                pchn -> flags |= PCM_FLG_ABORT;
               else
              if ( TIMEOUT( pcm, playback ) )
                PRINTK( "pcm: flush failed (playback)\n" );
               else
                continue;
              ess_stop_playback( card, pcm );
              card -> ess.pflags = PFLG_NONE;
            }
        }
      card -> ess.pflags = PFLG_NONE;
    }
}

static void ess_init_record( gus_pcm_t *pcm )
{
  gus_card_t *card;

  card = pcm -> card;
  if ( !( card -> ess.rflags & RFLG_USED ) )
    ess_start_record( card, pcm );
}

static void ess_done_record( gus_pcm_t *pcm )
{
  gus_card_t *card;
  gus_pcm_channel_t *pchn;

  card = pcm -> card;
  if ( card -> ess.rflags & RFLG_USED )
    {
      pchn = &pcm -> chn[ PCM_RECORD ];
      card -> ess.rflags |= RFLG_FLUSH;
      while ( card -> ess.rflags & PFLG_USED )
        {
          pchn -> flags |= PCM_FLG_SLEEP;
          SLEEP( pcm, record, HZ * 10 );
          pchn -> flags &= ~PCM_FLG_SLEEP;
          if ( card -> ess.rflags & RFLG_USED )
            {
              if ( TABORT( pcm, record ) )
                pchn -> flags |= PCM_FLG_ABORT;
               else
              if ( TIMEOUT( pcm, record ) )
                PRINTK( "pcm: flush failed (record)\n" );
               else
                continue;
              ess_stop_record( card, pcm );
              card -> ess.rflags = PFLG_NONE;
            }
        }
    }
}

static void ess_dma_playback( gus_pcm_t *pcm, gus_pcm_channel_t *pchn, char *buffer, unsigned int count )
{
  unsigned char *dest;

  dest = pchn -> buffer + ( pchn -> head * pchn -> block_size ) + pchn -> sizes[ pchn -> head ];
  if ( pchn -> mode & PCM_MODE_ULAW ) {
    gus_translate_memcpy_fromfs( gus_ulaw_dsp, dest, buffer, count );
  } else {
    MEMCPY_FROMFS( dest, buffer, count );
  }
}

static void ess_dma_playback_neutral( gus_pcm_t *pcm, gus_pcm_channel_t *pchn )
{
  gus_card_t *card;
  unsigned int size;
  unsigned int count;
  unsigned char *dest;
  unsigned char neutral;

  card = pcm -> card;  
  if ( card -> ess.pflags & PFLG_USED )
    {
      size = pchn -> sizes[ pchn -> head ];
      count = pchn -> block_size - size;
      neutral = pchn -> neutral_byte;
      if ( count > 0 )
        {
          dest = pchn -> buffer + ( pchn -> head * pchn -> block_size ) + size;
          if ( size < pchn -> block_size ) neutral = *( dest - ( pchn -> mode & PCM_MODE_BIG ? 2 : 1 ) );
          MEMSET( dest, neutral, count );
        }
    }
}

static void ess_dma_record( gus_pcm_t *pcm, gus_pcm_channel_t *pchn, char *buffer, unsigned int count )
{
  unsigned char *src;

  src = pchn -> buffer + ( pchn -> tail * pchn -> block_size ) + ( pchn -> block_size - pchn -> sizes[ pchn -> tail ] );
  if ( pchn -> mode & PCM_MODE_ULAW ) {
    gus_translate_memcpy_tofs( gus_dsp_ulaw, buffer, src, count );
  } else {
    MEMCPY_TOFS( buffer, src, count );
  }
}

static void ess_sync_playback( gus_pcm_t *pcm )
{
  ess_init_playback( pcm );
  ess_done_playback( pcm );
}

static void ess_sync_record( gus_pcm_t *pcm )
{
  ess_done_record( pcm );
}

static unsigned int ess_pointer_playback( gus_pcm_t *pcm )
{
  gus_card_t *card;
  unsigned int res = 0;
  gus_pcm_channel_t *pchn;

  card = pcm -> card;
  pchn = &pcm -> chn[ PCM_PLAYBACK ];
  if ( ( pchn -> flags & PCM_FLG_TRIGGER ) == PCM_FLG_TRIGGER )
    res = pchn -> used_size - get_dma_residue( card -> dmas[ GUS_DMA_ESS ] -> dma );
  return res;
}

static unsigned int ess_pointer_record( gus_pcm_t *pcm )
{
  gus_card_t *card;
  unsigned int res = 0;
  gus_pcm_channel_t *pchn;

  card = pcm -> card;
  pchn = &pcm -> chn[ PCM_RECORD ];
  if ( ( pchn -> flags & PCM_FLG_TRIGGER ) == PCM_FLG_TRIGGER )
    res = pchn -> used_size - get_dma_residue( card -> dmas[ GUS_DMA_ESS ] -> dma );
  return res;
}

/*
 *
 */

void gus_init_ess_pcm( gus_pcm_t *pcm )
{
  gus_card_t *card;
  gus_pcm_channel_t *pchn;

  card = pcm -> card;
  card -> pcm_ess = pcm;

  card -> ess.pflags = PFLG_NONE;
  card -> ess.rflags = RFLG_NONE;

  pcm -> flags |= PCM_LFLG_USED;
  pcm -> info_flags = GUS_PCM_INFO_CODEC | GUS_PCM_INFO_MMAP |
  		      GUS_PCM_INFO_PLAYBACK | GUS_PCM_INFO_RECORD;

  pcm -> info_name = "ES 1688";
  
  pchn = &pcm -> chn[ PCM_PLAYBACK ];
  pchn -> formats = AFMT_MU_LAW | AFMT_U8 | AFMT_S16_LE;
  pchn -> min_fragment = 4;
  pchn -> max_rate = 48000;
  pchn -> max_voices = 2;
  pchn -> hw_open = ess_open_playback;
  pchn -> hw_close = ess_close_playback;
  pchn -> hw_init = ess_init_playback;
  pchn -> hw_done = ess_done_playback;
  pchn -> hw_dma = ess_dma_playback;
  pchn -> hw_dma_neutral = ess_dma_playback_neutral;
  pchn -> hw_sync = ess_sync_playback;
  pchn -> hw_pointer = ess_pointer_playback;

  pchn++;
  pchn -> formats = AFMT_MU_LAW | AFMT_U8 | AFMT_S16_LE;
  pchn -> min_fragment = 4;
  pchn -> max_rate = 48000;
  pchn -> max_voices = 2;
  pchn -> hw_open = ess_open_record;
  pchn -> hw_close = ess_close_record;
  pchn -> hw_init = ess_init_record;
  pchn -> hw_done = ess_done_record;
  pchn -> hw_dma = ess_dma_record;
  pchn -> hw_dma_neutral = NULL;
  pchn -> hw_sync = ess_sync_record;
  pchn -> hw_pointer = ess_pointer_record;
}

#endif /* GUSCFG_ESS */
