/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 *  Routines for GF1 patches
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include "libgus.h"
#include "libgus_local.h"

/*
 *  defines
 */

#define WAVE_MODE_16BIT		0x01
#define WAVE_MODE_UNSIGNED	0x02
#define WAVE_MODE_LOOP		0x04
#define WAVE_MODE_BIDIRECT	0x08
#define WAVE_MODE_BACKWARD	0x10
#define WAVE_MODE_SUSTAIN	0x20
#define WAVE_MODE_ENVELOPE	0x40

/*
 *  structures
 */

struct patch_header {
  unsigned char header[ 12 ];		/* "GF1PATCH110" */
  unsigned char gravis_id[ 10 ];	/* "ID#000002" */
  unsigned char description[ 60 ];
  unsigned char instruments;
  unsigned char voices;
  unsigned char channels;
  unsigned short wave_forms;
  unsigned short master_volume;
  unsigned int data_size;
};

/*
 *  variables
 */


/*
 *  local functions
 */

int gus_instr_patch_load( gus_instrument_t *instr, char *filename, int s8bit )
{
  int i, fd, offset;
  unsigned int mode, length;
  struct stat info;
  unsigned char buf[ 0xef ];
  struct patch_header phdr;
  gus_layer_t *layer;
  gus_wave_t *wave, *fwave, *pwave;

  if ( stat( filename, &info ) == -1 )
    {
      gus_dprintf( "patch file '%s' not found\n", filename );
      return -1;
    }
  if ( ( fd = open( filename, O_RDONLY ) ) < 0 )
    {
      gus_dprintf( "could not open patch file '%s'\n", filename );
      return -1;
    }
  if ( read( fd, buf, 0xef ) != 0xef )
    {
      gus_dprintf( "patch read error (1) '%s'\n", filename );
      close( fd );
      return -1;
    }
  memcpy( (char *)&phdr, buf, sizeof( phdr ) );
  if ( strncmp( phdr.header, "GF1PATCH110", 12 ) ||
       strncmp( phdr.gravis_id, "ID#000002", 10 ) )
    {
      gus_dprintf( "wrong patch signature or version '%s'\n", filename );
      close( fd );
      return -1;
    }
  phdr.wave_forms = gus_get_word( buf, 85 );
  phdr.master_volume = gus_get_word( buf, 87 );
  offset = 0xef;
  fwave = pwave = NULL;
  for ( i = 0; i < phdr.wave_forms; i++ )
    {
      if ( lseek( fd, offset, SEEK_SET ) < 0 )
        {
          gus_dprintf( "patch seek error '%s'\n", filename );
          close( fd );
          return -1;
        }
      if ( read( fd, buf, 60 ) != 60 )
        {
          gus_dprintf( "patch read error (2) '%s'\n", filename );
          close( fd );
          return -1;
        }
              
      wave = calloc( 1, sizeof( gus_wave_t ) );
      if ( !wave )
        {
          offset += 96 + gus_get_dword( buf, 8 );
          continue;
        }
        
      wave -> mode = GUS_INSTR_PATCH;

      wave -> data.patch.sample_rate = gus_get_word( buf, 20 );
      wave -> data.patch.low_frequency = gus_get_dword( buf, 22 );
      wave -> data.patch.high_frequency = gus_get_dword( buf, 26 );
      wave -> data.patch.root_frequency = gus_get_dword( buf, 30 );
      wave -> data.patch.tune = gus_get_word( buf, 34 );
      wave -> data.patch.balance = gus_get_byte( buf, 36 ) << 4;
      if ( gus_get_byte( buf, 36 ) > 7 )
        wave -> data.patch.balance += 15;
      
      mode = gus_get_byte( buf, 55 );
      if ( mode & WAVE_MODE_16BIT )	wave -> format |= GUS_WAVE_16BIT;
      if ( mode & WAVE_MODE_UNSIGNED )	wave -> format |= GUS_WAVE_UNSIGNED;
      if ( mode & WAVE_MODE_LOOP )	wave -> format |= GUS_WAVE_LOOP;
      if ( mode & WAVE_MODE_BIDIRECT )	wave -> format |= GUS_WAVE_BIDIR;
      if ( mode & WAVE_MODE_BACKWARD )	wave -> format |= GUS_WAVE_BACKWARD;
      if ( mode & WAVE_MODE_ENVELOPE )
        {
          int idx, same, old;
        
          wave -> data.patch.flags |= GUS_WAVE_PATCH_ENVELOPE;
          if ( mode & WAVE_MODE_SUSTAIN )
            wave -> data.patch.flags |= GUS_WAVE_PATCH_SUSTAIN;
          same = 0; 
          old = buf[ 43 ];
          for ( idx = 0; idx < 6; idx++ )
            {
              wave -> data.patch.envelope_rate[ idx ] = buf[ 37 + idx ];
              wave -> data.patch.envelope_offset[ idx ] = buf[ 43 + idx ];
              if ( old == buf[ 43 + idx ] ) same++;
            }
          if ( same == 6 )	/* ok.. ProPatches set have this ugly patches.. */
            {
              /* attack */
              wave -> data.patch.envelope_rate[ 0 ] = 20;
              /* final release */
              wave -> data.patch.envelope_rate[ 5 ] = 40;
              wave -> data.patch.flags |= GUS_WAVE_PATCH_SUSTAIN;
              wave -> format &= ~GUS_WAVE_LOOP;
            }
        }
      wave -> loop_start = gus_get_dword( buf, 12 ) << 4;
      wave -> loop_end = gus_get_dword( buf, 16 ) << 4;
      if ( !(wave -> format & GUS_WAVE_LOOP) )
        wave -> loop_end -= wave -> format & GUS_WAVE_16BIT ? 40 : 24;
      wave -> size = length = gus_get_dword( buf, 8 );
      wave -> data.patch.tremolo_sweep = gus_get_byte( buf, 49 );
      wave -> data.patch.tremolo_rate = gus_get_byte( buf, 50 );
      wave -> data.patch.tremolo_depth = gus_get_byte( buf, 51 );
      wave -> data.patch.vibrato_sweep = gus_get_byte( buf, 52 );
      wave -> data.patch.vibrato_rate = gus_get_byte( buf, 53 );
      wave -> data.patch.vibrato_depth = gus_get_byte( buf, 54 );
      wave -> data.patch.scale_frequency = gus_get_word( buf, 56 );
      wave -> data.patch.scale_factor = gus_get_word( buf, 58 );

      offset += 96;
      if ( lseek( fd, offset, SEEK_SET ) < 0 )
        {
          gus_dprintf( "patch seek error '%s'\n", filename );
          close( fd );
          return -1;
        }
      offset += length;
      if ( ( wave -> begin.ptr = malloc( length ) ) == NULL ) continue;

#if 0
      {
        char x[ 8 ];
        x[ 7 ] = 0;
        strncpy( x, buf, 7 );
        printf( ">>%s<< - p=0x%x, s=0x%x, e=0x%x, l=0x%x\n", x, (unsigned int)sample -> g.ptr, sample -> loop_start, sample -> loop_end, sample -> length );
      }
#endif

      if ( read( fd, wave -> begin.ptr, length ) != length )
        {
          gus_dprintf( "patch read error (3) '%s'\n", filename );
          close( fd );
          return -1;
        }

      if ( s8bit && ( mode & GUS_WAVE_16BIT ) )
        {
          int idx;
          unsigned char *pb;
          
          wave -> format &= ~GUS_WAVE_16BIT;
          pb = wave -> begin.ptr;
          for ( idx = 0; idx < wave -> size >> 1; idx++ )
            pb[ idx ] = pb[ ( idx << 1 ) | 1 ];
          wave -> loop_start >>= 1;
          wave -> loop_end >>= 1;
          wave -> size >>= 1;
        }

#if 0
      if ( wave -> format & GUS_WAVE_16BIT )
        {
          unsigned short *pw;
          pw = (unsigned short *)wave -> begin.ptr;
          pw[ ( wave -> loop_end >> 4 ) / 2 ] = pw[ ( ( wave -> loop_end >> 4 ) / 2 ) + 1 ] =
      		pw[ ( wave -> loop_start >> 4 ) / 2 ];
        }
       else
        {
          unsigned char *pb;
          pb = (unsigned char *)wave -> begin.ptr;
          pb[ ( wave -> loop_end >> 4 ) ] = pb[ ( wave -> loop_end >> 4 ) + 1 ] =
      		pb[ ( wave -> loop_start >> 4 ) ];          
        }
#endif
      if ( !fwave )
        fwave = wave;
       else
        pwave -> next = wave;
      pwave = wave;
    }
  close( fd );
  if ( !fwave ) return -1;
  instr -> info.layer = layer = calloc( 1, sizeof( gus_layer_t ) );
  layer -> mode = GUS_INSTR_PATCH;
  layer -> wave = fwave;
  if ( !layer )
    {
      while ( fwave )
        {
          pwave = fwave;
          fwave = fwave -> next;
          free( pwave -> begin.ptr );
          free( pwave );
        }
      instr -> info.layer = NULL;
      return -1;
    }
  instr -> mode = GUS_INSTR_PATCH;
  return 0;
}
