// pvsynthesizer.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/
#ifdef __GNUG__
#pragma implementation
#endif

#include "application.h"
#include "envelope.h"
#include "localdefs.h"
#include "pvocrequester.h"
#include "request.h"
#include "phasevocoder.h"
#include "pipeaction.h"
#include "pvsynthesizer.h"
#include "pvocdata.h"

class PVSynthesisRequester : public PvocRequester {
	friend PVSynthesizer;
protected:
	PVSynthesisRequester(const char* , PhaseVocoder::Info &,
	                     PvocData *, double dur);
	redefined void configureRequest(Request *);
	redefined boolean confirmValues();
private:
	PvocData* pvdata;
	float lowLimit;
	float highLimit;
	double synthDur;
};

PVSynthesisRequester::PVSynthesisRequester(const char* title,
                                           PhaseVocoder::Info &info,
                                           PvocData* data,
                                           double dur)
		: PvocRequester(title, info), pvdata(data), 
		  lowLimit(0.0), highLimit(pvdata->sRate()/2.0),
		  synthDur(dur) {}
		
void
PVSynthesisRequester::configureRequest(Request* request) {
	request->appendLabel("All \"0\" values will be set to defaults.");
	request->appendValue("Output frame size (samples):",
		&pvocInfo.outputFrameSize, NonNegativeIntegers);
	request->appendValue("Spectral envelope warp factor:", &pvocInfo.warp);
	Range allFreqs(0.0F, pvdata->sRate()/2.0);
	request->appendValue("Lower band limit (hz.):", &lowLimit, allFreqs);
	request->appendValue("Upper band limit: (hz.)", &highLimit, allFreqs);
	PvocRequester::configureRequest(request);
}

boolean
PVSynthesisRequester::confirmValues() {
	int status = true;
	char msg[120];
	char msg2[120];
	double sampleRate = pvocInfo.samplingRate;
	if(sampleRate != double(pvdata->sRate())) {
	    sprintf(msg,
	        "Warning: Phase Vocoder data samprate (%d) != selection samprate (%f).",
	            pvdata->sRate(), sampleRate);
		sprintf(msg2,
			"Resynthesized length and spectrum will be skewed by a factor of %f.",
			sampleRate / double(pvdata->sRate()));
	    status = Application::confirm(msg, msg2, "Continue anyway?", Cancel);
	}
	if(!status)
		return status;
	PvocRequester::confirmValues();
	pvocInfo.firstBand = pvdata->getBandNumber(lowLimit);
	pvocInfo.lastBand = pvdata->getBandNumber(highLimit);
	double analDur = pvdata->duration();
	printf("anal dur = %f, selected dur = %f, ratio = %f\n",
	       analDur, synthDur, synthDur/analDur);
	printf("new timeScaleFactor = %f\n\n",
	       pvocInfo.timeScaleFactor /= synthDur/analDur);
	return status;
}

//********

PVSynthesizer::PVSynthesizer(Data* data, PvocData* anal)
	: QueuedOutputFunction(data, 1024),
	  pvocInfo(
	  	data->sRate(),                                   // srate
	  	(data->dataType() == FloatData) ? 1.0 : 32767.0, // scaling
	  	anal->channels() - 2,                            // fftsize
	  	0,                                               // fundfrq
	  	anal->frameOffset(),                             // infrmoffset
	  	anal->channels() - 2                             // outfrmsize
	  ),
	  pvoc(nil), pvdata(anal) {
	pvdata->ref();
}

PVSynthesizer::PVSynthesizer(
	Data* data,  PvocData* anal,
	int L, int I, double T, int i, int j, boolean useKaiser)
		: QueuedOutputFunction(data, 2*L),
		  pvocInfo(
			data->sRate(),                           // srate	
			(data->dataType() == FloatData) ? 1.0 : 32767.0,
		  	anal->channels() - 2,                    // fftsize
		  	0,
		  	anal->frameOffset(),                     // infrmoffset
			L, I, T, i, j,
			0.0,                                     // warp
			useKaiser
		  ),
		  pvoc(nil), pvdata(anal) {
	pvdata->ref();
	initialize();
}

Modifier *
PVSynthesizer::create(DataEditor *de) {
	return nil;
}

PVSynthesizer::~PVSynthesizer() {
	Resource::unref(pvdata);
	delete pvoc;
}

void
PVSynthesizer::restoreState() {
	Super::restoreState();
	pvoc->reset();
	currentFrame.set(0, 0);
}

Requester *
PVSynthesizer::createRequester() {
	return new PVSynthesisRequester(
		"Phase Vocoder Resynthesis into Selected Region:",
		pvocInfo,
		pvdata,
		double(target()->length()) / sampRate()
	);
}

void
PVSynthesizer::initialize() {		
	pvoc = new PhaseVocoder(pvocInfo);
	if(pvoc->isGood()) {
		setOutQueueSize(2 * pvoc->getOutputFrameOffset());
		Super::initialize();
	}
}

typedef int (QueuedOutputFunction::*(InQueueFunction))(double);

class PVInPipeAction : public InPipeAction {
public:
	PVInPipeAction(PVSynthesizer *pvs, InQueueFunction fun)
		: pv(pvs), pfun(fun) {}
	redefined int add(double val) { return (pv->*pfun)(val); }
private:
	PVSynthesizer* pv;
	InQueueFunction pfun;
};

int
PVSynthesizer::doProcessing() {
	BUG("PVSynthesizer::doProcessing");
	int status = (pvdata->nFrames() > currentFrame);
	if(status) {
		Data* frame = pvdata->clone(currentFrame);
		InPipeAction* action = new PVInPipeAction(
			this, &PVSynthesizer::addToOutQueue
		);
		status = pvoc->runSynthesis(frame, action);
		Resource::unref(frame);
		currentFrame += 1;
	}
	return status;
}
