/*
 * ident_vars.c --
 *
 * This file contains the implementation of the Ident Resources MIB
 * (RFC 1414).
 *
 * Copyright (c) 1996-1997
 *
 * Erik Schoenfelder		TU Braunschweig, Germany
 *
 * 
 * Permission to use, copy, modify, and distribute this software and its 
 * documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in 
 * supporting documentation, and that the name of CMU not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  
 * 
 * THE AUTHORS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * THE AUTHORS BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 * 
 */

#include "mib_module.h"
#include "ident_vars.h"

#ifdef HAVE_IDENT

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <malloc.h>
#include <pwd.h>

#include "snmp_vars.h"

static struct variable id_variables[] = {
  {IDIDENTSTATUS, INTEGER, RONLY, var_id, 4, {1, 1, 1, 1}},
  {IDIDENTOPSYS, STRING, RONLY, var_id, 4, {1, 1, 1, 2}},
  {IDIDENTCHARSET, STRING, RONLY, var_id, 4, {1, 1, 1, 3}},
  {IDIDENTUSERID, STRING, RONLY, var_id, 4, {1, 1, 1, 4}},
  {IDIDENTMISC, STRING, RONLY, var_id, 4, {1, 1, 1, 5}}
};

static oid id_base [] = { MIB, 24 };

/*
 * initialize and register ident-mib:
 */
void
ident_init()
{
  /* Identification MIB (rfc 1414): */
  mib_register (id_base, sizeof(id_base) / sizeof(oid),
		id_variables,
		sizeof(id_variables)/sizeof(*id_variables),
		sizeof(*id_variables));
}


static char *
xstrdup (s)
char *s;
{
    char *n = malloc (strlen (s) + 1);
    if (! s)
      return "(unknown)";
    strcpy (n, s);
    return n;
}


/* 
 * format uid to name or gecos; return the result in buf with max-len
 * len. 
 */
static void
uid_to_str (buf, uid, len, flag)
char *buf;
int uid, len, flag;
{
    struct passwd *paw;
    typedef struct _pwds {
	int uid;
	char *name;
	char *gecos;
	struct _pwds *next;
    } pwds;
    static pwds *all_pw = 0;
    pwds *p;
    char *s;

    for (p = all_pw; p && p->uid != uid; p = p->next) {
      continue;
    }

    if (! p) {
      if (! (paw = getpwuid (uid))) {
	strcpy (buf, "(unknown)");
	return;
      }
    
      if (paw) {
	pwds *pnew = (pwds *) malloc (sizeof (pwds));
	if (! pnew) {
	  strncpy (buf, "(unknown)", len);
	  buf [len - 1] = 0;
	  return;
	}
	pnew->uid = paw->pw_uid;
	pnew->name = xstrdup (paw->pw_name);
	pnew->gecos = xstrdup (paw->pw_gecos);
	pnew->next = all_pw;
	p = all_pw = pnew;
      }
    }
    strncpy (buf, flag ? p->gecos : p->name, len - 1);
    buf [len - 1] = 0;

    /* 
     * for US-ASCII strip 8'th bit -- germans please don't even think
     * about complaining... -- send a diff :-)  
     */
    for (s = buf; *s; s++) {
      *s = *s & 0x7f;
    }
}


/*
 * entry for the identification mib:
 */

u_char *
var_id(vp, name, length, exact, var_len, write_method)
     struct variable *vp;    /* IN - ptr to variable entry that points here */
     oid     *name;	     /* IN/OUT - input name req, output name found */
     int     *length;	     /* IN/OUT - length of input and output oid's */
     int     exact;	     /* IN - TRUE if an exact match was requested. */
     int     *var_len;	     /* OUT - length of var or 0 if function ret. */
     int     (**write_method)();   /* OUT - ptr to func to set var, else 0 */
{
  int i;
  /* static struct tcpstat tcpstat; */
  static char ret_str [256];
  oid newname[MAX_NAME_LEN], lowest[MAX_NAME_LEN], *op;
  u_char *cp;
  int State, LowState;
  static struct inpcb inpcb, Lowinpcb;
  
  /* mib-2.ident.identInfo.identTable.identEntry == 1.3.6.1.2.1.24.1.1.1.x */
    
  bcopy((char *)vp->name, (char *)newname, (int)vp->namelen * sizeof(oid));

  /* find "next" connection */
  Again:
    LowState = -1;	    /* Don't have one yet */
    TCP_Scan_Init();
    for (;;) {
      if ((i = TCP_Scan_Next(&State, &inpcb)) < 0) 
	goto Again;
      if (i == 0) 
	break;	    /* Done */
      cp = (u_char *)&inpcb.inp_laddr.s_addr;
      op = newname + 11;
      *op++ = *cp++, *op++ = *cp++, *op++ = *cp++, *op++ = *cp++;
      
      newname[15] = ntohs(inpcb.inp_lport);
	
      cp = (u_char *)&inpcb.inp_faddr.s_addr;
      op = newname + 16;
      *op++ = *cp++, *op++ = *cp++, *op++ = *cp++, *op++ = *cp++;
      
      newname[20] = ntohs(inpcb.inp_fport);
      
      if (exact) {
	if (compare(newname, 21, name, *length) == 0) {
	  bcopy((char *)newname, (char *)lowest, 21 * sizeof(oid));
	  LowState = State;
	  Lowinpcb = inpcb;
	  break;  /* no need to search further */
	}
      } else {
	if ((compare(newname, 21, name, *length) > 0) &&
	    ((LowState < 0) || (compare(newname, 21, lowest, 21) < 0))){
	  /*
	   * if new one is greater than input and closer to input than
	   * previous lowest, save this one as the "next" one.
	   */
	  bcopy((char *)newname, (char *)lowest, 21 * sizeof(oid));
	  LowState = State;
	  Lowinpcb = inpcb;
	}
      }
    }
    if (LowState < 0)
      return(NULL);

    bcopy((char *)lowest, (char *)name, ((int)vp->namelen + 10) * sizeof(oid));
    *length = vp->namelen + 10;
    *write_method = 0;
    *var_len = sizeof(long);

    switch (vp->magic) {

      case IDIDENTSTATUS: 
	long_return = (Lowinpcb.inp_state == 6) + 1;
	return (u_char *) &long_return;

      case IDIDENTOPSYS:
	*var_len = 4;
	return "unix";

      case IDIDENTCHARSET:
	*var_len = 8;
	return "US-ASCII";

      case IDIDENTUSERID:
	if (Lowinpcb.inp_state == 6) {
	  *ret_str = 0;
	} else {
	  uid_to_str (ret_str, Lowinpcb.uid, 256, 0);
	}
	*var_len = strlen (ret_str); 
	return ret_str;

      case IDIDENTMISC:
	if (Lowinpcb.inp_state == 6) {
	  *ret_str = 0;
	} else {
	  uid_to_str (ret_str, Lowinpcb.uid, 256, 1);
	}
	*var_len = strlen (ret_str); 
	return ret_str;
    }
    return NULL;
}

#endif /* HAVE_IDENT */
