/*
 * ProFTPD - FTP server daemon
 * Copyright (c) 1997, 1998 Public Flood Software
 * Copyright (C) 1999, MacGyver aka Habeeb J. Dihu <macgyver@tos.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307, USA.
 */

/* Shows a count of "who" is online via proftpd.  Uses the /var/run/proftpd*
 * log files.
 *
 * $Id: ftpcount.c,v 1.4 1999/10/11 04:45:46 macgyver Exp $
 */

#include "conf.h"

#ifdef HAVE_GETOPT_H
# include <getopt.h>
#endif

static char *percent_complete(unsigned long size, unsigned long done)
{
  static char sbuf[32];

  if(done < size) {
    snprintf(sbuf, sizeof(sbuf), "%.0f",
	     ((double)done / (double)size) * 100.0);
    sbuf[sizeof(sbuf) - 1] = '\0';
  } else {
    sstrncpy(sbuf, "100", sizeof(sbuf));
  }
  
  return sbuf;
}

static char *idle_time(time_t *i)
{
  time_t now;
  long l;
  static char sbuf[7];

  if(!i || !*i)
    return "-";
  else {
    time(&now);
    l = now - *i;

    if(l < 3600) 
      snprintf(sbuf, sizeof(sbuf), "%ldm%lds",(l / 60),(l % 60));
    else
      snprintf(sbuf, sizeof(sbuf), "%ldh%ldm",(l / 3600),
      ((l - (l / 3600) * 3600) / 60));
  }

  return sbuf;
}

struct option_help {
  char *long_opt,*short_opt,*desc;
} opts_help[] = {
  { "--help","-h", NULL },
  { "--verbose","-v","display add'l information for each connection" },
  { "--path","-p","specify full path to scoreboard directory" },
  { NULL }
};

struct option opts[] = {
  { "help",    0, NULL, 'h' },
  { "verbose", 0, NULL, 'v' },
  { "path",    1, NULL, 'p' },
  { NULL,      0, NULL, 0   }
};

void show_usage(const char *progname, int exit_code)
{
  struct option_help *h;

  printf("usage: %s [options]\n",progname);
  for(h = opts_help; h->long_opt; h++) {
    printf("  %s,%s\n",h->long_opt,h->short_opt);
    if(!h->desc)
      printf("    display %s usage\n",progname);
    else
      printf("    %s\n",h->desc);
  }

  exit(exit_code);
}

int main(int argc, char **argv)
{
  logrun_t *l;
  pid_t oldpid = 0,mpid;
  int c = 0,tot = 0;
  int verbose = 0;

  char *cp,*progname = *argv;

  if((cp = rindex(progname,'/')) != NULL)
    progname = cp+1;

  opterr = 0;
  while((c = getopt_long(argc,argv,"hp:v",opts,NULL)) != -1) {
    switch(c) {
    case 'h':
      show_usage(progname,0);
    case 'v':
      verbose++; break;
    case 'p':
      log_run_setpath(optarg);
      break;
    case '?':
      fprintf(stderr,"unknown option: %c\n",(char)optopt);
      show_usage(progname,1);
    }
  }

  /* sanity check on scoreboard path */
  if(log_run_checkpath() < 0) {
    fprintf(stderr,"%s: %s\n",log_run_getpath(),strerror(errno));
    fprintf(stderr,"(Perhaps you need to specify the scoreboard path with --path, or change\n");
    fprintf(stderr," the compile-time default directory?)\n");
    exit(1);
  }

  c = 0;
  while((l = log_read_run(&mpid)) != NULL) {
    if(errno)
      break;

    if(!c++ || oldpid != mpid) {
      if(tot)
        printf("   -  %d user%s\n\n", tot, tot > 1 ? "s" : "");
      if(!mpid)
        printf("inetd FTP connections:\n");
      else
        printf("Master proftpd process %d:\n",(int)mpid);
      oldpid = mpid; tot = 0;
    }

#ifdef BUILDAS_FTPWHO
    /* We're really running as ftpwho.
     */
    if(l->transfer_size)
      printf("%5d %-6s (%s%%) %s\n",(int)l->pid,idle_time(&l->idle_since),
	     percent_complete(l->transfer_size,l->transfer_complete),
	     l->op);
    else
      printf("%5d %-6s %s\n",(int)l->pid,idle_time(&l->idle_since),l->op);
    
    if(verbose) {
      if(l->address[0])
	printf("             (host: %s)\n",l->address);
      if(l->cwd[0])
	printf("              (cwd: %s)\n",l->cwd);
    }
#endif

    tot++;
  }
  if(tot)
    printf("   -  -      %d user%s\n\n", tot, tot > 1 ? "s" : "");
  return 0;
}
