/*
 * pftp -- sends files from host to host through free choosable ports
 *
 * Copyright (C) 1996, 1997, 1998 Ben Schluricke
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the emplied warranty of MERCHANT-
 * ABILITY OF FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 * Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 *    Written by Ben Schluricke
 *    E-Mail:    bhor0533@lehr.chem.TU-Berlin.DE
 *
 * This program is dedicated to my girl-friend, Heather O'Rourke.
 *
 *
 */
#ifdef USE_POSIX_THREAD
#define _REENTRANT
#include <pthread.h>
#endif
#ifdef FreeBSD
#include <sys/errno.h>
#include <sys/types.h>
#endif
#include <dirent.h>
#if defined Linux || defined NEXTSTEP
#include <sys/dir.h>
#endif
#ifdef unicos
#include <fcntl.h>
#endif
#include <time.h>
#include "main.h"

#define DATESTRING(num) ((*(pmain+num))+4)
#define USERSTRING(num) ((*(pmain+num))+25)

extern short get_var_from_pftprc(const char *, char *, short);
extern void  set_tty(int);
extern char *tabdir(char *);
extern short create_dir(char *, char *);
extern short save_dir(char *, char *);
extern short copy_file(char *, char *);
extern short save_tags(char *);
extern char **run_prog(char **, char *);
extern void mhelp(char *);
extern short set_prog(char **, char *);
extern void display_pa(char **, char *, char *, int);
extern char *get_subject(char *, char *, char *);
extern long pftp_pager(char *, char *, char *);
extern void warranty(void);
char PFTP_SORT_BY='a';
void rm_empty_dirs(void);
short empty_dir(char *);
int  read_pinfo(char **);
void write_pinfo(char **);
char browse_dir(char *);
int set_entry_in_files(char **, char *, int *);
void rm_entry_from_files(char **, char *, int *);
void printlistentry(char *, char *, char *);
short mycmp(char *, char *);
void bubble_sort(char **, char *, long);

int tlastentry;
int dlastentry;
char *tfiles[HUNAME]; /* holds the selected files */
char *dfiles[HUNAME]; /* holds the selected files */
char *pager[SONAME]; /* holds the pager program and its arguments */
char *headline;
char *lastline;
char *saving_str;
char *blankline;
char *curdir; /* holds the selected files */
int pinfo_size;
short USE_PFTPPAGER;


void pmanager(short PFTP_PEEK)
{
   char *pmain[LONAME]; /* holds Pinfo entries */
   char *Pinfodir=NULL, *Puni=NULL, *cdir=NULL;
   char **tmp=NULL, *subj=NULL;
   char *secondline=NULL;
   char c='\0', *ctmp=NULL;
   int num=0, cur=1, update=1;
   long endnum=0, j=0;
   int height=_WINROWS_-4;
   int firstline=0;
   struct stat buf;
   *pmain  = NULL;
   *tfiles = NULL;

   for (endnum=HUNAME, tmp=tfiles; j < endnum; j++, tmp++) *tmp=NULL;
   for (endnum=HUNAME, tmp=dfiles; j < endnum; j++, tmp++) *tmp=NULL;
   endnum=j=0;

   tlastentry=dlastentry=pinfo_size=0;
   (*statstr)->OVERWRITE = 0;
   (*statstr)->_SKIP_ = 0;
   USE_PFTPPAGER=0;

   MEM_CHECK((cdir = (char *)calloc(LONAME, sizeof(char))));
   MEM_CHECK((headline = (char *)calloc(LONAME, sizeof(char))));
   sprintf(headline, "****** Port-FTP Version %s ****** file and directory manager ******", VERSION);
   for (ctmp=headline; *ctmp; ctmp++);
   for (; ctmp - headline < _WINCOLS_; ctmp++) *ctmp = '*';
   *(ctmp-1) = ' ';
   *ctmp = '\0';
   MEM_CHECK((lastline = (char *)calloc(LONAME, sizeof(char))));
   sprintf(lastline, "[%dH**** <q> quit, <t> tag, <s> save, <space> more, <b> back, <?> help ***", _WINROWS_);
   for (num=0, ctmp=lastline; *ctmp != 'H'; ctmp++, num++);
   for (ctmp=lastline; *ctmp; ctmp++);
   for (; ctmp - lastline < _WINCOLS_+num; ctmp++) *ctmp = '*';
   *ctmp = '\r';
   *++ctmp = '\0';
   MEM_CHECK((saving_str = (char *)calloc(SONAME, sizeof(char))));
   sprintf(saving_str, "[%dH* Saving ...", _WINROWS_);
   for (num=0, ctmp=saving_str; *ctmp != 'H'; ctmp++, num++);
   for (ctmp=saving_str; *ctmp; ctmp++);
   for (; ctmp - saving_str < _WINCOLS_+num; ctmp++) *ctmp = ' ';
   *ctmp = '\r';
   *++ctmp = '\0';
   MEM_CHECK((blankline = (char *)calloc(SONAME, sizeof(char))));
   for (ctmp=blankline; ctmp - blankline < _WINCOLS_; ctmp++) *ctmp = ' ';
   *ctmp='\r';
   *++ctmp='\0';
   MEM_CHECK((curdir = (char *)calloc(LONAME, sizeof(char))));
   MEM_CHECK((Puni = (char *)calloc(SONAME, sizeof(char))));
   MEM_CHECK((Pinfodir = (char *)calloc(LONAME, sizeof(char))));

   /*
    * Hold current directory.
    */
#ifdef NEXTSTEP
   getwd(curdir);
#else
   getcwd(curdir, LONAME);
#endif

   /*
    * Get the name of the upload directory.
    */
   if (!get_var_from_pftprc("PFTPRECEIVE", Pinfodir, 0)) {
      if (!PFTP_PEEK) fprintf(stderr,  "** PFTPRECEIVE is not defined.  Please, rtfm!\n");
      exit(1);
   }

   /*
    * Check if directory exists.  If not create it.
    */
   if (stat(Pinfodir, &buf)) {
      if (mkdir(Pinfodir, 0755) < 0) {
         fprintf(stderr, "** %s: %s!\n", Pinfodir, _PFTP_ERROR_ARRAY_);
         exit(1);
      }
   }

   /*
    * Change to the upload directory.
    */
   if (chdir(Pinfodir) < 0) {
      fprintf(stderr, "** %s: %s\n", Pinfodir, _PFTP_ERROR_ARRAY_);
      exit(1);
   }

   if (!PFTP_PEEK) {
      /*
       * Lock the upload directory if it's unlocked.
       */
      sprintf(Puni, "%s:%d",
      getenv("HOSTNAME")? getenv("HOSTNAME"): getenv("HOST")? getenv("HOST"): "unknown_host"
      , getpid());
      if ((num = open(Puni, O_CREAT, 0600)) < 0) {
         fprintf(stderr, "** %s: %s\n", Puni, _PFTP_ERROR_ARRAY_);
         exit(1);
      }
      close(num);
      link(Puni, "lock");
      buf.st_mode = 0;
      if (stat(Puni, &buf) < 0) {
         fprintf(stderr, "** %s: %s\n", Puni, _PFTP_ERROR_ARRAY_);
         exit(1);
      }
      if (buf.st_nlink != 2) {
         fprintf(stderr, "**\n** There is another pftp manager already running!\n");
         fprintf(stderr, "** Look in `%s' to see the host name and the pid.\n",
         Pinfodir);
         fprintf(stderr, "** If the lock file is outdated remove the `hostname:pid' file\n");
         fprintf(stderr, "** and the `lock' file.\n**\n");
         unlink(Puni);
         if (Pinfodir) free(Pinfodir);
         if (Puni) free(Puni);
         if (curdir) free(curdir);
         if (blankline) free(blankline);
         if (saving_str) free(saving_str);
         if (lastline) free(lastline);
         if (headline) free(headline);
         if (cdir) free(cdir);
         exit(1);
      }
      
      /*
       * Set PFTP_SORT_BY.
       */
      if (get_var_from_pftprc("PFTPSORTBY", Pinfodir, 0)) {
         if (*Pinfodir == 's') PFTP_SORT_BY = 's';
         else if (*Pinfodir == 't') PFTP_SORT_BY = 't';
      }

      /*
       * Set the pager.
       */
      if (!set_prog(pager, "PFTPPAGER")) USE_PFTPPAGER = 1;
   }
   if (Pinfodir) free(Pinfodir);

   /*
    * Read the Pinfo file.
    */
   if ((endnum = read_pinfo(pmain)) < 0) {
      if (!PFTP_PEEK) {
         unlink(Puni);
         unlink("lock");
      }
      exit(1);
   }

   /*
    * Just peeking.
    */
   if (PFTP_PEEK) {
      for (tmp=pmain; *tmp && **tmp != 'N' && **tmp != 'O'; tmp++);
      if (*tmp) {
         fprintf(stdout, PFTP_NEW_DATA_MSG);
         exit(0);
      }
      exit(1);
   }

   /*
    * Start interaction with the user.
    */
   num=0;
   cur= *pmain ? 0 : -1;
   fprintf(stdout, "[?25l\r");
   MEM_CHECK((subj = (char *)calloc(LONAME, sizeof(char))));
   set_tty(2);
   do {
      if (update) {
         firstline = num;
         fprintf(stdout, "[H[J%s\n", headline);
         fprintf(stdout, "%s\n", blankline);
         for (j=0; j < height && *(pmain+num); j++, num++) {
            *subj = '\0';
            sprintf(cdir, "%s %s %.15s  %s  %s", num==cur? "-->": "   ",
            *(pmain+num), DATESTRING(num), USERSTRING(num),
            get_subject(subj, USERSTRING(num), DATESTRING(num)) ? subj: "");
            *(cdir+_WINCOLS_) = '\0';
            fprintf(stdout, "%s%s", cdir, *subj ? "" : "\n");
         }
         fprintf(stdout, lastline);
         update=0;
      }
      while ((c = fgetc(stdin)) == '' || c == '[');
      do {
         switch(c) {
            case 'j':
            case 'B':
               c = 0;
               if (cur < 0) break;
               if (cur < firstline + height - 1 && cur < endnum) {
                  fprintf(stdout, "[%dH   ", cur+3-firstline);
                  cur++;
                  fprintf(stdout, "[%dH-->[%dH", cur+3-firstline, _WINROWS_);
                  break;
               }
               else if (cur >= endnum) {
                  cur = endnum;
                  break;
               }
               else if (cur < firstline + height - 1) break;
               else cur++;
            case ' ':
               if (num > endnum) update = 2;
               if (num + height > endnum) num = endnum - height + 1;
               if (num < 0) num = 0;
               if (c == ' ') {
                  if (update == 2) cur = endnum;
                  else cur = num;
               }
               update=1;
               break;
            case 'k':
            case 'A':
               if (cur < 0) break;
               if (cur > firstline) {
                  fprintf(stdout, "[%dH   ", cur+3-firstline);
                  cur--;
                  fprintf(stdout, "[%dH-->[%dH", cur+3-firstline, _WINROWS_);
                  break;
               }
               else if (cur > firstline) break;
            case 'b':
               if (cur < 0) break;
               num -= 2 * height;
               if (num < 0) num = 0;
               if (c == 'b') cur = num;
               else cur--;
               if (cur < 0) cur = 0;
               update = 1;
               break;
            case '':
               buf.st_mode = 0;
               stat("Pinfo", &buf);
               if (buf.st_size > pinfo_size) {
                  pinfo_size = buf.st_size;
                  endnum = read_pinfo(pmain);
                  num=0;
                  cur= *pmain ? 0 : -1;
               }
               num -= height;
               if (num < 0) num = 0;
               update = 1;
               break;
            case 'l':
            case 'C':
               if (cur < 0) break;
               sprintf(cdir, "%s/%s", USERSTRING(cur), DATESTRING(cur));
               buf.st_mode = 0;
               if (stat(cdir, &buf) < 0) {
                  fprintf(stderr, "[%dH** %s: %s\n",
                  _WINROWS_, cdir, _PFTP_ERROR_ARRAY_);
               }
               **(pmain+cur) = ' ';
               if (buf.st_mode & S_IFDIR) {
                  if ((c = browse_dir(cdir)) >= 0) {
                     num -= height;
                     if (num < 0) num = 0;
                     update = 1;
                  }
                  else c = 'd';
               }
               break;
            case 't':
               if (cur < 0) break;
               if (*(*(pmain+cur)+2) == '*') *(*(pmain+cur)+2) = ' ';
               else {
                  *(*(pmain+cur)+2) = '*';
                  **(pmain+cur) = ' ';
               }
               sprintf(cdir, "%s/%s", USERSTRING(cur), DATESTRING(cur));
               if ((j = set_entry_in_files(tfiles, cdir, &tlastentry)) >= 0) {
                  rm_entry_from_files(tfiles, *(tfiles+j), &tlastentry);
               }
               sprintf(cdir, " %s %.15s  %s  %s", 
               *(pmain+cur), DATESTRING(cur), USERSTRING(cur),
               get_subject(subj, USERSTRING(cur), DATESTRING(cur)) ? subj: "");
               *(cdir+_WINCOLS_-4) = '\0';
               fprintf(stdout, "[%dH-->%s[%dH",
               cur+3-firstline, cdir, _WINROWS_);
               c = 'j';
               break;
            case 'd':
               if (cur < 0) break;
               if (*(*(pmain+cur)+1) == 'D') *(*(pmain+cur)+1) = ' ';
               else {
                  *(*(pmain+cur)+1) = 'D';
                  **(pmain+cur) = ' ';
               }
               sprintf(cdir, "%s/%s", USERSTRING(cur), DATESTRING(cur));
               if ((j = set_entry_in_files(dfiles, cdir, &dlastentry)) >= 0) {
                  rm_entry_from_files(dfiles, *(dfiles+j), &dlastentry);
               }
               sprintf(cdir, " %s %.15s  %s  %s", 
               *(pmain+cur), DATESTRING(cur), USERSTRING(cur),
               get_subject(subj, USERSTRING(cur), DATESTRING(cur)) ? subj: "");
               *(cdir+_WINCOLS_-4) = '\0';
               fprintf(stdout, "[%dH-->%s[%dH",
               cur+3-firstline, cdir, _WINROWS_);
               c = 'j';
               break;
            case 'n':
            case 'N':
               if (cur < 0) break;
               if (**(pmain+cur) != 'N') {
                  **(pmain+cur) = 'N';
                  if (*(*(pmain+cur)+1) == 'D') c = 'd';
                  sprintf(cdir, " %s %.15s  %s  %s", 
                  *(pmain+cur), DATESTRING(cur), USERSTRING(cur),
                  get_subject(subj, USERSTRING(cur), DATESTRING(cur)) ? subj: "");
                  *(cdir+_WINCOLS_-4) = '\0';
                  fprintf(stdout, "[%dH-->%s[%dH",
                  cur+3-firstline, cdir, _WINROWS_);
               }
               if (c != 'd') c = 'j';
               break;
            case '\n':
               if (cur < 0) break;
               sprintf(cdir, "%s/%s/._pftp_info_file", USERSTRING(cur), DATESTRING(cur));
               MEM_CHECK((secondline = (char *)calloc(SONAME, sizeof(char))));
               sprintf(secondline, " From: %s   Date: %s", USERSTRING(cur), DATESTRING(cur));
               for (ctmp=secondline; *ctmp && ctmp - secondline < _WINCOLS_; ctmp++);
               for (; ctmp - secondline < _WINCOLS_; ctmp++) *ctmp = ' ';
               *ctmp = '\0';
               buf.st_mode = 0;
               if (!stat(cdir, &buf) && pftp_pager(cdir, secondline, blankline)) {
                     num -= height;
                     if (num < 0) num = 0;
                     update = 1;
               }
               else {
                  fprintf(stderr, "[%dH%s", _WINROWS_, blankline);
                  fprintf(stderr, "** No information available.");
               }
               free(secondline);
               break;
            case 'S':
               if (cur < 0) break;
               sprintf(cdir, "%s/%s/._pftp_info_file", USERSTRING(cur), DATESTRING(cur));
               buf.st_mode = 0;
               if (!stat(cdir, &buf) && (buf.st_mode & S_IFREG)) {
                  char *dest=NULL;
                  MEM_CHECK((ctmp = (char *)calloc(LONAME, sizeof(char))));
                  fprintf(stdout, "[?25h\r");
                  if ((dest = tabdir(curdir))) {
                     buf.st_mode = 0;
                     if (!stat(dest, &buf) && (buf.st_mode & S_IFDIR)) {
                        sprintf(ctmp, "%s/mesg_from_%s", dest, USERSTRING(cur));
                     }
                     else strcpy(ctmp, dest);
                     if (create_dir(dest, NULL)) {
                        fprintf(stderr, "%s", saving_str);
                        copy_file(cdir, ctmp);
                     }
                     rm_entry_from_files(dfiles, cdir, &dlastentry);
                  }
                  fprintf(stdout, "[?25l\r");
                  free(ctmp);
                  num -= height;
                  if (num < 0) num = 0;
                  update = 1;
               }
               else {
                  fprintf(stderr, "[%dH%s", _WINROWS_, blankline);
                  fprintf(stderr, "** No message file available.");
               }
               break;
            case 's':
               if (cur < 0) break;
               for (j=0,tmp=tfiles; tmp - tfiles <= tlastentry; tmp++) {
                  buf.st_mode = 0;
                  if (*tmp && !stat(*tmp, &buf) && (buf.st_mode & S_IFREG)) {
                     j = 1;
                     break;
                  }
               }
               /*
                * Make cursor visible.
                */
               fprintf(stdout, "[?25h\r");
               if (j) {
                  fprintf(stdout, "[%dH%s", _WINROWS_, blankline);
                  fprintf(stdout, " Save a) taged files  b) current directory  c) cancel: ");
                  while ((c = fgetc(stdin)) != 'a' && c != 'b' && c != 'c');
               }
               if (c != 'c') {
                  char *dest=NULL;
                  if ((dest = tabdir(curdir))) {
                     if (c == 'b' || !j) {
                        /*
                         * Save current directory.
                         */
                        sprintf(cdir, "%s/%s", USERSTRING(cur), DATESTRING(cur));
                        buf.st_mode = 0;
                        if (!stat(dest, &buf)) {
                           if (buf.st_mode & S_IFDIR) {
                              fprintf(stderr, "%s", saving_str);
                              save_dir(cdir, dest);
                           }
                           else {
                              fprintf(stdout, "[%dH%s", _WINROWS_, blankline);
                              fprintf(stdout, "** `%s' is not a directory as needed.", dest);
                           }
                        }
                        else if (create_dir(dest, "")) {
                           fprintf(stderr, "%s", saving_str);
                           save_dir(cdir, dest);
                        }
                        c = 'd';
                     }
                     else if (create_dir(dest, "")) {
                        fprintf(stderr, "%s", saving_str);
                        save_tags(dest);
                     }
                  }
               }
               fprintf(stdout, "[?25l\r");
               (*statstr)->OVERWRITE = 0;
               (*statstr)->_SKIP_ = 0;
               num -= height;
               if (num < 0) num = 0;
               update = 1;
               break;
            case '?':
            if (cdir) {
               for (ctmp=cdir; ctmp - cdir < _WINCOLS_; ctmp++) *ctmp = ' ';
               *ctmp = '\0';
               mhelp(cdir);
               num -= height;
               if (num < 0) num = 0;
               update = 1;
            }
               break;
            case 'w':
               warranty();
               num -= height;
               if (num < 0) num = 0;
               update = 1;
               break;
         }
      } while (c == 'j' || c == 'd');
   } while (c != 'q');
   unlink(Puni);
   unlink("lock");
   if (Puni) free(Puni);

   /*
    * Remove files from dfiles.
    */
   num = 0; 
   for (tmp=dfiles; tmp-dfiles <= dlastentry; tmp++) {
      if (*tmp) {
         num = 1;
         break;
      }
   }
   fprintf(stdout, "[%dH%s", _WINROWS_, blankline);
   if (num) {
      fprintf(stdout, " Delete marked files (Y|n)? ");
      while ((c = fgetc(stdin)) != 'y' && c != 'Y' && c != 'n' && c != 'N' && c != '\n');
      if (c == 'y' || c == 'Y' || c == '\n') {
         for (tmp=dfiles; tmp - dfiles <= dlastentry; tmp++) {
            if (*tmp) {
               buf.st_mode = 0;
               if (!stat(*tmp, &buf)) {
                  if ((buf.st_mode & S_IFDIR) && empty_dir(*tmp)) rmdir(*tmp);
                  else unlink(*tmp);
               }
            }
         }
      }
   }

   if (subj) free(subj);
   if (curdir) free(curdir);
   if (blankline) free(blankline);
   if (saving_str) free(saving_str);
   if (lastline) free(lastline);
   if (headline) free(headline);
   if (cdir) free(cdir);
   /*
    * Read and write `Pinfo'.
    */
   write_pinfo(pmain);

   fprintf(stdout, "[?25h\r");
   fprintf(stdout, "[%dH*** Have a nice day! ;^)                                              \n", _WINROWS_);
   set_tty(0);
}


short empty_dir(char *s)
{
   DIR *dp;
   char *pif=NULL;
#if defined Linux || defined NEXSTEP
   struct direct *dir;
#else
   struct dirent *dir;
#endif

   if (!(dp = opendir(s))) {
      return 0;
   }
   while ((dir = readdir(dp)))
   {
      if (dir->d_ino) {
         if (!strcmp(dir->d_name, "._pftp_info_file")) {
            MEM_CHECK((pif = (char *)calloc(LONAME, sizeof(char))));
            sprintf(pif, "%s/._pftp_info_file", s);
         }
         else if (strcmp(dir->d_name, "..") && strcmp(dir->d_name, ".")) {
            closedir(dp);
            if (pif) free(pif);
            return 0;
         }
      }
   }
   closedir(dp);

   if (pif) unlink(pif);
   free(pif);

   return 1;
}


void rm_empty_dirs(void)
{
   struct stat buf;
   DIR *dp;
#if defined Linux || defined NEXSTEP
   struct direct *dir;
#else
   struct dirent *dir;
#endif

   if (!(dp = opendir("."))) {
      return;
   }
   while ((dir = readdir(dp)))
   {
      if (dir->d_ino && strcmp(dir->d_name, "..") && strcmp(dir->d_name, ".")) {
         buf.st_mode = 0;
         if (!stat(dir->d_name, &buf) && (buf.st_mode & S_IFDIR)) {
            if (empty_dir(dir->d_name)) rmdir(dir->d_name);
         }
      }
   }
   closedir(dp);
}


int read_pinfo(char **pmain)
{
   FILE *fp=NULL;
   char *ptmp=NULL, *tmpstr=NULL;
   struct stat buf;
   int i=0;

   if ((fp = fopen("Pinfo", "r")) == NULL) return 0;
   buf.st_mode = 0;
   stat("Pinfo", &buf);
   if (!buf.st_size) {
      fclose(fp);
      return 0;
   }
   pinfo_size = buf.st_size;

   MEM_CHECK((ptmp = (char *)calloc(SONAME, sizeof(char))));
   for (i=0; *(pmain+i); i++) {
      if (!fgets(ptmp, SONAME, fp)) {
         fclose(fp);
         return i-1;
      }
   }
   for (;fgets(ptmp, SONAME, fp); i++) {
      for (tmpstr=ptmp+5; *tmpstr && *tmpstr != '\n'; tmpstr++);
      *tmpstr = '\0';
      *(ptmp+3)  = '\0';
      *(ptmp+24) = '\0';
      *(pmain+i) = ptmp;
      MEM_CHECK((ptmp = (char *)calloc(SONAME, sizeof(char))));
   }
   if (ptmp) free(ptmp);
   *(pmain+i) = NULL;
   fclose(fp);

   return i-1;
}


void write_pinfo(char **pmain)
{
   FILE *fp=NULL;
   char *wuni=NULL, *cdir=NULL;
   char *tmp=NULL;
   int num=0;
   struct stat buf;

   MEM_CHECK((wuni = (char *)calloc(SONAME, sizeof(char))));
   MEM_CHECK((cdir = (char *)calloc(LONAME, sizeof(char))));
   sprintf(wuni, "%d:%s", getpid(),
   getenv("HOSTNAME")? getenv("HOSTNAME"): getenv("HOST")? getenv("HOST"): "unknown_host");
   if ((num = open(wuni, O_CREAT, 0600)) < 0) {
      fprintf(stderr, "** %s: %s\n", wuni, _PFTP_ERROR_ARRAY_);
      set_tty(0);
      exit(1);
   }
   close(num);
   for (buf.st_nlink=0, num=0; buf.st_nlink != 2; num++) {
      link(wuni, "wlock");
      buf.st_mode = 0;
      if (stat(wuni, &buf) < 0) {
         fprintf(stderr, "** %s: %s\n", wuni, _PFTP_ERROR_ARRAY_);
         set_tty(0);
         exit(1);
      }
      if (buf.st_nlink != 2) sleep(1);
      if (num > 2) {
         fprintf(stderr, "** Can't write to Pinfo file!\n** Please, have a look in your upload directory.\n");
         fprintf(stderr, "** If necessary remove the lock file `wlock' and the file `pid:hostname'.\n");
         unlink(wuni);
         return;
      }
   }

   /*
    * Check if new files arrived.
    */
   num = read_pinfo(pmain);

   /*
    * Remove empty `user@date' directories.
    */
   rm_empty_dirs();

   if ((fp = fopen("Pinfo", "w")) == NULL) {
      fprintf(stderr, "** Pinfo: %s\n", _PFTP_ERROR_ARRAY_);
      set_tty(0);
      exit(1);
   }
   for (; (tmp = *pmain) && num >= 0; num--, pmain++) {
      if (*(tmp+1) == 'D') {
         sprintf(cdir, "%s/%s", USERSTRING(0), DATESTRING(0));
         buf.st_mode = 0;
         if (stat(cdir, &buf)) continue;
      }
      else if (*tmp == 'N') *tmp = 'O';
      *(tmp+1)  = ' ';
      *(tmp+2)  = ' ';
      *(tmp+3)  = ' ';
      *(tmp+24) = ' ';
      fprintf(fp, "%s\n", tmp);
   }
   fclose(fp);
   unlink("wlock");
   unlink(wuni);
   if (cdir) free(cdir);
   if (wuni) free(wuni);
}


char browse_dir(char *pdir)
{
   char *cdir=NULL, **tmp=NULL, c='\0', *ctmp=NULL;
   char *direntries[HUNAME];
   char *secondline=NULL;
   int endnum=0, height=_WINROWS_-5;
   int num=0, cur=0, j=0, update=1;
   int firstline=0;
   DIR *dp=NULL;
#if defined Linux || defined NEXSTEP
   struct direct *dir;
#else
   struct dirent *dir;
#endif
   struct stat buf;

   MEM_CHECK((cdir = (char *)calloc(LONAME, sizeof(char))));
   MEM_CHECK((*direntries = (char *)calloc(SONAME, sizeof(char))));

   /*
    * Open the directory and read entries.
    */
   if ((dp = opendir(pdir)) == NULL) {
      fprintf(stdout, "[%dH** %s: %s\n", _WINROWS_, pdir, _PFTP_ERROR_ARRAY_);
      return -1;
   }
   for (endnum=0; (dir = readdir(dp)); endnum++)
   {
      if (!dir->d_ino || !mycmp(".", dir->d_name) || !mycmp("..", dir->d_name) \
         || !mycmp("._pftp_info_file", dir->d_name)) {
         endnum--;
      }
      else {
         MEM_CHECK((*(direntries+endnum) = (char *)calloc(LONAME, sizeof(char))));
         strcpy(*(direntries+endnum), dir->d_name);
      }
   }
   closedir(dp);
   *(direntries+endnum) = NULL;

   if (!endnum) {
      fprintf(stdout, "[%dH%s", _WINROWS_, blankline);
      fprintf(stdout, "** Contains no data!");
      return -1;
   }

   /*
    * Sort directory entries.
    */
   bubble_sort(direntries, pdir, endnum);

   endnum--;
   do {
      if (update > 0) {
         firstline = num;
         fprintf(stdout, "[H[J%s\n", headline);
         strcpy(cdir, pdir);
         *(cdir+(2*_WINCOLS_)-2) = '\0';
         fprintf(stdout, "%s:\n", cdir);
         if (strlen(cdir) < _WINCOLS_) fprintf(stdout, "\n");
         for (j=0; j < height && *(direntries+num); j++, num++) {
            printlistentry(num==cur? "-->":"   ", pdir, *(direntries+num));
         }
         fprintf(stdout, lastline);
         update=0;
      }
      else if (update < 0) {
         num -= height;
         if (num < 0) num = 0;
         update = 1;
      }
      while ((c = fgetc(stdin)) == '' || c == '[');
      do {
         switch(c) {
            case 'j':
            case 'B':
               c = '\0';
               if (cur < firstline + height - 1 && cur < endnum) {
                  fprintf(stdout, "[%dH   ", cur+4-firstline);
                  cur++;
                  fprintf(stdout, "[%dH-->[%dH", cur+4-firstline, _WINROWS_);
                  break;
               }
               else if (cur >= endnum) {
                  cur = endnum;
                  break;
               }
               else if (cur < firstline + height - 1) break;
               else cur++;
            case ' ':
               if (cur >= endnum) break;
               if (num >= endnum) update = 2;
               if (num + height > endnum) num = endnum - height + 1;
               if (num < 0) num = 0;
               if (c == ' ') {
                  if (update == 2) cur = endnum;
                  else cur = num;
               }
               update=1;
               break;
            case 'k':
            case 'A':
               if (cur <= 0) break;
               if (cur > firstline) {
                  fprintf(stdout, "[%dH   ", cur+4-firstline);
                  cur--;
                  fprintf(stdout, "[%dH-->[%dH", cur+4-firstline, _WINROWS_);
                  break;
               }
               else if (cur > firstline) break;
            case 'b':
               if (cur <= 0) break;
               num -= 2 * height;
               if (num < 0) num = 0;
               if (c == 'b') cur = num;
               else cur--;
               if (cur < 0) cur = 0;
               update = 1;
               break;
            case '':
               num -= height;
               if (num < 0) num = 0;
               update = 1;
               break;
            case 'l':
            case 'C':
            c = '0';
               sprintf(cdir, "%s/%s", pdir, *(direntries+cur));
               buf.st_mode = 0;
               if (stat(cdir, &buf) < 0) {
                  fprintf(stdout, "[%dH** %s: %s\n",
                  _WINROWS_, cdir, _PFTP_ERROR_ARRAY_);
               }
               if (buf.st_mode & S_IFDIR) {
                  char c='\0';
                  if ((c = browse_dir(cdir)) == 'q' || c == 'm') {
                     if (cdir) free(cdir);
                     for (; endnum; endnum--) {
                        if (*(direntries+endnum)) free(*(direntries+endnum));
                     }
                     return c;
                  }
                  else if (c >= 0) {
                     num -= height;
                     if (num < 0) num = 0;
                     update = 1;
                  }
               }
               else c = '\n';
               break;
            case 'm': 
               if (cdir) free(cdir);
               for (; endnum; endnum--) {
                  if (*(direntries+endnum)) free(*(direntries+endnum));
               }
               return 'm';
            case 'h':
            case 'D':
               if (cdir) free(cdir);
               for (; endnum; endnum--) {
                  if (*(direntries+endnum)) free(*(direntries+endnum));
               }
               c = 'h';
               break;
            case 't':
               sprintf(cdir, "%s/%s", pdir, *(direntries+cur));
               if ((j = set_entry_in_files(tfiles, cdir, &tlastentry)) >= 0) {
                  rm_entry_from_files(tfiles, *(tfiles+j), &tlastentry);
               }
               fprintf(stdout, "[%dH", cur+4-firstline);
               printlistentry("-->", pdir, *(direntries+cur));
               fprintf(stdout, "[%dH", _WINROWS_);
               c = 'j';
               break;
            case 'd':
               sprintf(cdir, "%s/%s", pdir, *(direntries+cur));
               if ((j = set_entry_in_files(dfiles, cdir, &dlastentry)) >= 0) {
                  rm_entry_from_files(dfiles, *(dfiles+j), &dlastentry);
               }
               fprintf(stdout, "[%dH", cur+4-firstline);
               printlistentry("-->", pdir, *(direntries+cur));
               fprintf(stdout, "[%dH", _WINROWS_);
               c = 'j';
               break;
            case 'q':
               fprintf(stdout, "[?25h\r[%dH%s", _WINROWS_, blankline);
               fprintf(stdout, "*** Really quit (Y|n)? ");
               c = fgetc(stdin);
               if (c == 'y' || c == 'Y' || c == '\n') {
                  if (cdir) free(cdir);
                  for (; endnum; endnum--) {
                     if (*(direntries+endnum)) free(*(direntries+endnum));
                  }
                  return 'q';
               }
               else {
                  num -= height;
                  if (num < 0) num = 0;
                  update = 1;
               }
               fprintf(stdout, "[?25l\r");
               break;
            case '\n':
               sprintf(cdir, "%s/%s", pdir, *(direntries+cur));
               if (USE_PFTPPAGER) {
                  MEM_CHECK((secondline = (char *)calloc(SONAME, sizeof(char))));
                  sprintf(secondline, " File: `%s'", *(direntries+cur));
                  for (ctmp=secondline; *ctmp && ctmp - secondline < _WINCOLS_; ctmp++);
                  for (; ctmp - secondline < _WINCOLS_; ctmp++) *ctmp = ' ';
                  *ctmp = '\0';
                  buf.st_mode = 0;
                  if (!stat(cdir, &buf) && !(buf.st_mode & S_IFDIR) \
                     && pftp_pager(cdir, secondline, blankline)) {
                        num -= height;
                        if (num < 0) num = 0;
                        update = 1;
                  }
                  else {
                     fprintf(stderr, "[%dH%s", _WINROWS_, blankline);
                     if (buf.st_mode & S_IFDIR) {
                        c = 'l';
                        break;
                     }
                     else fprintf(stderr, "** Can't open file `%s'.", cdir);
                  }
                  free(secondline);
               }
               else {
                  buf.st_mode = 0;
                  if (!stat(cdir, &buf)) {
                     if (!(buf.st_mode & S_IFDIR)) {
                        fprintf(stderr, "[?25h\r[H[J");
                        run_prog(pager, cdir); 
                        fprintf(stderr, "[?25l\r");
                     }
                     else {
                        fprintf(stdout, "[%dH%s", _WINROWS_, blankline);
                        if (buf.st_mode & S_IFDIR) {
                           c = 'l';
                           break;
                        }
                        else fprintf(stderr, "** Can't open file `%s'.", cdir);
                     }
                  }
                  num -= height;
                  if (num < 0) num = 0;
                  update = 1;
               }
               c = 0;
               break;
            case 's':
               j = 0; 
               for (tmp=tfiles; tmp-tfiles <= tlastentry; tmp++) {
                  buf.st_mode = 0;
                  if (*tmp && !stat(*tmp, &buf) && (buf.st_mode & S_IFREG)) {
                     j = 1;
                     break;
                  }
               }
               sprintf(cdir, "%s/%s", pdir, *(direntries+cur));
               buf.st_mode = 0;
               stat(cdir, &buf);
               /*
                * Make cursor visible.
                */
               fprintf(stdout, "[?25h\r");
               fprintf(stdout, "[%dH%s", _WINROWS_, blankline);
               if (j) {
                  fprintf(stdout, " Save a) taged files  b) current %s c) cancel: ",
                  (buf.st_mode & S_IFDIR) ? "directory recursively": "file");
                  while ((c = fgetc(stdin)) != 'a' && c != 'b' && c != 'c');
               }
               if (c != 'c') {
                  char *dest=NULL;
                  if ((dest = tabdir(curdir))) {
                     if (c == 'b' || !j) {
                        /*
                         * Save current directory.
                         */
                        if (buf.st_mode & S_IFDIR) {
                           char *ddir=NULL;
                           /*
                            * Set destination directory.
                            */
                           for (ddir=cdir; *ddir; ddir++);
                           for (ddir--; *ddir != '/' && ddir != cdir; ddir--);
                           buf.st_mode = 0;
                           if (!stat(dest, &buf)) {
                              if (buf.st_mode & S_IFDIR) {
                                 if (create_dir(dest, ddir)) {
                                    strcat(dest, ddir);
                                    fprintf(stderr, "%s", saving_str);
                                    save_dir(cdir, dest);
                                 }
                              }
                              else {
                                 fprintf(stdout, "[%dH%s", _WINROWS_, blankline);
                                 fprintf(stdout, "** `%s' is not a directory as needed.", dest);
                              }
                           }
                           else if (create_dir(dest, ddir)) {
                              strcat(dest, ddir);
                              fprintf(stderr, "%s", saving_str);
                              save_dir(cdir, dest);
                           }
                           c = 'd';
                        }
                        /*
                         * Save current file.
                         */
                        else {
                           if (create_dir(dest, NULL)) {
                              buf.st_mode = 0;
                              if (!stat(dest, &buf) && (buf.st_mode & S_IFDIR)) {
                                 strcat(dest, "/");
                                 for (ctmp=cdir; *ctmp; ctmp++);
                                 for (ctmp--; *ctmp != '/' && ctmp != cdir; ctmp--);
                                 strcat(dest, ++ctmp);
                              }
                              fprintf(stderr, "%s", saving_str);
                              copy_file(cdir, dest);
                           }
                           c = 'j';
                        }
                     }
                     else if (create_dir(dest, "")) {
                        fprintf(stderr, "%s", saving_str);
                        save_tags(dest);
                     }
                  }
               }
               fprintf(stdout, "[?25l\r");
               (*statstr)->OVERWRITE = 0;
               (*statstr)->_SKIP_ = 0;
               num -= height;
               if (num < 0) num = 0;
               update = 1;
               break;
            case '?':
               if (cdir) {
                  char *ctmp=NULL;
                  for (ctmp=cdir; ctmp - cdir < _WINCOLS_; ctmp++) *ctmp = ' ';
                  *ctmp = '\0';
                  mhelp(cdir);
                  num -= height;
                  if (num < 0) num = 0;
                  update = 1;
               }
               break;
            case 'w':
               warranty();
               num -= height;
               if (num < 0) num = 0;
               update = 1;
               break;
         }
      } while (c == 'j' || c == 'd' || c == '\n' || c == 'l');
   } while (c != 'h');
   return '\0';
}


int set_entry_in_files(char **files, char *str, int *lastentry)
{
   char **tmp;
   struct stat buf;

   buf.st_mode = 0;
   stat(str, &buf);

   if (buf.st_mode & S_IFDIR) {
      char *tmpdir[HUNAME];
#if defined Linux || defined NEXSTEP
      struct direct *dir;
#else
      struct dirent *dir;
#endif
      DIR *dp=NULL;

      tmp = tmpdir;
      if ((dp = opendir(str)) == NULL) {
         fprintf(stderr, "** %s: %s\n", str, _PFTP_ERROR_ARRAY_);
         set_tty(0);
         exit(1);
      }
      while ((dir = readdir(dp)))
      {
         if (dir->d_ino && mycmp(".", dir->d_name) && mycmp("..", dir->d_name) \
            && mycmp("._pftp_info_file", dir->d_name)) {
            MEM_CHECK((*tmp = (char *)calloc(LONAME, sizeof(char))));
            sprintf(*tmp++, "%s/%s", str, dir->d_name);
         }
      }
      closedir(dp);
      for (tmp--; tmp+1 != tmpdir; tmp--) {
         set_entry_in_files(files, *tmp, lastentry);
         free(*tmp);
      }
   }
   /*
    * Check if files was already taged.
    */
   for (tmp=files; tmp - files <= *lastentry; tmp++) {
      if (*tmp && !mycmp(*tmp, str)) return tmp - files;
   }

   /*
    * Search an empty field and set the new entry.
    */
   for (tmp=files; *tmp; tmp++);
   if (tmp - files >= *lastentry) *lastentry = tmp - files;
   MEM_CHECK((*tmp = (char *)calloc(LONAME, sizeof(char))));
   strcpy(*tmp, str);
   return -1;
}


void rm_entry_from_files(char **files, char *str, int *lastentry)
{
   char **tmp;
   struct stat buf;

   buf.st_mode = 0;
   stat(str, &buf);

   if (buf.st_mode & S_IFDIR) {
      char *tmpdir[HUNAME];
#if defined Linux || defined NEXSTEP
      struct direct *dir;
#else
      struct dirent *dir;
#endif
      DIR *dp=NULL;

      tmp = tmpdir;
      if ((dp = opendir(str)) == NULL) {
         fprintf(stderr, "** %s: %s\n", str, _PFTP_ERROR_ARRAY_);
         set_tty(0);
         exit(1);
      }
      while ((dir = readdir(dp)))
      {
         if (dir->d_ino && mycmp(".", dir->d_name) && mycmp("..", dir->d_name) \
            && mycmp("._pftp_info_file", dir->d_name)) {
            MEM_CHECK((*tmp = (char *)calloc(LONAME, sizeof(char))));
            sprintf(*tmp++, "%s/%s", str, dir->d_name);
         }
      }
      closedir(dp);
      for (tmp--; tmp+1 != tmpdir; tmp--) {
         rm_entry_from_files(files, *tmp, lastentry);
         free(*tmp);
      }
   }
   for (tmp=files; tmp - files <= *lastentry; tmp++) {
      if (*tmp && !mycmp(*tmp, str)) {
         free(*tmp);
         *tmp=NULL;
         if (tmp - files == *lastentry) {
            for (;!*tmp && tmp != files; tmp--, (*lastentry)--);
            if (*lastentry < 0) *lastentry = 0;
         }
         return;
      }
   }
}


void printlistentry(char *arrow, char *dir, char *file)
{
   int i=0,j=0;
   char perms[10];
   char name[LONAME];
   char **tmp;
   char type='\0', deletef=' ', tagf=' ';
   struct stat buf;
   char *modes[] = {
      "---", "--x", "-w-", "-wx",
      "r--", "r-x", "rw-", "rwx"
   };

   
   /*
    * Set full path name.
    */
   sprintf(name, "%s/%s", dir, file);

   /*
    * Get inode contents.
    */
   buf.st_mode = 0;
   stat(name, &buf);
   
   switch(buf.st_mode & S_IFMT) {
      case S_IFREG:
         type = '-';
         break;
      case S_IFDIR:
         type = 'd';
         break;
      default:
         type = '?';
         break;
   }

   *perms = '\0';
   for (i=2; i >= 0; i--) {
      j = (buf.st_mode >> (i*3)) & 07;
      strcat(perms, *(modes+j));
   }

   /*
    * Check if the file was taged or will
    * be deleted.
    */
   for (tmp=tfiles; tmp - tfiles <= tlastentry; tmp++) {
      if (*tmp && !mycmp(*tmp, name)) {
         tagf='*';
         break;
      }
   }
   for (tmp=dfiles; tmp - dfiles <= dlastentry; tmp++) {
      if (*tmp && !mycmp(*tmp, name)) {
         deletef='D';
         break;
      }
   }

   /*
    * Finally print out the line.
    */
   sprintf(name, "%s %c%c %c%s %9ld %.12s %s",
   arrow, deletef, tagf, type, perms, buf.st_size, ctime(&buf.st_mtime)+4, file);
   *(name+_WINCOLS_-1) = '\0';
   fprintf(stdout, "%s\n", name);
}


short mycmp(char *s1, char *s2)
{
   for (; *s1 == *s2 && *s1 && *s2; s1++, s2++);
   if (!*s1 && !*s2) return 0;
   return 1;
}


void bubble_sort(char **vec, char *pdir, long size)
{
   long i, j;
   char *tmp=NULL;
   char *str=NULL;
   struct stat buf;

   if (PFTP_SORT_BY == 'a') { /* sort alphabetically */
      for (i=1; i < size; i++) {
         for (j=size-1; j >= i; j--) {
            if (strcmp(*(vec+j-1), *(vec+j)) > 0) {
               tmp = *(vec+j-1);
               *(vec+j-1) = *(vec+j);
               *(vec+j) = tmp;
            }
         }
      }
   }
   else if (PFTP_SORT_BY == 't') { /* sort by time */
      time_t first=0, second=0;
      MEM_CHECK((str = (char *)calloc(LONAME, sizeof(char))));
      for (i=1; i < size; i++) {
         for (j=size-1; j >= i; j--) {
            sprintf(str, "%s/%s", pdir, *(vec+j-1));
            stat(str, &buf);
            first = buf.st_mtime;
            sprintf(str, "%s/%s", pdir, *(vec+j));
            stat(str, &buf);
            second = buf.st_mtime;
            if (first > second) {
               tmp = *(vec+j-1);
               *(vec+j-1) = *(vec+j);
               *(vec+j) = tmp;
            }
         }
      }
      free(str);
   }
   else if (PFTP_SORT_BY == 's') { /* sort by size */
      off_t first=0, second=0;
      MEM_CHECK((str = (char *)calloc(LONAME, sizeof(char))));
      for (i=1; i < size; i++) {
         for (j=size-1; j >= i; j--) {
            sprintf(str, "%s/%s", pdir, *(vec+j-1));
            stat(str, &buf);
            first = buf.st_size;
            sprintf(str, "%s/%s", pdir, *(vec+j));
            stat(str, &buf);
            second = buf.st_size;
            if (first > second) {
               tmp = *(vec+j-1);
               *(vec+j-1) = *(vec+j);
               *(vec+j) = tmp;
            }
         }
      }
      free(str);
   }
}
