/*
 *  nwlsobj.c
 *
 *  List bindery objects
 *
 *  Copyright (C) 1996 by Volker Lendecke
 *
 */

#include <stdio.h>
#include <unistd.h>
#include <ctype.h>
#include <stdlib.h>
#include "nwcalls.h"

static char *progname;

static void
usage(void)
{
	fprintf(stderr, "usage: %s [options] pattern\n", progname);
	return;
}

static void
help(void)
{
	printf("\n");
	printf("usage: %s [options]\n", progname);
	printf("\n"
	       "-h             Print this help text\n"
	       "-S server      Server name to be used\n"
	       "-U username    Username sent to server\n"
	       "-P password    Use this password\n"
	       "-n             Do not use any password\n"
	       "-C             Don't convert password to uppercase\n"
	       "\n"
	       "-t type        Object type to be listed (decimal)\n"
	       "-o object      Object pattern\n"
	       "-v             Verbose listing\n"
	       "-a             Alternative output format\n"
	       "\n");
}

int
main(int argc, char **argv)
{
	struct ncp_conn *conn;
	struct ncp_bindery_object o;
	int found = 0;

	char default_pattern[] = "*";
	char *pattern = default_pattern;
	char *p;
	long err;
	int opt;
	int verbose = 0;
	int fmt = 0;
	__u16 type = 0xffff;
	int useConn = 0;

	progname = argv[0];

	if ((conn = ncp_initialize_2(&argc, argv, 1, NCP_BINDERY_USER, &err, 0)) == NULL)
	{
		useConn = 1;
	}
	while ((opt = getopt(argc, argv, "h?vt:o:a")) != EOF)
	{
		switch (opt)
		{
		case 'h':
		case '?':
			help();
			exit(1);
		case 't':
			type = strtol(optarg, NULL, 0);
			break;
		case 'o':
			pattern = optarg;
			break;
		case 'v':
			verbose = 1;
			break;
		case 'a':
			fmt = 1;
			break;
		default:
			usage();
			exit(1);
		}
	}

	if (useConn) {
		char* path;

		if (optind < argc)
			path = argv[optind];
		else
			path = ".";
		if (NWParsePath(path, NULL, &conn, NULL, NULL) || !conn) {
			fprintf(stderr, "%s: %s is not on ncpfs filesystem\n", progname, path);
			return 1;
		}
	} else {
		if (optind < argc)
		{
			usage();
			exit(1);
		}
	}
	for (p = pattern; *p != '\0'; p++)
	{
		*p = toupper(*p);
	}

	o.object_id = 0xffffffff;

	while (ncp_scan_bindery_object(conn, o.object_id,
				       type, pattern, &o) == 0)
	{
		found = 1;
		if (verbose != 0)
		{
			if (fmt) {
				printf("%08X %04X %-48s %d %02X %d\n",
				       (u_int32_t)o.object_id,
				       o.object_type,
				       o.object_name,
				       o.object_flags,
				       o.object_security,
				       o.object_has_prop);
			} else {
				printf("%s %08X %04X %d %02X %d\n",
				       o.object_name, (unsigned int) o.object_id,
				       (unsigned int) o.object_type,
				       o.object_flags, o.object_security,
				       o.object_has_prop);
			}
		} else
		{
			if (fmt) {
				printf("%08X %04X %s\n",
				       (u_int32_t)o.object_id,
				       o.object_type,
				       o.object_name);
			} else {
				printf("%s %08X %04X\n",
				      o.object_name, (unsigned int) o.object_id,
				      (unsigned int) o.object_type);
			}
		}
	}

	ncp_close(conn);
	return 0;
}
