/*
 * iprule.c		"ip rule".
 *
 *		This program is free software; you can redistribute it and/or
 *		modify it under the terms of the GNU General Public License
 *		as published by the Free Software Foundation; either version
 *		2 of the License, or (at your option) any later version.
 *
 * Authors:	Alexey Kuznetsov, <kuznet@ms2.inr.ac.ru>
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <syslog.h>
#include <fcntl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <arpa/inet.h>
#include <string.h>
#include <resolv.h>

#include "utils.h"

static void usage(void)
{
	fprintf(stderr, "Usage: ip rule [ list | add | del ] SELECTOR ACTION\n");
	fprintf(stderr, "SELECTOR := [ from PREFIX ] [ to PREFIX ] [ tos TOS ]\n");
	fprintf(stderr, "            [ dev STRING ] [ pref NUMBER ]\n");
	fprintf(stderr, "ACTION := [ table TABLE_ID ] [ nat ADDRESS ]\n");
	fprintf(stderr, "          [ prohibit | reject | unreachable ]\n");
	fprintf(stderr, "TABLE_ID := [ local | main | default | NUMBER ]\n");
	exit(-1);
}

int print_rule(struct sockaddr_nl *who, struct nlmsghdr *n, void *arg)
{
	FILE *fp = (FILE*)arg;
	struct rtmsg *r = NLMSG_DATA(n);
	int len = n->nlmsg_len;
	struct rtattr * tb[RTA_MAX+1];
	char abuf[256];
	SPRINT_BUF(b1);

	if (n->nlmsg_type != RTM_NEWRULE)
		return 0;

	len -= NLMSG_LENGTH(sizeof(*r));
	if (len < 0)
		return -1;
/*
	if (len < r->rtm_optlen)
		return -1;
 */

	memset(tb, 0, sizeof(tb));
	parse_rtattr(tb, RTA_MAX, RTM_RTA(r), len /*r->rtm_optlen*/);

	if (tb[RTA_PRIORITY])
		fprintf(fp, "%u:\t", *(unsigned*)RTA_DATA(tb[RTA_PRIORITY]));
	else
		fprintf(fp, "0:\t");

	if (tb[RTA_SRC]) {
		if (inet_ntop(r->rtm_family, RTA_DATA(tb[RTA_SRC]),
			      abuf, sizeof(abuf))) {
			if (r->rtm_src_len != 32)
				fprintf(fp, "from %s/%d ", abuf, r->rtm_src_len);
			else
				fprintf(fp, "from %s ", abuf);
		} else
			fprintf(fp, "from ???/%d ", r->rtm_src_len);
	} else if (r->rtm_src_len) {
		fprintf(fp, "from 0/%d ", r->rtm_dst_len);
	} else {
		fprintf(fp, "from all ");
	}

	if (tb[RTA_DST]) {
		if (inet_ntop(r->rtm_family, RTA_DATA(tb[RTA_DST]),
			      abuf, sizeof(abuf))) {
			if (r->rtm_dst_len != 32)
				fprintf(fp, "to %s/%d ", abuf, r->rtm_dst_len);
			else
				fprintf(fp, "to %s ", abuf);
		} else
			fprintf(fp, "to 0/%d ", r->rtm_dst_len);
	} else if (r->rtm_dst_len) {
		fprintf(fp, "to 0/%d ", r->rtm_dst_len);
	}

	if (r->rtm_tos)
		fprintf(fp, "tos %02x ", r->rtm_tos);

	if (tb[RTA_IIF]) {
		fprintf(fp, "iif %s ", (char*)RTA_DATA(tb[RTA_IIF]));
	}

	if (r->rtm_table)
		fprintf(fp, "lookup %s ", rttable_name(r->rtm_table));

	if (tb[RTA_FLOW])
		fprintf(fp, "flowid %s ", sprint_tc_classid(*(__u32*)RTA_DATA(tb[RTA_FLOW]), b1));

	if (r->rtm_type == RTN_NAT) {
		if (tb[RTA_GATEWAY]) {
			if (inet_ntop(r->rtm_family, RTA_DATA(tb[RTA_GATEWAY]),
				      abuf, sizeof(abuf)))
				fprintf(fp, "map-to %s", abuf);
			else
				fprintf(fp, "map-to ???");
		} else
			fprintf(fp, "masquerade");
	} else if (r->rtm_type != RTN_UNICAST)
		fprintf(fp, "%s", rtntype_name(r->rtm_type));

	fprintf(fp, "\n");
	fflush(fp);
	return 0;
}

int iprule_list(int argc, char **argv)
{
	struct rtnl_handle rth;

	if (argc > 0)
		usage();

	if (rtnl_open(&rth, 0) < 0) {
		fprintf(stderr, "cannot open rtnetlink\n");
		return 1;
	}

	if (rtnl_wilddump_request(&rth, AF_INET, RTM_GETRULE) < 0) {
		perror("cannot send dump request");
		return 1;
	}

	if (rtnl_dump_filter(&rth, print_rule, stdout, NULL, NULL) < 0) {
		fprintf(stderr, "dump terminated\n");
		return 1;
	}

	return 0;
}


int iprule_modify(int cmd, int argc, char **argv)
{
	struct rtnl_handle rth;
	struct {
		struct nlmsghdr 	n;
		struct rtmsg 		r;
		char   			buf[1024];
	} req;

	memset(&req, 0, sizeof(req));

	req.n.nlmsg_type = cmd;
	req.n.nlmsg_len = NLMSG_LENGTH(sizeof(struct rtmsg));
	req.n.nlmsg_flags = NLM_F_REQUEST;
	req.r.rtm_family = AF_INET;
	req.r.rtm_protocol = RTPROT_BOOT;
	req.r.rtm_scope = RT_SCOPE_UNIVERSE;
	req.r.rtm_table = 0;
	req.r.rtm_type = RTN_UNICAST;

	if (cmd == RTM_NEWRULE) {
		req.n.nlmsg_flags |= NLM_F_CREATE|NLM_F_EXCL;
		req.r.rtm_type = RTN_UNICAST;
	}

	while (argc > 0) {
		if (strcmp(*argv, "from") == 0) {
			inet_prefix dst;
			NEXT_ARG();
			get_prefix(&dst, *argv, AF_INET);
			req.r.rtm_src_len = dst.bitlen;
			addattr_l(&req.n, sizeof(req), RTA_SRC, &dst.data, dst.bytelen);
		} else if (strcmp(*argv, "to") == 0) {
			inet_prefix dst;
			NEXT_ARG();
			get_prefix(&dst, *argv, AF_INET);
			req.r.rtm_dst_len = dst.bitlen;
			addattr_l(&req.n, sizeof(req), RTA_DST, &dst.data, dst.bytelen);
		} else if (matches(*argv, "preference") == 0) {
			NEXT_ARG();
			addattr32(&req.n, sizeof(req), RTA_PRIORITY, atoi(*argv));
		} else if (strcmp(*argv, "tos") == 0) {
			__u8 tos;
			NEXT_ARG();
			if (strcmp(*argv, "mincost") == 0)
				tos = IPTOS_MINCOST;
			else if (strcmp(*argv, "reliability") == 0)
				tos = IPTOS_RELIABILITY;
			else if (strcmp(*argv, "throughput") == 0)
				tos = IPTOS_THROUGHPUT;
			else if (strcmp(*argv, "lowdelay") == 0)
				tos = IPTOS_LOWDELAY;
			else if (get_u8(&tos, *argv, 16))
				invarg("tos value is invalid\n");
			req.r.rtm_tos = tos;
		} else if (matches(*argv, "classid") == 0 ||
			   matches(*argv, "flowid") == 0) {
			__u32 flowid;
			NEXT_ARG();
			if (get_tc_classid(&flowid, *argv))
				invarg("invalid flowid\n");
			addattr32(&req.n, sizeof(req), RTA_FLOW, flowid);
		} else if (matches(*argv, "table") == 0) {
			NEXT_ARG();
			if (strcmp(*argv, "main") == 0)
				req.r.rtm_table = RT_TABLE_MAIN;
			else if (strcmp(*argv, "local") == 0)
				req.r.rtm_table = RT_TABLE_LOCAL;
			else if (strcmp(*argv, "default") == 0)
				req.r.rtm_table = RT_TABLE_DEFAULT;
			else
				req.r.rtm_table = atoi(*argv);
		} else if (strcmp(*argv, "nat") == 0) {
			NEXT_ARG();
			addattr32(&req.n, sizeof(req), RTA_GATEWAY, get_addr32(*argv));
			req.r.rtm_type = RTN_NAT;
		} else if (strcmp(*argv, "local") == 0)
			req.r.rtm_type = RTN_LOCAL;
		else if (matches(*argv, "broadcast") == 0 ||
			 strcmp(*argv, "brd") == 0)
			req.r.rtm_type = RTN_BROADCAST;
		else if (matches(*argv, "anycast") == 0)
			req.r.rtm_type = RTN_ANYCAST;
		else if (matches(*argv, "multicast") == 0)
			req.r.rtm_type = RTN_MULTICAST;
		else if (matches(*argv, "prohibit") == 0)
			req.r.rtm_type = RTN_PROHIBIT;
		else if (matches(*argv, "unreachable") == 0)
			req.r.rtm_type = RTN_UNREACHABLE;
		else if (matches(*argv, "unicast") == 0)
			req.r.rtm_type = RTN_UNICAST;
		else if (strcmp(*argv, "throw") == 0)
			req.r.rtm_type = RTN_THROW;
		else if (strcmp(*argv, "dev") == 0) {
			NEXT_ARG();
			addattr_l(&req.n, sizeof(req), RTA_IIF, *argv, strlen(*argv)+1);
		} else
			usage();
		argc--;
		argv++;
	}

	if (rtnl_open(&rth, 0) < 0) {
		fprintf(stderr, "cannot open rtnetlink\n");
		return 1;
	}

	if (rtnl_talk(&rth, &req.n, 0, 0, NULL, NULL, NULL) < 0)
		return 2;

	return 0;
}

int do_iprule(int argc, char **argv)
{
	if (argc < 1) {
		return iprule_list(0, NULL);
	} else if (matches(argv[0], "list") == 0 ||
		   matches(argv[0], "lst") == 0 ||
		   matches(argv[0], "show") == 0) {
		return iprule_list(argc-1, argv+1);
	} else if (matches(argv[0], "add") == 0) {
		return iprule_modify(RTM_NEWRULE, argc-1, argv+1);
	} else if (matches(argv[0], "delete") == 0) {
		return iprule_modify(RTM_DELRULE, argc-1, argv+1);
	}
	usage();
}

