/*
 * ifaconf.c		"ip address".
 *
 *		This program is free software; you can redistribute it and/or
 *		modify it under the terms of the GNU General Public License
 *		as published by the Free Software Foundation; either version
 *		2 of the License, or (at your option) any later version.
 *
 * Authors:	Alexey Kuznetsov, <kuznet@ms2.inr.ac.ru>
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <syslog.h>
#include <fcntl.h>
#include <linux/netdevice.h>
#include <net/if_arp.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <string.h>

#include "utils.h"

char filter_dev[16];
int  filter_family;

static void usage(void)
{
	fprintf(stderr, "Usage: ip addr [ add | del ] IFADDR dev STRING\n");
	fprintf(stderr, "       ip addr show [ dev STRING ] [ ipv4 | ipv6 | link | all ] [txqueuelen]\n");
	fprintf(stderr, "IFADDR := PREFIX [ local ADDR ]\n");
	fprintf(stderr, "          [ broadcast ADDR ] [ anycast ADDR ]\n");
	fprintf(stderr, "          [ label STRING ] [ scope SCOPE ]\n");
	fprintf(stderr, "SCOPE := [ host | link | global | NUMBER ]\n");
	exit(-1);
}

void get_arphrd_name(int type, char *buf)
{
#define __PF(f) { ARPHRD_##f, #f },
static struct {
	int type;
	char *name;
} arphrd_names[] = {
__PF(NETROM)
__PF(ETHER)
__PF(EETHER)
__PF(AX25)
__PF(PRONET)
__PF(CHAOS)
__PF(IEEE802)
__PF(ARCNET)
__PF(APPLETLK)
__PF(DLCI)
__PF(METRICOM)
__PF(SLIP)
__PF(CSLIP)
__PF(SLIP6)
__PF(CSLIP6)
__PF(RSRVD)
__PF(ADAPT)
__PF(ROSE)
__PF(X25)
__PF(PPP)
__PF(TUNNEL)
__PF(TUNNEL6)
__PF(FRAD)
__PF(SKIP)
__PF(LOOPBACK)
__PF(LOCALTLK)
__PF(FDDI)
__PF(BIF)
__PF(SIT)
__PF(IPDDP)
__PF(IPGRE)
__PF(PIMREG)
__PF(HIPPI)
__PF(ASH)
};
#undef __PF

	int i;
        for (i=0; i<sizeof(arphrd_names)/sizeof(arphrd_names[0]); i++) {
		if (arphrd_names[i].type == type) {
			strcpy(buf, arphrd_names[i].name);
			return;
		}
	}
	sprintf(buf, "[%d]", type);
}

void print_lla(FILE *fp, int len, unsigned char *addr)
{
	int i;
	for (i=0; i<len; i++) {
		if (i==0)
			fprintf(fp, "%02x", addr[i]);
		else
			fprintf(fp, ":%02x", addr[i]);
	}
}

void print_link_flags(FILE *fp, unsigned flags)
{
	fprintf(fp, "<");
	flags &= ~IFF_RUNNING;
#define _PF(f) if (flags&IFF_##f) { \
                  flags &= ~IFF_##f ; \
                  fprintf(fp, #f "%s", flags ? "," : ""); }
	_PF(LOOPBACK);
	_PF(BROADCAST);
	_PF(POINTOPOINT);
	_PF(MULTICAST);
	_PF(NOARP);
	_PF(ALLMULTI);
	_PF(PROMISC);
	_PF(MASTER);
	_PF(SLAVE);
	_PF(DEBUG);
	_PF(UP);
#undef _PF
        if (flags)
		fprintf(fp, "%x", flags);
	fprintf(fp, "> ");
}


int print_linkinfo(struct sockaddr_nl *who, struct nlmsghdr *n, void *arg)
{
	FILE *fp = (FILE*)arg;
	struct ifinfomsg *ifi = NLMSG_DATA(n);
	struct rtattr * tb[IFLA_MAX+1];
	int len = n->nlmsg_len;

	if (n->nlmsg_type != RTM_NEWLINK && n->nlmsg_type != RTM_DELLINK)
		return 0;

	len -= NLMSG_LENGTH(sizeof(*ifi));
	if (len < 0)
		return -1;

	memset(tb, 0, sizeof(tb));
	parse_rtattr(tb, IFLA_MAX, IFLA_RTA(ifi), len);
	if (tb[IFLA_IFNAME] == NULL)
		return -1;

	if (filter_dev[0] && strcmp(filter_dev, RTA_DATA(tb[IFLA_IFNAME])))
		return -1;

	if (n->nlmsg_type == RTM_DELLINK)
		fprintf(fp, "Deleted ");
		
	fprintf(fp, "%d: %s", ifi->ifi_index,
		tb[IFLA_IFNAME] ? (char*)RTA_DATA(tb[IFLA_IFNAME]) : "<nil>");

	if (tb[IFLA_LINK]) {
		int iflink = *(int*)RTA_DATA(tb[IFLA_LINK]);
		if (iflink == 0)
			fprintf(fp, "@NONE: ");
		else
			fprintf(fp, "@%d: ", iflink);
	} else {
		fprintf(fp, ": ");
	}
	print_link_flags(fp, ifi->ifi_flags);

	if (tb[IFLA_MTU])
		fprintf(fp, "mtu %u ", *(int*)RTA_DATA(tb[IFLA_MTU]));
	if (tb[IFLA_QDISC])
		fprintf(fp, "qdisc %s", (char*)RTA_DATA(tb[IFLA_QDISC]));
	fprintf(fp, "\n");
	if (1) {
		char type_buf[32];
		get_arphrd_name(ifi->ifi_type, type_buf);
		fprintf(fp, "    link/%s ", type_buf);
	}
	if (tb[IFLA_ADDRESS]) {
		print_lla(fp, RTA_PAYLOAD(tb[IFLA_ADDRESS]),
			  RTA_DATA(tb[IFLA_ADDRESS]));
	}
	if (tb[IFLA_BROADCAST]) {
		fprintf(fp, " brd ");
		print_lla(fp, RTA_PAYLOAD(tb[IFLA_BROADCAST]),
			  RTA_DATA(tb[IFLA_BROADCAST]));
	}
	fprintf(fp, "\n");
	if (tb[IFLA_STATS] && show_stats) {
		struct net_device_stats slocal;
		struct net_device_stats *s = RTA_DATA(tb[IFLA_STATS]);
		if (((unsigned long)s) & (sizeof(unsigned long)-1)) {
			memcpy(&slocal, s, sizeof(slocal));
			s = &slocal;
		}
		fprintf(fp, "    RX: bytes  packets  errors  dropped overrun mcast   \n");
		fprintf(fp, "    %-10ld %-8ld %-7ld %-7ld %-7ld %-7ld\n",
			s->rx_bytes, s->rx_packets, s->rx_errors,
			s->rx_dropped, s->rx_over_errors,
			s->multicast
			);
		if (show_stats > 1) {
			fprintf(fp, "    RX err: length  crc     frame   fifo    missed\n");
			fprintf(fp, "            %-7ld %-7ld %-7ld %-7ld %-7ld\n",
				s->rx_length_errors,
				s->rx_crc_errors,
				s->rx_frame_errors,
				s->rx_fifo_errors,
				s->rx_missed_errors
				);
		}
		fprintf(fp, "    TX: bytes  packets  errors  dropped carrier collsns \n");
		fprintf(fp, "    %-10ld %-8ld %-7ld %-7ld %-7ld %-7ld\n",
			s->tx_bytes, s->tx_packets, s->tx_errors,
			s->tx_dropped, s->tx_carrier_errors, s->collisions);
		if (show_stats > 1) {
			fprintf(fp, "    TX err: aborted fifo    window  heartbeat\n");
			fprintf(fp, "            %-7ld %-7ld %-7ld %-7ld\n",
				s->tx_aborted_errors,
				s->tx_fifo_errors,
				s->tx_window_errors,
				s->tx_heartbeat_errors
				);
		}
	}
	fflush(fp);
	return 0;
}

int print_addrinfo(struct sockaddr_nl *who, struct nlmsghdr *n, void *arg)
{
	FILE *fp = (FILE*)arg;
	struct ifaddrmsg *ifa = NLMSG_DATA(n);
	struct rtattr *rta = IFA_RTA(ifa);
	int len = n->nlmsg_len;
	struct rtattr * rta_tb[IFA_MAX+1];
	char abuf[256];

	if (n->nlmsg_type != RTM_NEWADDR && n->nlmsg_type != RTM_DELADDR)
		return 0;
	len -= NLMSG_LENGTH(sizeof(*ifa));
	if (len < 0)
		return -1;

	if (n->nlmsg_type == RTM_DELADDR)
		fprintf(fp, "Deleted ");

	memset(rta_tb, 0, sizeof(rta_tb));
	while (RTA_OK(rta, len)) {
		if (rta->rta_type <= IFA_MAX)
			rta_tb[rta->rta_type] = rta;
		rta = RTA_NEXT(rta, len);
	}
	if (len)
		fprintf(stderr, "!!!Deficit %d\n", len);

	if (ifa->ifa_family == AF_INET)
		fprintf(fp, "    inet ");
	else if (ifa->ifa_family == AF_INET6)
		fprintf(fp, "    inet6 ");
	else
		fprintf(fp, "    family %d ", ifa->ifa_family);

	if (rta_tb[IFA_ADDRESS] &&
	    inet_ntop(ifa->ifa_family, RTA_DATA(rta_tb[IFA_ADDRESS]),
		      abuf, sizeof(abuf))) {
		fprintf(fp, "%s/%d ", abuf, ifa->ifa_prefixlen);
	} else if (rta_tb[IFA_ADDRESS]) {
		int i;
		for (i=0; i<16; i++)
			printf("%02x" , ((__u8*)RTA_DATA(rta_tb[IFA_ADDRESS]))[i]);
		fprintf(fp, "???/%d ", ifa->ifa_prefixlen);
	}
	if (rta_tb[IFA_LOCAL]) {
		if (rta_tb[IFA_ADDRESS] == NULL ||
		    memcmp(RTA_DATA(rta_tb[IFA_ADDRESS]), RTA_DATA(rta_tb[IFA_LOCAL]), 4)) {
			if (inet_ntop(ifa->ifa_family, RTA_DATA(rta_tb[IFA_LOCAL]),
				      abuf, sizeof(abuf)))
				fprintf(fp, "local %s ", abuf);
			else
				fprintf(fp, "local ??? ");
		}
	}
	if (rta_tb[IFA_BROADCAST]) {
		if (inet_ntop(ifa->ifa_family, RTA_DATA(rta_tb[IFA_BROADCAST]),
			      abuf, sizeof(abuf)))
			fprintf(fp, "brd %s ", abuf);
		else
			fprintf(fp, "brd ??? ");
	}
	if (rta_tb[IFA_ANYCAST]) {
		if (inet_ntop(ifa->ifa_family, RTA_DATA(rta_tb[IFA_ANYCAST]),
			      abuf, sizeof(abuf)))
			fprintf(fp, "any %s ", abuf);
		else
			fprintf(fp, "any ??? ");
	}
	switch (ifa->ifa_scope) {
	case RT_SCOPE_HOST:
		fprintf(fp, "scope host ");
		break;
	case RT_SCOPE_LINK:
		fprintf(fp, "scope link ");
		break;
	case RT_SCOPE_SITE:
		fprintf(fp, "scope site ");
		break;
	case RT_SCOPE_UNIVERSE:
		fprintf(fp, "scope global ");
		break;
	default:
		fprintf(fp, "scope %d ", ifa->ifa_scope);
		break;
	}
	if (ifa->ifa_flags&IFA_F_SECONDARY) {
		ifa->ifa_flags &= ~IFA_F_SECONDARY;
		fprintf(fp, "secondary ");
	}
	if (ifa->ifa_flags&IFA_F_TENTATIVE) {
		ifa->ifa_flags &= ~IFA_F_TENTATIVE;
		fprintf(fp, "tentative ");
	}
	if (ifa->ifa_flags&IFA_F_DEPRECATED) {
		ifa->ifa_flags &= ~IFA_F_DEPRECATED;
		fprintf(fp, "deprecated ");
	}
	if (!(ifa->ifa_flags&IFA_F_PERMANENT)) {
		fprintf(fp, "dynamic ");
	} else
		ifa->ifa_flags &= ~IFA_F_PERMANENT;
	if (ifa->ifa_flags)
		fprintf(fp, "flags %02x ", ifa->ifa_flags);
	if (rta_tb[IFA_LABEL])
		fprintf(fp, "%s", (char*)RTA_DATA(rta_tb[IFA_LABEL]));
	fprintf(fp, "\n");
	if (rta_tb[IFA_CACHEINFO]) {
		struct ifa_cacheinfo *ci = RTA_DATA(rta_tb[IFA_CACHEINFO]);
		char buf[128];
		if (ci->ifa_valid == 0xFFFFFFFFU)
			sprintf(buf, "valid_lft forever");
		else
			sprintf(buf, "valid_lft %dsec", ci->ifa_valid);
		if (ci->ifa_prefered == 0xFFFFFFFFU)
			sprintf(buf+strlen(buf), " prefered_lft forever");
		else
			sprintf(buf+strlen(buf), " prefered_lft %dsec", ci->ifa_prefered);
		fprintf(fp, "       %s\n", buf);
	}
	fflush(fp);
	return 0;
}


struct nlmsg_list
{
	struct nlmsg_list *next;
	struct nlmsghdr	  h;
};

int print_selected_addrinfo(int ifindex, struct nlmsg_list *ainfo, FILE *fp)
{
	for ( ;ainfo ;  ainfo = ainfo->next) {
		struct nlmsghdr *n = &ainfo->h;
		struct ifaddrmsg *ifa = NLMSG_DATA(n);

		if (n->nlmsg_type != RTM_NEWADDR)
			continue;

		if (n->nlmsg_len < NLMSG_LENGTH(sizeof(ifa)))
			return -1;

		if (ifa->ifa_index != ifindex || 
		    (filter_family && filter_family != ifa->ifa_family))
			continue;

		print_addrinfo(NULL, n, fp);
	}
	return 0;
}

void print_queuelen(int ifindex)
{
	struct ifreq ifr = {0};
	int s;

	s = socket(AF_INET, SOCK_STREAM, 0);
	if (s < 0) return;

	ifr.ifr_ifindex = ifindex;
	if (ioctl(s, SIOCGIFNAME, &ifr) < 0) {
		perror("SIOCGIFNAME"); return; 
	}
	if (ioctl(s, SIOCGIFTXQLEN, &ifr) < 0) { 
		perror("SIOCGIFXQLEN");
		return;
	}
	close(s); 

	printf("    txqueuelen %d\n", ifr.ifr_qlen);
}

int store_nlmsg(struct sockaddr_nl *who, struct nlmsghdr *n, void *arg)
{
	struct nlmsg_list **linfo = (struct nlmsg_list**)arg;
	struct nlmsg_list *h;
	struct nlmsg_list **lp;

	h = malloc(n->nlmsg_len+sizeof(void*));
	if (h == NULL)
		return -1;

	memcpy(&h->h, n, n->nlmsg_len);
	h->next = NULL;

	for (lp = linfo; *lp; lp = &(*lp)->next) /* NOTHING */;
	*lp = h;
	return 0;
}

int ipaddr_list(int argc, char **argv)
{
	struct nlmsg_list *linfo = NULL;
	struct nlmsg_list *ainfo = NULL;
	struct nlmsg_list *l;
	struct rtnl_handle rth;
	int showqueue = 0 ; 

	while (argc > 0) {
		if (strcmp(*argv, "dev") == 0) {
			NEXT_ARG();
			if (filter_dev[0])
				usage();
			strcpy(filter_dev, *argv);
		} else if (strcmp(*argv, "all") == 0) {
			filter_family = AF_UNSPEC;
		} else if (strcmp(*argv, "ipv4") == 0) {
			filter_family = AF_INET;
		} else if (strcmp(*argv, "ipv6") == 0) {
			filter_family = AF_INET6;
		} else if (strcmp(*argv, "link") == 0) {
			filter_family = AF_PACKET;
		} else if (strcmp(*argv, "txqueuelen") == 0) {
			showqueue++; 		
		} else {
			if (filter_dev[0])
				usage();
			strcpy(filter_dev, *argv);
		}
		argv++; argc--;
	}

	if (rtnl_open(&rth, 0) < 0) {
		fprintf(stderr, "cannot open rtnetlink\n");
		exit(1);
	}

	if (rtnl_wilddump_request(&rth, preferred_family, RTM_GETLINK) < 0) {
		perror("cannot send dump request");
		exit(1);
	}

	if (rtnl_dump_filter(&rth, store_nlmsg, &linfo, NULL, NULL) < 0) {
		fprintf(stderr, "dump terminated\n");
		exit(1);
	}

	if (filter_family != AF_PACKET) {
		if (rtnl_wilddump_request(&rth, filter_family, RTM_GETADDR) < 0) {
			perror("cannot send dump request");
			exit(1);
		}

		if (rtnl_dump_filter(&rth, store_nlmsg, &ainfo, NULL, NULL) < 0) {
			fprintf(stderr, "dump terminated\n");
			exit(1);
		}
	}

	for (l=linfo; l; l = l->next) {
		if (print_linkinfo(NULL, &l->h, stdout) == 0) {
			struct ifinfomsg *ifi = NLMSG_DATA(&l->h);

			if (showqueue)
				print_queuelen(ifi->ifi_index); 
			if (filter_family != AF_PACKET)
				print_selected_addrinfo(ifi->ifi_index, ainfo, stdout);
		}
		fflush(stdout);
	}

	exit(0);
}


int ipaddr_modify(int cmd, int argc, char **argv)
{
	struct rtnl_handle rth;
	struct {
		struct nlmsghdr 	n;
		struct ifaddrmsg 	ifa;
		char   			buf[256];
	} req;
	char  d[16];
	inet_prefix dst;
	int local_len = 0;
	int brd_len = 0;
	int any_len = 0;
	int dst_len = 0;

	memset(&req, 0, sizeof(req));
	d[0] = 0;

	req.n.nlmsg_len = NLMSG_LENGTH(sizeof(struct ifaddrmsg));
	req.n.nlmsg_flags = NLM_F_REQUEST;
	req.n.nlmsg_type = cmd;
	req.ifa.ifa_family = preferred_family;

	while (argc > 0) {
		if (strcmp(*argv, "local") == 0) {
			inet_prefix addr;
			NEXT_ARG();
			if (local_len)
				usage();
			get_addr(&addr, *argv, req.ifa.ifa_family);
			if (req.ifa.ifa_family == AF_UNSPEC)
				req.ifa.ifa_family = addr.family;
			addattr_l(&req.n, sizeof(req), IFA_LOCAL, &addr.data, addr.bytelen);
			local_len = addr.bytelen;
		} else if (strcmp(*argv, "broadcast") == 0) {
			inet_prefix addr;
			NEXT_ARG();
			if (brd_len)
				usage();
			get_addr(&addr, *argv, req.ifa.ifa_family);
			if (req.ifa.ifa_family == AF_UNSPEC)
				req.ifa.ifa_family = addr.family;
			addattr_l(&req.n, sizeof(req), IFA_BROADCAST, &addr.data, addr.bytelen);
			brd_len = addr.bytelen;
		} else if (strcmp(*argv, "anycast") == 0) {
			inet_prefix addr;
			NEXT_ARG();
			if (any_len)
				usage();
			get_addr(&addr, *argv, req.ifa.ifa_family);
			if (req.ifa.ifa_family == AF_UNSPEC)
				req.ifa.ifa_family = addr.family;
			addattr_l(&req.n, sizeof(req), IFA_ANYCAST, &addr.data, addr.bytelen);
			any_len = addr.bytelen;
		} else if (strcmp(*argv, "scope") == 0) {
			NEXT_ARG();
			if (strcmp(*argv, "host") == 0)
				req.ifa.ifa_scope = RT_SCOPE_HOST;
			else if (strcmp(*argv, "link") == 0)
				req.ifa.ifa_scope = RT_SCOPE_LINK;
			else if (strcmp(*argv, "site") == 0)
				req.ifa.ifa_scope = RT_SCOPE_SITE;
			else if (strcmp(*argv, "global") == 0)
				req.ifa.ifa_scope = RT_SCOPE_UNIVERSE;
			else if (strcmp(*argv, "universe") == 0)
				req.ifa.ifa_scope = RT_SCOPE_UNIVERSE;
			else
				req.ifa.ifa_scope = atoi(*argv);
		} else if (strcmp(*argv, "dev") == 0) {
			NEXT_ARG();
			strcpy(d, *argv);
		} else if (strcmp(*argv, "label") == 0) {
			NEXT_ARG();
			addattr_l(&req.n, sizeof(req), IFA_LABEL, *argv, strlen(*argv));
		} else {
			if (dst_len)
				usage();
			get_prefix(&dst, *argv, req.ifa.ifa_family);
			dst_len = dst.bytelen;
			if (req.ifa.ifa_family == AF_UNSPEC)
				req.ifa.ifa_family = dst.family;
			addattr_l(&req.n, sizeof(req), IFA_ADDRESS, &dst.data, dst.bytelen);
			req.ifa.ifa_prefixlen = dst.bitlen;
		}
		argc--; argv++;
	}
	if (d[0] == 0)
		usage();

	if (local_len == 0 && dst_len)
		addattr_l(&req.n, sizeof(req), IFA_LOCAL, &dst.data, dst.bytelen);

	if (rtnl_open(&rth, 0) < 0) {
		fprintf(stderr, "cannot open rtnetlink\n");
		exit(1);
	}

	ll_init_map(&rth);

	if ((req.ifa.ifa_index = ll_name_to_index(d)) == 0)
		usage();

	if (rtnl_talk(&rth, &req.n, 0, 0, NULL, NULL, NULL) < 0)
		exit(2);

	exit(0);
}

int do_ipaddr(int argc, char **argv)
{
	if (argc < 1)
		return ipaddr_list(0, NULL);
	if (matches(*argv, "add") == 0)
		return ipaddr_modify(RTM_NEWADDR, argc-1, argv+1);
	if (matches(*argv, "delete") == 0)
		return ipaddr_modify(RTM_DELADDR, argc-1, argv+1);
	if (matches(*argv, "list") == 0 || matches(*argv, "show") == 0
	    || matches(*argv, "lst") == 0)
		return ipaddr_list(argc-1, argv+1);
	usage();
}

