/* Bezerk
 * Copyright (C) 1998 Tony Gale.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <gtk/gtk.h>
#include <glib.h>
#include <string.h>

#include "gtkfontsel.h"
 
#include "dialogs.h"
#include "servers.h"
#include "reply.h"
#include "irc.h"
#include "prefs.h"
#include "servers.h"
#include "persona.h"
#include "util.h"
#include "ch_utils.h"
#include "send.h"
#include "bezerk.h"
#include "version.h"
#include "debug.h"

#include "book_open.xpm"
#include "book_closed.xpm"

extern GSList *aliases;

extern GdkFont *font_bold;
extern GdkFont *font_normal;
extern GdkFont *font_italic;
extern GdkFont *font_bold_italic;

void choice_destroy(GtkWidget *widget, ChoiceData *choice_data)
{
  bs_function_enter();

  if (!choice_data) {
    bs_function_leave();
    return;
  }

  if (choice_data->negative_func) {
    choice_data->negative_func(choice_data->user_data);
  }

  g_free(choice_data);

  bs_function_leave();
  return;
}

void choice_positive(GtkWidget *widget, ChoiceData *choice_data)
{

  bs_function_enter();

  if (choice_data->positive_func) {
    choice_data->positive_func(choice_data->user_data);
  }

  choice_data->negative_func = NULL;
  gtk_widget_destroy(choice_data->dialog);

  bs_function_leave();
  return;
}

void choice_negative(GtkWidget *widget, ChoiceData *choice_data)
{

  bs_function_enter();

  if (choice_data->negative_func) {
    choice_data->negative_func(choice_data->user_data);
    choice_data->negative_func = NULL;
  }

  gtk_widget_destroy(choice_data->dialog);
  
  bs_function_leave();
  return;
}

void choice_dialog(void *user_data,
		   char *title,
		   char *question,
		   char *positive,
		   char *negative,
		   void (*positive_func)(void *data),
		   void (*negative_func)(void *data))
{
  ChoiceData *choice_data;

  GtkWidget *label;
  GtkWidget *button;

  bs_function_enter();

  choice_data = g_malloc(sizeof(ChoiceData));
  choice_data->dialog = gtk_dialog_new();
  choice_data->user_data = user_data;
  choice_data->positive_func = positive_func;
  choice_data->negative_func = negative_func;

  if (title) {
    gtk_window_set_title (GTK_WINDOW (choice_data->dialog), title);
  }
  gtk_window_position (GTK_WINDOW (choice_data->dialog), GTK_WIN_POS_CENTER);
  gtk_signal_connect (GTK_OBJECT (choice_data->dialog), "destroy",
		      GTK_SIGNAL_FUNC (choice_destroy), choice_data);
  
  if (question) {
    label = gtk_label_new(question);
    gtk_label_set_justify( GTK_LABEL(label), GTK_JUSTIFY_LEFT);
    gtk_container_border_width (GTK_CONTAINER (GTK_DIALOG(choice_data->dialog)->vbox), 10);
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG(choice_data->dialog)->vbox), label, TRUE, FALSE, 10);
    gtk_widget_show (label);
  }

  if (positive) {
    button = gtk_button_new_with_label (positive);
  } else {
    button = gtk_button_new_with_label ("Ok");
  }
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG(choice_data->dialog)->action_area), button, TRUE, FALSE, 0);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (choice_positive),
		      choice_data);
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_widget_grab_default (button);
  gtk_widget_show (button);

  if (negative) {
    button = gtk_button_new_with_label (negative);
  } else {
    button = gtk_button_new_with_label ("Cancel");
  }
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG(choice_data->dialog)->action_area), button, TRUE, FALSE, 0);
  gtk_signal_connect(GTK_OBJECT (button), "clicked",
		     GTK_SIGNAL_FUNC (choice_negative),
		     choice_data);
  gtk_widget_show (button);

  gtk_widget_show (choice_data->dialog);

  bs_function_leave();
  return;
}

/* --------------------------------------------------------------------- */

void prefs_font_fixed (GtkWidget *widget, FontPrefsData *font_prefs)
{
  bs_function_enter();

  font_prefs->type[0] = 'f';

  bs_function_leave();
  return;
}

void prefs_font_variable (GtkWidget *widget, FontPrefsData *font_prefs)
{
  bs_function_enter();

  font_prefs->type[0] = 'v';

  bs_function_leave();
  return;
}

void prefs_font_8 (GtkWidget *widget, FontPrefsData *font_prefs)
{
  bs_function_enter();

  g_free(font_prefs->size);
  font_prefs->size = g_strdup("8");

  bs_function_leave();
  return;
}

void prefs_font_10 (GtkWidget *widget, FontPrefsData *font_prefs)
{
  bs_function_enter();

  g_free(font_prefs->size);
  font_prefs->size = g_strdup("10");

  bs_function_leave();
  return;
}

void prefs_font_12 (GtkWidget *widget, FontPrefsData *font_prefs)
{
  bs_function_enter();

  g_free(font_prefs->size);
  font_prefs->size = g_strdup("12");

  bs_function_leave();
  return;
}

void prefs_font_14 (GtkWidget *widget, FontPrefsData *font_prefs)
{
  bs_function_enter();

  g_free(font_prefs->size);
  font_prefs->size = g_strdup("14");

  bs_function_leave();
  return;
}

void prefs_font_16 (GtkWidget *widget, FontPrefsData *font_prefs)
{
  bs_function_enter();

  g_free(font_prefs->size);
  font_prefs->size = g_strdup("16");

  bs_function_leave();
  return;
}

void prefs_font_change_ok (GtkWidget *widget, FontPrefsData *font_prefs)
{
  char buff[BUFFLEN];

  bs_function_enter();

  switch (font_prefs->changing_font)
    {
    case FONT_NORMAL:
      if (font_prefs->normal_font) {
	g_free(font_prefs->normal_font);
      }
      font_prefs->normal_font = gtk_font_selection_get_font_name(GTK_FONT_SELECTION(font_prefs->font_sel));
      g_snprintf(buff, BUFFLEN, "Normal:\n%s", font_prefs->normal_font);
      gtk_label_set ( GTK_LABEL(font_prefs->adv_normal_label), buff);
      break;
    case FONT_BOLD:
      if (font_prefs->bold_font) {
	g_free(font_prefs->bold_font);
      }
      font_prefs->bold_font = gtk_font_selection_get_font_name(GTK_FONT_SELECTION(font_prefs->font_sel));
      g_snprintf(buff, BUFFLEN, "Bold:\n%s", font_prefs->bold_font);
      gtk_label_set ( GTK_LABEL(font_prefs->adv_bold_label), buff);
      break;
    case FONT_ITALIC:
      if (font_prefs->italic_font) {
	g_free(font_prefs->italic_font);
      }
      font_prefs->italic_font = gtk_font_selection_get_font_name(GTK_FONT_SELECTION(font_prefs->font_sel));
      g_snprintf(buff, BUFFLEN, "Italic:\n%s", font_prefs->italic_font);
      gtk_label_set ( GTK_LABEL(font_prefs->adv_italic_label), buff);
      break;
    case FONT_BOLD_ITALIC:
      if (font_prefs->bold_italic_font) {
	g_free(font_prefs->bold_italic_font);
      }
      font_prefs->bold_italic_font = gtk_font_selection_get_font_name(GTK_FONT_SELECTION(font_prefs->font_sel));
      g_snprintf(buff, BUFFLEN, "Bold Italic:\n%s", font_prefs->bold_italic_font);
      gtk_label_set ( GTK_LABEL(font_prefs->adv_bold_italic_label), buff);
      break;
    }

  gtk_grab_remove(font_prefs->font_sel);
  gtk_widget_destroy( GTK_WIDGET(font_prefs->font_sel) );

  bs_function_leave();
  return;
}
  
void prefs_font_adv_change_normal (GtkWidget *widget, FontPrefsData *font_prefs)
{

  bs_function_enter();

  font_prefs->changing_font = FONT_NORMAL;
  font_prefs->font_sel = gtk_font_selection_new(NULL);
  gtk_font_selection_set_font_name(GTK_FONT_SELECTION(font_prefs->font_sel), font_prefs->normal_font);
  gtk_signal_connect (GTK_OBJECT (GTK_FONT_SELECTION(font_prefs->font_sel)->ok_button),
                      "clicked",
                      GTK_SIGNAL_FUNC(prefs_font_change_ok),
                      font_prefs);
  gtk_signal_connect_object (GTK_OBJECT (GTK_FONT_SELECTION(font_prefs->font_sel)->cancel_button),
			     "clicked",
			     GTK_SIGNAL_FUNC(gtk_widget_destroy),
			     GTK_OBJECT(font_prefs->font_sel) );
  gtk_widget_show(font_prefs->font_sel);

  gtk_grab_add(font_prefs->font_sel);

  bs_function_leave();
  return;
}

void prefs_font_adv_change_bold (GtkWidget *widget, FontPrefsData *font_prefs)
{

  bs_function_enter();

  font_prefs->changing_font = FONT_BOLD;
  font_prefs->font_sel = gtk_font_selection_new(NULL);
  gtk_font_selection_set_font_name(GTK_FONT_SELECTION(font_prefs->font_sel), font_prefs->bold_font);
  gtk_signal_connect (GTK_OBJECT (GTK_FONT_SELECTION(font_prefs->font_sel)->ok_button),
                      "clicked",
                      GTK_SIGNAL_FUNC(prefs_font_change_ok),
                      font_prefs);
  gtk_signal_connect_object (GTK_OBJECT (GTK_FONT_SELECTION(font_prefs->font_sel)->cancel_button),
			     "clicked",
			     GTK_SIGNAL_FUNC(gtk_widget_destroy),
			     GTK_OBJECT(font_prefs->font_sel) );
  gtk_widget_show(font_prefs->font_sel);

  gtk_grab_add(font_prefs->font_sel);

  bs_function_leave();
  return;
}

void prefs_font_adv_change_italic (GtkWidget *widget, FontPrefsData *font_prefs)
{

  bs_function_enter();

  font_prefs->changing_font = FONT_ITALIC;
  font_prefs->font_sel = gtk_font_selection_new(NULL);
  gtk_font_selection_set_font_name(GTK_FONT_SELECTION(font_prefs->font_sel), font_prefs->italic_font);
  gtk_signal_connect (GTK_OBJECT (GTK_FONT_SELECTION(font_prefs->font_sel)->ok_button),
                      "clicked",
                      GTK_SIGNAL_FUNC(prefs_font_change_ok),
                      font_prefs);
  gtk_signal_connect_object (GTK_OBJECT (GTK_FONT_SELECTION(font_prefs->font_sel)->cancel_button),
			     "clicked",
			     GTK_SIGNAL_FUNC(gtk_widget_destroy),
			     GTK_OBJECT(font_prefs->font_sel) );
  gtk_widget_show(font_prefs->font_sel);

  gtk_grab_add(font_prefs->font_sel);

  bs_function_leave();
  return;
}

void prefs_font_adv_change_bold_italic (GtkWidget *widget, FontPrefsData *font_prefs)
{

  bs_function_enter();

  font_prefs->changing_font = FONT_BOLD_ITALIC;
  font_prefs->font_sel = gtk_font_selection_new(NULL);
  gtk_font_selection_set_font_name(GTK_FONT_SELECTION(font_prefs->font_sel), font_prefs->bold_italic_font);
  gtk_signal_connect (GTK_OBJECT (GTK_FONT_SELECTION(font_prefs->font_sel)->ok_button),
                      "clicked",
                      GTK_SIGNAL_FUNC(prefs_font_change_ok),
                      font_prefs);
  gtk_signal_connect_object (GTK_OBJECT (GTK_FONT_SELECTION(font_prefs->font_sel)->cancel_button),
			     "clicked",
			     GTK_SIGNAL_FUNC(gtk_widget_destroy),
			     GTK_OBJECT(font_prefs->font_sel) );
  gtk_widget_show(font_prefs->font_sel);

  gtk_grab_add(font_prefs->font_sel);

  bs_function_leave();
  return;
}

void prefs_font_adv_toggle (GtkToggleButton *button, FontPrefsData *font_prefs)
{
  bs_function_enter();

  if (button->active) {
    font_prefs->adv[0] = '1';
    gtk_widget_set_sensitive(font_prefs->adv_frame, TRUE);
    gtk_widget_set_sensitive(font_prefs->std_frame, FALSE);
  } else {
    font_prefs->adv[0] = '0';
    gtk_widget_set_sensitive(font_prefs->adv_frame, FALSE);
    gtk_widget_set_sensitive(font_prefs->std_frame, TRUE);
  }

  bs_function_leave();
  return;
}

GtkWidget *prefs_fonts_page(void **pref_data)
{
  FontPrefsData *font_prefs;
  GtkWidget *vbox;
  GtkWidget *table;
  GtkWidget *menu;
  GtkWidget *menuitem;
  GtkWidget *label;
  GtkWidget *button;

  bs_function_enter();

  font_prefs = g_malloc(sizeof(FontPrefsData));
  /* These will get set by the prefs_fonts_reset function */
  font_prefs->adv = NULL;
  font_prefs->type = NULL;
  font_prefs->size = NULL;
  font_prefs->normal_font = NULL;
  font_prefs->bold_font = NULL;
  font_prefs->italic_font = NULL;
  font_prefs->bold_italic_font = NULL;
		    
  vbox = gtk_vbox_new(FALSE, 0);
  gtk_container_border_width (GTK_CONTAINER (vbox), 4);
  
  font_prefs->std_frame = gtk_frame_new ("Standard");
  gtk_container_border_width (GTK_CONTAINER (font_prefs->std_frame), 3);
  gtk_box_pack_start (GTK_BOX (vbox), font_prefs->std_frame, FALSE, FALSE, 3);
  gtk_widget_show (font_prefs->std_frame);

  table = gtk_table_new (2, 2, FALSE);
  gtk_container_add (GTK_CONTAINER(font_prefs->std_frame), table);
  gtk_widget_show (table);

  label = gtk_label_new ("Font:");
  gtk_table_attach_defaults (GTK_TABLE(table), label, 0, 1, 0, 1);
  gtk_widget_show (label);

  font_prefs->std_type_omenu = gtk_option_menu_new ();
  menu = gtk_menu_new ();

  menuitem = gtk_menu_item_new_with_label ("Fixed Width");
  gtk_signal_connect ( GTK_OBJECT (menuitem), "activate",
		       GTK_SIGNAL_FUNC (prefs_font_fixed),
		       font_prefs);
  gtk_menu_append (GTK_MENU (menu), menuitem);
  gtk_widget_show (menuitem);

  menuitem = gtk_menu_item_new_with_label ("Variable Width");
  gtk_signal_connect ( GTK_OBJECT (menuitem), "activate",
		       GTK_SIGNAL_FUNC (prefs_font_variable),
		       font_prefs);
  gtk_menu_append (GTK_MENU (menu), menuitem);
  gtk_widget_show (menuitem);

  gtk_option_menu_set_menu (GTK_OPTION_MENU (font_prefs->std_type_omenu), menu);
  gtk_widget_show (font_prefs->std_type_omenu);

  gtk_table_attach ( GTK_TABLE(table), font_prefs->std_type_omenu, 1, 2, 0, 1,
		     GTK_SHRINK|GTK_EXPAND,  GTK_SHRINK|GTK_EXPAND, 0, 2);

  label = gtk_label_new ("Size:");
  gtk_table_attach_defaults (GTK_TABLE(table), label, 0, 1, 1, 2);
  gtk_widget_show (label);

  font_prefs->std_size_omenu = gtk_option_menu_new ();
  menu = gtk_menu_new ();

  menuitem = gtk_menu_item_new_with_label ("8");
  gtk_signal_connect( GTK_OBJECT (menuitem), "activate",
		      GTK_SIGNAL_FUNC (prefs_font_8),
		      font_prefs);
  gtk_menu_append (GTK_MENU (menu), menuitem);
  gtk_widget_show (menuitem);

  menuitem = gtk_menu_item_new_with_label ("10");
  gtk_signal_connect( GTK_OBJECT (menuitem), "activate",
		      GTK_SIGNAL_FUNC (prefs_font_10),
		      font_prefs);
  gtk_menu_append (GTK_MENU (menu), menuitem);
  gtk_widget_show (menuitem);

  menuitem = gtk_menu_item_new_with_label ("12");
  gtk_signal_connect( GTK_OBJECT (menuitem), "activate",
		      GTK_SIGNAL_FUNC (prefs_font_12),
		      font_prefs);
  gtk_menu_append (GTK_MENU (menu), menuitem);
  gtk_widget_show (menuitem);

  menuitem = gtk_menu_item_new_with_label ("14");
  gtk_signal_connect( GTK_OBJECT (menuitem), "activate",
		      GTK_SIGNAL_FUNC (prefs_font_14),
		      font_prefs);
  gtk_menu_append (GTK_MENU (menu), menuitem);
  gtk_widget_show (menuitem);

  menuitem = gtk_menu_item_new_with_label ("16");
  gtk_signal_connect( GTK_OBJECT (menuitem), "activate",
		      GTK_SIGNAL_FUNC (prefs_font_16),
		      font_prefs);
  gtk_menu_append (GTK_MENU (menu), menuitem);
  gtk_widget_show (menuitem);

  gtk_option_menu_set_menu (GTK_OPTION_MENU (font_prefs->std_size_omenu), menu);
  gtk_widget_show (font_prefs->std_size_omenu);

  gtk_table_attach ( GTK_TABLE(table), font_prefs->std_size_omenu, 1, 2, 1, 2,
		     GTK_SHRINK|GTK_EXPAND,  GTK_SHRINK|GTK_EXPAND, 0, 2);
 
  font_prefs->adv_button = gtk_check_button_new_with_label ( "Advanced" );
  gtk_signal_connect (GTK_OBJECT(font_prefs->adv_button), "clicked",
		      GTK_SIGNAL_FUNC (prefs_font_adv_toggle),
		      font_prefs );
  gtk_box_pack_start (GTK_BOX (vbox), font_prefs->adv_button, FALSE, FALSE, 3);
  GTK_WIDGET_UNSET_FLAGS(font_prefs->adv_button, GTK_CAN_FOCUS);
  gtk_widget_show (font_prefs->adv_button);

  font_prefs->adv_frame = gtk_frame_new ("Advanced");
  gtk_container_border_width (GTK_CONTAINER (font_prefs->adv_frame), 3);
  gtk_box_pack_start (GTK_BOX (vbox), font_prefs->adv_frame, FALSE, FALSE, 3);
  gtk_widget_show (font_prefs->adv_frame);

  table = gtk_table_new (4, 2, FALSE);
  gtk_container_add (GTK_CONTAINER(font_prefs->adv_frame), table);
  gtk_widget_show (table);

  font_prefs->adv_normal_label = gtk_label_new ("Normal:");
  gtk_label_set_justify( GTK_LABEL(font_prefs->adv_normal_label) , GTK_JUSTIFY_LEFT);
  gtk_table_attach ( GTK_TABLE(table), font_prefs->adv_normal_label, 0, 1, 0, 1,
		     GTK_SHRINK|GTK_EXPAND,  GTK_SHRINK|GTK_EXPAND, 2, 6);
  gtk_widget_show (font_prefs->adv_normal_label);
  
  button = gtk_button_new_with_label("Change...");
  gtk_table_attach ( GTK_TABLE(table), button, 1, 2, 0, 1,
		     GTK_SHRINK|GTK_EXPAND,  GTK_SHRINK|GTK_EXPAND, 5, 5);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (prefs_font_adv_change_normal),
		      font_prefs);
  gtk_widget_show (button);
  
  font_prefs->adv_bold_label = gtk_label_new ("Bold:");
  gtk_label_set_justify( GTK_LABEL(font_prefs->adv_bold_label), GTK_JUSTIFY_LEFT);
  gtk_table_attach ( GTK_TABLE(table), font_prefs->adv_bold_label, 0, 1, 1, 2,
		     GTK_SHRINK|GTK_EXPAND,  GTK_SHRINK|GTK_EXPAND, 2, 6);
  gtk_widget_show (font_prefs->adv_bold_label);

  button = gtk_button_new_with_label("Change...");
  gtk_table_attach ( GTK_TABLE(table), button, 1, 2, 1, 2,
		     GTK_SHRINK|GTK_EXPAND,  GTK_SHRINK|GTK_EXPAND, 5, 5);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (prefs_font_adv_change_bold),
		      font_prefs);
  gtk_widget_show (button);

  font_prefs->adv_italic_label = gtk_label_new ("Italic:");
  gtk_label_set_justify( GTK_LABEL(font_prefs->adv_italic_label), GTK_JUSTIFY_LEFT);
  gtk_table_attach ( GTK_TABLE(table), font_prefs->adv_italic_label, 0, 1, 2, 3,
		     GTK_SHRINK|GTK_EXPAND,  GTK_SHRINK|GTK_EXPAND, 2, 6);
  gtk_widget_show (font_prefs->adv_italic_label);

  button = gtk_button_new_with_label("Change...");
  gtk_table_attach ( GTK_TABLE(table), button, 1, 2, 2, 3,
		     GTK_SHRINK|GTK_EXPAND,  GTK_SHRINK|GTK_EXPAND, 5, 5);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (prefs_font_adv_change_italic),
		      font_prefs);
  gtk_widget_show (button);

  font_prefs->adv_bold_italic_label = gtk_label_new ("Bold Italic:");
  gtk_label_set_justify( GTK_LABEL(font_prefs->adv_bold_italic_label), GTK_JUSTIFY_LEFT);
  gtk_table_attach ( GTK_TABLE(table), font_prefs->adv_bold_italic_label, 0, 1, 3, 4,
		     GTK_SHRINK|GTK_EXPAND,  GTK_SHRINK|GTK_EXPAND, 2, 6);
  gtk_widget_show (font_prefs->adv_bold_italic_label);

  button = gtk_button_new_with_label("Change...");
  gtk_table_attach ( GTK_TABLE(table), button, 1, 2, 3, 4,
		     GTK_SHRINK|GTK_EXPAND,  GTK_SHRINK|GTK_EXPAND, 5, 5);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (prefs_font_adv_change_bold_italic),
		      font_prefs);
  gtk_widget_show (button);

  gtk_widget_show (vbox);

  *pref_data = font_prefs;


  bs_function_leave();
  return(vbox);
}

gint prefs_fonts_save(void *pref_data)
{
  FontPrefsData *font_prefs = pref_data;
  char *normal, *bold, *italic, *bolditalic, *new_font;
  bs_function_enter();

  if ( *font_prefs->adv == '1' ) {
    preference_set("font.advanced", "1");

    preference_set("font.normal", font_prefs->normal_font);
    font_normal = gdk_font_load(font_prefs->normal_font);
    if (!font_normal) {
      new_font = preference_get_default("font.normal");
      font_normal = gdk_font_load( new_font );
      preference_set("font.normal", new_font);
    }

    preference_set("font.bold", font_prefs->bold_font);
    font_bold = gdk_font_load(font_prefs->bold_font);
    if (!font_bold) {
      new_font = preference_get_default("font.bold");
      font_bold = gdk_font_load( new_font );
      preference_set("font.bold", new_font);
    }

    preference_set("font.italic", font_prefs->italic_font);
    font_italic = gdk_font_load(font_prefs->italic_font);
    if (!font_italic) {
      new_font = preference_get_default("font.italic");
      font_italic = gdk_font_load( new_font );
      preference_set("font.italic", new_font);
    }

    preference_set("font.bolditalic", font_prefs->bold_italic_font);
    font_bold_italic = gdk_font_load(font_prefs->bold_italic_font);
    if (!font_normal) {
      new_font = preference_get_default("font.bolditalic");
      font_bold_italic = gdk_font_load( new_font );
      preference_set("font.bolditalic", new_font);
    }
  } else {
    preference_set("font.advanced", "0");
    preference_set("font.type", font_prefs->type);
    preference_set("font.size", font_prefs->size);

    if ( *font_prefs->type == 'f' ) {
      normal = preference_get("font.std_f_normal");
      bold = preference_get("font.std_f_bold");
      italic =  preference_get("font.std_f_italic");
      bolditalic = preference_get("font.std_f_bolditalic");
    } else {
      normal = preference_get("font.std_v_normal");
      bold = preference_get("font.std_v_bold");
      italic =  preference_get("font.std_v_italic");
      bolditalic = preference_get("font.std_v_bolditalic");
    }

    new_font = g_malloc(strlen(normal) + strlen(font_prefs->size) + 15);
    sprintf(new_font, "%s%s-*-*-*-*-*-*-*", normal, font_prefs->size);
    font_normal = gdk_font_load(new_font);
    if (!font_normal) {
      new_font = preference_get_default("font.normal");
      font_normal = gdk_font_load( new_font );
    }
    preference_set("font.normal", new_font);
    g_free(new_font);

    new_font = g_malloc(strlen(bold) + strlen(font_prefs->size) + 15);
    sprintf(new_font, "%s%s-*-*-*-*-*-*-*", bold, font_prefs->size);
    font_bold = gdk_font_load(new_font);
    if (!font_normal) {
      new_font = preference_get_default("font.bold");
      font_bold = gdk_font_load( new_font );
    }
    preference_set("font.bold", new_font);
    g_free(new_font);

    new_font = g_malloc(strlen(italic) + strlen(font_prefs->size) + 15);
    sprintf(new_font, "%s%s-*-*-*-*-*-*-*", italic, font_prefs->size);
    font_italic = gdk_font_load(new_font);
    if (!font_normal) {
      new_font = preference_get_default("font.italic");
      font_italic = gdk_font_load( new_font );
    }
    preference_set("font.italic", new_font);
    g_free(new_font);

    new_font = g_malloc(strlen(bolditalic) + strlen(font_prefs->size) + 15);
    sprintf(new_font, "%s%s-*-*-*-*-*-*-*", bolditalic, font_prefs->size);
    font_bold_italic = gdk_font_load(new_font);
    if (!font_normal) {
      new_font = preference_get_default("font.bolditalic");
      font_bold_italic = gdk_font_load( new_font );
    }
    preference_set("font.bolditalic", new_font);
    g_free(new_font);
  }

  return(TRUE);
}

void prefs_fonts_reset(void *pref_data)
{
  FontPrefsData *font_prefs = pref_data;
  char buff[BUFFLEN];

  bs_function_enter();

  if ( font_prefs->adv ) {
    g_free(font_prefs->adv);
  }
  if ( font_prefs->type ) {
    g_free(font_prefs->type);
  }
  if ( font_prefs->size ) {
    g_free(font_prefs->size);
  }
  if ( font_prefs->normal_font ) {
    g_free(font_prefs->normal_font);
  }
  if ( font_prefs->bold_font ) {
    g_free(font_prefs->bold_font);
  }
  if ( font_prefs->italic_font ) {
    g_free(font_prefs->italic_font);
  }
  if ( font_prefs->bold_italic_font ) {
    g_free(font_prefs->bold_italic_font);
  }

  font_prefs->adv = g_strdup( preference_get("font.advanced") );
  font_prefs->type = g_strdup( preference_get("font.type") );
  font_prefs->size = g_strdup( preference_get("font.size") );
  font_prefs->normal_font = g_strdup( preference_get("font.normal") );
  font_prefs->bold_font = g_strdup( preference_get("font.bold") );
  font_prefs->italic_font = g_strdup( preference_get("font.italic") );
  font_prefs->bold_italic_font = g_strdup( preference_get("font.bolditalic") );

  g_snprintf(buff, BUFFLEN, "Normal:\n%s", font_prefs->normal_font);
  gtk_label_set ( GTK_LABEL(font_prefs->adv_normal_label), buff);

  g_snprintf(buff, BUFFLEN, "Bold:\n%s", font_prefs->bold_font);
  gtk_label_set ( GTK_LABEL(font_prefs->adv_bold_label), buff);

  g_snprintf(buff, BUFFLEN, "Italic:\n%s", font_prefs->italic_font);
  gtk_label_set ( GTK_LABEL(font_prefs->adv_italic_label), buff);

  g_snprintf(buff, BUFFLEN, "Bold Italic:\n%s", font_prefs->bold_italic_font);
  gtk_label_set ( GTK_LABEL(font_prefs->adv_bold_italic_label), buff);

  if ( *font_prefs->adv == '1' ) {
    gtk_widget_set_sensitive(font_prefs->std_frame, FALSE);
    gtk_widget_set_sensitive(font_prefs->adv_frame, TRUE);
    gtk_toggle_button_set_state( GTK_TOGGLE_BUTTON(font_prefs->adv_button), TRUE);
  } else {
    gtk_widget_set_sensitive(font_prefs->adv_frame, FALSE);
    gtk_widget_set_sensitive(font_prefs->std_frame, TRUE);
    gtk_toggle_button_set_state( GTK_TOGGLE_BUTTON(font_prefs->adv_button), FALSE);
  }

  if ( *font_prefs->type != 'f' ) {
    gtk_option_menu_set_history ( GTK_OPTION_MENU(font_prefs->std_type_omenu), 1);
  }

  if ( !strcmp("8", font_prefs->size) ) {
    gtk_option_menu_set_history ( GTK_OPTION_MENU(font_prefs->std_size_omenu), 0);
  } else if ( !strcmp("10", font_prefs->size) ) {
    gtk_option_menu_set_history ( GTK_OPTION_MENU(font_prefs->std_size_omenu), 1);
  } else if ( !strcmp("12", font_prefs->size) ) {
    gtk_option_menu_set_history ( GTK_OPTION_MENU(font_prefs->std_size_omenu), 2);
  } else if ( !strcmp("14", font_prefs->size) ) {
    gtk_option_menu_set_history ( GTK_OPTION_MENU(font_prefs->std_size_omenu), 3);
  } else if ( !strcmp("16", font_prefs->size) ) {
    gtk_option_menu_set_history ( GTK_OPTION_MENU(font_prefs->std_size_omenu), 4);
  }

  bs_function_leave();
  return;
}

NotebookPage pages[] = {
  { "Persona", NULL, persona_prefs_create, persona_prefs_save, persona_prefs_reset },
  { "Fonts",   NULL, prefs_fonts_page,     prefs_fonts_save,   prefs_fonts_reset   },
  { "Servers", NULL, servers_prefs_create, servers_prefs_save, servers_prefs_reset },
  { NULL,      NULL, NULL,                 NULL,               NULL                },
};

void prefs_notebook_page_switch(GtkWidget *widget, GtkNotebookPage *page, gint page_num, PreferencesData *preferences_data)
{
  GtkNotebookPage *oldpage;
  GtkWidget *pixwid;

  bs_function_enter();

  oldpage = GTK_NOTEBOOK(widget)->cur_page;

  if (page == oldpage)
    return;

  pixwid = ((GtkBoxChild*)(GTK_BOX (page->tab_label)->children->data))->widget;
  gtk_pixmap_set (GTK_PIXMAP (pixwid), preferences_data->book_open, preferences_data->book_open_mask);
  pixwid = ((GtkBoxChild*) (GTK_BOX (page->menu_label)->children->data))->widget;
  gtk_pixmap_set (GTK_PIXMAP (pixwid), preferences_data->book_open, preferences_data->book_open_mask);

  if (oldpage)
    {
      pixwid = ((GtkBoxChild*) (GTK_BOX
                                (oldpage->tab_label)->children->data))->widget;
      gtk_pixmap_set (GTK_PIXMAP (pixwid), preferences_data->book_closed, preferences_data->book_closed_mask);
      pixwid = ((GtkBoxChild*) (GTK_BOX (oldpage->menu_label)->children->data))->widget;
      gtk_pixmap_set (GTK_PIXMAP (pixwid), preferences_data->book_closed, preferences_data->book_closed_mask);
    }

  bs_function_leave();
  return;
}

void prefs_notebook_create_pages(PreferencesData *preferences_data)
{

  int page_num = 0;
  GtkWidget *page;
  GtkWidget *label;
  GtkWidget *label_box;
  GtkWidget *menu_box;
  GtkWidget *pixmap;

  bs_function_enter();

  while(pages[page_num].name) {
    page = pages[page_num].create(&pages[page_num].data);
    gtk_widget_show_all(page);
    pages[page_num].reset(pages[page_num].data);

    label_box = gtk_hbox_new (FALSE, 0);
    pixmap = gtk_pixmap_new (preferences_data->book_closed, preferences_data->book_closed_mask);
    gtk_box_pack_start (GTK_BOX (label_box), pixmap, FALSE, TRUE, 0);
    gtk_misc_set_padding (GTK_MISC (pixmap), 3, 1);
    label = gtk_label_new(pages[page_num].name);
    gtk_box_pack_start (GTK_BOX (label_box), label, FALSE, TRUE, 0);
    gtk_widget_show_all (label_box);
     
    menu_box = gtk_hbox_new (FALSE, 0);
    pixmap = gtk_pixmap_new (preferences_data->book_closed, preferences_data->book_closed_mask);
    gtk_box_pack_start (GTK_BOX (menu_box), pixmap, FALSE, TRUE, 0);
    gtk_misc_set_padding (GTK_MISC (pixmap), 3, 1);
    label = gtk_label_new (pages[page_num].name);
    gtk_box_pack_start (GTK_BOX (menu_box), label, FALSE, TRUE, 0);
    gtk_widget_show_all (menu_box);

    gtk_notebook_append_page_menu ( GTK_NOTEBOOK(preferences_data->notebook),
				    page, label_box, menu_box);
    page_num++;
  }        

  bs_function_leave();
  return;
}

void preferences_ok(GtkWidget *widget, PreferencesData *preferences_data)
{
  int page_num = 0;

  bs_function_enter();

  while(pages[page_num].name) {
    if ( !pages[page_num].save(pages[page_num].data) ) {
      gtk_notebook_set_page ( GTK_NOTEBOOK(preferences_data->notebook), page_num);
      gdk_beep();
      return;
    }
    page_num++;
  }

  preference_save();

  gtk_widget_hide ( GTK_WIDGET(preferences_data->dialog) );
  gtk_grab_remove ( GTK_WIDGET(preferences_data->dialog) );

  bs_function_leave();
  return;
}

gint preferences_delete(GtkWidget *widget, GdkEvent *event, PreferencesData *preferences_data)
{
  bs_function_enter();

  gtk_widget_hide ( GTK_WIDGET(preferences_data->dialog) );
  gtk_grab_remove ( GTK_WIDGET(preferences_data->dialog) );

  bs_function_leave();
  return(TRUE);
}

void preferences_cancel(GtkWidget *widget, PreferencesData *preferences_data)
{
  bs_function_enter();

  gtk_widget_hide ( GTK_WIDGET(preferences_data->dialog) );
  gtk_grab_remove ( GTK_WIDGET(preferences_data->dialog) );

  bs_function_leave();
  return;
}

void preferences_dialog(Prefs_Page start_page)
{
  static PreferencesData *preferences_data = NULL;
  GtkWidget *button;
  GdkColor *transparent = NULL;
  int page_num = 0;
 
  bs_function_enter();

  if (!preferences_data) {
    preferences_data = g_malloc(sizeof(PreferencesData));
    preferences_data->dialog = gtk_dialog_new ();
    gtk_widget_set_usize(preferences_data->dialog, 500, -1);

    gtk_window_set_title (GTK_WINDOW (preferences_data->dialog), "Preferences");
    gtk_window_position (GTK_WINDOW (preferences_data->dialog), GTK_WIN_POS_CENTER);
    gtk_signal_connect (GTK_OBJECT (preferences_data->dialog), "delete_event",
			GTK_SIGNAL_FUNC (preferences_delete), preferences_data);
    gtk_signal_connect (GTK_OBJECT (preferences_data->dialog), "destroy",
			GTK_SIGNAL_FUNC(gtk_widget_destroyed),
			&preferences_data);
 
    preferences_data->notebook = gtk_notebook_new ();
    gtk_notebook_popup_enable ( GTK_NOTEBOOK(preferences_data->notebook) );
    gtk_signal_connect (GTK_OBJECT (preferences_data->notebook), "switch_page",
			GTK_SIGNAL_FUNC (prefs_notebook_page_switch), preferences_data);
    gtk_notebook_set_tab_pos (GTK_NOTEBOOK (preferences_data->notebook), GTK_POS_TOP);
    gtk_container_border_width (GTK_CONTAINER (preferences_data->notebook), 10);
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG(preferences_data->dialog)->vbox), preferences_data->notebook,
			FALSE, FALSE, 0);
    gtk_widget_realize (preferences_data->notebook);
    
    preferences_data->book_open = gdk_pixmap_create_from_xpm_d (preferences_data->notebook->window,
								&preferences_data->book_open_mask,
								transparent,
								book_open_xpm);
    preferences_data->book_closed = gdk_pixmap_create_from_xpm_d (preferences_data->notebook->window,
								  &preferences_data->book_closed_mask,
								  transparent,
								  book_closed_xpm);

    prefs_notebook_create_pages (preferences_data);

    button = gtk_button_new_with_label ("Ok");
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG(preferences_data->dialog)->action_area), button, TRUE, FALSE, 0);
    gtk_signal_connect( GTK_OBJECT (button), "clicked",
			GTK_SIGNAL_FUNC (preferences_ok), 
			preferences_data );
    GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
    gtk_widget_grab_default (button);
    gtk_widget_show (button);
    
    button = gtk_button_new_with_label ("Cancel");
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG(preferences_data->dialog)->action_area), button, TRUE, FALSE, 0);
    gtk_signal_connect( GTK_OBJECT (button), "clicked",
			GTK_SIGNAL_FUNC (preferences_cancel),
			preferences_data );
    gtk_widget_show (button);
  } else {
    while(pages[page_num].name) {
      pages[page_num].reset(pages[page_num].data);
      page_num++;
    }
  }
  if ( start_page != PREFS_DEFAULT ) {
    gtk_notebook_set_page ( GTK_NOTEBOOK(preferences_data->notebook), (gint) start_page);
  }

  gtk_grab_add (GTK_WIDGET (preferences_data->dialog));
  gtk_widget_show_all(preferences_data->dialog);

  bs_function_leave();
  return;
}

/* --------------------------------------------------------------------- */

void nick_ok(GtkWidget *widget, Nick_Data *nick_data)
{
  char *nick_entry_text;

  bs_function_enter();

  nick_entry_text = gtk_entry_get_text(GTK_ENTRY(nick_data->entry));

  if (strlen(nick_entry_text) == 0) {
    bs_function_leave();
    return;
  }

  g_free(nick_data->connection->nick);
  nick_data->connection->nick = g_strdup(nick_entry_text);
  irc_send_format(nick_data->connection->sd, "NICK %s", nick_entry_text);

  gtk_widget_destroy(nick_data->dialog);
/*   gtk_main_quit(); */

  bs_function_leave();
  return;
}

void nick_dialog(Connection *connection, char *message)
{
  GtkWidget *vbox;
  GtkWidget *label;
  GtkWidget *button;
  Nick_Data *nick_data;

  bs_function_enter();

  nick_data = g_malloc(sizeof(Nick_Data));
  nick_data->connection = connection;

  nick_data->dialog = gtk_dialog_new();
  gtk_window_set_title (GTK_WINDOW (nick_data->dialog), "Nick");
  gtk_window_position (GTK_WINDOW (nick_data->dialog), GTK_WIN_POS_CENTER);
/*   gtk_signal_connect (GTK_OBJECT (nick_data->dialog), "delete_event", */
/* 		      GTK_SIGNAL_FUNC (gtk_true), NULL); */
/*   gtk_signal_connect (GTK_OBJECT (nick_data->dialog), "destroy", */
/* 			GTK_SIGNAL_FUNC (gtk_main_quit), NULL); */
  
  vbox = GTK_DIALOG(nick_data->dialog)->vbox;

  label = gtk_label_new(message);
  gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 10);
  gtk_widget_show (label);

  nick_data->entry = gtk_entry_new();
  gtk_box_pack_start (GTK_BOX (vbox), nick_data->entry, TRUE, TRUE, 10);
  gtk_widget_show (nick_data->entry);

  button = gtk_button_new_with_label ("Ok");
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG(nick_data->dialog)->action_area), button, TRUE, FALSE, 0);
  gtk_signal_connect ( GTK_OBJECT (button), "clicked",
		       GTK_SIGNAL_FUNC (nick_ok),
		       nick_data);
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_widget_grab_default (button);
  gtk_widget_show (button);

  gtk_widget_show (nick_data->dialog);
 
/*   gtk_main(); */

/*   gtk_widget_destroy(nick_data->dialog); */

  bs_function_leave();
  return;
} 

void about_dialog()
{
  GtkWidget *window;
  GtkWidget *vbox;
  GtkWidget *label;
  GtkWidget *button;

  char buff[BUFFLEN];

  bs_function_enter();

  window = gtk_window_new (GTK_WINDOW_DIALOG);
  gtk_window_set_title (GTK_WINDOW (window), "About");
  gtk_window_position (GTK_WINDOW (window), GTK_WIN_POS_CENTER);
/*   gtk_signal_connect (GTK_OBJECT (window), "delete_event", */
/*                       GTK_SIGNAL_FUNC (gtk_true), NULL); */
/*   gtk_signal_connect (GTK_OBJECT (window), "destroy", */
/*                       GTK_SIGNAL_FUNC (gtk_main_quit), NULL); */
  gtk_grab_add (GTK_WIDGET (window));

  vbox = gtk_vbox_new (FALSE, 0);
  gtk_container_border_width (GTK_CONTAINER (vbox), 8);
  gtk_container_add (GTK_CONTAINER (window), vbox);

  g_snprintf(buff, BUFFLEN, "Bezerk version %s", BEZERK_VERSION);
  label = gtk_label_new (buff);
  gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 3);
  gtk_widget_show (label);
  
  label = gtk_label_new ("Copyright  1998 Tony Gale");
  gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 5);
  gtk_widget_show (label);

  label = gtk_label_new ("trog@gtk.org");
  gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 5);
  gtk_widget_show (label);

  label = gtk_label_new ("http://www.gtk.org/~trog");
  gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 5);
  gtk_widget_show (label);

  button = gtk_button_new_with_label ("Ok");
  gtk_widget_set_usize (button, 96, -1);
  gtk_box_pack_start (GTK_BOX (vbox), button, TRUE, FALSE, 0);
  gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                    GTK_SIGNAL_FUNC (gtk_widget_destroy), GTK_OBJECT (window));
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_widget_grab_default (button);
  gtk_widget_show (button);

  gtk_widget_show (vbox);

  gtk_widget_show (window);

/*   gtk_main (); */

  bs_function_leave();
  return;
}

GtkWidget *new_alias_entry;
char *new_alias_name;

void new_alias_ok(GtkWidget *widget, gpointer dialog)
{
  char *entry_text;

  bs_function_enter();

  entry_text = gtk_entry_get_text(GTK_ENTRY(new_alias_entry));
  if (!strlen(entry_text) || strchr(entry_text, ' ') || strchr(entry_text, '\t') ) {
    bs_function_leave();
    return;
  }

  new_alias_name = g_strdup(entry_text);
    
  gtk_widget_destroy(dialog);
  gtk_main_quit();

  bs_function_leave();
  return;
}
 
char *new_alias_dialog()
{
  GtkWidget *dialog;
  GtkWidget *vbox;
  GtkWidget *label;
  GtkWidget *button;

  bs_function_enter();

  if (new_alias_name) {
    g_free(new_alias_name);
  }
  new_alias_name = NULL;

  dialog = gtk_dialog_new();
  gtk_window_set_title (GTK_WINDOW (dialog), "New Alias");
  gtk_window_position (GTK_WINDOW (dialog), GTK_WIN_POS_CENTER);
/*   gtk_signal_connect (GTK_OBJECT (dialog), "delete_event", */
/* 		      GTK_SIGNAL_FUNC (gtk_true), NULL); */
/*   gtk_signal_connect (GTK_OBJECT (dialog), "destroy", */
/* 			GTK_SIGNAL_FUNC (gtk_main_quit), NULL); */
  
  vbox = GTK_DIALOG(dialog)->vbox;

  label = gtk_label_new("Enter a single word name for the new alias");
  gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 10);
  gtk_widget_show (label);

  new_alias_entry = gtk_entry_new();
  gtk_box_pack_start (GTK_BOX (vbox), new_alias_entry, TRUE, TRUE, 10);
  gtk_widget_show (new_alias_entry);

  button = gtk_button_new_with_label ("Ok");
  /*   gtk_widget_set_usize (button, 96, -1);  */
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG(dialog)->action_area), button, TRUE, FALSE, 0);
  gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
			     GTK_SIGNAL_FUNC (new_alias_ok), GTK_OBJECT (dialog));
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_widget_grab_default (button);
  gtk_widget_show (button);

  button = gtk_button_new_with_label ("Cancel");
  /*   gtk_widget_set_usize (button, 96, -1);  */
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG(dialog)->action_area), button, TRUE, FALSE, 0);
  gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy), GTK_OBJECT (dialog));
  gtk_widget_show (button);

  gtk_widget_show (dialog);
 
  gtk_main();

  bs_function_leave();
  return(new_alias_name);
} 

typedef struct _TreeItemData {
  Alias *item;
  GSList *list;
}TreeItemData;

GtkWidget *alias_value_entry;

void set_alias(Alias *new_alias)
{
  static Alias *current_alias=NULL;
  char *new_value;

  bs_function_enter();

  if (new_alias == (Alias *) -1) {
    current_alias=NULL;
  } else {
    if (current_alias) {
      new_value = g_strdup(gtk_entry_get_text( GTK_ENTRY(alias_value_entry) ));
      g_free(current_alias->value);
      current_alias->value = new_value;
    } else if (new_alias) {
      gtk_entry_set_editable(GTK_ENTRY(alias_value_entry), TRUE);
    }
    current_alias = new_alias;
  }
  bs_function_leave();
  return;
}

void tree_selection_changed(GtkWidget *widget, gpointer data)
{
  GtkWidget *item;
  GList *selected_list;
  Alias *script_item;
  TreeItemData *item_data;

  bs_function_enter();

  selected_list = (GList *) GTK_TREE_SELECTION( GTK_TREE(widget) );
  if (!selected_list) {
    set_alias((Alias *) -1);
    bs_function_leave();
    return;
  }
  item = selected_list->data;
  item_data = gtk_object_get_user_data(GTK_OBJECT(item));
  if (!item_data) {
    set_alias(NULL);
    gtk_entry_set_text( GTK_ENTRY(alias_value_entry), "");
    gtk_entry_set_editable(GTK_ENTRY(alias_value_entry), FALSE);
    bs_function_leave();
    return;
  }
  script_item = (Alias *) item_data->item;
  set_alias(script_item);
 
  gtk_entry_set_text( GTK_ENTRY(alias_value_entry), script_item->value);

  bs_function_leave();
  return;
}

void tree_delete_item(GtkWidget *widget, GtkTree* tree)
{
  GtkWidget *item;
  GList *selected_list;
  Alias *script_item;
  TreeItemData *item_data;

  bs_function_enter();

  selected_list = (GList *) GTK_TREE_SELECTION( GTK_TREE(tree) );
  if (!selected_list) {
    bs_function_leave();
    return;
  }
  item = selected_list->data;
  item_data = gtk_object_get_user_data(GTK_OBJECT(item));
  if (!item_data) {
    bs_function_leave();
    return;
  }
  script_item = (Alias *) item_data->item;
  gtk_container_remove(GTK_CONTAINER(item->parent), item);
  g_free(script_item->name);
  g_free(script_item->value);
  aliases = g_slist_remove(aliases, script_item);

  bs_function_leave();
  return;
}

void tree_new_alias(GtkWidget *widget, GtkWidget* tree)
{
  char *alias_name;
  GtkWidget *tree_item;
  TreeItemData *item_data;
  Alias *new_alias;

  bs_function_enter();

  alias_name = new_alias_dialog();

  if (alias_name) {
    new_alias = g_malloc(sizeof(Alias));
    new_alias->name = g_strdup(alias_name);
    new_alias->value = g_strdup("");
    aliases = g_slist_append( aliases, new_alias);
    item_data = g_malloc(sizeof(TreeItemData));
    item_data->list = aliases;
    item_data->item = new_alias;
    tree_item = gtk_tree_item_new_with_label(alias_name);
    gtk_object_set_user_data(GTK_OBJECT(tree_item), item_data);
    gtk_tree_append(GTK_TREE(tree), tree_item);
    gtk_widget_show(tree_item);
  }
    
  bs_function_leave();
  return;
}
 
void tree_ok(GtkWidget *widget, GtkWidget *dialog)
{

  bs_function_enter();

  set_alias(NULL);  
  prefs_save_aliases();
  gtk_widget_destroy(dialog);

  bs_function_leave();
  return;
}

GtkWidget *create_subtree(GSList *members, GtkWidget *root)
{
  GtkWidget *subtree, *item;
  GSList *member_entry;
  TreeItemData *item_data;

  /* TODO: This will need changing to a script type */
  Alias *alias;

  bs_function_enter();

  subtree = gtk_tree_new();

  member_entry = members;
  while(member_entry) {
    item_data = g_malloc(sizeof(TreeItemData));
    item_data->list = members;
    alias = member_entry->data;
    item_data->item = alias;
    item = gtk_tree_item_new_with_label(alias->name);
    gtk_object_set_user_data(GTK_OBJECT(item), item_data);
    gtk_tree_append(GTK_TREE(subtree), item);
    gtk_widget_show(item);
    member_entry = g_slist_next(member_entry);
  }

  gtk_tree_item_set_subtree(GTK_TREE_ITEM(root), subtree);

  bs_function_leave();
  return(subtree);
}

int script_dialog()
{
  GtkWidget *dialog;
  GtkWidget *hbox;
  GtkWidget *scrolled_win;
  GtkWidget *tree;
  GtkWidget *tree_item;
  GtkWidget *subtree;
  GtkWidget *button;

  bs_function_enter();
  
  dialog = gtk_dialog_new();
  gtk_window_set_title (GTK_WINDOW (dialog), "Aliases");
  gtk_widget_set_usize (dialog, 400, 250);
  gtk_window_position (GTK_WINDOW (dialog), GTK_WIN_POS_CENTER);
  gtk_signal_connect (GTK_OBJECT (dialog), "delete_event",
		      GTK_SIGNAL_FUNC (gtk_true), NULL);
  gtk_signal_connect (GTK_OBJECT (dialog), "destroy",
			GTK_SIGNAL_FUNC (gtk_main_quit), NULL);
  
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_container_border_width (GTK_CONTAINER (hbox), 4);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->vbox), hbox);
  gtk_widget_show (hbox);

  scrolled_win = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_win),
                                  GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_box_pack_start (GTK_BOX (hbox), scrolled_win, FALSE, TRUE, 0);
  gtk_widget_set_usize (scrolled_win, 120, -1);
  gtk_widget_show (scrolled_win);

  tree = gtk_tree_new();
  gtk_signal_connect(GTK_OBJECT(tree), "selection_changed",
                     GTK_SIGNAL_FUNC(tree_selection_changed),
                     (gpointer)NULL);
/*   gtk_object_set_user_data(GTK_OBJECT(tree), tree_buttons); */
  gtk_container_add(GTK_CONTAINER(scrolled_win), tree);
  gtk_tree_set_selection_mode(GTK_TREE(tree), GTK_SELECTION_BROWSE);
   gtk_tree_set_view_lines(GTK_TREE(tree), TRUE); 
/*   gtk_tree_set_view_mode(GTK_TREE(tree), !view_line); */
  gtk_widget_show(tree);
 
  tree_item = gtk_tree_item_new_with_label("Aliases");
  gtk_tree_append(GTK_TREE(tree), tree_item);
  gtk_widget_show(tree_item);       
  
  subtree = create_subtree(aliases, tree_item);

  alias_value_entry = gtk_entry_new();
  gtk_box_pack_start (GTK_BOX (hbox), alias_value_entry, TRUE, TRUE, 0);
  gtk_entry_set_editable(GTK_ENTRY(alias_value_entry), FALSE);
  gtk_widget_grab_focus(alias_value_entry);
  gtk_widget_show (alias_value_entry);

  button = gtk_button_new_with_label ("Ok");
  /*   gtk_widget_set_usize (button, 96, -1);  */
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG(dialog)->action_area), button, TRUE, FALSE, 0);
  gtk_signal_connect(GTK_OBJECT (button), "clicked",
		     GTK_SIGNAL_FUNC (tree_ok), 
		     GTK_OBJECT (dialog));
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_widget_grab_default (button);
  gtk_widget_show (button);

  button = gtk_button_new_with_label ("New...");
  /*   gtk_widget_set_usize (button, 96, -1);  */
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG(dialog)->action_area), button, TRUE, FALSE, 0);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (tree_new_alias), 
		      (gpointer)subtree);
  gtk_widget_show (button);

  button = gtk_button_new_with_label ("Delete");
  /*   gtk_widget_set_usize (button, 96, -1);  */
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG(dialog)->action_area), button, TRUE, FALSE, 0);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (tree_delete_item), 
		      (gpointer)tree);
  gtk_widget_show (button);

  gtk_widget_show (dialog);
 
  gtk_main();

  bs_function_leave();
  return(0);
}

void file_sel_ok (GtkWidget *widget, FileSelData *fs_data)
{

  bs_function_enter();

  printf("file_sel_ok %s\n", gtk_file_selection_get_filename(GTK_FILE_SELECTION (fs_data->window)));
  if ( (fs_data->ok_func(gtk_file_selection_get_filename(GTK_FILE_SELECTION (fs_data->window)), 
			 fs_data->user_data)) ) {

    /* set this to NULL so that when the window is destroyed, file_sel_cancel doesn't
     * try to free the user_data */ 
    fs_data->user_data = NULL;

    gtk_widget_destroy(fs_data->window);
  }

  gdk_beep();

  bs_function_leave();
  return;
}

void file_sel_cancel (GtkWidget *w, FileSelData *fs_data)
{

  bs_function_enter();

  if (fs_data->user_data) {
    if (fs_data->cancel_func) {
      fs_data->cancel_func(fs_data->user_data);
    } else {
      g_free(fs_data->user_data);
    }
    /* set this to NULL so that when the window is destroyed, we don't call 
     * gtk_widget_destroy twice */
    fs_data->user_data = NULL;
    gtk_widget_destroy(fs_data->window);
  } else {
    g_free(fs_data);
  }

  bs_function_leave();
  return;
}
 
void file_selection_dialog(void *user_data, int (*ok_func)(char *filename, void *data), 
			   void (*cancel_func)(void *data))
{
  FileSelData *dialog_data;

  bs_function_enter();

  if (!ok_func) {
      bs_function_leave();
      return;
  }

  dialog_data = g_malloc(sizeof(FileSelData));
  dialog_data->user_data = user_data;
  dialog_data->ok_func = ok_func;
  dialog_data->cancel_func = cancel_func;

  dialog_data->window = gtk_file_selection_new ("Select File");
  gtk_file_selection_hide_fileop_buttons( GTK_FILE_SELECTION (dialog_data->window) );
  gtk_signal_connect (GTK_OBJECT (dialog_data->window), "destroy",
		      (GtkSignalFunc) file_sel_cancel, dialog_data);

  /* Connect the ok_button to file_sel_ok function */
  gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (dialog_data->window)->ok_button),
		      "clicked", (GtkSignalFunc) file_sel_ok, dialog_data );

  /* Connect the cancel_button to file_sel_cancel function */
  gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (dialog_data->window)->cancel_button),
		      "clicked", (GtkSignalFunc) file_sel_cancel, dialog_data);

  gtk_widget_show(dialog_data->window);

  bs_function_leave();
  return;
}

gint progress_delete(GtkWidget *widget, GdkEvent *event, ProgressData *progress_data)
{
  bs_function_enter();
  
  if (!progress_data) {
    return(FALSE);
  }

  if (progress_data->cancel_func) {
    progress_data->cancel_func(progress_data->user_data);
  }

  bs_function_leave();
  return(FALSE);
}

void progress_destroy(GtkWidget *widget, ProgressData *progress_data)
{
  bs_function_enter();

  if (!progress_data) {
    return;
  }
  if (progress_data->cancel_func) {
    progress_data->cancel_func(progress_data->user_data);
  }

  g_free(progress_data);

  bs_function_leave();
  return;
}

void progress_cancel(GtkWidget *widget, ProgressData *progress_data)
{
  bs_function_enter();

  if (!progress_data) {
    return;
  }

  gtk_widget_destroy(progress_data->dialog);

  bs_function_leave();
  return;
}

void progress_done(ProgressData *progress_data)
{
  bs_function_enter();
  
  if (!progress_data) {
    return;
  }

  progress_data->cancel_func = NULL;
  gtk_widget_destroy(progress_data->dialog);

  bs_function_leave();
  return;
}

void progress_update(float value, ProgressData *progress_data)
{
  bs_function_enter();
  
  if (value > 1.0) {
    value = 1.0;
  } else if (value < 0.0) {
    value = 0.0;
  }

  gtk_progress_bar_update (GTK_PROGRESS_BAR (progress_data->progress_bar), value);
 
  bs_function_leave();
  return;
}
  
ProgressData *progress_dialog(void *user_data, char *title, char *text, void (*cancel_func)(void *data))
{
  GtkWidget *label;
  GtkWidget *button;
  ProgressData *progress_data;

  bs_function_enter();

  progress_data = g_malloc(sizeof(ProgressData));
  progress_data->user_data = user_data;
  progress_data->cancel_func = cancel_func;

  progress_data->dialog = gtk_dialog_new();
  gtk_window_set_title (GTK_WINDOW (progress_data->dialog), title);
  gtk_window_position (GTK_WINDOW (progress_data->dialog), GTK_WIN_POS_CENTER);
/*   gtk_signal_connect (GTK_OBJECT (progress_data->dialog), "delete_event", */
/* 			GTK_SIGNAL_FUNC (progress_delete), progress_data); */
  gtk_signal_connect (GTK_OBJECT (progress_data->dialog), "destroy",
			GTK_SIGNAL_FUNC (progress_destroy), progress_data);
  
  label = gtk_label_new(text);
  gtk_label_set_justify( GTK_LABEL(label), GTK_JUSTIFY_LEFT);
  gtk_container_border_width (GTK_CONTAINER (GTK_DIALOG(progress_data->dialog)->vbox), 10);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG(progress_data->dialog)->vbox), label, TRUE, TRUE, 10);
  gtk_widget_show (label);

  progress_data->progress_bar = gtk_progress_bar_new ();
  gtk_widget_set_usize (progress_data->progress_bar, 200, 20);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG(progress_data->dialog)->vbox), 
		      progress_data->progress_bar, TRUE, TRUE, 10);
  gtk_widget_show (progress_data->progress_bar);

  if (cancel_func) {
    button = gtk_button_new_with_label ("Cancel");
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG(progress_data->dialog)->action_area), button, TRUE, FALSE, 0);
    gtk_signal_connect (GTK_OBJECT (button), "clicked",
			GTK_SIGNAL_FUNC (progress_cancel),
			progress_data );
    gtk_widget_show (button);
  }

  gtk_widget_show (progress_data->dialog);

  bs_function_leave();
  return(progress_data);
}
