C
C ------------------------------------------------------------
C  Mod by PJ 18.8.97: Added SAVE statements in all subroutines
C ------------------------------------------------------------
C
C============================================================================
C============================================================================
C============================================================================
c
c##########################################################################
c#                                                                        #
c#                              GEOPACK                                   #
c#                     (MAIN SET OF FORTRAN CODES)                        #
c#                                                                        #
c##########################################################################
C

c
c    This collection of subroutines is a result of several upgrades
c    of the original package by N.A. Tsyganenko, including that by
c    M. Peredo at the Science Planning and Operations Facility (SPOF)
c    of the ISTP program (Sept.1992).
c
c    This release is dated by April 16, 1996.
c
c    Please see the additional file geopack.doc for descriptions of
c    the individual subroutines.
c
c    Major changes include the following.

c       (1) A modification of the subroutine IGRF was made, to accept
c    dates beyond 1990.

c       (2) A modification of RECALC:  some time ago it was replaced by
c    a faster subroutine RECOMP, which we have renamed back to RECALC. So,
c    in fact, we have the new subroutine under the old name.

c        (3) A new subroutine GEIGEO was included, which makes transformations
c    between Geocentric Equatorial Inertial and Geographical Cartesian systems.

C        (4) In this version of the package, the COMMON-block containing
c    elements of the coordinate transformation matrices has the symbolic name
c    GEOPACK (this common-block was unnamed in the previous releases). It is
c    therefore important that the codes using the older versions of the GEOPACK
c    software be accordingly modified.
C
c        (5) The subroutines RHAND, STEP, and TRACE have now additional input
c    parameter PARMOD, which is a 10-element array, reserved for specifying
c    input parameters of new external field models, such as the solar wind
c    pressure, IMF components, Dst- and AE-indices, and other quantities which
c    can be added later on, as the magnetospheric models become more and more
c    sophisticated.  Users should make a corresponding modification of the
c    list of input parameters of the older model subroutines for the T87 and
c    T89 models (do not forget to describe PARMOD(10) as a dimension !), to
c    make them compatible with this release of GEOPACK.
C
c
c       (6) In this edition of the package, all versions of the external field
c     models, compiled by M. Peredo in the previous release of GEOPACK, were
c     taken out. The present package contains only the most "stable" software,
c     which is not supposed to undergo major changes in the future.  Routines
c     for the external field models are provided now separately.
c----------------------------------------------------------------------------
c
c
      SUBROUTINE IGRF(IY,NM,R,T,F,BR,BT,BF)
C
c
C       MODIFIED TO ACCEPT DATES BETWEEN 1965 AND 2000; COEFFICIENTS
C       FOR 1985, 1990, AND 1995 CORRESPOND TO DGRF1985, DGRF1990, AND
C       IGRF1995 MODELS, RESPECTIVELY.  ALSO, THE CODE WAS MODIFIED
C       TO ACCEPT DATES BEYOND 1995,  USING LINEAR EXTRAPOLATION BETWEEN
C       1995 AND 2000 BASED ON THE IGRF SECULAR VARIATION COEFFICIENTS.
c
c
c       Modified by:    Mauricio Peredo
c                       Hughes STX at NASA/GSFC
c                       September 1992 and April 1996.
c
c
C  CALCULATES COMPONENTS OF MAIN GEOMAGNETIC FIELD IN SPHERICAL
C  GEOGRAPHICAL COORD SYSTEM BY USING THIRD GENERATION IGRF MODEL
C  (J. GEOMAG. GEOELECTR.(1982), V.34, P.313-315,
C  GEOMAGN. AND AERONOMY (1986), V.26, P.523-525).
C  UPDATING THE COEFFICIENTS TO A GIVEN EPOCH IS MADE DURING THE FIRST
C  CALL AND AFTER EVERY CHANGE OF PARAMETER IY.
C------INPUT PARAMETERS:
C  IY - YEAR NUMBER (FOUR-DIGIT; FROM 1965 UP TO 1990)
C  NM - MAXIMAL ORDER OF HARMONICS TAKEN INTO ACCOUNT (NOT MORE THAN 10)
C  R,T,F - SPHERICAL COORDINATES OF THE POINT (R IN UNITS RE=6371.2 KM,
C  COLATITUDE T AND LONGITUDE F IN RADIANS)
C----- OUTPUT PARAMETERS:
C  BR,BT,BF - SPHERICAL COMPONENTS OF MAIN GEOMAGN.FIELD IN NANOTESLA
C
C
C                   AUTHOR: NIKOLAI A. TSYGANENKO
C                           INSTITUTE OF PHYSICS
C                           ST.-PETERSBURG STATE UNIVERSITY
C                           STARY PETERGOF 198904
C                           ST.-PETERSBURG
C                           RUSSIA
C
        IMPLICIT NONE
        SAVE
C
      REAL A(11),B(11),G(66),H(66),REC(66),G65(66),H65(66),G70(66),
     *H70(66),G75(66),H75(66),G80(66),H80(66),G85(66),H85(66),G90(66),
     *H90(66),G95(66),H95(66),DG95(45),DH95(45)

        REAL R,T,F,BR,BT,BF,DT,F2,F1,S,P,AA,PP,D,BBR,BBF,U,CF,SF,
     1       C,W,X,Y,Z,Q,BI,P2,D2,AN,E,HH,BBT,QQ,XK,DP,PM

      LOGICAL BK,BM

        INTEGER IY,NM,MA,IPR,IYR,KNM,N,N2,M,MNN,MN,K,MM
c
      DATA G65/0.,-30334.,-2119.,-1662.,2997.,1594.,1297.,-2038.,1292.,
     *856.,957.,804.,479.,-390.,252.,-219.,358.,254.,-31.,-157.,-62.,
     *45.,61.,8.,-228.,4.,1.,-111.,75.,-57.,4.,13.,-26.,-6.,13.,1.,13.,
     *5.,-4.,-14.,0.,8.,-1.,11.,4.,8.,10.,2.,-13.,10.,-1.,-1.,5.,1.,-2.,
     *-2.,-3.,2.,-5.,-2.,4.,4.,0.,2.,2.,0./
      DATA H65/0.,0.,5776.,0.,-2016.,114.,0.,-404.,240.,-165.,0.,148.,
     *-269.,13.,-269.,0.,19.,128.,-126.,-97.,81.,0.,-11.,100.,68.,-32.,
     *-8.,-7.,0.,-61.,-27.,-2.,6.,26.,-23.,-12.,0.,7.,-12.,9.,-16.,4.,
     *24.,-3.,-17.,0.,-22.,15.,7.,-4.,-5.,10.,10.,-4.,1.,0.,2.,1.,2.,
     *6.,-4.,0.,-2.,3.,0.,-6./
c
      DATA G70/0.,-30220.,-2068.,-1781.,3000.,1611.,1287.,-2091.,1278.,
     *838.,952.,800.,461.,-395.,234.,-216.,359.,262.,-42.,-160.,-56.,
     *43.,64.,15.,-212.,2.,3.,-112.,72.,-57.,1.,14.,-22.,-2.,13.,-2.,
     *14.,6.,-2.,-13.,-3.,5.,0.,11.,3.,8.,10.,2.,-12.,10.,-1.,0.,3.,
     *1.,-1.,-3.,-3.,2.,-5.,-1.,6.,4.,1.,0.,3.,-1./
      DATA H70/0.,0.,5737.,0.,-2047.,25.,0.,-366.,251.,-196.,0.,167.,
     *-266.,26.,-279.,0.,26.,139.,-139.,-91.,83.,0.,-12.,100.,72.,-37.,
     *-6.,1.,0.,-70.,-27.,-4.,8.,23.,-23.,-11.,0.,7.,-15.,6.,-17.,6.,
     *21.,-6.,-16.,0.,-21.,16.,6.,-4.,-5.,10.,11.,-2.,1.,0.,1.,1.,3.,
     *4.,-4.,0.,-1.,3.,1.,-4./
c
      DATA G75/0.,-30100.,-2013.,-1902.,3010.,1632.,1276.,-2144.,1260.,
     *830.,946.,791.,438.,-405.,216.,-218.,356.,264.,-59.,-159.,-49.,
     *45.,66.,28.,-198.,1.,6.,-111.,71.,-56.,1.,16.,-14.,0.,12.,-5.,
     *14.,6.,-1.,-12.,-8.,4.,0.,10.,1.,7.,10.,2.,-12.,10.,-1.,-1.,4.,
     *1.,-2.,-3.,-3.,2.,-5.,-2.,5.,4.,1.,0.,3.,-1./
      DATA H75/0.,0.,5675.,0.,-2067.,-68.,0.,-333.,262.,-223.,0.,191.,
     *-265.,39.,-288.,0.,31.,148.,-152.,-83.,88.,0.,-13.,99.,75.,-41.,
     *-4.,11.,0.,-77.,-26.,-5.,10.,22.,-23.,-12.,0.,6.,-16.,4.,-19.,6.,
     *18.,-10.,-17.,0.,-21.,16.,7.,-4.,-5.,10.,11.,-3.,1.,0.,1.,1.,3.,
     *4.,-4.,-1.,-1.,3.,1.,-5./
c
      DATA G80/0.,-29992.,-1956.,-1997.,3027.,1663.,1281.,-2180.,1251.,
     *833.,938.,782.,398.,-419.,199.,-218.,357.,261.,-74.,-162.,-48.,
     *48.,66.,42.,-192.,4.,14.,-108.,72.,-59.,2.,21.,-12.,1.,11.,-2.,
     *18.,6.,0.,-11.,-7.,4.,3.,6.,-1.,5.,10.,1.,-12.,9.,-3.,-1.,7.,2.,
     *-5.,-4.,-4.,2.,-5.,-2.,5.,3.,1.,2.,3.,0./
      DATA H80/0.,0.,5604.,0.,-2129.,-200.,0.,-336.,271.,-252.,0.,212.,
     *-257.,53.,-297.,0.,46.,150.,-151.,-78.,92.,0.,-15.,93.,71.,-43.,
     *-2.,17.,0.,-82.,-27.,-5.,16.,18.,-23.,-10.,0.,7.,-18.,4.,-22.,9.,
     *16.,-13.,-15.,0.,-21.,16.,9.,-5.,-6.,9.,10.,-6.,2.,0.,1.,0.,3.,
     *6.,-4.,0.,-1.,4.,0.,-6./
c
      DATA G85/0.,-29873.,-1905.,-2072.,3044.,1687.,1296.,-2208.,1247.,
     *829.,936.,780.,361.,-424.,170.,-214.,355.,253.,-93.,-164.,-46.,
     *53.,65.,51.,-185.,4.,16.,-102.,74.,-62.,3.,24.,-6.,4.,10.,0.,21.,
     *6.,0.,-11.,-9.,4.,4.,4.,-4.,5.,10.,1.,-12.,9.,-3.,-1.,7.,1.,-5.,
     *-4.,-4.,3.,-5.,-2.,5.,3.,1.,2.,3.,0./
      DATA H85/0.,0.,5500.,0.,-2197.,-306.,0.,-310.,284.,-297.,0.,232.,
     *-249.,69.,-297.,0.,47.,150.,-154.,-75.,95.,0.,-16.,88.,69.,-48.,
     *-1.,21.,0.,-83.,-27.,-2.,20.,17.,-23.,-7.,0.,8.,-19.,5.,-23.,11.,
     *14.,-15.,-11.,0.,-21.,15.,9.,-6.,-6.,9.,9.,-7.,2.,0.,1.,0.,3.,
     *6.,-4.,0.,-1.,4.,0.,-6./
c
      DATA G90/0., -29775.,  -1848.,  -2131.,   3059.,   1686.,   1314.,
     *     -2239.,   1248.,    802.,    939.,    780.,    325.,   -423.,
     *       141.,   -214.,    353.,    245.,   -109.,   -165.,    -36.,
     *        61.,     65.,     59.,   -178.,      3.,     18.,    -96.,
     *        77.,    -64.,      2.,     26.,     -1.,      5.,      9.,
     *         0.,     23.,      5.,     -1.,    -10.,    -12.,      3.,
     *         4.,      2.,     -6.,      4.,      9.,      1.,    -12.,
     *         9.,     -4.,     -2.,      7.,      1.,     -6.,     -3.,
     *        -4.,      2.,     -5.,     -2.,      4.,      3.,      1.,
     *         3.,      3.,      0./

      DATA H90/0.,      0.,   5406.,      0.,  -2279.,   -373.,      0.,
     *      -284.,    293.,   -352.,      0.,    247.,   -240.,     84.,
     *      -299.,      0.,     46.,    154.,   -153.,    -69.,     97.,
     *         0.,    -16.,     82.,     69.,    -52.,      1.,     24.,
     *         0.,    -80.,    -26.,      0.,     21.,     17.,    -23.,
     *        -4.,      0.,     10.,    -19.,      6.,    -22.,     12.,
     *        12.,    -16.,    -10.,      0.,    -20.,     15.,     11.,
     *        -7.,     -7.,      9.,      8.,     -7.,      2.,      0.,
     *         2.,      1.,      3.,      6.,     -4.,      0.,     -2.,
     *         3.,     -1.,     -6./

      DATA G95/0., -29682.,  -1789.,  -2197.,   3074.,   1685.,   1329.,
     *     -2268.,   1249.,    769.,    941.,    782.,    291.,   -421.,
     *       116.,   -210.,    352.,    237.,   -122.,   -167.,    -26.,
     *        66.,     64.,     65.,   -172.,      2.,     17.,    -94.,
     *        78.,    -67.,      1.,     29.,      4.,      8.,     10.,
     *        -2.,     24.,      4.,     -1.,     -9.,    -14.,      4.,
     *         5.,      0.,     -7.,      4.,      9.,      1.,    -12.,
     *         9.,     -4.,     -2.,      7.,      0.,     -6.,     -3.,
     *        -4.,      2.,     -5.,     -2.,      4.,      3.,      1.,
     *         3.,      3.,      0./

      DATA H95/0.,      0.,   5318.,      0.,  -2356.,   -425.,      0.,
     *      -263.,    302.,   -406.,      0.,    262.,   -232.,     98.,
     *      -301.,      0.,     44.,    157.,   -152.,    -64.,     99.,
     *         0.,    -16.,     77.,     67.,    -57.,      4.,     28.,
     *         0.,    -77.,    -25.,      3.,     22.,     16.,    -23.,
     *        -3.,      0.,     12.,    -20.,      7.,    -21.,     12.,
     *        10.,    -17.,    -10.,      0.,    -19.,     15.,     11.,
     *        -7.,     -7.,      9.,      7.,     -8.,      1.,      0.,
     *         2.,      1.,      3.,      6.,     -4.,      0.,     -2.,
     *         3.,     -1.,     -6./

      DATA DG95/0.0,  17.6,    13.0,   -13.2,     3.7,    -0.8,     1.5,
     *         -6.4,  -0.2,    -8.1,     0.8,     0.9,    -6.9,     0.5,
     *         -4.6,   0.8,     0.1,    -1.5,    -2.0,    -0.1,     2.3,
     *          0.5,  -0.4,     0.6,     1.9,    -0.2,    -0.2,     0.0,
     *         -0.2,  -0.8,    -0.6,     0.6,     1.2,     0.1,     0.2,
     *         -0.6,   0.3,    -0.2,     0.1,     0.4,    -1.1,     0.3,
     *          0.2,  -0.9,    -0.3/

      DATA DH95/0.0,   0.0,   -18.3,     0.0,   -15.0,    -8.8,     0.0,
     *          4.1,   2.2,   -12.1,     0.0,     1.8,     1.2,     2.7,
     *         -1.0,   0.0,     0.2,     1.2,     0.3,     1.8,     0.9,
     *          0.0,   0.3,    -1.6,    -0.2,    -0.9,     1.0,     2.2,
     *          0.0,   0.8,     0.2,     0.6,    -0.4,     0.0,    -0.3,
     *          0.0,   0.0,     0.4,    -0.2,     0.2,     0.7,     0.0,
     *         -1.2,  -0.7,    -0.6/
c
c
      DATA MA,IYR,IPR/0,0,0/
      IF(MA.NE.1) GOTO 10
      IF(IY.NE.IYR) GOTO 30
      GOTO 130
10    MA=1
      KNM=15
C
      DO 20 N=1,11
         N2=2*N-1
         N2=N2*(N2-2)
         DO 20 M=1,N
            MN=N*(N-1)/2+M
20    REC(MN)=FLOAT((N-M)*(N+M-2))/FLOAT(N2)
C
30    IYR=IY
      IF (IYR.LT.1965) IYR=1965
      IF (IYR.GT.2000) IYR=2000
      IF (IY.NE.IYR.AND.IPR.EQ.0) write(*,999)IY,IYR
      IF (IYR.NE.IY) IPR=1
      IF (IYR.LT.1970) GOTO 50          !INTERPOLATE BETWEEN 1965 - 1970
      IF (IYR.LT.1975) GOTO 60          !INTERPOLATE BETWEEN 1970 - 1975
      IF (IYR.LT.1980) GOTO 70          !INTERPOLATE BETWEEN 1975 - 1980
      IF (IYR.LT.1985) GOTO 80          !INTERPOLATE BETWEEN 1980 - 1985
      IF (IYR.LT.1990) GOTO 90          !INTERPOLATE BETWEEN 1985 - 1990
      IF (IYR.LT.1995) GOTO 100         !INTERPOLATE BETWEEN 1990 - 1995
C
C
C       EXTRAPOLATE BETWEEN 1995 - 2000
C
      DT=FLOAT(IYR)-1995.
      DO 40 N=1,66
         G(N)=G95(N)
         H(N)=H95(N)
         IF (N.GT.45) GOTO 40
         G(N)=G(N)+DG95(N)*DT
         H(N)=H(N)+DH95(N)*DT
40    CONTINUE
      GOTO 300
C
C
C       INTERPOLATE BETWEEEN 1965 - 1970
C
50    F2=(IYR-1965)/5.
      F1=1.-F2
      DO 55 N=1,66
         G(N)=G65(N)*F1+G70(N)*F2
55       H(N)=H65(N)*F1+H70(N)*F2
      GOTO 300
C
C
C       INTERPOLATE BETWEEN 1970 - 1975
C
60    F2=(IYR-1970)/5.
      F1=1.-F2
      DO 65 N=1,66
         G(N)=G70(N)*F1+G75(N)*F2
65       H(N)=H70(N)*F1+H75(N)*F2
      GOTO 300
C
C
C       INTERPOLATE BETWEEN 1975 - 1980
C
70    F2=(IYR-1975)/5.
      F1=1.-F2
      DO 75 N=1,66
         G(N)=G75(N)*F1+G80(N)*F2
75       H(N)=H75(N)*F1+H80(N)*F2
      GOTO 300
C
C
C       INTERPOLATE BETWEEN 1980 - 1985
C
80    F2=(IYR-1980)/5.
      F1=1.-F2
      DO 85 N=1,66
         G(N)=G80(N)*F1+G85(N)*F2
85       H(N)=H80(N)*F1+H85(N)*F2
      GOTO 300
C
C
C       INTERPOLATE BETWEEN 1985 - 1990
C
90    F2=(IYR-1985)/5.
      F1=1.-F2
      DO 95 N=1,66
         G(N)=G85(N)*F1+G90(N)*F2
95       H(N)=H85(N)*F1+H90(N)*F2
      GOTO 300
C
C
C       INTERPOLATE BETWEEN 1990 - 1995
C
100   F2=(IYR-1990)/5.
      F1=1.-F2
      DO 105 N=1,66
         G(N)=G90(N)*F1+G95(N)*F2
105      H(N)=H90(N)*F1+H95(N)*F2
      GOTO 300
C
C
C    GET HERE WHEN COEFFICIENTS FOR APPROPRIATE IGRF MODEL HAVE BEEN ASSIGNED
C
300   S=1.
      DO 120 N=2,11
         MN=N*(N-1)/2+1
         S=S*FLOAT(2*N-3)/FLOAT(N-1)
         G(MN)=G(MN)*S
         H(MN)=H(MN)*S
         P=S
         DO 120 M=2,N
            AA=1.
            IF (M.EQ.2) AA=2.
            P=P*SQRT(AA*FLOAT(N-M+1)/FLOAT(N+M-2))
            MNN=MN+M-1
            G(MNN)=G(MNN)*P
120         H(MNN)=H(MNN)*P
C
130   IF(KNM.EQ.NM) GO TO 140
      KNM=NM
      K=KNM+1
140   PP=1./R
      P=PP
      DO 150 N=1,K
         P=P*PP
         A(N)=P
150      B(N)=P*N
      P=1.
      D=0.
      BBR=0.
      BBT=0.
      BBF=0.
      U=T
      CF=COS(F)
      SF=SIN(F)
      C=COS(U)
      S=SIN(U)
      BK=(S.LT.1.E-5)
      DO 200 M=1,K
         BM=(M.EQ.1)
         IF(BM) GOTO 160
         MM=M-1
         W=X
         X=W*CF+Y*SF
         Y=Y*CF-W*SF
         GOTO 170
160      X=0.
         Y=1.
170      Q=P
         Z=D
         BI=0.
         P2=0.
         D2=0.
         DO 190 N=M,K
            AN=A(N)
            MN=N*(N-1)/2+M
            E=G(MN)
            HH=H(MN)
            W=E*Y+HH*X
            BBR=BBR+B(N)*W*Q
            BBT=BBT-AN*W*Z
            IF(BM) GOTO 180
            QQ=Q
            IF(BK) QQ=Z
            BI=BI+AN*(E*X-HH*Y)*QQ
180         XK=REC(MN)
            DP=C*Z-S*Q-XK*D2
            PM=C*Q-XK*P2
            D2=Z
            P2=Q
            Z=DP
190        Q=PM
         D=S*D+C*P
         P=S*P
         IF(BM) GOTO 200
         BI=BI*MM
         BBF=BBF+BI
200   CONTINUE
C
      BR=BBR
      BT=BBT
      IF(BK) GOTO 210
      BF=BBF/S
      GOTO 220
210   IF(C.LT.0.) BBF=-BBF
      BF=BBF
220   CONTINUE
      RETURN
C
999   FORMAT(//1X,
     * 'IGRF WARNS:**** YEAR IS OUT OF INTERVAL 1965-2000: IY =',I5,/,
     *',        CALCULATIONS WILL BE DONE FOR IYR =',I5,' ****'//)
      END
C
C
c---------------------------------------------------------------------
c
       SUBROUTINE DIP(PS,X,Y,Z,BX,BY,BZ)
C
C  CALCULATES GSM COMPONENTS OF GEODIPOLE FIELD WITH THE DIPOLE MOMENT
C  CORRESPONDING TO THE EPOCH OF 1980.
C------------INPUT PARAMETERS:
C   PS - GEODIPOLE TILT ANGLE IN RADIANS, X,Y,Z - GSM COORDINATES IN RE
C------------OUTPUT PARAMETERS:
C   BX,BY,BZ - FIELD COMPONENTS IN GSM SYSTEM, IN NANOTESLA.
C
C
C                   AUTHOR: NIKOLAI A. TSYGANENKO
C                           INSTITUTE OF PHYSICS
C                           ST.-PETERSBURG STATE UNIVERSITY
C                           STARY PETERGOF 198904
C                           ST.-PETERSBURG
C                           RUSSIA
C
      IMPLICIT NONE
      SAVE
C
      REAL PS,X,Y,Z,BX,BY,BZ,PSI,SPS,CPS,P,U,V,T,Q
      INTEGER M

      DATA M,PSI/0,5./
      IF(M.EQ.1.AND.ABS(PS-PSI).LT.1.E-5) GOTO 1
      SPS=SIN(PS)
      CPS=COS(PS)
      PSI=PS
      M=1
  1   P=X**2
      U=Z**2
      V=3.*Z*X
      T=Y**2
      Q=30574./SQRT(P+T+U)**5
      BX=Q*((T+U-2.*P)*SPS-V*CPS)
      BY=-3.*Y*Q*(X*SPS+Z*CPS)
      BZ=Q*((P+T-2.*U)*CPS-V*SPS)
      RETURN
      END
c------------------------------------------------------------------------
c
      SUBROUTINE SUN(IYR,IDAY,IHOUR,MIN,ISEC,GST,SLONG,SRASN,SDEC)
C
C  CALCULATES FOUR QUANTITIES NECESSARY FOR COORDINATE TRANSFORMATIONS
C  WHICH DEPEND ON SUN POSITION (AND, HENCE, ON UNIVERSAL TIME AND SEASON)
C
C-------  INPUT PARAMETERS:
C  IYR,IDAY,IHOUR,MIN,ISEC -  YEAR, DAY, AND UNIVERSAL TIME IN HOURS, MINUTES,
C    AND SECONDS  (IDAY=1 CORRESPONDS TO JANUARY 1).
C
C-------  OUTPUT PARAMETERS:
C  GST - GREENWICH MEAN SIDEREAL TIME, SLONG - LONGITUDE ALONG ECLIPTIC
C  SRASN - RIGHT ASCENSION,  SDEC - DECLINATION  OF THE SUN (RADIANS)
C  THIS SUBROUTINE HAS BEEN COMPILED FROM: RUSSELL C.T., COSM.ELECTRO-
C  DYN., 1971, V.2,PP.184-196.
C
C
C                   AUTHOR: Gilbert D. Mead
C
C
        IMPLICIT NONE
      SAVE

        REAL GST,SLONG,SRASN,SDEC,RAD,T,VL,G,OBLIQ,SOB,SLP,SIND,
     1       COSD,SC

        INTEGER IYR,IDAY,IHOUR,MIN,ISEC

      DOUBLE PRECISION DJ,FDAY
      DATA RAD/57.295779513/
      IF(IYR.LT.1901.OR.IYR.GT.2099) RETURN
      FDAY=DFLOAT(IHOUR*3600+MIN*60+ISEC)/86400.D0
      DJ=365*(IYR-1900)+(IYR-1901)/4+IDAY-0.5D0+FDAY
      T=DJ/36525.
      VL=DMOD(279.696678+0.9856473354*DJ,360.D0)
      GST=DMOD(279.690983+.9856473354*DJ+360.*FDAY+180.,360.D0)/RAD
      G=DMOD(358.475845+0.985600267*DJ,360.D0)/RAD
      SLONG=(VL+(1.91946-0.004789*T)*SIN(G)+0.020094*SIN(2.*G))/RAD
      IF(SLONG.GT.6.2831853) SLONG=SLONG-6.2831853
      IF (SLONG.LT.0.) SLONG=SLONG+6.2831853
      OBLIQ=(23.45229-0.0130125*T)/RAD
      SOB=SIN(OBLIQ)
      SLP=SLONG-9.924E-5
C
C   THE LAST CONSTANT IS A CORRECTION FOR THE ANGULAR ABERRATION  DUE TO
C   THE ORBITAL MOTION OF THE EARTH
C
      SIND=SOB*SIN(SLP)
      COSD=SQRT(1.-SIND**2)
      SC=SIND/COSD
      SDEC=ATAN(SC)
      SRASN=3.141592654-ATAN2(COS(OBLIQ)/SOB*SC,-COS(SLP)/COSD)
      RETURN
      END
c-----------------------------------------------------------------------
c
      SUBROUTINE SPHCAR(R,TETA,PHI,X,Y,Z,J)
C
C   CONVERTS SPHERICAL COORDS INTO CARTESIAN ONES AND VICA VERSA
C    (TETA AND PHI IN RADIANS).
C
C                  J>0            J<0
C-----INPUT:   J,R,TETA,PHI     J,X,Y,Z
C----OUTPUT:      X,Y,Z        R,TETA,PHI
C
C
C                   AUTHOR: NIKOLAI A. TSYGANENKO
C                           INSTITUTE OF PHYSICS
C                           ST.-PETERSBURG STATE UNIVERSITY
C                           STARY PETERGOF 198904
C                           ST.-PETERSBURG
C                           RUSSIA
C
        IMPLICIT NONE
      SAVE

        REAL R,TETA,PHI,X,Y,Z,SQ

        INTEGER J

      IF(J.GT.0) GOTO 3
      SQ=X**2+Y**2
      R=SQRT(SQ+Z**2)
      IF (SQ.NE.0.) GOTO 2
      PHI=0.
      IF (Z.LT.0.) GOTO 1
      TETA=0.
      RETURN
  1   TETA=3.141592654
      RETURN
  2   SQ=SQRT(SQ)
      PHI=ATAN2(Y,X)
      TETA=ATAN2(SQ,Z)
      IF (PHI.LT.0.) PHI=PHI+6.28318531
      RETURN
  3   SQ=R*SIN(TETA)
      X=SQ*COS(PHI)
      Y=SQ*SIN(PHI)
      Z=R*COS(TETA)
      RETURN
      END
c--------------------------------------------------------------------
c
      SUBROUTINE BSPCAR(TETA,PHI,BR,BTET,BPHI,BX,BY,BZ)
C   CALCULATES CARTESIAN FIELD COMPONENTS FROM SPHERICAL ONES
C-----INPUT:   TETA,PHI - SPHERICAL ANGLES OF THE POINT IN RADIANS
C              BR,BTET,BPHI -  SPHERICAL COMPONENTS OF THE FIELD
C-----OUTPUT:  BX,BY,BZ - CARTESIAN COMPONENTS OF THE FIELD
C
C
C                   AUTHOR: NIKOLAI A. TSYGANENKO
C                           INSTITUTE OF PHYSICS
C                           ST.-PETERSBURG STATE UNIVERSITY
C                           STARY PETERGOF 198904
C                           ST.-PETERSBURG
C                           RUSSIA
C
        IMPLICIT NONE
      SAVE

        REAL TETA,PHI,BR,BTET,BPHI,BX,BY,BZ,S,C,SF,CF,BE

      S=SIN(TETA)
      C=COS(TETA)
      SF=SIN(PHI)
      CF=COS(PHI)
      BE=BR*S+BTET*C
      BX=BE*CF-BPHI*SF
      BY=BE*SF+BPHI*CF
      BZ=BR*C-BTET*S
      RETURN
      END
c
C----------------------------------------------------------------------
C
      SUBROUTINE RECALC(IYR,IDAY,IHOUR,MIN,ISEC)
C
C
C  AN ALTERNATIVE VERSION OF THE SUBROUTINE RECALC FROM THE GEOPACK PACKAGE
C  BASED ON A DIFFERENT APPROACH TO DERIVATION OF ROTATION MATRIX ELEMENTS
C
C THIS SUBROUTINE WORKS BY 20% FASTER THAN THE ORIGINAL RECALC AND IS EASIER
C         TO UNDERSTAND
C
C   ################################################
C   #  WRITTEN BY  N.A. TSYGANENKO ON DEC.1, 1991  #
C   ################################################
C
c
c       Modified by:    Mauricio Peredo and Nikolai Tsyganenko
c                       Hughes STX at NASA/GSFC Code 695
c                       September 1992, March 4, April 16, 1996
c
c   Modified    (1) to accept dates up to year 2000 and updated IGRF
C                           coefficients for 1985-1995, and
c               (2) to provide matrix elements for the GEI-GEO  transformation
c
C   OTHER SUBROUTINES CALLED BY THIS ONE: SUN
C
C     IYR = YEAR NUMBER (FOUR DIGITS)
C     IDAY = DAY OF YEAR (DAY 1 = JAN 1)
C     IHOUR = HOUR OF DAY (00 TO 23)
C     MIN = MINUTE OF HOUR (00 TO 59)
C     ISEC = SECONDS OF DAY(00 TO 59)
C
        IMPLICIT NONE
      SAVE

        REAL ST0,CT0,SL0,CL0,CTCL,STCL,CTSL,STSL,SFI,CFI,SPS,CPS,
     1       SHI,CHI,HI,PSI,XMUT,A11,A21,A31,A12,A22,A32,A13,A23,
     2       A33,DS3,F2,F1,G10,G11,H11,DT,SQ,SQQ,SQR,S1,S2,
     3       S3,CGST,SGST,DIP1,DIP2,DIP3,Y1,Y2,Y3,Y,Z1,Z2,Z3,DJ,
     4       T,OBLIQ,DZ1,DZ2,DZ3,DY1,DY2,DY3,EXMAGX,EXMAGY,EXMAGZ,
     5       EYMAGX,EYMAGY,GST,SLONG,SRASN,SDEC,BA(6)

        INTEGER IYR,IDAY,IHOUR,MIN,ISEC,K,IY,IDE,IYE,IPR

       COMMON /GEOPACK/ ST0,CT0,SL0,CL0,CTCL,STCL,CTSL,STSL,SFI,CFI,SPS,
     * CPS,SHI,CHI,HI,PSI,XMUT,A11,A21,A31,A12,A22,A32,A13,A23,A33,DS3,
     * K,IY,CGST,SGST,BA
c
      DATA IYE,IDE,IPR/3*0/
      IF (IYR.EQ.IYE.AND.IDAY.EQ.IDE) GOTO 5
C
C  IYE AND IDE ARE THE CURRENT VALUES OF YEAR AND DAY NUMBER
C
      IY=IYR
      IDE=IDAY
      IF(IY.LT.1965) IY=1965
      IF(IY.GT.2000) IY=2000
C
C  WE ARE RESTRICTED BY THE INTERVAL 1965-2000,
C  FOR WHICH THE IGRF COEFFICIENTS ARE KNOWN; IF IYR IS OUTSIDE THIS INTERVAL
C   THE SUBROUTINE GIVES A WARNING (BUT DOES NOT REPEAT IT AT THE NEXT CALLS)
C
      IF(IY.NE.IYR.AND.IPR.EQ.0) PRINT 10,IYR,IY
      IF(IY.NE.IYR) IPR=1
      IYE=IY
C
C  LINEAR INTERPOLATION OF THE GEODIPOLE MOMENT COMPONENTS BETWEEN THE
C  VALUES FOR THE NEAREST EPOCHS:
C
        IF (IY.LT.1970) THEN                            !1965-1970
           F2=(FLOAT(IY)+FLOAT(IDAY)/365.-1965.)/5.
           F1=1.D0-F2
           G10=30334.*F1+30220.*F2
           G11=-2119.*F1-2068.*F2
           H11=5776.*F1+5737.*F2
        ELSEIF (IY.LT.1975) THEN                        !1970-1975
           F2=(FLOAT(IY)+FLOAT(IDAY)/365.-1970.)/5.
           F1=1.D0-F2
           G10=30220.*F1+30100.*F2
           G11=-2068.*F1-2013.*F2
           H11=5737.*F1+5675.*F2
        ELSEIF (IY.LT.1980) THEN                        !1975-1980
           F2=(DFLOAT(IY)+DFLOAT(IDAY)/365.-1975.)/5.
           F1=1.D0-F2
           G10=30100.*F1+29992.*F2
           G11=-2013.*F1-1956.*F2
           H11=5675.*F1+5604.*F2
        ELSEIF (IY.LT.1985) THEN                        !1980-1985
           F2=(FLOAT(IY)+FLOAT(IDAY)/365.-1980.)/5.
           F1=1.D0-F2
           G10=29992.*F1+29873.*F2
           G11=-1956.*F1-1905.*F2
           H11=5604.*F1+5500.*F2
        ELSEIF (IY.LT.1990) THEN                        !1985-1990
           F2=(FLOAT(IY)+FLOAT(IDAY)/365.-1985.)/5.
           F1=1.D0-F2
           G10=29873.*F1+29775.*F2
           G11=-1905.*F1-1848.*F2
           H11=5500.*F1+5406.*F2
        ELSEIF (IY.LT.1995) THEN                        !1990-1995
           F2=(FLOAT(IY)+FLOAT(IDAY)/365.-1990.)/5.
           F1=1.D0-F2
           G10=29775.*F1+29682.*F2
           G11=-1848.*F1-1789.*F2
           H11=5406.*F1+5318.*F2
        ELSE                                            !1995-2000
C
C   LINEAR EXTRAPOLATION BEYOND 1995 BY USING SECULAR VELOCITY COEFFICIENTS:
C
           DT=FLOAT(IY)+FLOAT(IDAY)/365.-1995.
           G10=29682.-17.6*DT
           G11=-1789.+13.0*DT
           H11=5318.-18.3*DT
        ENDIF
C
C  NOW CALCULATE THE COMPONENTS OF THE UNIT VECTOR EzMAG IN GEO COORD.SYSTEM:
C   SIN(TETA0)*COS(LAMBDA0), SIN(TETA0)*SIN(LAMBDA0), AND COS(TETA0)
C         ST0 * CL0                ST0 * SL0                CT0
C
      SQ=G11**2+H11**2
      SQQ=SQRT(SQ)
      SQR=SQRT(G10**2+SQ)
      SL0=-H11/SQQ
      CL0=-G11/SQQ
      ST0=SQQ/SQR
      CT0=G10/SQR
      STCL=ST0*CL0
      STSL=ST0*SL0
      CTSL=CT0*SL0
      CTCL=CT0*CL0
C
C      THE CALCULATIONS ARE TERMINATED IF ONLY GEO-MAG TRANSFORMATION
C       IS TO BE DONE  (IHOUR>24 IS THE AGREED CONDITION FOR THIS CASE):
C
   5   IF (IHOUR.GT.24) RETURN
C
      CALL SUN(IY,IDAY,IHOUR,MIN,ISEC,GST,SLONG,SRASN,SDEC)
C
C  S1,S2, AND S3 ARE THE COMPONENTS OF THE UNIT VECTOR EXGSM=EXGSE IN THE
C   SYSTEM GEI POINTING FROM THE EARTH'S CENTER TO THE SUN:
C
      S1=COS(SRASN)*COS(SDEC)
      S2=SIN(SRASN)*COS(SDEC)
      S3=SIN(SDEC)
      CGST=COS(GST)
      SGST=SIN(GST)
C
C  DIP1, DIP2, AND DIP3 ARE THE COMPONENTS OF THE UNIT VECTOR EZSM=EZMAG
C   IN THE SYSTEM GEI:
C
      DIP1=STCL*CGST-STSL*SGST
      DIP2=STCL*SGST+STSL*CGST
      DIP3=CT0
C
C  NOW CALCULATE THE COMPONENTS OF THE UNIT VECTOR EYGSM IN THE SYSTEM GEI
C   BY TAKING THE VECTOR PRODUCT D x S AND NORMALIZING IT TO UNIT LENGTH:
C
      Y1=DIP2*S3-DIP3*S2
      Y2=DIP3*S1-DIP1*S3
      Y3=DIP1*S2-DIP2*S1
      Y=SQRT(Y1*Y1+Y2*Y2+Y3*Y3)
      Y1=Y1/Y
      Y2=Y2/Y
      Y3=Y3/Y
C
C   THEN IN THE GEI SYSTEM THE UNIT VECTOR Z = EZGSM = EXGSM x EYGSM = S x Y
C    HAS THE COMPONENTS:
C
      Z1=S2*Y3-S3*Y2
      Z2=S3*Y1-S1*Y3
      Z3=S1*Y2-S2*Y1
C
C    THE VECTOR EZGSE (HERE DZ) IN GEI HAS THE COMPONENTS (0,-SIN(DELTA),
C     COS(DELTA)) = (0.,-0.397823,0.917462); HERE DELTA = 23.44214 DEG FOR
C   THE EPOCH 1978 (SEE THE BOOK BY GUREVICH OR OTHER ASTRONOMICAL HANDBOOKS).
C    HERE THE MOST ACCURATE TIME-DEPENDENT FORMULA IS USED:
C
      DJ=FLOAT(365*(IY-1900)+(IY-1901)/4 +IDAY)-0.5+FLOAT(ISEC)/86400.
      T=DJ/36525.
      OBLIQ=(23.45229-0.0130125*T)/57.2957795
      DZ1=0.
      DZ2=-SIN(OBLIQ)
      DZ3=COS(OBLIQ)
C
C  THEN THE UNIT VECTOR EYGSE IN GEI SYSTEM IS THE VECTOR PRODUCT DZ x S :
C
      DY1=DZ2*S3-DZ3*S2
      DY2=DZ3*S1-DZ1*S3
      DY3=DZ1*S2-DZ2*S1
C
C   THE ELEMENTS OF THE MATRIX GSE TO GSM ARE THE SCALAR PRODUCTS:
C   CHI=EM22=(EYGSM,EYGSE), SHI=EM23=(EYGSM,EZGSE), EM32=(EZGSM,EYGSE)=-EM23,
C     AND EM33=(EZGSM,EZGSE)=EM22
C
      CHI=Y1*DY1+Y2*DY2+Y3*DY3
      SHI=Y1*DZ1+Y2*DZ2+Y3*DZ3
      HI=ASIN(SHI)
C
C    TILT ANGLE: PSI=ARCSIN(DIP,EXGSM)
C
      SPS=DIP1*S1+DIP2*S2+DIP3*S3
      CPS=SQRT(1.-SPS**2)
      PSI=ASIN(SPS)
C
C    THE ELEMENTS OF THE MATRIX MAG TO SM ARE THE SCALAR PRODUCTS:
C CFI=GM22=(EYSM,EYMAG), SFI=GM23=(EYSM,EXMAG); THEY CAN BE DERIVED AS FOLLOWS:
C
C IN GEO THE VECTORS EXMAG AND EYMAG HAVE THE COMPONENTS (CT0*CL0,CT0*SL0,-ST0)
C  AND (-SL0,CL0,0), RESPECTIVELY.    HENCE, IN GEI THE COMPONENTS ARE:
C  EXMAG:    CT0*CL0*COS(GST)-CT0*SL0*SIN(GST)
C            CT0*CL0*SIN(GST)+CT0*SL0*COS(GST)
C            -ST0
C  EYMAG:    -SL0*COS(GST)-CL0*SIN(GST)
C            -SL0*SIN(GST)+CL0*COS(GST)
C             0
C  THE COMPONENTS OF EYSM IN GEI WERE FOUND ABOVE AS Y1, Y2, AND Y3;
C  NOW WE ONLY HAVE TO COMBINE THE QUANTITIES INTO SCALAR PRODUCTS:
C
      EXMAGX=CT0*(CL0*CGST-SL0*SGST)
      EXMAGY=CT0*(CL0*SGST+SL0*CGST)
      EXMAGZ=-ST0
      EYMAGX=-(SL0*CGST+CL0*SGST)
      EYMAGY=-(SL0*SGST-CL0*CGST)
      CFI=Y1*EYMAGX+Y2*EYMAGY
      SFI=Y1*EXMAGX+Y2*EXMAGY+Y3*EXMAGZ
C
      XMUT=(ATAN2(SFI,CFI)+3.1415926536)*3.8197186342
C
C  THE ELEMENTS OF THE MATRIX GEO TO GSM ARE THE SCALAR PRODUCTS:
C
C   A11=(EXGEO,EXGSM), A12=(EYGEO,EXGSM), A13=(EZGEO,EXGSM),
C   A21=(EXGEO,EYGSM), A22=(EYGEO,EYGSM), A23=(EZGEO,EYGSM),
C   A31=(EXGEO,EZGSM), A32=(EYGEO,EZGSM), A33=(EZGEO,EZGSM),
C
C   ALL THE UNIT VECTORS IN BRACKETS ARE ALREADY DEFINED IN GEI:
C
C  EXGEO=(CGST,SGST,0), EYGEO=(-SGST,CGST,0), EZGEO=(0,0,1)
C  EXGSM=(S1,S2,S3),  EYGSM=(Y1,Y2,Y3),   EZGSM=(Z1,Z2,Z3)
C                                                           AND  THEREFORE:
C
      A11=S1*CGST+S2*SGST
      A12=-S1*SGST+S2*CGST
      A13=S3
      A21=Y1*CGST+Y2*SGST
      A22=-Y1*SGST+Y2*CGST
      A23=Y3
      A31=Z1*CGST+Z2*SGST
      A32=-Z1*SGST+Z2*CGST
      A33=Z3
C
 10   FORMAT(//1X,
     * '****RECALC WARNS:  YEAR IS OUT OF INTERVAL 1965-2000: IYR=',I4,
     * /,6X,'CALCULATIONS WILL BE DONE FOR IYR=',I4,/)
      RETURN
      END
C--------------------------------------------------------------------------
C
      SUBROUTINE GEOMAG(XGEO,YGEO,ZGEO,XMAG,YMAG,ZMAG,J,IYR)
C
C CONVERTS GEOGRAPHIC (GEO) TO DIPOLE (MAG) COORDINATES OR VICA VERSA.
C IYR IS YEAR NUMBER (FOUR DIGITS).
C
C                           J>0                J<0
C-----INPUT:  J,XGEO,YGEO,ZGEO,IYR   J,XMAG,YMAG,ZMAG,IYR
C-----OUTPUT:    XMAG,YMAG,ZMAG        XGEO,YGEO,ZGEO
C
C
C                   AUTHOR: NIKOLAI A. TSYGANENKO
C                           INSTITUTE OF PHYSICS
C                           ST.-PETERSBURG STATE UNIVERSITY
C                           STARY PETERGOF 198904
C                           ST.-PETERSBURG
C                           RUSSIA
C
        IMPLICIT NONE
      SAVE

        REAL XGEO,YGEO,ZGEO,XMAG,YMAG,ZMAG,ST0,CT0,SL0,CL0,CTCL,
     1       STCL,CTSL,STSL,AB(19),BB(8)

        INTEGER J,IYR,K,IY,II

      COMMON /GEOPACK/ ST0,CT0,SL0,CL0,CTCL,STCL,CTSL,STSL,AB,K,IY,BB
      DATA II/1/
      IF(IYR.EQ.II) GOTO 1
      II=IYR
      CALL RECALC(II,0,25,0,0)
  1   CONTINUE
      IF(J.LT.0) GOTO 2
      XMAG=XGEO*CTCL+YGEO*CTSL-ZGEO*ST0
      YMAG=YGEO*CL0-XGEO*SL0
      ZMAG=XGEO*STCL+YGEO*STSL+ZGEO*CT0
      RETURN
  2   XGEO=XMAG*CTCL-YMAG*SL0+ZMAG*STCL
      YGEO=XMAG*CTSL+YMAG*CL0+ZMAG*STSL
      ZGEO=ZMAG*CT0-XMAG*ST0
      RETURN
      END
c
c######################################################################
c
      SUBROUTINE GEIGEO(XGEI,YGEI,ZGEI,XGEO,YGEO,ZGEO,J)
C
C CONVERTS EQUATORIAL INERTIAL (GEI) TO GEOGRAPHICAL (GEO) COORDS
C   OR VICA VERSA.
C                    J>0                J<0
C-----INPUT: J,XGEI,YGEI,ZGEI    J,XGEO,YGEO,ZGEO
C----OUTPUT:   XGEO,YGEO,ZGEO      XGEI,YGEI,ZGEI
C  ATTENTION:  SUBROUTINE  RECALC  MUST BE CALLED BEFORE GEIGEO IN TWO CASES:
C     /A/  BEFORE THE FIRST CALL OF GEIGEO
C     /B/  IF THE CURRENT VALUES OF IYEAR,IDAY,IHOUR,MIN,ISEC ARE DIFFERENT
C          FROM THOSE IN THE PRECEDING  CALL  OF GEIGEO
C
C                   AUTHOR: NIKOLAI A. TSYGANENKO
C                           HUGHES STX CORPORATION
C                           NASA GODDARD SPACE FLIGHT CENTER
C                           GREENBELT, MD 20771
C                           USA
C
        IMPLICIT NONE
      SAVE

        REAL XGEI,YGEI,ZGEI,XGEO,YGEO,ZGEO,A,CGST,SGST,B
        INTEGER J,KKKK

      COMMON /GEOPACK/ A(27),KKKK(2),CGST,SGST,B(6)
      IF(J.LT.0) GOTO 1
      XGEO=XGEI*CGST+YGEI*SGST
      YGEO=YGEI*CGST-XGEI*SGST
      ZGEO=ZGEI
      RETURN
  1   XGEI=XGEO*CGST-YGEO*SGST
      YGEI=YGEO*CGST+XGEO*SGST
      ZGEI=ZGEO
      RETURN
      END
c
C------------------------------------------------------------------------
C
      SUBROUTINE MAGSM(XMAG,YMAG,ZMAG,XSM,YSM,ZSM,J)
C
C CONVERTS DIPOLE (MAG) TO SOLAR MAGNETIC (SM) COORDINATES OR VICA VERSA
C
C                    J>0              J<0
C-----INPUT: J,XMAG,YMAG,ZMAG     J,XSM,YSM,ZSM
C----OUTPUT:    XSM,YSM,ZSM       XMAG,YMAG,ZMAG
C  ATTENTION:  SUBROUTINE  RECALC  MUST BE CALLED BEFORE MAGSM IN TWO CASES:
C     /A/  BEFORE THE FIRST USE OF MAGSM
C     /B/  IF THE CURRENT VALUES OF IYEAR,IDAY,IHOUR,MIN,ISEC ARE DIFFERENT
C          FROM THOSE IN THE PRECEDING CALL OF  MAGSM
C
C                   AUTHOR: NIKOLAI A. TSYGANENKO
C                           INSTITUTE OF PHYSICS
C                           ST.-PETERSBURG STATE UNIVERSITY
C                           STARY PETERGOF 198904
C                           ST.-PETERSBURG
C                           RUSSIA
C
        IMPLICIT NONE
      SAVE

        REAL XMAG,YMAG,ZMAG,XSM,YSM,ZSM,SFI,CFI,A(8),B(7),
     1       AB(10),BA(8)

        INTEGER J,K,IY

      COMMON /GEOPACK/ A,SFI,CFI,B,AB,K,IY,BA
      IF (J.LT.0) GOTO 1
      XSM=XMAG*CFI-YMAG*SFI
      YSM=XMAG*SFI+YMAG*CFI
      ZSM=ZMAG
      RETURN
  1   XMAG=XSM*CFI+YSM*SFI
      YMAG=YSM*CFI-XSM*SFI
      ZMAG=ZSM
      RETURN
      END
C-------------------------------------------------------------------------
C
       SUBROUTINE GSMGSE(XGSM,YGSM,ZGSM,XGSE,YGSE,ZGSE,J)
C
C CONVERTS SOLAR MAGNETOSPHERIC (GSM) TO SOLAR ECLIPTICAL (GSE) COORDS
C   OR VICA VERSA.
C                    J>0                J<0
C-----INPUT: J,XGSM,YGSM,ZGSM    J,XGSE,YGSE,ZGSE
C----OUTPUT:   XGSE,YGSE,ZGSE      XGSM,YGSM,ZGSM
C  ATTENTION:  SUBROUTINE  RECALC  MUST BE CALLED BEFORE GSMGSE IN TWO CASES:
C     /A/  BEFORE THE FIRST CALL OF GSMGSE
C     /B/  IF THE CURRENT VALUES OF IYEAR,IDAY,IHOUR,MIN,ISEC ARE DIFFERENT
C          FROM THOSE IN THE PRECEDING  CALL  OF GSMGSE
C
C
C                   AUTHOR: NIKOLAI A. TSYGANENKO
C                           INSTITUTE OF PHYSICS
C                           ST.-PETERSBURG STATE UNIVERSITY
C                           STARY PETERGOF 198904
C                           ST.-PETERSBURG
C                           RUSSIA
C
        IMPLICIT NONE
       SAVE

        REAL XGSM,YGSM,ZGSM,XGSE,YGSE,ZGSE,SHI,CHI,
     1       A(12),AB(13),BA(8)
        INTEGER J,K,IY

      COMMON /GEOPACK/ A,SHI,CHI,AB,K,IY,BA
      IF(J.LT.0) GOTO 1
      XGSE=XGSM
      YGSE=YGSM*CHI-ZGSM*SHI
      ZGSE=YGSM*SHI+ZGSM*CHI
      RETURN
1     XGSM=XGSE
      YGSM=YGSE*CHI+ZGSE*SHI
      ZGSM=ZGSE*CHI-YGSE*SHI
      RETURN
      END
C--------------------------------------------------------------------------
C
       SUBROUTINE SMGSM(XSM,YSM,ZSM,XGSM,YGSM,ZGSM,J)
C
C CONVERTS SOLAR MAGNETIC (SM) TO SOLAR MAGNETOSPHERIC (GSM) COORDINATES
C   OR VICA VERSA.
C                  J>0                 J<0
C-----INPUT: J,XSM,YSM,ZSM        J,XGSM,YGSM,ZGSM
C----OUTPUT:  XGSM,YGSM,ZGSM       XSM,YSM,ZSM
C
C  ATTENTION:  SUBROUTINE RECALC MUST BE CALLED BEFORE SMGSM IN TWO CASES:
C     /A/  BEFORE THE FIRST USE OF SMGSM
C     /B/  IF THE CURRENT VALUES OF IYEAR,IDAY,IHOUR,MIN,ISEC ARE DIFFERENT
C          FROM THOSE IN THE PRECEDING CALL OF SMGSM
C
C
C                AUTHOR: NIKOLAI A. TSYGANENKO
C                        INSTITUTE OF PHYSICS
C                        ST.-PETERSBURG  UNIVERSITY
C                        STARY PETERGOF 198904
C                        ST.-PETERSBURG
C                        U.S.S.R.
C
        IMPLICIT NONE
       SAVE

        REAL XSM,YSM,ZSM,XGSM,YGSM,ZGSM,SPS,CPS,A(10),B(15),AB(8)
        INTEGER J,K,IY

      COMMON /GEOPACK/ A,SPS,CPS,B,K,IY,AB
      IF (J.LT.0) GOTO 1
      XGSM=XSM*CPS+ZSM*SPS
      YGSM=YSM
      ZGSM=ZSM*CPS-XSM*SPS
      RETURN
  1   XSM=XGSM*CPS-ZGSM*SPS
      YSM=YGSM
      ZSM=XGSM*SPS+ZGSM*CPS
      RETURN
      END
C---------------------------------------------------------------------------
C
      SUBROUTINE GEOGSM(XGEO,YGEO,ZGEO,XGSM,YGSM,ZGSM,J)
C
C CONVERTS GEOGRAPHIC TO SOLAR MAGNETOSPHERIC COORDINATES OR VICA VERSA.
C
C                   J>0                   J<0
C----- INPUT:  J,XGEO,YGEO,ZGEO    J,XGSM,YGSM,ZGSM
C---- OUTPUT:    XGSM,YGSM,ZGSM      XGEO,YGEO,ZGEO
C  ATTENTION:  SUBROUTINE  RECALC  MUST BE CALLED BEFORE GEOGSM IN TWO CASES:
C     /A/  BEFORE THE FIRST USE OF GEOGSM
C     /B/  IF THE CURRENT VALUES OF IYEAR,IDAY,IHOUR,MIN,ISEC  ARE DIFFERENT
C            FROM THOSE IN THE PREVIOUS CALL OF THIS SUBROUTINE
C
C
C                   AUTHOR: NIKOLAI A. TSYGANENKO
C                           INSTITUTE OF PHYSICS
C                           ST.-PETERSBURG STATE UNIVERSITY
C                           STARY PETERGOF 198904
C                           ST.-PETERSBURG
C                           RUSSIA
C
        IMPLICIT NONE
      SAVE

        REAL XGEO,YGEO,ZGEO,XGSM,YGSM,ZGSM,A11,A21,A31,A12,
     1       A22,A32,A13,A23,A33,D,AA(17),B(8)
        INTEGER J,K,IY

      COMMON /GEOPACK/ AA,A11,A21,A31,A12,A22,A32,A13,A23,A33,D,K,IY,B
      IF (J.LT.0) GOTO 1
      XGSM=A11*XGEO+A12*YGEO+A13*ZGEO
      YGSM=A21*XGEO+A22*YGEO+A23*ZGEO
      ZGSM=A31*XGEO+A32*YGEO+A33*ZGEO
      RETURN
  1   XGEO=A11*XGSM+A21*YGSM+A31*ZGSM
      YGEO=A12*XGSM+A22*YGSM+A32*ZGSM
      ZGEO=A13*XGSM+A23*YGSM+A33*ZGSM
      RETURN
      END
C-------------------------------------------------------------------------
C
      SUBROUTINE RHAND(X,Y,Z,R1,R2,R3,IOPT,PARMOD,EXNAME)
C
C COMPUTES RIGHT HAND EXPRESSIONS IN THE GEOMAGNETIC FIELD LINE EQUATION
C      (a subsidiary subroutine for the subr. STEP)
C
C                   AUTHOR: NIKOLAI A. TSYGANENKO
C                           INSTITUTE OF PHYSICS
C                           ST.-PETERSBURG STATE UNIVERSITY
C                           STARY PETERGOF 198904
C                           ST.-PETERSBURG
C                           RUSSIA
C
        IMPLICIT NONE
      SAVE

        REAL X,Y,Z,R1,R2,R3,PSI,DS3,BX,BY,BZ,A(15),AA(10),BB(8),
     1       XG,YG,ZG,R,T,F,BR,BT,BF,FX,FY,FZ,HX,HY,HZ,B,PARMOD(10)
        INTEGER IOPT,K,IY
        EXTERNAL EXNAME

      COMMON /GEOPACK/ A,PSI,AA,DS3,K,IY,BB
      CALL EXNAME(IOPT,PARMOD,PSI,X,Y,Z,BX,BY,BZ)
      IF (K.EQ.0) GOTO 1
      CALL GEOGSM(XG,YG,ZG,X,Y,Z,-1)
      CALL SPHCAR(R,T,F,XG,YG,ZG,-1)
      CALL IGRF(IY,K,R,T,F,BR,BT,BF)
      CALL BSPCAR(T,F,BR,BT,BF,FX,FY,FZ)
      CALL GEOGSM(FX,FY,FZ,HX,HY,HZ,1)
      GOTO 2
  1   CALL DIP(PSI,X,Y,Z,HX,HY,HZ)
  2   BX=BX+HX
      BY=BY+HY
      BZ=BZ+HZ
      B=DS3/SQRT(BX**2+BY**2+BZ**2)
      R1=BX*B
      R2=BY*B
      R3=BZ*B
      RETURN
      END
C------------------------------------------------------------------------
C
      SUBROUTINE STEP(N,X,Y,Z,DS,ERRIN,IOPT,PARMOD,EXNAME)
C
C RE-CALCULATES COORDS X,Y,Z FOR ONE STEP ALONG FIELD LINE. N IS MAXIMUM
C ORDER OF HARMONICS IN MAIN FIELD EXPANSION, DS IS STEP SIZE, ERRIN IS
C PERMISSIBLE ERROR VALUE, IOPT AND EXNAME - SEE COMMENTS TO SUBROUTINE TRACE
C  ALL THE PARAMETERS ARE INPUT ONES; OUTPUT IS THE RENEWED TRIPLET X,Y,Z
C
C                   AUTHOR: NIKOLAI A. TSYGANENKO
C                           INSTITUTE OF PHYSICS
C                           ST.-PETERSBURG STATE UNIVERSITY
C                           STARY PETERGOF 198904
C                           ST.-PETERSBURG
C                           RUSSIA
C
        IMPLICIT NONE
      SAVE

        REAL X,Y,Z,DS,ERRIN,DS3,R11,R12,R13,R21,R22,R23,
     1  R31,R32,R33,R41,R42,R43,R51,R52,R53,ERRCUR,A(26),B(8),PARMOD(10)
        INTEGER N,IOPT,K,IY

      COMMON /GEOPACK/ A,DS3,K,IY,B
      EXTERNAL EXNAME
      K=N
  1   DS3=-DS/3.
      CALL RHAND(X,Y,Z,R11,R12,R13,IOPT,PARMOD,EXNAME)
      CALL RHAND(X+R11,Y+R12,Z+R13,R21,R22,R23,IOPT,PARMOD,EXNAME)
      CALL RHAND(X+.5*(R11+R21),Y+.5*(R12+R22),Z+.5*
     *(R13+R23),R31,R32,R33,IOPT,PARMOD,EXNAME)
      CALL RHAND(X+.375*(R11+3.*R31),Y+.375*(R12+3.*R32
     *),Z+.375*(R13+3.*R33),R41,R42,R43,IOPT,PARMOD,EXNAME)
      CALL RHAND(X+1.5*(R11-3.*R31+4.*R41),Y+1.5*(R12-
     *3.*R32+4.*R42),Z+1.5*(R13-3.*R33+4.*R43),
     *R51,R52,R53,IOPT,PARMOD,EXNAME)
      ERRCUR=ABS(R11-4.5*R31+4.*R41-.5*R51)+ABS(R12-4.5*R32+4.*R42-.5*
     *R52)+ABS(R13-4.5*R33+4.*R43-.5*R53)
      IF (ERRCUR.LT.ERRIN) GOTO 2
      DS=DS*.5
      GOTO 1
  2   X=X+.5*(R11+4.*R41+R51)
      Y=Y+.5*(R12+4.*R42+R52)
      Z=Z+.5*(R13+4.*R43+R53)
      IF(ERRCUR.LT.ERRIN*.04.AND.ABS(DS).LT.1.33) DS=DS*1.5
      RETURN
      END
C----------------------------------------------------------------------
C
      SUBROUTINE TRACE(XI,YI,ZI,DIR,RLIM,R0,IHARM,NP,IOPT,PARMOD,EXNAME,
     *XF,YF,ZF,XX,YY,ZZ,L)
C
C   TRACES FIELD LINE FROM ARBITRARY POINT OF SPACE UP TO THE EARTH
C   SURFACE OR UP TO MODEL LIMITING BOUNDARY.
C-------------- INPUT PARAMETERS:
C   XI,YI,ZI - GSM COORDS OF INITIAL POINT (IN EARTH RADII),
C   DIR - SIGN OF TRACING DIRECTION: IF DIR=1. THEN ANTIPARALLEL TO
C     B VECTOR (E.G. FROM NORTHERN TO SOUTHERN CONJUGATE POINT),
C     AND IF DIR=-1. THEN PARALLEL TO B.
C   R0 IS RADIUS OF SPHERE (IN RE) FOR WHICH 'LANDING POINT' COORDINATES
C     XF,YF,ZF  SHOULD BE CALCULATED
C   RLIM - UPPER GEOCENTRIC DISTANCE WHICH LIMITS TRACING REGION.
C   IHARM - MAXIMAL ORDER OF SPH.HARMONICS IN THE MAIN FIELD EXPANSION,
C     IT DEPENDS ON THE CURRENT VERSION OF INTERNAL FIELD MODEL. SUB-
C     ROUTINE IGRF OF THIS ISSUE CORRESPONDS TO IHARM=10. IF THE MAIN
C     FIELD SHOULD BE ASSUMED TO BE PURELY DIPOLAR, THEN PUT IHARM=0
C     AND SPECIFY VALUE OF GEODIPOLE TILT ANGLE PSI (IN RADIANS)
C     IN THE 16-TH ELEMENT OF THE COMMON BLOCK BEFORE CALLING TRACE.
C     OTHERWISE CALL SUBROUTINE RECALC BEFORE CALLING TRACE.
C   NP - UPPER ESTIMATE OF NUMBER OF STEPS ALONG THE FIELD LINE
C     (OF THE ORDER OF SEVERAL HUNDREDS).
C   IOPT - SPECIFIES OPTION OF EXTERNAL FIELD MODEL (E.G. KP-LEVEL OF
C     DISTURBANCE)
C    PARMOD -  A 10-ELEMENT ARRAY CONTAINING OTHER POSSIBLE MODEL
C     PARAMETERS; THE CONCRETE MEANING OF ITS COMPONENTS DEPENDS ON A
C     SPECIFIC VERSION OF THE EXTERNAL FIELD MODEL
C   EXNAME - NAME OF THE EXTRATERRESTRIAL FIELD MODEL SUBROUTINE
C
C-------------- OUTPUT PARAMETERS:
C   XF,YF,ZF - GSM COORDS OF FINAL POINT
C   XX,YY,ZZ - ARRAYS (LENGTH NP) CONTAINING COORDS OF FIELD LINE POINTS
C   L - ACTUAL NUMBER OF FIELD LINE POINTS. IF L EXCEEDS NP, TRACING
C     TERMINATES, AND A WARNING IS DISPLAYED
C
C
C                   AUTHOR: NIKOLAI A. TSYGANENKO
C                           INSTITUTE OF PHYSICS
C                           ST.-PETERSBURG STATE UNIVERSITY
C                           STARY PETERGOF 198904
C                           ST.-PETERSBURG
C                           RUSSIA
C
        IMPLICIT NONE
      SAVE

        REAL XI,YI,ZI,DIR,RLIM,R0,XF,YF,ZF,DD,ERR,DS,AA(26),BB(8),
     1       X,Y,Z,AL,R1,R2,R3,AD,RR,RYZ,R,FC,XR,YR,ZR, PARMOD(10)
        INTEGER IHARM,NP,IOPT,L,K1,K2,J,I

      REAL XX(NP),YY(NP),ZZ(NP)
      COMMON /GEOPACK/ AA,DD,K1,K2,BB
      EXTERNAL EXNAME
 10   FORMAT(//,1X,'**** COMPUTATIONS IN THE SUBROUTINE TRACE',
     *' ARE TERMINATED: NP IS TOO SMALL ****'//)
      J=IHARM
      ERR=0.0005
      L=0
      DS=0.5*DIR
      X=XI
      Y=YI
      Z=ZI
      DD=DIR
      K1=IHARM
      AL=0.
      CALL RHAND(X,Y,Z,R1,R2,R3,IOPT,PARMOD,EXNAME)
      AD=SIGN(0.01,X*R1+Y*R2+Z*R3)
      RR=SQRT(X**2+Y**2+Z**2)+AD
  1   L=L+1
      IF(L.GT.NP) GOTO 7
      XX(L)=X
      YY(L)=Y
      ZZ(L)=Z
      RYZ=Y**2+Z**2
      R2=X**2+RYZ
      R=SQRT(R2)
      IF(R.GT.RLIM.OR.RYZ.GT.900..OR.X.GT.15.) GOTO 8
      IF(R.LT.R0.AND.RR.GT.R) GOTO 6
      IF(R.GE.RR) GOTO 5
      IF(R.GT.5.) GOTO 5
      IF(R.GE.3.) GOTO 3
      FC=0.2
      IF(R-R0.LT.0.05) FC=0.05
      AL=FC*(R-R0+0.2)
      DS=DIR*AL
      GOTO 4
  3   DS=DIR
      AL=1.
  4   XR=X
      YR=Y
      ZR=Z
  5   RR=R
      IF(IHARM.NE.0) J=1+10./(R-AL)
      IF(J.GT.IHARM) J=IHARM
      CALL STEP(J,X,Y,Z,DS,ERR,IOPT,PARMOD,EXNAME)
      GOTO 1
  6   R1=(R0-R)/(RR-R)
      X=X-(X-XR)*R1
      Y=Y-(Y-YR)*R1
      Z=Z-(Z-ZR)*R1
      GOTO 8
  7   write(*,10)
      L=NP
      RETURN
  8   XF=X
      YF=Y
      ZF=Z
      DO 9 I=L,NP
      XX(I)=XF
      YY(I)=YF
  9   ZZ(I)=ZF
      RETURN
      END

c----------------------------------------------------------------------------
cAuthors and curators:

cNikolai Tsyganenko - Hughes STX Corporation (ys2nt@lepvax.gsfc.nasa.gov)

cDavid P. Stern - NASA/GSFC Code 695 (u5dps@lepvax.gsfc.nasa.gov)

cMauricio Peredo - Hughes STX Corporation (peredo@istp1.gsfc.nasa.gov)

cLast updated: July 3, 1996
