/*
   Pathetic Writer
   Copyright (C) 1997, 1998  Ulric Eriksson <ulric@edu.stockholm.se>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA.
 */

/*
   Module name:    window.c

   This module handles windows: creating, destroying, printing on windows.
*/

#include <stdio.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Xatom.h>
#include <X11/Shell.h>
#include <X11/Xmu/Atoms.h>
#include <X11/Xmu/StdSel.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Scrollbar.h>
#include <X11/Xaw/MenuButton.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/SmeBSB.h>
#include <X11/Xaw/SmeLine.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Toggle.h>
#include <X11/Xaw/Paned.h>
#include <X11/xpm.h>

#include "../common/common.h"
#include "../common/cmalloc.h"

#include "../common/bitmaps/rbm.xbm"
#include "../common/bitmaps/pw_fg.xpm"

#include "../xcommon/xcommon.h"
#include "../xcommon/xfonts.h"
#include "../xcommon/embed.h"
#include "../xcommon/Canvas.h"
#include "../xcommon/Richtext.h"
#include "../xcommon/Pw.h"
#include "../xcommon/Hsep.h"
#include "../xcommon/DragAndDrop.h"
#include "../xcommon/plugin.h"
#include "../xcommon/icon.h"

#include "../pw/pw.h"
#include "xpw.h"

#include "../siod/siod.h"

String fallback_resources[] = {
#include "app-defaults.h"
	NULL
};

Widget topLevel, topbox, form, gridpane,
        text1, label1, label2, label3;

/* The toolbar */
Widget tbOpen, tbSave, tbView, tbPrint, tbHelp;

Widget btnFont, mnuFont, btnSize, mnuSize, btnStyle, mnuStyle,
        btnColor, mnuColor;
Widget cmdBold, cmdItalic, cmdUline, cmdHLeft, cmdHCenter, cmdHRight;
Widget cmdVTop, cmdVBottom;
static Widget shortcuts;

typedef struct {
        Boolean plugin;
} AppData;

static AppData app_data;
static XtAppContext app_context;

#define XtNplugin "plugin"
#define XtCPlugin "Plugin"

static XtResource resources[] = {
        {
                XtNplugin,
                XtCPlugin,
                XtRBoolean,
                sizeof(Boolean),
                XtOffsetOf(AppData, plugin),
                XtRImmediate,
                (XtPointer)False
        }
};

static XrmOptionDescRec options[] = {
        {"-plugin", "*plugin", XrmoptionNoArg, "True"}
};

window *w_list;

int pr_scr_flag, pr_line_flag;

Display *display;

Window root;
static GC cursor_gc, cell_gc, clear_gc;
static Cursor grid_cursor;
static unsigned long blockbg, noblockbg;

Atom wm_delete_window;	/* Atom sent to destroy a window */
extern Atom target_atom;	/* used for selection */

void activate_window(window *w)
{
	char b[256];

	w_list = w;

	strcpy(b, "Pathetic Writer: ");
	strncat(b, w->buf->name, 200);
	XtVaSetValues(topLevel, XtNtitle, b, (char *)0);
	XtVaSetValues(w->ui->grid,
		XtNrichtextData, w->buf,
		(char *)0);
	XtSetKeyboardFocus(topLevel, w->ui->grid);
}
static void chomp(char *p)
{
	if ((p = strchr(p, '\n'))) *p = '\0';
}

void draw_input(Display * display, char *text)
{
	char b[256];
	sprintf(b, "[%d,%d - %d,%d]",
		w_list->blku.row, w_list->blku.col,
		w_list->blkl.row, w_list->blkl.col);
	label_set(label1, b);
	label_set(label3, "PW");
}

void draw_status(Display * display, char *text)
{
	chomp(text);
	label_set(label2, text);
	XFlush(display);
}

/*
   void llpr(char *p)
   Prints the string p on the bottom line of the screen.  If p is empty and
   the last string printed was also empty, the string isn't printed.
*/
void llpr(char *p)
{
	static int isclear = FALSE;

	if (isclear && p[0] == '\0')
		return;
	isclear = (p[0] == '\0');

	draw_status(display, p);
}

static void draw_vbar(Display * display, window *w)
{
	double float_value;

	float_value = w->top.row;
	float_value /= BUFFER_ROWS;

	XawScrollbarSetThumb(w->ui->vscroll, float_value, 0.0);
}

static void draw_hbar(Display * display, window *w)
{
	double float_value;

	float_value = w->top.col;
	float_value /= BUFFER_COLS;

	XawScrollbarSetThumb(w->ui->hscroll, float_value, 0.0);
}

static void draw_scrollbars(Display * display, window *w)
{
	draw_vbar(display, w);
	draw_hbar(display, w);
}

static int line_start(buffer *buf, int row)
{
	int x = 0;
	int w = paper_width-left_margin-right_margin;
	int hadj;

	hadj = ret_hadj(buf, row);

	if (hadj == HADJ_CENTER)
		x = (w-line_width(buf, row, line_length(buf, row))) / 2;
	else if (hadj == HADJ_RIGHT)
		x = w-line_width(buf, row, line_length(buf, row));

	return x+left_margin;
}

/* This function is a dog: it copies from the grid which may well be
   partially obscured during the copy. As a result it doesn't work.
   It should use the richtext_pixmap function instead. */
Pixmap draw_snapshot(void)
{
	int x, y;
	unsigned int width, height, border_width, depth;
	Window cell_win, root;
	Pixmap bitmap;
	GC gc;
	unsigned long valuemask = 0;
	XGCValues values;

	cell_win = XtWindow(w_list->ui->grid);
	XGetGeometry(display, cell_win, &root,
		&x, &y, &width, &height, &border_width, &depth);
	bitmap = XCreatePixmap(display, cell_win, width, height, 1);
	gc = XCreateGC(display, bitmap, valuemask, &values);
	XCopyPlane(display, cell_win, bitmap, gc, 0, 0, width, height,
			0, 0, 1);
	XFreeGC(display, gc);

	return bitmap;
}

static int row_height(XtPointer p, int r)
{
	buffer *b = (buffer *)p;
	if (r > b->used_lines) return 20;
	return b->text[r].height;
}

static int row_hadj(XtPointer p, int r)
{
	return ret_hadj((buffer *)p, r);
}

static int row_style(XtPointer p, int r)
{
	return ret_style((buffer *)p, r);
}

static rich_char *row_text(XtPointer p, int r)
{
	buffer *b = (buffer *)p;
	if (r > b->used_lines) return NULL;
	return b->text[r].p;
}

/* This will be the Redisplay function in the RichtextWidget */
void draw_buffer(Display *display, window *w)
{
	XtVaSetValues(w->ui->grid,
		XtNrichtextData, w->buf,
		XtNrichtextTopRow, w->top.row,
		XtNrichtextTopCol, w->top.col,
		XtNrichtextSelectTopRow, w->blku.row,
		XtNrichtextSelectTopCol, w->blku.col,
		XtNrichtextSelectBottomRow, w->blkl.row,
		XtNrichtextSelectBottomCol, w->blkl.col,
		XtNrichtextPointRow, w->point_pos.row,
		XtNrichtextPointCol, w->point_pos.col,
		XtNrichtextRedisplay, True,
		(char *)0);
	/* did top move? */
	/* unlike Siag we have no rownum or topnum, so it doesn't matter */
}

static GC get_gc(Display * display, Window wind,
       unsigned long fg, unsigned long bg, Font font)
{
	unsigned long valuemask = 0/*GCFunction | GCForeground | GCBackground |
		GCLineWidth | GCFont*/;
	XGCValues values;
	GC gc = XCreateGC(display, wind, valuemask, &values);

	XSetForeground(display, gc, fg);
	XSetBackground(display, gc, bg);
	if (font != -1)
		XSetFont(display, gc, font);
	return gc;
}

window *find_window_by_widget(Widget wdg)
{
	window *w = w_list;
	do {
		if (w->ui->viewport == wdg || w->ui->grid == wdg ||
			w->ui->vscroll == wdg || w->ui->hscroll == wdg)
			return w;
		w = w->next;
	} while (w != w_list);
	return NULL;
}

void free_window(window *w)
{
	window *pw;

	for (pw = w_list; pw->next != w && pw->next != pw; pw = pw->next);
	pw->next = w->next;

	if (w_list == w) w_list = w_list->next;
	if (w_list == w) w_list = NULL;
	XtDestroyWidget(w->ui->viewport);
	cfree(w);
}

static void expose_ruler(Widget w, window *win)
{
	int i;

	for (i = left_margin; i < paper_width-right_margin; i += 36)
		XDrawLine(XtDisplay(w), XtWindow(w), cell_gc,
				i, 0, i, height_get(w));
}

/* used for the XtNtablePluginCoords resource */
static void plugin_coordinates(Widget w, XtPointer p, long *x, long *y)
{
        buffer *b = (buffer *)p;
        int n, ph;
        *x = *y = 0;
        ph = plugin_find_by_widget(w);
        if (ph == -1) return;
        n = buffer_plugin2index(b, ph);
        if (n == -1) return;
        buffer_global_coords(b, b->plugin[n].row, b->plugin[n].col, x, y);
}

window *new_window(buffer *b, window *prev)
{
	Dimension totalwidth, formheight, w1, h1;
	int d;	/* distance between widgets in form */
	window *w;

	w = (window *)cmalloc(sizeof(window));

	if (w == NULL) return NULL;	/* can't really happen */

	w->ui = (pw_ui *)cmalloc(sizeof(pw_ui));
	if (w->ui == NULL) {
		cfree(w);
		return NULL;
	}

	w->buf = b;
	set_point(w, make_position(1, 0));
	w->blku.row = w->blku.col = -1;
	w->blkl.row = w->blkl.col = -1;
	w->top = w->point_pos;

	if (prev == NULL) prev = w;
	else w->next = prev->next;
	prev->next = w;

	/* Figure out how big the new form should be.			*/
	/* The total width must be the width of the gridpane.		*/
	XtVaGetValues(gridpane,
		XtNwidth, &totalwidth, (char *)0);

	/* The form height is whatever we get, but if it is too small
	we cannot create the new window.				*/
	formheight = 100;
	w->ui->viewport = XtVaCreateManagedWidget("viewport",
		pwWidgetClass, gridpane,
		XtNwidth, totalwidth,
		(char *)0);
	XtVaGetValues(w->ui->viewport,
		XtNheight, &formheight, XtNdefaultDistance, &d, (char *)0);

	w1 = totalwidth-20-3*d;
	h1 = 20;

	w->ui->ruler = XtVaCreateManagedWidget("ruler",
		canvasWidgetClass, w->ui->viewport,
		/*XtNwidth, w1,*/
		/*XtNheight, h1,*/
		XtNcanvasExpose, expose_ruler,
		/*XtNtop, XawChainTop,*/
		/*XtNleft, XawChainLeft,*/
		/*XtNright, XawChainRight,*/
		/*XtNbottom, XawChainTop,*/
		(char *)0);

	h1 = formheight-height_get(w->ui->ruler)-20-3*d;
	w->ui->grid = XtVaCreateManagedWidget("grid",
		richtextWidgetClass, w->ui->viewport,
		/*XtNwidth, w1,*/
		/*XtNheight, h1,*/
		/* set nonchanging resources here */
		XtNrichtextData, (XtPointer)b,
		XtNrichtextRowHeight, row_height,
		XtNrichtextAdjHoriz, row_hadj,
		XtNrichtextStyle, row_style,
		XtNrichtextText, row_text,
		/* as well as changing ones */
		XtNrichtextTopRow, w->top.row,
		XtNrichtextTopCol, w->top.col,
		XtNrichtextSelectTopRow, w->blku.row,
		XtNrichtextSelectTopCol, w->blku.col,
		XtNrichtextSelectBottomRow, w->blkl.row,
		XtNrichtextSelectBottomCol, w->blkl.col,
		XtNrichtextPointRow, w->point_pos.row,
		XtNrichtextPointCol, w->point_pos.col,
		XtNrichtextPluginCoords, plugin_coordinates,
		/*XtNfromVert, w->ui->ruler,*/
		(char *)0);
	XDefineCursor(display, XtWindow(w->ui->grid), grid_cursor);
	h1 = height_get(w->ui->ruler)+height_get(w->ui->grid)+d;
	w->ui->vscroll = XtVaCreateManagedWidget("vscroll",
		scrollbarWidgetClass, w->ui->viewport,
		/*XtNheight, h1,*/
		/*XtNfromHoriz, w->ui->grid,*/ (char *)0);
	w1 = width_get(w->ui->grid);
	w->ui->hscroll = XtVaCreateManagedWidget("hscroll",
		scrollbarWidgetClass, w->ui->viewport,
		/*XtNwidth, w1,*/
		/*XtNfromVert, w->ui->grid,*/ (char *)0);
	XtAddCallback(w->ui->vscroll, XtNjumpProc, vscroll_jump, NULL);
	XtAddCallback(w->ui->vscroll, XtNscrollProc, vscroll_scroll, NULL);

	XtAddCallback(w->ui->hscroll, XtNjumpProc, hscroll_jump, NULL);
	XtAddCallback(w->ui->hscroll, XtNscrollProc, hscroll_scroll, NULL);

	XtVaGetValues(w->ui->viewport,
		XtNheight, &formheight, (char *)0);

	return w;
}

int remove_window(window *w)
{
	
	if (w == w->next) return FALSE;
	free_window(w);
	return TRUE;
}

int split_window(window *w)
{
	window *w2 = new_window(w->buf, w);

	if (w2 == NULL) return FALSE;
	w2->point_pos = w->point_pos;
	w2->top = w->top;
	return TRUE;
}

/* stuff snarfed from main.c */

static void set_point_rc(int row, int col)
{
        position p;

        p.row = row;
        p.col = col;
        set_point(w_list, p);
}

static void set_mark_rc(int row, int col)
{
        position p;

        p.row = row;
        p.col = col;
        set_point(w_list, p);
}

static void set_block(int row1, int col1, int row2, int col2)
{
        int ur = w_list->blku.row, uc = w_list->blku.col;
        int lr = w_list->blkl.row, lc = w_list->blkl.col;

        w_list->blku.row = row1;
        w_list->blkl.row = row2;
        w_list->blku.col = col1;
        w_list->blkl.col = col2;

        /* Redraw if any change */
        if (ur != w_list->blku.row || uc != w_list->blku.col ||
                lr != w_list->blkl.row || lc != w_list->blkl.col) {
                pr_scr_flag = TRUE;
        }

        /* Become selection owner */
        /* this function should be integrated with the one in cmds.c */
        if (XtOwnSelection(w_list->ui->grid, XA_PRIMARY,
		CurrentTime, convert_proc,
                lose_ownership_proc, transfer_done_proc) == False) {
                XtWarning("PW: failed to become selection owner\n");
                w_list->blku.row = w_list->blku.col = -1;
                w_list->blkl.row = w_list->blkl.col = -1;
        }
}


#ifndef ABS
#define ABS(a) ((a)>0?(a):-(a))
#endif

void GridButtonAction(Widget w, XEvent *event, String *params, Cardinal *n)
{
        int col, row;
        int x, y;

        x = event->xbutton.x;
        y = event->xbutton.y;
        hide_cur(w_list);
        activate_window(find_window_by_widget(w));
        get_coords_cell(w_list, w_list->top.row, w_list->top.col,
                        &row, &col, x, y);
        if (*n < 1 || !strcmp(params[0], "point")) {
                set_point_rc(row, col);
        } else if (!strcmp(params[0], "mark")) {
                set_mark_rc(row, col);
        } else if (!strcmp(params[0], "block")) {
                set_block(row, col,
                        get_mark(w_list).row, get_mark(w_list).col);
                pr_scr_flag = TRUE;
        } else if (!strcmp(params[0], "paste")) {
                set_point_rc(row, col);
                XtGetSelectionValue(w, XA_PRIMARY, target_atom,
                        requestor_callback, event, event->xbutton.time);
                pr_scr_flag = TRUE;
        } else if (!strcmp(params[0], "select")) {
                int r1 = w_list->blku.row, c1 = w_list->blku.col;
                int r2 = w_list->blkl.row, c2 = w_list->blkl.col;
                /* is the selection set to something already? */
                if (r1 == -1 || c1 == -1 || r2 == -1 || c2 == -1) {
			int r = get_point(w_list).row;
			int c = get_point(w_list).col;
			if (row < r || (row == r && col < c)) {
				set_block(row, col, r, c);
			} else {
				set_block(r, c, row, col);
			}
                } else {
			if (row < r1 || (row == r1 && col <= c1)) {
				/* grow at the start */
				r1 = row, c1 = col;
			} else if (row > r2 || (row == r2 && col >= c2)) {
				/* grow at the end */
				r2 = row, c2 = col;
			} else if (ABS(r1-row) < ABS(r2-row)) {
				/* shrink at the start */
				r1 = row, c1 = col;
			} else {
				/* shrink at the end */
				r2 = row, c2 = col;
			}
                        set_block(r1, c1, r2, c2);
                }
        } else if (!strcmp(params[0], "adjust")) {
		int r = get_point(w_list).row, c = get_point(w_list).col;
		if (row < r || (row == r && col < c)) {
			set_block(row, col, r, c);
		} else {
			set_block(r, c, row, col);
		}
        }
        show_cur(w_list);
}

void SelfInsertCharAction(Widget w, XEvent *event, String *params, Cardinal *n)
{
        int count, bufsiz = 10;
        unsigned char buf[12];
        KeySym keysym;
        XKeyEvent kevent;

        kevent = event->xkey;
        hide_cur(w_list);
        count = XLookupString(&kevent, (char *)buf, bufsiz, &keysym, NULL);
        if (keysym >= ' ' && count == 1) {
                buf[count] = '\0';
                ins_text(w_list->buf, w_list->point_pos, buf, w_list->current_fmt);
                rebreak_line(w_list->buf, w_list->point_pos.row);
                w_list->point_pos.col += count;
                /* point position may now be hosed */
                if (w_list->point_pos.col >
                                line_length(w_list->buf, w_list->point_pos.row))
 {
                        w_list->point_pos.col -=
                                line_length(w_list->buf, w_list->point_pos.row);
                        w_list->point_pos.col -= 1;     /* we deleted a space too */
                        w_list->point_pos.row++;
                }
                /* and if that didn't help... */
                if (w_list->point_pos.col >
                                line_length(w_list->buf, w_list->point_pos.row))
 {
                        w_list->point_pos.col =
                                line_length(w_list->buf, w_list->point_pos.row);
                }
                pr_line_flag = TRUE;
        }
        show_cur(w_list);
}

static LISP ltooltip_mode(LISP newmode)
{
        tooltip_mode_set(get_c_long(newmode), label1);
        return NIL;
}

static void siaghelp_action(Widget w, XEvent *event,
        String *params, Cardinal *num_params)
{
        char b[256];

        sprintf(b, "file://localhost%s/pw/docs/%s", siagdocs, params[0]);
        if (!fork()) {
                execlp(siaghelp, "Siaghelp", b, (char *)0);
                exit(0);
        }
}

static void
execute_callback(Widget w, XtPointer client_data, XtPointer call_data)
{
        char *b = (char *) client_data;

        execute(b);
}

void vscroll_jump(Widget w, XtPointer client_data, XtPointer call_data)
{
        float top;
        int gridtop;

        hide_cur(w_list);
        activate_window(find_window_by_widget(w));
        XtVaGetValues(w, XtNtopOfThumb, &top, (char *)0);
        gridtop = top*BUFFER_ROWS;
        if (gridtop < 1) gridtop = 1;
        w_list->point_pos.row = w_list->top.row = gridtop;
        pr_scr_flag = TRUE;
        show_cur(w_list);
}

void vscroll_scroll(Widget w, XtPointer client_data, XtPointer call_data)
{
        int i = (int) call_data;
        Dimension length;

        activate_window(find_window_by_widget(w));
        XtVaGetValues(w, XtNlength, &length, (char *)0);
        if (i < 0) {
                if ((length / -i) > 15) {
                        execute("(scroll-char-up)");
                } else {
                        execute("(scroll-up)");
                }
        } else {
                if ((length / i) > 15) {
                        execute("(scroll-char-down)");
                } else {
                        execute("(scroll-down)");
                }
        }
}

void hscroll_jump(Widget w, XtPointer client_data, XtPointer call_data)
{
        float top;
        int gridtop;

        hide_cur(w_list);
        activate_window(find_window_by_widget(w));
        XtVaGetValues(w, XtNtopOfThumb, &top, (char *)0);
        gridtop = top*BUFFER_COLS;
        if (gridtop < 1) gridtop = 1;
        w_list->point_pos.col = w_list->top.col = gridtop;
        pr_scr_flag = TRUE;
        show_cur(w_list);
}

void hscroll_scroll(Widget w, XtPointer client_data, XtPointer call_data)
{
        int i = (int) call_data;
        Dimension length;

        activate_window(find_window_by_widget(w));
        XtVaGetValues(w, XtNlength, &length, (char *)0);
        if (i < 0) {
                if ((length / -i) > 15) {
                        execute("(scroll-char-left)");
                } else {
                        execute("(scroll-left)");
                }
        } else {
                if ((length / i) > 15) {
                        execute("(scroll-char-right)");
                } else {
                        execute("(scroll-right)");
                }
        }
}

static struct {
        char *label;
        Widget button, menu;
} *menubar;

static int menucount = 0;

static void make_menu(char *label)
{
        char button_name[80];

        sprintf(button_name, "btn%s", label);
	menubar = crealloc(menubar, (menucount+1)*(sizeof *menubar));
        menubar[menucount].label = cstrdup(label);
        menubar[menucount].button = XtVaCreateManagedWidget(button_name,
                menuButtonWidgetClass, topbox,
                XtNtop, XawChainTop,
                XtNleft, XawChainLeft,
                XtNright, XawChainLeft,
                XtNbottom, XawChainTop,
		XtNheight, 24,
		(char *)0);
        if (menucount) {
                XtVaSetValues(menubar[menucount].button,
                        XtNfromHoriz, menubar[menucount-1].button, (char *)0);
        }
        menubar[menucount].menu = XtVaCreatePopupShell("menu",
                simpleMenuWidgetClass, menubar[menucount].button, (char *)0);
	label_set(menubar[menucount].button, label);
        menucount++;
}

static void init_menu()
{
        make_menu("File");
        make_menu("Edit");
        make_menu("Block");
        make_menu("Format");
        make_menu("Plugin");
        make_menu("Window");
        make_menu("Tools");
        make_menu("Help");
}

static Widget find_menu_by_name(char *label)
{
        int i;

	if (!cstrcasecmp("Shortcuts", label))
		return shortcuts;

        for (i = 0; i < menucount; i++) {
                if (!cstrcasecmp(menubar[i].label, label))
                        return menubar[i].menu;
        }
        return NULL;
}

static LISP add_menu_entry(LISP menu, LISP label, LISP function)
{
        Widget entry;
        Widget menuw = find_menu_by_name(get_c_string(menu));
        if (!menuw) {
#if 0
                fprintf(stderr, "No menu! Bailing out.\n");
#endif
                return NIL;
        }

        if (!strcmp(get_c_string(label), "-"))  /* line pane */
                entry = XtCreateManagedWidget("-",
                                smeLineObjectClass,
                                menuw,
                                NULL, 0);
        else {
                entry = XtVaCreateManagedWidget(get_c_string(function),
                                smeBSBObjectClass, menuw,
                                (char *)0, 0);
                XtAddCallback(entry,
                        XtNcallback, execute_callback,
			cstrdup(get_c_string(function)));
		label_set(entry, get_c_string(label));
        }
        return NIL;
}

static struct {
	char *label, *sublabel;
	Widget entry, menu;
} *submenus;

static int submenucount = 0;

static Widget find_submenu_by_name(char *label, char *sublabel)
{
        int i;

        for (i = 0; i < submenucount; i++) {
                if (!cstrcasecmp(submenus[i].label, label) &&
                                !cstrcasecmp(submenus[i].sublabel, sublabel))
                        return submenus[i].menu;
        }
        return None;
}

static Widget find_submenu_by_entry(Widget entry)
{
        int i;

        for (i = 0; i < submenucount; i++) {
                if (entry == submenus[i].entry)
                        return submenus[i].menu;
        }
        return None;
}

static LISP add_submenu(LISP label, LISP sublabel)
{
        Widget menuw;
        static Pixmap rbm = None;

        menuw = find_menu_by_name(get_c_string(label));

        if (!menuw) {
#if 0
                fprintf(stderr, "add_submenu: No menu! Bailing out.\n");
#endif
                return NIL;
        }
        if (rbm == None) {
                rbm = XCreatePixmapFromBitmapData(XtDisplay(menuw),
                        XDefaultRootWindow(XtDisplay(menuw)),
                        rbm_bits, rbm_width, rbm_height, 1, 0, 1);
        }
        submenus = crealloc(submenus, (submenucount+1)*(sizeof *submenus));
        submenus[submenucount].label = cstrdup(get_c_string(label));
        submenus[submenucount].sublabel = cstrdup(get_c_string(sublabel));
        submenus[submenucount].entry =
                XtVaCreateManagedWidget(get_c_string(sublabel),
                        smeBSBObjectClass, menuw,
                        XtNlabel, translate(get_c_string(sublabel)),
                        XtNrightBitmap, rbm,
                        (char *)0);
        submenus[submenucount].menu = XtVaCreatePopupShell("submenu",
                simpleMenuWidgetClass, menuw, (char *)0);
        submenucount++;
        return NIL;
}

static LISP add_submenu_entry(LISP menu, LISP submenu,
                                LISP label, LISP function)
{
        Widget entry, menuw;

#if 0
        if (app_data.grid_only) return NIL;
#endif

        menuw = find_submenu_by_name(get_c_string(menu),
                                get_c_string(submenu));
        if (!menuw) {
#if 0
                fprintf(stderr, "add_submenu_entry: No menu! Bailing out.\n");
#endif
                return NIL;
        }

        if (!strcmp(get_c_string(label), "-")) {        /* line pane */
                entry = XtCreateManagedWidget("-",
                        smeLineObjectClass,
                        menuw,
                        NULL, 0);
        } else {
                entry = XtVaCreateManagedWidget(get_c_string(function),
                        smeBSBObjectClass, menuw,
                        XtNlabel, translate(get_c_string(label)),
                        (char *)NULL);
                XtAddCallback(entry,
                        XtNcallback, execute_callback,
                        cstrdup(get_c_string(function)));
        }
        return NIL;
}


static Widget tbNew, tbCopy;    /* needed as fromVert and fromHoriz */

static Widget make_toggle(char *name, char *cmd, Widget pw, 
        Widget below, Widget after, char *pm)
{
        Widget w;
        XpmAttributes xa;
        Pixmap pm_return;
        int result;
        XpmColorSymbol symbol;
        Pixel color;
        char fn[1024];

        if (pm[0] == '/') strcpy(fn, pm);
        else sprintf(fn, "%s/common/bitmaps/%s", siaghome, pm);

        XtVaGetValues(pw, XtNbackground, &color, (char *)0);
        xa.closeness = 40000;
        xa.exactColors = False;
        xa.valuemask = XpmCloseness | XpmExactColors;

        if (pw == None) return None;

        w = XtVaCreateManagedWidget(name,
                toggleWidgetClass, pw, (char *)NULL);
        symbol.name = NULL;
        symbol.value = "none";
        symbol.pixel = color;
        xa.colorsymbols = &symbol;
        xa.numsymbols = 1;
        xa.valuemask |= XpmColorSymbols;

        result = XpmReadFileToPixmap(XtDisplay(w),
                XRootWindowOfScreen(XtScreen(w)),
                fn, &pm_return, NULL, &xa);

        if (result != XpmSuccess) {
                fprintf(stderr, "XpmReadFileToPixmap returns %s for %s\n",
                        XpmGetErrorString(result), fn);
                return NULL;
        }

        XtVaSetValues(w,
                XtNfromVert, below,
                XtNtop, XawChainTop,
                XtNleft, XawChainLeft,
                XtNright, XawChainLeft,
                XtNbottom, XawChainTop,
                XtNbitmap, pm_return,
                (char *)0);
        if (after) {
                XtVaSetValues(w,
                        XtNfromHoriz, after, (char *)NULL);
        }
        XtAddCallback(w,
                XtNcallback, execute_callback, (XtPointer)cmd);
        return w;
}

static Widget make_command(char *name, char *cmd, Widget pw, 
        Widget below, Widget after, char *pm)
{
        Widget w;
        XpmAttributes xa;
        Pixmap pm_return;
        int result;
        XpmColorSymbol symbol;
        Pixel color;
        char fn[1024];

        if (pm[0] == '/') strcpy(fn, pm);
        else sprintf(fn, "%s/common/bitmaps/%s", siaghome, pm);

        XtVaGetValues(pw, XtNbackground, &color, (char *)0);
        xa.closeness = 40000;
        xa.exactColors = False;
        xa.valuemask = XpmCloseness | XpmExactColors;
        if (pw == None) return None;

        w = XtVaCreateManagedWidget(name,
                commandWidgetClass, pw, (char *)NULL);
        symbol.name = NULL;
        symbol.value = "none";
        symbol.pixel = color;
        xa.colorsymbols = &symbol;
        xa.numsymbols = 1;
        xa.valuemask |= XpmColorSymbols;

        result = XpmReadFileToPixmap(XtDisplay(w),
                XRootWindowOfScreen(XtScreen(w)),
                fn, &pm_return, NULL, &xa);

        if (result != XpmSuccess) {
                fprintf(stderr, "XpmReadFileToPixmap returns %s for %s\n",
                        XpmGetErrorString(result), fn);
                return NULL;
        }

        XtVaSetValues(w,
                XtNfromVert, below,
                XtNtop, XawChainTop,
                XtNleft, XawChainLeft,
                XtNright, XawChainLeft,
                XtNbottom, XawChainTop,
                XtNbitmap, pm_return,
                (char *)0);

        if (after) {
                XtVaSetValues(w,
                        XtNfromHoriz, after, (char *)NULL);
        }
        XtAddCallback(w,
                XtNcallback, execute_callback, (XtPointer)cmd);
        return w;
}


#define BELOW_WIDGET1 hsep1
#define BELOW_WIDGET2 hsep2
static Widget hsep1, hsep2, hsep3;

/* The toolbar */
static void init_toolbar()
{
        Widget tbOpen, tbSave, tbView, tbPrint, tbHelp;

        hsep1 = XtVaCreateManagedWidget("hsep1",
                hsepWidgetClass, topbox,
                XtNfromVert, menubar[0].button,
                XtNtop, XawChainTop,
                XtNleft, XawChainLeft,
                XtNright, XawChainRight,
                XtNbottom, XawChainTop,
                (char *)0);

        tbNew = make_command("tbNew", "(new-pw)",
                topbox, BELOW_WIDGET1, NULL, "new.xpm");

        hsep2 = XtVaCreateManagedWidget("hsep2",
                hsepWidgetClass, topbox,
                XtNfromVert, tbNew,
                XtNtop, XawChainTop,
                XtNleft, XawChainLeft,
                XtNright, XawChainRight,
                XtNbottom, XawChainTop,
                (char *)0);

        tbOpen = make_command("tbOpen", "(load-buffer)",
                topbox, BELOW_WIDGET1, tbNew, "fld_open.xpm");
        tbSave = make_command("tbSave", "(save-buffer-as)",
                topbox, BELOW_WIDGET1, tbOpen, "floppy3.xpm");
        tbView = make_command("tbView", "(preview)",
                topbox, BELOW_WIDGET1, tbSave, "preview.xpm");
        tbPrint = make_command("tbPrint", "(print)",
                topbox, BELOW_WIDGET1, tbView, "printer.xpm");
        tbHelp = make_command("tbHelp", "(help-contents)",
                topbox, BELOW_WIDGET1, tbPrint, "info.xpm");
        tbCopy = make_command("tbCopy", "(help-copyright)",
                topbox, BELOW_WIDGET1, tbHelp, "copyright.xpm");
}

static void init_toggle()
{
        cmdBold = make_toggle("cmdBold", "(toggle-fontflag BOLD)",
                topbox, BELOW_WIDGET2, btnColor, "bold.xpm");
        cmdItalic = make_toggle("cmdItalic", "(toggle-fontflag ITALIC)",
                topbox, BELOW_WIDGET2, cmdBold, "italic.xpm");
        cmdUline = make_toggle("cmdUline", "(toggle-fontflag ULINE)",
                topbox, BELOW_WIDGET2, cmdItalic, "uchar.xpm");
        cmdHLeft = make_toggle("cmdHLeft", "(change-hadjust HADJ_LEFT)",
                topbox, BELOW_WIDGET1, tbCopy, "hleft.xpm");
        cmdHCenter = make_toggle("cmdHCenter", "(change-hadjust HADJ_CENTER)",
                topbox, BELOW_WIDGET1, cmdHLeft, "hcenter.xpm");
        cmdHRight = make_toggle("cmdHRight", "(change-hadjust HADJ_RIGHT)",
                topbox, BELOW_WIDGET1, cmdHCenter, "hright.xpm");
        cmdVTop = make_toggle("cmdVTop", "(change-vadjust VADJ_TOP)",
                topbox, BELOW_WIDGET2, cmdUline, "vtop.xpm");
        cmdVBottom = make_toggle("cmdVBottom", "(change-vadjust VADJ_BOTTOM)",
                topbox, BELOW_WIDGET2, cmdVTop, "vbottom.xpm");
}

void font_menu(Widget w, char *label, char *cmd)
{
        Widget entry = XtVaCreateManagedWidget(cmd,
                        smeBSBObjectClass, w,
                        (char *)0);
        XtAddCallback(entry,
                XtNcallback, execute_callback, cmd);
	label_set(entry, label);
}

void setup_buttons()
{
        btnFont = XtVaCreateManagedWidget("btnFont",
                menuButtonWidgetClass, topbox,
                XtNtop, XawChainTop,
                XtNleft, XawChainLeft,
                XtNright, XawChainLeft,
                XtNbottom, XawChainTop,
                XtNfromVert, BELOW_WIDGET2,
                XtNwidth, 160,
		XtNlabel, "Font",
                (char *)0);

        mnuFont = XtVaCreatePopupShell("menu",
                simpleMenuWidgetClass, btnFont, (char *)0);

        font_menu(mnuFont, "Courier",
                "(change-font COURIER FONT_MASK)");
        font_menu(mnuFont, "Helvetica",
                "(change-font HELVETICA FONT_MASK)");
        font_menu(mnuFont, "New Century Schoolbook",
                "(change-font NEW_CENTURY FONT_MASK)");
        font_menu(mnuFont, "Times",
                "(change-font TIMES FONT_MASK)");

        btnSize = XtVaCreateManagedWidget("btnSize",
                menuButtonWidgetClass, topbox,
                XtNtop, XawChainTop,
                XtNleft, XawChainLeft,
                XtNright, XawChainLeft,
                XtNbottom, XawChainTop,
                XtNfromVert, BELOW_WIDGET2,
                XtNfromHoriz, btnFont,
                XtNwidth, 40,
		XtNlabel, "Size",
                (char *)0);

        mnuSize = XtVaCreatePopupShell("menu",
                simpleMenuWidgetClass, btnSize, (char *)0);

        font_menu(mnuSize, "8",
                "(change-font SIZE_8 SIZE_MASK)");
        font_menu(mnuSize, "10",
                "(change-font SIZE_10 SIZE_MASK)");
        font_menu(mnuSize, "12",
                "(change-font SIZE_12 SIZE_MASK)");
        font_menu(mnuSize, "14",
                "(change-font SIZE_14 SIZE_MASK)");
        font_menu(mnuSize, "18",
                "(change-font SIZE_18 SIZE_MASK)");
        font_menu(mnuSize, "20",
                "(change-font SIZE_20 SIZE_MASK)");
        font_menu(mnuSize, "24",
                "(change-font SIZE_24 SIZE_MASK)");
        font_menu(mnuSize, "30",
                "(change-font SIZE_30 SIZE_MASK)");

        btnStyle = XtVaCreateManagedWidget("btnStyle",
                menuButtonWidgetClass, topbox,
                XtNtop, XawChainTop,
                XtNleft, XawChainLeft,
                XtNright, XawChainLeft,
                XtNbottom, XawChainTop,
                XtNfromVert, BELOW_WIDGET2,
                XtNfromHoriz, btnSize,
                XtNwidth, 100,
		XtNlabel, "Style",
                (char *)0);

        mnuStyle = XtVaCreatePopupShell("menu",
                simpleMenuWidgetClass, btnStyle, (char *)0);

        font_menu(mnuStyle, "Default",
                "(change-style STY_DEFAULT)");
        font_menu(mnuStyle, "Header 1",
                "(change-style STY_HEADER1)");
        font_menu(mnuStyle, "Header 2",
                "(change-style STY_HEADER2)");
        font_menu(mnuStyle, "Header 3",
                "(change-style STY_HEADER3)");
        font_menu(mnuStyle, "Header 4",
                "(change-style STY_HEADER4)");
        font_menu(mnuStyle, "Header 5",
                "(change-style STY_HEADER5)");
        font_menu(mnuStyle, "Header 6",
                "(change-style STY_HEADER6)");
        font_menu(mnuStyle, "Address",
                "(change-style STY_ADDRESS)");
        font_menu(mnuStyle, "Title",
                "(change-style STY_TITLE)");
        font_menu(mnuStyle, "Abstract",
                "(change-style STY_ABSTRACT)");
        font_menu(mnuStyle, "Preformatted",
                "(change-style STY_PREFORMAT)");
        font_menu(mnuStyle, "User 1",
                "(change-style STY_USER1)");
        font_menu(mnuStyle, "User 2",
                "(change-style STY_USER2)");
        font_menu(mnuStyle, "User 3",
                "(change-style STY_USER3)");
        font_menu(mnuStyle, "User 4",
                "(change-style STY_USER4)");
        font_menu(mnuStyle, "Embed",
                "(change-style STY_EMBED)");

        btnColor = XtVaCreateManagedWidget("btnColor",
                menuButtonWidgetClass, topbox,
                XtNtop, XawChainTop,
                XtNleft, XawChainLeft,
                XtNright, XawChainLeft,
                XtNbottom, XawChainTop,
                XtNfromVert, BELOW_WIDGET2,
                XtNfromHoriz, btnStyle,
                XtNwidth, 80,
		XtNlabel, "Color",
                (char *)0);

        mnuColor = XtVaCreatePopupShell("menu",
                simpleMenuWidgetClass, btnColor, (char *)0);

        font_menu(mnuColor, "Black",
                "(change-font COLOR_BLACK COLOR_MASK)");
        font_menu(mnuColor, "Red",
                "(change-font COLOR_RED COLOR_MASK)");
        font_menu(mnuColor, "Green",
                "(change-font COLOR_GREEN COLOR_MASK)");
        font_menu(mnuColor, "Blue",
                "(change-font COLOR_BLUE COLOR_MASK)");
        font_menu(mnuColor, "Yellow",
                "(change-font COLOR_YELLOW COLOR_MASK)");
        font_menu(mnuColor, "Magenta",
                "(change-font COLOR_MAGENTA COLOR_MASK)");
        font_menu(mnuColor, "Cyan",
                "(change-font COLOR_CYAN COLOR_MASK)");
        font_menu(mnuColor, "White",
                "(change-font COLOR_WHITE COLOR_MASK)");
}

void drop_handler(Widget w, XtPointer data,
                XEvent *event, Boolean *b)
{
        unsigned char *Data, *filename;
        unsigned long DataSize;
        int DataType = DndDataType(event);

        if (DataType == DndNotDnd) return;
        DndGetData(&Data, &DataSize);
        switch (DataType) {
        case DndFile:
                filename = Data;
                if (!fork()) {
                        execlp("pw", "Pw", filename, (char *)0);
                        exit(1);
                }
        default:
                ;
        }
}

static void place_shortcuts(Widget w, XEvent *event,
                String *p, Cardinal *n)
{
        XButtonEvent *bev = (XButtonEvent *)event;
        int col, row;
        int x, y;

        activate_window(find_window_by_widget(w));
        x = event->xbutton.x;
        y = event->xbutton.y;
        get_coords_cell(w_list, w_list->top.row, w_list->top.col, 
                        &row, &col, x, y);
        XtVaSetValues(shortcuts,
                XtNx, bev->x_root-10,
                XtNy, bev->y_root-10,
                (char *)0);
}

static void menu_motion(Widget w, XEvent *event,
                String *p, Cardinal *n)
{
        int fromright;
        Dimension width;
        XMotionEvent *mev = (XMotionEvent *)event;
        Widget menu, submenu;

        menu = XawSimpleMenuGetActiveEntry(w);
        if (menu == None) return;
        submenu = find_submenu_by_entry(menu);
        if (submenu == None) return;
        XtVaGetValues(w, XtNwidth, &width, (char *)0);
        fromright = width-mev->x;
        if (fromright > 0 && fromright < 10) {
                Position x, y, x_root, y_root;
                XtVaGetValues(menu,
                        XtNx, &x,
                        XtNy, &y,
                        (char *)0);
                XtTranslateCoords(w,
                        x+width-20, y, &x_root, &y_root);
                XtVaSetValues(submenu,
                        XtNx, x_root,
                        XtNy, y_root,
                        (char *)0);
                XtPopup(submenu, XtGrabNonexclusive);
        }
}

static void popdown_submenu(Widget w, XEvent *event,
                String *p, Cardinal *n)
{
        XtPopdown(w);
        XtPopdown(XtParent(w));
}


static XtActionsRec actions[] =
{
        {"grid-button", GridButtonAction},
        {"self-insert-char", SelfInsertCharAction},
        {"siaghelp", siaghelp_action},
        {"place-shortcuts", place_shortcuts},
	{"menu-motion", menu_motion},
	{"popdown-submenu", popdown_submenu}
};

void init_windows1(int *argc, char **argv)
{
        unsigned long highlight_color;
        unsigned long unhighlight_color;

        XtSetLanguageProc(NULL, (XtLanguageProc) NULL, NULL);

        topLevel = XtVaAppInitialize(
                    &app_context,       /* application context */
                    "Pw",               /* application class */
                    options,            /* command line options list */
		    XtNumber(options),
                    argc, argv,        /* command line args */
                    fallback_resources, /* for missing app-defaults file */
                    (char *)0);         /* terminate varargs list */

	XtGetApplicationResources(topLevel, &app_data, resources,
			XtNumber(resources), NULL, 0);

        XtAppAddActions(app_context, actions, XtNumber(actions));

	shortcuts = XtVaCreatePopupShell("shortcuts",
                simpleMenuWidgetClass, topLevel, (char *)0);
        XtVaCreateManagedWidget("-",
                smeLineObjectClass, shortcuts, (char *)0);

        topbox = XtCreateManagedWidget("topbox",
                formWidgetClass, topLevel, NULL, 0);

        highlight_color = WhitePixel(XtDisplay(topLevel),
                                DefaultScreen(XtDisplay(topLevel)));
        XtVaGetValues(topbox,   /* or any old widget with a grey background */
                XtNbackground, &unhighlight_color,
                (char *)0);
        tooltip_init(topLevel, highlight_color, unhighlight_color);

        init_menu();
        init_toolbar();
        setup_buttons();
        init_toggle();

	hsep3 = XtVaCreateManagedWidget("hsep3",
		hsepWidgetClass, topbox,
		XtNtop, XawChainTop,
		XtNleft, XawChainLeft,
		XtNright, XawChainRight,
		XtNbottom, XawChainTop,
		XtNfromVert, btnFont, (char *)0);

        label1 = XtVaCreateManagedWidget("label1",
                labelWidgetClass, topbox,
                XtNtop, XawChainTop,
                XtNleft, XawChainLeft,
                XtNright, XawChainRight,
                XtNbottom, XawChainTop,
                XtNfromVert, hsep3, (char *)0);

        gridpane = XtVaCreateManagedWidget("gridpane",
                panedWidgetClass, topbox,
                XtNtop, XawChainTop,
                XtNleft, XawChainLeft,
                XtNright, XawChainRight,
                XtNbottom, XawChainBottom,
                XtNfromVert, label1,
                XtNallowResize, True,
                /*XtNmin, 40,
                XtNmax, 600,
                XtNheight, 200,*/ (char *)0);

        label2 = XtVaCreateManagedWidget("label2",
                labelWidgetClass, topbox,
                XtNtop, XawChainBottom,
                XtNleft, XawChainLeft,
                XtNright, XawChainRight,
                XtNbottom, XawChainBottom,
                XtNfromVert, gridpane,
		XtNwidth, 496,
		(char *)0);

	label3 = XtVaCreateManagedWidget("label3",
		labelWidgetClass, topbox,
		XtNtop, XawChainBottom,
		XtNleft, XawChainRight,
		XtNright, XawChainRight,
		XtNbottom, XawChainBottom,
		XtNfromVert, gridpane,
		XtNfromHoriz, label2,
		XtNwidth, 100,
		(char *)0);
}

/* this does probably not belong here in window.c,
   because it doesn't depend on X */
static void save_plugin(char *p)
{
        if (*p++ != ' ' || *p == '\0') printf("501 File name missing\n");
        else {
                if (savematrix(p, w_list->buf, NULL)) {
                        printf("501 Can't save %s\n", p);
                } else {
                        printf("250 Saved %s\n", p);
                }
        }
}

static void load_plugin(char *p)
{
        if (*p++ != ' ' || *p == '\0') printf("501 File name missing\n");
        else {
                if (loadmatrix(p, w_list->buf, NULL)) {
                        printf("501 Can't load %s\n", p);
                } else {
                        printf("250 Loaded %s\n", p);
                }
        }
}

static void exec_plugin(char *p)
{
        if (*p++ != ' ' || *p == '\0') printf("501 Command missing\n");
        else {
                execute(p);
                printf("250 OK\n");
        }
}

static void help_plugin(char *p)
{
        printf("214 SAVE LOAD EXEC HELP NOOP QUIT PRNT\n");
}

static void noop_plugin(char *p)
{
        printf("250 OK\n");
}

static void win_plugin(char *p)
{
	Window win = XtWindow(topLevel);
	printf("250 %lx\n", (unsigned long)win);
}

static void quit_plugin(char *p)
{
        printf("221 Over and out\n");
        execute("(quit-pw)");
}

static void prnt_plugin(char *p)
{
        printf("502 Can't print yet\n");
}

static struct {
        char *verb;
        void (*cb)(char *);
} plugin_cmds[] = {
        {"SAVE", save_plugin},
        {"LOAD", load_plugin},
        {"EXEC", exec_plugin},
        {"HELP", help_plugin},
        {"NOOP", noop_plugin},
	{"WIN", win_plugin},
        {"QUIT", quit_plugin},
        {"PRNT", prnt_plugin},
        {NULL, NULL}
};

static void read_plugin_cmd(XtPointer client_data, int *fid, XtInputId *id)
{
        char b[1024], *p;
        int i, n;

        if ((n = read(*fid, b, 1020)) == -1) return;

        b[n] = '\0';
        if ((p = strchr(b, '\n')) == NULL) {
                printf("501 Incomplete command\n");
                return;
        }

        *p = '\0';
        for (i = 0; plugin_cmds[i].verb; i++) {
                if (!strncmp(b, plugin_cmds[i].verb,
                                strlen(plugin_cmds[i].verb)))
                        break;
        }
        if (plugin_cmds[i].verb)
                (*plugin_cmds[i].cb)(b+strlen(plugin_cmds[i].verb));
        else
                printf("500 What are you talking about\n");
        fflush(stdout);
}

void mainloop(void)
{
	if (app_data.plugin) {
		XtAppAddInput(XtWidgetToApplicationContext(topLevel),
                        fileno(stdin), (XtPointer)XtInputReadMask,
                        read_plugin_cmd, NULL);
		printf("220 %s\n", VERSION);
		fflush(stdout);
        }

        XtAppMainLoop(app_context);

        exit(0);
}

/* handle spontaneous exit of plugins */
static void handle_plugin_exit(int ph)
{
        buffer *b = b_list;

        do {
                int n = buffer_plugin2index(b, ph);
                if (n != -1) {
                        cfree(b->plugin[n].name);
                        b->nplugin--;
                        for (; n < b->nplugin; n++)
                                b->plugin[n] = b->plugin[n+1];
                        b->change = pr_scr_flag = TRUE;
                }
                b = b->next;
        } while (b != b_list);
}

/*
   void init_windows(buffer *b)
   Sets up the whole initial window structure and initializes scrupd.
   The window list w_list is set to a list with a single window with the
   buffer b.
*/
void init_windows(buffer *b, int argc, char **argv)
{
	unsigned long foreground;	/* pixel value */
	unsigned long background;
	Window cell_win;
	XColor screen_color, exact_color;

	XtRealizeWidget(topLevel);

	plugin_init(topLevel, handle_plugin_exit);

	display = XtDisplay(topLevel);

	grid_cursor = XCreateFontCursor(display, XC_xterm);

	activate_window(new_window(b, NULL));

	wm_delete_window = XInternAtom(display, "WM_DELETE_WINDOW", False);
	XtOverrideTranslations(topLevel,
		XtParseTranslationTable(
			"<Message>WM_PROTOCOLS: execute(quit-pw)"));

	XSetWMProtocols(display, XtWindow(topLevel), &wm_delete_window, 1);

	root = DefaultRootWindow(display);

	init_color(display);
	XAllocNamedColor(display,
			DefaultColormap(display, DefaultScreen(display)),
			"grey", &screen_color, &exact_color);
	blockbg = screen_color.pixel;
	noblockbg = WhitePixelOfScreen(XtScreen(topLevel));

	foreground = BlackPixelOfScreen(XtScreen(topLevel));

	cell_win = XtWindow(w_list->ui->grid);

	XtVaGetValues(w_list->ui->grid,
		XtNbackground, &background, (char *)0);

	clear_gc = get_gc(display, cell_win, background, foreground, -1);

	cell_gc = get_gc(display, cell_win, foreground, blockbg,
			 get_font(display, CELL_FONT));

	cursor_gc = get_gc(display, cell_win, foreground ^ background, 0, -1);
	XSetFunction(display, cursor_gc, GXxor);
	XSetLineAttributes(display, cursor_gc, 1, LineSolid, CapButt, JoinMiter);

	draw_scrollbars(display, w_list);
	draw_status(display, "");

	/* Set up selection */
	target_atom = XInternAtom(display, "PW_BLOCK", False);

	/* stuff originally in realmain */

        embed_init(topLevel);

        ok2print = 1;

        init_subr_3("add-menu-entry", add_menu_entry);
        init_subr_2("add-submenu", add_submenu);
        init_subr_4("add-submenu-entry", add_submenu_entry);
        init_subr_1("tooltip-mode", ltooltip_mode);

        init_form(topLevel, app_context);

        icon_set(topLevel, pw_fg_xpm);
        DndInitialize(topLevel);
        DndRegisterOtherDrop(drop_handler);

        font_init(topLevel);
}

/*
   void exit_windows()
   Cleans up after Calc before exit.  All buffers and windows are freed.
*/
void exit_windows(void)
{
	/* free all buffers */
	while (b_list != NULL)
		free_buffer(b_list);
	while (w_list != NULL)
		free_window(w_list);
}

/*
   static void pr_scr()
   Prints and refreshes all the windows.
   Sets pr_scr_flag to FALSE.
*/
static void pr_scr(void)
{
	int i;
	window *w;
	draw_status(display, "");
	w = w_list;
	do {
		/* plugin size fixer-upper */
		for (i = 0; i < w->buf->nplugin; i++) {
			unsigned long wi, he;
			long r = w->buf->plugin[i].row;
			alloc_line(w->buf, r);
			pw_plugin_size_get(w->buf->plugin[i].ph, &wi, &he);
			w->buf->text[r].height = he;
		}
		draw_scrollbars(display, w);
		draw_buffer(display, w);
                /* check for nondisplayed plugins */
                /* should also *remove* plugins that can't be displayed */
                for (i = 0; i < w->buf->nplugin; i++) {
                        if (!w->buf->plugin[i].displayed) {
                                plugin_show(w->buf->plugin[i].ph,
                                                w->ui->grid);
                                w->buf->plugin[i].displayed = 1;
                        }
                }
		w = w->next;
	} while (w != w_list);
	pr_scr_flag = pr_line_flag = FALSE;
}	/* pr_scr */

void get_char_coords(window *w, int top_row, int top_col,
			int cell_row, int cell_col,
			int *cell_x, int *cell_y)
{
	int r;
	*cell_y = 0;
	for (r = top_row; r < cell_row; r++)
		*cell_y += cell_height(w->buf, r);

	*cell_x = line_start(w->buf, cell_row)
			+line_width(w->buf, cell_row, cell_col);
}

void get_coords_cell(window *w, int top_row, int top_col,
			int *cur_row, int *cur_col,
			int cur_x, int cur_y)
{
	int r, c, h;
	r = top_row;
	h = line_height(w->buf, r);
	while (cur_y > h) {
		r++;
		h += line_height(w->buf, r);
	}
	*cur_row = r;

	c = 0;
	cur_x -= line_start(w->buf, r);
	cur_x += top_col;	/* it's in pixels, y'know */

	while (c < line_length(w->buf, r) &&
			cur_x > line_width(w->buf, r, c)) {
		c++;
	}
	*cur_col = c;
}

char *style2name(int sty)
{
	return styles[sty].name;
}

void show_format(void)
{
	static int last_fmt = -1;
	int fmt = w_list->current_fmt;
	int sty = ret_style(w_list->buf, w_list->point_pos.row);
	int bold = fmt & BOLD;
	int italic = fmt & ITALIC;
	int uline = fmt & ULINE;
	static int last_hadj = -1;
	static int last_vadj = -1;
	int hadj = ret_hadj(w_list->buf, w_list->point_pos.row);
	int hadj_center = (hadj == HADJ_CENTER);
	int hadj_right = (hadj == HADJ_RIGHT);
	int hadj_left = !(hadj_center || hadj_right);
	int vadj = fmt & VADJ_MASK;
	int vadj_top = (vadj == VADJ_TOP);
	int vadj_bottom = (vadj == VADJ_BOTTOM);

	/* thou shalt not waste time on things you have already done */
	if (fmt == last_fmt && hadj == last_hadj && vadj == last_vadj &&
		!pr_scr_flag && !pr_line_flag) return;

	last_fmt = fmt;
	last_hadj = hadj;
	last_vadj = vadj;

	/* menus */
	label_set(btnFont, family2name(fmt));
	label_set(btnSize, size2name(fmt));
	label_set(btnStyle, style2name(sty));
	label_set(btnColor, color2name(fmt));

	/* toggle buttons */
	state_set(cmdBold, (bold?1:0));
	state_set(cmdItalic, (italic?1:0));
	state_set(cmdUline, (uline?1:0));
	state_set(cmdHLeft, (hadj_left?1:0));
	state_set(cmdHCenter, (hadj_center?1:0));
	state_set(cmdHRight, (hadj_right?1:0));
	state_set(cmdVTop, (vadj_top?1:0));
	state_set(cmdVBottom, (vadj_bottom?1:0));
}

int cursor_visible = FALSE;

/*
   void show_cur(window *w)
   Moves the cursor to reflect the position of point in w.
   If point is not visible, the window is moved so that point is in
   the middle of the screen.
*/
void show_cur(window *w)
{
	int top_row = w->top.row, top_col = w->top.col;
	char *p = "";

	XtVaSetValues(w->ui->grid,
		XtNrichtextPointRow, w->point_pos.row,
		XtNrichtextPointCol, w->point_pos.col,
		XtNrichtextVisibleCursor, True,
		(char *)0);
	if (pr_scr_flag) {
		pr_scr();
	} else {
		/* this may have moved top */
		XtVaGetValues(w->ui->grid,
			XtNrichtextTopRow, &(w->top.row),
			XtNrichtextTopCol, &(w->top.col),
			(char *)0);
		if (top_row != w->top.row) {
			draw_vbar(display, w);
		}
		if (top_col != w->top.col) {
			draw_hbar(display, w);
		}
		if (pr_line_flag) {
			rt_pr_line(w->ui->grid, w->point_pos.row);
			pr_line_flag = FALSE;
		}
	}
	show_format();
	draw_input(display, p);
	cursor_visible = TRUE;
}	/* show_cur */

void hide_cur(window *w)
{
	;
}

