/*
   Siag, Scheme In A Grid
   Copyright (C) 1996-1998  Ulric Eriksson <ulric@edu.stockholm.se>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>

#include <unistd.h>
#include <sys/stat.h>

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "../siod/siod.h"

#include "../common/cmalloc.h"
#include "../common/common.h"
#include "../common/fonts.h"

#include "../common/bitmaps/new.xpm"
#include "../common/bitmaps/fld_open.xpm"
#include "../common/bitmaps/floppy3.xpm"
#include "../common/bitmaps/preview.xpm"
#include "../common/bitmaps/printer.xpm"
#include "../common/bitmaps/plotter.xpm"
#include "../common/bitmaps/sigma.xpm"
#include "../common/bitmaps/info.xpm"
#include "../common/bitmaps/copyright.xpm"
#include "../common/bitmaps/bold.xpm"
#include "../common/bitmaps/italic.xpm"
#include "../common/bitmaps/hleft.xpm"
#include "../common/bitmaps/hcenter.xpm"
#include "../common/bitmaps/hright.xpm"
#include "../common/bitmaps/grid.xpm"
#include "../common/bitmaps/borders.xpm"
#include "../common/bitmaps/uline.xpm"
#include "../common/bitmaps/none.xpm"

#include "../siag/types.h"
#include "../siag/calc.h"

#include "gsiag.h"
#include "sgrid.h"

textbuf kbd_macro = {0, 0, 0};

enum {ABORT = 0, DONE, WAITING};

window *w_list;
int pr_scr_flag;
int grid_lines = 1;

int lastc, macro_flag;

GtkWidget *topLevel;

static GtkWidget *topbox, *menu_bar, *toolbar,*toolbarhandlebox, *formatbar, *formatbarhandlebox,*gridpane, *text1, *label1, *label2, *tbNew, *tbCopy;

static GtkWidget *btnFont, *mnuFont, *btnSize, *mnuSize, *btnStyle, *mnuStyle, *btnColor, *mnuColor;
static GtkWidget *cmdBold, *cmdItalic, *cmdHLeft, *cmdHCenter, *cmdHRight;
static GtkWidget *cmdBorders, *cmdGrid, *cmdUline, *cmdNone;

static GtkTooltips *helptext;

extern int recalc;
int input_warp_pointer = 1;     /* move pointer to input field */

static int status;

static void draw_input(char *text)
{
	int r = w_list->point_pos.row, c = w_list->point_pos.col;
	int type = ret_type(w_list->buf, r, c);
	int intp = ret_interpreter(w_list->buf, r, c);
	char b[1024];
	if (type == EXPRESSION || type == STRING)
		sprintf(b, "[%d,%d] [%s] %s",
			r, c, interpreter2name(intp), text);
	else
		sprintf(b, "[%d,%d] %s",
			r, c, text);
	gtk_label_set(GTK_LABEL(label1), "");
	gtk_label_set(GTK_LABEL(label1), b);
}

static void draw_status(char *text)
{
	char *p = strchr(text, '\n');
	if (p) *p = '\0';
	gtk_label_set(GTK_LABEL(label2), "");
	gtk_label_set(GTK_LABEL(label2), text);
}

void llpr(char *p)
{
	static int isclear = FALSE;

	if (isclear && p[0] == '\0')
		return;
	isclear = (p[0] == '\0');

	draw_status(p);
}

static void draw_vbar(window *w)
{
}

static void draw_hbar(window *w)
{
}

static void draw_scrollbars(window *w)
{
	draw_vbar(w);
	draw_hbar(w);
}
#if 0	/* FIXME */
static char *colnum_text(buffer *buf, int row, int col)
{
        static char b[80];
        sprintf(b, "%d", col);
        return b;
}
#endif

static gint rowcol_format(GtkSGrid *dummy, gpointer data,
					 gint row, gint col)
{
        return HELVETICA | SIZE_12 | BOLD | HADJ_CENTER;
}

static gint cell_format(GtkSGrid *dummy, gpointer data,
					 gint row, gint col)
{
	buffer *buf = (buffer *)data;
	return ret_format(buf, row, col);
}

static void draw_colnums(window *w)
{
  GtkSGrid *colnum = GTK_SGRID(w->ui->colnum);
  colnum->prot_col = w->prot.col;
  colnum->top_col = w->top.col;
  colnum->data = w->buf;
  gtk_sgrid_set_point(colnum, 0, 0);
}

static void draw_rownums(window *w)
{
  GtkSGrid *rownum = GTK_SGRID(w->ui->rownum);
  rownum->prot_row = w->prot.row;
  rownum->top_row = w->top.row;
  rownum->data = w->buf;
  gtk_sgrid_set_point(rownum, 0, 0);
}
#if 0	/* FIXME */
static char *cell_text(buffer *b, int row, int col)
{
        static char s[1024];
        s[0] = '\0';
        ret_pvalue(s, b, row, col, -1);
        return s;
}

static unsigned long cell_format(buffer *b, int row, int col)
{
        return ret_format(b, row, col);
}
#endif
static gint ng_col_width(GtkSGrid *dummy, gpointer data, gint col)
{
	buffer *b = (buffer *)data;
	if (col < 1) return b->sw;
	return cell_width(b, col);
}

static gint ng_row_height(GtkSGrid *dummy, gpointer data, gint row)
{
	buffer *b = (buffer *)data;
	if (row < 1) return b->sh;
	return cell_height(b, row);
}

static gchar *ng_text(GtkSGrid *dummy, gpointer b, gint row, gint col)
{
	static char s[1024];
        s[0] = '\0';
        ret_pvalue(s, (buffer *)b, row, col, -1);
        return s;
}

static void draw_cells(window *w)
{
	GTK_SGRID(w->ui->grid)->max_row = BUFFER_ROWS;
	GTK_SGRID(w->ui->grid)->max_col = BUFFER_COLS;
	GTK_SGRID(w->ui->grid)->prot_row = w->prot.row;
	GTK_SGRID(w->ui->grid)->prot_col = w->prot.col;
	GTK_SGRID(w->ui->grid)->top_row = w->top.row;
	GTK_SGRID(w->ui->grid)->top_col = w->top.col;
	GTK_SGRID(w->ui->grid)->sel_top_row = w->blku.row;
	GTK_SGRID(w->ui->grid)->sel_bottom_row = w->blkl.row;
	GTK_SGRID(w->ui->grid)->sel_left_col = w->blku.col;
	GTK_SGRID(w->ui->grid)->sel_right_col = w->blkl.col;
	GTK_SGRID(w->ui->grid)->point_row = w->point_pos.row;
	GTK_SGRID(w->ui->grid)->point_col = w->point_pos.col;
	GTK_SGRID(w->ui->grid)->row_height = ng_row_height;
	GTK_SGRID(w->ui->grid)->col_width = ng_col_width;
/*	GTK_SGRID(w->ui->grid)->type =*/
	GTK_SGRID(w->ui->grid)->text = ng_text;
/*	GTK_SGRID(w->ui->grid)->format =*/
	GTK_SGRID(w->ui->grid)->data = w->buf;
	GTK_SGRID(w->ui->grid)->grid_lines = grid_lines;
	gtk_sgrid_set_point(GTK_SGRID(w->ui->grid),
				 w->point_pos.row, w->point_pos.col);

	/* there is a chance that this moved top */
	if (GTK_SGRID(w->ui->grid)->top_row != w->top.row) {
		w->top.row = GTK_SGRID(w->ui->grid)->top_row;
		draw_rownums(w);
	}
	if (GTK_SGRID(w->ui->grid)->top_col != w->top.col) {
		w->top.col = GTK_SGRID(w->ui->grid)->top_col;
		draw_colnums(w);
	}
}

static int dialog_key(GtkWidget *w, GdkEventKey *event, gpointer user_data)
{
	int c = event->keyval;
	if (c == GDK_Escape) status = ABORT;
	if (c == GDK_Return) status = DONE;
	return TRUE;
}

int ask_for_str_comp(char *prompt, char *buffr, int (*comp)(char *))
{
	char *p;
	status = WAITING;

	gtk_label_set(GTK_LABEL(label1), prompt);
	gtk_entry_set_text(GTK_ENTRY(text1), buffr);
	gtk_widget_grab_focus(text1);
	gtk_grab_add(text1);
	while (status == WAITING)
		gtk_main_iteration();

	p = gtk_entry_get_text(GTK_ENTRY(text1));
	if (p) strcpy(buffr, p);
	gtk_entry_set_text(GTK_ENTRY(text1), "");
	gtk_grab_remove(text1);
	activate_window(w_list);
	return status;
}

int edit_cell(char *p, char *b)
{
	return ask_for_str(p, b);
}

static int nocomp(char *b)
{
        return TRUE;
}

int ask_for_str(char *prompt, char *buffr)
{
        return ask_for_str_comp(prompt, buffr, nocomp);
}

int add_str_to_input_queue(textbuf buf)
{
	return FALSE;
}

static void set_block(int row1, int col1, int row2, int col2)
{
        int ur = w_list->blku.row, uc = w_list->blku.col;
        int lr = w_list->blkl.row, lc = w_list->blkl.col;

        if (row1 < row2) {
                w_list->blku.row = row1;
                w_list->blkl.row = row2;
        } else {
                w_list->blku.row = row2;
                w_list->blkl.row = row1;
        }
        if (col1 < col2) {
                w_list->blku.col = col1;
                w_list->blkl.col = col2;
        } else {
                w_list->blku.col = col2;
                w_list->blkl.col = col1;
        }


        /* Redraw if any change */
        if (ur != w_list->blku.row || uc != w_list->blku.col ||
                lr != w_list->blkl.row || lc != w_list->blkl.col) {
                pr_scr_flag = TRUE;
        }
}

#ifndef ABS
#define ABS(a) ((a)>0?(a):-(a))
#endif
#if 0
static LISP lhelptext_mode(LISP newmode)
{
        return NIL;
}
#endif
static void execute_callback(GtkWidget *w, char *client_data)
{
        execute((char *)client_data);
}

static struct {
        char *label;
        GtkWidget *menu;
} menubar[10];

static menucount = 0;

static void make_menu(char *label)
{
	GtkWidget *root_menu;

	menubar[menucount].label = cstrdup(label);
	menubar[menucount].menu = gtk_menu_new();
	root_menu = gtk_menu_item_new_with_label(label);
	gtk_widget_show(root_menu);
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(root_menu),
				menubar[menucount].menu);
	gtk_menu_bar_append(GTK_MENU_BAR(menu_bar), root_menu);
	menucount++;
}

static void init_menu(void)
{
        make_menu("File");
        make_menu("Edit");
        make_menu("Block");
        make_menu("Format");
        make_menu("Data");
        make_menu("Window");
        make_menu("Plot");
        make_menu("Tools");
        make_menu("Help");
}

static GtkWidget *find_menu_by_name(char *label)
{
        int i;

        for (i = 0; i < menucount; i++) {
                if (!cstrcasecmp(menubar[i].label, label))
                        return menubar[i].menu;
        }
        return NULL;
}

static LISP add_menu_entry(LISP menu, LISP label, LISP function)
{
        GtkWidget *entry;
        GtkWidget *menuw = find_menu_by_name(get_c_string(menu));
        if (!menuw) {
                fprintf(stderr, "No menu! Bailing out.\n");
                return NIL;
        }

        if (!strcmp(get_c_string(label), "-")) {        /* line pane */
		entry = gtk_menu_item_new();
		gtk_container_add(GTK_CONTAINER(menuw), entry);
		gtk_widget_show(entry);
        } else {
		entry = gtk_menu_item_new_with_label(get_c_string(label));
		gtk_menu_append(GTK_MENU(menuw), entry);
		gtk_signal_connect_object(GTK_OBJECT(entry),
				"activate",
				GTK_SIGNAL_FUNC(execute_callback),
				(gpointer)g_strdup(get_c_string(function)));
		gtk_widget_show(entry);
        }
        return NIL;
}

static GtkWidget *make_toggle(char *name, char *cmd, GtkWidget *pw,
			char **bits, char *tips)
{
  GtkWidget *w = gtk_toggle_button_new(); 
	GdkWindow *win = pw->window;
	GtkStyle *style = gtk_widget_get_default_style();
	GdkBitmap *mask;
	GdkPixmap *pm = gdk_pixmap_create_from_xpm_d(win,
		&mask, &style->bg[GTK_STATE_NORMAL], bits);
	GtkWidget *pmw = gtk_pixmap_new(pm, mask);

	
	  gtk_container_add(GTK_CONTAINER(w), pmw);
	  gtk_widget_show(pmw); 
	  /* gtk_box_pack_start(GTK_BOX(pw), w, FALSE, FALSE, 2); */
	   gtk_widget_show(w); 
	   gtk_signal_connect_object(GTK_OBJECT(w),
	   "clicked",
	   GTK_SIGNAL_FUNC(execute_callback),
	   (gpointer)g_strdup(cmd));
	
	 gtk_toolbar_append_widget  (GTK_TOOLBAR(formatbar),
                                        GTK_WIDGET(w),
                                        tips,
                                        NULL);

	GTK_WIDGET_UNSET_FLAGS(w, GTK_CAN_FOCUS);
	GTK_WIDGET_UNSET_FLAGS(w, GTK_CAN_DEFAULT);
	return w;
}

/*changed RW */

static GtkWidget *make_command(char *name, char *cmd, GtkWidget *pw,
			char **bits, char *tips)
{
  /*GtkWidget *w = gtk_button_new(); */
	GdkWindow *win = pw->window;
	GtkStyle *style = gtk_widget_get_default_style();
	GdkBitmap *mask;
	GdkPixmap *pm = gdk_pixmap_create_from_xpm_d(win,
		&mask, &style->bg[GTK_STATE_NORMAL], bits);
	GtkWidget *pmw = gtk_pixmap_new(pm, mask);
	/*	GTK_WIDGET_UNSET_FLAGS(w, GTK_CAN_FOCUS); 
	GTK_WIDGET_UNSET_FLAGS(w, GTK_CAN_DEFAULT); 
	gtk_container_add(GTK_CONTAINER(w), pmw); 
	gtk_widget_show(pmw); 
	gtk_box_pack_start(GTK_BOX(pw), w, FALSE, FALSE, 2); 
	gtk_widget_show(w);
	
	gtk_signal_connect_object(
	               GTK_OBJECT(w),
		       "clicked",
		       GTK_SIGNAL_FUNC(execute_callback),
		       (gpointer)g_strdup(cmd));
	
	gtk_tooltips_set_tip(helptext, w, tips, NULL); */
	
 GtkWidget     *w = gtk_toolbar_append_item (
				     GTK_TOOLBAR (toolbar),
				     NULL,
				     tips,
				     NULL,
				     GTK_WIDGET(pmw),
				     GTK_SIGNAL_FUNC(execute_callback),
				     (gpointer)g_strdup(cmd)
				     );


	return w;
}
/* stopped changing */

static void init_toolbar()
{
        GtkWidget *tbOpen, *tbSave, *tbView, *tbPrint, *tbPlot, *tbSum, *tbHelp;

        tbNew = make_command("tbNew", "(new-siag)", toolbar,
                (char **)new_xpm, "Start another instance of Siag");
        tbOpen = make_command("tbOpen", "(load-buffer)", toolbar,
                (char **)fld_open_xpm, "Open a Siag document");
        tbSave = make_command("tbSave", "(save-buffer-as)", toolbar,
                (char **)floppy3_xpm, "Save the contents of the current buffer");
        tbView = make_command("tbView", "(preview)", toolbar,
                (char **)preview_xpm, "Preview the contents of the current buffer");
        tbPrint = make_command("tbPrint", "(print)", toolbar,
                (char **)printer_xpm,
		"Print the contents of the current buffer");
        tbPlot = make_command("tbPlot", "(plot \"lines\")", toolbar,
                (char **)plotter_xpm,
		"Plot the contents of the block using lines");
        tbSum = make_command("tbSum", "(block-sum)", toolbar,
                (char **)sigma_xpm,
		"Add the contents of the block");
        tbHelp = make_command("tbHelp", "(help-contents)", toolbar,
                (char **)info_xpm,
		"Display the Siag online documentation");
        tbCopy = make_command("tbCopy", "(help-copyright)", toolbar,
                (char **)copyright_xpm,
		"Display the GNU general public licence");
        cmdBorders = make_command("cmdBorders", "(block-borders 1)", toolbar,
                (char **)borders_xpm,
		"Draw borders around the block");
        cmdGrid = make_command("cmdGrid", "(block-borders 2)", toolbar,
                (char **)grid_xpm,
		"Draw grid lines in the block");
        cmdUline = make_command("cmdUline", "(block-borders 3)", toolbar,
                (char **)uline_xpm,
		"Underline the block");
        cmdNone = make_command("cmdNone", "(block-borders 0)", toolbar,
                (char **)none_xpm,
		"Remove grid lines from the block");
}

static void init_toggle()
{
        cmdBold = make_toggle("cmdBold", "(toggle-fontflag BOLD)", formatbar,
                (char **)bold_xpm, "Bold text");
        cmdItalic = make_toggle("cmdItalic", "(toggle-fontflag ITALIC)", formatbar,
                (char **)italic_xpm, "Italic text");
        cmdHLeft = make_toggle("cmdHLeft", "(change-font HADJ_LEFT HADJ_MASK)", formatbar,
                (char **)hleft_xpm, "Left adjusted text");
        cmdHCenter = make_toggle("cmdHCenter", "(change-font HADJ_CENTER HADJ_MASK)", formatbar,
                (char **)hcenter_xpm, "Centered text");
        cmdHRight = make_toggle("cmdHRight", "(change-font HADJ_RIGHT HADJ_MASK)", formatbar,
                (char **)hright_xpm, "Right adjusted text");
}

static void font_menu(GtkWidget *w, char *label, char *cmd)
{
	GtkWidget *entry = gtk_menu_item_new_with_label(label);
	gtk_menu_append(GTK_MENU(w), entry);
	gtk_signal_connect_object(GTK_OBJECT(entry),
			"activate",
			GTK_SIGNAL_FUNC(execute_callback),
			(gpointer)g_strdup(cmd));
	gtk_widget_show(entry);
}

static GtkWidget *create_font_menu()
{
	mnuFont = gtk_menu_new();
        font_menu(mnuFont, "Courier",
                "(change-font COURIER FONT_MASK)");
        font_menu(mnuFont, "Helvetica",
                "(change-font HELVETICA FONT_MASK)");
        font_menu(mnuFont, "New Century Schoolbook",
                "(change-font NEW_CENTURY FONT_MASK)");
        font_menu(mnuFont, "Times",
                "(change-font TIMES FONT_MASK)");
	return mnuFont;
}

static GtkWidget *create_size_menu()
{
	mnuSize = gtk_menu_new();
        font_menu(mnuSize, "8",
                "(change-font SIZE_8 SIZE_MASK)");
        font_menu(mnuSize, "10",
                "(change-font SIZE_10 SIZE_MASK)");
        font_menu(mnuSize, "12",
                "(change-font SIZE_12 SIZE_MASK)");
        font_menu(mnuSize, "14",
                "(change-font SIZE_14 SIZE_MASK)");
        font_menu(mnuSize, "18",
                "(change-font SIZE_18 SIZE_MASK)");
        font_menu(mnuSize, "20",
                "(change-font SIZE_20 SIZE_MASK)");
        font_menu(mnuSize, "24",
                "(change-font SIZE_24 SIZE_MASK)");
        font_menu(mnuSize, "30",
                "(change-font SIZE_30 SIZE_MASK)");
	return mnuSize;
}

static GtkWidget *create_style_menu()
{
	mnuStyle = gtk_menu_new();
        font_menu(mnuStyle, "Default",
                "(change-font FMT_DEFAULT FMT_MASK)");
        font_menu(mnuStyle, "Invisible",
                "(change-font FMT_INVISIBLE FMT_MASK)");
        font_menu(mnuStyle, "Integer",
                "(change-font FMT_INTEGER FMT_MASK)");
        font_menu(mnuStyle, "Scientific",
                "(change-font FMT_SCIENTIFIC FMT_MASK)");
        font_menu(mnuStyle, "Fixed",
                "(change-font FMT_FIXED FMT_MASK)");
        font_menu(mnuStyle, "Date",
                "(change-font FMT_DATE FMT_MASK)");
        font_menu(mnuStyle, "Time",
                "(change-font FMT_TIME FMT_MASK)");
        font_menu(mnuStyle, "Comma",
                "(change-font FMT_COMMA FMT_MASK)");
        font_menu(mnuStyle, "Percent",
                "(change-font FMT_PERCENT FMT_MASK)");
        font_menu(mnuStyle, "Hex",
                "(change-font FMT_HEX FMT_MASK)");
        font_menu(mnuStyle, "Currency",
                "(change-font FMT_CURRENCY FMT_MASK)");
        font_menu(mnuStyle, "User 1",
                "(change-font FMT_USER1 FMT_MASK)");
        font_menu(mnuStyle, "User 2",
                "(change-font FMT_USER2 FMT_MASK)");
        font_menu(mnuStyle, "User 3",
                "(change-font FMT_USER3 FMT_MASK)");
        font_menu(mnuStyle, "User 4",
                "(change-font FMT_USER4 FMT_MASK)");
        font_menu(mnuStyle, "User 5",
                "(change-font FMT_USER5 FMT_MASK)");
	return mnuStyle;
}

static GtkWidget *create_color_menu()
{
	mnuColor = gtk_menu_new();
        font_menu(mnuColor, "Black",
                "(change-font COLOR_BLACK COLOR_MASK)");
        font_menu(mnuColor, "Red",
                "(change-font COLOR_RED COLOR_MASK)");
        font_menu(mnuColor, "Green",
                "(change-font COLOR_GREEN COLOR_MASK)");
        font_menu(mnuColor, "Blue",
                "(change-font COLOR_BLUE COLOR_MASK)");
        font_menu(mnuColor, "Yellow",
                "(change-font COLOR_YELLOW COLOR_MASK)");
        font_menu(mnuColor, "Magenta",
                "(change-font COLOR_MAGENTA COLOR_MASK)");
        font_menu(mnuColor, "Cyan",
                "(change-font COLOR_CYAN COLOR_MASK)");
        font_menu(mnuColor, "White",
                "(change-font COLOR_WHITE COLOR_MASK)");
	return mnuColor;
}

static void setup_buttons()
{
  btnFont = gtk_option_menu_new();
  
  gtk_option_menu_set_menu(GTK_OPTION_MENU(btnFont),
			   create_font_menu(btnFont));
  
    /*  gtk_box_pack_start(GTK_BOX(formatbar), btnFont, FALSE, FALSE, 2);
 	gtk_widget_show(btnFont);
 	gtk_tooltips_set_tip(helptext, btnFont, "Change the font family",NULL);
    */ 
  gtk_toolbar_append_widget  (GTK_TOOLBAR(formatbar),
			      GTK_WIDGET(btnFont),
			      "Change the font family",
			      NULL);
  gtk_widget_show(btnFont);
		
  btnSize = gtk_option_menu_new();
  gtk_option_menu_set_menu(GTK_OPTION_MENU(btnSize),
			   create_size_menu(btnSize));
  
        /* gtk_box_pack_start(GTK_BOX(formatbar), btnSize, FALSE, FALSE, 2);
	gtk_widget_show(btnSize);
	gtk_tooltips_set_tip(helptext, btnSize, "Change the font size",NULL);
	*/
  gtk_toolbar_append_widget  (GTK_TOOLBAR(formatbar),
			      GTK_WIDGET(btnSize),
			      "Change the font size",
			      NULL);
  gtk_widget_show(btnSize);
		
  btnStyle = gtk_option_menu_new();
  gtk_option_menu_set_menu(GTK_OPTION_MENU(btnStyle),
			   create_style_menu(btnStyle));
  /*gtk_box_pack_start(GTK_BOX(formatbar), btnStyle, FALSE, FALSE, 2);
    gtk_widget_show(btnStyle);
    gtk_tooltips_set_tip(helptext, btnStyle, "Change the display style",NULL);
  */
	
  gtk_toolbar_append_widget  (GTK_TOOLBAR(formatbar),
			      GTK_WIDGET(btnStyle),
			      "Change the display style",
			      NULL);
  gtk_widget_show(btnStyle);
  
  btnColor = gtk_option_menu_new();
  gtk_option_menu_set_menu(GTK_OPTION_MENU(btnColor),
			   create_color_menu(btnColor));
  /*
    gtk_box_pack_start(GTK_BOX(formatbar), btnColor, FALSE, FALSE, 2);
    gtk_widget_show(btnColor);
    gtk_tooltips_set_tip(helptext, btnColor, "Change the color",NULL);
  */
  
  gtk_toolbar_append_widget  (GTK_TOOLBAR(formatbar),
			      GTK_WIDGET(btnColor),
			      "Change the color",
			      NULL);
  
  gtk_widget_show(btnColor);
  
}

void mainloop()
{
	gtk_main();
}

void init_windows1(int *argc, char **argv)
{
	gtk_set_locale();
	gtk_init(argc, &argv);
	gtk_rc_parse("gsiagrc");
	topLevel = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_widget_set_usize(topLevel, 600, 400);
      	gtk_signal_connect (GTK_OBJECT (topLevel), "destroy",
                          (GtkSignalFunc) execute_callback,
                          (gpointer)"(quit-siag)");
      	gtk_signal_connect (GTK_OBJECT (topLevel), "delete_event",
                          (GtkSignalFunc) execute_callback,
                          (gpointer)"(quit-siag)");

        topbox = gtk_vbox_new(FALSE, 0);        /* not homogenous, no spacing */
        gtk_container_add (GTK_CONTAINER (topLevel), topbox);
        gtk_widget_show(topbox);

        menu_bar = gtk_menu_bar_new();
        gtk_box_pack_start(GTK_BOX(topbox), menu_bar, FALSE, FALSE, 0);
        gtk_widget_show(menu_bar);
	init_menu();

	helptext = gtk_tooltips_new();
	/*changed this bit - RW*/
	toolbarhandlebox  = gtk_handle_box_new ();
	gtk_box_pack_start (GTK_BOX (topbox), toolbarhandlebox,FALSE,FALSE, 2);
	gtk_signal_connect (GTK_OBJECT (toolbarhandlebox),
			    "child_attached",
                            NULL,
                            "attached");
        gtk_signal_connect (GTK_OBJECT (toolbarhandlebox),
                            "child_detached",
                            NULL,
                            "detached");
        gtk_widget_show (toolbarhandlebox);

        toolbar = gtk_toolbar_new (GTK_ORIENTATION_HORIZONTAL, GTK_TOOLBAR_ICONS);
	gtk_container_add(GTK_CONTAINER(toolbarhandlebox), toolbar);
	/*gtk_box_pack_start(GTK_BOX(topbox),toolbar, FALSE, FALSE, 2);
	  toolbar = gtk_hbox_new(FALSE, 0);
         gtk_container_add (GTK_CONTAINER (realtoolbar), toolbar);
	 gtk_widget_show(toolbar); */
	gtk_widget_show(toolbar);
	init_toolbar();
        /*stopped changing */
	


        formatbarhandlebox  = gtk_handle_box_new ();
	gtk_box_pack_start (GTK_BOX (topbox), formatbarhandlebox,FALSE,FALSE, 2);
	gtk_signal_connect (GTK_OBJECT (formatbarhandlebox),
			    "child_attached",
                            NULL,
                            "attached");
        gtk_signal_connect (GTK_OBJECT (formatbarhandlebox),
                            "child_detached",
                            NULL,
                            "detached");
        gtk_widget_show (formatbarhandlebox);
	
	formatbar = gtk_toolbar_new(GTK_ORIENTATION_HORIZONTAL, GTK_TOOLBAR_ICONS);
        gtk_container_add(GTK_CONTAINER(formatbarhandlebox),formatbar);
        gtk_widget_show(formatbar);
	setup_buttons();
	init_toggle();



        label1 = gtk_label_new("Label 1");
        gtk_misc_set_alignment(GTK_MISC(label1), 0, 0);
        gtk_box_pack_start(GTK_BOX(topbox), label1, FALSE, FALSE, 2);
        gtk_widget_show(label1);

        text1 = gtk_entry_new();
	GTK_WIDGET_UNSET_FLAGS(text1, GTK_CAN_DEFAULT);
	gtk_signal_connect_after(GTK_OBJECT(text1),
		"key_press_event",
		(GtkSignalFunc)dialog_key, GTK_OBJECT(text1));
        gtk_box_pack_start(GTK_BOX(topbox), text1, FALSE, FALSE, 2);
        gtk_widget_show(text1);
	gridpane = gtk_vbox_new(TRUE, 0);
        gtk_box_pack_start(GTK_BOX(topbox), gridpane, TRUE, TRUE, 2);
        gtk_widget_show(gridpane);

        label2 = gtk_label_new("Label 2");
        gtk_misc_set_alignment(GTK_MISC(label2), 0, 0);
        gtk_box_pack_start(GTK_BOX(topbox), label2, FALSE, FALSE, 2);
        gtk_widget_show(label2);
}

void activate_window(window *w)
{
	char b[256];

	if (w_list)
		GTK_SGRID(w_list->ui->grid)->visible_cursor = FALSE;
	w_list = w;
	strcpy(b, "Siag: ");
	strncat(b, w->buf->name, 200);
	gtk_window_set_title(GTK_WINDOW(topLevel), b);
	GTK_SGRID(w->ui->grid)->visible_cursor = TRUE;
	gtk_widget_grab_focus(w->ui->grid);	/* grab_default? */
}

window *find_window_by_widget(GtkWidget *wdg)
{
        window *w = w_list;
        do {
                if (w->ui->viewport == wdg || w->ui->selectall == wdg ||
                        w->ui->colnum == wdg || w->ui->rownum == wdg ||
                        w->ui->grid == wdg || w->ui->vscroll == wdg ||
                        w->ui->hscroll == wdg)
                        return w;
                w = w->next;
        } while (w != w_list);
        return NULL;
}

void free_window(window *w)
{
        window *pw;

        for (pw = w_list; pw->next != w && pw->next != pw; pw = pw->next);
        pw->next = w->next;

        if (w_list == w) w_list = w_list->next;
        if (w_list == w) w_list = NULL;
	gtk_widget_destroy(w->ui->viewport);
        cfree(w->ui);
        cfree(w);
}
#if 0
static int width(GtkWidget *w)
{
	gint w1, h1;
	gdk_window_get_size(w->window, &w1, &h1);
	return w1;
}

static int height(GtkWidget *w)
{
	gint w1, h1;
	gdk_window_get_size(w->window, &w1, &h1);
	return h1;
}
#endif
static gchar *ng_colnum_text(GtkSGrid *dummy, gpointer data, gint row, gint col)
{
        static char b[80];
        sprintf(b, "%d", col);
        return b;
}

static gchar *ng_rownum_text(GtkSGrid *dummy, gpointer data, gint row, gint col)
{
        static char b[80];
        sprintf(b, "%d", row);
        return b;
}

static int colnum_click(GtkWidget *w, GdkEventButton *event, gpointer user_data)
{
	gint tx, ty, row, col;
	hide_cur(w_list);
	activate_window(find_window_by_widget(w));
	tx = event->x;
	ty = event->y;
	gtk_sgrid_coord_to_rowcol(GTK_SGRID(w), tx, ty, &row, &col);
	set_point_col(w_list, col);
	set_block(1, col, BUFFER_ROWS, col);
	pr_scr_flag = TRUE;
	show_cur(w_list);
	return TRUE;
}

static int rownum_click(GtkWidget *w, GdkEventButton *event, gpointer user_data)
{
	gint tx, ty, row, col;
	hide_cur(w_list);
	activate_window(find_window_by_widget(w));
	tx = event->x;
	ty = event->y;
	gtk_sgrid_coord_to_rowcol(GTK_SGRID(w), tx, ty, &row, &col);
	set_point_row(w_list, row);
	set_block(row, 1, row, BUFFER_COLS);
	pr_scr_flag = TRUE;
	show_cur(w_list);
	return TRUE;
}

static int grid_click(GtkWidget *w, GdkEventButton *event, gpointer user_data)
{
	gint tx, ty, row, col;
	hide_cur(w_list);
	activate_window(find_window_by_widget(w));
	tx = event->x;
	ty = event->y;
	gtk_sgrid_coord_to_rowcol(GTK_SGRID(w), tx, ty, &row, &col);
	set_point_row(w_list, row);
	set_point_col(w_list, col);
	pr_scr_flag = TRUE;
	show_cur(w_list);
	return TRUE;
}

#define CTRL(c) ((c)&31)
#define CONTROL_MASK 4
#define ALT(c) ((c)|0x80)
#define ALT_MASK 8

static int grid_key(GtkWidget *w, GdkEventKey *event, gpointer user_data)
{
	lastc = event->keyval;

	switch (lastc) {
	case GDK_Home:		lastc = CTRL('a'); break;
	case GDK_Left:		lastc = CTRL('b'); break;
	case GDK_Delete:	lastc = CTRL('d'); break;
	case GDK_End:		lastc = CTRL('e'); break;
	case GDK_Right:		lastc = CTRL('f'); break;
	case GDK_Down:		lastc = CTRL('n'); break;
	case GDK_Up:		lastc = CTRL('p'); break;
	case GDK_Page_Up:	lastc = ALT('v'); break;
	case GDK_Page_Down:	lastc = CTRL('v'); break;
	default:		if (lastc > 255) return TRUE;
	}
	if (event->state & GDK_CONTROL_MASK) lastc = CTRL(lastc);
	if (event->state & ALT_MASK) lastc = ALT(lastc);
	do_cmd(lastc);
	return TRUE;
}
/* These signal handlers for the scrollbars know about the rownum
   and colnum widgets, so that they are scrolled along with the
   grid. Point is also moved to keep it visible. */
static void siag_hadjustment_value_changed(GtkAdjustment *adjustment,
					gpointer data)
{
	window *w;

	g_return_if_fail (adjustment != NULL);
	g_return_if_fail (data != NULL);

	w = (window *)data;
	w->top.col = w->point_pos.col = adjustment->value;
	draw_cells(w);
	draw_colnums(w);
}

static void siag_vadjustment_value_changed(GtkAdjustment *adjustment,
					gpointer data)
{
	window *w;

	g_return_if_fail (adjustment != NULL);
	g_return_if_fail (data != NULL);

	w = (window *)data;
	w->top.row = w->point_pos.row = adjustment->value;
	draw_cells(w);
	draw_rownums(w);
}

static void siag_set_hadjustment (window *w, GtkAdjustment *adjustment)
{
  g_return_if_fail (w != NULL);
  g_return_if_fail (adjustment != NULL);

  if (w->ui->hadjustment) {
    gtk_signal_disconnect_by_data (GTK_OBJECT (w->ui->hadjustment),
                                     (gpointer) w);
    gtk_object_unref (GTK_OBJECT (w->ui->hadjustment));
  }

  w->ui->hadjustment = adjustment;
  gtk_object_ref (GTK_OBJECT (w->ui->hadjustment));

  gtk_signal_connect (GTK_OBJECT (adjustment), "value_changed",
                      (GtkSignalFunc) siag_hadjustment_value_changed,
                      (gpointer) w);
}

static void siag_set_vadjustment (window *w, GtkAdjustment *adjustment)
{
  g_return_if_fail (w != NULL);
  g_return_if_fail (adjustment != NULL);

  if (w->ui->vadjustment) {
    gtk_signal_disconnect_by_data (GTK_OBJECT (w->ui->vadjustment),
                                     (gpointer) w);
    gtk_object_unref (GTK_OBJECT (w->ui->vadjustment));
  }

  w->ui->vadjustment = adjustment;
  gtk_object_ref (GTK_OBJECT (w->ui->vadjustment));

  gtk_signal_connect (GTK_OBJECT (adjustment), "value_changed",
                      (GtkSignalFunc) siag_vadjustment_value_changed,
                      (gpointer) w);
}

static int colnum_height()
{
	return w_list->ui->selectall->allocation.height;
}

window *new_window(buffer *b, window *prev)
{
        window *w;
	int h1 = 20, w1 = 60;
	GtkAdjustment *hadjustment, *vadjustment;

TRACEME((f, "new_window()\n"));

        w = (window *)cmalloc(sizeof(window));

        if (w == NULL) return NULL;

        w->ui = (siag_ui *)cmalloc(sizeof(siag_ui));
        if (w->ui == NULL) {
                cfree(w);
                return NULL;
        }

        w->buf = b;
        w->point_pos.row = 1;
        w->point_pos.col = 1;
        w->blku.row = w->blku.col = 2;
        w->blkl.row = w->blkl.col = 1;
        w->prot = w->top = w->point_pos;

        if (prev == NULL) prev = w;
        else w->next = prev->next;
        prev->next = w;

	w->ui->viewport = gtk_table_new(3, 3, FALSE);
	gtk_table_set_row_spacings(GTK_TABLE(w->ui->viewport), 0);
	gtk_box_pack_start(GTK_BOX(gridpane), w->ui->viewport,
		TRUE, TRUE, 0);
	gtk_widget_show(w->ui->viewport);

	w->ui->selectall = gtk_button_new();
	GTK_WIDGET_UNSET_FLAGS(w->ui->selectall, GTK_CAN_FOCUS);
	gtk_widget_set_usize(w->ui->selectall, 60, 20);
	gtk_table_attach(GTK_TABLE(w->ui->viewport),
		w->ui->selectall, 0, 1, 0, 1,
		GTK_FILL, GTK_FILL, 0, 0);
	gtk_signal_connect_object(GTK_OBJECT(w->ui->selectall),
		"clicked", GTK_SIGNAL_FUNC(execute_callback),
		(gpointer)"(select-all)");
	gtk_tooltips_set_tip(helptext, w->ui->selectall, "Select everything",NULL);
	gtk_widget_show(w->ui->selectall);

	w->ui->colnum = gtk_sgrid_new(1, BUFFER_COLS);
	GTK_SGRID(w->ui->colnum)->default_height = h1;
	GTK_SGRID(w->ui->colnum)->row_height = colnum_height;
	GTK_SGRID(w->ui->colnum)->col_width = ng_col_width;
	GTK_SGRID(w->ui->colnum)->format = rowcol_format;
	GTK_SGRID(w->ui->colnum)->text = ng_colnum_text;
	GTK_SGRID(w->ui->colnum)->max_row = 1;
	GTK_SGRID(w->ui->colnum)->max_col = BUFFER_COLS;
	GTK_SGRID(w->ui->colnum)->data = b;
#if 1	/* lethal */
	gtk_signal_connect(GTK_OBJECT(w->ui->colnum), "button_press_event",
		(GtkSignalFunc)colnum_click, GTK_OBJECT(w->ui->colnum));
#endif
	gtk_table_attach(GTK_TABLE(w->ui->viewport),
		w->ui->colnum, 1, 2, 0, 1,
		GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
	GTK_SGRID(w->ui->colnum)->bg = &w->ui->colnum->style->bg[GTK_STATE_NORMAL];
	gtk_widget_show(w->ui->colnum);

	w->ui->rownum = gtk_sgrid_new(BUFFER_ROWS, 1);
	GTK_SGRID(w->ui->rownum)->default_width = w1;
	GTK_SGRID(w->ui->rownum)->row_height = ng_row_height;
	GTK_SGRID(w->ui->rownum)->format = rowcol_format;
	GTK_SGRID(w->ui->rownum)->text = ng_rownum_text;
	GTK_SGRID(w->ui->rownum)->max_row = BUFFER_ROWS;
	GTK_SGRID(w->ui->rownum)->max_col = 1;
	GTK_SGRID(w->ui->rownum)->data = b;
	gtk_signal_connect(GTK_OBJECT(w->ui->rownum), "button_press_event",
		(GtkSignalFunc)rownum_click, GTK_OBJECT(w->ui->rownum));
	gtk_table_attach(GTK_TABLE(w->ui->viewport),
		w->ui->rownum, 0, 1, 1, 2,
		GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
	GTK_SGRID(w->ui->rownum)->bg = &w->ui->rownum->style->bg[GTK_STATE_NORMAL];
	gtk_widget_show(w->ui->rownum);

	w->ui->grid = gtk_sgrid_new(BUFFER_ROWS, BUFFER_COLS);
	GTK_SGRID(w->ui->grid)->row_height = ng_row_height;
	GTK_SGRID(w->ui->grid)->col_width = ng_col_width;
	GTK_SGRID(w->ui->grid)->format = cell_format;
	GTK_SGRID(w->ui->grid)->text = ng_text;
	GTK_SGRID(w->ui->grid)->max_row = BUFFER_ROWS;
	GTK_SGRID(w->ui->grid)->max_col = BUFFER_COLS;
	GTK_SGRID(w->ui->grid)->data = b;
	GTK_WIDGET_SET_FLAGS(w->ui->grid, GTK_CAN_FOCUS);
	GTK_WIDGET_SET_FLAGS(w->ui->grid, GTK_CAN_DEFAULT);
	gtk_signal_connect(GTK_OBJECT(w->ui->grid), "button_press_event",
		(GtkSignalFunc)grid_click, GTK_OBJECT(w->ui->grid));
	gtk_signal_connect(GTK_OBJECT(w->ui->grid), "key_press_event",
		(GtkSignalFunc)grid_key, GTK_OBJECT(w->ui->grid));
	gtk_table_attach(GTK_TABLE(w->ui->viewport),
		w->ui->grid, 1, 2, 1, 2,
		GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
	gtk_widget_show(w->ui->grid);
	w->ui->hadjustment = w->ui->vadjustment = NULL;
	hadjustment = (GtkAdjustment *)gtk_adjustment_new(1, 1,
				 BUFFER_COLS, 1, 1, 5);
	vadjustment = (GtkAdjustment *)gtk_adjustment_new(1, 1,
				 BUFFER_ROWS, 1, 1, 5);
	siag_set_hadjustment(w, hadjustment);
	siag_set_vadjustment(w, vadjustment);

	w->ui->vscroll = gtk_vscrollbar_new(w->ui->vadjustment);
	GTK_WIDGET_UNSET_FLAGS(w->ui->vscroll, GTK_CAN_FOCUS);
	gtk_range_set_update_policy(GTK_RANGE(w->ui->vscroll), GTK_UPDATE_CONTINUOUS);
	gtk_table_attach(GTK_TABLE(w->ui->viewport),
		w->ui->vscroll, 2, 3, 0, 2,
		0, GTK_EXPAND | GTK_FILL, 0, 0);
	gtk_widget_show(w->ui->vscroll);

	w->ui->hscroll = gtk_hscrollbar_new(w->ui->hadjustment);
	GTK_WIDGET_UNSET_FLAGS(w->ui->hscroll, GTK_CAN_FOCUS);
	gtk_range_set_update_policy(GTK_RANGE(w->ui->hscroll), GTK_UPDATE_CONTINUOUS);
	gtk_table_attach(GTK_TABLE(w->ui->viewport),
		w->ui->hscroll, 0, 2, 2, 3,
		GTK_EXPAND | GTK_FILL, 0, 0, 0);
	gtk_widget_show(w->ui->hscroll);

        return w;
}

int remove_window(window *w)
{
	if (w == w->next) return FALSE;
        free_window(w);
        return TRUE;
}

int split_window(window *w)
{
        window *w2 = new_window(w->buf, w);

        if (w2 == NULL) return FALSE;
        w2->point_pos = w->point_pos;
        w2->top = w->top;
        w2->prot = w->prot;
        return TRUE;
}

void init_windows(buffer * b, int argc, char **argv)
{
        char *p;

	gtk_widget_show(topLevel);
	activate_window(new_window(b, NULL));

        draw_scrollbars(w_list);
        p = ret_text(w_list->buf,
                      w_list->point_pos.row, w_list->point_pos.col);
        if (p == NULL)
                p = "";
        draw_input(p);
        draw_status("");
        draw_cells(w_list);
        draw_colnums(w_list);
        draw_rownums(w_list);

        init_subr_3("add-menu-entry", add_menu_entry);
/*	init_subr_1("helptext-mode", lhelptext_mode);
*/
	init_calc_cmds();
}

void exit_windows()
{
        /* free all buffers */
        while (b_list != NULL)
                free_buffer(b_list);
        while (w_list != NULL)
                free_window(w_list);
}

static void pr_scr()
{
        window *w;
        buffer *b;

        draw_status("");
        b = b_list;
        do {
                if (b->recalc) {
                        int i;
                        b->recalc = 0;
                        for (i = 0; i < recalc; i++)
                                calc_matrix(b);
                }
                b = b->next;
        } while (b != b_list);

        w = w_list;
        do {
                draw_cells(w);
                draw_colnums(w);
                draw_rownums(w);
                draw_scrollbars(w);
                w = w->next;
        } while (w != w_list);
        pr_scr_flag = FALSE;
}       /* pr_scr */

static int cell_next_row(window *w, int row)
{
        if (row+1 == w->prot.row) return w->top.row;
        return row+1;
}

static int cell_next_col(window *w, int col)
{
        if (col+1 == w->prot.col) return w->top.col;
        return col+1;
}

static int cell_prev_row(window *w, int row)
{
        if (row == w->top.row) return w->prot.row-1;
        return row-1;
}

static int cell_prev_col(window *w, int col)
{
        if (col == w->top.col) return w->prot.col-1;
        return col-1;
}

void get_cell_coords(window *w, int top_row, int top_col,
                     int cell_row, int cell_col,
                     int *cell_x, int *cell_y)
{
        int i;

        *cell_y = 0;
        for (i = 1; i < w->prot.row; i++)
                *cell_y += cell_height(w->buf, i);

        while (cell_row < top_row) {
                cell_row = cell_next_row(w, cell_row);
                *cell_y -= cell_height(w->buf, cell_row);
        }
        while (cell_row > top_row) {
                cell_row = cell_prev_row(w, cell_row);
                *cell_y += cell_height(w->buf, cell_row);
        }
        *cell_x = 0;
        for (i = 1; i < w->prot.col; i++)
                *cell_x += cell_width(w->buf, i);

        while (cell_col < top_col) {
                cell_col = cell_next_col(w, cell_col);
                *cell_x -= cell_width(w->buf, cell_col);
        }
        while (cell_col > top_col) {
                cell_col = cell_prev_col(w, cell_col);
                *cell_x += cell_width(w->buf, cell_col);
        }
}

void get_coords_cell(window *w, int top_row, int top_col,
                     int *cur_row, int *cur_col,
                     int cur_x, int cur_y)
{
        int prot_x = 0, prot_y = 0, i;

        for (i = 1; i < w->prot.col; i++)
                cur_x -= cell_width(w->buf, i);
        for (i = 1; i < w->prot.row; i++)
                cur_y -= cell_height(w->buf, i);

        *cur_row = top_row;
        *cur_col = top_col;
        while (cur_y < prot_y && *cur_row > 1) {
                cur_y += cell_height(w->buf, *cur_row);
                (*cur_row) = cell_prev_row(w, *cur_row);
        }
        while (cur_y > cell_height(w->buf, *cur_row) && *cur_row < BUFFER_ROWS) {
                cur_y -= cell_height(w->buf, *cur_row);
                (*cur_row) = cell_next_row(w, *cur_row);
        }
        while (cur_x < prot_x && *cur_col > 1) {
                cur_x += cell_width(w->buf, *cur_col);
                (*cur_col) = cell_prev_col(w, *cur_col);
        }
        while (cur_x > cell_width(w->buf, *cur_col) && *cur_col < BUFFER_COLS) {
                cur_x -= cell_width(w->buf, *cur_col);
                (*cur_col) = cell_next_col(w, *cur_col);
        }
}

void show_format()
{
        static int last_fmt = -1;
        int fmt = ret_format(w_list->buf,
                                w_list->point_pos.row,
                                w_list->point_pos.col);
#if 0	/* FIXME */
        int bold = fmt & BOLD;
        int italic = fmt & ITALIC;
        int hadj = fmt & HADJ_MASK;
        int hadj_left = (hadj == HADJ_LEFT);
        int hadj_center = (hadj == HADJ_CENTER);
        int hadj_right = (hadj == HADJ_RIGHT);
#endif

	if (fmt != last_fmt) {
		gtk_option_menu_set_history(GTK_OPTION_MENU(btnFont),
				(fmt & FONT_MASK) >> 5);
		gtk_option_menu_set_history(GTK_OPTION_MENU(btnSize),
				(fmt & SIZE_MASK));
		gtk_option_menu_set_history(GTK_OPTION_MENU(btnStyle),
				(fmt & FMT_MASK) >> FMT_SHIFT);
		gtk_option_menu_set_history(GTK_OPTION_MENU(btnColor),
				(fmt & COLOR_MASK) >> COLOR_SHIFT);
	}

        last_fmt = fmt;
#if 0
	/* toggle buttons */
	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(cmdBold),
		(bold?1:0));
	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(cmdItalic),
		(italic?1:0));
	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(cmdHLeft),
		(hadj_left?1:0));
	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(cmdHCenter),
		(hadj_center?1:0));
	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(cmdHRight),
		(hadj_right?1:0));
#endif
}

int cursor_visible = FALSE;

void show_cur(window *w)
{
        char *p;
        int top_row = w->top.row;
        int top_col = w->top.col;

	gtk_sgrid_set_point(GTK_SGRID(w->ui->grid),
				w->point_pos.row, w->point_pos.col);
        if (pr_scr_flag) {
                pr_scr();
        } else {
		/* this may have moved the top, so we must check that */
		w->top.row = GTK_SGRID(w->ui->grid)->top_row;
		w->top.col = GTK_SGRID(w->ui->grid)->top_col;
                if (top_row != w->top.row) {
                        draw_rownums(w);
                        draw_vbar(w);
                }
                if (top_col != w->top.col) {
                        draw_colnums(w);
                        draw_hbar(w);
                }
        }
        p = ret_text(w->buf, w->point_pos.row,
                      w->point_pos.col);
        if (p == NULL)
                p = "";
        draw_input(p);
        show_format();
        cursor_visible = TRUE;
}       /* show_cur */

void hide_cur(window *w)
{
}

