/* PSPP - computes sample statistics.
   Copyright (C) 1997, 1998 Free Software Foundation, Inc.
   Written by Ben Pfaff <blp@gnu.org>.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA. */

#include <config.h>
#include <assert.h>
#include <stdio.h>
#include "common.h"
#include "error.h"
#include "getline.h"
#include "output.h"
#include "lexer.h"
#include "lexerP.h"

#include "arena.h"
#include <stdlib.h>

#undef DEBUGGING
/*#define DEBUGGING 1*/
#include "debug-print.h"

void parse (void);
void test_arena (void);
void test_table (void);

/* Program entry point. */
int
main (int argc, char **argv)
{
  void init_glob (int, char **);	/* Exported by glob.c. */
  void parse_command_line (int, char **);	/* Exported by cmdline.c */
  void init_lex (void);		/* Exported by lexer.c. */
  void init_cmd_parser (void);	/* Exported by command.c. */

  /* Initialization. */
  if (!outp_init ())
    hcf (EXIT_FAILURE);
  init_glob (argc, argv);
  parse_command_line (argc, argv);
  if (!outp_read_devices ())
    msg (FE, _("Error initializing output drivers."));
  init_lex ();
  init_cmd_parser ();

  /* Execution. */
  parse ();

  /* Termination. */
  hcf (EXIT_SUCCESS);
}

int parse_cmd (void);

/* Parses the entire script. */
void
parse (void)
{
  void go_interactive (void);	/* Exported by lexer.c */
  int code;

  while (!finished)
    {
      getl_prompt = GETL_PRPT_STANDARD;
      get_token ();

      while (token == STOP)
	if (getl_perform_delayed_reset ())
	  get_token ();
	else
	  return;

      getl_prompt = GETL_PRPT_CONTINUATION;
      code = parse_cmd ();

      assert (code >= -3 && code <= 1);
      if (code <= 0)
	{
	  if (am_reading_script)
	    {
	      static const char *warnings[] =
		{
		  N_("This command not executed."),
		  N_("Skipping the rest of this command.  Part of this "
		     "command may have been executed."),
		  N_("Skipping the rest of this command.  This command was "
		     "fully executed up to this point."),
		  N_("Trailing garbage was encountered following this "
		     "command.  The command was fully executed to this "
		     "point."),
		};

	      msg (SW, gettext (warnings[-code]));
	      error_break ();
	      while (token != STOP && token != '.')
		get_token ();
	    }
	  else
	    discard_line ();
	}

      check_error_count ();
    }
}

#if DEBUGGING
/* Reads one token from the lexer and writes a textual representation
   on stdout for debugging purposes. */
int
dump_token ()
{
  token = yylex ();

  if (curfn)
    printf ("%s:%d\t", curfn, curln);
  switch (token)
    {
    case STOP:
      printf ("dump_token(): STOP\n");
      break;
    case ID:
      printf ("ID\t%s\n", tokstr);
      break;
    case NUM:
      printf ("NUM\t%f\n", tokval);
      break;
    case STRING:
      printf ("STRING\t%s\n", tokstr);
      break;
    case EXP:
      puts ("MISC\tEXP");
      break;
    case 0:
      puts ("MISC\tEOF");
      break;
    case TAGGED_QUOTE:
      puts ("TAGGED-QUOTE");
      break;
    default:
      if (token >= AND && token <= WITH)
	{
	  static const char *kwtab[] =
	    {
	      "AND", "OR", "NOT", "EQ", "GE", "GT", "LE", "LT",
	      "NE", "ALL", "BY", "TO", "WITH",
	    };
	  printf ("RES-KW\t%s\n", kwtab[token - AND]);
	  break;
	}
      printf ("PUNCT\t%c\n", token);
      break;
    }
  return token;
}
#else /* !DEBUGGING */
/* Stubbed out version of dump_token() that simply reads the next
   token. */
int
dump_token (void)
{
  token = yylex ();

  return token;
}
#endif /* !DEBUGGING */
