/* PSPP - computes sample statistics.
   Copyright (C) 1997, 1998 Free Software Foundation, Inc.
   Written by Ben Pfaff <blp@gnu.org>.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA. */

/* AIX requires this to be the first thing in the file.  */
#include <config.h>
#if __GNUC__
#define alloca __builtin_alloca
#else
#if HAVE_ALLOCA_H
#include <alloca.h>
#else
#ifdef _AIX
#pragma alloca
#else
#ifndef alloca			/* predefined by HP cc +Olibcalls */
char *alloca ();
#endif
#endif
#endif
#endif

#include <assert.h>
#include <stdlib.h>
#include <ctype.h>
#include <errno.h>
#include "common.h"
#include "error.h"
#include "str.h"
#include "filename.h"
#include "settings.h"
#include "version.h"

#undef DEBUGGING
/*#define DEBUGGING 1 */
#include "debug-print.h"

/* PORTME: Everything in this file is system dependent. */

#if unix
#include <pwd.h>
#if HAVE_UNISTD_H
#include <unistd.h>
#endif
#include "stat.h"
#endif

#if __WIN32__
#define NOGDI
#define NOUSER
#define NONLS
#include <win32/windows.h>
#endif

#if __DJGPP__
#include <sys/stat.h>
#endif

const char *config_path;

void
init_filename (void)
{
  config_path = getenv_default ("STAT_CONFIG_PATH", default_config_path);
}

/* Allocates room for at least AMT + *M_OUT characters in *OUTPUT and
   sets *END and *OP appropriately. */
static void
expand_line (char **output, char **end, char **op, int *m_out, int amt)
{
  int n_out = *op - *output;

  amt += *m_out;
  do
    *m_out *= 2;
  while (*m_out < amt);
  *output = xrealloc (*output, *m_out);
  *end = &(*output)[*m_out];
  *op = &(*output)[n_out];
}

#define EXPAND_LINE(AMT)					\
	do							\
	  {							\
	    if (op + AMT >= end)				\
	      expand_line (&output, &end, &op, &m_out, AMT);	\
	  }							\
	while (0)

/* Substitutes $variables as defined by GETENV into INPUT and returns
   a copy of the resultant string.  Supports $var and ${var} syntaxes;
   $$ substitutes as $. */
char *
interp_vars (const char *input, const char *(*getenv) (const char *))
{
  char *output, *end;
  const char *ip;
  char *op;
  int m_out = 1024;

  if (NULL == strchr (input, '$'))
    return xstrdup (input);
  output = xmalloc (m_out);
  end = &output[m_out];
  ip = input;
  op = output;
  while (*ip)
    if (*ip != '$')
      {
	EXPAND_LINE (1);
	*op++ = *ip++;
	continue;
      }
    else
      {
	ip++;
	if (*ip == '$')
	  {
	    EXPAND_LINE (1);
	    *op++ = *ip++;
	    continue;
	  }
	if (*ip == '{')
	  {
	    char *sop = op;
	    const char *val;

	    ip++;
	    while (*ip && *ip != '}')
	      {
		EXPAND_LINE (1);
		*op++ = *ip++;
	      }
	    *op = 0;

	    val = getenv (sop);
	    op = sop;
	    if (val)
	      {
		EXPAND_LINE (strlen (val));
		op = stpcpy (op, val);
	      }
	    if (*ip == '}')
	      ip++;
	  }
	else
	  {
	    char *sop = op;
	    const char *val;

	    if (isalpha ((unsigned char) *ip))
	      while (*ip && isalpha ((unsigned char) *ip))
		{
		  EXPAND_LINE (1);
		  *op++ = *ip++;
		}
	    else if (*ip)
	      {
		EXPAND_LINE (1);
		*op++ = *ip++;
	      }
	    *op = 0;

	    val = getenv (sop);
	    op = sop;
	    if (val)
	      {
		EXPAND_LINE (strlen (val));
		op = stpcpy (op, val);
	      }
	  }
      }
  *op++ = 0;
  return xrealloc (output, op - output);
}

/* Returns the current working directory, as a malloc()'d string.
   From libc.info. */
char *
gnu_getcwd (void)
{
  int size = 100;
  char *buffer = (char *) xmalloc (size);
     
  while (1)
    {
      char *value = getcwd (buffer, size);
      if (value != 0)
	return buffer;
      size *= 2;
      free (buffer);
      buffer = (char *) xmalloc (size);
    }
}

#if unix
/* Expands csh tilde notation from the path INPUT into a malloc()'d
   returned string. */
char *
tilde_expand (const char *input)
{
  char *output, *end;
  const char *ip;
  char *op;
  int m_out = 1024;

  if (NULL == strchr (input, '~'))
    return xstrdup (input);
  output = xmalloc (m_out);
  end = &output[m_out];
  ip = input;
  op = output;
  for (ip = input; *ip;)
    {
      if (*ip != '~')
	{
	  EXPAND_LINE (1);
	  *op++ = *ip++;
	}
      else if (ip == input || ip[-1] == PATH_DELIMITER)
	{
	  char *cp;

	  ip++;
	  cp = strchr (ip, DIR_SEPARATOR);
	  if (!cp)
	    cp = strchr (ip, 0);
	  if (cp > ip)
	    {
	      struct passwd *pwd;
	      char username[9];

	      strncpy (username, ip, cp - ip + 1);
	      username[8] = 0;
	      pwd = getpwnam (username);

	      if (!pwd || !pwd->pw_dir)
		{
		  EXPAND_LINE (1);
		  *op++ = *ip++;
		}
	      else
		{
		  EXPAND_LINE (strlen (pwd->pw_dir));
		  op = stpcpy (op, pwd->pw_dir);
		}
	    }
	  else
	    {
	      const char *home = blp_getenv ("HOME");
	      if (!home)
		{
		  EXPAND_LINE (1);
		  *op++ = *ip++;
		}
	      else
		{
		  EXPAND_LINE (strlen (home));
		  op = stpcpy (op, home);
		}
	    }
	  ip = cp;
	}
      else
	{
	  EXPAND_LINE (1);
	  *op++ = *ip++;
	}
    }
  *op++ = 0;
  return xrealloc (output, op - output);
}
#else /* !unix */
char *
tilde_expand (char *input)
{
  return xstrdup (input);
}
#endif /* !unix */

/* normalize_filename(): This very OS-dependent routine canonicalizes
   filename FN1.  The filename should not need to be the name of an
   existing file.  Returns a malloc()'d copy of the canonical name.
   This function must always succeed; if it needs to bail out then it
   should return xstrdup(FN1).  */
#if unix
/* Stolen from libc.info but heavily modified, this is a wrapper
   around readlink() that allows for arbitrary filename length. */
char *
readlink_malloc (const char *filename)
{
  int size = 128;

  while (1)
    {
      char *buffer = (char *) xmalloc (size);
      int nchars;
#if __CHECKER__
      memset (buffer, 0, size);
#endif
      nchars = readlink (filename, buffer, size);
      if (nchars == -1)
	{
	  free (buffer);
	  return NULL;
	}
      if (nchars < size - 1)
	{
	  buffer[nchars] = 0;
	  return buffer;
	}
      free (buffer);
      size *= 2;
    }
}

char *
normalize_filename (const char *filename)
{
  const char *src;
  char *fn1, *fn2, *dest;
  int maxlen;

  if (is_special_filename (filename))
    return xstrdup (filename);
  
  fn1 = tilde_expand (filename);

  /* Follow symbolic links. */
  while (1)
    {
      fn2 = fn1;
      fn1 = readlink_malloc (fn1);
      if (!fn1)
	{
	  fn1 = fn2;
	  break;
	}
      free (fn2);
    }

  maxlen = strlen (fn1) * 2;
  if (maxlen < 31)
    maxlen = 31;
  dest = fn2 = xmalloc (maxlen + 1);
  src = fn1;

  if (*src == DIR_SEPARATOR)
    *dest++ = *src++;
  else
    {
      errno = 0;
#if __CHECKER__
      memset (dest, 0, maxlen);
#endif
      while (getcwd (dest, maxlen - (dest - fn2)) == NULL && errno == ERANGE)
	{
	  maxlen *= 2;
	  dest = fn2 = xrealloc (fn2, maxlen + 1);
#if __CHECKER__
	  memset (dest, 0, maxlen);
#endif
	  errno = 0;
	}
      if (errno)
	{
	  free (fn1);
	  free (fn2);
	  return NULL;
	}
      dest = strchr (fn2, '\0');
      if (dest - fn2 >= maxlen)
	{
	  int ofs = dest - fn2;
	  maxlen *= 2;
	  fn2 = xrealloc (fn2, maxlen + 1);
	  dest = fn2 + ofs;
	}
      if (dest[-1] != DIR_SEPARATOR)
	*dest++ = DIR_SEPARATOR;
    }

  while (1)
    {
      int c, f;

      c = *src++;

      f = 0;
      if (c == DIR_SEPARATOR || c == 0)
	{
	  /* remove `./', `../' from directory */
	  if (dest[-1] == '.' && dest[-2] == DIR_SEPARATOR)
	    dest--;
	  else if (dest[-1] == '.' && dest[-2] == '.' && dest[-3] == DIR_SEPARATOR)
	    {
	      dest -= 3;
	      if (dest == fn2)
		dest++;
	      while (dest[-1] != DIR_SEPARATOR)
		dest--;
	    }
	  else if (dest[-1] != DIR_SEPARATOR)	/* remove extra slashes */
	    f = 1;

	  if (c == 0)
	    {
	      if (dest[-1] == DIR_SEPARATOR && dest > fn2 + 1)
		dest--;
	      *dest = 0;
	      free (fn1);

	      return xrealloc (fn2, strlen (fn2) + 1);
	    }
	}
      else
	f = 1;

      if (f)
	{
	  if (dest - fn2 >= maxlen)
	    {
	      int ofs = dest - fn2;
	      maxlen *= 2;
	      fn2 = xrealloc (fn2, maxlen + 1);
	      dest = fn2 + ofs;
	    }
	  *dest++ = c;
	}
    }
}
#elif __WIN32__
char *
normalize_filename (char *fn1)
{
  DWORD len;
  DWORD success;
  char *fn2;

  /* Don't change special filenames. */
  if (is_special_filename (filename))
    return xstrdup (filename);

  /* First find the required buffer length. */
  len = GetFullPathName (fn1, 0, NULL, NULL);
  if (!len)
    {
      fn2 = xstrdup (fn1);
      return fn2;
    }

  /* Then make a buffer that big. */
  fn2 = xmalloc (len);
  success = GetFullPathName (fn1, len, fn2, NULL);
  if (success >= len || success == 0)
    {
      free (fn2);
      fn2 = xstrdup (fn1);
      return fn2;
    }
  return fn2;
}
#elif __BORLANDC__
char *
normalize_filename (char *fn1)
{
  char *fn2 = _fullpath (NULL, fn1, 0);
  if (fn2)
    {
      char *cp;
      for (cp = fn2; *cp; cp++)
	*cp = toupper ((unsigned char) (*cp));
      return fn2;
    }
  return xstrdup (fn1);
}
#elif __DJGPP__
char *
normalize_filename (char *fn1)
{
  char *fn2 = xmalloc (1024);
  _fixpath (fn1, fn2);
  fn2 = xrealloc (fn2, strlen (fn2) + 1);
  return fn2;
}
#else /* not Lose32, Unix, or DJGPP */
char *
normalize_filename (char *fn)
{
  return xstrdup (fn);
}
#endif /* not Lose32, Unix, or DJGPP */

/* Simulates $VER and $ARCH environment variables. */
const char *
blp_getenv (const char *s)
{
  if (streq (s, "VER"))
    return getenv_default ("STAT_VER", bare_version);
  else if (streq (s, "ARCH"))
    return getenv_default ("STAT_ARCH", host_system);
  else
    return getenv (s);
}

/* Returns nonzero if file with name NAME exists. */
int
file_exists (const char *name)
{
#if unix
  struct stat temp;

  return stat (name, &temp) == 0;
#else
  FILE *f = fopen (name, "r");
  if (!f)
    return 0;
  fclose (f);
  return 1;
#endif
}

/* Returns nonzero iff NAME specifies an absolute filename. */
int
absolute_filename_p (const char *name)
{
#if unix
  if (name[0] == '/'
      || !strncmp (name, "./", 2)
      || !strncmp (name, "../", 3)
      || name[0] == '~')
    return 1;
#elif __MSDOS__
  if (name[0] == '\\'
      || !strncmp (name, ".\\", 2)
      || !strncmp (name, "..\\", 3)
      || (name[0] && name[1] == ':'))
    return 1;
#endif
  
  return 0;
}
  
/* Searches for a configuration file with name NAME in the path given
   by PATH, which is tilde- and environment-interpolated.  Directories
   in PATH are delimited by PATH_DELIMITER, defined in <pref.h>.
   Returns the malloc'd full name of the first file found, or NULL if
   none is found.

   If PREPEND is non-NULL, then it is prepended to each filename;
   i.e., it looks like PREPEND/PATH_COMPONENT/NAME.  This is not done
   with absolute directories in the path. */
#if unix || __MSDOS__ || __WIN32__
char *
search_path (const char *name, const char *path, const char *prepend)
{
  char *subst_path;

  char *fn = NULL;
  int fn_size = 0;
  int fn_len = 0;

  char *bp, *ep;

  int boilerplate_len;

  boilerplate_len = 1 + strlen (name);
  if (prepend)
    boilerplate_len += 1 + strlen (prepend);

  if (absolute_filename_p (name))
    return tilde_expand (name);
  
  {
    char *temp = interp_vars (path, blp_getenv);
    bp = subst_path = tilde_expand (temp);
    free (temp);
  }

  verbose_msg (4, _("Searching for `%s'..."), name);

  while (1)
    {
      if (0 == *bp)
	{
	  verbose_msg (4, _("Search unsuccessful!"));
	  free (fn);
	  free (subst_path);
	  return NULL;
	}

      for (ep = bp; *ep && *ep != PATH_DELIMITER; ep++)
	;

      fn_len = (ep - bp) + boilerplate_len;
      if (fn_len > fn_size)
	{
	  fn = xrealloc (fn, fn_len + 1);
	  fn_size = fn_len;
	}

      /* Paste together PREPEND/PATH/NAME. */
      {
	char *cp = fn;

	if (prepend && !absolute_filename_p (bp))
	  {
	    cp = stpcpy (cp, prepend);
	    *cp++ = DIR_SEPARATOR;
	  }
	memcpy (cp, bp, ep - bp);
	cp += ep - bp;
	if (ep - bp && cp[-1] != DIR_SEPARATOR)
	  *cp++ = DIR_SEPARATOR;
	strcpy (cp, name);
      }
      
      verbose_msg (5, " - %s", fn);
      if (file_exists (fn))
	{
	  verbose_msg (4, _("Found `%s'."), fn);
	  free (subst_path);
	  return fn;
	}

      if (0 == *ep)
	{
	  verbose_msg (4, _("Search unsuccessful!"));
	  free (subst_path);
	  free (fn);
	  return NULL;
	}
      bp = ep + 1;
    }
}
#else /* not unix, msdog, lose32 */
char *
search_path (char *name, char *path, char *prepend)
{
  size_t size = strlen (path) + 1 + strlen (name) + 1;
  char *string;
  char *cp;
  
  if (prepend)
    size += strlen (prepend) + 1;
  string = xmalloc (size);
  
  cp = string;
  if (prepend)
    {
      cp = stpcpy (cp, prepend);
      *cp++ = DIR_SEPARATOR;
    }
  cp = stpcpy (cp, path);
  *cp++ = DIR_SEPARATOR;
  strcpy (cp, name);

  return string;
}
#endif /* not unix, msdog, lose32 */

/* Returns getenv(KEY) if that's non-NULL; else returns DEF. */
const char *
getenv_default (const char *key, const char *def)
{
  const char *value = getenv (key);
  return value ? value : def;
}

/* Used for giving an error message on a set_safer security
   violation. */
static FILE *
safety_violation (const char *fn)
{
  msg (SE, _("Not opening pipe file `%s' because SAFER option set."), fn);
  errno = EPERM;
  return NULL;
}

/* As a general comment on the following routines, a `sensible value'
   for errno includes 0 if there is no associated system error.  The
   routines will only set errno to 0 if there is an error in a
   callback that sets errno to 0; they themselves won't. */

/* File open routine that understands `-' as stdin/stdout and `|cmd'
   as a pipe to command `cmd'.  Returns resultant FILE on success,
   NULL on failure.  If NULL is returned then errno is set to a
   sensible value.  */
FILE *
open_file (const char *fn, const char *mode)
{
  struct stdfile
    {
      const char *s;
      FILE *f;
      char mode;
    };

  static struct stdfile filetab[] =
    {
      {"-", stdin, 'r'},
      {"-", stdout, 'w'},
      {"stdin", stdin, 'r'},
      {"stdout", stdout, 'w'},
      {"stderr", stderr, 'w'},
      {NULL, 0, 0},
    };

  struct stdfile *iter;

  assert (mode[0] == 'r' || mode[0] == 'w');

  for (iter = filetab; iter->s; iter++)
    if (streq (iter->s, fn) && iter->mode == mode[0])
      return iter->f;
  
#if unix
  if (fn[0] == '|')
    {
      if (set_safer)
	return safety_violation (fn);
      return popen (&fn[1], mode);
    }
  else if (*fn && fn[strlen (fn) - 1] == '|')
    {
      char *s;
      FILE *f;

      if (set_safer)
	return safety_violation (fn);
      
      s = local_alloc (strlen (fn));
      memcpy (s, fn, strlen (fn) - 1);
      s[strlen (fn) - 1] = 0;
      
      f = popen (s, mode);

      local_free (s);

      return f;
    }
  else
#endif
    {
      FILE *f = fopen (fn, mode);
      if (f && mode[0] == 'w')
	setvbuf (f, NULL, _IOLBF, 0);
      return f;
    }
}

/* Counterpart to open_file that closes file F with name FN; returns 0
   on success, EOF on failure.  If EOF is returned, errno is set to a
   sensible value. */
int
close_file (FILE *f, const char *fn)
{
  if (streq (fn, "-"))
    return 0;
#if unix
  else if (fn[0] == '|' || (*fn && fn[strlen (fn) - 1] == '|'))
    return pclose (f), 0;
#endif
  else
    return fclose (f);
}

/* File open routine that extends open_file().  Opens or reopens a
   file according to the contents of file_ext F.  Returns nonzero on
   success.  If 0 is returned, errno is set to a sensible value. */
int
open_file_ext (file_ext * f)
{
  char *p;

  p = strstr (f->filename, "%d");
  if (p)
    {
      char *s = local_alloc (strlen (f->filename) + INT_DIGITS - 1);
      char *cp;

      memcpy (s, f->filename, p - f->filename);
      cp = spprintf (&s[p - f->filename], "%d", *f->sequence_no);
      strcpy (cp, &p[2]);

      if (f->file)
	{
	  int error = 0;

	  if (f->preclose)
	    if (f->preclose (f) == 0)
	      error = errno;

	  if (EOF == close_file (f->file, f->filename) || error)
	    {
	      f->file = NULL;
	      local_free (s);
	      if (error)
		errno = error;
	      return 0;
	    }
	  f->file = NULL;
	}

      f->file = open_file (s, f->mode);
      local_free (s);

      if (f->file && f->postopen)
	if (f->postopen (f) == 0)
	  {
	    int error = errno;
	    close_file (f->file, f->filename);
	    errno = error;
	    return 0;
	  }
      return (f->file != NULL);
    }
  else if (f->file)
    return 1;
  else
    {
      f->file = open_file (f->filename, f->mode);
      if (f->file && f->postopen)
	if (f->postopen (f) == 0)
	  {
	    int error = errno;
	    close_file (f->file, f->filename);
	    errno = error;
	    return 0;
	  }
      return (f->file != NULL);
    }
}

/* Properly closes the file associated with file_ext F, if any.
   Return nonzero on success.  If zero is returned, errno is set to a
   sensible value. */
int
close_file_ext (file_ext * f)
{
  if (f->file)
    {
      int error = 0;

      if (f->preclose)
	if (f->preclose (f) == 0)
	  error = errno;

      if (EOF == close_file (f->file, f->filename) || error)
	{
	  f->file = NULL;
	  if (error)
	    errno = error;
	  return 0;
	}
      f->file = NULL;
    }
  return 1;
}

/* Returns the directory part of the file FILE, as a malloc()'d
   string. */
char *
blp_dirname (const char *file)
{
  const char *p = strrchr (file, DIR_SEPARATOR);
  char *s;
  
  if (p == NULL)
    p = file;
  s = xmalloc (p - file + 1);
  strncpy (s, file, p - file);
  s[p - file] = 0;

  return s;
}

/* Prepends directory DIR to filename FILE and returns a malloc()'d
   copy of it. */
char *
prepend_dir (const char *file, const char *dir)
{
  char *temp;
  char *cp;
  
  if (absolute_filename_p (file))
    return xstrdup (file);

  temp = xmalloc (strlen (file) + 1 + strlen (dir) + 1);
  cp = stpcpy (temp, dir);
  if (cp != temp && cp[-1] != DIR_SEPARATOR)
    *cp++ = DIR_SEPARATOR;
  cp = stpcpy (cp, file);

  return temp;
}

/* Returns 1 if the filename specified is a virtual file that doesn't
   really exist on disk, 0 if it's a real filename. */
int
is_special_filename (const char *filename)
{
  if (streq (filename, "-") || streq (filename, "stdin")
      || streq (filename, "stdout") || streq (filename, "stderr")
#if unix
      || filename[0] == '|'
      || (*filename && filename[strlen (filename) - 1] == '|')
#endif
      )
    return 1;

  return 0;
}


